// Types for the streaming response
interface StreamChunk {
  type: 'text' | 'complete' | 'error';
  content: string;
  full_text?: string;
  status: 'generating' | 'complete' | 'error';
  model_info?: {
    model_name: string;
    device: string;
    max_tokens: number;
  };
}

interface QuestionRequest {
  question: string;
  streaming: boolean;
  context?: Array<{ question: string; answer: string }>;
}

// Export the main class
export class StudentQuestionClient {
  private baseUrl: string;

  constructor(baseUrl: string = 'https://myadvisor.cs.uct.ac.za/bad') {
    this.baseUrl = baseUrl;
  }

  // Method 1: Using fetch with ReadableStream (Recommended)
  async askQuestionStream(
    question: string,
    context: Array<{ question: string; answer: string }> = [],
    onChunk: (chunk: StreamChunk) => void,
    onComplete: (fullText: string) => void,
    onError: (error: string) => void
  ): Promise<void> {
    try {
      console.log(`Attempting to connect to: ${this.baseUrl}/ask/stream-json`);
      console.log(`Sending question: ${question}`);
      console.log(`Context:`, context);
      
      const response = await fetch(`${this.baseUrl}/ask/stream-json`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          question,
          context,
        } as QuestionRequest),
        // Add timeout and other fetch options
        signal: AbortSignal.timeout(60000), // 60 second timeout
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const reader = response.body?.getReader();
      if (!reader) {
        throw new Error('No response body');
      }

      const decoder = new TextDecoder();
      let buffer = '';

      try {
        while (true) {
          const { done, value } = await reader.read();
          
          if (done) break;

          // Decode the chunk and add to buffer
          buffer += decoder.decode(value, { stream: true });
          
          // Process complete lines (NDJSON format)
          const lines = buffer.split('\n');
          buffer = lines.pop() || ''; // Keep incomplete line in buffer

          for (const line of lines) {
            if (line.trim()) {
              try {
                const chunk: StreamChunk = JSON.parse(line);
                onChunk(chunk);

                if (chunk.type === 'complete') {
                  onComplete(chunk.full_text || '');
                  return;
                } else if (chunk.type === 'error') {
                  onError(chunk.content);
                  return;
                }
              } catch (parseError) {
                console.error('Error parsing JSON chunk:', parseError, 'Line:', line);
              }
            }
          }
        }
      } finally {
        reader.releaseLock();
      }
    } catch (error) {
      console.error('Fetch error:', error);
      onError(error instanceof Error ? error.message : 'Unknown error occurred');
    }
  }

  async getFaculties(): Promise<any[]> {
    const res = await fetch(`${this.baseUrl}/faculties/`, {
      method: "GET",
    });

    if (!res.ok) {
      throw new Error(`Failed to fetch faculties: ${res.statusText}`);
    }

    return res.json();
  }

  // Method 2: Using EventSource (if you modify backend to use SSE format)
  askQuestionSSE(
    question: string,
    context: Array<{ question: string; answer: string }> = [],
    onChunk: (chunk: StreamChunk) => void,
    onComplete: (fullText: string) => void,
    onError: (error: string) => void
  ): EventSource {
    // Note: This would require modifying your backend to use SSE format
    // Instead of NDJSON. For now, use the fetch method above.
    const eventSource = new EventSource(
      `${this.baseUrl}/ask/stream-sse?question=${encodeURIComponent(question)}`
    );

    eventSource.onmessage = (event) => {
      try {
        const chunk: StreamChunk = JSON.parse(event.data);
        onChunk(chunk);

        if (chunk.type === 'complete') {
          onComplete(chunk.full_text || '');
          eventSource.close();
        } else if (chunk.type === 'error') {
          onError(chunk.content);
          eventSource.close();
        }
      } catch (error) {
        onError('Error parsing server response');
      }
    };

    eventSource.onerror = () => {
      onError('Connection error');
      eventSource.close();
    };

    return eventSource;
  }

  // Method 3: Simple async generator approach
  async* streamQuestion(
    question: string,
    context: Array<{ question: string; answer: string }> = []
  ): AsyncGenerator<StreamChunk, void, unknown> {
    const response = await fetch(`${this.baseUrl}/ask/stream-json`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        question,
        context,
      } as QuestionRequest),
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const reader = response.body?.getReader();
    if (!reader) {
      throw new Error('No response body');
    }

    const decoder = new TextDecoder();
    let buffer = '';

    try {
      while (true) {
        const { done, value } = await reader.read();
        
        if (done) break;

        buffer += decoder.decode(value, { stream: true });
        const lines = buffer.split('\n');
        buffer = lines.pop() || '';

        for (const line of lines) {
          if (line.trim()) {
            try {
              const chunk: StreamChunk = JSON.parse(line);
              yield chunk;
              
              if (chunk.type === 'complete' || chunk.type === 'error') {
                return;
              }
            } catch (parseError) {
              console.error('Error parsing JSON chunk:', parseError);
            }
          }
        }
      }
    } finally {
      reader.releaseLock();
    }
  }

  // Test connection method
  async testConnection(): Promise<boolean> {
    try {
      console.log(`Testing connection to: ${this.baseUrl}`);
      const response = await fetch(`${this.baseUrl}/health`, {
        method: 'GET',
        signal: AbortSignal.timeout(5000), // 5 second timeout for health check
      });
      console.log(`Health check response: ${response.status}`);
      return response.ok;
    } catch (error) {
      console.error('Health check failed:', error);
      return false;
    }
  }

  // Non-streaming method for comparison
  async askQuestion(question: string): Promise<any> {
    const response = await fetch(`${this.baseUrl}/ask`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        question,
        streaming: false
      } as QuestionRequest),
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    return response.json();
  }
}

// Export interfaces as well
export type { StreamChunk, QuestionRequest };

// Example usage in a React component or vanilla JS
export class ChatInterface {
  private client: StudentQuestionClient;
  private currentAnswer: string = '';

  constructor() {
    this.client = new StudentQuestionClient();
  }

  // Example usage with callbacks
  async handleQuestion(question: string, updateUI: (text: string) => void) {
    this.currentAnswer = '';

    await this.client.askQuestionStream(
      question,
      [],
      // onChunk
      (chunk) => {
        if (chunk.type === 'text') {
          this.currentAnswer += chunk.content;
          updateUI(this.currentAnswer);
        }
      },
      // onComplete
      (fullText) => {
        console.log('Generation complete:', fullText);
        updateUI(fullText);
      },
      // onError
      (error) => {
        console.error('Error:', error);
        updateUI(`Error: ${error}`);
      }
    );
  }

  // Example usage with async generator
  async handleQuestionWithGenerator(question: string, updateUI: (text: string) => void) {
    this.currentAnswer = '';
    
    try {
      for await (const chunk of this.client.streamQuestion(question)) {
        if (chunk.type === 'text') {
          this.currentAnswer += chunk.content;
          updateUI(this.currentAnswer);
        } else if (chunk.type === 'complete') {
          console.log('Generation complete');
          break;
        } else if (chunk.type === 'error') {
          updateUI(`Error: ${chunk.content}`);
          break;
        }
      }
    } catch (error) {
      updateUI(`Error: ${error}`);
    }
  }
}

// Example vanilla JS usage
export function setupChatInterface() {
  const client = new StudentQuestionClient();
  const questionInput = document.getElementById('question-input') as HTMLInputElement;
  const answerDiv = document.getElementById('answer') as HTMLDivElement;
  const askButton = document.getElementById('ask-button') as HTMLButtonElement;

  askButton.addEventListener('click', async () => {
    const question = questionInput.value;
    if (!question.trim()) return;

    answerDiv.innerHTML = '';
    askButton.disabled = true;

    await client.askQuestionStream(
      question,
      [],
      // onChunk - update UI with each chunk
      (chunk) => {
        if (chunk.type === 'text') {
          answerDiv.innerHTML += chunk.content;
        }
      },
      // onComplete
      (fullText) => {
        console.log('Complete answer:', fullText);
        askButton.disabled = false;
      },
      // onError
      (error) => {
        answerDiv.innerHTML = `<div style="color: red;">Error: ${error}</div>`;
        askButton.disabled = false;
      }
    );
  });
}