export async function getFaculties(): Promise<any[]> {
  const res = await fetch(`https://myadvisor.cs.uct.ac.za/backend/faculties/`, {
    method: "GET",
  });

  if (!res.ok) {
    throw new Error(`Failed to fetch faculties: ${res.statusText}`);
  }

  return res.json();
}

export interface Program {
  program_code: string;
  program_name: string;
  description?: string;
  // add more fields if your API returns them
}

export async function getProgramsByFaculty(facultyCode: string): Promise<Program[]> {
  const res = await fetch(
    `https://myadvisor.cs.uct.ac.za/backend/faculties/${facultyCode}/programs/`,
    { method: "GET" }
  );

  if (!res.ok) {
    throw new Error(`Failed to fetch programs for faculty ${facultyCode}: ${res.statusText}`);
  }

  return res.json();
}

export interface Course {
  code: string;
  name: string;
  credits: number;
  level: number;
  semester: string;
  department: string;
  convener: string;
  entry_requirements?: string;
  corequisites?: string;
  objective?: string;
  course_outline: string;
  lecture_times: string;
  dp_requirements: string;
  assessment: string;
  type: string;
  notes?: string;
}

export interface YearCourses {
  year: number;
  items: Course[];
  required_count?: number;
  courses?: Course[]; // sometimes grouped differently
  groups?: { required_count: number; courses: Course[] }[];
}

export interface Degree {
  degree_code: string;
  degree_name: string;
  notes?: string;
  courses: YearCourses[];
}

export async function getDegreesByProgram(
  facultyCode: string,
  programCode: string
): Promise<Degree[]> {
  const res = await fetch(
    `https://myadvisor.cs.uct.ac.za/backend/faculties/${facultyCode}/programs/${programCode}/degrees/`,
    { method: "GET" }
  );

  if (!res.ok) {
    throw new Error(
      `Failed to fetch degrees for program ${programCode} in faculty ${facultyCode}: ${res.statusText}`
    );
  }

  return res.json();
}

export interface Major {
  major_code: string;
  major_name: string;
  major_notes?: string;
  courses?: YearCourses[]; // Add courses structure to match Degree
}

export async function getMajorsByProgram(
  facultyCode: string,
  programCode: string
): Promise<Major[]> {
  const res = await fetch(
    `https://myadvisor.cs.uct.ac.za/backend/faculties/${facultyCode}/programs/${programCode}/majors/`,
    { method: "GET" }
  );

  if (!res.ok) {
    throw new Error(
      `Failed to fetch majors for program ${programCode} in faculty ${facultyCode}: ${res.statusText}`
    );
  }

  return res.json();
}

// Unified Academic Program Types
export interface AcademicProgramBase {
  notes?: string;
  courses?: YearCourses[];
}

export interface DegreeProgram extends AcademicProgramBase {
  degree_code: string;
  degree_name: string;
  major_code?: never;
  major_name?: never;
}

export interface MajorProgram extends AcademicProgramBase {
  major_code: string;
  major_name: string;
  degree_code?: never;
  degree_name?: never;
}

export type AcademicProgram = DegreeProgram | MajorProgram;

// Helper function to check if it's a major
export function isMajorProgram(program: AcademicProgram): program is MajorProgram {
  return 'major_code' in program;
}

// Helper function to get the unified code
export function getProgramCode(program: AcademicProgram): string {
  return isMajorProgram(program) ? program.major_code : program.degree_code;
}

// Helper function to get the unified name
export function getProgramName(program: AcademicProgram): string {
  return isMajorProgram(program) ? program.major_name : program.degree_name;
}

// Helper function to convert Degree to AcademicProgram
export function degreeToAcademicProgram(degree: Degree): AcademicProgram {
  return {
    degree_code: degree.degree_code,
    degree_name: degree.degree_name,
    notes: degree.notes,
    courses: degree.courses
  };
}

// Helper function to convert Major to AcademicProgram
export function majorToAcademicProgram(major: Major): AcademicProgram {
  return {
    major_code: major.major_code,
    major_name: major.major_name,
    notes: major.major_notes,
    courses: major.courses || []
  };
}