// services/api/auth.ts - UPDATED with correct types based on actual API response

export interface LoginRequest {
  username: string;
  password: string;
}

export interface SignupRequest {
  username: string;
  email: string;
  password: string;
  confirmPassword?: string; // Optional since we don't send this to backend
}

// Updated interface to match your ACTUAL API response structure
export interface ApiResponse {
  status: number;
  message?: string;
  data?: {
    is_advisor: boolean;
    username: string;
    email?: string; // Make optional since it might not always be present
    red_dot: boolean; // Now included based on your actual response
    inbox: Array<{
      category: string;
      subject: string;
      question: string;
      answer: string | null;
      student: string;
      advisor: string;
      read: boolean;
    }>;
  };
}

export class AuthAPI {
  // Use the Next.js API routes (your proxy)
  private static readonly BASE_URL = '/api/auth';

  static async login(credentials: LoginRequest): Promise<ApiResponse> {
    try {
      console.log('Making login request to:', `${this.BASE_URL}/login`);
      
      const response = await fetch(`${this.BASE_URL}/login`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(credentials),
      });

      const data = await response.json();
      console.log('Login response:', data);

      return {
        status: response.status,
        message: data.message || 'Login request completed',
        data: data.data || data, // Handle both response formats
      };
    } catch (error) {
      console.error('Login API error:', error);
      return {
        status: 500,
        message: error instanceof Error ? error.message : 'Network error occurred during login',
      };
    }
  }

  static async signup(userData: SignupRequest): Promise<ApiResponse> {
    try {
      // Remove confirmPassword before sending to backend
      const { confirmPassword, ...dataToSend } = userData;

      console.log('Making signup request to:', `${this.BASE_URL}/signup`);

      const response = await fetch(`${this.BASE_URL}/signup`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(dataToSend),
      });

      const data = await response.json();
      console.log('Signup response:', data);

      return {
        status: response.status,
        message: data.message || 'Signup request completed',
        data: data.data || data, // Handle both response formats
      };
    } catch (error) {
      console.error('Signup API error:', error);
      return {
        status: 500,
        message: error instanceof Error ? error.message : 'Network error occurred during signup',
      };
    }
  }

  // Optional: Add token validation method
  static async validateToken(token: string): Promise<boolean> {
    try {
      const response = await fetch(`${this.BASE_URL}/validate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
      });

      return response.ok;
    } catch {
      return false;
    }
  }
}