// @/services/api/student-advisor.ts

export interface AdvisorQuestionRequest {
  category: string;
  department?: string;
  faculty?: string;
  program?: string;
  degree?: string;
  major?: string;
  subject: string;
  question: string;
  student: string;
}

export interface AdvisorQuestionResponse {
  success: boolean;
  message: string;
  data?: any;
  status: number;
}

export class StudentAdvisorAPI {
  private static readonly BASE_URL = 'https://myadvisor.cs.uct.ac.za/backend/ask_question/';

  static async submitQuestion(
    questionData: AdvisorQuestionRequest
  ): Promise<AdvisorQuestionResponse> {
    try {
      console.log('Submitting question data:', questionData);

      const response = await fetch(this.BASE_URL, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(questionData),
      });

      console.log('Backend response status:', response.status);

      // Get the response text first to check what we're receiving
      const responseText = await response.text();
      console.log('Backend response text:', responseText);

      // Try to parse as JSON
      let responseData;
      try {
        responseData = responseText ? JSON.parse(responseText) : {};
      } catch (parseError) {
        console.error('Failed to parse response as JSON:', parseError);
        console.error('Response text was:', responseText);
        
        return {
          success: false,
          message: `Invalid JSON response: ${responseText.substring(0, 100)}...`,
          status: response.status,
        };
      }

      if (!response.ok) {
        console.error('Backend API error:', responseText);
        return {
          success: false,
          message: responseData?.message || `HTTP ${response.status}: ${responseData?.details || 'Unknown error'}`,
          status: response.status,
          data: responseData,
        };
      }

      return {
        success: true,
        message: 'Question submitted successfully',
        status: response.status,
        data: responseData,
      };

    } catch (error) {
      console.error('Network/API error:', error);
      
      return {
        success: false,
        message: error instanceof Error ? error.message : 'Network error occurred',
        status: 0,
      };
    }
  }
}