"use server";

import { 
  type Department as ImportedDepartment, 
  type Course as DeptCourse, 
  type DepartmentWithCourses 
} from "@/services/api/courses";
import { 
  type Program, 
  type Course as FacultyCourse, 
  type Degree, 
  type Major 
} from "@/services/api/faculties";

// Use the imported Department type instead of defining a new one
export type Department = ImportedDepartment;

// Faculty-related server actions
export async function getFacultiesAction(): Promise<any[]> {
  try {
    const res = await fetch(`https://myadvisor.cs.uct.ac.za/backend/faculties/`, {
      method: "GET",
    });

    if (!res.ok) {
      throw new Error(`Failed to fetch faculties: ${res.statusText}`);
    }

    return res.json();
  } catch (error) {
    console.error('Error fetching faculties:', error);
    throw error;
  }
}

export async function getProgramsByFacultyAction(facultyCode: string): Promise<Program[]> {
  try {
    const res = await fetch(
      `https://myadvisor.cs.uct.ac.za/backend/faculties/${facultyCode}/programs/`,
      { method: "GET" }
    );

    if (!res.ok) {
      throw new Error(`Failed to fetch programs for faculty ${facultyCode}: ${res.statusText}`);
    }

    return res.json();
  } catch (error) {
    console.error('Error fetching programs:', error);
    throw error;
  }
}

export async function getDegreesByProgramAction(
  facultyCode: string,
  programCode: string
): Promise<Degree[]> {
  const res = await fetch(
    `https://myadvisor.cs.uct.ac.za/backend/faculties/${facultyCode}/programs/${programCode}/degrees/`,
    { method: "GET" }
  );

  if (!res.ok) {
    throw new Error(
      `Failed to fetch degrees for program ${programCode} in faculty ${facultyCode}: ${res.statusText}`
    );
  }

  return res.json();
}

export async function getMajorsByProgramAction(
  facultyCode: string,
  programCode: string
): Promise<Major[]> {
  const res = await fetch(
    `https://myadvisor.cs.uct.ac.za/backend/faculties/${facultyCode}/programs/${programCode}/majors/`,
    { method: "GET" }
  );

  if (!res.ok) {
    throw new Error(
      `Failed to fetch majors for program ${programCode} in faculty ${facultyCode}: ${res.statusText}`
    );
  }

  return res.json();
}

// Department-related server actions
export async function getDepartmentsAction(): Promise<Department[]> {
  try {
    const res = await fetch(`https://myadvisor.cs.uct.ac.za/backend/departments/`, {
      method: "GET",
      cache: 'no-store',
    });

    if (!res.ok) {
      throw new Error(`Failed to fetch departments: ${res.statusText}`);
    }

    return res.json();
  } catch (error) {
    console.error('Error fetching departments:', error);
    throw error;
  }
}

export async function getCoursesByDepartmentAction(department_code: string): Promise<DeptCourse[]> {
  try {
    const res = await fetch(`https://myadvisor.cs.uct.ac.za/backend/departments/${department_code}/courses`, {
      method: "GET",
      cache: 'no-store',
    });

    if (!res.ok) {
      throw new Error(`Failed to fetch department courses: ${res.statusText}`);
    }

    return res.json();
  } catch (error) {
    console.error('Error fetching department courses:', error);
    throw error;
  }
}

export async function getDeptByCoursesAction(department_code: string): Promise<DepartmentWithCourses> {
  try {
    // Get all departments to find the specific one
    const departments = await getDepartmentsAction();
    const department = departments.find(dept => dept.department_code === department_code);
    
    if (!department) {
      throw new Error(`Department ${department_code} not found`);
    }

    // Get courses for this department
    const courses = await getCoursesByDepartmentAction(department_code);

    return {
      department,
      courses
    };
  } catch (error) {
    console.error('Error fetching department with courses:', error);
    throw error;
  }
}