"use client";

import { useState } from "react";
import Image from "next/image";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";

interface Advisor {
  username: string;
  first_name: string;
  last_name: string | null;
  title: string | null;
  email: string | null;
  phone_number: string | null;
  office_number: string | null;
  building: string | null;
  notes: string;
  faculties: Array<{ faculty_name: string }>;
  departments: Array<{ department_name: string }>;
  programs: any[];
  degrees: any[];
  majors: any[];
}

interface AdvisorsClientProps {
  advisors: Advisor[];
}

export function AdvisorsClient({ advisors }: AdvisorsClientProps) {
  // Helper function to get display name
  const getDisplayName = (advisor: Advisor) => {
    const fullName = `${advisor.first_name}${
      advisor.last_name ? ` ${advisor.last_name}` : ""
    }`;
    return advisor.title ? `${advisor.title} ${fullName}` : fullName;
  };

  // Helper function to get department or faculty
  const getDepartmentOrFaculty = (advisor: Advisor) => {
    if (advisor.departments.length > 0) {
      return advisor.departments[0].department_name;
    }
    if (advisor.faculties.length > 0) {
      return advisor.faculties[0].faculty_name;
    }
    return null;
  };

  return (
    <div className="w-full max-w-6xl mx-auto mt-16 mb-16 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8 z-10 px-8">
      {advisors.map((advisor, idx) => (
        <Card
          key={idx}
          className="flex flex-col items-center p-4 transition-transform duration-200 hover:scale-105 border-2 border-[#222222]/40 hover:shadow-medium"
        >
          <CardHeader className="flex flex-col items-center">
            <div className="w-32 h-32 rounded-full overflow-hidden mb-2 bg-gray-50 flex items-center justify-center border drop-shadow-sm">
              <Image
                src="/images/no_avatar.png"
                alt="Advisor Image"
                width={250}
                height={250}
                className="object-cover w-full h-full"
              />
            </div>
          </CardHeader>
          <CardContent className="flex flex-col items-center gap-1 text-center">
            <div className="outfit-500">{getDisplayName(advisor)}</div>
            {getDepartmentOrFaculty(advisor) && (
              <div className="outfit-300">
                {getDepartmentOrFaculty(advisor)}
              </div>
            )}
            {advisor.email && (
              <div className="text-blue-400 outfit-300">{advisor.email}</div>
            )}
          </CardContent>
        </Card>
      ))}
    </div>
  );
}
