import React, { useState, useEffect } from "react";
import { Card } from "@/components/ui/card";
import { Component, Check, X, AlertCircle } from "lucide-react";
import type { CourseSelectionTableProps, CourseGroup } from "../types";

export function CourseSelectionTable({
  courseSections,
  selectedCourses,
  selectedGroups,
  onCourseToggle,
  onGroupToggle,
}: CourseSelectionTableProps) {
  // Wrapper for course toggle
  const handleCourseToggle = (
    courseCode: string,
    sectionIndex: number,
    groupIndex?: number,
    parentGroupIndex?: number
  ) => {
    onCourseToggle(courseCode, sectionIndex, groupIndex, parentGroupIndex);
  };

  // Wrapper for group toggle
  const handleGroupToggle = (
    sectionIndex: number,
    groupIndex: number,
    subGroupIndex?: number
  ) => {
    onGroupToggle(sectionIndex, groupIndex, subGroupIndex);
  };

  // Add console logging to debug
  console.log("CourseSelectionTable - courseSections:", courseSections);
  console.log("CourseSelectionTable - selectedCourses:", selectedCourses);
  console.log("CourseSelectionTable - selectedGroups:", selectedGroups);

  // Helper function to generate unique group key
  const getGroupKey = (
    sectionIndex: number,
    groupIndex: number,
    subGroupIndex?: number
  ) => {
    return subGroupIndex !== undefined
      ? `${sectionIndex}-${groupIndex}-${subGroupIndex}`
      : `${sectionIndex}-${groupIndex}`;
  };

  // Helper function to check if a parent group's requirements are met
  const isParentGroupComplete = (
    sectionIndex: number,
    groupIndex: number
  ): boolean => {
    if (!courseSections || !courseSections[sectionIndex]) return false;

    const section = courseSections[sectionIndex];
    if (!section || !section.groups || !section.groups[groupIndex])
      return false;

    const parentGroup = section.groups[groupIndex];
    if (!parentGroup || !parentGroup.groups) return false;

    // Count completed nested groups
    let completedNestedGroups = 0;
    parentGroup.groups.forEach((_, subGroupIndex) => {
      if (isNestedGroupComplete(sectionIndex, groupIndex, subGroupIndex)) {
        completedNestedGroups++;
      }
    });

    return completedNestedGroups >= (parentGroup.required_count || 0);
  };

  // Helper function to check if a nested group is complete
  const isNestedGroupComplete = (
    sectionIndex: number,
    groupIndex: number,
    subGroupIndex: number
  ): boolean => {
    if (!courseSections || !courseSections[sectionIndex]) return false;

    const section = courseSections[sectionIndex];
    if (!section || !section.groups || !section.groups[groupIndex])
      return false;

    const parentGroup = section.groups[groupIndex];
    if (
      !parentGroup ||
      !parentGroup.groups ||
      !parentGroup.groups[subGroupIndex]
    )
      return false;

    const nestedGroup = parentGroup.groups[subGroupIndex];
    if (!nestedGroup) return false;

    if (nestedGroup.required_count === 100) {
      // For required groups, all courses must be selected
      return areAllRequiredCoursesSelected(nestedGroup);
    } else {
      // For regular nested groups, check if required count is met
      const groupCourses = getAllCoursesFromGroup(nestedGroup);
      const selectedGroupCourses = selectedCourses.filter((c) =>
        groupCourses.some((gc) => gc.code === c.code)
      );
      return selectedGroupCourses.length >= (nestedGroup.required_count || 0);
    }
  };

  // Helper function to check if a nested group should be disabled
  const isNestedGroupDisabled = (
    sectionIndex: number,
    groupIndex: number,
    subGroupIndex: number
  ): boolean => {
    const isCurrentGroupComplete = isNestedGroupComplete(
      sectionIndex,
      groupIndex,
      subGroupIndex
    );

    // If current group is complete, it should never be disabled
    if (isCurrentGroupComplete) return false;

    // If parent group is complete, disable this group
    return isParentGroupComplete(sectionIndex, groupIndex);
  };

  // Helper function to get all courses from a group
  const getAllCoursesFromGroup = (group: CourseGroup): any[] => {
    const courses: any[] = [];

    if (group.courses) {
      courses.push(...group.courses);
    }

    if (group.groups) {
      group.groups.forEach((subGroup) => {
        courses.push(...getAllCoursesFromGroup(subGroup));
      });
    }

    return courses;
  };

  // Helper function to check if all courses in a required group are selected
  const areAllRequiredCoursesSelected = (group: CourseGroup): boolean => {
    const allCourses = getAllCoursesFromGroup(group);
    return allCourses.every((course) =>
      selectedCourses.some((c) => c.code === course.code)
    );
  };

  // Helper function to check if a group is complete
  const isGroupComplete = (
    group: CourseGroup,
    sectionIndex: number,
    groupIndex?: number,
    parentGroupIndex?: number
  ): boolean => {
    // If this is a leaf group (no nested groups), check based on courses
    if (group.required_count === 100) {
      // For required groups, all courses must be selected
      return areAllRequiredCoursesSelected(group);
    } else {
      // For regular groups, check if required count is met
      const groupCourses = getAllCoursesFromGroup(group);
      const selectedGroupCourses = selectedCourses.filter((c) =>
        groupCourses.some((gc) => gc.code === c.code)
      );
      return selectedGroupCourses.length >= group.required_count;
    }
  };

  // Helper function to render individual courses
  const renderCourse = (
    course: any,
    sectionIndex: number,
    groupIndex?: number,
    parentGroupIndex?: number,
    isDisabled: boolean = false
  ) => {
    // Check if course is in a required group (required_count === 100)
    const isInRequiredGroup = () => {
      const section = courseSections[sectionIndex];

      // Check immediate parent group
      const immediateGroup =
        parentGroupIndex !== undefined
          ? section.groups?.[parentGroupIndex]
          : groupIndex !== undefined
          ? section.groups?.[groupIndex]
          : null;

      // Check if immediate group or any parent group has required_count === 100
      if (immediateGroup?.required_count === 100) {
        return true;
      }

      // If this is a nested course, check the parent group too
      if (parentGroupIndex !== undefined && groupIndex !== undefined) {
        const parentGroup = section.groups?.[groupIndex];
        if (parentGroup?.required_count === 100) {
          return true;
        }
      }

      return false;
    };

    const isSelected = selectedCourses.some((c) => c.code === course.code);
    const isInRequired = isInRequiredGroup();
    const uniqueKey = `${sectionIndex}-${groupIndex || "direct"}-${
      parentGroupIndex || "main"
    }-${course.code || Math.random()}`;

    return (
      <div
        key={uniqueKey}
        className={`bg-white rounded-lg border border-gray-200 shadow-sm transition-all 
        ${isInRequired ? "cursor-default" : "cursor-pointer"} 
        ${
          isSelected
            ? "border-blue-500 bg-blue-50 shadow-md"
            : isInRequired
            ? "border-gray-200"
            : "hover:border-gray-300"
        }
        ${isDisabled ? "opacity-50 cursor-not-allowed bg-gray-50" : ""}`}
        onClick={() => {
          // Only allow course selection if not in a required group and not disabled
          if (!isInRequired && !isDisabled && course.code) {
            handleCourseToggle(
              course.code,
              sectionIndex,
              groupIndex,
              parentGroupIndex
            );
          }
        }}
      >
        <div className="p-3">
          <div className="flex items-start justify-between">
            <div className="flex items-start space-x-3">
              {/* Only show selection indicator if not in a required group */}
              {!isInRequired && (
                <div
                  className={`flex-shrink-0 w-6 h-6 rounded-full border-2 flex items-center justify-center mt-0.5 
                ${
                  isSelected
                    ? "bg-blue-500 border-blue-500"
                    : isDisabled
                    ? "border-gray-200 bg-gray-100"
                    : "border-gray-300 bg-white"
                }`}
                >
                  {isSelected && <Check className="w-4 h-4 text-white" />}
                </div>
              )}

              {/* Show automatic selection indicator for required groups */}
              {isInRequired && (
                <div className="flex-shrink-0 w-6 h-6 rounded-full bg-green-500 border-2 border-green-500 flex items-center justify-center mt-0.5">
                  <Check className="w-4 h-4 text-white" />
                </div>
              )}

              <div className="flex-1 min-w-0">
                <h4
                  className={`text-sm font-medium mb-2 ${
                    isDisabled
                      ? "text-gray-500"
                      : isInRequired
                      ? "text-green-900"
                      : "text-gray-900"
                  }`}
                >
                  {course.name || "Unnamed Course"}
                  {isInRequired && (
                    <span className="ml-2 text-xs px-2 py-1 rounded-full bg-green-200 text-green-700">
                      Required
                    </span>
                  )}
                </h4>
                <div className="flex items-center flex-wrap gap-x-4 gap-y-1 text-xs">
                  <div className="flex items-center space-x-1">
                    <span className="text-gray-500">Code:</span>
                    <span
                      className={`font-medium ${
                        isDisabled ? "text-gray-500" : "text-gray-900"
                      }`}
                    >
                      {course.code || "N/A"}
                    </span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <span className="text-gray-500">Credits:</span>
                    <span
                      className={`font-medium ${
                        isDisabled ? "text-gray-500" : "text-gray-900"
                      }`}
                    >
                      {course.credits || "N/A"}
                    </span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <span className="text-gray-500">Level:</span>
                    <span
                      className={`font-medium ${
                        isDisabled ? "text-gray-500" : "text-gray-900"
                      }`}
                    >
                      {course.level || "N/A"}
                    </span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <span className="text-gray-500">Semester:</span>
                    <span
                      className={`font-medium ${
                        isDisabled ? "text-gray-500" : "text-gray-900"
                      }`}
                    >
                      {course.semester || "N/A"}
                    </span>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  };

  // Helper function to render course groups
  const renderCourseGroup = (
    item: any,
    depth = 0,
    sectionIndex: number,
    groupIndex?: number,
    parentGroupIndex?: number
  ): React.ReactNode => {
    const indentClass = depth > 0 ? `ml-${depth * 4}` : "";

    // Regular course
    if ("code" in item) {
      if (!courseSections || !courseSections[sectionIndex]) return null;

      const section = courseSections[sectionIndex];
      if (!section) return null;

      const group =
        parentGroupIndex !== undefined &&
        section.groups &&
        section.groups[parentGroupIndex]
          ? section.groups[parentGroupIndex]
          : groupIndex !== undefined &&
            section.groups &&
            section.groups[groupIndex]
          ? section.groups[groupIndex]
          : null;

      // For required groups (100%), don't render individual course selection
      // The courses will be managed entirely by the group header
      if (group && group.required_count === 100) {
        return null;
      }

      // Check if this is a nested course and if its parent group is disabled
      const isDisabled =
        parentGroupIndex !== undefined && groupIndex !== undefined
          ? isNestedGroupDisabled(sectionIndex, parentGroupIndex, groupIndex)
          : false;

      return renderCourse(
        item,
        sectionIndex,
        groupIndex,
        parentGroupIndex,
        isDisabled
      );
    }

    // Group with required_count
    if ("required_count" in item) {
      const group = item as CourseGroup;
      const groupCourses = getAllCoursesFromGroup(group);
      const isAllRequired = group.required_count === 100;

      // Determine if this is a nested group (has a parent group)
      const isNestedGroup = parentGroupIndex !== undefined;

      // For nested groups, check if they should be disabled
      const isGroupDisabled =
        isNestedGroup &&
        parentGroupIndex !== undefined &&
        groupIndex !== undefined
          ? isNestedGroupDisabled(sectionIndex, parentGroupIndex, groupIndex)
          : false;

      // Check if group is complete based on selected courses
      const selectedGroupCourses = selectedCourses.filter((c) =>
        groupCourses.some((gc) => gc.code === c.code)
      );

      // Check if this is a parent group (has nested groups)
      const isParentGroup = group.groups && group.groups.length > 0;

      // Only show completion for leaf groups (with courses), not parent groups
      const shouldShowComplete =
        !isParentGroup &&
        (isAllRequired
          ? areAllRequiredCoursesSelected(group)
          : selectedGroupCourses.length >= group.required_count);

      const uniqueKey = `group-${sectionIndex}-${groupIndex ?? ""}-${
        parentGroupIndex ?? ""
      }`;

      return (
        <div key={uniqueKey} className={`${indentClass} space-y-3`}>
          <div
            className={`bg-blue-50 rounded-lg border-2 border-blue-200 shadow-sm ${
              isGroupDisabled ? "opacity-50" : ""
            }`}
          >
            <div
              className={`bg-blue-100 rounded-t-lg px-4 py-2 border-b border-blue-200 cursor-default`}
            >
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-2">
                  <Component className="h-4 w-4 text-blue-600" />
                  <h4 className="text-sm font-semibold text-blue-900">
                    {isAllRequired
                      ? "All courses required in this group"
                      : isParentGroup
                      ? `Choose ${group.required_count} from the groups below`
                      : `Select ${group.required_count} item${
                          group.required_count > 1 ? "s" : ""
                        }`}
                  </h4>
                  {/* Only show completion badge for leaf groups, not parent groups */}
                  {shouldShowComplete && (
                    <span className="ml-2 text-xs px-2 py-1 rounded-full bg-green-200 text-green-700">
                      Complete
                    </span>
                  )}
                </div>
              </div>
            </div>

            <div className="p-3 space-y-3">
              {/* For required groups (100%), show courses as selectable items */}
              {isAllRequired ? (
                <div className="space-y-2">
                  {group.courses?.map((course, idx) => (
                    <div
                      key={`required-course-${course.code || idx}-${uniqueKey}`}
                    >
                      {renderCourse(
                        course,
                        sectionIndex,
                        groupIndex,
                        parentGroupIndex,
                        isGroupDisabled
                      )}
                    </div>
                  ))}
                </div>
              ) : (
                // For regular groups, render courses normally
                <>
                  {group.courses?.map((course, idx) => (
                    <div key={`course-${course.code || idx}-${uniqueKey}`}>
                      {renderCourseGroup(
                        course,
                        depth + 1,
                        sectionIndex,
                        groupIndex,
                        parentGroupIndex
                      )}
                    </div>
                  ))}
                </>
              )}

              {/* Render nested groups */}
              {group.groups?.map((subGroup, idx) => (
                <div key={`subgroup-${idx}-${uniqueKey}`}>
                  {renderCourseGroup(
                    subGroup,
                    depth + 1,
                    sectionIndex,
                    groupIndex, // Current group becomes parent
                    idx // Subgroup index
                  )}
                </div>
              ))}
            </div>
          </div>
        </div>
      );
    }

    return null;
  };

  // Helper function to handle required group toggle (100% groups)
  const handleRequiredGroupToggle = async (
    group: CourseGroup,
    sectionIndex: number,
    groupIndex?: number,
    parentGroupIndex?: number
  ) => {
    const isAllRequired = group.required_count === 100;

    if (isAllRequired) {
      // Handle required group: select/deselect ALL courses at once
      const allCourses = getAllCoursesFromGroup(group);
      const areAllSelected = areAllRequiredCoursesSelected(group);

      if (areAllSelected) {
        // If all are selected, deselect all courses
        allCourses.forEach((course) => {
          const isSelected = selectedCourses.some(
            (c) => c.code === course.code
          );
          if (isSelected) {
            onCourseToggle(
              course.code,
              sectionIndex,
              groupIndex,
              parentGroupIndex
            );
          }
        });
      } else {
        // If not all are selected, select ALL courses
        allCourses.forEach((course) => {
          const isSelected = selectedCourses.some(
            (c) => c.code === course.code
          );
          if (!isSelected) {
            onCourseToggle(
              course.code,
              sectionIndex,
              groupIndex,
              parentGroupIndex
            );
          }
        });
      }
    }
  };

  // Helper function to check if a section's requirements are met
  const isSectionRequirementMet = (sectionIndex: number): boolean => {
    if (!courseSections || !courseSections[sectionIndex]) return false;

    const section = courseSections[sectionIndex];
    if (!section) return false;

    if (section.groups) {
      // Count completed groups
      let completedGroups = 0;

      section.groups.forEach((group, groupIndex) => {
        if (group) {
          const isComplete = isGroupComplete(group, sectionIndex, groupIndex);
          if (isComplete) {
            completedGroups++;
          }
        }
      });

      return completedGroups >= (section.required_count || 0);
    } else {
      const selectedCoursesCount = selectedCourses.filter((c) =>
        section.courses?.some((sc) => sc && sc.code === c.code)
      ).length;
      return selectedCoursesCount >= (section.required_count || 0);
    }
  };

  return (
    <div className="space-y-6">
      {/* Early return if courseSections is not available */}
      {!courseSections || courseSections.length === 0 ? (
        <div className="p-4 text-center text-gray-500">
          Loading course data...
        </div>
      ) : (
        <>
          {/* Single container for all sections */}
          <Card className="w-full border-2 rounded-xl overflow-hidden drop-shadow-sm border-[#222222]/40">
            {/* Combined sections content */}
            <div className="p-4 space-y-6">
              {courseSections.map((section, sectionIndex) => {
                if (!section) return null;

                console.log("Rendering section:", sectionIndex, section);
                const sectionMet = isSectionRequirementMet(sectionIndex);

                return (
                  <div key={sectionIndex} className="space-y-4">
                    {/* Section Content */}
                    <div>
                      {section.courses ? (
                        // Direct course selection (no groups)
                        <div className="space-y-4">
                          {section.courses.map((course, courseIndex) => {
                            if (!course) return null;

                            console.log(
                              "Direct course rendering:",
                              course,
                              "courseIndex:",
                              courseIndex
                            );
                            return renderCourse(
                              course,
                              sectionIndex,
                              undefined,
                              undefined,
                              false
                            );
                          })}
                        </div>
                      ) : (
                        // Group-based selection
                        <div className="space-y-4">
                          {section.groups?.map((group, groupIndex) => {
                            if (!group) return null;

                            console.log(
                              "Group rendering:",
                              group,
                              "groupIndex:",
                              groupIndex
                            );
                            return renderCourseGroup(
                              group,
                              0,
                              sectionIndex,
                              groupIndex
                            );
                          })}
                        </div>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>
          </Card>

          {/* Summary card */}
          <Card className="mt-6 p-4 bg-gradient-to-r from-green-50 to-emerald-50 border-green-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-green-800">
                  Selected courses: {selectedCourses?.length || 0}
                </p>
                <p className="text-xs text-green-700">
                  {selectedCourses?.reduce(
                    (sum, c) => sum + (c.credits || 0),
                    0
                  ) || 0}{" "}
                  total credits
                </p>
              </div>

              {/* Credit distribution */}
              <div className="flex items-center space-x-2">
                <span className="text-xs text-green-700">Credits:</span>
                <div className="w-24 bg-green-200 rounded-full h-2">
                  <div
                    className="bg-green-600 h-2 rounded-full transition-all"
                    style={{
                      width: `${Math.min(
                        100,
                        ((selectedCourses?.reduce(
                          (sum, c) => sum + (c.credits || 0),
                          0
                        ) || 0) /
                          120) *
                          100
                      )}%`,
                    }}
                  />
                </div>
                <span className="text-xs text-green-700">
                  {selectedCourses?.reduce(
                    (sum, c) => sum + (c.credits || 0),
                    0
                  ) || 0}
                  /120
                </span>
              </div>
            </div>
          </Card>
        </>
      )}
    </div>
  );
}
