import React, { useState, useMemo } from "react";
import { CourseCard } from "./CourseCard";
import { Card } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Loader2, CheckCircle, XCircle } from "lucide-react";
import type { Course } from "../types";
import { getRowStyling } from "../types";

interface CourseRoadmapProps {
  selectedYear: {
    year: number;
    title: string;
  };
  degreeCode: string | null;
  semOneCourses: Course[];
  semTwoCourses: Course[];
  semThreeCourses: Course[];
  onStatusChange: (courseCode: string, status: number) => void;
}

interface SubmissionResult {
  success: boolean;
  evaluation?: string;
  error?: string;
}

export function CourseRoadmap({
  selectedYear,
  degreeCode,
  semOneCourses,
  semTwoCourses,
  semThreeCourses,
  onStatusChange,
}: CourseRoadmapProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [submissionResult, setSubmissionResult] =
    useState<SubmissionResult | null>(null);

  // Get all courses from all semesters
  const allCourses = [...semOneCourses, ...semTwoCourses, ...semThreeCourses];

  // Check if all courses have been marked (status 1 = pass, status 2 = fail)
  const allCoursesMarked = useMemo(() => {
    return allCourses.every(
      (course) => course.status === 1 || course.status === 2
    );
  }, [allCourses]);

  const updateCourseStatus = (courseCode: string, status: number) => {
    if (onStatusChange) {
      onStatusChange(courseCode, status);
    }
  };

  const handleSubmit = async () => {
    if (!degreeCode) {
      setSubmissionResult({
        success: false,
        error: "Missing degree code. Please go back and reselect your program.",
      });
      setDialogOpen(true);
      return;
    }

    if (!allCoursesMarked) {
      setSubmissionResult({
        success: false,
        error: "Please mark all courses as pass or fail before submitting.",
      });
      setDialogOpen(true);
      return;
    }

    // Open dialog and start loading
    setDialogOpen(true);
    setIsSubmitting(true);
    setSubmissionResult(null);

    try {
      // Create the submission data
      const submissionData = {
        degree_code: degreeCode,
        year: selectedYear.year,
        completed_courses: allCourses.map((course) => ({
          code: course.code,
          passed: course.status === 1, // status 1 = pass, status 2 = fail
        })),
      };

      console.log("Submitting roadmap data:", submissionData);

      const response = await fetch("/api/roadmap", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(submissionData),
      });

      const result = await response.json();
      console.log("Roadmap response:", result);

      if (result.success) {
        setSubmissionResult({
          success: true,
          evaluation: result.evaluation || "Roadmap submitted successfully!",
        });
      } else {
        setSubmissionResult({
          success: false,
          error: result.error || "Failed to submit roadmap",
        });
      }
    } catch (error) {
      console.error("Error submitting roadmap:", error);
      setSubmissionResult({
        success: false,
        error:
          error instanceof Error ? error.message : "Failed to submit roadmap",
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleDialogClose = () => {
    setDialogOpen(false);
    setSubmissionResult(null);
  };

  return (
    <div className="container mx-auto px-4 md:w-3xl">
      <div className="w-full md:max-w-4xl mx-auto border-b border-[#000000] mb-8"></div>

      {/* Desktop View - Timeline Layout */}
      <div className="hidden md:block relative">
        {/* Timeline */}
        <div className="absolute left-1/2 top-0 bottom-0 w-0.5 bg-[#e2e2e2] -translate-x-1/2"></div>

        {/* First Semester */}
        {semOneCourses.length > 0 && (
          <div className="flex items-center mb-6 relative z-10">
            <div className="w-1/2 flex justify-end translate-x-15">
              <div className="bg-uct-green border-2 outfit-300 border-[#222222]/10 text-white px-6 py-2 rounded-full">
                {selectedYear.title}, 1st semester
              </div>
            </div>
            <div className="absolute left-1/2 -translate-x-1/2 w-5 h-5 bg-uct-blue rounded-full border-4 border-white"></div>
            <div className="w-full translate-x-50 py-0">
              <CourseCard
                courses={semOneCourses}
                allCourses={allCourses}
                onStatusChange={onStatusChange}
              />
            </div>
          </div>
        )}

        {/* Second Semester */}
        {semTwoCourses.length > 0 && (
          <div className="flex items-center mb-6 relative z-10">
            <div className="w-full justify-end -translate-x-50 py-0">
              <CourseCard
                courses={semTwoCourses}
                allCourses={allCourses}
                onStatusChange={onStatusChange}
              />
            </div>
            <div className="absolute left-1/2 -translate-x-1/2 w-5 h-5 bg-uct-blue rounded-full border-4 border-white"></div>
            <div className="w-1/2 flex justify-start -translate-x-20 pl-8">
              <div className="bg-uct-green border-2 outfit-300 border-[#222222]/10 text-white px-6 py-2 rounded-full">
                {selectedYear.title}, 2nd semester
              </div>
            </div>
          </div>
        )}

        {/* Third Semester (if exists) */}
        {semThreeCourses.length > 0 && (
          <div className="flex items-center mb-6 relative z-10">
            <div className="w-1/2 flex justify-end translate-x-15">
              <div className="bg-uct-green border-2 outfit-300 border-[#222222]/10 text-white px-6 py-2 rounded-full">
                {selectedYear.title}, 3rd semester
              </div>
            </div>
            <div className="absolute left-1/2 -translate-x-1/2 w-5 h-5 bg-uct-blue rounded-full border-4 border-white"></div>
            <div className="w-full translate-x-50 py-0">
              <CourseCard
                courses={semThreeCourses}
                allCourses={allCourses}
                onStatusChange={onStatusChange}
              />
            </div>
          </div>
        )}
      </div>

      {/* Mobile View - Single Card with All Courses */}
      <div className="md:hidden mb-8">
        <div className="text-center mb-6">
          <h2 className="text-lg font-semibold text-gray-800">
            {selectedYear.title} Courses
          </h2>
        </div>

        <Card className="w-full max-w-sm mx-auto border-2 border-[#222222]/40 rounded-xl overflow-hidden drop-shadow-sm">
          <div className="p-4 space-y-3">
            {allCourses.map((course, index) => (
              <div
                key={index}
                className={`border border-gray-200 rounded-lg p-3 ${getRowStyling(
                  course.status
                )}`}
              >
                <div className="space-y-3">
                  <div className="font-medium text-gray-900">{course.code}</div>

                  <div className="flex flex-wrap gap-x-4 gap-y-1 text-xs text-gray-600">
                    <div className="flex items-center space-x-1">
                      <span className="text-gray-500">Code:</span>
                      <span className="font-medium">{course.code}</span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <span className="text-gray-500">Credits:</span>
                      <span className="font-medium">{course.credits}</span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <span className="text-gray-500">Pass Req:</span>
                      <span className="font-medium">
                        {course.passRequirement}
                      </span>
                    </div>
                  </div>

                  <div className="flex gap-2 w-full">
                    <button
                      onClick={() => updateCourseStatus(course.code, 1)}
                      disabled={course.status === 3}
                      className={`
                        ${
                          course.status === 1 ? "bg-green-600" : "bg-[#1EA558]"
                        } 
                        ${
                          course.status === 3
                            ? "bg-gray-400 cursor-not-allowed"
                            : "hover:bg-green-600"
                        }
                        text-white px-3 py-2 rounded-md flex-1 text-sm font-medium transition-colors
                      `}
                    >
                      Pass
                    </button>
                    <button
                      onClick={() => updateCourseStatus(course.code, 2)}
                      disabled={course.status === 3}
                      className={`
                        ${course.status === 2 ? "bg-red-600" : "bg-[#EF0B0B]"} 
                        ${
                          course.status === 3
                            ? "bg-gray-400 cursor-not-allowed"
                            : "hover:bg-red-600"
                        }
                        text-white px-3 py-2 rounded-md flex-1 text-sm font-medium transition-colors
                      `}
                    >
                      Fail
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </Card>
      </div>

      <div className="w-full flex justify-center mt-20">
        <button
          onClick={handleSubmit}
          disabled={!allCoursesMarked || isSubmitting}
          className={`w-fit outfit-400 text-lg border-2 border-[#222222]/10 py-2 px-12 rounded-3xl drop-shadow-lg text-white transition-all duration-200 ${
            allCoursesMarked && !isSubmitting
              ? "bg-uct-green hover:bg-uct-green/90 cursor-pointer"
              : "bg-gray-400 cursor-not-allowed"
          }`}
        >
          {isSubmitting ? "Submitting..." : "Submit"}
        </button>
      </div>

      {/* Progress indicator */}
      {!allCoursesMarked && (
        <div className="w-full flex justify-center mt-4">
          <p className="text-sm text-gray-600">
            Please mark all courses as pass or fail to enable submission
          </p>
        </div>
      )}

      {/* Submission Dialog */}
      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="sm:max-w-[45rem]">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              {isSubmitting && <Loader2 className="h-5 w-5 animate-spin" />}
              {submissionResult?.success === true && (
                <CheckCircle className="h-5 w-5 text-green-600" />
              )}
              {submissionResult?.success === false && (
                <XCircle className="h-5 w-5 text-red-600" />
              )}

              {isSubmitting && "Processing Roadmap..."}
              {submissionResult?.success === true && "Roadmap Evaluation"}
              {submissionResult?.success === false && "Submission Error"}
            </DialogTitle>

            {isSubmitting && (
              <DialogDescription>
                Please wait while we process your roadmap submission...
              </DialogDescription>
            )}
          </DialogHeader>

          <div className="py-6">
            {isSubmitting && (
              <div className="flex flex-col items-center space-y-4">
                <Loader2 className="h-12 w-12 animate-spin text-uct-blue" />
                <p className="text-center text-gray-600">
                  Analyzing your course progress and generating
                  recommendations...
                </p>
              </div>
            )}

            {submissionResult?.success === true && (
              <div className="space-y-4">
                <div className="flex items-center justify-center w-16 h-16 mx-auto bg-green-100 rounded-full">
                  <CheckCircle className="h-8 w-8 text-green-600" />
                </div>
                <div className="text-center">
                  <p className="text-lg font-medium text-green-800 mb-4">
                    Roadmap Successfully Submitted!
                  </p>
                  <div className="bg-gray-50 p-4 max-h-[18rem] overflow-y-auto rounded-lg text-left">
                    <p className="text-gray-700 whitespace-pre-wrap">
                      {submissionResult.evaluation}
                    </p>
                  </div>
                </div>
              </div>
            )}

            {submissionResult?.success === false && (
              <div className="space-y-4">
                <div className="flex items-center justify-center w-16 h-16 mx-auto bg-red-100 rounded-full">
                  <XCircle className="h-8 w-8 text-red-600" />
                </div>
                <div className="text-center">
                  <p className="text-lg font-medium text-red-800 mb-4">
                    Submission Failed
                  </p>
                  <div className="bg-red-50 p-4 max-h-[35rem] overflow-y-auto rounded-lg text-left">
                    <p className="text-red-700">{submissionResult.error}</p>
                  </div>
                </div>
              </div>
            )}
          </div>

          {submissionResult && (
            <DialogFooter>
              <Button onClick={handleDialogClose} className="w-full">
                {submissionResult.success ? "Continue" : "Try Again"}
              </Button>
            </DialogFooter>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
