"use client";

import React, { useState, useEffect } from "react";
import ErrorPage from "@/components/ErrorPage";
import {
  Breadcrumb,
  BreadcrumbList,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbSeparator,
} from "@/components/ui/breadcrumb";
import {
  degreeToAcademicProgram,
  majorToAcademicProgram,
  type AcademicProgram,
  type Program,
} from "@/services/api/faculties";

// Import your components
import { LoadingSpinner } from "@/components/roadmap/ui/LoadingSpinner";
import { PageHeader } from "@/components/roadmap/ui/PageHeader";
import { FacultySelection } from "@/components/roadmap/selections/FacultySelection";
import { ProgramSelection } from "@/components/roadmap/selections/ProgramSelection";
import { AcademicProgramSelection } from "@/components/roadmap/selections/AcademicProgramSelection";
import { YearSelection } from "@/components/roadmap/selections/YearSelection";
import { CoursePicker } from "@/components/roadmap/courses/CoursePicker";
import { CourseRoadmap } from "@/components/roadmap/courses/CourseRoadmap";
import {
  Faculty,
  Course,
  CourseSection,
  convertApiCourseToLocal,
} from "@/components/roadmap/types";

interface RoadmapClientProps {
  initialFaculties: Faculty[];
}

export function RoadmapClient({ initialFaculties }: RoadmapClientProps) {
  // State management
  const [faculties] = useState<Faculty[]>(initialFaculties);
  const [programs, setPrograms] = useState<Program[]>([]);
  const [academicPrograms, setAcademicPrograms] = useState<AcademicProgram[]>(
    []
  );
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  // Selection states
  const [selectedFaculty, setSelectedFaculty] = useState<Faculty | null>(null);
  const [selectedProgram, setSelectedProgram] = useState<Program | null>(null);
  const [selectedAcademicProgram, setSelectedAcademicProgram] =
    useState<AcademicProgram | null>(null);
  const [selectedYear, setSelectedYear] = useState<{
    year: number;
    title: string;
  } | null>(null);
  const [selectedDegreeCode, setSelectedDegreeCode] = useState<string | null>(
    null
  );

  // Course management states
  const [courseSections, setCourseSections] = useState<CourseSection[]>([]);
  const [selectedCourses, setSelectedCourses] = useState<Course[]>([]);
  const [selectedGroups, setSelectedGroups] = useState<Set<string>>(new Set());
  const [showRoadmap, setShowRoadmap] = useState(false);

  // Roadmap semester states
  const [semOneCourses, setSemOneCourses] = useState<Course[]>([]);
  const [semTwoCourses, setSemTwoCourses] = useState<Course[]>([]);
  const [semThreeCourses, setSemThreeCourses] = useState<Course[]>([]);

  // Load programs when faculty is selected - UPDATED TO USE API ROUTES
  useEffect(() => {
    const loadPrograms = async () => {
      if (!selectedFaculty) return;

      try {
        setLoading(true);
        const response = await fetch(
          `/api/faculties/${selectedFaculty.faculty_code}/programs`
        );

        if (!response.ok) {
          throw new Error("Failed to fetch programs");
        }

        const programsData = await response.json();
        setPrograms(programsData);
      } catch (err) {
        setError("Failed to load programs");
        console.error(err);
      } finally {
        setLoading(false);
      }
    };

    loadPrograms();
  }, [selectedFaculty]);

  // Load degrees and majors when program is selected - UPDATED TO USE API ROUTES
  useEffect(() => {
    const loadAcademicPrograms = async () => {
      if (!selectedFaculty || !selectedProgram) return;

      try {
        setLoading(true);

        const degreesUrl = `/api/faculties/${selectedFaculty.faculty_code}/programs/${selectedProgram.program_code}/degrees`;
        const majorsUrl = `/api/faculties/${selectedFaculty.faculty_code}/programs/${selectedProgram.program_code}/majors`;

        console.log("Fetching degrees from:", degreesUrl);
        console.log("Fetching majors from:", majorsUrl);

        // Load both degrees and majors using API routes
        const [degreesResponse, majorsResponse] = await Promise.all([
          fetch(degreesUrl),
          fetch(majorsUrl),
        ]);

        console.log(
          "Degrees response status:",
          degreesResponse.status,
          degreesResponse.ok
        );
        console.log(
          "Majors response status:",
          majorsResponse.status,
          majorsResponse.ok
        );

        if (!degreesResponse.ok) {
          const degreesError = await degreesResponse.text();
          console.error("Degrees response error:", degreesError);
        }

        if (!majorsResponse.ok) {
          const majorsError = await majorsResponse.text();
          console.error("Majors response error:", majorsError);
        }

        if (!degreesResponse.ok || !majorsResponse.ok) {
          throw new Error(
            `Failed to fetch academic programs. Degrees: ${degreesResponse.status}, Majors: ${majorsResponse.status}`
          );
        }

        const [degreesData, majorsData] = await Promise.all([
          degreesResponse.json(),
          majorsResponse.json(),
        ]);

        console.log("Degrees data:", degreesData);
        console.log("Majors data:", majorsData);

        // Convert to unified format
        const allPrograms: AcademicProgram[] = [
          ...degreesData.map(degreeToAcademicProgram),
          ...majorsData.map(majorToAcademicProgram),
        ];

        setAcademicPrograms(allPrograms);
      } catch (err) {
        console.error("Full academic programs error:", err);
        setError("Failed to load academic programs");
      } finally {
        setLoading(false);
      }
    };

    loadAcademicPrograms();
  }, [selectedFaculty, selectedProgram]);

  // Generate year options based on selected academic program
  const getYearOptions = () => {
    if (!selectedAcademicProgram?.courses) return [];

    return selectedAcademicProgram.courses.map((yearCourse) => ({
      year: yearCourse.year,
      title: `${yearCourse.year}${
        yearCourse.year === 1
          ? "st"
          : yearCourse.year === 2
          ? "nd"
          : yearCourse.year === 3
          ? "rd"
          : "th"
      } Year`,
    }));
  };

  const handleStatusChange = (courseCode: string, status: number) => {
    // Update the course status in semester one
    const updatedSemOne = semOneCourses.map((course) =>
      course.code === courseCode ? { ...course, status: status } : course
    );
    setSemOneCourses(updatedSemOne);

    // Update the course status in semester two
    const updatedSemTwo = semTwoCourses.map((course) =>
      course.code === courseCode ? { ...course, status: status } : course
    );
    setSemTwoCourses(updatedSemTwo);

    // Update the course status in semester three
    const updatedSemThree = semThreeCourses.map((course) =>
      course.code === courseCode ? { ...course, status: status } : course
    );
    setSemThreeCourses(updatedSemThree);

    // Find the course that was updated across all semesters
    const updatedCourse = [
      ...semOneCourses,
      ...semTwoCourses,
      ...semThreeCourses,
    ].find((c) => c.code === courseCode);

    if (updatedCourse) {
      // Update postrequisites in semester two based on semester one changes
      if (semOneCourses.some((c) => c.code === courseCode)) {
        const updatedSemTwoWithPostreq = semTwoCourses.map((course) => {
          const isPostrequisite = updatedCourse.postrequisites.includes(
            course.code
          );
          if (isPostrequisite) {
            if (status === 2) {
              return { ...course, status: 3 };
            } else if (status === 1) {
              return {
                ...course,
                status: course.status === 3 ? 0 : course.status,
              };
            }
          }
          return course;
        });
        setSemTwoCourses(updatedSemTwoWithPostreq);
      }

      // Update postrequisites in semester three based on semester two changes
      if (semTwoCourses.some((c) => c.code === courseCode)) {
        const updatedSemThreeWithPostreq = semThreeCourses.map((course) => {
          const isPostrequisite = updatedCourse.postrequisites.includes(
            course.code
          );
          if (isPostrequisite) {
            if (status === 2) {
              return { ...course, status: 3 };
            } else if (status === 1) {
              return {
                ...course,
                status: course.status === 3 ? 0 : course.status,
              };
            }
          }
          return course;
        });
        setSemThreeCourses(updatedSemThreeWithPostreq);
      }
    }
  };

  // Navigation handlers
  const handleHomeClick = (e: React.MouseEvent) => {
    e.preventDefault();
    setSelectedFaculty(null);
    setSelectedProgram(null);
    setSelectedAcademicProgram(null);
    setSelectedYear(null);
    setSelectedDegreeCode(null);
    setSelectedCourses([]);
    setCourseSections([]);
    setSelectedGroups(new Set<string>());
    setShowRoadmap(false);
    setPrograms([]);
    setAcademicPrograms([]);
  };

  const handleFacultyClick = (e: React.MouseEvent) => {
    e.preventDefault();
    setSelectedProgram(null);
    setSelectedAcademicProgram(null);
    setSelectedYear(null);
    setSelectedDegreeCode(null);
    setSelectedCourses([]);
    setCourseSections([]);
    setSelectedGroups(new Set<string>());
    setShowRoadmap(false);
    setAcademicPrograms([]);
  };

  const handleProgramClick = (e: React.MouseEvent) => {
    e.preventDefault();
    setSelectedAcademicProgram(null);
    setSelectedYear(null);
    setSelectedDegreeCode(null);
    setSelectedCourses([]);
    setCourseSections([]);
    setSelectedGroups(new Set<string>());
    setShowRoadmap(false);
  };

  const handleAcademicProgramClick = (e: React.MouseEvent) => {
    e.preventDefault();
    setSelectedYear(null);
    setSelectedCourses([]);
    setCourseSections([]);
    setSelectedGroups(new Set<string>());
    setShowRoadmap(false);
  };

  const handleAcademicProgramSelect = (program: AcademicProgram) => {
    setSelectedAcademicProgram(program);
    // Store the degree code for later submission
    if ("degree_code" in program && program.degree_code) {
      setSelectedDegreeCode(program.degree_code);
    } else if ("major_code" in program && program.major_code) {
      setSelectedDegreeCode(program.major_code);
    } else {
      console.warn("No valid code found for academic program:", program);
      setSelectedDegreeCode(null);
    }
  };

  const handleYearSelect = (year: { year: number; title: string }) => {
    setSelectedYear(year);

    // Find the year data from the selected academic program
    const yearData = selectedAcademicProgram?.courses?.find(
      (y) => y.year === year.year
    );

    console.log("handleYearSelect - yearData:", yearData);

    if (yearData) {
      // Convert year data to course sections
      const sections: CourseSection[] = [];

      if (yearData.items) {
        console.log("Processing yearData.items:", yearData.items);

        // Separate courses from groups
        const courses: any[] = [];
        const groups: any[] = [];

        yearData.items.forEach((item, index) => {
          console.log(
            `Item ${index}:`,
            item,
            "has code:",
            "code" in item,
            "has required_count:",
            "required_count" in item
          );

          if ("code" in item) {
            // This is a course
            courses.push(item);
          } else if ("required_count" in item) {
            // This is a group
            groups.push(item);
          } else {
            console.warn("Unknown item type:", item);
          }
        });

        console.log("Separated courses:", courses);
        console.log("Separated groups:", groups);

        // Create section with proper structure
        if (courses.length > 0 && groups.length > 0) {
          // Mixed content - create separate sections or combine appropriately
          sections.push({
            required_count: courses.length, // Assume all direct courses are required
            courses: courses,
          });
          sections.push({
            required_count: yearData.required_count || 1,
            groups: groups,
          });
        } else if (courses.length > 0) {
          // Only courses
          sections.push({
            required_count: yearData.required_count || courses.length,
            courses: courses,
          });
        } else if (groups.length > 0) {
          // Only groups
          sections.push({
            required_count: yearData.required_count || 1,
            groups: groups,
          });
        }
      } else if (yearData.courses) {
        // Direct courses array
        console.log("Processing yearData.courses:", yearData.courses);
        sections.push({
          required_count: yearData.required_count || yearData.courses.length,
          courses: yearData.courses,
        });
      }

      // Handle groups if they exist separately
      if (yearData.groups) {
        console.log("Processing yearData.groups:", yearData.groups);
        sections.push({
          required_count: yearData.required_count || 1,
          groups: yearData.groups.map((group, index) => ({
            ...group,
            groupIndex: index,
          })),
        });
      }

      console.log("Final sections:", sections);
      setCourseSections(sections);
    } else {
      console.log("No yearData found");
      setCourseSections([]);
    }

    setSelectedCourses([]);
    setSelectedGroups(new Set<string>());
    setShowRoadmap(false);
  };

  const handleCourseToggle = (
    courseCode: string,
    sectionIndex: number,
    groupIndex?: number,
    subGroupIndex?: number
  ) => {
    const section = courseSections[sectionIndex];
    if (!section) return;

    let apiCourse: any;

    // Navigate through nested structure to find the course
    if (
      subGroupIndex !== undefined &&
      groupIndex !== undefined &&
      section.groups
    ) {
      const group = section.groups[groupIndex];
      if (group?.groups) {
        apiCourse = group.groups[subGroupIndex]?.courses?.find(
          (c) => c.code === courseCode
        );
      }
    } else if (groupIndex !== undefined && section.groups) {
      apiCourse = section.groups[groupIndex]?.courses?.find(
        (c) => c.code === courseCode
      );
    } else if (section.courses) {
      apiCourse = section.courses.find((c) => c.code === courseCode);
    }

    if (!apiCourse) return;

    const course = convertApiCourseToLocal(apiCourse);
    const isSelected = selectedCourses.some((c) => c.code === courseCode);

    if (isSelected) {
      setSelectedCourses(selectedCourses.filter((c) => c.code !== courseCode));
    } else {
      // Determine the maximum allowed courses for this selection context
      let maxAllowed = 1;
      let currentSelections: Course[] = [];

      if (
        subGroupIndex !== undefined &&
        groupIndex !== undefined &&
        section.groups
      ) {
        const subGroup = section.groups[groupIndex]?.groups?.[subGroupIndex];
        maxAllowed = subGroup?.required_count || 1;
        currentSelections = selectedCourses.filter((c) =>
          subGroup?.courses?.some((sc) => sc.code === c.code)
        );
      } else if (groupIndex !== undefined && section.groups) {
        const group = section.groups[groupIndex];
        maxAllowed = group?.required_count || 1;
        currentSelections = selectedCourses.filter((c) =>
          group?.courses?.some((sc) => sc.code === c.code)
        );
      } else {
        maxAllowed = section.required_count;
        currentSelections = selectedCourses.filter((c) =>
          section.courses?.some((sc) => sc.code === c.code)
        );
      }

      if (currentSelections.length < maxAllowed) {
        setSelectedCourses([...selectedCourses, course]);
      }
    }
  };

  const handleGroupToggle = (
    sectionIndex: number,
    groupIndex: number,
    subGroupIndex?: number
  ) => {
    const groupKey =
      subGroupIndex !== undefined
        ? `${sectionIndex}-${groupIndex}-${subGroupIndex}`
        : `${sectionIndex}-${groupIndex}`;

    const newSelectedGroups = new Set(selectedGroups);

    if (selectedGroups.has(groupKey)) {
      newSelectedGroups.delete(groupKey);

      // Remove all courses from this group and its subgroups
      const section = courseSections[sectionIndex];
      if (section.groups) {
        let groupToProcess;

        if (subGroupIndex !== undefined) {
          groupToProcess = section.groups[groupIndex]?.groups?.[subGroupIndex];
        } else {
          groupToProcess = section.groups[groupIndex];
        }

        if (groupToProcess) {
          // Remove courses from this group
          if (groupToProcess.courses) {
            setSelectedCourses(
              selectedCourses.filter(
                (c) => !groupToProcess.courses?.some((gc) => gc.code === c.code)
              )
            );
          }

          // Remove any selected subgroups
          if (groupToProcess.groups) {
            const subGroupKeys = Array.from(selectedGroups).filter((key) =>
              key.startsWith(`${sectionIndex}-${groupIndex}-`)
            );
            subGroupKeys.forEach((key) => newSelectedGroups.delete(key));
          }
        }
      }
    } else {
      // Check if we can add more groups at this level
      const section = courseSections[sectionIndex];
      let maxAllowed = section.required_count;
      let currentGroupCount = 0;

      if (subGroupIndex !== undefined) {
        // This is a subgroup selection
        const parentGroup = section.groups?.[groupIndex];
        maxAllowed = parentGroup?.required_count || 1;
        currentGroupCount = Array.from(selectedGroups).filter((key) =>
          key.startsWith(`${sectionIndex}-${groupIndex}-`)
        ).length;
      } else {
        // This is a top-level group selection
        currentGroupCount = Array.from(selectedGroups).filter(
          (key) =>
            key.startsWith(`${sectionIndex}-`) &&
            !key.includes("-", key.indexOf("-") + 1)
        ).length;
      }

      if (currentGroupCount < maxAllowed) {
        newSelectedGroups.add(groupKey);
      }
    }

    setSelectedGroups(newSelectedGroups);
  };

  const handleContinueToRoadmap = () => {
    // Distribute selected courses into semesters
    const sem1 = selectedCourses.slice(
      0,
      Math.ceil(selectedCourses.length / 2)
    );
    const sem2 = selectedCourses.slice(Math.ceil(selectedCourses.length / 2));
    setSemOneCourses(sem1);
    setSemTwoCourses(sem2);
    setSemThreeCourses([]);
    setShowRoadmap(true);
  };

  // Get page title based on current state
  const getPageTitle = () => {
    if (!selectedFaculty) return "Select a Faculty:";
    if (!selectedProgram) return "Select a Program:";
    if (!selectedAcademicProgram) return "Select a Degree/Major:";
    if (!selectedYear) return "Select a Year:";
    if (!showRoadmap) return "Pick Courses";
    return `${selectedYear.title} Roadmap`;
  };

  // Show error page if there's an error
  if (error) {
    return (
      <div className="w-full h-screen flex items-center justify-center">
        <ErrorPage />
      </div>
    );
  }

  return (
    <>
      <PageHeader />

      {/* Loading indicator */}
      {loading && <LoadingSpinner />}

      {/* Breadcrumb */}
      <div className="w-full max-w-6xl mt-8 mb-8 z-10 justify-center flex items-center px-8">
        <Breadcrumb>
          <BreadcrumbList>
            <BreadcrumbItem>
              <BreadcrumbLink
                href="#"
                onClick={handleHomeClick}
                aria-current={!selectedFaculty ? "page" : undefined}
              >
                Home
              </BreadcrumbLink>
            </BreadcrumbItem>
            {selectedFaculty && (
              <>
                <BreadcrumbSeparator />
                <BreadcrumbItem>
                  <BreadcrumbLink
                    href="#"
                    onClick={handleFacultyClick}
                    aria-current={
                      selectedFaculty && !selectedProgram ? "page" : undefined
                    }
                  >
                    {selectedFaculty.faculty_name}
                  </BreadcrumbLink>
                </BreadcrumbItem>
              </>
            )}
            {selectedProgram && (
              <>
                <BreadcrumbSeparator />
                <BreadcrumbItem>
                  <BreadcrumbLink
                    href="#"
                    onClick={handleProgramClick}
                    aria-current={
                      selectedProgram && !selectedAcademicProgram
                        ? "page"
                        : undefined
                    }
                  >
                    {selectedProgram.program_name}
                  </BreadcrumbLink>
                </BreadcrumbItem>
              </>
            )}
            {selectedAcademicProgram && (
              <>
                <BreadcrumbSeparator />
                <BreadcrumbItem>
                  <BreadcrumbLink
                    href="#"
                    onClick={handleAcademicProgramClick}
                    aria-current={
                      selectedAcademicProgram && !selectedYear
                        ? "page"
                        : undefined
                    }
                  >
                    {selectedAcademicProgram.degree_name ||
                      selectedAcademicProgram.major_name}
                  </BreadcrumbLink>
                </BreadcrumbItem>
              </>
            )}
            {selectedYear && (
              <>
                <BreadcrumbSeparator />
                <BreadcrumbItem>
                  <BreadcrumbLink
                    aria-current={
                      selectedYear && selectedCourses.length === 0
                        ? "page"
                        : undefined
                    }
                  >
                    {selectedYear.title}
                  </BreadcrumbLink>
                </BreadcrumbItem>
              </>
            )}
          </BreadcrumbList>
        </Breadcrumb>
      </div>

      {/* Main content */}
      <div className="w-full max-w-6xl mx-auto mt-2 mb-16 z-10">
        <h2 className="text-3xl mb-3 text-center outfit-400">
          {getPageTitle()}
        </h2>

        {!loading && (
          <>
            {!selectedFaculty ? (
              <FacultySelection
                faculties={faculties}
                onSelect={setSelectedFaculty}
              />
            ) : !selectedProgram ? (
              <ProgramSelection
                programs={programs}
                onSelect={setSelectedProgram}
              />
            ) : !selectedAcademicProgram ? (
              <AcademicProgramSelection
                academicPrograms={academicPrograms}
                onSelect={handleAcademicProgramSelect}
              />
            ) : !selectedYear ? (
              <YearSelection
                yearOptions={getYearOptions()}
                onSelect={handleYearSelect}
              />
            ) : !showRoadmap ? (
              <CoursePicker
                courseSections={courseSections}
                selectedCourses={selectedCourses}
                selectedGroups={selectedGroups}
                onCourseToggle={handleCourseToggle}
                onGroupToggle={handleGroupToggle}
                onContinue={handleContinueToRoadmap}
              />
            ) : (
              <CourseRoadmap
                selectedYear={selectedYear}
                degreeCode={selectedDegreeCode}
                semOneCourses={semOneCourses}
                semTwoCourses={semTwoCourses}
                semThreeCourses={semThreeCourses}
                onStatusChange={handleStatusChange}
              />
            )}
          </>
        )}
      </div>
    </>
  );
}
