// Fixed login-form.tsx - Properly handle API response structure

"use client";

import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import {
  AuthAPI,
  type LoginRequest,
  type SignupRequest,
} from "@/services/api/auth";
import { useAuth } from "@/components/AuthContext";
import { useRouter } from "next/navigation";

// Login form schema
const loginSchema = z.object({
  username: z.string().min(1, "Username is required"),
  password: z.string().min(1, "Password is required"),
});

// Signup form schema
const signupSchema = z
  .object({
    username: z.string().min(3, "Username must be at least 3 characters"),
    email: z.string().email("Invalid email address"),
    password: z.string().min(6, "Password must be at least 6 characters"),
    confirmPassword: z.string().min(1, "Please confirm your password"),
  })
  .refine((data) => data.password === data.confirmPassword, {
    message: "Passwords don't match",
    path: ["confirmPassword"],
  });

type LoginFormData = z.infer<typeof loginSchema>;
type SignupFormData = z.infer<typeof signupSchema>;

// Define the expected API response structure
interface APIInboxMessage {
  question: string;
  answer: string | null;
  student: string;
  advisor: string;
  // API might be missing these properties
  category?: string;
  subject?: string;
  read?: boolean;
}

interface APILoginResponse {
  username: string;
  email?: string;
  is_advisor: boolean;
  red_dot?: boolean; // Make this optional since it might not be in the API
  inbox?: APIInboxMessage[];
}

export function LoginForm({
  className,
  ...props
}: React.ComponentProps<"form">) {
  const [isSignup, setIsSignup] = useState(false);
  const { login } = useAuth();
  const router = useRouter();

  // Login form
  const loginForm = useForm<LoginFormData>({
    resolver: zodResolver(loginSchema),
    mode: "onSubmit",
    defaultValues: {
      username: "",
      password: "",
    },
  });

  // Signup form
  const signupForm = useForm<SignupFormData>({
    resolver: zodResolver(signupSchema),
    mode: "onSubmit",
    defaultValues: {
      username: "",
      email: "",
      password: "",
      confirmPassword: "",
    },
  });

  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  // Helper function to transform API inbox to User inbox format
  const transformInboxData = (apiInbox: APIInboxMessage[] = []) => {
    return apiInbox.map((msg) => ({
      category: msg.category || "GENERAL", // Provide default if missing
      subject: msg.subject || "No Subject", // Provide default if missing
      question: msg.question,
      answer: msg.answer,
      student: msg.student,
      advisor: msg.advisor,
      read: msg.read ?? false, // Default to false if missing
    }));
  };

  const onLoginSubmit = async (data: LoginFormData) => {
    setIsLoading(true);
    setError(null);

    try {
      const result = await AuthAPI.login(data);

      if (result.status === 200 && result.data) {
        console.log("Login successful:", result.message);
        console.log("API Response data:", result.data);

        // Type assertion with the expected API structure
        const apiData = result.data as APILoginResponse;

        // Create user object matching the User interface with all required fields
        const userData = {
          id: apiData.username, // Use username as ID since no user_id is provided
          username: apiData.username,
          email: apiData.email || `${apiData.username}@uct.ac.za`, // Fallback email if not provided
          is_advisor: apiData.is_advisor,
          red_dot: apiData.red_dot ?? false, // Default to false if not provided by API
          inbox: transformInboxData(apiData.inbox), // Transform API inbox to match User interface
        };

        console.log("Created userData:", userData);

        // Generate a session token since the API doesn't provide one
        const sessionToken = `session_${Date.now()}_${apiData.username}`;

        login(userData, sessionToken);
        router.push("/");
      } else if (result.status === 400) {
        setError(result.message || "Login failed");
      } else {
        setError("Login failed. Please try again.");
      }
    } catch (err) {
      console.error("Login error:", err);
      setError(err instanceof Error ? err.message : "An error occurred");
    } finally {
      setIsLoading(false);
    }
  };

  const onSignupSubmit = async (data: SignupFormData) => {
    setIsLoading(true);
    setError(null);

    try {
      const result = await AuthAPI.signup(data);

      if (result.status === 200 && result.data) {
        console.log("Signup successful:", result.message);
        console.log("Signup API Response data:", result.data);

        // Create user object using the data from your API response
        const userData = {
          id: result.data.username, // Use username as ID
          username: result.data.username,
          email: result.data.email || data.email, // Use signup email or fallback to API response
          is_advisor: result.data.is_advisor,
          red_dot: result.data.red_dot,
          inbox: result.data.inbox, // Direct access - structures match perfectly
        };

        console.log("Created signup userData:", userData);

        // Generate a session token since the API doesn't provide one
        const sessionToken = `session_${Date.now()}_${result.data.username}`;

        login(userData, sessionToken);
        router.push("/");
      } else if (result.status === 400) {
        setError(result.message || "Signup failed");
      } else {
        setError("Signup failed. Please try again.");
      }
    } catch (err) {
      console.error("Signup error:", err);
      setError(err instanceof Error ? err.message : "An error occurred");
    } finally {
      setIsLoading(false);
    }
  };

  if (isSignup) {
    return (
      <form
        className={cn("flex flex-col gap-6", className)}
        onSubmit={signupForm.handleSubmit(onSignupSubmit)}
        {...props}
      >
        <div className="flex flex-col items-center gap-2 text-center">
          <h1 className="text-2xl font-bold outfit-400">Create your account</h1>
          <p className="text-muted-foreground text-sm text-balance outfit-500">
            Enter your details below to create your account
          </p>
        </div>
        <div className="grid gap-6">
          {error && (
            <div className="p-3 text-sm text-red-600 bg-red-50 border border-red-200 rounded-md">
              {error}
            </div>
          )}
          <div className="grid gap-3">
            <Label htmlFor="username">Username</Label>
            <Input
              id="username"
              type="text"
              placeholder="johndoe"
              {...signupForm.register("username")}
            />
            {signupForm.formState.errors.username && (
              <p className="text-sm text-red-600">
                {signupForm.formState.errors.username.message}
              </p>
            )}
          </div>
          <div className="grid gap-3">
            <Label htmlFor="email">Email</Label>
            <Input
              id="email"
              type="email"
              placeholder="m@example.com"
              {...signupForm.register("email")}
            />
            {signupForm.formState.errors.email && (
              <p className="text-sm text-red-600">
                {signupForm.formState.errors.email.message}
              </p>
            )}
          </div>
          <div className="grid gap-3">
            <Label htmlFor="password">Password</Label>
            <Input
              id="password"
              type="password"
              {...signupForm.register("password")}
            />
            {signupForm.formState.errors.password && (
              <p className="text-sm text-red-600">
                {signupForm.formState.errors.password.message}
              </p>
            )}
          </div>
          <div className="grid gap-3">
            <Label htmlFor="confirmPassword">Confirm Password</Label>
            <Input
              id="confirmPassword"
              type="password"
              {...signupForm.register("confirmPassword")}
            />
            {signupForm.formState.errors.confirmPassword && (
              <p className="text-sm text-red-600">
                {signupForm.formState.errors.confirmPassword.message}
              </p>
            )}
          </div>
          <Button type="submit" className="w-full" disabled={isLoading}>
            {isLoading ? "Creating Account..." : "Create Account"}
          </Button>
          <div className="after:border-border relative text-center text-sm after:absolute after:inset-0 after:top-1/2 after:z-0 after:flex after:items-center after:border-t">
            <span className="bg-background text-uct-heading-grey relative z-10 px-2 bg-gradient-to-br from-sky-50 via-sky-50 to-violet-50">
              Already have an account?
            </span>
          </div>
          <Button
            type="button"
            variant="outline"
            className="w-full bg-white"
            onClick={() => {
              setIsSignup(false);
              loginForm.reset();
              loginForm.clearErrors();
              setError(null);
            }}
          >
            Sign In
          </Button>
        </div>
      </form>
    );
  }

  return (
    <form
      className={cn("flex flex-col gap-6", className)}
      onSubmit={loginForm.handleSubmit(onLoginSubmit)}
      {...props}
    >
      <div className="flex flex-col items-center gap-2 text-center">
        <h1 className="text-2xl font-bold outfit-400">Login to your account</h1>
        <p className="text-muted-foreground text-sm text-balance outfit-500">
          Enter your username below to login to your account
        </p>
      </div>
      <div className="grid gap-6">
        {error && (
          <div className="p-3 text-sm text-red-600 bg-red-50 border border-red-200 rounded-md">
            {error}
          </div>
        )}
        <div className="grid gap-3">
          <Label htmlFor="username">Username</Label>
          <Input
            id="username"
            type="text"
            placeholder="johndoe"
            {...loginForm.register("username")}
          />
          {loginForm.formState.errors.username && (
            <p className="text-sm text-red-600">
              {loginForm.formState.errors.username.message}
            </p>
          )}
        </div>
        <div className="grid gap-3">
          <div className="flex items-center">
            <Label htmlFor="password">Password</Label>
            <a
              href="#"
              className="ml-auto text-sm underline-offset-4 hover:underline"
            >
              Forgot your password?
            </a>
          </div>
          <Input
            id="password"
            type="password"
            {...loginForm.register("password")}
          />
          {loginForm.formState.errors.password && (
            <p className="text-sm text-red-600">
              {loginForm.formState.errors.password.message}
            </p>
          )}
        </div>
        <Button type="submit" className="w-full" disabled={isLoading}>
          {isLoading ? "Logging in..." : "Login"}
        </Button>
        <div className="after:border-border relative text-center text-sm after:absolute after:inset-0 after:top-1/2 after:z-0 after:flex after:items-center after:border-t">
          <span className="bg-background text-uct-heading-grey relative z-10 px-2 bg-gradient-to-br from-sky-50 via-sky-50 to-violet-50">
            Or continue with
          </span>
        </div>
        <Button
          type="button"
          variant="outline"
          className="w-full bg-white"
          onClick={() => {
            setIsSignup(true);
            signupForm.reset();
            signupForm.clearErrors();
          }}
        >
          Create An Account
        </Button>
      </div>
    </form>
  );
}
