"use client";

import { useState, useEffect } from "react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import {
  ArrowLeft,
  Reply,
  Send,
  User,
  Clock,
  Mail,
  MailOpen,
  FileText,
} from "lucide-react";
import { useAuth } from "@/components/AuthContext";

interface InboxMessage {
  category: string;
  subject: string;
  question: string;
  answer: string | null;
  student: string;
  advisor: string;
  read: boolean;
  question_timestamp?: string;
  answer_timestamp?: string;
}

interface ProcessedMessage {
  id: string;
  from: string;
  subject: string;
  body: string;
  timestamp: string;
  isRead: boolean;
  isReply?: boolean;
  category: string;
  originalIndex: number; // Track original position for read status updates
  question: string;
  answer: string | null;
  student: string;
  advisor: string;
}

interface InboxWindowProps {
  isOpen: boolean;
  onOpenChange: (open: boolean) => void;
}

const getCategoryBadgeStyle = (category: string) => {
  const normalizedCategory = category.toUpperCase();
  const styles: Record<string, string> = {
    FACULTY: "bg-purple-100 text-purple-800 border-purple-200",
    PROGRAM: "bg-blue-100 text-blue-800 border-blue-200",
    DEGREE: "bg-green-100 text-green-800 border-green-200",
    COURSE: "bg-orange-100 text-orange-800 border-orange-200",
    DEPARTMENT: "bg-indigo-100 text-indigo-800 border-indigo-200",
    REQUEST: "bg-cyan-100 text-cyan-800 border-cyan-200",
    EMAIL: "bg-gray-100 text-gray-800 border-gray-200",
    SOLVED: "bg-emerald-100 text-emerald-800 border-emerald-200",
    "DAMAGE REPORT": "bg-red-100 text-red-800 border-red-200",
    POSTAL: "bg-yellow-100 text-yellow-800 border-yellow-200",
    SMS: "bg-pink-100 text-pink-800 border-pink-200",
    SCHADENSMELDUNG: "bg-amber-100 text-amber-800 border-amber-200",
  };
  return styles[normalizedCategory] || styles["EMAIL"];
};

const getRelativeTime = (timestamp: string) => {
  const now = new Date();
  const messageTime = new Date(timestamp);
  const diffInMilliseconds = now.getTime() - messageTime.getTime();
  const diffInMinutes = Math.floor(diffInMilliseconds / (1000 * 60));
  const diffInHours = Math.floor(diffInMilliseconds / (1000 * 60 * 60));
  const diffInDays = Math.floor(diffInMilliseconds / (1000 * 60 * 60 * 24));

  if (diffInMinutes < 1) return "Just now";
  if (diffInMinutes < 60) return `${diffInMinutes}m ago`;
  if (diffInHours < 24) return `${diffInHours}h ago`;
  if (diffInDays === 1) return "Yesterday";
  if (diffInDays < 7) return `${diffInDays} days ago`;
  if (diffInDays < 30) return `${Math.floor(diffInDays / 7)} weeks ago`;
  return `${Math.floor(diffInDays / 30)} months ago`;
};

export function InboxWindow({ isOpen, onOpenChange }: InboxWindowProps) {
  const { user, updateMessageReadStatus, refreshUserData } = useAuth();
  const [messages, setMessages] = useState<ProcessedMessage[]>([]);
  const [selectedMessage, setSelectedMessage] =
    useState<ProcessedMessage | null>(null);
  const [replyText, setReplyText] = useState("");
  const [isReplying, setIsReplying] = useState(false);
  const [isSendingReply, setIsSendingReply] = useState(false);

  // Handle closing the inbox and refreshing user data
  const handleCloseInbox = async (open: boolean) => {
    if (!open) {
      console.log("📱 Closing inbox and refreshing user data...");

      // Reset any local state
      setSelectedMessage(null);
      setIsReplying(false);
      setReplyText("");

      // Refresh user data to sync with database
      try {
        await refreshUserData();
        console.log("✅ User data refreshed after closing inbox");
      } catch (error) {
        console.error("⚠️ Error refreshing user data:", error);
        // Don't show error to user as this is background sync
      }
    }

    // Always call the original onOpenChange
    onOpenChange(open);
  };

  // Function to call refresh API with specific message marked as read
  const callRefreshWithMessage = async (message: ProcessedMessage) => {
    if (!user) return;

    try {
      console.log("📤 Calling refresh API for message:", message.subject);

      // Get the original message from user.inbox
      const originalMessage = user.inbox[message.originalIndex];

      // Create the message object with read: true
      const messageToSend = {
        question: originalMessage.question,
        answer: originalMessage.answer || "",
        subject: originalMessage.subject,
        read: true, // Always set to true
        // Include other fields if they exist
        ...(originalMessage.category && { category: originalMessage.category }),
        ...(originalMessage.student && { student: originalMessage.student }),
        ...(originalMessage.advisor && { advisor: originalMessage.advisor }),
      };

      const refreshData = {
        username: user.username,
        is_advisor: user.is_advisor,
        inbox: [messageToSend],
      };

      console.log("📤 Sending refresh data:", refreshData);

      const response = await fetch("/api/refresh/", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(refreshData),
      });

      if (response.ok) {
        const result = await response.json();
        console.log("✅ Refresh API call successful (data ignored):", result);
      } else {
        console.error("⚠️ Refresh API call failed:", response.statusText);
      }
    } catch (error) {
      console.error("⚠️ Error calling refresh API:", error);
    }
  };

  // Process inbox data from user context
  useEffect(() => {
    console.log("📬 InboxWindow - useEffect triggered");
    console.log("📬 InboxWindow - user:", user);
    console.log("📬 InboxWindow - user?.inbox:", user?.inbox);

    if (user?.inbox) {
      console.log(
        "📬 InboxWindow - Processing inbox with",
        user.inbox.length,
        "messages"
      );

      const processedMessages: ProcessedMessage[] = user.inbox.map(
        (msg, index) => {
          // Determine who the message is "from" based on user type
          const isUserAdvisor = user.is_advisor;
          const fromPerson = isUserAdvisor ? msg.student : msg.advisor;

          // Determine which timestamp to use
          let messageTimestamp: string;
          if (msg.answer && msg.answer.trim() !== "" && msg.answer_timestamp) {
            // Use answer timestamp if message has an answer
            messageTimestamp = msg.answer_timestamp;
          } else if (msg.question_timestamp) {
            // Use question timestamp if no answer or answer is empty
            messageTimestamp = msg.question_timestamp;
          } else {
            // Fallback to synthetic timestamp if no timestamps available
            const baseDate = new Date();
            baseDate.setHours(baseDate.getHours() - index * 2);
            messageTimestamp = baseDate.toISOString();
          }

          console.log(`📬 InboxWindow - Message ${index}:`, {
            subject: msg.subject,
            from: fromPerson,
            isUserAdvisor,
            read: msg.read,
            hasAnswer: !!(msg.answer && msg.answer.trim() !== ""),
            usingTimestamp:
              msg.answer && msg.answer.trim() !== ""
                ? "answer_timestamp"
                : "question_timestamp",
            timestamp: messageTimestamp,
          });

          return {
            id: `msg_${index}`,
            from: fromPerson,
            subject: msg.subject,
            body: msg.question,
            timestamp: messageTimestamp,
            isRead: msg.read,
            category: msg.category,
            originalIndex: index,
            question: msg.question,
            answer: msg.answer,
            student: msg.student,
            advisor: msg.advisor,
          };
        }
      );

      console.log("📬 InboxWindow - Processed messages:", processedMessages);
      setMessages(processedMessages);
    } else {
      console.log(
        "📬 InboxWindow - No inbox data found, setting empty messages"
      );
      setMessages([]);
    }
  }, [user]);

  const handleMessageClick = async (message: ProcessedMessage) => {
    // Call refresh API with this message marked as read (only if it's currently unread)
    if (!message.isRead) {
      await callRefreshWithMessage(message);
    }

    // Mark as read when opening
    if (!message.isRead) {
      updateMessageReadStatus(message.originalIndex);

      // Update local state
      setMessages((prev) =>
        prev.map((msg) =>
          msg.id === message.id ? { ...msg, isRead: true } : msg
        )
      );
    }
    setSelectedMessage(message);
    setIsReplying(false);
  };

  const handleReply = () => {
    setIsReplying(true);
  };

  const handleSendReply = async () => {
    if (replyText.trim() && selectedMessage && user) {
      setIsSendingReply(true);

      try {
        // Prepare the data for the API call
        const answerData = {
          subject: selectedMessage.subject,
          question: selectedMessage.question,
          answer: replyText.trim(),
          student: selectedMessage.student,
          advisor: user.username, // Use the current user's username as advisor
        };

        console.log("Submitting answer:", answerData);

        // Call the API route
        const response = await fetch("/api/answer-question", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(answerData),
        });

        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || "Failed to submit answer");
        }

        const result = await response.json();
        console.log("Answer submitted successfully:", result);

        // Update the current selected message with the answer
        const updatedMessage = {
          ...selectedMessage,
          answer: replyText.trim(),
          advisor: user.username,
        };

        // Update the selected message state
        setSelectedMessage(updatedMessage);

        // Update the messages array to reflect the answer
        setMessages((prev) =>
          prev.map((msg) =>
            msg.id === selectedMessage.id
              ? {
                  ...msg,
                  answer: replyText.trim(),
                  advisor: user.username,
                }
              : msg
          )
        );

        // Clear reply state
        setReplyText("");
        setIsReplying(false);

        // Show success message (optional)
        console.log("Reply sent successfully!");
      } catch (error) {
        console.error("Error sending reply:", error);
        // You might want to show an error toast/notification here
        alert(
          `Failed to send reply: ${
            error instanceof Error ? error.message : "Unknown error"
          }`
        );
      } finally {
        setIsSendingReply(false);
      }
    }
  };

  const handleBackToList = () => {
    setSelectedMessage(null);
    setIsReplying(false);
    setReplyText("");
  };

  const unreadCount = messages.filter((msg) => !msg.isRead).length;

  // Helper function to determine if reply button should be shown
  const shouldShowReplyButton = (message: ProcessedMessage) => {
    return (
      user?.is_advisor && // User must be an advisor
      message.question && // Message must have a question
      !message.answer && // Message must NOT have an answer yet
      !message.isReply // Don't show for reply messages
    );
  };

  // Show empty state if no user or no messages
  if (!user) {
    return (
      <Dialog open={isOpen} onOpenChange={handleCloseInbox}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Please log in</DialogTitle>
          </DialogHeader>
          <p className="text-center text-gray-500">
            You need to be logged in to view your inbox.
          </p>
        </DialogContent>
      </Dialog>
    );
  }

  return (
    <Dialog open={isOpen} onOpenChange={handleCloseInbox}>
      <DialogContent
        className="max-w-none w-[90vw] md:w-[80vw] max-h-[90vh] h-[80vh] p-0 overflow-hidden"
        style={{ width: "98vw", maxWidth: "80vw" }}
      >
        <div className="flex h-full overflow-hidden w-full md:w-[80vw]">
          {/* Messages List */}
          <div
            className={`${
              selectedMessage ? "hidden md:flex" : "flex"
            } w-[90vw] md:w-1/4 border-r flex-col overflow-hidden `}
          >
            <DialogHeader className="p-4 border-b flex-shrink-0">
              <DialogTitle className="flex items-center gap-2">
                <Mail className="h-5 w-5" />
                Inbox
                {unreadCount > 0 && (
                  <Badge variant="destructive" className="ml-2">
                    {unreadCount}
                  </Badge>
                )}
              </DialogTitle>
            </DialogHeader>

            <div className="flex-1 overflow-hidden ">
              {messages.length === 0 ? (
                <div className="flex-1 flex items-center justify-center p-8">
                  <div className="text-center text-gray-500">
                    <Mail className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p className="text-lg font-medium mb-2">No messages yet</p>
                    <p className="text-sm">Your inbox is empty</p>
                  </div>
                </div>
              ) : (
                <ScrollArea className="h-full">
                  <div className="divide-y divide-gray-200">
                    {messages.map((message) => (
                      <div
                        key={message.id}
                        onClick={() => handleMessageClick(message)}
                        className={`p-4 cursor-pointer transition-colors hover:bg-gray-50 border-l-4 w-full max-w-full ${
                          !message.isRead
                            ? "bg-blue-50 border-l-blue-500"
                            : "bg-white border-l-transparent"
                        } ${
                          selectedMessage?.id === message.id
                            ? "bg-blue-100 border-l-blue-600"
                            : ""
                        }`}
                      >
                        <div className="flex items-start justify-between mb-2 min-w-0">
                          <div className="flex items-center gap-2 flex-1 min-w-0">
                            {/* Read/unread indicator dot */}
                            <div
                              className={`w-2 h-2 rounded-full flex-shrink-0 ${
                                !message.isRead ? "bg-blue-500" : "bg-gray-400"
                              }`}
                            />
                            <div className="flex items-center gap-1 flex-1 min-w-0">
                              {message.isReply ? (
                                <User className="h-4 w-4 text-green-600 flex-shrink-0" />
                              ) : (
                                <User className="h-4 w-4 text-gray-500 flex-shrink-0" />
                              )}
                              <span
                                className={`text-sm ${
                                  !message.isRead
                                    ? "font-semibold"
                                    : "font-medium"
                                } truncate min-w-0`}
                              >
                                {message.from}
                              </span>
                            </div>
                            {!message.isRead && (
                              <MailOpen className="h-4 w-4 text-blue-600 flex-shrink-0" />
                            )}
                          </div>
                          <div className="text-xs text-gray-500 ml-4 flex-shrink-0">
                            {getRelativeTime(message.timestamp)}
                          </div>
                        </div>

                        <div
                          className={`text-sm ${
                            !message.isRead ? "font-semibold" : "font-medium"
                          } mb-2 text-gray-900 truncate`}
                        >
                          {message.subject}
                        </div>

                        <p className="text-xs text-gray-600 line-clamp-2 mb-3 break-words">
                          {message.body}
                        </p>

                        <div className="flex justify-between items-center flex-wrap gap-2">
                          <Badge
                            variant="outline"
                            className={`text-xs px-2 py-1 flex-shrink-0 ${getCategoryBadgeStyle(
                              message.category
                            )}`}
                          >
                            {message.category.toUpperCase()}
                          </Badge>
                          {/* Show answered indicator */}
                          {message.answer && (
                            <Badge
                              variant="outline"
                              className="text-xs px-2 py-1 bg-green-100 text-green-800 border-green-200 flex-shrink-0"
                            >
                              ANSWERED
                            </Badge>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                </ScrollArea>
              )}
            </div>
          </div>

          {/* Message Detail */}
          <div
            className={`${
              selectedMessage ? "flex" : "hidden"
            } w-full md:w-3/4 flex-col overflow-hidden`}
          >
            {selectedMessage ? (
              <>
                <div className="p-6 border-b bg-gradient-to-r from-gray-50 to-white flex-shrink-0">
                  <div className="flex items-center gap-3 mb-4">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={handleBackToList}
                      className="md:hidden flex-shrink-0"
                    >
                      <ArrowLeft className="h-4 w-4" />
                    </Button>
                    <div className="flex-1 min-w-0">
                      <h3 className="font-semibold md:text-xl text-gray-900 mb-1 break-words">
                        {selectedMessage.subject}
                      </h3>
                      <div className="flex gap-2 flex-wrap">
                        <Badge
                          variant="outline"
                          className={`text-xs ${getCategoryBadgeStyle(
                            selectedMessage.category
                          )}`}
                        >
                          {selectedMessage.category.toUpperCase()}
                        </Badge>
                        {selectedMessage.answer && (
                          <Badge
                            variant="outline"
                            className="text-xs bg-green-100 text-green-800 border-green-200"
                          >
                            ANSWERED
                          </Badge>
                        )}
                      </div>
                    </div>
                    {/* Only show reply button if conditions are met */}
                    {shouldShowReplyButton(selectedMessage) && (
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={handleReply}
                        disabled={isReplying}
                        className="bg-white hover:bg-gray-50 flex-shrink-0"
                      >
                        <Reply className="h-4 w-4 mr-2" />
                        Reply
                      </Button>
                    )}
                  </div>
                  <div className="flex items-center gap-4 text-sm text-gray-600 flex-wrap">
                    <div className="flex items-center gap-2 min-w-0">
                      <User className="h-4 w-4 flex-shrink-0" />
                      <span className="font-medium truncate">
                        {selectedMessage.from}
                      </span>
                    </div>
                    <Separator
                      orientation="vertical"
                      className="h-4 hidden sm:block"
                    />
                    <div className="flex items-center gap-2 min-w-0">
                      <Clock className="h-4 w-4 flex-shrink-0" />
                      <span className="truncate text-xs sm:text-sm">
                        {new Date(selectedMessage.timestamp).toLocaleString()}
                      </span>
                    </div>
                  </div>
                </div>

                <ScrollArea
                  className={`flex-1 w-full ${
                    isReplying && shouldShowReplyButton(selectedMessage)
                      ? "max-h-[calc(48vh-8rem)] md:max-h-[calc(58vh-8rem)]" // Dynamic height based on viewport minus header/padding
                      : "max-h-[calc(80vh-8rem)]" // Full height when not replying
                  }`}
                >
                  <div className="p-6 space-y-6 w-full max-w-full">
                    {/* Question Section */}
                    {selectedMessage.question && (
                      <div className="bg-white rounded-lg border border-gray-200 shadow-sm w-full max-w-full">
                        <div className="bg-blue-50 px-6 py-3 rounded-t-lg border-b border-gray-200">
                          <div className="flex items-center gap-2 min-w-0">
                            <User className="h-4 w-4 text-blue-600 flex-shrink-0" />
                            <span className="font-medium text-blue-800 truncate">
                              {selectedMessage.student}
                            </span>
                            <span className="text-sm text-blue-600 flex-shrink-0">
                              asked:
                            </span>
                          </div>
                        </div>
                        <ScrollArea className=" w-full">
                          <div className="p-6 w-full max-w-full">
                            <p className="text-gray-800 leading-relaxed whitespace-pre-wrap text-sm md:text-base break-words w-full">
                              {selectedMessage.question}
                            </p>
                          </div>
                        </ScrollArea>
                      </div>
                    )}

                    {/* Answer Section - Show existing answer or reply being composed */}
                    {selectedMessage.answer && (
                      <div className="bg-white rounded-lg border border-gray-200 shadow-sm w-full max-w-full">
                        <div className="bg-green-50 px-6 py-3 rounded-t-lg border-b border-gray-200">
                          <div className="flex items-center gap-2 min-w-0">
                            <User className="h-4 w-4 text-green-600 flex-shrink-0" />
                            <span className="font-medium text-green-800 truncate">
                              {selectedMessage.advisor}
                            </span>
                            <span className="text-sm text-green-600 flex-shrink-0">
                              answered:
                            </span>
                          </div>
                        </div>
                        <ScrollArea className="max-h-[20rem] w-full">
                          <div className="p-6 w-full max-w-full">
                            <p className="text-gray-800 leading-relaxed whitespace-pre-wrap text-sm  md:text-base break-words w-full">
                              {selectedMessage.answer}
                            </p>
                          </div>
                        </ScrollArea>
                      </div>
                    )}

                    {/* For reply messages, show the original format */}
                    {selectedMessage.isReply && (
                      <div className="bg-white rounded-lg border border-gray-200 shadow-sm w-full max-w-full">
                        <ScrollArea className="max-h-[20rem] w-full">
                          <div className="p-6 w-full max-w-full">
                            <p className="text-gray-800 leading-relaxed whitespace-pre-wrap text-sm  md:text-base break-words w-full">
                              {selectedMessage.body}
                            </p>
                          </div>
                        </ScrollArea>
                      </div>
                    )}
                  </div>
                </ScrollArea>

                {/* Reply Section - Only shown if user is advisor and conditions are met */}
                {isReplying && shouldShowReplyButton(selectedMessage) && (
                  <div className="border-t bg-gray-50 p-4 md:p-6 space-y-4 max-h-[calc(30vh)] w-full max-w-full flex flex-col">
                    <div className="flex items-center gap-2 text-sm font-medium text-gray-700">
                      <Reply className="h-4 w-4" />
                      Reply to {selectedMessage.from}
                    </div>
                    <Textarea
                      placeholder="Type your reply here..."
                      value={replyText}
                      onChange={(e) => setReplyText(e.target.value)}
                      className="flex-1  max-h-[calc(20vh-4rem)] text-sm resize-none bg-white w-full"
                    />
                    <div className="flex justify-end gap-3">
                      <Button
                        variant="outline"
                        onClick={() => {
                          setIsReplying(false);
                          setReplyText("");
                        }}
                      >
                        Cancel
                      </Button>
                      <Button
                        onClick={handleSendReply}
                        disabled={!replyText.trim() || isSendingReply}
                        className="bg-blue-600 hover:bg-blue-700 disabled:opacity-50"
                      >
                        <Send className="h-4 w-4 mr-2" />
                        {isSendingReply ? "Sending..." : "Send Reply"}
                      </Button>
                    </div>
                  </div>
                )}
              </>
            ) : (
              <div className="flex-1 flex items-center justify-center text-gray-500 bg-gray-50">
                <div className="text-center">
                  <FileText className="h-16 w-16 mx-auto mb-4 text-gray-300" />
                  <p className="text-lg font-medium mb-2">
                    Select a message to read
                  </p>
                  <p className="text-sm text-gray-400">
                    Choose a message from the list to view its contents
                  </p>
                </div>
              </div>
            )}
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
