"use client";

import { useState } from "react";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";

interface GlossaryTerm {
  term: string;
  definition: string;
}

interface GlossaryClientProps {
  glossaryTerms: GlossaryTerm[];
}

export function GlossaryClient({ glossaryTerms }: GlossaryClientProps) {
  const [search, setSearch] = useState("");

  // Group terms by first letter and filter based on search
  const filteredGlossary = Array.from({ length: 26 }, (_, i) => {
    const letter = String.fromCharCode(65 + i);

    // Get all terms that start with this letter
    const termsForLetter = glossaryTerms.filter((term) =>
      term.term.toUpperCase().startsWith(letter)
    );

    // If searching, filter terms by term name or definition (case-insensitive)
    const filteredTerms = search.trim()
      ? termsForLetter.filter(
          (entry) =>
            entry.term.toLowerCase().includes(search.trim().toLowerCase()) ||
            entry.definition.toLowerCase().includes(search.trim().toLowerCase())
        )
      : termsForLetter;

    return { letter, words: filteredTerms };
  }).filter((section) => section.words.length > 0);

  return (
    <>
      {/* Search Bar */}
      <form
        className="w-full md:w-1/3 px-8 md:px-0 h-fit mt-10 z-10"
        onSubmit={(e) => e.preventDefault()}
        autoComplete="off"
      >
        <Input
          type="text"
          value={search}
          onChange={(e) => setSearch(e.target.value)}
          placeholder="Search Here..."
          className="rounded-full border-1 border-gray-300 bg-white text-sm"
        />
      </form>

      {/* Alphabet buttons */}
      <div className="w-full flex justify-center mt-8 border-b-2 border-gray-200 pb-8 z-10 px-4 md:px-0">
        <div className="flex flex-wrap gap-1 md:gap-2 justify-center">
          {Array.from({ length: 26 }, (_, i) => {
            const letter = String.fromCharCode(65 + i);
            return (
              <a key={letter} href={`#${letter}`}>
                <Button
                  variant="outline"
                  className="w-8 h-8 md:w-9 md:h-9 p-0 text-sm md:text-lg outfit-500 text-gray-700 hover:bg-uct-blue hover:text-white transition-colors duration-150 bg-white"
                >
                  {letter}
                </Button>
              </a>
            );
          })}
        </div>
      </div>

      {/* Glossary Content */}
      <div className="w-full flex flex-col items-center mt-12 z-10">
        {filteredGlossary.length === 0 && (
          <div className="text-gray-500 mt-8 text-xl">No results found.</div>
        )}
        {filteredGlossary.map(({ letter, words }) => (
          <div
            key={letter}
            id={letter}
            className="w-11/12 md:w-9/12 flex flex-col md:flex-row md:items-start mb-16 scroll-mt-32"
          >
            {/* Big Letter */}
            <div className="w-full md:w-32 flex-shrink-0 flex items-center justify-start px-4 md:px-0  md:pr-8 mb-6 md:mb-0">
              <span className="text-8xl font-semibold text-uct-blue">
                {letter}
              </span>
            </div>
            {/* Cards */}
            <div className="flex-1 grid grid-cols-1 md:grid-cols-2 gap-6">
              {words.map((entry, idx) => (
                <Card key={idx} className="text-left">
                  <CardHeader>
                    <CardTitle className="text-xl outfit-500">
                      {entry.term}
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-gray-700 text-sm md:text-base outfit-300">
                      {entry.definition}
                    </p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        ))}
      </div>
    </>
  );
}
