"use client";

import React, { useState } from "react";
import { Input } from "@/components/ui/input";
import { Course } from "@/services/api/courses";

import { ArrowUpDown } from "lucide-react";
import {
  ColumnDef,
  flexRender,
  getCoreRowModel,
  useReactTable,
  getFilteredRowModel,
  getSortedRowModel,
  SortingState,
} from "@tanstack/react-table";

import {
  Dialog,
  DialogContent,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";

import { Button } from "@/components/ui/button";

import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";

const columns: ColumnDef<Course>[] = [
  {
    accessorKey: "course_name",
    header: ({ column }) => (
      <div className="flex items-center gap-2">
        Course Name
        <button
          type="button"
          onClick={() => column.toggleSorting(column.getIsSorted() === "asc")}
          className="ml-2"
        >
          <ArrowUpDown className="h-4 w-4" />
        </button>
      </div>
    ),
    enableGlobalFilter: true,
    enableSorting: true,
  },
  {
    accessorKey: "course_code",
    header: "Code",
    enableGlobalFilter: true,
  },
  {
    accessorKey: "semester",
    header: () => <div className="text-center w-full">Semester</div>,
    enableGlobalFilter: true,
    enableSorting: true,
    cell: ({ getValue }) => (
      <span className="flex justify-center">{getValue<string>()}</span>
    ),
  },
];

export function CourseTable({ courses }: { courses: Course[] }) {
  const [globalFilter, setGlobalFilter] = React.useState("");
  const [sorting, setSorting] = React.useState<SortingState>([]);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [selectedCourse, setSelectedCourse] = useState<Course | null>(null);

  const table = useReactTable({
    data: courses,
    columns,
    getCoreRowModel: getCoreRowModel(),
    getFilteredRowModel: getFilteredRowModel(),
    getSortedRowModel: getSortedRowModel(),
    state: {
      globalFilter,
      sorting,
    },
    onGlobalFilterChange: setGlobalFilter,
    onSortingChange: setSorting,
    globalFilterFn: "includesString",
  });

  const renderCourseDetails = () => {
    if (!selectedCourse) return <div>No course selected.</div>;

    return (
      <div className="space-y-6">
        <div className="overflow-y-auto space-y-8">
          <div className="space-y-4">
            <div className="">
              <div className="w-full flex flex-col items-center justify-center h-fit border-b-1 border-gray-300 py-4">
                <span className="text-xl font-semibold capitalize text-gray-800 text-center">
                  {selectedCourse.course_name.toLowerCase()}
                </span>
                <span className="mt-1 text-base text-uct-heading-grey text-center">
                  {selectedCourse.course_code}
                </span>
              </div>

              {/* Desktop Table View */}
              <div className="hidden md:block w-full h-fit border-b border-gray-300">
                <Table>
                  <TableBody>
                    <TableRow>
                      <TableCell className="w-40 font-medium text-center py-6">
                        Convener
                      </TableCell>
                      <TableCell className="text-center border-l border-gray-300">
                        {selectedCourse.convener}
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell className="w-40 font-medium text-center py-6">
                        NQF Level
                      </TableCell>
                      <TableCell className="text-center border-l border-gray-300">
                        {selectedCourse.nqf_level}
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell className="w-40 font-medium text-center py-6">
                        Credits
                      </TableCell>
                      <TableCell className="text-center border-l border-gray-300">
                        {selectedCourse.nqf_credits}
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell className="w-40 font-medium text-center py-6">
                        Semester
                      </TableCell>
                      <TableCell className="text-center border-l border-gray-300">
                        {selectedCourse.semester}
                      </TableCell>
                    </TableRow>
                    {selectedCourse.notes && (
                      <TableRow>
                        <TableCell className="w-40 font-medium text-center py-6">
                          Notes
                        </TableCell>
                        <TableCell className="text-center whitespace-pre-line border-l border-gray-300">
                          {selectedCourse.notes}
                        </TableCell>
                      </TableRow>
                    )}
                    {selectedCourse.outline && (
                      <TableRow>
                        <TableCell className="w-40 font-medium text-center py-6">
                          Course outline
                        </TableCell>
                        <TableCell className="text-center whitespace-pre-line border-l border-gray-300">
                          {selectedCourse.outline}
                        </TableCell>
                      </TableRow>
                    )}
                    {selectedCourse.entryRequirements && (
                      <TableRow>
                        <TableCell className="w-40 font-medium text-center py-6">
                          Course entry requirements
                        </TableCell>
                        <TableCell className="text-center border-l border-gray-300">
                          {selectedCourse.entryRequirements}
                        </TableCell>
                      </TableRow>
                    )}
                    {selectedCourse.dpRequirements && (
                      <TableRow>
                        <TableCell className="w-40 font-medium text-center py-6">
                          DP requirements
                        </TableCell>
                        <TableCell className="text-center border-l border-gray-300">
                          {selectedCourse.dpRequirements}
                        </TableCell>
                      </TableRow>
                    )}
                    {selectedCourse.assessment && (
                      <TableRow>
                        <TableCell className="w-40 font-medium text-center py-6">
                          Assessment
                        </TableCell>
                        <TableCell className="text-center whitespace-pre-line border-l border-gray-300">
                          {selectedCourse.assessment}
                        </TableCell>
                      </TableRow>
                    )}
                  </TableBody>
                </Table>
              </div>

              {/* Mobile Card View */}
              <div className="md:hidden max-h-[70vh] px-4 py-4 border-b border-gray-300">
                <div className="bg-gray-50 rounded-lg p-4 space-y-3">
                  <div className="space-y-3">
                    <div>
                      <span className="text-sm font-medium text-gray-500">
                        Convener
                      </span>
                      <p className="text-sm text-gray-900 mt-1">
                        {selectedCourse.convener}
                      </p>
                    </div>
                    <div>
                      <span className="font-medium text-gray-400">
                        NQF Level
                      </span>
                      <p className="text-sm text-gray-900 mt-1">
                        {selectedCourse.nqf_level}
                      </p>
                    </div>
                    <div>
                      <span className="font-medium text-gray-400">
                        NQF Credits
                      </span>
                      <p className="text-sm text-gray-900 mt-1">
                        {selectedCourse.nqf_credits}
                      </p>
                    </div>
                    <div>
                      <span className="font-medium text-gray-400">
                        Semester
                      </span>
                      <p className="text-sm text-gray-900 mt-1">
                        {selectedCourse.semester}
                      </p>
                    </div>
                    {selectedCourse.notes && (
                      <div>
                        <span className="font-medium text-gray-400">Notes</span>
                        <p className="text-sm text-gray-900 mt-1 whitespace-pre-line">
                          {selectedCourse.notes}
                        </p>
                      </div>
                    )}
                    {selectedCourse.outline && (
                      <div>
                        <span className="font-medium text-gray-400">
                          Course outline
                        </span>
                        <p className="text-sm text-gray-900 mt-1 whitespace-pre-line">
                          {selectedCourse.outline}
                        </p>
                      </div>
                    )}
                    {selectedCourse.entryRequirements && (
                      <div>
                        <span className="font-medium text-gray-400">
                          Course entry requirements
                        </span>
                        <p className="text-sm text-gray-900 mt-1">
                          {selectedCourse.entryRequirements}
                        </p>
                      </div>
                    )}
                    {selectedCourse.dpRequirements && (
                      <div>
                        <span className="font-medium text-gray-400">
                          DP requirements
                        </span>
                        <p className="text-sm text-gray-900 mt-1">
                          {selectedCourse.dpRequirements}
                        </p>
                      </div>
                    )}
                    {selectedCourse.assessment && (
                      <div>
                        <span className="font-medium text-gray-400">
                          Assessment
                        </span>
                        <p className="text-sm text-gray-900 mt-1 whitespace-pre-line">
                          {selectedCourse.assessment}
                        </p>
                      </div>
                    )}
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  };

  return (
    <div className="w-full">
      <div className="flex items-center justify-center py-4 px-8 md:px-0">
        <Input
          placeholder="Filter courses..."
          value={globalFilter ?? ""}
          onChange={(e) => setGlobalFilter(e.target.value)}
          className="max-w-2xl border-2 rounded-full bg-white"
        />
      </div>

      {/* Desktop Table View */}
      <div className="hidden md:block rounded-md border bg-white overflow-x-auto">
        <Table className="w-full min-w-[800px]">
          <TableHeader>
            {table.getHeaderGroups().map((headerGroup) => (
              <TableRow key={headerGroup.id}>
                {headerGroup.headers.map((header, index) => (
                  <TableHead
                    className={`text-base text-uct-heading-grey ${
                      index === 0
                        ? "w-[50%] px-6" // Course Name - 50% width
                        : index === 1
                        ? "w-[25%] px-4" // Code - 25% width
                        : "w-[25%] px-4" // Semester - 25% width
                    }`}
                    key={header.id}
                  >
                    {header.isPlaceholder
                      ? null
                      : flexRender(
                          header.column.columnDef.header,
                          header.getContext()
                        )}
                  </TableHead>
                ))}
              </TableRow>
            ))}
          </TableHeader>
          <TableBody>
            {table.getRowModel().rows.length ? (
              table.getRowModel().rows.map((row) => (
                <TableRow
                  key={row.id}
                  className="h-12 cursor-pointer hover:bg-gray-50"
                  onClick={() => {
                    setSelectedCourse(row.original);
                    setDialogOpen(true);
                  }}
                >
                  {row.getVisibleCells().map((cell, index) => (
                    <TableCell
                      key={cell.id}
                      className={`py-0 text-base font-medium ${
                        index === 0
                          ? "w-[50%] px-6" // Course Name
                          : index === 1
                          ? "w-[25%] px-4" // Code
                          : "w-[25%] px-4" // Semester
                      }`}
                    >
                      {flexRender(
                        cell.column.columnDef.cell,
                        cell.getContext()
                      )}
                    </TableCell>
                  ))}
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell
                  colSpan={columns.length}
                  className="h-24 text-center"
                >
                  No results.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>

      {/* Mobile Card View */}
      <div className="md:hidden space-y-4">
        {table.getRowModel().rows.length ? (
          table.getRowModel().rows.map((row) => (
            <div
              key={row.id}
              className="bg-white rounded-lg border border-gray-200 p-4 mx-4 cursor-pointer hover:bg-gray-50 transition-colors shadow-sm"
              onClick={() => {
                setSelectedCourse(row.original);
                setDialogOpen(true);
              }}
            >
              <div className="space-y-3">
                <div className="flex items-start justify-between">
                  <div className="flex-1 min-w-0">
                    <h3 className="text-base w-10/12 font-semibold text-gray-900 leading-tight">
                      {row.original.course_name}
                    </h3>
                  </div>
                  <div className="flex-shrink-0 ml-2">
                    <svg
                      className="w-5 h-5 text-gray-400"
                      fill="none"
                      stroke="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path
                        strokeLinecap="round"
                        strokeLinejoin="round"
                        strokeWidth={2}
                        d="M9 5l7 7-7 7"
                      />
                    </svg>
                  </div>
                </div>

                <div className="space-y-2 text-sm">
                  <div className="flex justify-between items-center py-1">
                    <span className="text-gray-500 font-medium">Code</span>
                    <span className="text-gray-900 font-medium">
                      {row.original.course_code}
                    </span>
                  </div>
                  <div className="flex justify-between items-center py-1">
                    <span className="text-gray-500 font-medium">Semester</span>
                    <span className="text-gray-900 font-medium">
                      {row.original.semester}
                    </span>
                  </div>
                </div>
              </div>
            </div>
          ))
        ) : (
          <div className="text-center py-12">
            <p className="text-gray-500">No results found.</p>
          </div>
        )}
      </div>

      {/* Course Details Dialog */}
      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="!max-w-full md:!max-w-4/6 h-full md:h-fit p-0 rounded-lg overflow-hidden">
          <DialogTitle className="sr-only">
            {selectedCourse ? selectedCourse.course_name : "Course Details"}
          </DialogTitle>
          <div className="flex-1 flex flex-col bg-white min-w-0">
            <div className="flex-1 overflow-y-auto p-6 md:p-5">
              {renderCourseDetails()}
            </div>
            <DialogFooter className="hidden px-4 py-4 md:py-6 md:sticky md:bottom-0 bg-white z-10 md:flex justify-center md:justify-end border-t md:border-t-0">
              <Button onClick={() => setDialogOpen(false)} type="button">
                Close
              </Button>
            </DialogFooter>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
