// components/ask-advisor/forms/ProgramQuestionForm.tsx

import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { QuestionFormFields } from "../QuestionFormFields";
import type { Faculty, Program, QuestionFormData } from "../types";

interface ProgramQuestionFormProps {
  faculties: Faculty[];
  programs: Program[];
  selectedFaculty: string;
  selectedProgram: string;
  onFacultyChange: (value: string) => void;
  onProgramChange: (value: string) => void;
  formData: QuestionFormData;
  onFormDataChange: (data: QuestionFormData) => void;
  onSubmit: () => void;
  isSubmitting: boolean;
  loading: boolean;
}

export function ProgramQuestionForm({
  faculties,
  programs,
  selectedFaculty,
  selectedProgram,
  onFacultyChange,
  onProgramChange,
  formData,
  onFormDataChange,
  onSubmit,
  isSubmitting,
  loading,
}: ProgramQuestionFormProps) {
  const isValid =
    selectedFaculty &&
    selectedProgram &&
    formData.subject.trim() &&
    formData.question.trim();

  return (
    <div className="space-y-4">
      <h3 className="text-lg font-semibold text-gray-800">
        Program of Study Question
      </h3>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div className="space-y-2">
          <Label htmlFor="program-faculty-select">Select Faculty</Label>
          <Select value={selectedFaculty} onValueChange={onFacultyChange}>
            <SelectTrigger id="program-faculty-select" className="h-11">
              <SelectValue placeholder="Choose a faculty" />
            </SelectTrigger>
            <SelectContent>
              {faculties.map((faculty) => (
                <SelectItem
                  key={faculty.faculty_code}
                  value={faculty.faculty_code}
                >
                  {faculty.faculty_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="space-y-2">
          <Label htmlFor="program-study-select">Select Program of Study</Label>
          <Select
            value={selectedProgram}
            onValueChange={onProgramChange}
            disabled={!selectedFaculty || loading}
          >
            <SelectTrigger id="program-study-select" className="h-11">
              <SelectValue placeholder="Choose a program" />
            </SelectTrigger>
            <SelectContent>
              {programs.map((program) => (
                <SelectItem
                  key={program.program_code}
                  value={program.program_code}
                >
                  {program.program_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      <QuestionFormFields
        subject={formData.subject}
        question={formData.question}
        onSubjectChange={(value) =>
          onFormDataChange({ ...formData, subject: value })
        }
        onQuestionChange={(value) =>
          onFormDataChange({ ...formData, question: value })
        }
        onSubmit={onSubmit}
        isSubmitting={isSubmitting}
        isValid={!!isValid}
        submitButtonText="Submit Program Question"
        questionPlaceholder="Ask your question about this program of study..."
      />
    </div>
  );
}
