// components/ask-advisor/forms/MajorQuestionForm.tsx

import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { QuestionFormFields } from "../QuestionFormFields";
import type { Faculty, Program, Major, QuestionFormData } from "../types";

interface MajorQuestionFormProps {
  faculties: Faculty[];
  programs: Program[];
  majors: Major[];
  selectedFaculty: string;
  selectedProgram: string;
  selectedMajor: string;
  onFacultyChange: (value: string) => void;
  onProgramChange: (value: string) => void;
  onMajorChange: (value: string) => void;
  formData: QuestionFormData;
  onFormDataChange: (data: QuestionFormData) => void;
  onSubmit: () => void;
  isSubmitting: boolean;
  loading: boolean;
}

export function MajorQuestionForm({
  faculties,
  programs,
  majors,
  selectedFaculty,
  selectedProgram,
  selectedMajor,
  onFacultyChange,
  onProgramChange,
  onMajorChange,
  formData,
  onFormDataChange,
  onSubmit,
  isSubmitting,
  loading,
}: MajorQuestionFormProps) {
  const isValid =
    selectedFaculty &&
    selectedProgram &&
    selectedMajor &&
    formData.subject.trim() &&
    formData.question.trim();

  return (
    <div className="space-y-4">
      <h3 className="text-lg font-semibold text-gray-800">Major Question</h3>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="space-y-2">
          <Label htmlFor="major-faculty-select">Select Faculty</Label>
          <Select value={selectedFaculty} onValueChange={onFacultyChange}>
            <SelectTrigger id="major-faculty-select" className="h-11">
              <SelectValue placeholder="Choose a faculty" />
            </SelectTrigger>
            <SelectContent>
              {faculties.map((faculty) => (
                <SelectItem
                  key={faculty.faculty_code}
                  value={faculty.faculty_code}
                >
                  {faculty.faculty_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="space-y-2">
          <Label htmlFor="major-program-select">Select Program</Label>
          <Select
            value={selectedProgram}
            onValueChange={onProgramChange}
            disabled={!selectedFaculty || loading}
          >
            <SelectTrigger id="major-program-select" className="h-11">
              <SelectValue placeholder="Choose a program" />
            </SelectTrigger>
            <SelectContent>
              {programs.map((program) => (
                <SelectItem
                  key={program.program_code}
                  value={program.program_code}
                >
                  {program.program_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="space-y-2">
          <Label htmlFor="major-select">Select Major</Label>
          <Select
            value={selectedMajor}
            onValueChange={onMajorChange}
            disabled={!selectedProgram || loading}
          >
            <SelectTrigger id="major-select" className="h-11">
              <SelectValue placeholder="Choose a major" />
            </SelectTrigger>
            <SelectContent>
              {majors.map((major) => (
                <SelectItem key={major.major_code} value={major.major_code}>
                  {major.major_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      <QuestionFormFields
        subject={formData.subject}
        question={formData.question}
        onSubjectChange={(value) =>
          onFormDataChange({ ...formData, subject: value })
        }
        onQuestionChange={(value) =>
          onFormDataChange({ ...formData, question: value })
        }
        onSubmit={onSubmit}
        isSubmitting={isSubmitting}
        isValid={!!isValid}
        submitButtonText="Submit Major Question"
        questionPlaceholder="Ask your question about this major..."
      />
    </div>
  );
}
