// components/ask-advisor/forms/DegreeQuestionForm.tsx

import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { QuestionFormFields } from "../QuestionFormFields";
import type { Faculty, Program, Degree, QuestionFormData } from "../types";

interface DegreeQuestionFormProps {
  faculties: Faculty[];
  programs: Program[];
  degrees: Degree[];
  selectedFaculty: string;
  selectedProgram: string;
  selectedDegree: string;
  onFacultyChange: (value: string) => void;
  onProgramChange: (value: string) => void;
  onDegreeChange: (value: string) => void;
  formData: QuestionFormData;
  onFormDataChange: (data: QuestionFormData) => void;
  onSubmit: () => void;
  isSubmitting: boolean;
  loading: boolean;
}

export function DegreeQuestionForm({
  faculties,
  programs,
  degrees,
  selectedFaculty,
  selectedProgram,
  selectedDegree,
  onFacultyChange,
  onProgramChange,
  onDegreeChange,
  formData,
  onFormDataChange,
  onSubmit,
  isSubmitting,
  loading,
}: DegreeQuestionFormProps) {
  const isValid =
    selectedFaculty &&
    selectedDegree &&
    formData.subject.trim() &&
    formData.question.trim();

  return (
    <div className="space-y-4">
      <h3 className="text-lg font-semibold text-gray-800">Degree Question</h3>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="space-y-2">
          <Label htmlFor="degree-faculty-select">Select Faculty</Label>
          <Select value={selectedFaculty} onValueChange={onFacultyChange}>
            <SelectTrigger id="degree-faculty-select" className="h-11">
              <SelectValue placeholder="Choose a faculty" />
            </SelectTrigger>
            <SelectContent>
              {faculties.map((faculty) => (
                <SelectItem
                  key={faculty.faculty_code}
                  value={faculty.faculty_code}
                >
                  {faculty.faculty_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="space-y-2">
          <Label htmlFor="degree-program-select">Select Program</Label>
          <Select
            value={selectedProgram}
            onValueChange={onProgramChange}
            disabled={!selectedFaculty || loading}
          >
            <SelectTrigger id="degree-program-select" className="h-11">
              <SelectValue placeholder="Choose a program" />
            </SelectTrigger>
            <SelectContent>
              {programs.map((program) => (
                <SelectItem
                  key={program.program_code}
                  value={program.program_code}
                >
                  {program.program_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="space-y-2">
          <Label htmlFor="final-degree-select">Select Degree</Label>
          <Select
            value={selectedDegree}
            onValueChange={onDegreeChange}
            disabled={!selectedFaculty || loading}
          >
            <SelectTrigger id="final-degree-select" className="h-11">
              <SelectValue placeholder="Choose a degree" />
            </SelectTrigger>
            <SelectContent>
              {degrees.map((degree) => (
                <SelectItem key={degree.degree_code} value={degree.degree_code}>
                  {degree.degree_name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      <QuestionFormFields
        subject={formData.subject}
        question={formData.question}
        onSubjectChange={(value) =>
          onFormDataChange({ ...formData, subject: value })
        }
        onQuestionChange={(value) =>
          onFormDataChange({ ...formData, question: value })
        }
        onSubmit={onSubmit}
        isSubmitting={isSubmitting}
        isValid={!!isValid}
        submitButtonText="Submit Degree Question"
        questionPlaceholder="Ask your question about this degree..."
      />
    </div>
  );
}
