"use client";

import { useEffect, useState } from "react";
import { Button } from "@/components/ui/button";
import ErrorPage from "@/components/ErrorPage";
import { Card, CardContent } from "@/components/ui/card";
import { toast, Toaster } from "sonner";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Users,
  Building2,
  BookOpen,
  Award,
  NotebookPen,
  ChevronDown,
} from "lucide-react";

// Import hooks
import { useAskAdvisorData } from "@/app/hooks/userAskAdvisorData";
import { useAskAdvisorForms } from "@/app/hooks/useAskAdvisorForms";

// Import form components
import { FacultyQuestionForm } from "@/components/ask-advisor/forms/FacultyQuestionForm";
import { DepartmentQuestionForm } from "@/components/ask-advisor/forms/DepartmentQuestionForm";
import { ProgramQuestionForm } from "@/components/ask-advisor/forms/ProgramQuestionForm";
import { DegreeQuestionForm } from "@/components/ask-advisor/forms/DegreeQuestionForm";
import { MajorQuestionForm } from "@/components/ask-advisor/forms/MajorQuestionForm";

// Import types
import type { Faculty } from "@/components/ask-advisor/types";

interface AskAdvisorClientProps {
  initialFaculties: Faculty[];
}

const tabOptions = [
  { value: "faculty", label: "Faculty", icon: Users },
  { value: "department", label: "Department", icon: Building2 },
  { value: "program", label: "Program", icon: BookOpen },
  { value: "degree", label: "Degree", icon: Award },
  { value: "major", label: "Major", icon: NotebookPen },
];

export function AskAdvisorClient({ initialFaculties }: AskAdvisorClientProps) {
  const [activeTab, setActiveTab] = useState("faculty");

  // Data management hook
  const {
    faculties,
    programs,
    degrees,
    majors,
    departments,
    loading,
    error,
    loadPrograms,
    loadDegreesData,
    loadProgramDegrees,
    loadMajors,
    loadProgramMajors, // ADD THIS
    clearPrograms,
    clearDegrees,
    clearMajors,
  } = useAskAdvisorData({ initialFaculties });

  // Form management hook
  const {
    isSubmitting,
    facultySelection,
    setFacultySelection,
    departmentSelection,
    setDepartmentSelection,
    programFacultySelection,
    programStudySelection,
    setProgramStudySelection,
    degreeFacultySelection,
    degreeProgramSelection,
    finalDegreeSelection,
    setFinalDegreeSelection,
    majorFacultySelection,
    majorDepartmentSelection, // This will now represent the program selection for majors
    finalMajorSelection,
    setFinalMajorSelection,
    facultyFormData,
    setFacultyFormData,
    departmentFormData,
    setDepartmentFormData,
    programFormData,
    setProgramFormData,
    degreeFormData,
    setDegreeFormData,
    majorFormData,
    setMajorFormData,
    handleProgramFacultyChange,
    handleDegreeFacultyChange,
    handleDegreeProgramChange,
    handleMajorFacultyChange,
    handleMajorProgramChange, // CHANGED FROM handleMajorDepartmentChange
    handleSubmit,
  } = useAskAdvisorForms({
    faculties,
    programs,
    degrees,
    majors,
    departments,
    onDataLoad: {
      loadPrograms,
      loadDegreesData,
      loadProgramDegrees,
      loadMajors,
      loadProgramMajors, // ADD THIS
      clearPrograms,
      clearDegrees,
      clearMajors,
    },
  });

  // Show error page if there's a critical error
  if (error && !faculties.length) {
    return (
      <div className="w-full h-screen flex items-center justify-center">
        <ErrorPage />
      </div>
    );
  }

  const handleTabChange = (value: string) => {
    setActiveTab(value);
  };

  return (
    <>
      <Toaster position="top-center" richColors expand={false} />

      {/* Header Section */}
      <div className="w-full flex flex-col items-center justify-center text-center mt-24 md:mt-16 z-10">
        <div className="text-5xl md:text-8xl outfit-400 mb-1 mt-3">
          Ask An <span className="text-uct-blue">Advisor</span>
        </div>

        <div className="max-w-2xl text-sm md:text-base text-gray-700 outfit-300 mt-5 px-8 md:px-0">
          Explore everything you need to know about UCT courses in one place.
          Search by name, filter by faculty, and find key details like
          prerequisites, credits, and course summaries—all made simple and easy
          to understand.
        </div>
      </div>

      <div className="w-full max-w-6xl mx-auto mb-10 h-auto px-6 md:px-0">
        {/* Main Form Card */}
        <Card className="drop-shadow-sm border-2 border-[#222222]/40 mt-5 rounded-3xl">
          <CardContent className="p-8">
            <Tabs
              value={activeTab}
              onValueChange={handleTabChange}
              className="w-full"
            >
              {/* Desktop Tabs */}
              <TabsList className="hidden md:grid w-full grid-cols-5 mb-8">
                {tabOptions.map(({ value, label, icon: Icon }) => (
                  <TabsTrigger
                    key={value}
                    value={value}
                    className="flex items-center gap-2"
                  >
                    <Icon className="h-4 w-4" />
                    {label}
                  </TabsTrigger>
                ))}
              </TabsList>

              {/* Mobile Dropdown */}
              <div className="md:hidden mb-8">
                <h3 className="text-lg font-medium text-gray-900 mb-4">
                  Select Question Type
                </h3>
                <Select value={activeTab} onValueChange={handleTabChange}>
                  <SelectTrigger className="w-full">
                    <SelectValue>
                      {(() => {
                        const currentTab = tabOptions.find(
                          (tab) => tab.value === activeTab
                        );
                        const Icon = currentTab?.icon || Users;
                        return (
                          <div className="flex items-center gap-2">
                            <Icon className="h-4 w-4" />
                            {currentTab?.label}
                          </div>
                        );
                      })()}
                    </SelectValue>
                  </SelectTrigger>
                  <SelectContent>
                    {tabOptions.map(({ value, label, icon: Icon }) => (
                      <SelectItem key={value} value={value}>
                        <div className="flex items-center gap-2">
                          <Icon className="h-4 w-4" />
                          {label}
                        </div>
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <TabsContent value="faculty" className="space-y-6">
                <FacultyQuestionForm
                  faculties={faculties}
                  selectedFaculty={facultySelection}
                  onFacultyChange={setFacultySelection}
                  formData={facultyFormData}
                  onFormDataChange={setFacultyFormData}
                  onSubmit={() => handleSubmit("faculty")}
                  isSubmitting={isSubmitting}
                />
              </TabsContent>

              <TabsContent value="department" className="space-y-6">
                <DepartmentQuestionForm
                  departments={departments}
                  selectedDepartment={departmentSelection}
                  onDepartmentChange={setDepartmentSelection}
                  formData={departmentFormData}
                  onFormDataChange={setDepartmentFormData}
                  onSubmit={() => handleSubmit("department")}
                  isSubmitting={isSubmitting}
                />
              </TabsContent>

              <TabsContent value="program" className="space-y-6">
                <ProgramQuestionForm
                  faculties={faculties}
                  programs={programs}
                  selectedFaculty={programFacultySelection}
                  selectedProgram={programStudySelection}
                  onFacultyChange={handleProgramFacultyChange}
                  onProgramChange={setProgramStudySelection}
                  formData={programFormData}
                  onFormDataChange={setProgramFormData}
                  onSubmit={() => handleSubmit("program")}
                  isSubmitting={isSubmitting}
                  loading={loading}
                />
              </TabsContent>

              <TabsContent value="degree" className="space-y-6">
                <DegreeQuestionForm
                  faculties={faculties}
                  programs={programs}
                  degrees={degrees}
                  selectedFaculty={degreeFacultySelection}
                  selectedProgram={degreeProgramSelection}
                  selectedDegree={finalDegreeSelection}
                  onFacultyChange={handleDegreeFacultyChange}
                  onProgramChange={handleDegreeProgramChange}
                  onDegreeChange={setFinalDegreeSelection}
                  formData={degreeFormData}
                  onFormDataChange={setDegreeFormData}
                  onSubmit={() => handleSubmit("degree")}
                  isSubmitting={isSubmitting}
                  loading={loading}
                />
              </TabsContent>

              <TabsContent value="major" className="space-y-6">
                <MajorQuestionForm
                  faculties={faculties}
                  programs={programs} // Make sure this is programs, not departments
                  majors={majors}
                  selectedFaculty={majorFacultySelection}
                  selectedProgram={majorDepartmentSelection} // This state holds the program selection
                  selectedMajor={finalMajorSelection}
                  onFacultyChange={handleMajorFacultyChange}
                  onProgramChange={handleMajorProgramChange} // Make sure this is the right handler
                  onMajorChange={setFinalMajorSelection}
                  formData={majorFormData}
                  onFormDataChange={setMajorFormData}
                  onSubmit={() => handleSubmit("major")}
                  isSubmitting={isSubmitting}
                  loading={loading}
                />
              </TabsContent>
            </Tabs>
          </CardContent>
        </Card>
      </div>
    </>
  );
}
