"use client";

import React, {
  useRef,
  useState,
  useCallback,
  useMemo,
  useEffect,
} from "react";
import { Input } from "@/components/ui/input";
import {
  ArrowUpDown,
  ChevronLeft,
  ChevronRight,
  ChevronDown,
  ChevronUp,
  Home,
  Users,
  Component,
} from "lucide-react";
import {
  type ColumnDef,
  flexRender,
  getCoreRowModel,
  useReactTable,
  getFilteredRowModel,
  getSortedRowModel,
  type SortingState,
} from "@tanstack/react-table";

import {
  Dialog,
  DialogContent,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";

import {
  NavigationMenuLink,
  NavigationMenu,
  NavigationMenuList,
  NavigationMenuItem,
} from "@/components/ui/navigation-menu";

import { Button } from "@/components/ui/button";

import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";

// Types
type ProgramItem = {
  program_name: string;
  program_code: string;
};

type DetailedCourse = {
  code: string;
  name: string;
  credits: number;
  level: number;
  semester: string;
  department?: string;
  convener?: string;
  course_outline?: string;
  entry_requirements?: string;
  dp_requirements?: string;
  assessment?: string;
  corequisites?: string;
  objective?: string;
  lecture_times?: string;
  notes?: string;
  type: string;
};

type CourseGroup = {
  required_count: number;
  courses?: DetailedCourse[];
  groups?: CourseGroup[];
};

type YearItem = DetailedCourse | CourseGroup;

type YearData = {
  year: number;
  items: YearItem[];
};

type ProgramData = {
  degree_code?: string;
  degree_name?: string;
  major_code?: string;
  major_name?: string;
  notes?: string;
  courses?: YearData[];
} & (
  | {
      degree_code: string;
      degree_name: string;
      major_code?: never;
      major_name?: never;
    }
  | {
      major_code: string;
      major_name: string;
      degree_code?: never;
      degree_name?: never;
    }
);

interface ProgramClientContentProps {
  programs: ProgramData[];
  isScienceFaculty?: boolean;
}

// ListItem helper component
const ListItem = ({
  className,
  title,
  children,
  ...props
}: React.ComponentPropsWithoutRef<"a"> & { title: string }) => (
  <li>
    <NavigationMenuLink asChild>
      <a
        className={
          "block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors hover:bg-accent hover:text-accent-foreground focus:bg-accent focus:text-accent-foreground " +
          (className || "")
        }
        {...props}
      >
        <div className="text-sm font-medium leading-none">{title}</div>
        <p className="line-clamp-2 text-sm leading-snug text-muted-foreground">
          {children}
        </p>
      </a>
    </NavigationMenuLink>
  </li>
);

// Custom Collapsible Sidebar Component
interface CustomSidebarProps {
  collapsed: boolean;
  onToggle: () => void;
  activeTab: string;
  onTabChange: (tab: string) => void;
  selectedProgram: ProgramData | null;
  isScienceFaculty: boolean;
}

const CustomSidebar: React.FC<CustomSidebarProps> = React.memo(
  ({
    collapsed,
    onToggle,
    activeTab,
    onTabChange,
    selectedProgram,
    isScienceFaculty,
  }) => {
    const menuItems = useMemo(
      () => [
        { id: "overview", icon: Home, label: "Course Details" },
        {
          id: "details",
          icon: Users,
          label: isScienceFaculty ? "Major Details" : "Degree Details",
        },
      ],
      [isScienceFaculty]
    );

    return (
      <div
        className={`bg-gray-50 text-black transition-all duration-300 ease-in-out border-r border-gray-300 flex-shrink-0 ${
          collapsed ? "w-12" : "w-64"
        } flex-col rounded-l-lg hidden md:flex`}
      >
        {/* Sidebar Header */}
        <div
          className={`border-gray-300 flex min-h-[64px] border-b
    ${
      collapsed
        ? "justify-center items-center p-0"
        : "items-center justify-between p-4"
    }
  `}
        >
          {!collapsed && (
            <span className="flex flex-col items-start">
              <span className="font-semibold text-lg">
                {selectedProgram
                  ? isScienceFaculty
                    ? selectedProgram.major_code
                    : selectedProgram.degree_code
                  : isScienceFaculty
                  ? "Major"
                  : "Degree"}
              </span>
              <span className="text-sm font-normal text-gray-700">
                {selectedProgram
                  ? isScienceFaculty
                    ? selectedProgram.major_name
                    : selectedProgram.degree_name
                  : isScienceFaculty
                  ? "Select a Major"
                  : "Select a Degree"}
              </span>
            </span>
          )}
          <button
            onClick={onToggle}
            className="p-1 hover:bg-gray-200 rounded flex-shrink-0"
          >
            {collapsed ? (
              <ChevronRight className="w-4 h-4" />
            ) : (
              <ChevronLeft className="w-4 h-4" />
            )}
          </button>
        </div>

        {/* Navigation Items */}
        <nav className="flex-1 p-2">
          <ul className="space-y-1">
            {menuItems.map((item) => {
              const Icon = item.icon;
              return (
                <li key={item.id}>
                  <button
                    onClick={() => onTabChange(item.id)}
                    className={`w-full flex items-center rounded-lg transition-colors
      ${
        activeTab === item.id
          ? "bg-blue-100 text-blue-700"
          : "text-black hover:bg-gray-200"
      }
      ${collapsed ? "justify-center p-3" : "space-x-3 p-3 "}
    `}
                    title={collapsed ? item.label : ""}
                  >
                    <Icon className="w-4 h-4 flex-shrink-0 text-black" />
                    {!collapsed && (
                      <span className="truncate text-sm">{item.label}</span>
                    )}
                  </button>
                </li>
              );
            })}
          </ul>
        </nav>
      </div>
    );
  }
);

function DataTableDemo({
  programs,
  isScienceFaculty = false,
}: ProgramClientContentProps) {
  // Transform programs data for table display - memoized
  const tableData: ProgramItem[] = useMemo(
    () =>
      programs.map((program) => ({
        program_name: isScienceFaculty
          ? program.major_name!
          : program.degree_name!,
        program_code: isScienceFaculty
          ? program.major_code!
          : program.degree_code!,
      })),
    [programs, isScienceFaculty]
  );

  const [globalFilter, setGlobalFilter] = useState("");
  const [sorting, setSorting] = useState<SortingState>([]);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [selectedProgram, setSelectedProgram] = useState<ProgramData | null>(
    null
  );
  const [sidebarCollapsed, setSidebarCollapsed] = useState(false);
  const [activeTab, setActiveTab] = useState("overview");
  const [openAccordion, setOpenAccordion] = useState<string | undefined>(
    undefined
  );
  const accordionRefs = useRef<{ [key: string]: HTMLDivElement | null }>({});
  const [expandedCourse, setExpandedCourse] = useState<string | null>(null);
  const [activeYear, setActiveYear] = useState(0);
  const courseCardRefs = useRef<{ [key: string]: HTMLDivElement | null }>({});

  // Table columns - memoized to prevent recreation
  const columns: ColumnDef<ProgramItem>[] = [
    {
      accessorKey: "program_name",
      header: ({ column }) => (
        <div className="flex items-center gap-2">
          {isScienceFaculty ? "Major Name" : "Degree Name"}
          <button
            type="button"
            onClick={() => column.toggleSorting(column.getIsSorted() === "asc")}
            className="ml-2"
          >
            <ArrowUpDown className="h-4 w-4" />
          </button>
        </div>
      ),
      enableGlobalFilter: true,
      enableSorting: true,
    },
    {
      accessorKey: "program_code",
      header: isScienceFaculty ? "Major Code" : "Degree Code",
      enableGlobalFilter: true,
    },
  ];

  // Memoized helper functions
  const getCoursesForYear = useCallback(
    (yearNumber: number): YearItem[] => {
      if (!selectedProgram || !selectedProgram.courses) return [];

      const yearData = selectedProgram.courses.find(
        (year) => year.year === yearNumber
      );
      return yearData ? yearData.items : [];
    },
    [selectedProgram]
  );

  // Update getAvailableYears
  const getAvailableYears = useCallback((): number[] => {
    if (!selectedProgram || !selectedProgram.courses) return [];
    return selectedProgram.courses
      .map((year) => year.year)
      .sort((a, b) => a - b);
  }, [selectedProgram]);

  // Scroll to opened accordion item
  useEffect(() => {
    if (openAccordion && accordionRefs.current[openAccordion]) {
      const timeoutId = setTimeout(() => {
        accordionRefs.current[openAccordion]?.scrollIntoView({
          behavior: "smooth",
          block: "nearest",
        });
      }, 180);

      return () => clearTimeout(timeoutId);
    }
  }, [openAccordion]);

  // Scroll to expanded course card
  useEffect(() => {
    if (expandedCourse && courseCardRefs.current[expandedCourse]) {
      const timeoutId = setTimeout(() => {
        courseCardRefs.current[expandedCourse]?.scrollIntoView({
          behavior: "smooth",
          block: "center",
        });
      }, 100);

      return () => clearTimeout(timeoutId);
    }
  }, [expandedCourse]);

  const table = useReactTable({
    data: tableData,
    columns,
    getCoreRowModel: getCoreRowModel(),
    getFilteredRowModel: getFilteredRowModel(),
    getSortedRowModel: getSortedRowModel(),
    state: {
      globalFilter,
      sorting,
    },
    onGlobalFilterChange: setGlobalFilter,
    onSortingChange: setSorting,
    globalFilterFn: "includesString",
  });

  // Memoized render function for course groups
  const renderCourseGroup = useCallback(
    (item: YearItem, depth = 0): React.ReactNode => {
      const indentClass =
        depth > 0 ? `pl-${depth * 2} border-l-2 border-blue-500` : "";

      // Regular course
      if ("code" in item) {
        const course = item as DetailedCourse;
        const isExpanded = expandedCourse === course.code;

        return (
          <div
            key={course.code}
            ref={(el) => {
              courseCardRefs.current[course.code] = el;
            }}
            className={`bg-white rounded-lg border border-gray-200 shadow-sm transition-colors ${indentClass}`}
          >
            <div
              className="p-3 hover:bg-gray-50 cursor-pointer"
              onClick={() => setExpandedCourse(isExpanded ? null : course.code)}
            >
              <div className="flex items-start justify-between">
                <div className="flex-1 min-w-0">
                  <h4 className="text-sm font-medium text-gray-900 mb-2">
                    {course.name}
                  </h4>
                  <div className="flex items-center flex-wrap gap-x-4 gap-y-1 text-xs">
                    <div className="flex items-center space-x-1">
                      <span className="text-gray-500">Code:</span>
                      <span className="text-gray-900 font-medium">
                        {course.code}
                      </span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <span className="text-gray-500">Credits:</span>
                      <span className="text-gray-900 font-medium">
                        {course.credits}
                      </span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <span className="text-gray-500">Level:</span>
                      <span className="text-gray-900 font-medium">
                        {course.level}
                      </span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <span className="text-gray-500">Semester:</span>
                      <span className="text-gray-900 font-medium">
                        {course.semester}
                      </span>
                    </div>
                  </div>
                </div>
                <div className="flex-shrink-0 ml-3">
                  {isExpanded ? (
                    <ChevronUp className="h-6 w-6 text-gray-400" />
                  ) : (
                    <ChevronDown className="h-6 w-6 text-gray-400" />
                  )}
                </div>
              </div>
            </div>

            {/* Expanded course details */}
            {isExpanded && (
              <div className="px-3 pb-3 border-t border-gray-200 mt-2 pt-3">
                <div className="space-y-3 text-xs">
                  {course.department && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Department:
                      </span>
                      <span className="text-gray-900 ml-2">
                        {course.department}
                      </span>
                    </div>
                  )}
                  {course.convener && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Convener:
                      </span>
                      <span className="text-gray-900 ml-2">
                        {course.convener}
                      </span>
                    </div>
                  )}
                  {course.objective && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Objective:
                      </span>
                      <p className="text-gray-900 mt-1">{course.objective}</p>
                    </div>
                  )}
                  {course.course_outline && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Course outline:
                      </span>
                      <p className="text-gray-900 mt-1 whitespace-pre-line">
                        {course.course_outline}
                      </p>
                    </div>
                  )}
                  {course.lecture_times && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Lecture times:
                      </span>
                      <p className="text-gray-900 mt-1">
                        {course.lecture_times}
                      </p>
                    </div>
                  )}
                  {course.entry_requirements && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Entry requirements:
                      </span>
                      <p className="text-gray-900 mt-1">
                        {course.entry_requirements}
                      </p>
                    </div>
                  )}
                  {course.corequisites && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Corequisites:
                      </span>
                      <p className="text-gray-900 mt-1">
                        {course.corequisites}
                      </p>
                    </div>
                  )}
                  {course.dp_requirements && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        DP requirements:
                      </span>
                      <p className="text-gray-900 mt-1">
                        {course.dp_requirements}
                      </p>
                    </div>
                  )}
                  {course.assessment && (
                    <div>
                      <span className="text-gray-500 font-medium">
                        Assessment:
                      </span>
                      <p className="text-gray-900 mt-1 whitespace-pre-line">
                        {course.assessment}
                      </p>
                    </div>
                  )}
                  {course.notes && (
                    <div>
                      <span className="text-gray-500 font-medium">Notes:</span>
                      <p className="text-gray-900 mt-1">{course.notes}</p>
                    </div>
                  )}
                </div>
              </div>
            )}
          </div>
        );
      }

      // Group with required_count
      if ("required_count" in item) {
        const group = item as CourseGroup;
        const requirementText =
          group.required_count === 100
            ? "All required"
            : `Choose ${group.required_count}`;

        return (
          <div
            key={`group-${Math.random()}`}
            className={`space-y-3 ${indentClass}`}
          >
            <div className="bg-blue-50 rounded-lg border-2 border-blue-200 shadow-sm">
              <div className="bg-blue-100 rounded-t-lg px-4 py-2 border-b border-blue-200">
                <div className="flex items-center space-x-2">
                  <Component className="h-4 w-4 text-blue-600" />
                  <h4 className="text-sm font-semibold text-blue-900">
                    {requirementText}
                  </h4>
                </div>
              </div>
              <div className="p-3">
                {/* Render direct courses */}
                {group.courses && (
                  <div className="space-y-2">
                    {group.courses.map((course, idx) => (
                      <div key={`course-${course.code || idx}`}>
                        {renderCourseGroup(course, depth + 1)}
                      </div>
                    ))}
                  </div>
                )}

                {/* Render nested groups */}
                {group.groups && (
                  <div className="space-y-2 mt-3">
                    {group.groups.map((subGroup, idx) => (
                      <div key={`subgroup-${idx}`}>
                        {renderCourseGroup(subGroup, depth + 1)}
                      </div>
                    ))}
                  </div>
                )}
              </div>
            </div>
          </div>
        );
      }

      return null;
    },
    [expandedCourse]
  );

  // Mobile Tab Navigation Component (memoized)
  const MobileTabNavigation = React.memo(() => {
    const menuItems = [
      { id: "overview", label: "Course Details" },
      {
        id: "details",
        label: isScienceFaculty ? "Major Details" : "Degree Details",
      },
    ];

    return (
      <div className="md:hidden border-b border-gray-200 mb-4">
        <div className="flex overflow-x-auto">
          {menuItems.map((item) => (
            <button
              key={item.id}
              onClick={() => setActiveTab(item.id)}
              className={`flex-shrink-0 px-4 py-3 text-sm font-medium border-b-2 transition-colors ${
                activeTab === item.id
                  ? "border-blue-500 text-blue-600"
                  : "border-transparent text-gray-500 hover:text-gray-700"
              }`}
            >
              {item.label}
            </button>
          ))}
        </div>
      </div>
    );
  });

  // Mobile Course Card Component (memoized)
  const MobileCourseCard = React.memo(({ course }: { course: YearItem }) => {
    // Regular course
    if ("code" in course) {
      const detailedCourse = course as DetailedCourse;
      const isExpanded = expandedCourse === detailedCourse.code;

      return (
        <div
          className="bg-white rounded-lg border border-gray-200 shadow-sm mx-auto max-w-sm"
          ref={(el) => {
            courseCardRefs.current[detailedCourse.code] = el;
          }}
        >
          <div
            className="p-4 hover:bg-gray-50 cursor-pointer transition-colors"
            onClick={() =>
              setExpandedCourse(isExpanded ? null : detailedCourse.code)
            }
          >
            <div className="space-y-3">
              <div className="flex items-start justify-between">
                <div className="flex-1 min-w-0">
                  <h3 className="text-base font-semibold text-gray-900 leading-tight">
                    {detailedCourse.name}
                  </h3>
                </div>
                <div className="flex-shrink-0 ml-3">
                  {isExpanded ? (
                    <ChevronUp className="h-6 w-6 text-gray-400" />
                  ) : (
                    <ChevronDown className="h-6 w-6 text-gray-400" />
                  )}
                </div>
              </div>
              <div className="flex flex-wrap items-center gap-x-4 gap-y-1 text-sm">
                <div className="flex items-center space-x-1">
                  <span className="text-gray-500 font-medium">Code:</span>
                  <span className="text-gray-900 font-medium">
                    {detailedCourse.code}
                  </span>
                </div>
                <div className="flex items-center space-x-1">
                  <span className="text-gray-500 font-medium">Credits:</span>
                  <span className="text-gray-900 font-medium">
                    {detailedCourse.credits}
                  </span>
                </div>
                <div className="flex items-center space-x-1">
                  <span className="text-gray-500 font-medium">Level:</span>
                  <span className="text-gray-900 font-medium">
                    {detailedCourse.level}
                  </span>
                </div>
                <div className="flex items-center space-x-1">
                  <span className="text-gray-500 font-medium">Semester:</span>
                  <span className="text-gray-900 font-medium">
                    {detailedCourse.semester}
                  </span>
                </div>
              </div>
            </div>
          </div>

          {/* Expanded course details */}
          {isExpanded && (
            <div className="px-4 pb-4 border-t border-gray-200">
              <div className="space-y-3 text-sm pt-3">
                {detailedCourse.department && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      Department:
                    </span>
                    <span className="text-gray-900 ml-2">
                      {detailedCourse.department}
                    </span>
                  </div>
                )}
                {detailedCourse.convener && (
                  <div>
                    <span className="text-gray-500 font-medium">Convener:</span>
                    <span className="text-gray-900 ml-2">
                      {detailedCourse.convener}
                    </span>
                  </div>
                )}
                {detailedCourse.objective && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      Objective:
                    </span>
                    <p className="text-gray-900 mt-1">
                      {detailedCourse.objective}
                    </p>
                  </div>
                )}
                {detailedCourse.course_outline && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      Course outline:
                    </span>
                    <p className="text-gray-900 mt-1 whitespace-pre-line">
                      {detailedCourse.course_outline}
                    </p>
                  </div>
                )}
                {detailedCourse.lecture_times && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      Lecture times:
                    </span>
                    <p className="text-gray-900 mt-1">
                      {detailedCourse.lecture_times}
                    </p>
                  </div>
                )}
                {detailedCourse.entry_requirements && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      Entry requirements:
                    </span>
                    <p className="text-gray-900 mt-1">
                      {detailedCourse.entry_requirements}
                    </p>
                  </div>
                )}
                {detailedCourse.corequisites && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      Corequisites:
                    </span>
                    <p className="text-gray-900 mt-1">
                      {detailedCourse.corequisites}
                    </p>
                  </div>
                )}
                {detailedCourse.dp_requirements && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      DP requirements:
                    </span>
                    <p className="text-gray-900 mt-1">
                      {detailedCourse.dp_requirements}
                    </p>
                  </div>
                )}
                {detailedCourse.assessment && (
                  <div>
                    <span className="text-gray-500 font-medium">
                      Assessment:
                    </span>
                    <p className="text-gray-900 mt-1 whitespace-pre-line">
                      {detailedCourse.assessment}
                    </p>
                  </div>
                )}
                {detailedCourse.notes && (
                  <div>
                    <span className="text-gray-500 font-medium">Notes:</span>
                    <p className="text-gray-900 mt-1">{detailedCourse.notes}</p>
                  </div>
                )}
              </div>
            </div>
          )}
        </div>
      );
    }

    // Group with required_count
    if ("required_count" in course) {
      const group = course as CourseGroup;
      const requirementText =
        group.required_count === 100
          ? "All courses required"
          : `Choose ${group.required_count}`;

      return (
        <div className="bg-blue-50 rounded-lg border-2 border-blue-200 shadow-sm mx-auto max-w-sm">
          <div className="bg-blue-100 rounded-t-lg px-4 py-3 border-b border-blue-200">
            <div className="flex items-center space-x-2">
              <Component className="h-4 w-4 text-blue-600" />
              <h3 className="text-base font-semibold text-blue-900 leading-tight">
                {requirementText}
              </h3>
            </div>
          </div>
          <div className="p-4">
            {/* Render courses and nested groups */}
            <div className="space-y-2">
              {group.courses &&
                group.courses.map((subCourse, idx) => (
                  <div key={`course-${subCourse.code || idx}`} className="pl-2">
                    {renderCourseGroup(subCourse, 1)}
                  </div>
                ))}
              {group.groups &&
                group.groups.map((subGroup, idx) => (
                  <div key={idx} className="pl-2">
                    {renderCourseGroup(subGroup, 1)}
                  </div>
                ))}
            </div>
          </div>
        </div>
      );
    }

    return null;
  });

  // Mobile Year Navigation (memoized)
  const MobileYearNavigation = React.memo(() => {
    const availableYears = getAvailableYears();

    return (
      <div className="md:hidden mb-6">
        <div className="flex overflow-x-auto pb-2">
          {availableYears.map((year) => (
            <button
              key={year}
              onClick={() => setActiveYear(year - 1)} // Convert to 0-based index
              className={`flex-shrink-0 px-4 py-2 mx-1 text-sm font-medium rounded-full transition-colors ${
                activeYear === year - 1
                  ? "bg-blue-100 text-blue-700"
                  : "bg-gray-100 text-gray-700 hover:bg-gray-200"
              }`}
            >
              {year === 1
                ? "1st Yr"
                : year === 2
                ? "2nd Yr"
                : year === 3
                ? "3rd Yr"
                : `${year}th Yr`}
            </button>
          ))}
        </div>
      </div>
    );
  });

  // Memoized tab content renderer
  const renderTabContent = useCallback(() => {
    if (!selectedProgram)
      return <div>No {isScienceFaculty ? "major" : "degree"} selected.</div>;

    const availableYears = getAvailableYears();

    switch (activeTab) {
      case "overview":
        if (!selectedProgram?.courses || selectedProgram.courses.length === 0) {
          return (
            <div className="w-full h-full flex items-center justify-center">
              <div className="text-gray-500 text-center">
                No course information available for this{" "}
                {isScienceFaculty ? "major" : "degree"}.
              </div>
            </div>
          );
        }
        return (
          <div className="space-y-6">
            {/* Desktop Navigation */}
            <div className="hidden md:block">
              <div className="hidden md:flex w-full justify-center items-center my-4">
                <NavigationMenu>
                  <NavigationMenuList className="gap-6">
                    {availableYears.map((year, idx) => (
                      <NavigationMenuItem key={year}>
                        <NavigationMenuLink asChild active={activeYear === idx}>
                          <button
                            onClick={() => setActiveYear(idx)}
                            className={`px-6 py-0 !rounded-none font-semibold transition-colors border-b
                        ${
                          activeYear === idx
                            ? "border-black text-black"
                            : "border-transparent text-gray-700 hover:text-black"
                        }
                      `}
                            style={{ background: "none" }}
                          >
                            {year === 1
                              ? "First Year"
                              : year === 2
                              ? "Second Year"
                              : year === 3
                              ? "Third Year"
                              : year === 4
                              ? "Fourth Year"
                              : `Year ${year}`}
                          </button>
                        </NavigationMenuLink>
                      </NavigationMenuItem>
                    ))}
                  </NavigationMenuList>
                </NavigationMenu>
              </div>
            </div>

            {/* Mobile Navigation */}
            <MobileYearNavigation />

            {/* Desktop Table View */}
            <div className="hidden md:block max-h-[70vh] md:max-h-[60vh] overflow-y-auto pr-4">
              <div className="space-y-4">
                {getCoursesForYear(availableYears[activeYear]).map(
                  (courseItem, i) => (
                    <React.Fragment key={`course-item-${i}`}>
                      {renderCourseGroup(courseItem)}
                    </React.Fragment>
                  )
                )}
              </div>
            </div>

            {/* Mobile Card View */}
            <div className="md:hidden max-h-[80vh] overflow-y-auto">
              <div className="space-y-3 overflow-x-hidden">
                {getCoursesForYear(availableYears[activeYear]).map(
                  (course, i) => (
                    <MobileCourseCard
                      key={`mobile-course-${i}`}
                      course={course}
                    />
                  )
                )}
              </div>
            </div>
          </div>
        );

      case "details":
        return (
          <div className="w-full h-full flex flex-col items-center space-y-4">
            <div className="md:text-lg w-40 font-semibold border-b border-black pb-2 text-center mt-8">
              {isScienceFaculty ? "Major Details" : "Degree Details"}
            </div>
            <div className="max-w-2xl text-sm md:text-base text-gray-800 text-center px-4">
              {selectedProgram.notes ||
                `No additional ${
                  isScienceFaculty ? "major" : "degree"
                } information available.`}
            </div>
          </div>
        );
      default:
        return <div>Select a tab</div>;
    }
  }, [
    selectedProgram,
    activeTab,
    activeYear,
    getAvailableYears,
    getCoursesForYear,
    renderCourseGroup,
    isScienceFaculty,
  ]);

  // Handle row click to select program and open dialog - memoized
  const handleRowClick = useCallback(
    (programCode: string) => {
      const program = programs.find((p) =>
        isScienceFaculty
          ? p.major_code === programCode
          : p.degree_code === programCode
      );
      if (program) {
        setSelectedProgram(program);
        setActiveTab("overview");
        setActiveYear(0);
        setDialogOpen(true);
      }
    },
    [programs, isScienceFaculty]
  );

  return (
    <div className="w-full max-w-4xl mx-auto mt-12 mb-10 z-10">
      <div className="flex items-center justify-center py-4 px-8 md:px-0">
        <Input
          placeholder={
            isScienceFaculty ? "Filter Majors..." : "Filter Degrees..."
          }
          value={globalFilter ?? ""}
          onChange={(e) => setGlobalFilter(e.target.value)}
          className="max-w-2xl border-2 rounded-full bg-white"
        />
      </div>
      <div>
        <div>
          {/* Desktop Table View */}
          <div className="hidden md:block rounded-xl border bg-white drop-shadow-lg">
            <Table>
              <TableHeader>
                {table.getHeaderGroups().map((headerGroup) => (
                  <TableRow key={headerGroup.id}>
                    {headerGroup.headers.map((header) => (
                      <TableHead
                        className="text-base text-gray-600 px-10"
                        key={header.id}
                      >
                        {header.isPlaceholder
                          ? null
                          : flexRender(
                              header.column.columnDef.header,
                              header.getContext()
                            )}
                      </TableHead>
                    ))}
                  </TableRow>
                ))}
              </TableHeader>
              <TableBody>
                {table.getRowModel().rows.length ? (
                  table.getRowModel().rows.map((row) => (
                    <TableRow
                      key={row.id}
                      className="h-12 cursor-pointer"
                      onClick={() => handleRowClick(row.original.program_code)}
                    >
                      {row.getVisibleCells().map((cell) => (
                        <TableCell
                          key={cell.id}
                          className="py-0 text-base font-medium px-10"
                        >
                          {flexRender(
                            cell.column.columnDef.cell,
                            cell.getContext()
                          )}
                        </TableCell>
                      ))}
                    </TableRow>
                  ))
                ) : (
                  <TableRow>
                    <TableCell
                      colSpan={columns.length}
                      className="h-24 text-center"
                    >
                      No results.
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>
          </div>

          {/* Mobile Card View */}
          <div className="md:hidden space-y-4">
            {table.getRowModel().rows.length ? (
              table.getRowModel().rows.map((row) => (
                <div
                  key={row.id}
                  className="bg-white rounded-lg border border-gray-200 p-4 mx-4 cursor-pointer hover:bg-gray-50 transition-colors shadow-sm"
                  onClick={() => handleRowClick(row.original.program_code)}
                >
                  <div className="space-y-3">
                    <div className="flex items-start justify-between">
                      <div className="flex-1 min-w-0">
                        <h3 className="text-base w-10/12 font-semibold text-gray-900 leading-tight">
                          {row.original.program_name}
                        </h3>
                      </div>
                      <div className="flex-shrink-0 ml-2">
                        <svg
                          className="w-5 h-5 text-gray-400"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M9 5l7 7-7 7"
                          />
                        </svg>
                      </div>
                    </div>

                    <div className="space-y-2 text-sm">
                      <div className="flex justify-between items-center py-1">
                        <span className="text-gray-500 font-medium">Code</span>
                        <span className="text-gray-900 font-medium">
                          {row.original.program_code}
                        </span>
                      </div>
                    </div>
                  </div>
                </div>
              ))
            ) : (
              <div className="text-center py-12">
                <p className="text-gray-500">No results found.</p>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Fixed Dialog with Custom Sidebar */}
      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="!max-w-full md:!max-w-[80%] h-full md:h-[80vh] p-0 rounded-lg overflow-hidden">
          <DialogTitle className="sr-only">
            {selectedProgram
              ? isScienceFaculty
                ? selectedProgram.major_name
                : selectedProgram.degree_name
              : "Program Details"}
          </DialogTitle>
          <div className="flex h-full">
            {/* Custom Collapsible Sidebar - Hidden on mobile */}
            <CustomSidebar
              collapsed={sidebarCollapsed}
              onToggle={() => setSidebarCollapsed(!sidebarCollapsed)}
              activeTab={activeTab}
              onTabChange={setActiveTab}
              selectedProgram={selectedProgram}
              isScienceFaculty={isScienceFaculty}
            />

            {/* Main Content Area */}
            <div className="flex-1 flex flex-col bg-white min-w-0">
              {/* Mobile Tab Navigation */}
              <MobileTabNavigation />

              {/* Content */}
              <div className="flex-1 overflow-y-auto p-6 md:p-6 pt-0 md:pt-6">
                {renderTabContent()}
              </div>

              {/* Sticky DialogFooter */}
              <DialogFooter className="px-4 py-4 md:sticky md:bottom-0 bg-white border-t justify-center z-10 hidden md:flex md:justify-end">
                <Button
                  className="w-fit"
                  onClick={() => setDialogOpen(false)}
                  type="button"
                >
                  Close
                </Button>
              </DialogFooter>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}

export function ProgramClientContent({
  programs,
  isScienceFaculty = false,
}: ProgramClientContentProps) {
  return (
    <DataTableDemo programs={programs} isScienceFaculty={isScienceFaculty} />
  );
}
