"use client";

import React, { useState, useEffect } from "react";
import Link from "next/link";
import Image from "next/image";
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
} from "@/components/ui/sheet";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import { Button } from "@/components/ui/button";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import { Menu, ChevronDown, ChevronRight, Inbox } from "lucide-react";
import { useAuth } from "@/components/AuthContext";
import { InboxWindow } from "@/components/inboxWindow";

// Props interface for the Navbar component
interface NavbarProps {
  hideOnScroll?: boolean; // Default true - navbar hides on scroll down
}

// Custom dropdown component to replace NavigationMenu
const DropdownMenu = ({
  trigger,
  children,
  isOpen,
  onToggle,
}: {
  trigger: React.ReactNode;
  children: React.ReactNode;
  isOpen: boolean;
  onToggle: () => void;
}) => {
  const dropdownRef = React.useRef<HTMLDivElement>(null);

  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        dropdownRef.current &&
        !dropdownRef.current.contains(event.target as Node)
      ) {
        if (isOpen) {
          onToggle();
        }
      }
    };

    document.addEventListener("mousedown", handleClickOutside);
    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [isOpen, onToggle]);

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        onClick={onToggle}
        className={`
          flex items-center gap-1 px-4 py-2 rounded-md transition-colors min-w-[140px] justify-center
          ${
            isOpen
              ? "bg-white text-black"
              : "bg-[#222222] text-white hover:bg-[#2b2b2b] hover:text-white"
          }
        `}
      >
        <span className="w-fit whitespace-nowrap">{trigger}</span>
        <ChevronDown
          className={`h-4 w-4 transition-transform duration-200 ${
            isOpen ? "rotate-180" : ""
          }`}
        />
      </button>

      {isOpen && (
        <div className="absolute top-full left-1/2 transform -translate-x-1/2 mt-3 z-50">
          <div className="bg-popover text-popover-foreground border rounded-md shadow-md min-w-[400px] md:min-w-[500px] lg:min-w-[600px]">
            {children}
          </div>
        </div>
      )}
    </div>
  );
};

// ListItem helper for desktop navigation
const ListItem = ({
  className,
  title,
  children,
  href,
  onClick,
  ...props
}: React.ComponentPropsWithoutRef<"a"> & {
  title: string;
  href: string;
  onClick?: () => void;
}) => (
  <li>
    <Link
      href={href}
      className={
        "block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors hover:bg-accent hover:text-accent-foreground focus:bg-accent focus:text-accent-foreground " +
        (className || "")
      }
      onClick={onClick}
      {...props}
    >
      <div className="text-sm font-medium leading-none">{title}</div>
      <p className="line-clamp-2 text-sm leading-snug text-muted-foreground">
        {children}
      </p>
    </Link>
  </li>
);

// Top-level mobile sidebar item (no indentation)
const TopLevelSidebarItem = ({
  href,
  children,
  onClick,
}: {
  href: string;
  children: React.ReactNode;
  onClick?: () => void;
}) => (
  <Link
    href={href}
    className="block px-3 py-2 text-sm  hover:bg-gray-100 rounded-md transition-colors"
    onClick={onClick}
  >
    {children}
  </Link>
);

// Mobile sidebar item component (with indentation for sub-items)
const SidebarItem = ({
  href,
  children,
  onClick,
}: {
  href: string;
  children: React.ReactNode;
  onClick?: () => void;
}) => (
  <div className="relative">
    <div className="absolute left-0 top-0 bottom-0 w-px bg-gray-300"></div>
    <div className="absolute left-0 top-1/2 w-3 h-px bg-gray-300"></div>
    <Link
      href={href}
      className="block ml-4 pl-2 py-2 text-sm hover:bg-gray-100 rounded-md transition-colors"
      onClick={onClick}
    >
      {children}
    </Link>
  </div>
);

// Collapsible section for mobile sidebar
const SidebarSection = ({
  title,
  children,
}: {
  title: string;
  children: React.ReactNode;
}) => {
  const [isOpen, setIsOpen] = useState(false);

  return (
    <Collapsible open={isOpen} onOpenChange={setIsOpen}>
      <CollapsibleTrigger asChild>
        <Button
          variant="ghost"
          className="w-full justify-between px-4 py-2 h-auto font-normal"
        >
          <span className="w-fit whitespace-nowrap">{title}</span>
          {isOpen ? (
            <ChevronDown className="h-4 w-4" />
          ) : (
            <ChevronRight className="h-4 w-4" />
          )}
        </Button>
      </CollapsibleTrigger>
      <CollapsibleContent className="space-y-1">{children}</CollapsibleContent>
    </Collapsible>
  );
};

const stud_components: { title: string; href: string; description: string }[] =
  [
    {
      title: "FAQ",
      href: "/student_help/faq",
      description: "Quick answers to common student and admission questions.",
    },
    {
      title: "Glossary",
      href: "/student_help/glossary",
      description: "Definitions of key university and academic terms.",
    },
    {
      title: "Advisors Info",
      href: "/student_help/advisor_info",
      description: "Learn about available advisors and how to contact them.",
    },
  ];

const degree_components: {
  title: string;
  href: string;
  description: string;
}[] = [
  {
    title: "Roadmap",
    href: "/degree_planning/roadmap",
    description: "Plan your degree journey step by step with a guided roadmap.",
  },
  {
    title: "Ask a Advisor",
    href: "/degree_planning/ask_advisor",
    description:
      "Chat directly with a virtual advisor for personalized guidance.",
  },
];

export function Navbar({ hideOnScroll = true }: NavbarProps) {
  const [isVisible, setIsVisible] = useState(true);
  const [lastScrollY, setLastScrollY] = useState(0);
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [inboxOpen, setInboxOpen] = useState(false);

  // State for dropdown menus
  const [openDropdown, setOpenDropdown] = useState<string | null>(null);

  // Use auth context properly
  const { isLoggedIn, user, logout, isLoading } = useAuth();

  // Calculate if there are unread messages from user data
  const hasUnreadMessages = user?.red_dot || false;

  // Calculate the count of unanswered messages
  const unansweredMessagesCount =
    user?.inbox?.filter((message) => !message.read).length || 0;

  // Create tooltip text for inbox
  const getInboxTooltipText = () => {
    if (unansweredMessagesCount === 0) {
      return "No unread messages";
    } else if (unansweredMessagesCount === 1) {
      return "1 unread message";
    } else {
      return `${unansweredMessagesCount} unread messages`;
    }
  };

  useEffect(() => {
    // If hideOnScroll is false, navbar should always be visible
    if (!hideOnScroll) {
      setIsVisible(true);
      return;
    }

    const handleScroll = () => {
      const currentScrollY = window.scrollY;

      // Show navbar when scrolling up or at the top
      if (currentScrollY < lastScrollY || currentScrollY < 10) {
        setIsVisible(true);
      }
      // Hide navbar when scrolling down and past a certain threshold
      else if (currentScrollY > lastScrollY && currentScrollY > 100) {
        setIsVisible(false);
      }

      setLastScrollY(currentScrollY);
    };

    window.addEventListener("scroll", handleScroll, { passive: true });

    return () => window.removeEventListener("scroll", handleScroll);
  }, [lastScrollY, hideOnScroll]);

  const closeSidebar = () => setSidebarOpen(false);

  const handleSignOut = () => {
    logout(); // This will clear the global auth state and localStorage
    console.log("User signed out");
    // Optional: Close any open modals/sidebars
    setSidebarOpen(false);
    setInboxOpen(false);
  };

  // Handle dropdown toggle
  const handleDropdownToggle = (dropdownName: string) => {
    setOpenDropdown(openDropdown === dropdownName ? null : dropdownName);
  };

  // Close dropdown when clicking on a menu item
  const handleMenuItemClick = () => {
    setOpenDropdown(null);
  };

  // Show loading state while checking auth
  if (isLoading) {
    return (
      <div className="w-full h-20 justify-center flex items-center">
        <div className="fixed top-4 z-50 flex w-9/12 md:w-[90rem] items-center justify-center gap-4 bg-[#222222] text-white rounded-3xl px-6 py-3">
          <div className="animate-pulse flex items-center gap-4">
            <div className="w-12 h-12 bg-gray-600 rounded"></div>
            <div className="w-32 h-6 bg-gray-600 rounded"></div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <TooltipProvider>
      {/* Desktop Navigation */}
      <div className="w-full h-20 justify-center hidden md:flex items-center">
        <div
          className={`
            fixed top-4 z-50
            flex w-9/12 md:w-[90rem] items-center justify-center gap-4 bg-[#222222] text-white rounded-3xl px-6 py-3
            transition-transform duration-300 ease-in-out
            ${isVisible ? "translate-y-0" : "-translate-y-[calc(100%+1rem)]"}
          `}
        >
          <div className="flex flex-1  justify-start items-center gap-4">
            <Image
              src="/images/uct_logo.png"
              width={50}
              height={50}
              alt="UCT Logo"
            />
            <Link href={"/"} className="text-center flex flex-col">
              <span className="text-xl outfit-500 ">MyAdvisor</span>
              <span className="text-xs text-[#bdbdbd] outfit-300">
                University of Cape town
              </span>
            </Link>
          </div>

          <div className=" flex-1 flex items-center justify-center outfit-500 gap-4">
            <Link
              href={"/helpbot"}
              className="px-6 py-2 rounded-md transition-colors hover:bg-[#2b2b2b] hover:text-white min-w-[100px] text-center"
            >
              HelpBot
            </Link>

            <DropdownMenu
              trigger="Academic Info"
              isOpen={openDropdown === "academic"}
              onToggle={() => handleDropdownToggle("academic")}
            >
              <ul className="grid gap-3 p-4 md:grid-cols-1">
                <ListItem
                  href="/academic_info/handbooks"
                  title="HandBooks"
                  onClick={handleMenuItemClick}
                >
                  Access detailed information on every degree and its
                  requirements.
                </ListItem>
                <ListItem
                  href="/course_info/courses"
                  title="Courses"
                  onClick={handleMenuItemClick}
                >
                  Explore available courses, their prerequisites, and credit
                  details.
                </ListItem>
              </ul>
            </DropdownMenu>

            <DropdownMenu
              trigger="Degree Planning"
              isOpen={openDropdown === "degree"}
              onToggle={() => handleDropdownToggle("degree")}
            >
              <ul className="grid gap-3 p-4 md:grid-cols-1">
                {degree_components.map((component) => (
                  <ListItem
                    key={component.title}
                    title={component.title}
                    href={component.href}
                    onClick={handleMenuItemClick}
                  >
                    {component.description}
                  </ListItem>
                ))}
              </ul>
            </DropdownMenu>

            <DropdownMenu
              trigger="Student Help"
              isOpen={openDropdown === "student"}
              onToggle={() => handleDropdownToggle("student")}
            >
              <ul className="grid gap-3 p-4 md:grid-cols-1">
                {stud_components.map((component) => (
                  <ListItem
                    key={component.title}
                    title={component.title}
                    href={component.href}
                    onClick={handleMenuItemClick}
                  >
                    {component.description}
                  </ListItem>
                ))}
              </ul>
            </DropdownMenu>
          </div>

          <div className="flex flex-1 items-center justify-end gap-x-4 outfit-400 w-40">
            {isLoggedIn ? (
              <>
                {/* Show user info if available */}
                {user && (
                  <span className="text-sm text-gray-300 hidden lg:block">
                    Welcome, {user.username}
                  </span>
                )}

                {/* Desktop Inbox Button with Tooltip */}
                <div className="relative">
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <Button
                        variant="ghost"
                        size="icon"
                        className="text-white hover:bg-gray-500 rounded-full"
                        onClick={() => setInboxOpen(true)}
                      >
                        <Inbox className="h-5 w-5 size-0.5" />
                        <span className="sr-only">Inbox</span>
                      </Button>
                    </TooltipTrigger>
                    <TooltipContent>
                      <p>{getInboxTooltipText()}</p>
                    </TooltipContent>
                  </Tooltip>
                  {/* Notification Dot */}
                  {hasUnreadMessages && (
                    <div className="absolute -top-1 -right-1 h-3 w-3 bg-red-500 rounded-full border-2 border-[#222222]"></div>
                  )}
                </div>

                {/* Sign Out Button */}
                <Button
                  onClick={handleSignOut}
                  className="text-center bg-white text-base outift-400 text-black py-5 px-4 rounded-full transition-colors duration-300 hover:text-white hover:bg-red-600"
                >
                  Sign Out
                </Button>
              </>
            ) : (
              <Link
                href="/login"
                className="text-center bg-white text-base outift-400 text-black py-2 px-4 rounded-full transition-colors duration-300 hover:text-white hover:bg-uct-blue"
                style={{ maxWidth: 160 }}
              >
                Sign In
              </Link>
            )}
          </div>
        </div>
      </div>

      {/* Mobile Navigation */}
      <div className="md:hidden">
        <div
          className={`
            fixed top-0 left-0 right-0 z-50
            flex items-center justify-between bg-[#222222] text-white px-4 py-4
            transition-transform duration-300 ease-in-out
            ${isVisible ? "translate-y-0" : "-translate-y-full"}
          `}
        >
          {/* Sidebar Trigger */}
          <Sheet open={sidebarOpen} onOpenChange={setSidebarOpen}>
            <SheetTrigger asChild>
              <Button
                variant="ghost"
                size="icon"
                className="text-white hover:bg-[#2b2b2b]"
              >
                <Menu className="h-6 w-6" />
                <span className="sr-only">Open menu</span>
              </Button>
            </SheetTrigger>
            <SheetContent side="left" className="w-[280px] sm:w-[350px] px-2">
              <SheetHeader>
                <SheetTitle className="text-2xl outfit-500">
                  MyAdvisor
                </SheetTitle>
              </SheetHeader>
              <nav className="mt-6 space-y-2">
                {/* HelpBot as top-level item (no indentation) */}
                <TopLevelSidebarItem href="/helpbot" onClick={closeSidebar}>
                  HelpBot
                </TopLevelSidebarItem>

                <SidebarSection title="Academic Info">
                  <SidebarItem
                    href="/academic_info/handbooks"
                    onClick={closeSidebar}
                  >
                    HandBooks
                  </SidebarItem>
                  <SidebarItem
                    href="/course_info/courses"
                    onClick={closeSidebar}
                  >
                    Courses
                  </SidebarItem>
                </SidebarSection>

                <SidebarSection title="Degree Planning">
                  {degree_components.map((component) => (
                    <SidebarItem
                      key={component.title}
                      href={component.href}
                      onClick={closeSidebar}
                    >
                      {component.title}
                    </SidebarItem>
                  ))}
                </SidebarSection>

                <SidebarSection title="Student Help">
                  {stud_components.map((component) => (
                    <SidebarItem
                      key={component.title}
                      href={component.href}
                      onClick={closeSidebar}
                    >
                      {component.title}
                    </SidebarItem>
                  ))}
                </SidebarSection>

                <div className="mt-6 pt-6 border-t">
                  {isLoggedIn ? (
                    <div className="space-y-4 flex flex-col items-center">
                      {/* Show user info in mobile */}
                      {user && (
                        <div className="text-sm text-gray-600 text-center">
                          Logged in as: <strong>{user.username}</strong>
                        </div>
                      )}

                      {/* Mobile Inbox Button with Tooltip - Centered */}
                      <div className="relative">
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Button
                              onClick={() => {
                                setInboxOpen(true);
                                closeSidebar();
                              }}
                              className="flex items-center gap-2 bg-[#222222] text-white font-semibold py-2 px-6 rounded-full transition-colors duration-300 hover:bg-[#2b2b2b]"
                            >
                              <Inbox className="h-4 w-4" />
                              Inbox
                            </Button>
                          </TooltipTrigger>
                          <TooltipContent>
                            <p>{getInboxTooltipText()}</p>
                          </TooltipContent>
                        </Tooltip>
                        {/* Mobile Notification Dot */}
                        {hasUnreadMessages && (
                          <div className="absolute -top-1 -right-1 h-3 w-3 bg-red-500 rounded-full border-2 border-white"></div>
                        )}
                      </div>

                      {/* Mobile Sign Out Button - Centered */}
                      <Button
                        onClick={() => {
                          handleSignOut();
                          closeSidebar();
                        }}
                        className="bg-red-600 text-white font-semibold py-2 px-12 rounded-full transition-colors duration-300 hover:bg-red-700"
                      >
                        Sign Out
                      </Button>
                    </div>
                  ) : (
                    <div className="flex justify-center items-center">
                      <Link
                        href="/login"
                        className="bg-[#222222] text-white font-semibold py-2 px-12 rounded-full transition-colors duration-300 hover:bg-[#2b2b2b]"
                        onClick={closeSidebar}
                      >
                        Sign In
                      </Link>
                    </div>
                  )}
                </div>
              </nav>
            </SheetContent>
          </Sheet>

          {/* Center Logo */}
          <Link href="/" className="font-semibold text-xl outfit-500 ">
            MyAdvisor
          </Link>

          {/* Right Side - Mobile Icons */}
          <div className="flex items-center gap-2">
            {isLoggedIn && (
              <div className="relative">
                <Tooltip>
                  <TooltipTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      className="text-white hover:bg-[#2b2b2b]"
                      onClick={() => setInboxOpen(true)}
                    >
                      <Inbox className="h-5 w-5" />
                      <span className="sr-only">Inbox</span>
                    </Button>
                  </TooltipTrigger>
                  <TooltipContent>
                    <p className="!text-lg !font-medium">
                      {getInboxTooltipText()}
                    </p>
                  </TooltipContent>
                </Tooltip>
                {/* Mobile Header Notification Dot */}
                {hasUnreadMessages && (
                  <div className="absolute -top-1 -right-1 h-3 w-3 bg-red-500 rounded-full border-2 border-[#222222]"></div>
                )}
              </div>
            )}
            <Image
              src="/images/uct_logo.png"
              width={40}
              height={40}
              alt="UCT Logo"
            />
          </div>
        </div>
      </div>

      {/* Inbox Dialog */}
      <InboxWindow isOpen={inboxOpen} onOpenChange={setInboxOpen} />
    </TooltipProvider>
  );
}
