"use client";

import { useEffect, useRef, useState } from "react";
import lottie, { type AnimationItem } from "lottie-web";
import { Navbar } from "./Navbar";

const ErrorPage = ({
  title = "Hang on! We are under maintenance",
  message = "Dont worry, We cooking up something",
  animationPath = "/images/error_page.json",
  animationWidth = 500, // desktop default
  animationHeight = 500,
  mobileWidth = 300, // mobile default
  mobileHeight = 300,
  showBackButton = true,
  backButtonText = "Go Back",
  onBackClick = null,
}) => {
  const animationRef = useRef<HTMLDivElement | null>(null);
  const lottieInstance = useRef<AnimationItem | undefined>(undefined);
  const [animationError, setAnimationError] = useState("");
  const [isMobile, setIsMobile] = useState(false);

  // Detect mobile (tailwind sm breakpoint ~640px)
  useEffect(() => {
    const handleResize = () => setIsMobile(window.innerWidth < 640);
    handleResize(); // run once on mount
    window.addEventListener("resize", handleResize);
    return () => window.removeEventListener("resize", handleResize);
  }, []);

  // Load Lottie animation
  useEffect(() => {
    if (animationRef.current && animationPath) {
      try {
        if (lottieInstance.current) {
          lottieInstance.current.destroy();
        }

        lottieInstance.current = lottie.loadAnimation({
          container: animationRef.current,
          path: animationPath,
          renderer: "svg",
          loop: true,
          autoplay: true,
          rendererSettings: {
            preserveAspectRatio: "xMidYMid meet", // scales with container
          },
        });

        setAnimationError("");
      } catch (err) {
        setAnimationError("Failed to load animation");
        console.error("Lottie animation error:", err);
      }
    }

    return () => {
      if (lottieInstance.current) {
        lottieInstance.current.destroy();
      }
    };
  }, [animationPath, isMobile]); // ✅ safe dependencies

  // Pick size based on screen
  const currentWidth = isMobile ? mobileWidth : animationWidth;
  const currentHeight = isMobile ? mobileHeight : animationHeight;

  return (
    <div className="relative min-h-screen w-full overflow-x-hidden bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50 flex items-center justify-center p-4">
      <div className="absolute inset-0 z-20">
        {/* Soft Gradient Overlay */}
        {/* <div className="absolute inset-0 bg-gradient-to-br from-rose-100/40 via-sky-100/30 to-violet-100/40" /> */}
        <Navbar />
      </div>

      <div className="flex flex-col justify-center items-center text-center max-w-md mx-auto w-full">
        {/* Lottie Animation */}
        <div className="flex justify-center  w-full z-10">
          {animationError ? (
            <div
              className="flex items-center justify-center bg-gray-50 border-2 border-dashed border-gray-300 rounded-lg drop-shadow-lg"
              style={{
                width: `${currentWidth}px`,
                height: `${currentHeight}px`,
              }}
            >
              <p className="text-gray-500 text-center text-sm px-4">
                Animation not available
              </p>
            </div>
          ) : (
            <div
              ref={animationRef}
              className="drop-shadow-lg w-full max-w-xs h-auto"
              style={{
                width: "100%",
                maxWidth: `${currentWidth}px`,
                height: `${currentHeight}px`,
              }}
            />
          )}
        </div>

        {/* Content */}
        <div className="flex w-full md:w-7xl justify-center items-center flex-col  px-8 z-10">
          <h1 className=" w-full text-xl md:text-8xl  text-[#222222]/80 mb-4 outfit-400">
            {title}
          </h1>

          <p className="text-gray-600  text-sm md:text-base mb-8 leading-relaxed outfit-300">
            {message}
          </p>
        </div>
      </div>
    </div>
  );
};

export default ErrorPage;
