// Updated AuthContext.tsx - Fixed refresh functionality to properly handle API response

"use client";

import React, {
  createContext,
  useContext,
  useState,
  useEffect,
  ReactNode,
} from "react";

interface InboxMessage {
  category: string;
  subject: string;
  question: string;
  answer: string | null;
  student: string;
  advisor: string;
  read: boolean;
  question_timestamp?: string; // Add optional timestamp fields
  answer_timestamp?: string;
}

interface User {
  id?: string; // Keep for compatibility
  username: string;
  email?: string; // Keep for compatibility
  is_advisor: boolean;
  red_dot: boolean;
  inbox: InboxMessage[];
}

interface AuthContextType {
  isLoggedIn: boolean;
  user: User | null;
  login: (user: User, token?: string) => void;
  logout: () => void;
  isLoading: boolean;
  updateMessageReadStatus: (messageIndex: number) => void;
  refreshUserData: () => Promise<void>;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

interface AuthProviderProps {
  children: ReactNode;
}

export function AuthProvider({ children }: AuthProviderProps) {
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [user, setUser] = useState<User | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  // Function to refresh user data from the server
  const refreshUserData = async () => {
    if (!user) {
      console.log("No user to refresh");
      return;
    }

    try {
      console.log("🔄 Refreshing user data for:", user.username);

      const response = await fetch("/api/refresh/", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          username: user.username,
          is_advisor: user.is_advisor,
          inbox: [], // Always send empty inbox as specified
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "Failed to refresh user data");
      }

      const apiResponse = await response.json();
      console.log("✅ API Response received:", apiResponse);

      // The API returns the structure: { data: { username, is_advisor, red_dot, inbox } }
      const refreshedData = apiResponse.data;
      console.log("✅ Refreshed data extracted:", refreshedData);

      // Update user with refreshed data, preserving existing user properties
      const updatedUser: User = {
        id: refreshedData.username, // Use username as ID
        username: refreshedData.username,
        email: user.email || `${refreshedData.username}@uct.ac.za`, // Keep existing email or generate fallback
        is_advisor: refreshedData.is_advisor,
        red_dot: refreshedData.red_dot ?? false,
        inbox: refreshedData.inbox || [],
      };

      console.log("✅ Updated user object:", updatedUser);

      setUser(updatedUser);

      // Update localStorage with fresh data
      localStorage.setItem("user", JSON.stringify(updatedUser));
      console.log("✅ User data refreshed and stored in localStorage");
    } catch (error) {
      console.error("❌ Error refreshing user data:", error);
      // Don't logout on refresh error, just log it
      // The user can continue with cached data
    }
  };

  // Check for existing auth on mount and refresh if user exists
  useEffect(() => {
    const checkAuthStatus = async () => {
      try {
        // Check if we're in the browser (not during SSR)
        if (typeof window === "undefined") {
          setIsLoading(false);
          return;
        }

        const storedUser = localStorage.getItem("user");
        const storedToken = localStorage.getItem("authToken");

        if (storedUser && storedToken) {
          try {
            const userData = JSON.parse(storedUser);
            console.log(
              "🔄 AuthContext - Restored user from localStorage:",
              userData
            );
            console.log(
              "🔄 AuthContext - Restored inbox data:",
              userData.inbox
            );

            setUser(userData);
            setIsLoggedIn(true);
            console.log("✅ User session restored from localStorage");

            // After restoring user, refresh their data from server
            try {
              console.log(
                "🔄 Calling refresh for restored user:",
                userData.username
              );

              const response = await fetch("/api/refresh/", {
                method: "POST",
                headers: {
                  "Content-Type": "application/json",
                },
                body: JSON.stringify({
                  username: userData.username,
                  is_advisor: userData.is_advisor,
                  inbox: [], // Always send empty inbox as specified
                }),
              });

              if (response.ok) {
                const refreshResponse = await response.json();
                console.log("✅ Refresh response received:", refreshResponse);

                // Extract the actual user data from the nested 'data' property
                const refreshedData = refreshResponse.data;
                console.log(
                  "✅ User data refreshed on restore:",
                  refreshedData
                );

                // Update user with refreshed data
                const updatedUser: User = {
                  id: refreshedData.username,
                  username: refreshedData.username,
                  email:
                    userData.email || `${refreshedData.username}@uct.ac.za`, // Keep existing or generate fallback
                  is_advisor: refreshedData.is_advisor,
                  red_dot: refreshedData.red_dot ?? false,
                  inbox: refreshedData.inbox || [],
                };

                console.log("✅ Final updated user object:", updatedUser);

                setUser(updatedUser);
                localStorage.setItem("user", JSON.stringify(updatedUser));
              } else {
                console.log("⚠️ Refresh failed, continuing with cached data");
                const errorData = await response.json();
                console.error("Refresh error:", errorData);
              }
            } catch (refreshError) {
              console.error(
                "❌ Error during refresh on restore:",
                refreshError
              );
              // Continue with cached data
            }
          } catch (parseError) {
            console.error("❌ Error parsing stored user data:", parseError);
            // Clear corrupted data
            localStorage.removeItem("user");
            localStorage.removeItem("authToken");
          }
        } else {
          console.log("ℹ️ No stored authentication found");
        }
      } catch (error) {
        console.error("Error checking auth status:", error);
        // Clear potentially corrupted data
        localStorage.removeItem("user");
        localStorage.removeItem("authToken");
      } finally {
        setIsLoading(false);
      }
    };

    checkAuthStatus();
  }, []);

  const login = (userData: User, token?: string) => {
    setUser(userData);
    setIsLoggedIn(true);

    try {
      // Store in localStorage for persistence
      localStorage.setItem("user", JSON.stringify(userData));
      if (token) {
        localStorage.setItem("authToken", token);
      } else {
        // Generate a simple session token if none provided
        const sessionToken = `session_${Date.now()}_${userData.username}`;
        localStorage.setItem("authToken", sessionToken);
      }

      console.log("User logged in and stored in localStorage");
    } catch (error) {
      console.error("Error storing user data:", error);
    }
  };

  const logout = () => {
    setUser(null);
    setIsLoggedIn(false);

    try {
      // Clear stored data
      localStorage.removeItem("user");
      localStorage.removeItem("authToken");
      console.log("User logged out and localStorage cleared");
    } catch (error) {
      console.error("Error during logout:", error);
    }
  };

  // Function to update message read status
  const updateMessageReadStatus = (messageIndex: number) => {
    if (!user) return;

    const updatedUser = {
      ...user,
      inbox: user.inbox.map((message, index) =>
        index === messageIndex ? { ...message, read: true } : message
      ),
      // Update red_dot status - set to false if all messages are read
      red_dot: user.inbox.some((message, index) =>
        index === messageIndex ? false : !message.read
      ),
    };

    setUser(updatedUser);

    // Update localStorage
    try {
      localStorage.setItem("user", JSON.stringify(updatedUser));
    } catch (error) {
      console.error("Error updating user data:", error);
    }
  };

  const value = {
    isLoggedIn,
    user,
    login,
    logout,
    isLoading,
    updateMessageReadStatus,
    refreshUserData,
  };

  return <AuthContext.Provider value={value}>{children}</AuthContext.Provider>;
}

// Custom hook to use the auth context
export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error("useAuth must be used within an AuthProvider");
  }
  return context;
}

// HOC for protected routes - improved with better loading handling
export function withAuth<T extends object>(Component: React.ComponentType<T>) {
  return function AuthenticatedComponent(props: T) {
    const { isLoggedIn, isLoading } = useAuth();

    // Show loading while checking auth status
    if (isLoading) {
      return (
        <div className="flex items-center justify-center min-h-screen">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900 mx-auto mb-4"></div>
            <p>Loading...</p>
          </div>
        </div>
      );
    }

    // Redirect to login if not authenticated
    if (!isLoggedIn) {
      // Use Next.js router instead of window.location for better UX
      if (typeof window !== "undefined") {
        window.location.href = "/login";
      }
      return null;
    }

    return <Component {...props} />;
  };
}
