"use client";

import Image from "next/image";
import type React from "react";
import Link from "next/link";

import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { User, ArrowRight, Clock } from "lucide-react";

import { motion } from "motion/react";
import { MovingBorder } from "@/components/ui/moving-border";
import { CircleHelp } from "lucide-react";
import { CheckCircle, BarChart3 } from "lucide-react";

import ErrorPage from "@/components/ErrorPage";
// Import your animation data

import { useEffect, useRef } from "react";
import lottie, { type AnimationItem } from "lottie-web";
import {
  Book,
  GraduationCap,
  HelpCircle,
  BookOpen,
  Users,
  MessageCircle,
  Bot,
  GitFork,
} from "lucide-react";

import { Card, CardContent } from "@/components/ui/card";

import { NavigationMenuLink } from "@/components/ui/navigation-menu";

import SplitText from "@/components/SplitText";

import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
} from "@/components/ui/carousel";
import { Navbar } from "@/components/Navbar";
import { Footer } from "@/components/Footer";

const pages = [
  {
    title: "Handbook",
    description:
      "Complete university handbook with all policies and procedures",
    icon: Book,
    color: "bg-blue-50 text-blue-600",
    href: "/academic_info/handbooks",
  },
  {
    title: "Courses",
    description: "Browse and explore all available courses and programs",
    icon: GraduationCap,
    color: "bg-green-50 text-green-600",
    href: "/course_info/courses",
  },
  {
    title: "FAQ",
    description: "Frequently asked questions and quick answers",
    icon: HelpCircle,
    color: "bg-purple-50 text-purple-600",
    href: "/student_help/faq",
  },
  {
    title: "Glossary",
    description: "Academic terms and definitions made simple",
    icon: BookOpen,
    color: "bg-orange-50 text-orange-600",
    href: "/student_help/glossary",
  },
  {
    title: "Advisors Info",
    description: "Meet your academic advisors and their specialties",
    icon: Users,
    color: "bg-teal-50 text-teal-600",
    href: "/student_help/advisor_info",
  },
];

// Add this or import your logo SVG/component
const Icons = {
  logo: (props: React.SVGProps<SVGSVGElement>) => (
    <svg {...props} viewBox="0 0 24 24" fill="none">
      <circle cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="2" />
      <path d="M8 12h8" stroke="currentColor" strokeWidth="2" />
      <path d="M12 8v8" stroke="currentColor" strokeWidth="2" />
    </svg>
  ),
};

// ListItem helper component from shadcn/ui docs
const ListItem = ({
  className,
  title,
  children,
  ...props
}: React.ComponentPropsWithoutRef<"a"> & { title: string }) => (
  <li>
    <NavigationMenuLink asChild>
      <a
        className={
          "block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors hover:bg-accent hover:text-accent-foreground focus:bg-accent focus:text-accent-foreground " +
          (className || "")
        }
        {...props}
      >
        <div className="text-sm font-medium leading-none">{title}</div>
        <p className="line-clamp-2 text-sm leading-snug text-muted-foreground">
          {children}
        </p>
      </a>
    </NavigationMenuLink>
  </li>
);

const transition = {
  duration: 0.8,
  delay: 0.5,
  ease: [0, 0.71, 0.2, 1.01],
};

export default function Home() {
  const ai_video = useRef<HTMLDivElement | null>(null);
  const ai_videoMin = useRef<HTMLDivElement | null>(null);

  useEffect(() => {
    let animation: AnimationItem | undefined;

    if (ai_video.current) {
      animation = lottie.loadAnimation({
        container: ai_video.current,
        path: "/images/Ai_video1.json",
        renderer: "svg",
        loop: true,
        autoplay: true,
      });
    }

    return () => {
      animation?.destroy(); // ✅ Prevent duplicate renders
    };
  }, []);

  useEffect(() => {
    let animation: AnimationItem | undefined;

    if (ai_videoMin.current) {
      animation = lottie.loadAnimation({
        container: ai_videoMin.current,
        path: "/images/Ai_video1.json",
        renderer: "svg",
        loop: true,
        autoplay: true,
      });
    }

    return () => {
      animation?.destroy(); // ✅ Prevent duplicate renders
    };
  }, []);
  return (
    <div className="flex flex-col items-center min-h-screen">
      {/* Unified Gradient Background Section */}
      <div className="relative w-full bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50">
        {/* Animated Background Elements - spans entire section */}
        <div className="absolute inset-0 z-0">
          {/* Soft Gradient Overlay */}
          <div className="absolute inset-0 bg-gradient-to-br from-rose-100/40 via-sky-100/30 to-violet-100/40" />
        </div>

        {/* Animated Floating Elements */}
        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute top-80 left-1/4 animate-pulse hidden md:block"
        >
          <div
            className="w-4 h-4 bg-gradient-to-r from-rose-300 to-pink-300 rounded-full animate-bounce"
            style={{ animationDelay: "1s" }}
          />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute top-42 right-1/3 animate-pulse hidden md:block"
        >
          <div
            className="w-3 h-3 bg-gradient-to-r from-sky-300 to-blue-300 rounded-full animate-bounce"
            style={{ animationDelay: "1.7s" }}
          />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute top-76 right-1/4 animate-pulse hidden md:block"
        >
          <div
            className="w-5 h-5 bg-gradient-to-r from-violet-300 to-purple-300 rounded-full animate-bounce"
            style={{ animationDelay: "2s" }}
          />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute top-40 left-150 animate-bounce hidden md:block"
          style={{ animationDelay: "1.5s" }}
        >
          <div className="w-6 h-6 bg-gradient-to-br from-rose-200 to-pink-300 rounded-lg opacity-70 rotate-12" />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute top-60 left-250 animate-pulse opacity-50 hidden md:block"
        >
          <div className="w-8 h-2 bg-gradient-to-r from-emerald-300 to-mint-400 rounded-full transform rotate-45" />
        </motion.div>

        {/* bottom icons */}

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute bottom-190 right-300 animate-bounce hidden md:block"
          style={{ animationDelay: "1.5s" }}
        >
          <div className="w-5 h-5 bg-gradient-to-br from-violet-200 to-purple-300 rounded-lg opacity-60 -rotate-12" />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute bottom-170 left-300 animate-bounce hidden md:block"
          style={{ animationDelay: "1.2s" }}
        >
          <div className="w-6 h-6 bg-gradient-to-br from-rose-200 to-pink-300 rounded-lg opacity-70 rotate-12" />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute bottom-170 left-1/5 animate-pulse hidden md:block"
        >
          <div
            className="w-3 h-3 bg-gradient-to-r from-sky-300 to-blue-300 rounded-full animate-bounce"
            style={{ animationDelay: "1.5s" }}
          />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute bottom-190 right-1/6 animate-pulse hidden md:block"
        >
          <div
            className="w-4 h-4 bg-gradient-to-r from-rose-300 to-pink-300 rounded-full animate-bounce"
            style={{ animationDelay: "1.5s" }}
          />
        </motion.div>

        <motion.div
          initial={{ scale: 0 }}
          animate={{ scale: 1, transition: { duration: 1 } }}
          className="absolute bottom-210 right-200 animate-pulse opacity-40 hidden md:block"
        >
          <div className="w-6 h-2 bg-gradient-to-r from-sky-200 to-blue-300 rounded-full transform -rotate-12" />
        </motion.div>

        <div className="absolute top-10 left-1/2 opacity-30">
          <div className="flex space-x-1">
            <div className="w-8 h-4 bg-white rounded-full" />
            <div className="w-6 h-3 bg-white rounded-full -ml-2 mt-1" />
            <div className="w-4 h-2 bg-white rounded-full -ml-1 mt-1" />
          </div>
        </div>

        <div className="absolute top-20 right-10 opacity-25">
          <div className="flex space-x-1">
            <div className="w-6 h-3 bg-white rounded-full" />
            <div className="w-8 h-4 bg-white rounded-full -ml-2" />
            <div className="w-5 h-3 bg-white rounded-full -ml-1 mt-1" />
          </div>
        </div>

        {/* Nav Bar */}
        <div className="relative z-50">
          <Navbar />
        </div>

        {/* Hero Content */}
        <div className="relative w-full flex flex-col items-center justify-center min-h-[700px] md:min-h-[780px] overflow-hidden -mt-20 pt-20">
          {/* Original Content - Moved Higher */}
          <div className="md:w-[65rem] flex flex-col items-center justify-center md:text-center md:mt-8 md:mb-10">
            <SplitText
              text="Simplify Your Academic Journey"
              className="relative z-10 w-full text-4xl md:text-7xl  lg:text-8xl text-center font-base px-8 md:px-0 outfit-300  mt-8"
              delay={0}
              speed={0.05}
            />
            <SplitText
              text="Your personal guide to courses, credits, and academic support"
              className="relative z-10 text-base mt-3 text-gray-600 w-96 text-center md:text-lg lg:text-xl outfit-300 px-8 md:px-0 "
              delay={1.3}
              speed={0.02}
            />
          </div>

          {/*hero images for desktop - Moved Higher*/}
          <div className="absolute left-1/2 -translate-x-[820px] top-[290px] z-20 hidden md:block">
            <div className="w-[500px] h-[500px] flex items-center justify-center">
              <Image
                src="/images/myAdAvatarLeft.png"
                width={430}
                height={430}
                alt="Left Avatar"
                className="motion-preset-slide-right motion-duration-2500"
              />
            </div>
          </div>
          <div className="absolute left-1/2 translate-x-[300px] top-[360px] z-20 hidden md:block">
            <div className="w-[500px] h-[500px] flex items-center justify-center">
              <Image
                src="/images/myAdAvatarRight.png"
                width={430}
                height={430}
                alt="Right Avatar"
                className="motion-preset-slide-left motion-duration-2500"
              />
            </div>
          </div>

          {/*hero images for mobile*/}
          <div className="absolute left-1/2 -translate-x-[200px] top-[120px] z-20 md:hidden block overflow">
            <div className="w-[390px] h-[390px] flex items-center justify-center"></div>
          </div>
        </div>

        {/*Our Services section - Seamlessly integrated*/}
        <div className="relative z-10 pb-10 md:pb-32 pt-16 md:pt-50 ">
          <motion.div
            className="text-4xl md:text-7xl mb-10 text-center outfit-300"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
            viewport={{ once: true, margin: "-100px" }}
          >
            Our Services
          </motion.div>

          <div className="w-full md:w-[90rem] mt-5 mb-10 md:mt-20 grid grid-cols-1 md:grid-cols-4 gap-6 px-10 md:px-0 md:gap-x-8 md:gap-y-7 lg:px-0 mx-auto">
            <motion.div
              className="flex-1"
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.1,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <MovingBorderCard className="md:min-h-[18rem] h-[16rem] hover:scale-105 transition-transform">
                <Link
                  href="/academic_info/handbooks"
                  className="flex flex-col items-center justify-center h-full p-4 md:p-6 text-center w-full"
                >
                  <div className="w-full h-24 md:h-24 flex items-center justify-center">
                    <BookOpen className="text-[#222222] w-12 h-12" />
                  </div>
                  <div className="text-center text-[#222222] text-xl md:text-2xl mt-2 md:mt-4 font-semibold outfit-400">
                    Handbook Explorer
                  </div>
                  <div className="text-center text-[#222222]   text-sm md:text-sm px-2 md:px-4 mt-2 md:mt-4 outfit-300">
                    Quickly search faculty handbooks for course info, rules, and
                    requirements.
                  </div>
                </Link>
              </MovingBorderCard>
            </motion.div>

            <motion.div
              className="flex-1"
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.2,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <MovingBorderCard className="md:min-h-[18rem] h-[16rem] hover:scale-105 transition-transform">
                <Link
                  href="/helpbot"
                  className="flex flex-col items-center justify-center h-full p-4 md:p-6 text-center w-full"
                >
                  <div className="w-full h-24 md:h-24 flex items-center justify-center">
                    <Bot className="text-[#222222] w-12 h-12" />
                  </div>
                  <div className="text-center text-xl md:text-2xl mt-2 md:mt-4 font-semibold outfit-400 text-[#222222]">
                    HelpBot
                  </div>
                  <div className="text-center text-[#222222] text-sm md:text-sm px-2 md:px-9 mt-2 md:mt-4 outfit-300 ">
                    Your virtual advisor for course, credit, and policy support.
                  </div>
                </Link>
              </MovingBorderCard>
            </motion.div>

            <motion.div
              className="flex-1"
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.3,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <MovingBorderCard className="md:min-h-[18rem] h-[16rem] hover:scale-105 transition-transform">
                <Link
                  href="/degree_planning/roadmap"
                  className="flex flex-col items-center justify-center h-full p-4 md:p-6 text-center w-full"
                >
                  <div className="w-full h-24 md:h-24 flex items-center justify-center">
                    <GitFork className="text-[#222222] w-12 h-12" />
                  </div>
                  <div className="text-center text-xl md:text-2xl mt-2 md:mt-4 font-semibold outfit-400 text-[#222222]">
                    Degree Roadmap
                  </div>
                  <div className="text-center text-[#222222] text-sm md:text-sm px-2 md:px-9 mt-2 md:mt-4 outfit-300">
                    Plan your degree journey, track credits, and stay on course
                  </div>
                </Link>
              </MovingBorderCard>
            </motion.div>

            <motion.div
              className="flex-1"
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.4,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <MovingBorderCard className="md:min-h-[18rem] h-[16rem] hover:scale-105 transition-transform">
                <Link
                  href="/degree_planning/ask_advisor"
                  className="flex flex-col items-center justify-center h-full p-4 md:p-6 text-center w-full"
                >
                  <div className="w-full h-24 md:h-24 flex items-center justify-center">
                    <CircleHelp className="text-[#222222] w-12 h-12" />
                  </div>
                  <div className="text-center text-xl md:text-2xl mt-2 md:mt-4 font-semibold outfit-400 text-[#222222]">
                    Ask Advisor
                  </div>
                  <div className="text-center text-[#222222] text-sm md:text-sm px-2 md:px-9 mt-2 md:mt-4 outfit-300 ">
                    Your virtual advisor for course, credit, and policy support.
                  </div>
                </Link>
              </MovingBorderCard>
            </motion.div>
          </div>
        </div>

        {/* Gradient fade to white at bottom */}
        <div className="absolute bottom-0 left-0 right-0 h-10 bg-gradient-to-t from-violet-50 to-transparent z-10" />
      </div>
      {/*HelpBot section*/}
      <div className="relative w-full h-fit bg-gradient-to-bl from-violet-50 via-sky-50 to-rose-50 ">
        {/* Gradient fade from white at top */}
        <div className="absolute top-0 left-0 right-0 h-10 bg-gradient-to-b from-violet-50 to-transparent z-10" />

        <div className="w-full flex flex-col items-center justify-center text-center mt-8">
          <motion.div
            className=" text-sm md:text-base border-2 border-black px-3 md:mb-2 rounded-full outfit-300"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
            viewport={{ once: true, margin: "-100px" }}
          >
            HelpBot
          </motion.div>
          <motion.div
            className="text-4xl md:text-6xl  mb-1 mt-5 outfit-400"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
            viewport={{ once: true, margin: "-100px" }}
          >
            Get Answers.
          </motion.div>
          <motion.div
            className="text-4xl md:text-6xl  mb-1 text-uct-blue outfit-400"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
            viewport={{ once: true, margin: "-100px" }}
          >
            Anywhere, anytime
          </motion.div>
          <motion.div
            className="max-w-3xl text-sm md:text-base px-8 md:px-0 text-gray-700 outfit-300 mt-5"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
            viewport={{ once: true, margin: "-100px" }}
          >
            HelpBot is your personal academic advisor, available 24/7 to answer
            your questions about courses, credits, registration rules,
            prerequisites, and more. Whether you're planning your degree, unsure
            about a policy, or just need quick info—HelpBot is here to make
            student life simpler.
          </motion.div>
        </div>
        <motion.div
          className="max-w-7xl min-h-[20rem] flex flex-col md:flex-row  items-stretch justify-center mt-10 rounded-4xl mx-auto w-full mb-18 px-8 md:px-0 "
          initial={{ opacity: 0, y: 50 }}
          whileInView={{ opacity: 1, y: 0 }}
          transition={{
            duration: 0.7,
            delay: 0.1,
            ease: [0.25, 0.46, 0.45, 0.94],
          }}
          viewport={{ once: true, margin: "-100px" }}
        >
          <div className="flex-1 min-h-[20rem] md:min-h-[27rem] bg-[#222222]  rounded-t-4xl md:rounded-t-none md:rounded-l-4xl flex items-center justify-center">
            <div
              ref={ai_video}
              className="flex-1 min-h-[22rem] md:min-h-[27rem] bg-[#222222]  rounded-t-4xl md:rounded-t-none md:rounded-l-4xl  items-center justify-center hidden md:flex"
              style={{ width: "75%", height: "75%" }}
            />
            <div
              ref={ai_videoMin}
              className=" min-h-[22rem] md:min-h-[27rem] bg-[#222222]  rounded-t-4xl md:rounded-t-none md:rounded-l-4xl flex items-center justify-center md:hidden"
              style={{ width: "75%", height: "75%" }}
            />
          </div>
          <div className="md:flex-1 min-h-[5rem] md:min-h-[27rem] bg-uct-blue rounded-b-4xl md:rounded-bl-none md:rounded-r-4xl">
            <div className="flex flex-col h-full py-8 gap-y-10 md:gap-y-14">
              {/* First main div: Functions */}
              <div className="text-3xl md:text-6xl  text-white md:mb-4 self-start ml-8 mt-5 md:mt-10 outfit-400">
                Functions
              </div>

              {/* Second main div: Two lists side by side */}
              <div className="flex flex-row md:gap-12 pl-8 pr-8 outfit-300">
                <ul className="list-disc pl-6 text-white text-sm md:text-base space-y-3 ">
                  <li>Answer FAQs</li>
                  <li>Explain course requirements</li>
                  <li>Help calculate credits</li>
                  <div className=" space-y-3 hidden md:block">
                    <li>Break down handbook content</li>
                    <li> Provide degree structure info</li>
                  </div>
                </ul>
                <ul className="list-disc pl-6 text-white text-sm md:text-base space-y-3 ">
                  <li>Guide students</li>
                  <li>Suggest courses</li>
                  <li>Clarify academic terminology </li>
                  <div className=" space-y-3 hidden md:block">
                    <li>Help resolve common issues</li>
                    <li>Redirect complex queries</li>
                  </div>
                </ul>
              </div>

              {/* Third main div: Get assistance button */}
              <Link
                href="/helpbot"
                className="self-start outfit-400  ml-8 md:mt-4 bg-white text-black font-semibold px-4 py-2 rounded-full shadow hover:bg-black hover:text-white transition-colors duration-300"
              >
                Get assistance
              </Link>
            </div>
          </div>
        </motion.div>
        {/* Gradient fade to white at bottom */}
        {/* <div className="absolute bottom-0 left-0 right-0 h-14 bg-gradient-to-t from-white to-transparent z-10" /> */}
      </div>
      {/*Degree Roadmap section*/}
      <div className="w-full flex justify-center flex-col items-center  bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50 md:pt-10 ">
        {/* Gradient fade from white at top */}

        <div className="w-fit md:mt-15 md:mb-36">
          <div className="w-full flex flex-col items-center justify-center text-center mt-8 mb-10">
            <motion.div
              className=" text-sm md:text-base border-2 border-black px-3 md:mb-2 rounded-full outfit-300"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Degree Roadmap
            </motion.div>
            <motion.div
              className="text-4xl md:text-6xl  mb-1 mt-5 outfit-400"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Your academic path
            </motion.div>
            <motion.div
              className="text-4xl md:text-6xl font-semibold mb-1 text-uct-green outfit-400"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              fully mapped
            </motion.div>
            <motion.div
              className="max-w-3xl text-sm md:text-base px-8 md:px-0 text-gray-700 outfit-300 mt-5"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Roadmap gives you a clear view of your entire degree—from
              first-year courses to final credit requirements. Track your
              progress, understand pass criteria, and plan each semester with
              confidence. No guesswork, just your academic path—mapped out.
            </motion.div>
          </div>
          <div className="max-w-7xl mx-auto">
            <div className="relative">
              {/* Connecting Lines - Only show on desktop */}
              <motion.div
                className="hidden md:block md:absolute top-1/2 left-1/4 right-1/4 h-0.5 bg-green-300 -translate-y-1/2 z-0"
                initial={{ opacity: 0, y: 60, scale: 0.9 }}
                whileInView={{ opacity: 1, y: 0, scale: 1 }}
                transition={{
                  duration: 0.7,
                  delay: 0.4,
                  ease: [0.25, 0.46, 0.45, 0.94],
                }}
                viewport={{ once: true, margin: "-50px" }}
              >
                <div className="absolute left-1/2 top-1/2 w-3 h-3 bg-green-600 rounded-full -translate-x-1/2 -translate-y-1/2"></div>
              </motion.div>

              {/* Step Cards - Remove fixed height, use auto spacing */}
              <div className="grid md:grid-cols-3 gap-8 md:relative z-10 md:h-96 px-8 md:px-0">
                {/* Step 1 */}
                <motion.div
                  className="flex-1"
                  initial={{ opacity: 0, y: 60, scale: 0.9 }}
                  whileInView={{ opacity: 1, y: 0, scale: 1 }}
                  transition={{
                    duration: 0.7,
                    delay: 0.1,
                    ease: [0.25, 0.46, 0.45, 0.94],
                  }}
                  viewport={{ once: true, margin: "-50px" }}
                >
                  <Card className="text-center p-4 md:p-6 md:h-[24rem]  bg-card hover:shadow-lg border-2 border-[#222222]/10 transition-shadow">
                    <CardContent className="md:pt-6 pt-2 px-3 md:px-0">
                      <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <BookOpen className="w-8 h-8 text-green-600" />
                      </div>
                      <div className="w-8 h-8 border-2 border-[#222222]/10 bg-uct-green text-white rounded-full flex items-center justify-center mx-auto mb-4 text-sm font-bold">
                        1
                      </div>
                      <h3 className="text-xl font-semibold mb-3">
                        Select Your Degree
                      </h3>
                      <p className="text-muted-foreground text-sm leading-relaxed w-full">
                        Choose your degree type through our guided selection
                        process. We'll help you identify the right academic path
                        based on your goals and interests.
                      </p>
                    </CardContent>
                  </Card>
                </motion.div>

                {/* Step 2 */}
                <motion.div
                  className="flex-1"
                  initial={{ opacity: 0, y: 60, scale: 0.9 }}
                  whileInView={{ opacity: 1, y: 0, scale: 1 }}
                  transition={{
                    duration: 0.7,
                    delay: 0.2,
                    ease: [0.25, 0.46, 0.45, 0.94],
                  }}
                  viewport={{ once: true, margin: "-50px" }}
                >
                  <Card className="text-center p-4 md:h-[24rem] md:p-6 bg-card hover:shadow-lg border-2 border-[#222222]/10 transition-shadow">
                    <CardContent className="md:pt-6 pt-2 px-3 md:px-0">
                      <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <CheckCircle className="w-8 h-8 text-green-600" />
                      </div>
                      <div className="w-8 h-8 border-2 border-[#222222]/10 bg-uct-green text-white rounded-full flex items-center justify-center mx-auto mb-4 text-sm font-bold">
                        2
                      </div>
                      <h3 className="text-xl font-semibold mb-3">
                        Track Your Progress
                      </h3>
                      <p className="text-muted-foreground text-sm leading-relaxed">
                        Use our Pass/Fail feature to monitor your course
                        performance throughout each academic year. Stay on top
                        of requirements and deadlines.
                      </p>
                    </CardContent>
                  </Card>
                </motion.div>

                {/* Step 3 */}
                <motion.div
                  className="flex-1"
                  initial={{ opacity: 0, y: 60, scale: 0.9 }}
                  whileInView={{ opacity: 1, y: 0, scale: 1 }}
                  transition={{
                    duration: 0.7,
                    delay: 0.3,
                    ease: [0.25, 0.46, 0.45, 0.94],
                  }}
                  viewport={{ once: true, margin: "-50px" }}
                >
                  <Card className="text-center p-4 md:p-6 md:h-[24rem]  bg-card hover:shadow-lg border-2 border-[#222222]/10 transition-shadow">
                    <CardContent className="md:pt-6 pt-2 px-3 md:px-0">
                      <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <BarChart3 className="w-8 h-8 text-green-600" />
                      </div>
                      <div className="w-8 h-8 border-2 border-[#222222]/10 bg-uct-green text-white rounded-full flex items-center justify-center mx-auto mb-4 text-sm font-bold">
                        3
                      </div>
                      <h3 className="text-xl font-semibold mb-3">
                        Get Your Evaluation
                      </h3>
                      <p className="text-muted-foreground text-sm leading-relaxed">
                        Receive a comprehensive evaluation showing what you can
                        and can't do based on your completed coursework and
                        academic progress.
                      </p>
                    </CardContent>
                  </Card>
                </motion.div>
              </div>
            </div>

            {/* Call to Action */}
            <motion.div
              className="text-center mt-16 mb-8"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              <Link
                href="/degree_planning/roadmap"
                className="bg-uct-green hover:bg-green-700 text-white py-2  px-4 md:px-12 border-2 border-[#222222]/10 outfit-400 md:text-lg rounded-4xl transition-colors"
              >
                Start Your Academic Journey
              </Link>
            </motion.div>
          </div>
        </div>

        <div className="flex flex-col items-center justify-center w-full mb-32">
          <div className="w-full flex flex-col items-center justify-center text-center mt-8 mb-10">
            <motion.div
              className=" text-sm md:text-base border-2 border-black px-3 md:mb-2 rounded-full outfit-300"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Ask Advisor
            </motion.div>
            <motion.div
              className="text-4xl md:text-6xl  mb-1 mt-5 outfit-400"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Ask smarter
            </motion.div>
            <motion.div
              className="text-4xl md:text-6xl font-semibold mb-1 text-uct-blue outfit-400"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Get matched instantly
            </motion.div>
            <motion.div
              className="max-w-3xl text-sm md:text-base px-8 md:px-0 text-gray-700 outfit-300 mt-5"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              No more digging through emails or waiting for replies. Ask Advisor
              connects you directly to the right academic advisor—matched to
              your specific question in real time. Whether it’s about modules,
              requirements, or next steps, you’ll get clear, reliable guidance
              without the hassle.
            </motion.div>
          </div>
          {/* Chat Messages Section */}
          <div className="flex flex-col lg:flex-row items-center justify-center gap-12 mb-20 max-w-[90rem] mx-auto px-8 md:px-0">
            {/* Student Message */}
            <motion.div
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.1,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <Card className="bg-white border-gray-200 shadow-lg p-6 max-w-[25rem] w-full">
                <div className="flex items-start gap-3 mb-4">
                  <Avatar className="bg-blue-100 w-12 h-12">
                    <AvatarFallback className="bg-blue-100 text-blue-700">
                      <GraduationCap className="w-6 h-6" />
                    </AvatarFallback>
                  </Avatar>
                  <div>
                    <div className="text-blue-600 outfit-500">Student</div>
                    <div className="text-gray-500 text-sm outfit-400">
                      Ethan Allies
                    </div>
                  </div>
                </div>
                <div className="bg-gray-100 rounded-lg p-4 mb-3">
                  <p className="text-gray-800 text-sm leading-relaxed outfit-300">
                    "Hi! I need help choosing my courses for next semester.
                    Could we schedule a meeting?"
                  </p>
                </div>
                <div className="flex items-center gap-1 text-gray-400 text-xs">
                  <Clock className="w-3 h-3" />2 minutes ago
                </div>
              </Card>
            </motion.div>

            {/* Arrows */}
            <motion.div
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.2,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <div className="hidden lg:flex flex-col items-center justify-center w-64 gap-20">
                <div className="flex items-center w-full">
                  <div className="h-[0.215rem] bg-gradient-to-r from-blue-500 to-green-500 flex-1"></div>
                  <ArrowRight
                    className="w-9 h-9 text-green-500 -ml-2"
                    strokeWidth={2.5}
                    style={{ filter: "none" }}
                  />
                </div>
                <div className="relative flex items-center w-full">
                  <ArrowRight
                    className="w-9 h-9 text-blue-500 rotate-180 -mr-2 z-40"
                    strokeWidth={2.5}
                    style={{ filter: "none" }}
                  />
                  <div className="h-[0.215rem] bg-gradient-to-r from-blue-500/80 to-green-500 flex-1 z-10"></div>
                </div>
              </div>
            </motion.div>

            {/* Advisor Message */}
            <motion.div
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.3,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <Card className="bg-white border-gray-200 shadow-lg p-6 max-w-[25rem] w-full">
                <div className="flex items-start gap-3 mb-4">
                  <Avatar className="bg-green-100 w-12 h-12">
                    <AvatarFallback className="bg-green-100 text-green-700">
                      <User className="w-6 h-6" />
                    </AvatarFallback>
                  </Avatar>
                  <div>
                    <div className="text-green-600 outfit-500">Advisor</div>
                    <div className="text-gray-500 text-sm outfit-400">
                      Dr. Johnson
                    </div>
                  </div>
                </div>
                <div className="bg-gray-100 rounded-lg p-4 mb-3">
                  <p className="text-gray-800 text-sm leading-relaxed  outfit-300">
                    "I'm available tomorrow at 2 PM or Friday at 10 AM. What
                    works better for you?"
                  </p>
                </div>
                <div className="flex items-center gap-1 text-gray-400 text-xs">
                  <Clock className="w-3 h-3" />
                  Just now
                </div>
              </Card>
            </motion.div>
          </div>

          {/* Features Section */}
          <div className="grid md:grid-cols-3 gap-8 max-w-5xl mx-auto px-8 md:px-0">
            {/* Instant Messaging */}
            <motion.div
              className="text-center"
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.1,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <div className="w-16 h-16 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-4 drop-shadow-sm">
                <MessageCircle className="w-8 h-8 text-purple-700" />
              </div>
              <h3 className="text-xl mb-3 text-gray-900 outfit-500">
                Instant Messaging
              </h3>
              <p className="text-gray-600 leading-relaxed">
                Real-time communication with typing indicators and read receipts
              </p>
            </motion.div>

            {/* Smart Matching */}
            <motion.div
              className="text-center"
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.2,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4 drop-shadow-sm">
                <User className="w-8 h-8 text-green-700" />
              </div>
              <h3 className="text-xl outfit-500 mb-3 text-gray-900">
                Smart Matching
              </h3>
              <p className="text-gray-600 leading-relaxed">
                Automatically connect with the right advisor for your needs
              </p>
            </motion.div>

            {/* Academic Support */}
            <motion.div
              className="text-center"
              initial={{ opacity: 0, y: 60, scale: 0.9 }}
              whileInView={{ opacity: 1, y: 0, scale: 1 }}
              transition={{
                duration: 0.7,
                delay: 0.3,
                ease: [0.25, 0.46, 0.45, 0.94],
              }}
              viewport={{ once: true, margin: "-50px" }}
            >
              <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4 drop-shadow-sm">
                <GraduationCap className="w-8 h-8 text-blue-700 " />
              </div>
              <h3 className="text-xl outfit-500 mb-3 text-gray-900">
                Academic Support
              </h3>
              <p className="text-gray-600 leading-relaxed">
                Get guidance on courses, career planning, and academic goals
              </p>
            </motion.div>
          </div>
          {/* Call to Action */}
          <motion.div
            className="text-center mt-16"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
            viewport={{ once: true, margin: "-100px" }}
          >
            <Link
              href="/degree_planning/ask_advisor"
              className="bg-uct-blue  hover:bg-blue-800 text-white py-2 px-12  border-2 border-[#222222]/10 outfit-400 text-lg  rounded-4xl transition-colors duration-300"
            >
              Ask an Advisor
            </Link>
          </motion.div>
        </div>

        {/*Handbook Explorer section*/}
        <div className="w-full h-fit">
          <div className="w-full flex flex-col items-center justify-center text-center mt-8 mb-10">
            <motion.div
              className="text-sm md:text-base border-2 border-black px-3 md:mb-2 rounded-full outfit-400"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Handbook Explorer
            </motion.div>
            <motion.div
              className="text-4xl md:text-6xl  mb-1 mt-5 outfit-400"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Explore smarter.
            </motion.div>
            <motion.div
              className="text-4xl md:text-6xl  mb-1 text-uct-green outfit-400"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Find it faster
            </motion.div>
            <motion.div
              className="text-sm md:text-base max-w-3xl text-gray-700 mt-5 px-8 md:px-8 outfit-300"
              initial={{ opacity: 0, y: 50 }}
              whileInView={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, ease: "easeOut" }}
              viewport={{ once: true, margin: "-100px" }}
            >
              Handbook Explorer brings all the important UCT info together—clear
              course descriptions, personalized degree suggestions, FAQs, and
              easy-to-understand academic terms. It's where students go to get
              answers, plan smarter, and feel more in control of their
              university journey.
            </motion.div>
          </div>
          <div className="w-full h-fit bg-uct-blue rounded-t-4xl flex items-center justify-center">
            <Carousel className="w-full max-w-9/12 md:max-w-5xl py-20 px-4 md:px-0">
              <CarouselContent className="-ml-12">
                {pages.map((page, index) => {
                  const IconComponent = page.icon;
                  return (
                    <CarouselItem
                      key={index}
                      className="pl-12 md:basis-1/2 lg:basis-1/3"
                    >
                      <div className="p-1">
                        <Link href={page.href} className="block h-full">
                          <Card className="h-[18rem]  bg-white hover:shadow-lg transition-shadow duration-300 cursor-pointer">
                            <CardContent className="flex flex-col items-center justify-center p-6 h-full min-h-[200px]">
                              <div
                                className={`w-16 h-16 rounded-full ${page.color} flex items-center justify-center mb-4 border-2 border-[#222222]/5 shadow-xs`}
                              >
                                <IconComponent className="w-8 h-8" />
                              </div>
                              <h3 className="text-xl font-semibold mb-2 text-gray-900">
                                {page.title}
                              </h3>
                              <p className="text-sm text-gray-600 text-center leading-relaxed">
                                {page.description}
                              </p>
                            </CardContent>
                          </Card>
                        </Link>
                      </div>
                    </CarouselItem>
                  );
                })}
              </CarouselContent>
              <CarouselPrevious className="bg-white/20 border-white/30 text-white hover:bg-white hover:text-uct-blue transition-colors" />
              <CarouselNext className="bg-white/20 border-white/30 text-white hover:bg-white hover:text-uct-blue transition-colors" />
            </Carousel>
          </div>
        </div>
      </div>
      {/*Footer section*/}

      <Footer />
    </div>
    // <div className="w-full h-screen flex items-center justify-center">
    //   <ErrorPage />
    // </div>
  );
}

function MovingBorderCard({
  children,
  className,
  ...props
}: {
  children: React.ReactNode;
  className?: string;
  [key: string]: any;
}) {
  return (
    <div
      className={`relative overflow-hidden bg-black/10 p-[4px] ${className}`}
      style={{ borderRadius: "1rem" }}
      {...props}
    >
      <div
        className="absolute inset-0"
        style={{ borderRadius: "calc(1rem * 0.96)" }}
      >
        <MovingBorder duration={3000} rx="30%" ry="30%">
          <div className="h-40 w-40 bg-[radial-gradient(#0ea5e9_30%,transparent_60%)] opacity-[0.5]" />
        </MovingBorder>
      </div>
      <div
        className="relative flex h-full w-full items-center justify-center bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50  backdrop-blur-xl"
        style={{ borderRadius: "calc(1rem * 0.96)" }}
      >
        {children}
      </div>
    </div>
  );
}
