// hooks/useAskAdvisorData.tsx

import { useState, useEffect } from "react";
import type {
  Faculty,
  Program,
  Degree,
  Major,
  Department,
} from "@/components/ask-advisor/types";

interface UseAskAdvisorDataProps {
  initialFaculties: Faculty[];
}

export function useAskAdvisorData({
  initialFaculties,
}: UseAskAdvisorDataProps) {
  // Data states
  const [faculties] = useState<Faculty[]>(initialFaculties);
  const [programs, setPrograms] = useState<Program[]>([]);
  const [degrees, setDegrees] = useState<Degree[]>([]);
  const [majors, setMajors] = useState<Major[]>([]);
  const [departments, setDepartments] = useState<Department[]>([]);

  // Loading and error states
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  // Load departments on mount
  useEffect(() => {
    const loadDepartments = async () => {
      try {
        setLoading(true);
        const response = await fetch("/api/departments");

        if (!response.ok) {
          throw new Error("Failed to fetch departments");
        }

        const departmentsData = await response.json();
        setDepartments(departmentsData);
      } catch (err) {
        console.error("Error loading departments:", err);
        setError("Failed to load departments");
      } finally {
        setLoading(false);
      }
    };

    loadDepartments();
  }, []);

  // Load programs for a given faculty
  const loadPrograms = async (facultyCode: string) => {
    try {
      setLoading(true);
      const response = await fetch(`/api/faculties/${facultyCode}/programs`);

      if (!response.ok) {
        throw new Error("Failed to fetch programs");
      }

      const programsData = await response.json();
      setPrograms(programsData);
    } catch (err) {
      console.error("Error loading programs:", err);
      setError("Failed to load programs");
    } finally {
      setLoading(false);
    }
  };

  // Load degrees and programs for a given faculty
  const loadDegreesData = async (facultyCode: string) => {
    try {
      setLoading(true);
      const [programsResponse, degreesResponse] = await Promise.all([
        fetch(`/api/faculties/${facultyCode}/programs`),
        fetch(`/api/faculties/${facultyCode}/degrees`),
      ]);

      if (!programsResponse.ok || !degreesResponse.ok) {
        throw new Error("Failed to fetch programs or degrees");
      }

      const [programsData, degreesData] = await Promise.all([
        programsResponse.json(),
        degreesResponse.json(),
      ]);

      setPrograms(programsData);
      setDegrees(degreesData);
    } catch (err) {
      console.error("Error loading degrees data:", err);
      setError("Failed to load programs and degrees");
    } finally {
      setLoading(false);
    }
  };

  // Load specific degrees for a program
  const loadProgramDegrees = async (
    facultyCode: string,
    programCode: string
  ) => {
    try {
      setLoading(true);
      const response = await fetch(
        `/api/faculties/${facultyCode}/programs/${programCode}/degrees`
      );

      if (!response.ok) {
        throw new Error("Failed to fetch program degrees");
      }

      const degreesData = await response.json();
      setDegrees(degreesData);
    } catch (err) {
      console.error("Error loading program degrees:", err);
      setError("Failed to load program degrees");
    } finally {
      setLoading(false);
    }
  };

  // Load majors for a faculty and department (keeping the old method for department-based major loading)
  const loadMajors = async (facultyCode: string, departmentCode: string) => {
    try {
      setLoading(true);
      const response = await fetch(
        `/api/faculties/${facultyCode}/departments/${departmentCode}/majors`
      );

      if (!response.ok) {
        throw new Error("Failed to fetch majors");
      }

      const majorsData = await response.json();
      setMajors(majorsData);
    } catch (err) {
      console.error("Error loading majors:", err);
      setError("Failed to load majors");
    } finally {
      setLoading(false);
    }
  };

  // Load majors for a specific program (NEW METHOD)
  const loadProgramMajors = async (
    facultyCode: string,
    programCode: string
  ) => {
    try {
      setLoading(true);
      const response = await fetch(
        `/api/faculties/${facultyCode}/programs/${programCode}/majors`
      );

      if (!response.ok) {
        throw new Error("Failed to fetch program majors");
      }

      const majorsData = await response.json();
      setMajors(majorsData);
    } catch (err) {
      console.error("Error loading program majors:", err);
      setError("Failed to load program majors");
    } finally {
      setLoading(false);
    }
  };

  // Clear dependent data
  const clearPrograms = () => setPrograms([]);
  const clearDegrees = () => setDegrees([]);
  const clearMajors = () => setMajors([]);

  return {
    // Data
    faculties,
    programs,
    degrees,
    majors,
    departments,

    // States
    loading,
    error,

    // Actions
    loadPrograms,
    loadDegreesData,
    loadProgramDegrees,
    loadMajors,
    loadProgramMajors, // NEW METHOD ADDED
    clearPrograms,
    clearDegrees,
    clearMajors,
  };
}
