// hooks/useAskAdvisorForms.tsx

import { useState } from "react";
import { toast } from "sonner";
import { useAuth } from "@/components/AuthContext"; // Import the useAuth hook
import {
  StudentAdvisorAPI,
  type AdvisorQuestionRequest,
} from "@/services/api/ask-advisor";
import type {
  Faculty,
  Program,
  Degree,
  Major,
  Department,
  QuestionFormData,
  BaseQuestionData,
} from "@/components/ask-advisor/types";

interface UseAskAdvisorFormsProps {
  faculties: Faculty[];
  programs: Program[];
  degrees: Degree[];
  majors: Major[];
  departments: Department[];
  onDataLoad: {
    loadPrograms: (facultyCode: string) => Promise<void>;
    loadDegreesData: (facultyCode: string) => Promise<void>;
    loadProgramDegrees: (
      facultyCode: string,
      programCode: string
    ) => Promise<void>;
    loadMajors: (facultyCode: string, departmentCode: string) => Promise<void>;
    loadProgramMajors: (
      facultyCode: string,
      programCode: string
    ) => Promise<void>;
    clearPrograms: () => void;
    clearDegrees: () => void;
    clearMajors: () => void;
  };
}

export function useAskAdvisorForms({
  faculties,
  programs,
  degrees,
  majors,
  departments,
  onDataLoad,
}: UseAskAdvisorFormsProps) {
  // Get user data from AuthContext
  const { user } = useAuth();

  // Submission state
  const [isSubmitting, setIsSubmitting] = useState(false);

  // Selection states for each tab
  const [facultySelection, setFacultySelection] = useState("");
  const [departmentSelection, setDepartmentSelection] = useState("");
  const [programFacultySelection, setProgramFacultySelection] = useState("");
  const [programStudySelection, setProgramStudySelection] = useState("");
  const [degreeFacultySelection, setDegreeFacultySelection] = useState("");
  const [degreeProgramSelection, setDegreeProgramSelection] = useState("");
  const [finalDegreeSelection, setFinalDegreeSelection] = useState("");
  const [majorFacultySelection, setMajorFacultySelection] = useState("");
  const [majorDepartmentSelection, setMajorDepartmentSelection] = useState("");
  const [finalMajorSelection, setFinalMajorSelection] = useState("");

  // Form data states for each tab
  const [facultyFormData, setFacultyFormData] = useState<QuestionFormData>({
    subject: "",
    question: "",
  });
  const [departmentFormData, setDepartmentFormData] =
    useState<QuestionFormData>({
      subject: "",
      question: "",
    });
  const [programFormData, setProgramFormData] = useState<QuestionFormData>({
    subject: "",
    question: "",
  });
  const [degreeFormData, setDegreeFormData] = useState<QuestionFormData>({
    subject: "",
    question: "",
  });
  const [majorFormData, setMajorFormData] = useState<QuestionFormData>({
    subject: "",
    question: "",
  });

  // Handlers for program faculty changes
  const handleProgramFacultyChange = (value: string) => {
    setProgramFacultySelection(value);
    setProgramStudySelection("");
    onDataLoad.clearPrograms();
    if (value) {
      onDataLoad.loadPrograms(value);
    }
  };

  // Handlers for degree faculty changes - FIXED: Only load programs, not degrees
  const handleDegreeFacultyChange = (value: string) => {
    setDegreeFacultySelection(value);
    setDegreeProgramSelection("");
    setFinalDegreeSelection("");
    onDataLoad.clearPrograms();
    onDataLoad.clearDegrees();
    if (value) {
      // Only load programs for the faculty, not degrees yet
      onDataLoad.loadPrograms(value);
    }
  };

  // Handlers for degree program changes
  const handleDegreeProgramChange = (value: string) => {
    setDegreeProgramSelection(value);
    setFinalDegreeSelection("");
    if (degreeFacultySelection && value) {
      onDataLoad.loadProgramDegrees(degreeFacultySelection, value);
    }
  };

  // Handlers for major faculty changes
  const handleMajorFacultyChange = (value: string) => {
    setMajorFacultySelection(value);
    setMajorDepartmentSelection(""); // Clear program selection
    setFinalMajorSelection(""); // Clear major selection
    onDataLoad.clearPrograms(); // Clear existing programs
    onDataLoad.clearMajors(); // Clear existing majors
    if (value) {
      onDataLoad.loadPrograms(value); // Load programs for the selected faculty
    }
  };

  // Handlers for major program changes
  const handleMajorProgramChange = (value: string) => {
    setMajorDepartmentSelection(value); // Still using this state variable for the program selection
    setFinalMajorSelection("");
    if (majorFacultySelection && value) {
      onDataLoad.loadProgramMajors(majorFacultySelection, value);
    }
  };

  // Form submission handler
  const handleSubmit = async (category: string) => {
    setIsSubmitting(true);

    try {
      // Get the student name from the authenticated user
      const studentName =
        user && !user.is_advisor ? user.username : "Unknown Student";

      let questionData: AdvisorQuestionRequest = {
        category,
        subject: "",
        question: "",
        student: studentName, // Use the authenticated user's username
      };

      // Build the question data based on the category
      switch (category) {
        case "faculty":
          const selectedFacultyData = faculties.find(
            (f) => f.faculty_code === facultySelection
          );
          questionData = {
            ...questionData,
            faculty: selectedFacultyData?.faculty_name || "",
            subject: facultyFormData.subject,
            question: facultyFormData.question,
          };
          break;

        case "department":
          const selectedDepartmentData = departments.find(
            (d) => d.department_code === departmentSelection
          );
          questionData = {
            ...questionData,
            department: selectedDepartmentData?.department_name || "",
            subject: departmentFormData.subject,
            question: departmentFormData.question,
          };
          break;

        case "program":
          const selectedProgramFacultyData = faculties.find(
            (f) => f.faculty_code === programFacultySelection
          );
          const selectedProgramData = programs.find(
            (p) => p.program_code === programStudySelection
          );
          questionData = {
            ...questionData,
            faculty: selectedProgramFacultyData?.faculty_name || "",
            program: selectedProgramData?.program_name || "",
            subject: programFormData.subject,
            question: programFormData.question,
          };
          break;

        case "degree":
          const selectedDegreeFacultyData = faculties.find(
            (f) => f.faculty_code === degreeFacultySelection
          );
          const selectedDegreeProgramData = programs.find(
            (p) => p.program_code === degreeProgramSelection
          );
          const selectedDegreeData = degrees.find(
            (d) => d.degree_code === finalDegreeSelection
          );
          questionData = {
            ...questionData,
            faculty: selectedDegreeFacultyData?.faculty_name || "",
            program: selectedDegreeProgramData?.program_name || "",
            degree: selectedDegreeData?.degree_name || "",
            subject: degreeFormData.subject,
            question: degreeFormData.question,
          };
          break;

        case "major":
          const selectedMajorFacultyData = faculties.find(
            (f) => f.faculty_code === majorFacultySelection
          );
          const selectedMajorDepartmentData = departments.find(
            (d) => d.department_code === majorDepartmentSelection
          );
          const selectedMajorData = majors.find(
            (m) => m.major_code === finalMajorSelection
          );
          questionData = {
            ...questionData,
            faculty: selectedMajorFacultyData?.faculty_name || "",
            department: selectedMajorDepartmentData?.department_name || "",
            major: selectedMajorData?.major_name || "",
            subject: majorFormData.subject,
            question: majorFormData.question,
          };
          break;

        default:
          throw new Error("Invalid category");
      }

      // Validate required fields
      if (!questionData.subject.trim() || !questionData.question.trim()) {
        throw new Error("Subject and question are required");
      }

      const result = await StudentAdvisorAPI.submitQuestion(questionData);

      if (result.success) {
        toast.success("Question submitted successfully!", {
          description: result.message,
        });
        // Reset form fields after successful submission
        resetFormFields(category);
        console.log("Question submitted successfully:", result);
      } else {
        toast.error("Failed to submit question", {
          description: result.message,
        });
      }
    } catch (error) {
      const errorMessage =
        error instanceof Error ? error.message : "Failed to submit question";
      toast.error("Submission failed", {
        description: errorMessage,
      });
      console.error(`Error submitting ${category} question:`, error);
    } finally {
      setIsSubmitting(false);
    }
  };

  // Reset form fields
  const resetFormFields = (category: string) => {
    switch (category) {
      case "faculty":
        setFacultySelection("");
        setFacultyFormData({ subject: "", question: "" });
        break;
      case "department":
        setDepartmentSelection("");
        setDepartmentFormData({ subject: "", question: "" });
        break;
      case "program":
        setProgramFacultySelection("");
        setProgramStudySelection("");
        setProgramFormData({ subject: "", question: "" });
        onDataLoad.clearPrograms();
        break;
      case "degree":
        setDegreeFacultySelection("");
        setDegreeProgramSelection("");
        setFinalDegreeSelection("");
        setDegreeFormData({ subject: "", question: "" });
        onDataLoad.clearPrograms();
        onDataLoad.clearDegrees();
        break;
      case "major":
        setMajorFacultySelection("");
        setMajorDepartmentSelection("");
        setFinalMajorSelection("");
        setMajorFormData({ subject: "", question: "" });
        onDataLoad.clearPrograms();
        onDataLoad.clearMajors();
        break;
    }
  };

  return {
    // Submission state
    isSubmitting,

    // Selection states
    facultySelection,
    setFacultySelection,
    departmentSelection,
    setDepartmentSelection,
    programFacultySelection,
    programStudySelection,
    setProgramStudySelection,
    degreeFacultySelection,
    degreeProgramSelection,
    finalDegreeSelection,
    setFinalDegreeSelection,
    majorFacultySelection,
    majorDepartmentSelection,
    finalMajorSelection,
    setFinalMajorSelection,

    // Form data states
    facultyFormData,
    setFacultyFormData,
    departmentFormData,
    setDepartmentFormData,
    programFormData,
    setProgramFormData,
    degreeFormData,
    setDegreeFormData,
    majorFormData,
    setMajorFormData,

    // Handlers
    handleProgramFacultyChange,
    handleDegreeFacultyChange,
    handleDegreeProgramChange,
    handleMajorFacultyChange,
    handleMajorProgramChange,
    handleSubmit,
    resetFormFields,
  };
}
