"use client";

import type React from "react";

import { useEffect, useRef, useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Calculator,
  Calendar,
  Monitor,
  ArrowUp,
  Bot,
  User,
  Loader2,
} from "lucide-react";
import { Navbar } from "@/components/Navbar";
import lottie, { type AnimationItem } from "lottie-web";
import { AnimatePresence, motion } from "framer-motion";
import {
  Lightbulb,
  PenTool,
  FileText,
  MessageCircle,
  MoreHorizontal,
} from "lucide-react";
import { StudentQuestionClient } from "@/services/api/studentQuestionClient";

interface Message {
  id: string;
  role: "user" | "assistant";
  content: string;
  isStreaming?: boolean;
  timestamp: Date;
}

interface ChatContext {
  question: string;
  answer: string;
}

export default function Page() {
  const [selectedSection, setSelectedSection] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [showInitialContent, setShowInitialContent] = useState(true);
  const [messages, setMessages] = useState<Message[]>([]);
  const [isStreaming, setIsStreaming] = useState(false);
  const [chatContext, setChatContext] = useState<ChatContext[]>([]);
  const [client] = useState(
    () => new StudentQuestionClient("https://myadvisor.cs.uct.ac.za/bad")
  );
  const sphereContainerRef = useRef<HTMLDivElement | null>(null);
  const messagesEndRef = useRef<HTMLDivElement>(null);

  const MAX_CONTEXT_LENGTH = 2; // Keep last 2 Q&A pairs

  useEffect(() => {
    let animation: AnimationItem | undefined;

    if (sphereContainerRef.current) {
      animation = lottie.loadAnimation({
        container: sphereContainerRef.current,
        path: "/images/Circle_2.json",
        renderer: "svg",
        loop: true,
        autoplay: true,
      });
    }

    return () => {
      animation?.destroy(); // ✅ Prevent duplicate renders
    };
  }, []);

  const scrollToBottom = () => {
    setTimeout(() => {
      window.scrollTo({
        top: document.documentElement.scrollHeight,
        behavior: "smooth",
      });
    }, 100);
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  useEffect(() => {
    if (!isStreaming) {
      scrollToBottom();
    }
  }, [isStreaming]);

  const formatTime = (date: Date) => {
    return date.toLocaleTimeString([], { hour: "2-digit", minute: "2-digit" });
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!inputValue.trim() || isStreaming) return;

    // If this is the first message, trigger the fade animation
    if (showInitialContent) {
      setShowInitialContent(false);
    }

    const userMessage: Message = {
      id: Date.now().toString(),
      role: "user",
      content: inputValue.trim(),
      timestamp: new Date(),
    };

    const assistantMessage: Message = {
      id: (Date.now() + 1).toString(),
      role: "assistant",
      content: "",
      isStreaming: true,
      timestamp: new Date(),
    };

    setMessages((prev) => [...prev, userMessage, assistantMessage]);
    setInputValue("");
    setIsStreaming(true);

    // Prepare context for the API call
    const contextForApi = chatContext.slice(0, MAX_CONTEXT_LENGTH);

    await client.askQuestionStream(
      userMessage.content,
      contextForApi, // Pass the context
      // onChunk
      (chunk) => {
        if (chunk.type === "text") {
          setMessages((prev) =>
            prev.map((msg) =>
              msg.id === assistantMessage.id
                ? { ...msg, content: msg.content + chunk.content }
                : msg
            )
          );
        }
      },
      // onComplete
      (fullText) => {
        setMessages((prev) =>
          prev.map((msg) =>
            msg.id === assistantMessage.id
              ? { ...msg, content: fullText, isStreaming: false }
              : msg
          )
        );

        // Update the chat context with the new Q&A pair
        setChatContext((prev) => {
          const newContext = [
            { question: userMessage.content, answer: fullText },
            ...prev,
          ];

          // Keep only the last MAX_CONTEXT_LENGTH pairs (FIFO)
          return newContext.slice(0, MAX_CONTEXT_LENGTH);
        });

        setIsStreaming(false);
      },
      // onError
      (error) => {
        setMessages((prev) =>
          prev.map((msg) =>
            msg.id === assistantMessage.id
              ? {
                  ...msg,
                  content: `Error: ${error}`,
                  isStreaming: false,
                }
              : msg
          )
        );
        setIsStreaming(false);
      }
    );
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter") {
      handleSubmit(e);
    }
  };

  const handleCardClick = (prompt: string) => {
    setInputValue(prompt);
    setShowInitialContent(false);
    // Clear context when starting a new conversation from cards
    setChatContext([]);

    setTimeout(() => {
      const event = new Event("submit") as any;
      handleSubmit(event);
    }, 100);
  };

  const clearChatContext = () => {
    setChatContext([]);
    setMessages([]);
    setShowInitialContent(true);
  };

  return (
    <div className="relative flex flex-col items-center min-h-full bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50">
      <div className="absolute inset-0 z-0 min-h-full">
        {/* Soft Gradient Overlay */}
        <div className="absolute inset-0 bg-gradient-to-br from-rose-100/40 via-sky-100/30 to-violet-100/40 min-h-full" />
      </div>

      {/* Desktop Layout */}
      <div className="hidden md:flex min-h-screen w-full flex-col z-10">
        <Navbar hideOnScroll={false} />

        {/* Main content area - takes remaining height */}
        <div className="flex-1 flex flex-col relative">
          <div className="flex-1 w-4xl mx-auto px-6 flex flex-col min-h-0">
            {/* Content area that grows to fill available space */}
            <div
              className={`flex-1 relative flex flex-col ${
                !showInitialContent ? "" : "pb-24"
              }`}
            >
              {/* Initial content - State 1 (unchanged) */}
              <AnimatePresence initial={false}>
                {showInitialContent && (
                  <motion.div
                    initial={{ opacity: 1, y: 0 }}
                    animate={{ opacity: 1, y: 0 }}
                    exit={{
                      opacity: 0,
                      y: -50,
                      transition: {
                        duration: 0.5,
                        ease: "easeInOut",
                      },
                    }}
                    key="initial-content"
                    className="absolute inset-0 flex flex-col items-center justify-center pb-40"
                  >
                    {/* Lottie sphere animation */}
                    <motion.div
                      className="flex w-full items-center justify-center text-center gap-3"
                      exit={{
                        scale: 0.8,
                        opacity: 0,
                        transition: {
                          duration: 0.4,
                          ease: "easeInOut",
                        },
                      }}
                    >
                      <div
                        ref={sphereContainerRef}
                        style={{ width: 250, height: 250 }}
                      />
                    </motion.div>

                    {/* Welcome text */}
                    <motion.div
                      className="text-center "
                      exit={{
                        y: -30,
                        opacity: 0,
                        transition: {
                          duration: 0.4,
                          delay: 0.1,
                          ease: "easeInOut",
                        },
                      }}
                    >
                      <p className="text-black text-5xl outfit-300">
                        Good Evening
                      </p>
                      <p className="text-black text-5xl outfit-300">
                        Can I help you with anything?
                      </p>
                    </motion.div>

                    {/* Subtitle */}
                    <motion.div
                      className="flex justify-center items-center mx-auto text-center mb-6 w-md mt-6"
                      exit={{
                        y: -20,
                        opacity: 0,
                        transition: {
                          duration: 0.4,
                          delay: 0.15,
                          ease: "easeInOut",
                        },
                      }}
                    >
                      <h1 className="text-[#6e6e6e] outfit-300 text-lg font-lg">
                        Choose a prompt below or write your own to start
                        chatting with MyAdvisor
                      </h1>
                    </motion.div>

                    {/* Question cards */}
                    <motion.div
                      className="grid grid-cols-3 gap-6 h-32 mt-4"
                      exit={{
                        y: -40,
                        opacity: 0,
                        transition: {
                          duration: 0.5,
                          delay: 0.2,
                          ease: "easeInOut",
                        },
                      }}
                    >
                      {/* Card 1 */}
                      <motion.div
                        onClick={() =>
                          handleCardClick("Help me calculate credits")
                        }
                        className="bg-gradient-to-br from-pink-200 via-purple-100 to-purple-200 drop-shadow-sm border-4 border-neutral-100  p-4 rounded-xl shadow-sm hover:shadow-md transition-shadow cursor-pointer"
                        exit={{
                          scale: 0.9,
                          opacity: 0,
                          transition: {
                            duration: 0.3,
                            delay: 0.25,
                          },
                        }}
                      >
                        <div className="h-full flex flex-col items-start space-y-3">
                          <div className="text-black w-10/12 outfit-300 text-lg leading-tight px-4 ">
                            Help me calculate credits
                          </div>
                          <div className="ml-auto">
                            <div className="w-10 h-10 bg-white border-2 border-neutral-300 rounded-lg flex items-center justify-center">
                              <Calculator className="w-5 h-5 text-[#222222]" />
                            </div>
                          </div>
                        </div>
                      </motion.div>

                      {/* Card 2 */}
                      <motion.div
                        onClick={() =>
                          handleCardClick("Help me create a roadmap")
                        }
                        className=" bg-gradient-to-br from-yellow-50 via-orange-50 to-cyan-200 drop-shadow-sm p-4 border-4 border-neutral-100 rounded-xl shadow-sm hover:shadow-md transition-shadow cursor-pointer"
                        exit={{
                          scale: 0.9,
                          opacity: 0,
                          transition: {
                            duration: 0.3,
                            delay: 0.3,
                          },
                        }}
                      >
                        <div className="h-full flex flex-col items-start space-y-3">
                          <div className="text-black w-10/12 outfit-300 text-lg leading-tight px-4 ">
                            Help me create a roadmap
                          </div>
                          <div className="ml-auto">
                            <div className="w-10 h-10 bg-white border-2 border-neutral-300 rounded-lg flex items-center justify-center">
                              <Calendar className="w-5 h-5 text-[#222222]" />
                            </div>
                          </div>
                        </div>
                      </motion.div>

                      {/* Card 3 */}
                      <motion.div
                        onClick={() =>
                          handleCardClick("Help me find my courses")
                        }
                        className="bg-gradient-to-br from-violet-200 via-purple-100 to-pink-300 drop-shadow-sm p-4 border-4 border-neutral-100 rounded-xl shadow-sm hover:shadow-md transition-shadow cursor-pointer"
                        exit={{
                          scale: 0.9,
                          opacity: 0,
                          transition: {
                            duration: 0.3,
                            delay: 0.35,
                          },
                        }}
                      >
                        <div className="h-full flex flex-col items-start space-y-3">
                          <div className="text-black w-10/12 outfit-300 text-lg leading-tight px-4">
                            Help me find my courses
                          </div>
                          <div className="ml-auto">
                            <div className="w-10 h-10 bg-white border-2 border-neutral-300 rounded-lg flex items-center justify-center">
                              <Monitor className="w-5 h-5 text-[#222222]" />
                            </div>
                          </div>
                        </div>
                      </motion.div>
                    </motion.div>
                  </motion.div>
                )}
              </AnimatePresence>

              {/* Chat area - State 2 (fills available space, no internal scroll) */}
              <AnimatePresence>
                {!showInitialContent && (
                  <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.5, delay: 0.3 }}
                    className="flex-1 px-4 pt-12 py-6 space-y-6 min-h-0 pb-40"
                  >
                    {messages.map((message) => (
                      <div
                        key={message.id}
                        className={`flex gap-4 ${
                          message.role === "user"
                            ? "justify-end"
                            : "justify-start"
                        }`}
                      >
                        {message.role === "assistant" && (
                          <div className="w-10 h-10 rounded-full bg-gradient-to-br from-blue-100 to-purple-100 border-2 border-white shadow-sm flex items-center justify-center flex-shrink-0 mt-1">
                            <Bot className="w-5 h-5 text-blue-600" />
                          </div>
                        )}

                        <div
                          className={`max-w-[80%] ${
                            message.role === "user" ? "order-1" : ""
                          }`}
                        >
                          <div
                            className={`px-6 py-4 rounded-2xl shadow-sm ${
                              message.role === "user"
                                ? "bg-gradient-to-br from-blue-500 to-blue-600 text-white"
                                : "bg-white border border-gray-100"
                            }`}
                          >
                            <div className="whitespace-pre-wrap outfit-300 text-base leading-relaxed">
                              {message.content}
                              {message.isStreaming && (
                                <span className="inline-flex items-center ml-2">
                                  <div className="flex space-x-1">
                                    <div className="w-2 h-2 bg-blue-400 rounded-full animate-pulse"></div>
                                    <div className="w-2 h-2 bg-blue-400 rounded-full animate-pulse delay-75"></div>
                                    <div className="w-2 h-2 bg-blue-400 rounded-full animate-pulse delay-150"></div>
                                  </div>
                                </span>
                              )}
                            </div>
                          </div>
                          <div
                            className={`text-xs text-gray-400 mt-2 outfit-300 ${
                              message.role === "user"
                                ? "text-right"
                                : "text-left"
                            }`}
                          >
                            {formatTime(message.timestamp)}
                          </div>
                        </div>

                        {message.role === "user" && (
                          <div className="w-10 h-10 rounded-full bg-gradient-to-br from-gray-100 to-gray-200 border-2 border-white shadow-sm flex items-center justify-center flex-shrink-0 mt-1 order-2">
                            <User className="w-5 h-5 text-gray-600" />
                          </div>
                        )}
                      </div>
                    ))}
                    <div ref={messagesEndRef} className="h-8" />
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          </div>
        </div>

        {/* Input area - State 1 (with transparent background) */}
        {showInitialContent && (
          <>
            {/* Invisible placeholder with same dimensions as fixed input to prevent layout shift */}
            <div className="invisible fixed bottom-0 left-0 right-0 border-t border-gray-100 z-20">
              <div className="w-4xl mx-auto px-6 py-6">
                <div className="h-16"></div>
              </div>
            </div>

            <div className="fixed bottom-0 left-0 right-0 bg-transparent border-t border-transparent z-20">
              <div className="w-4xl mx-auto px-6 py-6">
                <form onSubmit={handleSubmit}>
                  <div className="relative">
                    <Input
                      type="text"
                      placeholder="Ask anything"
                      value={inputValue}
                      onChange={(e) => setInputValue(e.target.value)}
                      onKeyPress={handleKeyPress}
                      disabled={isStreaming}
                      className="w-full h-16 pl-6 pr-24 text-lg bg-white border-2 border-[#d6d6d6] text-black rounded-3xl shadow-sm focus:ring-2 focus:ring-[#008fd1] focus:border-transparent placeholder:text-[#a9a9a9]"
                    />
                    <div className="absolute right-5 top-1/2 transform -translate-y-1/2 flex items-center space-x-3">
                      <motion.button
                        type="submit"
                        className="w-8 h-8 bg-[#4a4a4a] rounded-full flex items-center justify-center cursor-pointer hover:bg-[#2f2f2f] disabled:opacity-50 disabled:cursor-not-allowed"
                        whileTap={{ scale: 0.95 }}
                        disabled={!inputValue.trim() || isStreaming}
                      >
                        {isStreaming ? (
                          <Loader2 className="w-4 h-4 text-white animate-spin" />
                        ) : (
                          <ArrowUp className="w-4 h-4 text-white" />
                        )}
                      </motion.button>
                    </div>
                  </div>
                </form>
              </div>
            </div>
          </>
        )}

        {/* Input area - State 2 (fixed at bottom) */}
        {!showInitialContent && (
          <div className="fixed bottom-0 left-0 right-0 bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50 border-t-2 border-gray-200 z-20">
            <div className="w-4xl mx-auto px-6 py-6">
              <form onSubmit={handleSubmit}>
                <div className="relative">
                  <Input
                    type="text"
                    placeholder="Ask anything"
                    value={inputValue}
                    onChange={(e) => setInputValue(e.target.value)}
                    onKeyPress={handleKeyPress}
                    disabled={isStreaming}
                    className="w-full h-16 pl-6 pr-24 text-lg bg-white border-2 border-[#d6d6d6] text-black rounded-3xl shadow-sm focus:ring-2 focus:ring-[#008fd1] focus:border-transparent placeholder:text-[#a9a9a9]"
                  />
                  <div className="absolute right-5 top-1/2 transform -translate-y-1/2 flex items-center space-x-3">
                    <motion.button
                      type="submit"
                      className="w-8 h-8 bg-[#4a4a4a] rounded-full flex items-center justify-center cursor-pointer hover:bg-[#2f2f2f] disabled:opacity-50 disabled:cursor-not-allowed"
                      whileTap={{ scale: 0.95 }}
                      disabled={!inputValue.trim() || isStreaming}
                    >
                      {isStreaming ? (
                        <Loader2 className="w-4 h-4 text-white animate-spin" />
                      ) : (
                        <ArrowUp className="w-4 h-4 text-white" />
                      )}
                    </motion.button>
                  </div>
                </div>
              </form>
            </div>
          </div>
        )}
      </div>

      {/* Mobile Layout (unchanged) */}
      {/* Mobile Layout */}
      <div className="min-h-screen w-full flex flex-col z-10 md:hidden">
        <Navbar hideOnScroll={false} />

        {/* Main content area - takes remaining height */}
        <div className="flex-1 flex flex-col relative">
          <div className="flex-1 w-full mx-auto px-4 flex flex-col min-h-0">
            {/* Content area that grows to fill available space */}
            <div
              className={`flex-1 relative flex flex-col ${
                !showInitialContent ? "" : "pb-32"
              }`}
            >
              {/* Initial content - Mobile */}
              <AnimatePresence initial={false}>
                {showInitialContent && (
                  <motion.div
                    initial={{ opacity: 1, y: 0 }}
                    animate={{ opacity: 1, y: 0 }}
                    exit={{
                      opacity: 0,
                      y: -50,
                      transition: {
                        duration: 0.5,
                        ease: "easeInOut",
                      },
                    }}
                    key="mobile-initial-content"
                    className="absolute inset-0 flex flex-col items-center justify-center"
                  >
                    {/* Mobile Welcome Content */}
                    <motion.div
                      className="text-center mb-8"
                      exit={{
                        y: -30,
                        opacity: 0,
                        transition: {
                          duration: 0.4,
                          delay: 0.1,
                          ease: "easeInOut",
                        },
                      }}
                    >
                      <div className="outfit-400 text-3xl mb-4">MyAdvisor</div>
                      <p className="text-[#6e6e6e] outfit-300 text-base">
                        Choose a prompt below or write your own to start
                      </p>
                    </motion.div>

                    {/* Mobile Buttons */}
                    <motion.div
                      className="space-y-3 max-w-sm mx-auto"
                      exit={{
                        y: -40,
                        opacity: 0,
                        transition: {
                          duration: 0.5,
                          delay: 0.2,
                          ease: "easeInOut",
                        },
                      }}
                    >
                      <div className="flex gap-3">
                        <Button
                          variant="outline"
                          onClick={() =>
                            handleCardClick("Help me calculate credits")
                          }
                          className="flex-1 bg-transparent border-1 border-[#585858] text-gray-800 rounded-full py-3 px-4 flex items-center gap-2 justify-start outfit-400"
                        >
                          <Calculator className="w-4 h-4 text-blue-600" />
                          <span>Calculate credits</span>
                        </Button>
                        <Button
                          variant="outline"
                          onClick={() =>
                            handleCardClick("Help me create a roadmap")
                          }
                          className="flex-1 bg-transparent border-1 border-[#585858] text-gray-800 rounded-full py-3 px-4 flex items-center gap-2 justify-start outfit-400"
                        >
                          <Calendar className="w-4 h-4 text-orange-500" />
                          <span>Create roadmap</span>
                        </Button>
                      </div>

                      <div className="flex gap-3">
                        <Button
                          variant="outline"
                          onClick={() =>
                            handleCardClick("Help me find my courses")
                          }
                          className="flex-1 bg-transparent border-1 border-[#585858] text-gray-800 rounded-full py-3 px-4 flex items-center gap-2 justify-start outfit-400"
                        >
                          <Monitor className="w-4 h-4 text-purple-500" />
                          <span>Find courses</span>
                        </Button>
                        <Button
                          variant="outline"
                          onClick={() => handleCardClick("Get academic advice")}
                          className="flex-1 bg-transparent border-1 border-[#585858] text-gray-800 rounded-full py-3 px-4 flex items-center gap-2 justify-start outfit-400"
                        >
                          <MessageCircle className="w-4 h-4 text-blue-500" />
                          <span>Get advice</span>
                        </Button>
                      </div>

                      <div className="flex justify-center">
                        <Button
                          variant="outline"
                          className="bg-transparent border-1 border-[#585858] text-gray-800 rounded-full py-3 px-8 flex items-center gap-2 outfit-400"
                        >
                          <MoreHorizontal className="w-4 h-4" />
                          <span>More</span>
                        </Button>
                      </div>
                    </motion.div>
                  </motion.div>
                )}
              </AnimatePresence>

              {/* Chat area - Mobile */}
              <AnimatePresence>
                {!showInitialContent && (
                  <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.5, delay: 0.3 }}
                    className="absolute top-20 left-0 right-0 bottom-20 px-2 py-4 space-y-4 overflow-y-auto"
                    style={{
                      minHeight: "calc(100vh - 160px)",
                      paddingTop: "2rem",
                    }}
                  >
                    {messages.map((message) => (
                      <div
                        key={message.id}
                        className={`flex gap-3 ${
                          message.role === "user"
                            ? "justify-end"
                            : "justify-start"
                        }`}
                      >
                        {message.role === "assistant" && (
                          <div className="w-8 h-8 rounded-full bg-gradient-to-br from-blue-100 to-purple-100 border-2 border-white shadow-sm flex items-center justify-center flex-shrink-0 mt-1">
                            <Bot className="w-4 h-4 text-blue-600" />
                          </div>
                        )}

                        <div
                          className={`max-w-[85%] ${
                            message.role === "user" ? "order-1" : ""
                          }`}
                        >
                          <div
                            className={`px-4 py-3 rounded-2xl shadow-sm ${
                              message.role === "user"
                                ? "bg-gradient-to-br from-blue-500 to-blue-600 text-white"
                                : "bg-white border border-gray-100"
                            }`}
                          >
                            <div className="whitespace-pre-wrap outfit-300 text-sm leading-relaxed">
                              {message.content}
                              {message.isStreaming && (
                                <span className="inline-flex items-center ml-2">
                                  <div className="flex space-x-1">
                                    <div className="w-1.5 h-1.5 bg-blue-400 rounded-full animate-pulse"></div>
                                    <div className="w-1.5 h-1.5 bg-blue-400 rounded-full animate-pulse delay-75"></div>
                                    <div className="w-1.5 h-1.5 bg-blue-400 rounded-full animate-pulse delay-150"></div>
                                  </div>
                                </span>
                              )}
                            </div>
                          </div>
                          <div
                            className={`text-xs text-gray-400 mt-1 outfit-300 ${
                              message.role === "user"
                                ? "text-right"
                                : "text-left"
                            }`}
                          >
                            {formatTime(message.timestamp)}
                          </div>
                        </div>

                        {message.role === "user" && (
                          <div className="w-8 h-8 rounded-full bg-gradient-to-br from-gray-100 to-gray-200 border-2 border-white shadow-sm flex items-center justify-center flex-shrink-0 mt-1 order-2">
                            <User className="w-4 h-4 text-gray-600" />
                          </div>
                        )}
                      </div>
                    ))}
                    <div ref={messagesEndRef} className="h-4" />
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          </div>
        </div>

        {/* Mobile Input Bar - Fixed at Bottom with Gradient Background */}
        <div className="fixed bottom-0 left-0 right-0 bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50 border-t-2 border-gray-200 z-20 md:hidden">
          <div className="w-full px-4 py-4">
            <form onSubmit={handleSubmit}>
              <div className="relative">
                <Input
                  type="text"
                  placeholder="Ask anything"
                  value={inputValue}
                  onChange={(e) => setInputValue(e.target.value)}
                  onKeyPress={handleKeyPress}
                  disabled={isStreaming}
                  className="w-full h-14 pl-5 pr-20 text-base bg-white border-2 border-[#d6d6d6] text-black rounded-3xl shadow-sm focus:ring-2 focus:ring-[#008fd1] focus:border-transparent placeholder:text-[#a9a9a9]"
                />
                <div className="absolute right-4 top-1/2 transform -translate-y-1/2 flex items-center space-x-3">
                  <motion.button
                    type="submit"
                    className="w-7 h-7 bg-[#4a4a4a] rounded-full flex items-center justify-center cursor-pointer hover:bg-[#2f2f2f] disabled:opacity-50 disabled:cursor-not-allowed"
                    whileTap={{ scale: 0.95 }}
                    disabled={!inputValue.trim() || isStreaming}
                  >
                    {isStreaming ? (
                      <Loader2 className="w-3.5 h-3.5 text-white animate-spin" />
                    ) : (
                      <ArrowUp className="w-3.5 h-3.5 text-white" />
                    )}
                  </motion.button>
                </div>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}
