"use client";

import React, { useState, useEffect } from "react";
import { Navbar } from "@/components/Navbar";
import Link from "next/link";
import { Input } from "@/components/ui/input";
import { Footer } from "@/components/Footer";
import { Button } from "@/components/ui/button";
import ErrorPage from "@/components/ErrorPage";
import { Department } from "@/services/api/courses";
import {
  Card,
  CardAction,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";

export default function Page() {
  const [departments, setDepartments] = useState<Department[]>([]);
  const [error, setError] = useState<string | null>(null);
  const [search, setSearch] = useState("");
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchDepartments = async () => {
      try {
        const response = await fetch("/api/departments", {
          method: "GET",
          headers: {
            Accept: "application/json",
          },
        });

        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || "Failed to fetch departments");
        }

        const fetchedDepartments = await response.json();
        setDepartments(fetchedDepartments);
      } catch (err) {
        console.error("Error fetching departments:", err);
        setError(
          err instanceof Error ? err.message : "Failed to fetch departments"
        );
      } finally {
        setLoading(false);
      }
    };

    fetchDepartments();
  }, []);

  // Group departments by first letter and filter based on search
  const filteredDepartments = Array.from({ length: 26 }, (_, i) => {
    const letter = String.fromCharCode(65 + i);

    // Get all departments that start with this letter
    const departmentsForLetter = departments.filter((dept) =>
      dept.department_name.toLowerCase().startsWith(letter.toLowerCase())
    );

    // If searching, filter departments by name or HOD (case-insensitive)
    const filteredDepts = search.trim()
      ? departmentsForLetter.filter(
          (dept) =>
            dept.department_name
              .toLowerCase()
              .includes(search.trim().toLowerCase()) ||
            (dept.hod &&
              dept.hod.toLowerCase().includes(search.trim().toLowerCase()))
        )
      : departmentsForLetter;

    return { letter, departments: filteredDepts };
  }).filter((section) => section.departments.length > 0);

  if (loading) {
    return (
      <div className="relative flex flex-col items-center min-h-screen pt-5 bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50">
        <div className="absolute inset-0 z-0">
          <div className="absolute inset-0 bg-gradient-to-br from-rose-100/40 via-sky-100/30 to-violet-100/40" />
        </div>
        <Navbar />
        <div className="flex items-center justify-center mt-32">
          <div className="text-xl text-gray-600">Loading departments...</div>
        </div>
      </div>
    );
  }

  if (error) {
    return <ErrorPage />;
  }
  return (
    <div className="relative flex flex-col items-center min-h-screen pt-5 bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50">
      <div className="absolute inset-0 z-0">
        {/* Soft Gradient Overlay */}
        <div className="absolute inset-0 bg-gradient-to-br from-rose-100/40 via-sky-100/30 to-violet-100/40" />
      </div>
      {/* Nav Bar */}
      <Navbar />
      {/*Big heading*/}
      <div className="w-full flex flex-col items-center justify-center text-center mt-16 md:mt-10 outfit-400 z-10">
        <div className="text-5xl md:text-8xl mb-1 mt-5">
          Course <span className="text-uct-blue">Finder</span>
        </div>
        <div className="text-xl md:text-3xl w-full text-gray-500 mt-5 md:mt-10">
          Select a Department
        </div>
      </div>

      {/* Search Bar */}
      <form
        className="w-full md:w-1/3 px-8 md:px-0 h-fit mt-10 z-10"
        onSubmit={(e) => e.preventDefault()}
        autoComplete="off"
      >
        <Input
          type="text"
          value={search}
          onChange={(e) => setSearch(e.target.value)}
          placeholder="Search departments or head of department..."
          className="rounded-full border-1 border-gray-300 bg-white text-sm"
        />
      </form>

      {/* Alphabet buttons */}
      <div className="w-full flex justify-center mt-8 border-b-2 border-gray-200 pb-8 z-10 px-4 md:px-0"></div>

      {/* Departments Content */}
      <div className="w-full flex flex-col items-center mt-12 z-10">
        {filteredDepartments.length === 0 && (
          <div className="text-gray-500 mt-8 text-xl">
            No departments found.
          </div>
        )}
        {filteredDepartments.map(({ letter, departments: depts }) => (
          <div
            key={letter}
            id={letter}
            className="w-11/12 md:w-9/12 flex flex-col md:flex-row md:items-start mb-16 scroll-mt-32"
          >
            {/* Big Letter */}
            <div className="w-full md:w-32 flex-shrink-0 flex items-center justify-start px-4 md:px-0 md:justify-center md:pr-8 mb-6 md:mb-0">
              <span className="text-8xl font-semibold text-uct-blue">
                {letter}
              </span>
            </div>
            {/* Cards */}
            <div className="flex-1 grid grid-cols-1 md:grid-cols-2 gap-6">
              {depts.map((entry) => (
                <Link
                  key={entry.department_code}
                  href={`/course_info/courses/${entry.department_code}`}
                  className="block"
                >
                  <Card className="text-left cursor-pointer hover:shadow-lg transition-shadow">
                    <CardHeader>
                      <CardTitle className="text-xl">
                        {entry.department_name}
                      </CardTitle>
                    </CardHeader>
                    {entry.hod && (
                      <CardContent>
                        <p className="text-gray-700">
                          Head of Department: {entry.hod}
                        </p>
                      </CardContent>
                    )}
                  </Card>
                </Link>
              ))}
            </div>
          </div>
        ))}
      </div>
      <Footer />
    </div>
  );
}
