// app/api/roadmap/route.ts
import { NextRequest, NextResponse } from 'next/server';

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    console.log('Roadmap submission data:', body);
    
    // Create AbortController for timeout
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 60000); // 60 seconds = 1 minute
    
    const res = await fetch(
      `https://myadvisor.cs.uct.ac.za/backend/roadmap_result/`,
      { 
        method: "POST",
        cache: 'no-store',
        headers: {
          'Accept': 'application/json',
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(body),
        signal: controller.signal // Add timeout signal
      }
    );

    // Clear the timeout since the request completed
    clearTimeout(timeoutId);

    console.log('External API response status:', res.status);

    if (!res.ok) {
      console.error('External API error:', res.statusText);
      return NextResponse.json(
        { error: `Failed to submit roadmap: ${res.statusText}` },
        { status: res.status }
      );
    }

    const data = await res.json();
    console.log('External API response data:', data);
    return NextResponse.json(data);
  } catch (error) {
    console.error('API Route error:', error);
    
    // Handle timeout specifically
    if (error instanceof Error && error.name === 'AbortError') {
      return NextResponse.json(
        { 
          error: 'Request timeout', 
          details: 'The request took longer than 60 seconds to complete'
        },
        { status: 408 }
      );
    }
    
    return NextResponse.json(
      { 
        error: 'Internal server error', 
        details: error instanceof Error ? error.message : 'Unknown error occurred'
      },
      { status: 500 }
    );
  }
}