import { NextRequest, NextResponse } from 'next/server';

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    // Validate required fields
    const { username, is_advisor, inbox } = body;
    
    if (!username || typeof is_advisor !== 'boolean' || !Array.isArray(inbox)) {
      return NextResponse.json(
        { error: 'Missing required fields: username, is_advisor, inbox' },
        { status: 400 }
      );
    }

    // console.log('Refreshing user data:', body);
    
    const res = await fetch(
      'https://myadvisor.cs.uct.ac.za/backend/refresh/',
      { 
        method: "POST",
        cache: 'no-store',
        headers: {
          'Accept': 'application/json',
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          username,
          is_advisor,
          inbox
        })
      }
    );

    // console.log('External API response status:', res.status);

    if (!res.ok) {
      console.error('External API error:', res.statusText);
      const errorText = await res.text();
      console.error('External API error details:', errorText);
      
      return NextResponse.json(
        { error: `Failed to refresh user data: ${res.statusText}` },
        { status: res.status }
      );
    }

    const data = await res.json();
    // console.log('User data refreshed successfully:', data);
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('API Route error:', error);
    return NextResponse.json(
      { 
        error: 'Internal server error', 
        details: error instanceof Error ? error.message : 'Unknown error occurred'
      },
      { status: 500 }
    );
  }
}