import { NextRequest, NextResponse } from 'next/server';

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    // Validate required fields
    const { subject, question, answer, student, advisor } = body;
    
    if (!subject || !question || !answer || !student || !advisor) {
      return NextResponse.json(
        { error: 'Missing required fields: subject, question, answer, student, advisor' },
        { status: 400 }
      );
    }

    console.log('Submitting answer with data:', body);
    
    const res = await fetch(
      'https://myadvisor.cs.uct.ac.za/backend/answer_question/',
      { 
        method: "POST",
        cache: 'no-store',
        headers: {
          'Accept': 'application/json',
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          subject,
          question,
          answer,
          student,
          advisor
        })
      }
    );

    console.log('External API response status:', res.status);

    if (!res.ok) {
      console.error('External API error:', res.statusText);
      const errorText = await res.text();
      console.error('External API error details:', errorText);
      
      return NextResponse.json(
        { error: `Failed to submit answer: ${res.statusText}` },
        { status: res.status }
      );
    }

    const data = await res.json();
    console.log('Answer submitted successfully:', data);
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('API Route error:', error);
    return NextResponse.json(
      { 
        error: 'Internal server error', 
        details: error instanceof Error ? error.message : 'Unknown error occurred'
      },
      { status: 500 }
    );
  }
}