import React from "react";
import Link from "next/link";
import Image from "next/image";
import { Footer } from "@/components/Footer";
import { Navbar } from "@/components/Navbar";
import { getFaculties } from "@/services/api/faculties";

import { Card, CardHeader, CardTitle } from "@/components/ui/card";

// --- static meta data ---
const facultyMeta = [
  {
    name: "Faculty of Commerce",
    borderColor: "#DBBB15",
    img: "/images/com_Fac.png",
    code: "COM",
  },
  {
    name: "Faculty of Science",
    borderColor: "#623E88",
    img: "/images/sci_Fac.png",
    code: "SCI",
  },
  {
    name: "Faculty of Humanities",
    borderColor: "#065391",
    img: "/images/hum_Fac.png",
    code: "HUM",
  },
  {
    name: "Faculty of Engineering",
    borderColor: "#1EA558",
    img: "/images/eng_Fac.png",
    code: "EBE",
  },
  {
    name: "Faculty of Law",
    borderColor: "#AD9B6D",
    img: "/images/law_Fac.png",
    code: "LAW",
  },
  {
    name: "Faculty of Health",
    borderColor: "#9E0031",
    img: "/images/health_Fac.png",
    code: "MED",
  },
];

export default async function Page() {
  // get live faculty codes from API
  const facultyCodes = await getFaculties();

  // merge API data with local meta info
  const mergedFaculties = facultyCodes.flatMap((entry) => {
    const meta = facultyMeta.find((f) => f.code === entry.faculty_code);
    return meta
      ? [{ ...meta, link: `/academic_info/fac/${entry.faculty_code}` }]
      : [];
  });

  return (
    <div className="relative flex flex-col items-center min-h-screen pt-5 bg-gradient-to-br from-rose-50 via-sky-50 to-violet-50">
      {/* Background gradient */}
      <div className="absolute inset-0 z-0">
        <div className="absolute inset-0 bg-gradient-to-br from-rose-100/40 via-sky-100/30 to-violet-100/40" />
      </div>

      {/* Navbar */}
      <Navbar />

      {/* Heading */}
      <div className="w-full flex flex-col items-center justify-center text-center mt-16 z-10">
        <div className="text-5xl md:text-8xl outfit-400 mb-1 mt-3">
          Handbook <span className="text-uct-blue">Information</span>
        </div>
        <div className="max-w-2xl text-sm md:text-base text-gray-700 outfit-300 mt-5 px-8 md:px-0">
          Explore everything you need to know about UCT courses in one place.
          Search by name, filter by faculty, and find key details like
          prerequisites, credits, and course summaries—all made simple and easy
          to understand.
        </div>
      </div>

      {/* Faculty Cards */}
      <div className="w-full max-w-6xl mt-16 grid grid-cols-2 md:grid-cols-1 lg:grid-cols-3 gap-10 mb-10 px-8 md:px-0">
        {mergedFaculties.map((faculty) => (
          <Link
            key={faculty.code}
            href={faculty.link}
            className="w-full"
            style={{ textDecoration: "none" }}
          >
            <Card
              className="w-full h-50 md:h-80 flex flex-col items-center drop-shadow-sm justify-center border-4 hover:scale-105 transition-transform md:p-4"
              style={{ borderColor: faculty.borderColor }}
            >
              <div className="flex-1 items-center justify-center hidden md:flex">
                <Image
                  src={faculty.img}
                  width={130}
                  height={130}
                  alt={`${faculty.name} logo`}
                  className="object-contain"
                />
              </div>
              <div className="flex items-center justify-center md:hidden">
                <Image
                  src={faculty.img}
                  width={90}
                  height={90}
                  alt={`${faculty.name} logo`}
                  className="object-contain"
                />
              </div>
              <CardHeader className="w-full">
                <CardTitle className="text-center outfit-400 text-sm md:text-2xl">
                  {faculty.name}
                </CardTitle>
              </CardHeader>
            </Card>
          </Link>
        ))}
      </div>

      {/* Footer */}
      <Footer />
    </div>
  );
}
