from django.shortcuts import render
from django.http import JsonResponse
from django.db.models import Q, F
from django.views.decorators.csrf import csrf_exempt
import json
import requests
from .helper_functions import process_courses_by_degree, process_courses_by_major
from .models import Faculty, Department, Program, Degree, Major, Course, CoursesPerDegree

def get_health(request):
    return JsonResponse({'status': 'ok'}, status=200)

def get_faculties(request):
    faculties = Faculty.objects.filter(
        program__isnull=False  # Only faculties that have programs
    ).distinct().values('faculty_code', 'faculty_name', 'colour').order_by('faculty_name')  # Get as dictionaries
    return JsonResponse(list(faculties), safe=False)

def get_programs_by_faculty(request, faculty_code):
    try:
        faculty = Faculty.objects.get(faculty_code=faculty_code)
        faculty_id = faculty.faculty_id
    except Faculty.DoesNotExist:
        return JsonResponse({'error': f'Faculty code {faculty_code} not found.'}, status=404)
    
    # Only get programs that have degrees or majors  
    # Get all degrees and majors to find which programs have content
    degrees_with_programs = Degree.objects.filter(
        coursesperdegree__isnull=False
    ).values_list('program', flat=True).distinct()
    
    majors_with_programs = Major.objects.filter(
        coursespermajor__isnull=False
    ).values_list('major_code', flat=True).distinct()
    
    # Extract program codes from major codes (assuming major codes start with program code)
    major_program_codes = set()
    for major_code in majors_with_programs:
        # Find programs that this major code starts with
        matching_programs = Program.objects.filter(
            faculty=faculty_id
        ).values_list('program_code', flat=True)
        for program_code in matching_programs:
            if major_code.startswith(program_code):
                major_program_codes.add(program_code)
                break
    
    # Combine program codes that have degrees or majors with courses
    program_codes_with_content = set(degrees_with_programs) | major_program_codes
    
    programs = Program.objects.filter(
        faculty=faculty_id,
        program_code__in=program_codes_with_content
    ).values('program_code', 'program_name', 'description', 'saqa_id', 'nqf_level', 'minimum_duration_years', 'notes').order_by('program_code')
    return JsonResponse(list(programs), safe=False)

def get_degrees_by_program(request, faculty_code, program_code):
    # Only get degrees that have courses
    degrees = Degree.objects.filter(
        program=program_code,
        coursesperdegree__isnull=False
    ).distinct().values(
        'degree_code',
        'degree_name',
        'notes'
    ).order_by('degree_code')  # Get as dictionaries
    degrees_list = list(degrees)
    
    # Add courses for each degree
    for degree in degrees_list:
        courses_data = process_courses_by_degree.get_courses_by_degree(degree['degree_code'])
        degree['courses'] = courses_data

    return JsonResponse(degrees_list, safe=False)

def get_majors_by_program(request, faculty_code, program_code):
    # Only get majors that have courses
    majors = Major.objects.filter(
        program=program_code,
        coursespermajor__isnull=False
    ).distinct().values(
        'major_code',
        'major_name',
        'notes'
    ).order_by('major_code')  # Get as dictionaries
    majors_list = list(majors)

    for major in majors_list:
        courses_data = process_courses_by_major.get_courses_by_major(major['major_code'])
        major['courses'] = courses_data

    return JsonResponse(majors_list, safe=False)

def get_departments(request):
    departments = Department.objects.filter(
        course__isnull=False  # Only departments that have courses
    ).distinct().values('department_code', 'department_name', 'faculty__faculty_name', 'building', 'hod').order_by('department_name')  # Get as dictionaries
    return JsonResponse(list(departments), safe=False)

def get_courses_by_department(request, department_code):
    courses = Course.objects.filter(
        department__department_code=department_code
    ).values('course_code', 'course_name', 'nqf_credits', 'nqf_level', 'semester__semester_name', 'convener', 'course_entry_requirements', 'corequisites', 'objective', 'course_outline', 'lecture_times', 'dp_requirements', 'assessment', 'notes')

    # Convert to list and sort using the same logic as process_courses_by_degree
    courses_list = list(courses)
    
    def sort_key(course):
        code = course['course_code']
        # Extract the numeric part from course code (e.g., 1002 from ECO1002F)
        numeric_part = ''.join(filter(str.isdigit, code))
        # Get first digit of the numeric part for primary sorting
        first_digit = int(numeric_part[0]) if numeric_part else 0
        return (first_digit, code.lower())
    
    sorted_courses = sorted(courses_list, key=sort_key)
    
    return JsonResponse(sorted_courses, safe=False)

def get_degrees_by_program_no_courses(request, faculty_code, program_code):
    # Only get degrees that have courses
    degrees = Degree.objects.filter(
        degree_code__startswith=program_code,
        courses_per_degree__isnull=False
    ).distinct().select_related('program').values(
        'degree_code',
        'degree_name',
        'program__program_abbreviation'
    ).order_by('degree_code')  # Get as dictionaries
    
    # Format the degree names and remove program__program_abbreviation from response
    degrees_list = list(degrees)
    for degree in degrees_list:
        program_abbreviation = degree.get('program__program_abbreviation', '')
        original_degree_name = degree.get('degree_name', '')
        degree['degree_name'] = f"{program_abbreviation} specialising in {original_degree_name}"
        # Remove program__program_abbreviation from the response
        degree.pop('program__program_abbreviation', None)
    
    return JsonResponse(degrees_list, safe=False)

def get_majors_by_program_no_courses(request, faculty_code, program_code):
    # Only get majors that have courses
    majors = Major.objects.filter(
        major_code__startswith=program_code,
        courses_per_major__isnull=False
    ).distinct().select_related('program').values(
        'major_code',
        'major_name',
        'program__program_abbreviation'
    ).order_by('major_code')  # Get as dictionaries
    
    # Format the major names and remove program__program_abbreviation from response
    majors_list = list(majors)
    for major in majors_list:
        program_abbreviation = major.get('program__program_abbreviation', '')
        original_major_name = major.get('major_name', '')
        major['major_name'] = f"{program_abbreviation} {original_major_name}"
        # Remove program__program_abbreviation from the response
        major.pop('program__program_abbreviation', None)
    
    return JsonResponse(majors_list, safe=False)

# Helper function for streaming API requests
# async def stream_from_api(question: str, api_base_url: str = "http://badkamer.cs.uct.ac.za"):
#     """
#     Helper function to stream JSON responses from the external API
#     """
#     async with httpx.AsyncClient(timeout=300.0) as client:
#         try:
#             async with client.stream(
#                 "POST",
#                 f"{api_base_url}/ask/stream-json",
#                 json={"question": question, "streaming": True},
#                 headers={"Content-Type": "application/json"}
#             ) as response:
#                 response.raise_for_status()
#                 async for chunk in response.aiter_text():
#                     if chunk.strip():
#                         yield chunk
#         except httpx.HTTPStatusError as e:
#             error_data = {
#                 "type": "error",
#                 "content": f"API server error: {e.response.status_code}",
#                 "status": "error"
#             }
#             yield f"{json.dumps(error_data)}\n"
#         except httpx.RequestError:
#             error_data = {
#                 "type": "error", 
#                 "content": "API server unavailable",
#                 "status": "error"
#             }
#             yield f"{json.dumps(error_data)}\n"

# # Streaming endpoint for real-time responses
# @csrf_exempt
# @require_http_methods(["POST"])
# def ask_question_stream(request):
#     """
#     Streaming endpoint that proxies the streaming response from the external API
#     """
#     try:
#         # Parse request body
#         data = json.loads(request.body)
#         question = data.get('question', '')
        
#         if not question:
#             return JsonResponse({'error': 'Question is required'}, status=400)
        
#         # Create generator for streaming response
#         def response_generator():
#             async def async_generator():
#                 async for chunk in stream_from_api(question):
#                     yield chunk
            
#             # Convert async generator to sync
#             loop = asyncio.new_event_loop()
#             asyncio.set_event_loop(loop)
#             try:
#                 async_gen = async_generator()
#                 while True:
#                     try:
#                         chunk = loop.run_until_complete(async_gen.__anext__())
#                         yield chunk
#                     except StopAsyncIteration:
#                         break
#             finally:
#                 loop.close()
        
#         return StreamingHttpResponse(
#             response_generator(),
#             content_type='application/x-ndjson',
#             headers={
#                 'Cache-Control': 'no-cache',
#                 'Connection': 'keep-alive',
#                 'Access-Control-Allow-Origin': '*'
#             }
#         )
        
#     except json.JSONDecodeError:
#         return JsonResponse({'error': 'Invalid JSON in request body'}, status=400)
#     except Exception as e:
#         return JsonResponse({'error': f'Server error: {str(e)}'}, status=500)


# # Class-based view for handling both GET and POST requests
# @method_decorator(csrf_exempt, name='dispatch')
# class ChatbotView(View):
#     """
#     Class-based view for handling chatbot interactions
#     Supports both streaming and non-streaming responses
#     """
    
#     def get(self, request):
#         """Health check for the chatbot endpoint"""
#         return JsonResponse({
#             'status': 'ok',
#             'message': 'Chatbot endpoint is available',
#             'endpoints': {
#                 'non_streaming': '/chatbot/ask/',
#                 'streaming': '/chatbot/ask/stream/'
#             }
#         })
    
#     def post(self, request):
#         """Default POST handler - redirects to non-streaming endpoint"""
#         return ask_question(request)

@csrf_exempt
def get_roadmap_result(request):
    """
    Takes student progress JSON and returns:
    1. Detailed information about completed courses
    2. All courses for the next year in the same degree
    
    Expected JSON format:
    {
        "degree_code": "CB001ACC03",
        "year": 1,
        "completed_courses": [
            {"code": "ACC1015F", "passed": true},
            {"code": "CSC1010F", "passed": false},
            ...
        ]
    }
    """
    if request.method != 'POST':
        return JsonResponse({'error': 'Only POST method allowed'}, status=405)
    
    try:
        data = json.loads(request.body)
        degree_code = data.get('degree_code')
        current_year = data.get('year')
        completed_courses = data.get('completed_courses', [])
        
        if not degree_code:
            return JsonResponse({'error': 'degree_code is required'}, status=400)
        if current_year is None:
            return JsonResponse({'error': 'year is required'}, status=400)
        
        # Validate degree exists
        try:
            degree = Degree.objects.get(degree_code=degree_code)
        except Degree.DoesNotExist:
            return JsonResponse({'error': f'Degree {degree_code} not found'}, status=404)
        
        # Get detailed information about completed courses
        completed_course_codes = [course['code'] for course in completed_courses]
        completed_course_details = []
        
        if completed_course_codes:
            courses = Course.objects.filter(course_code__in=completed_course_codes)
            course_lookup = {course.course_code: course for course in courses}
            
            for completed_course in completed_courses:
                course_code = completed_course['code']
                passed = completed_course.get('passed', False)
                
                if course_code in course_lookup:
                    course = course_lookup[course_code]
                    
                    # For all courses (passed or failed), only include code, name and passed status
                    completed_course_details.append({
                        'code': course.course_code,
                        'name': course.course_name,
                        'passed': passed
                    })
                else:
                    # Course not found in database
                    completed_course_details.append({
                        'code': course_code,
                        'name': 'Course not found',
                        'passed': passed,
                        'error': 'Course code not found in database'
                    })
        
        # Get all courses for the next year in the same degree
        next_year = current_year + 1
        next_year_courses = []
        
        # Get courses for the next year from CoursesPerDegree
        next_year_course_relations = CoursesPerDegree.objects.filter(
            degree=degree,
            year=next_year
        ).select_related('course', 'course__semester', 'course__department')
        
        for course_relation in next_year_course_relations:
            course = course_relation.course
            next_year_courses.append({
                'code': course.course_code,
                'name': course.course_name,
                'nqf_credits': course.nqf_credits,
                'nqf_level': course.nqf_level,
                'semester': course.semester.semester_name if course.semester else None,
                'department': course.department.department_name if course.department else None,
                'convener': course.convener,
                'course_entry_requirements': course.course_entry_requirements,
                'corequisites': course.corequisites,
                'objective': course.objective,
                'course_outline': course.course_outline,
                'lecture_times': course.lecture_times,
                'dp_requirements': course.dp_requirements,
                'assessment': course.assessment,
                'year': course_relation.year,
                'notes': course.notes,
                'restrictions': course_relation.notes if course_relation.notes else None
            })
        
        response_data = {
            'degree_info': {
                'degree_code': degree.degree_code,
                'degree_name': degree.degree_name,
                'program_name': degree.program.program_name if degree.program else None,
                'program_abbreviation': degree.program.program_abbreviation if degree.program else None,
                'department': degree.department.department_name if degree.department else None,
                'degree_notes': degree.notes if degree.notes else None
            },
            'current_year': current_year,
            'next_year': next_year,
            'completed_courses_details': completed_course_details,
            'next_year_courses': next_year_courses
        }
        
        # Send data to AI endpoint for evaluation
        ai_endpoint = "http://badkamer.cs.uct.ac.za/roadmap-evaluation"
        
        try:
            # Send POST request to AI endpoint
            ai_response = requests.post(
                ai_endpoint,
                json=response_data,
                headers={'Content-Type': 'application/json'},
                timeout=60  # 60 second timeout
            )
            
            # Check if the AI request was successful
            if ai_response.status_code == 200:
                # Return the AI's evaluation response
                try:
                    ai_data = ai_response.json()
                    return JsonResponse(ai_data, safe=False)
                except json.JSONDecodeError:
                    # If AI response is not valid JSON, return as text
                    return JsonResponse({
                        'ai_response': ai_response.text,
                        'original_data': response_data
                    }, safe=False)
            else:
                # AI endpoint failed, return original data with error info
                return JsonResponse({
                    'error': f'AI evaluation failed with status {ai_response.status_code}',
                    'ai_error_message': ai_response.text,
                    'original_data': response_data
                }, status=500)
                
        except requests.exceptions.Timeout:
            # AI endpoint timeout, return original data
            return JsonResponse({
                'error': 'AI evaluation timed out',
                'original_data': response_data
            }, status=408)
        except requests.exceptions.ConnectionError:
            # AI endpoint connection failed, return original data
            return JsonResponse({
                'error': 'Could not connect to AI evaluation service',
                'original_data': response_data
            }, status=503)
        except requests.exceptions.RequestException as e:
            # Other request errors, return original data
            return JsonResponse({
                'error': f'AI evaluation request failed: {str(e)}',
                'original_data': response_data
            }, status=500)
        
    except json.JSONDecodeError:
        return JsonResponse({'error': 'Invalid JSON in request body'}, status=400)
    except Exception as e:
        return JsonResponse({'error': f'Server error: {str(e)}'}, status=500)