# This is an auto-generated Django model module.
# You'll have to do the following manually to clean this up:
#   * Rearrange models' order
#   * Make sure each model has one field with primary_key=True
#   * Make sure each ForeignKey and OneToOneField has `on_delete` set to the desired behavior
#   * Remove `managed = False` lines if you wish to allow Django to create, modify, and delete the table
# Feel free to rename the models, but don't rename db_table values or field names.
from django.db import models
from django.core.exceptions import ValidationError

class Faculty(models.Model):
    faculty_id = models.AutoField(primary_key=True)
    faculty_code = models.CharField(unique=True, max_length=3, help_text="Unique 3-letter code e.g 'COM' for Commerce")
    faculty_name = models.TextField(unique=True, help_text="Must be unique")
    colour = models.CharField(max_length=7, default='#0000FF', help_text="Hex color code for the faculty e.g '#0000FF' for blue")
    icon = models.BinaryField(blank=True, null=True)

    def save(self, *args, **kwargs):
        self.faculty_code = self.faculty_code.upper()
        super().save(*args, **kwargs)

    def __str__(self):
        return str(self.faculty_name)
    
    class Meta:
        db_table = 'faculty'
        verbose_name_plural = 'Faculties'


class Department(models.Model):
    department_code = models.CharField(primary_key=True, max_length=3)
    department_name = models.TextField(unique=True)
    faculty = models.ForeignKey(Faculty, models.CASCADE)
    building = models.TextField(blank=True, null=True)
    hod = models.CharField(max_length=255, default='No information')

    def save(self, *args, **kwargs):
        self.department_code = self.department_code.upper()
        super().save(*args, **kwargs)

    
    def __str__(self):
        return str(self.department_name)

    class Meta:
        db_table = 'department'


class Semester(models.Model):
    semester_letter = models.CharField(primary_key=True, max_length=1)
    semester_name = models.CharField(max_length=255)
    
    def save(self, *args, **kwargs):
        self.semester_letter = self.semester_letter.upper()
        super().save(*args, **kwargs)

    def __str__(self):
        return str(self.semester_name)

    class Meta:
        db_table = 'semester'


class Course(models.Model):
    course_code = models.CharField(primary_key=True, max_length=8, help_text="Unique 8-digit code e.g. 'MAM1004F'. First 3 digits is the department code that the course falls under e.g. 'MAM' for Maths department. 4th digit is typically the year e.g. '1' for 1st year course. Next 3 digits is a number between 001 and 999. Last digit is a semester letter representing the semester the course is in. e.g. 'F' for First semester")
    course_name = models.CharField(max_length=255)
    nqf_credits = models.IntegerField(default=18)
    nqf_level = models.IntegerField(default=5)
    semester = models.ForeignKey(Semester, models.SET_NULL, blank=True, null=True)
    department = models.ForeignKey(Department, models.CASCADE, blank=True, null=True)
    convener = models.CharField(max_length=255, blank=True, null=True)
    course_entry_requirements = models.TextField(blank=True, null=True)
    corequisites = models.TextField(blank=True, null=True)
    objective = models.TextField(blank=True, null=True)
    course_outline = models.TextField(blank=True, null=True)
    lecture_times = models.TextField(blank=True, null=True)
    dp_requirements = models.TextField(blank=True, null=True)
    assessment = models.TextField(blank=True, null=True)
    notes = models.TextField(blank=True, null=True)

    def clean(self):
        """Validate the course before saving."""
        super().clean()
        
        if not self.course_code:
            return

        # Helpful validation for nqf_credits
        if self.nqf_credits is not None and (self.nqf_credits < 0 or self.nqf_credits > 999):
            raise ValidationError({
                'nqf_credits': "Nqf credits must be in between 0 and 999."
            })

        # Helpful validation for nqf_level
        if self.nqf_level is not None and (self.nqf_level < 0 or self.nqf_level > 99):
            raise ValidationError({
                'nqf_level': "Nqf level must be in between 0 and 99."
            })

        # Validate semester from last letter
        semester_letter = self.course_code[-1] if self.course_code else ''
            
        # Check if it's a valid semester character
        try:
            Semester.objects.get(semester_letter=semester_letter)
        except Semester.DoesNotExist:
            # Get available semester characters for helpful error message
            available_semesters = list(Semester.objects.values_list('semester_letter', 'semester_name'))
            available_letters = [f"'{letter}' ({name})" for letter, name in available_semesters]
            
            raise ValidationError({
                'course_code': (
                    f"Course code '{self.course_code}' ends with '{semester_letter}' which is not a valid semester letter. "
                    f"Available semester letters are: {', '.join(available_letters)}. "
                    "Please use a valid Semester letter or create a new Semester if needed."
                )
            })

        # Validate department from first 3 letters
        department_code = self.course_code[:3]
        try:
            Department.objects.get(department_code=department_code)
        except Department.DoesNotExist:
            raise ValidationError({
                'course_code': (
                    f"No Department matches '{department_code}' in course code '{self.course_code}'. "
                    "Please use a valid Department code or create a new Department if needed."
                )
            })

    def save(self, *args, **kwargs):
        # Call clean to validate before saving
        self.full_clean()
        # Auto-set semester from last letter
        semester_letter = self.course_code[-1] if self.course_code else ''
        self.semester = Semester.objects.get(semester_letter=semester_letter)

        # Auto-set department from first 3 letters
        department_code = self.course_code[:3]
        self.department = Department.objects.get(department_code=department_code)

        self.course_code = self.course_code.upper()

        super().save(*args, **kwargs)
    
    def __str__(self):
        return f"{self.course_code}"

    class Meta:
        db_table = 'course'


class Qualification(models.Model):
    qualification_id = models.AutoField(primary_key=True)
    qualification_name = models.CharField(max_length=255, unique=True)

    def __str__(self):
        return f"{self.qualification_name}"
    
    class Meta:
        db_table = "qualification"


class Program(models.Model):
    program_code = models.CharField(primary_key=True, max_length=5, help_text="Unique 5-digit code e.g. 'CB001' for Bachelor of Commerce. 1st digit is for faculty e.g. 'C' for Commerce. 2nd digit is for qualification e.g. 'B' for Bachelor's. Last 3 digits is a number between 001 and 999.")
    program_name = models.CharField(max_length=255, unique=True)
    program_abbreviation = models.CharField(max_length=255, unique=True, help_text="Unique abbreviation for the program. e.g. 'BCom' for Bachelor of Commerce.")
    description = models.TextField(blank=True, null=True)
    saqa_id = models.CharField(max_length=255, blank=True, null=True)
    minimum_duration_years = models.IntegerField(default=3)
    nqf_level = models.IntegerField(default=7, help_text="7 for Bachelor's, 8 for Honour's, 9 for Master's, 10 for Doctorate")
    faculty = models.ForeignKey(Faculty, models.CASCADE)
    qualification = models.ForeignKey(Qualification, models.CASCADE)
    notes = models.TextField(blank=True, null=True)
    
    def save(self, *args, **kwargs):
        self.program_code = self.program_code.upper()
        super().save(*args, **kwargs)
    
    def __str__(self):
        return f"{self.program_name} ({self.program_abbreviation})"

    class Meta:
        db_table = 'program'


class Degree(models.Model):
    degree_code = models.CharField(primary_key=True, max_length=10, help_text="Unique 10-digit code e.g. 'CB001MAM01' for Bachelor of Commerce specialising in Mathematics. First 5 digits are the program code e.g. 'CB001' for Bachelor of Commerce. Next 3 are the department code that the degree falls under e.g. 'MAM' for the Mathematics Department. Last 2 is a number between 01 and 99.")
    degree_name = models.CharField(max_length=255)
    department = models.ForeignKey(Department, models.CASCADE, blank=True, null=True)
    program = models.ForeignKey(Program, models.CASCADE, blank=True, null=True)
    notes = models.TextField(blank=True, null=True)
    
    
    def clean(self):
        """Validate the degree before saving."""
        super().clean()

        if not self.degree_code:
            return

        program_code = self.degree_code[:5] if self.degree_code else ''

        try:
            Program.objects.get(program_code=program_code)
        except Program.DoesNotExist:            
            raise ValidationError({
                'degree_code': (
                    f"No Program matches '{program_code}' in degree code '{self.degree_code}'. "
                    "Please use a valid Program code or create a new Program if needed."
                )
            })
            
        department_code = self.degree_code[5:8]
        try:
            Department.objects.get(department_code=department_code)
        except Department.DoesNotExist:
            raise ValidationError({
                'degree_code': (
                    f"No Department matches '{department_code}' in degree code '{self.degree_code}'. "
                    "Please use a valid Department code or create a new Department if needed."
                )
            })

    def save(self, *args, **kwargs):
        # Call clean to validate before saving
        self.full_clean()
        
        # Auto-set program from first 5 letters of degree code
        program_code = self.degree_code[:5] if self.degree_code else ''
        self.program = Program.objects.get(program_code=program_code)

        # Auto-set department from letters 5-8 of degree code
        department_code = self.degree_code[5:8]
        self.department = Department.objects.get(department_code=department_code)

        self.degree_code = self.degree_code.upper()

        super().save(*args, **kwargs)

    def __str__(self):
        return f"{self.program} specialising in {self.degree_name}"

    class Meta:
        db_table = 'degree'
        

class Major(models.Model):
    major_code = models.CharField(primary_key=True, max_length=10, help_text="Unique 10-digit code e.g. 'SB001MAM01' for major in Mathematics in the Bachelor of Science degree. First 5 digits are the program code e.g. 'SB001' for Bachelor of Science. Next 3 are the department code that the Major falls under e.g. 'MAM' for the Mathematics Department. Last 2 is a number between 01 and 99.")
    major_name = models.CharField(max_length=255)
    department = models.ForeignKey(Department, models.CASCADE, blank=True, null=True)
    program = models.ForeignKey(Program, models.CASCADE, blank=True, null=True)
    notes = models.TextField(blank=True, null=True)
    
    def clean(self):
        """Validate the major before saving."""
        super().clean()

        if not self.major_code:
            return

        program_code = self.major_code[:5] if self.major_code else ''

        try:
            Program.objects.get(program_code=program_code)
        except Program.DoesNotExist:            
            raise ValidationError({
                'major_code': (
                    f"No Program matches '{program_code}' in major code '{self.major_code}'. "
                    "Please use a valid Program code or create a new Program if needed."
                )
            })

        department_code = self.major_code[5:8]
        try:
            Department.objects.get(department_code=department_code)
        except Department.DoesNotExist:
            raise ValidationError({
                'major_code': (
                    f"No Department matches '{department_code}' in major code '{self.major_code}'. "
                    "Please use a valid Department code or create a new Department if needed."
                )
            })

    def save(self, *args, **kwargs):
        # Call clean to validate before saving
        self.full_clean()
        
        # Auto-set semester from last letter
        program_code = self.major_code[:5] if self.major_code else ''
        self.program = Program.objects.get(program_code=program_code)

        # Auto-set department from first 3 letters
        department_code = self.major_code[5:8]
        self.department = Department.objects.get(department_code=department_code)

        self.major_code = self.major_code.upper()

        super().save(*args, **kwargs)

    def __str__(self):
        return f"{self.program.program_abbreviation} {self.major_name}"

    class Meta:
        db_table = 'major'
        

class DegreeParentCourseGroup(models.Model):
    parent_group_id = models.AutoField(primary_key=True)
    degree = models.ForeignKey(Degree, models.CASCADE)
    required_count = models.IntegerField(default=1)
    description = models.TextField(default= "e.g. 'Choose 1 between both MAM1004F and MAM1008S and both MAM1031F and MAM1032S'")

    def __str__(self):
        return f"{self.description}"

    class Meta:
        db_table = 'degree_parent_course_group'


class MajorParentCourseGroup(models.Model):
    parent_group_id = models.AutoField(primary_key=True)
    major = models.ForeignKey(Major, models.CASCADE)
    required_count = models.IntegerField(default=1)
    description = models.TextField(default= "e.g. 'Choose 1 between both MAM1004F and MAM1008S and both MAM1031F and MAM1032S'")

    def __str__(self):
        return f"{self.description}"

    class Meta:
        db_table = 'major_parent_course_group'


class DegreeCourseGroup(models.Model):
    course_group_id = models.AutoField(primary_key=True)
    degree = models.ForeignKey(Degree, models.CASCADE, blank=True, null=True)
    required_count = models.IntegerField(default=1)
    description = models.TextField(default= "e.g. 'Choose 1 between MAM1004F and MAM1031F'")
    parent_group = models.ForeignKey(DegreeParentCourseGroup, models.SET_NULL, blank=True, null=True)

    def __str__(self):
        return f"{self.description}"

    class Meta:
        db_table = 'degree_course_group'
        

class MajorCourseGroup(models.Model):
    course_group_id = models.AutoField(primary_key=True)
    major = models.ForeignKey(Major, models.CASCADE)
    required_count = models.IntegerField(default=1)
    description = models.TextField(default= "e.g. 'Choose 1 between MAM1004F and MAM1031F'")
    parent_group = models.ForeignKey(MajorParentCourseGroup, models.SET_NULL, blank=True, null=True)

    def __str__(self):
        return f"{self.description}"

    class Meta:
        db_table = 'major_course_group'


class CoursesPerDegree(models.Model):
    course_degree_id = models.AutoField(primary_key=True)
    degree = models.ForeignKey(Degree, models.CASCADE)
    course = models.ForeignKey(Course, models.CASCADE)
    course_group = models.ForeignKey(DegreeCourseGroup, models.SET_NULL, blank=True, null=True)
    year = models.IntegerField(default=1)
    notes = models.TextField(blank=True, null=True)
    
    def __str__(self):
        return f"{self.degree} - {self.course}"

    class Meta:
        db_table = 'courses_per_degree'
        

class CoursesPerMajor(models.Model):
    course_major_id = models.AutoField(primary_key=True)
    major = models.ForeignKey(Major, models.CASCADE)
    course = models.ForeignKey(Course, models.CASCADE)
    course_group = models.ForeignKey(MajorCourseGroup, models.SET_NULL, blank=True, null=True)
    year = models.IntegerField(default=1)
    notes = models.TextField(blank=True, null=True)

    def __str__(self):
        return f"{self.major} - {self.course}"

    class Meta:
        db_table = 'courses_per_major'


class Elective(models.Model):
    elective_id = models.AutoField(primary_key=True)
    program = models.ForeignKey(Program, models.CASCADE)
    course = models.ForeignKey(Course, models.CASCADE)
    year = models.IntegerField(default=1)
    notes = models.TextField(blank=True, null=True)

    def __str__(self):
        return f"{self.program} - {self.course}"
    
    class Meta:
        db_table = 'elective'