from handbook.models import CoursesPerDegree, DegreeCourseGroup

def create_course_item(course):
    """Helper function to create a course item with consistent structure"""
    def none_if_null(value):
        return value if value is not None else "No information"

    course_item = {
        'code': none_if_null(course.get('code')),
        'name': none_if_null(course.get('name')),
        'credits': none_if_null(course.get('credits')),
        'level': none_if_null(course.get('level')),
        'semester': none_if_null(course.get('semester')),
        'department': none_if_null(course.get('department')),
        'convener': none_if_null(course.get('convener')),
        'entry_requirements': none_if_null(course.get('entry_requirements')),
        'corequisites': none_if_null(course.get('corequisites')),
        'objective': none_if_null(course.get('objective')),
        'course_outline': none_if_null(course.get('course_outline')),
        'lecture_times': none_if_null(course.get('lecture_times')),
        'dp_requirements': none_if_null(course.get('dp_requirements')),
        'assessment': none_if_null(course.get('assessment')),
        'notes': none_if_null(course.get('notes')),
        'restrictions': none_if_null(course.get('restrictions')),
        'type': none_if_null(course.get('type'))
    }
    return course_item

def get_courses_by_degree(degree_code):
    """Helper method to get courses for a specific degree. Not called by API directly."""
    courses = CoursesPerDegree.objects.filter(degree=degree_code).values(
        'course__course_code',
        'course__course_name',
        'course__nqf_credits',
        'course__nqf_level',
        'course__semester__semester_name',
        'course__department__department_name',
        'course__convener',
        'course__course_entry_requirements',
        'course__corequisites',
        'course__objective',
        'course__course_outline',
        'course__lecture_times',
        'course__dp_requirements',
        'course__assessment',
        'course__notes',
        'year',
        'course_group',
        'course_group__parent_group_id',
        'course_group__required_count',
        'notes',
        
    )

    # Group courses by group and handle ungrouped courses separately
    grouped_courses = {}
    ungrouped_courses = []
    
    for course in courses:
        
        # Determine course type based on course code
        course_type = 'elective' if 'ELE' in course['course__course_code'] else 'required'
        department_name = None if 'ELE' in course['course__course_code'] else course['course__department__department_name']
        
        course_data = {
                'code': course['course__course_code'],
                'name': course['course__course_name'],
                'credits': course['course__nqf_credits'],
                'level': course['course__nqf_level'],
                'year': course['year'],
                'semester': course['course__semester__semester_name'],
                'department': department_name,
                'convener': course['course__convener'],
                'entry_requirements': course['course__course_entry_requirements'],
                'corequisites': course['course__corequisites'],
                'objective': course['course__objective'],
                'course_outline': course['course__course_outline'],
                'lecture_times': course['course__lecture_times'],
                'dp_requirements': course['course__dp_requirements'],
                'assessment': course['course__assessment'],
                'notes': course['course__notes'],
                'restrictions': course['notes'],
                'type': course_type,
            }
        
        group = course['course_group']
        
        # If group_id is null, add as individual course (not grouped)
        if group is None:
            ungrouped_courses.append(course_data)
        else:
            # Initialize group if it doesn't exist
            if group not in grouped_courses:
                grouped_courses[group] = {
                    'course_group': group,
                    'parent_group_id': course['course_group__parent_group_id'],
                    'required_count': course['course_group__required_count'],
                    'course_count': 0,  # Initialize course count
                    'year': float('inf'),  # Initialize year tracking
                    'courses': [],
                }
            
            # Add course to the group
            grouped_courses[group]['courses'].append(course_data)
            
            # Update course count
            # grouped_courses[group]['course_count'] += 1

            # Update minimum year for the group
            current_year = grouped_courses[group]['year']
            if current_year == float('inf'):
                grouped_courses[group]['year'] = course['year']
            else:
                grouped_courses[group]['year'] = min(current_year, course['year'])

    # Now organize groups by parent_group_id
    parent_groups = {}
    standalone_groups = []
    
    # Get all unique parent group IDs to fetch their required counts
    parent_group_ids = set()
    for group in grouped_courses.values():
        if group['parent_group_id'] is not None:
            parent_group_ids.add(group['parent_group_id'])
    
    # Fetch parent group data from DegreeCourseGroup model
    parent_group_data = {}
    if parent_group_ids:
        parent_groups_query = DegreeCourseGroup.objects.filter(course_group_id__in=parent_group_ids).values(
            'course_group_id', 'required_count'
        )
        parent_group_data = {pg['course_group_id']: pg['required_count'] for pg in parent_groups_query}
    
    for group in grouped_courses.values():
        parent_id = group['parent_group_id']
        
        if parent_id is None:
            # No parent, add as standalone group
            standalone_groups.append(group)
        else:
            # Has parent, group under parent
            if parent_id not in parent_groups:
                parent_groups[parent_id] = {
                    'parent_group_id': parent_id,
                    'required_count': parent_group_data.get(parent_id, 0),  # Get actual parent required count
                    'year': float('inf'),  # Track minimum year across all subgroups
                    'subgroup_count': 0,  # Initialize subgroup count
                    'subgroups': []
                }
            
            parent_groups[parent_id]['subgroups'].append(group)
            parent_groups[parent_id]['subgroup_count'] += 1  # Increment subgroup count

            # Update parent group year to minimum of all subgroups
            if group['year'] < parent_groups[parent_id]['year']:
                parent_groups[parent_id]['year'] = group['year']
    
    # Create the new structure grouped by year
    result = []
    
    # Process each year (1-6)
    for year in [1, 2, 3, 4, 5, 6]:
        year_items = []
        
        # Add ungrouped courses for this year
        ungrouped_year_courses = [course for course in ungrouped_courses if course['year'] == year]
        # Sort ungrouped courses by course code number then alphabetically by course code
        def sort_key(course):
            code = course['code']
            # Extract the numeric part from course code (e.g., 1002 from ECO1002F)
            numeric_part = ''.join(filter(str.isdigit, code))
            # Get first digit of the numeric part for primary sorting
            first_digit = int(numeric_part[0]) if numeric_part else 0
            return (first_digit, code.lower())
        
        sorted_ungrouped_courses = sorted(ungrouped_year_courses, key=sort_key)
        
        for course in sorted_ungrouped_courses:
            # Create course item using helper function
            course_item = create_course_item(course)
            year_items.append(course_item)
        
        # Add standalone groups for this year
        for group in standalone_groups:
            if group['year'] == year:
                # Sort courses by course code number then alphabetically by course code
                def sort_key(course):
                    code = course['code']
                    # Extract the numeric part from course code (e.g., 1002 from ECO1002F)
                    numeric_part = ''.join(filter(str.isdigit, code))
                    # Get first digit of the numeric part for primary sorting
                    first_digit = int(numeric_part[0]) if numeric_part else 0
                    return (first_digit, code.lower())
                
                sorted_courses = sorted(group['courses'], key=sort_key)
                
                # Check if required_count should be 100 (all courses required)
                required_count = group['required_count']
                if len(group['courses']) == required_count and required_count > 0:
                    required_count = 100
                
                # Create group structure without year
                group_item = {
                    'required_count': required_count,
                    'courses': []
                }
                
                # Add courses to the group (remove year from individual courses)
                for course in sorted_courses:
                    course_item = create_course_item(course)
                    group_item['courses'].append(course_item)
                
                year_items.append(group_item)
        
        # Add parent groups for this year (grouped together)
        for parent_group in parent_groups.values():
            if parent_group['year'] == year:
                # Check if required_count should be 100 (all subgroups required)
                parent_required_count = parent_group['required_count']
                if parent_required_count == 0:  # Fix for 0 required_count
                    parent_required_count = 1
                if len(parent_group['subgroups']) == parent_required_count and parent_required_count > 0:
                    parent_required_count = 100
                
                # Create parent group structure
                parent_item = {
                    'required_count': parent_required_count,
                    'groups': []
                }
                
                # Add all subgroups under this parent
                # Sort subgroups by number of courses (ascending)
                sorted_subgroups = sorted(parent_group['subgroups'], key=lambda x: len(x['courses']))
                
                for subgroup in sorted_subgroups:
                    # Sort courses by course code number then alphabetically by course code
                    def sort_key(course):
                        code = course['code']
                        # Extract the numeric part from course code (e.g., 1002 from ECO1002F)
                        numeric_part = ''.join(filter(str.isdigit, code))
                        # Get first digit of the numeric part for primary sorting
                        first_digit = int(numeric_part[0]) if numeric_part else 0
                        return (first_digit, code.lower())
                    
                    sorted_courses = sorted(subgroup['courses'], key=sort_key)
                    
                    # Check if subgroup required_count should be 100
                    subgroup_required_count = subgroup['required_count']
                    if len(subgroup['courses']) == subgroup_required_count and subgroup_required_count > 0:
                        subgroup_required_count = 100
                    
                    subgroup_item = {
                        'required_count': subgroup_required_count,
                        'courses': []
                    }
                    
                    # Add courses to the subgroup
                    for course in sorted_courses:
                        course_item = create_course_item(course)
                        subgroup_item['courses'].append(course_item)
                    
                    parent_item['groups'].append(subgroup_item)
                
                year_items.append(parent_item)
        
        # Only add year data if there are items for this year
        if year_items:
            year_data = {
                'year': year,
                'items': year_items
            }
            result.append(year_data)
    
    return result