from django.contrib import admin
from .models import *
from django.utils.html import format_html


class DegreeParentCourseGroupInline(admin.TabularInline):
    model = DegreeParentCourseGroup
    extra = 1
    verbose_name = "Parent Course Group"
    verbose_name_plural = "Parent Course Groups"
    fk_name = 'degree'  # ensures the group is attached to this degree
    show_change_link = True  # Adds a link to edit the parent group separately if needed

class DegreeCourseGroupInline(admin.TabularInline):
    model = DegreeCourseGroup
    extra = 1
    verbose_name = "Course Group"
    verbose_name_plural = "Course Groups"
    fk_name = 'degree'  # ensures the group is attached to this degree
    show_change_link = True
    
    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "parent_group":
            if "change" in request.path:
                obj_id = request.path.rstrip("/").split("/")[-2]
                try:
                    degree = Degree.objects.get(pk=obj_id)
                    # Limit parent choices to groups of this degree
                    kwargs["queryset"] = DegreeParentCourseGroup.objects.filter(degree=degree)
                except Degree.DoesNotExist:
                    kwargs["queryset"] = DegreeParentCourseGroup.objects.none()
            else:
                kwargs["queryset"] = DegreeParentCourseGroup.objects.none()
        return super().formfield_for_foreignkey(db_field, request, **kwargs)

class MajorParentCourseGroupInline(admin.TabularInline):
    model = MajorParentCourseGroup
    extra = 1
    verbose_name = "Parent Course Group"
    verbose_name_plural = "Parent Course Groups"
    fk_name = 'major'  # ensures the group is attached to this major
    show_change_link = True  # Adds a link to edit the parent group separately if needed

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "course_group":
            major_id = request.resolver_match.kwargs.get('object_id')
            if major_id:
                try:
                    major = Major.objects.get(pk=major_id)
                    kwargs["queryset"] = MajorCourseGroup.objects.filter(major=major)
                except Major.DoesNotExist:
                    kwargs["queryset"] = MajorCourseGroup.objects.none()
            else:
                kwargs["queryset"] = MajorCourseGroup.objects.none()
        return super().formfield_for_foreignkey(db_field, request, **kwargs)

class MajorCourseGroupInline(admin.TabularInline):
    model = MajorCourseGroup
    extra = 1
    verbose_name = "Course Group"
    verbose_name_plural = "Course Groups"
    fk_name = 'major'  # ensures the group is attached to this major
    show_change_link = True  # allows opening the group to edit subgroups
    
    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "parent_group":
            # Get the current major from the URL
            # URL structure: /admin/handbook/major/TST01/change/
            if "change" in request.path:
                # Extract major id from URL - it's the part before '/change/'
                path_parts = request.path.rstrip("/").split("/")
                try:
                    # Find the major ID which comes after 'major' in the path
                    major_index = path_parts.index('major')
                    major_id = path_parts[major_index + 1]  # TST01
                    major = Major.objects.get(pk=major_id)
                    kwargs["queryset"] = MajorParentCourseGroup.objects.filter(major=major)
                except (ValueError, IndexError, Major.DoesNotExist):
                    kwargs["queryset"] = MajorParentCourseGroup.objects.none()
            else:
                kwargs["queryset"] = MajorParentCourseGroup.objects.none()
        return super().formfield_for_foreignkey(db_field, request, **kwargs)
    
class CoursesPerDegreeInline(admin.TabularInline):
    model = CoursesPerDegree
    extra = 1
    verbose_name = "Course"
    verbose_name_plural = "Courses"

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "course_group":
            # Get degree code from URL (object_id)
            degree_code = request.resolver_match.kwargs.get('object_id')
            if degree_code:
                try:
                    degree = Degree.objects.get(pk=degree_code)
                    kwargs["queryset"] = DegreeCourseGroup.objects.filter(degree=degree)
                except Degree.DoesNotExist:
                    kwargs["queryset"] = DegreeCourseGroup.objects.none()
            else:
                kwargs["queryset"] = DegreeCourseGroup.objects.none()
        return super().formfield_for_foreignkey(db_field, request, **kwargs)

class CoursesPerMajorInline(admin.TabularInline):
    model = CoursesPerMajor
    extra = 1
    verbose_name = "Course"
    verbose_name_plural = "Courses"

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "course_group":
            # Get major code from URL (object_id)
            view_kwargs = request.resolver_match.kwargs
            major_code = view_kwargs.get('object_id')

            if major_code:
                try:
                    major = Major.objects.get(pk=major_code)
                    kwargs["queryset"] = MajorCourseGroup.objects.filter(major=major)
                except Major.DoesNotExist:
                    kwargs["queryset"] = MajorCourseGroup.objects.none()
            else:
                kwargs["queryset"] = MajorCourseGroup.objects.none()
        return super().formfield_for_foreignkey(db_field, request, **kwargs)

  
@admin.register(Faculty)
class FacultyAdmin(admin.ModelAdmin):
    list_display = ('faculty_code', 'faculty_name', 'colour')
    search_fields = ['faculty_code', 'faculty_name']
    ordering = ['faculty_code']

@admin.register(Department)
class DepartmentAdmin(admin.ModelAdmin):
    list_display = ('department_code', 'department_name', 'faculty', 'building', 'hod')
    search_fields = ['department_code', 'department_name']
    list_filter = ['faculty', 'building']
    ordering = ['department_code']

@admin.register(Semester)
class SemesterAdmin(admin.ModelAdmin):
    list_display = ('semester_letter', 'semester_name')
    ordering = ['semester_letter']

@admin.register(Course)
class CoursesAdmin(admin.ModelAdmin):
    list_display = ('course_code', 'course_name', 'nqf_credits', 'nqf_level', 'semester', 'department', 'convener')
    search_fields = ['course_code', 'course_name', 'convener']
    list_filter = ['department', 'nqf_level', 'nqf_credits', 'semester']
    ordering = ['course_code']
    exclude = ('department', 'semester')
    
@admin.register(Program)
class ProgramsAdmin(admin.ModelAdmin):
    list_display = ('program_code', 'program_name', 'program_abbreviation', 'saqa_id', 'faculty', 'nqf_level', 'minimum_duration_years', 'qualification')
    search_fields = ['program_code', 'program_name', 'program_abbreviation', 'qualification__qualification_name']
    list_filter = ['faculty', 'qualification', 'minimum_duration_years']
    ordering = ['program_code']

@admin.register(Qualification)
class QualificationAdmin(admin.ModelAdmin):
    list_display = ('qualification_id', 'qualification_name')
    search_fields = ['qualification_name']
    ordering = ['qualification_id']

@admin.register(Degree)
class DegreesAdmin(admin.ModelAdmin):
    list_display = ('degree_code', 'degree_name', 'department', 'program')
    search_fields = ['degree_code', 'degree_name']
    list_filter = ['department', 'program']
    ordering = ['degree_code']
    inlines = [DegreeParentCourseGroupInline, DegreeCourseGroupInline, CoursesPerDegreeInline]

    readonly_fields = ("course_instructions", "program", "department")

    fieldsets = (
        (None, {
            'fields': ('course_instructions', 'degree_code', 'degree_name', 'program', 'department', 'notes'),
        }),
    )

    def course_instructions(self, obj):
        return format_html(
            "<b>Instructions for creating or editing degrees:</b><br>"
            "1. Add all information for the general degree (Degree code, Degree name, Notes).<br>"
            "2. The Program and Department will be automatically set based on the degree code:<br>"
            "   - Program: First 5 characters of degree code (e.g., 'CB001' from 'CB001BUS03')<br>"
            "   - Department: Characters 6-8 of degree code (e.g., 'BUS' from 'CB001BUS03')<br>"
            "3. Click 'Save and continue editing' before moving to the next tab.<br>"
            "<br>"
            "<b>Instructions for adding Courses for a Degree:</b><br>"
            "1. Add all courses than can be taken in the degree in the 'Courses' tab.<br>"
            "2. Add all information for the course (Course code, Year, Notes (specific to this degree, general notes for a course are stored in Courses)).<br>"
            "3. Click 'Save and continue editing' to save your progress.<br>"
            "4. To delete a course in a degree, check the 'Delete?' box and save your progress.<br>"
            "<br>"
            "<b>Information about how to make options between Courses:</b><br>"
            "<b>Course Groups:</b><br>"
            "Creates choices between multiple courses that a student must do in their degree<br>"
            "e.g. If a student must either do MAM1031F or MAM1004F.<br>"
            "<br>"
            "<b>Instructions for adding Course Groups</b><br>"
            "1. Create a course group with 'Required count' equal to the number of courses the student must take in this group.<br>"
            "2. Give it a meaningful description so you can differ between multiple Course groups.<br>"
            "3. Change all courses that must be a part of this group to all share the same 'Course group'.<br>"
            "4. To delete a course group, check the 'Delete?' box and save your progress. This will not delete the courses in that group.<br>"
            "5. When you are finished, click 'Save and continue editing' to apply your changes.<br>"
            "<br>"
            "<b>Parent Course Groups:</b><br>"
            "Creates paths of choices between multiple courses that a student must do in their degree<br>"
            "e.g. When a student must either do both MAM1004F and MAM1008S together or both MAM1031F and MAM1032S together.<br>"
            "<br>"
            "<b>Instructions for adding Parent Course Groups:</b><br>"
            "1. Create a parent course group with 'Required count' equal to the number of groups of courses the student must take in this parent group.<br>"
            "   i.e. MAM1004F and MAM1008S with Required count 1 is 1 group and MAM1031F and MAM1032S with Required count 1 is another group.<br>"
            "   Therefore, a student must do either group and then whatever number of courses the Required count is in the subgroup they choose.<br>"
            "2. Give it a meaningful description so you can differ between multiple Parent Course Groups.<br>"
            "3. Create a Course group for each set of courses that should be grouped together eg 'MAM1004F and MAM1008S' then 'MAM1031F and MAM1032S'.<br>"
            "   Required count in the subgroup is how many courses the student must take in that subgroup.<br>"
            "4. To delete a parent course group, check the 'Delete?' box and save your changes.<br>"
            "   This will delete only the parent group and not the courses or groups in the parent group.<br>"
        )

    course_instructions.short_description = "Instructions"
    
@admin.register(Major)
class MajorsAdmin(admin.ModelAdmin):
    list_display = ('major_code', 'major_name', 'department', 'program')
    search_fields = ['major_code', 'major_name']
    list_filter = ['department', 'program']
    ordering = ['major_code']
    inlines = [MajorParentCourseGroupInline, MajorCourseGroupInline, CoursesPerMajorInline]

    readonly_fields = ("course_instructions",)

    fieldsets = (
        (None, {
            'fields': ('course_instructions', 'major_code', 'major_name', 'notes'),
        }),
    )
    
    def course_instructions(self, obj):
        return format_html(
            "<b>Instructions for creating or editing majors:</b><br>"
            "1. Add all information for the general major (Major code, Major name, Notes).<br>"
            "2. Click 'Save and continue editing' before moving to the next tab.<br>"
            "<br>"
            "<b>Instructions for adding Courses for a Major:</b><br>"
            "1. Add all courses than can be taken in the major in the 'Courses' tab.<br>"
            "2. Add all information for the course (Course code, Year, Notes (specific to this major, general notes for a course are stored in Courses).<br>"
            "3. Click 'Save and continue editing' to save your progress.<br>"
            "4. To delete a course in a major, check the 'Delete?' box and save your progress.<br>"
            "<br>"
            "<b>Information about how to make options between Courses:</b><br>"
            "<b>Course Groups:</b><br>"
            "Creates choices between multiple courses that a student must do in their major<br>"
            "e.g. If a student must either do MAM1031F or MAM1004F.<br>"
            "<br>"
            "<b>Instructions for adding Course Groups</b><br>"
            "1. Create a course group with 'Required count' equal to the number of courses the student must take in this group.<br>"
            "2. Give it a meaningful description so you can differ between multiple Course groups.<br>"
            "3. Change all courses that must be a part of this group to all share the same 'Course group' that you have just created.<br>"
            "4. To delete a course group, check the 'Delete?' box and save your progress. This will not delete the courses in the group.<br>"
            "5. When you are finished, click 'Save and continue editing' to apply your changes.<br>"
            "<br>"
            "<b>Parent Course Groups:</b><br>"
            "Creates paths of choices between multiple courses that a student must do in their degree<br>"
            "e.g. When a student must either do both MAM1004F and MAM1008S together or both MAM1031F and MAM1032S together.<br>"
            "<br>"
            "<b>Instructions for adding Parent Course Groups:</b><br>"
            "1. Create a parent course group with 'Required count' equal to the number of groups of courses the student must take in this parent group.<br>"
            "   i.e. MAM1004F and MAM1008S with Required count 1 is 1 group and MAM1031F and MAM1032S with Required count 1 is another group.<br>"
            "   Therefore, a student must do either group and then whatever number of courses the Required count is in the subgroup they choose.<br>"
            "2. Give it a meaningful description so you can differ between multiple Parent Course Groups.<br>"
            "3. Create a Course group for each set of courses that should be grouped together eg 'MAM1004F and MAM1008S' then 'MAM1031F and MAM1032S'.<br>"
            "   Required count in the subgroup is how many courses the student must take in that subgroup.<br>"
            "4. To delete a parent course group, check the 'Delete?' box and save your changes.<br>"
            "   This will delete only the parent group and not the courses or groups in the parent group.<br>"
        )

    course_instructions.short_description = "Course Instructions"

# @admin.register(DegreeParentCourseGroup)
# class DegreeParentCourseGroupAdmin(admin.ModelAdmin):
#     list_display = ('description', 'degree', 'group_id')
#     search_fields = ['degree__degree_name', 'group_id']
#     list_filter = ['degree']

# @admin.register(DegreeCourseGroup)
# class DegreeCourseGroupAdmin(admin.ModelAdmin):
#     list_display = ('group_id', 'parent_group', 'degree', 'required_count', 'description')
#     search_fields = ['degree__degree_code', 'description']
#     list_filter = ['degree__degree_code', 'parent_group__group_id']

# @admin.register(CoursesPerDegree)
# class CoursesPerDegreeAdmin(admin.ModelAdmin):
#     list_display = ('course', 'degree', 'course_group', 'year', 'notes')
#     search_fields = ['course', 'degree__degree_name']
#     list_filter = ['degree', 'course_group__group_id', 'year']

# @admin.register(MajorParentCourseGroup)
# class MajorParentCourseGroupAdmin(admin.ModelAdmin):
#     list_display = ('description', 'major', 'group_id')
#     search_fields = ['major__major_name', 'group_id']
#     list_filter = ['major']

# @admin.register(MajorCourseGroup)
# class MajorCourseGroupAdmin(admin.ModelAdmin):
#     list_display = ('group_id', 'parent_group', 'major', 'required_count', 'description')
#     search_fields = ['major__major_code', 'description']
#     list_filter = ['major__major_code', 'parent_group__group_id']

# @admin.register(CoursesPerMajor)
# class CoursesPerMajorAdmin(admin.ModelAdmin):
#     list_display = ('course', 'major', 'course_group', 'year', 'notes')
#     search_fields = ['course__course_code', 'major__major_name']
#     list_filter = ['major__major_code', 'course_group__group_id', 'year']

# @admin.register(Elective)
# class ElectivesAdmin(admin.ModelAdmin):
#     list_display = ('program__program_name', 'course__course_code', 'year')
#     search_fields = ['course__course_code']
#     list_filter = ['program__program_code', 'year']

