from django.shortcuts import render
from .models import *
from handbook.models import Faculty, Department, Program, Degree, Major
from django.http import JsonResponse
import json
import random
from django.views.decorators.csrf import csrf_exempt
from django.contrib.auth.hashers import check_password
from django.utils import timezone

def get_advisors(request):
    advisors = Advisor.objects.all()
    advisors_data = []
    
    for advisor in advisors:
        advisor_data = {
            'username': advisor.username,
            'first_name': advisor.first_name,
            'last_name': advisor.last_name,
            'title': advisor.title,
            'email': advisor.email,
            'phone_number': advisor.phone_number,
            'office_number': advisor.office_number,
            'building': advisor.building,
            'notes': advisor.notes,
            'faculties': list(advisor.faculties.values('faculty_name')),
            'departments': list(advisor.departments.values('department_name')),
            'programs': list(advisor.programs.values('program_name')),
            'degrees': list(advisor.degrees.values('degree_name')),
            'majors': list(advisor.majors.values('major_name')),
        }
        advisors_data.append(advisor_data)
    
    return JsonResponse(advisors_data, safe=False)

def get_advisor_questions(request):
    questions = AdvisorQuestion.objects.all().values('question', 'answer')  # Get as dictionaries
    return JsonResponse(list(questions), safe=False)

def get_advisor_questions_chatbot(request):
    questions = AdvisorQuestion.objects.all().values('question_id','question')  # Get as dictionaries
    return JsonResponse(list(questions), safe=False)

@csrf_exempt
def get_advisor_questions_id(request):
    """
    Fetch advisor questions by their IDs
    Returns:
        "questions": [
            {"question_id": 1, "question": "How do I apply?", "answer": "..."},
            {"question_id": 2, "question": "What are the fees?", "answer": "..."},

    """
    if request.method != 'POST':
        return JsonResponse({'error': 'Only POST method allowed'}, status=405)
    
    try:
        # Parse JSON data from request body
        data = json.loads(request.body)
        question_ids = data.get('question_ids', [])
        if not question_ids:
            return JsonResponse({'error': 'No question_ids provided'}, status=400)
        # Fetch questions - INCLUDE question_id in values()
        questions = AdvisorQuestion.objects.filter(
            question_id__in=question_ids
        ).values('question_id', 'question', 'answer')
        questions_list = list(questions)
        # Optional: Preserve the order of question_ids in the response
        if len(questions_list) > 1:
            questions_dict = {q['question_id']: q for q in questions_list}
            ordered_questions = [
                questions_dict[qid] for qid in question_ids 
                if qid in questions_dict
            ]
            return JsonResponse({
                'questions': ordered_questions
            })
        else:
            return JsonResponse({
                'questions': questions_list
            })
            
    except json.JSONDecodeError as e:
        print(f"JSON decode error: {str(e)}")
        return JsonResponse({'error': 'Invalid JSON in request body'}, status=400)
    except Exception as e:
        print(f"Error in get_advisor_questions_id: {str(e)}")
        return JsonResponse({'error': f'Server error: {str(e)}'}, status=500)

def get_faqs(request):
    faqs = Faq.objects.select_related('section').all().values('section__section_name', 'question', 'answer')
    grouped = {}
    for faq in faqs:
        section_name = faq['section__section_name']
        if section_name not in grouped:
            grouped[section_name] = []
        grouped[section_name].append({
            'question': faq['question'],
            'answer': faq['answer']
        })
    return JsonResponse(grouped, safe=False)

def get_glossary(request):
    glossary = Glossary.objects.all().values('term', 'definition')  # Get as dictionaries
    return JsonResponse(list(glossary), safe=False)

@csrf_exempt
def register_user(request):
    if request.method == "POST":
        try:
            data = json.loads(request.body)
            username = data.get("username")
            password = data.get("password")
            first_name = data.get("first_name")
            last_name = data.get("last_name")
            email = data.get("email")
            if not username or not password:
                return JsonResponse({"error": "Username and password are required."}, status=400)
            if Student.objects.filter(username=username).exists():
                return JsonResponse({"error": "Username is already taken."}, status=400)
            if email and Student.objects.filter(email=email).exists():
                return JsonResponse({"error": "Email is already registered."}, status=400)
            student = Student(
                username=username,
                first_name=first_name,
                last_name=last_name,
                email=email
            )
            student.set_password(password)
            student.save()
            return JsonResponse({"data": {
                        "is_advisor": False,
                        "username": student.username,
                        "red_dot": False,
                        "inbox": []
                    }})
        except Exception as e:
            return JsonResponse({"error": str(e)}, status=400)
    return JsonResponse({"error": "Only POST allowed"}, status=405)

@csrf_exempt
def login_user(request):
    if request.method != "POST":
        return JsonResponse({"error": "Only POST allowed."}, status=405)

    try:
        data = json.loads(request.body)
        username = data.get("username")
        password = data.get("password")

        if not username or not password:
            return JsonResponse({"error": "Username and password required."}, status=400)

        # Try student first
        try:
            student = Student.objects.get(username=username)
            if student.check_password(password):
                # Get advisor questions for this student
                questions = AdvisorQuestion.objects.filter(student=student).order_by('-q_timestamp')
                # Red dot shows when there are unread messages (student_read = False)
                red_dot = any(not q.student_read for q in questions)
                inbox = [
                    {
                        "category": q.category if q.category else "",
                        "subject": q.subject if q.subject else "",
                        "question": q.question,
                        "answer": q.answer if q.answer else "",
                        "student": q.student.username,
                        "advisor": q.advisor.username if q.advisor else "",
                        "read": q.student_read,
                        "question_timestamp": q.q_timestamp.isoformat() if q.q_timestamp else None,
                        **({} if not q.a_timestamp or not q.answer or not q.answer.strip() or q.a_timestamp == q.q_timestamp else {"answer_timestamp": q.a_timestamp.isoformat()})
                    }
                    for q in questions
                ]
                return JsonResponse({
                    "data": {
                        "is_advisor": False,
                        "username": student.username,
                        "red_dot": red_dot, # whether the student has any notifications or not
                        "inbox": inbox # all advisor_question objects related to the this student
                    }
                })
            else:
                return JsonResponse({"error": "Incorrect password."}, status=401)
        except Student.DoesNotExist:
            pass

        # Try advisor
        try:
            advisor = Advisor.objects.get(username=username)
            if advisor.check_password(password):
                # Get advisor questions for this advisor
                questions = AdvisorQuestion.objects.filter(advisor=advisor).order_by('-q_timestamp')
                # Red dot shows when there are unread messages (advisor_read = False)
                red_dot = any(not q.advisor_read for q in questions)
                inbox = [
                    {
                        "category": q.category if q.category else "",
                        "subject": q.subject if q.subject else "",
                        "question": q.question,
                        "answer": q.answer if q.answer else "",
                        "student": q.student.username if q.student else "",
                        "advisor": q.advisor.username,
                        "read": q.advisor_read,
                        "question_timestamp": q.q_timestamp.isoformat() if q.q_timestamp else None,
                        **({} if not q.a_timestamp or not q.answer or not q.answer.strip() or q.a_timestamp == q.q_timestamp else {"answer_timestamp": q.a_timestamp.isoformat()})
                    }
                    for q in questions
                ]
                return JsonResponse({
                    "data": {
                        "is_advisor": True,
                        "username": advisor.username,
                        "red_dot": red_dot,
                        "inbox": inbox
                    }
                })
            else:
                return JsonResponse({"error": "Incorrect password."}, status=401)
        except Advisor.DoesNotExist:
            pass

        return JsonResponse({"error": "User does not exist."}, status=404)

    except Exception as e:
        return JsonResponse({"error": str(e)}, status=500)

@csrf_exempt
def ask_question(request):

    if request.method == "POST":
        try:
            data = json.loads(request.body)
            category = data.get("category")
            subject = data.get("subject")
            question = data.get("question")
            student_username = data.get("student")
            if not student_username:
                return JsonResponse({"error": "Student username required."}, status=400)
            try:
                student = Student.objects.get(username=student_username)
            except Student.DoesNotExist:
                return JsonResponse({"error": "Student not found."}, status=404)
            
            # Check for duplicate question (same student, category, and subject)
            if AdvisorQuestion.objects.filter(
                student=student,
                question=question,
                subject=subject
            ).exists():
                return JsonResponse({"error": "Question has already been asked"}, status=400)
            
            faculty_name_in = data.get("faculty") if data.get("faculty") else None
            department_name_in = data.get("department") if data.get("department") else None
            program_name_in = data.get("program") if data.get("program") else None
            degree = data.get("degree") if data.get("degree") else None
            major = data.get("major") if data.get("major") else None

            # Convert faculty name to faculty ID if provided
            faculty = None
            if faculty_name_in:
                try:
                    faculty_obj = Faculty.objects.get(faculty_name=faculty_name_in)
                    faculty = faculty_obj.faculty_id
                except Faculty.DoesNotExist:
                    return JsonResponse({"error": f"Faculty '{faculty_name_in}' not found."}, status=400)
            
            # Convert department name to department primary key if provided
            department = None
            if department_name_in:
                try:
                    department_obj = Department.objects.get(department_name=department_name_in)
                    department = department_obj.department_code
                except Department.DoesNotExist:
                    return JsonResponse({"error": f"Department '{department_name_in}' not found."}, status=400)
            
            program = None
            if program_name_in:
                try:
                    program_obj = Program.objects.get(program_name=program_name_in)
                    program = program_obj.program_code
                except Program.DoesNotExist:
                    return JsonResponse({"error": f"Program '{program_name_in}' not found."}, status=400)

            # find an advisor based on faculty, department, program etc based on if the advisor is linked to ALL of those categories.
            
            # Start with all advisors and apply all filters that were provided
            advisor_qs = Advisor.objects.all()
            filters_used = []
            
            # Apply all available filters - advisor must match ALL provided criteria
            if faculty:
                advisor_qs = advisor_qs.filter(faculties__pk=faculty)
                filters_used.append('faculty')
            if department:
                advisor_qs = advisor_qs.filter(departments__pk=department)
                filters_used.append('department')
            if program:
                advisor_qs = advisor_qs.filter(programs__pk=program)
                filters_used.append('program')
            if degree:
                advisor_qs = advisor_qs.filter(degrees__pk=degree)
                filters_used.append('degree')
            if major:
                advisor_qs = advisor_qs.filter(majors__pk=major)
                filters_used.append('major')
            
            # Check if we have any filters applied
            if not filters_used:
                return JsonResponse({"error": "At least one category (faculty, department, program, degree, or major) must be provided to find an advisor."}, status=400)
            
            # Random advisor selection when multiple advisors are found
            if advisor_qs.exists():
                advisor_list = list(advisor_qs)
                advisor = random.choice(advisor_list)
            else:
                advisor = None
                
            if not advisor:
                return JsonResponse({"error": "No advisor found for the provided categories. All provided criteria must match an advisor."}, status=404)
            # Create a new AdvisorQuestion
            advisor_question = AdvisorQuestion.objects.create(
                category=category,
                subject=subject,
                question=question,
                answer=None,
                student=student,
                advisor=advisor
            )
            return JsonResponse({"message": "Question submitted successfully.", "advisor": advisor.username, "filters_used": filters_used}, status=201)
        except Exception as e:
            return JsonResponse({"error": str(e)}, status=500)

@csrf_exempt
def answer_question(request):
    if request.method == "POST":
        try:
            data = json.loads(request.body)
            subject = data.get("subject")
            student_username = data.get("student")
            advisor_username = data.get("advisor")
            question_text = data.get("question")
            answer = data.get("answer")
            if not answer or not advisor_username:
                return JsonResponse({"error": "answer and advisor username required."}, status=400)
            try:
                advisor = Advisor.objects.get(username=advisor_username)
            except Advisor.DoesNotExist:
                return JsonResponse({"error": "Advisor not found."}, status=404)
            try:
                student = Student.objects.get(username=student_username)
            except Student.DoesNotExist:
                return JsonResponse({"error": "Student not found."}, status=404)
            # Find question by question_id or by subject, student, advisor, and question text
            question = AdvisorQuestion.objects.filter(
                subject=subject,
                advisor=advisor,
                student=student,
                question=question_text
            ).first()
            if not question:
                return JsonResponse({"error": "Question not found for this advisor and student."}, status=404)
            question.answer = answer
            question.student_read = False  # Student hasn't read the new answer yet
            question.advisor_read = True   # Advisor has provided the answer
            question.a_timestamp = timezone.now()  # Set answer timestamp
            question.save()
            return JsonResponse({"message": "Answer submitted successfully."}, status=200)
        except Exception as e:
            return JsonResponse({"error": str(e)}, status=500)
    return JsonResponse({"error": "Only POST allowed"}, status=405)

@csrf_exempt
def refresh_user_data(request):
    if request.method == "POST":
        try:
            data = json.loads(request.body)
            username = data.get("username")
            is_advisor = data.get("is_advisor", False)
            incoming_inbox = data.get("inbox", [])
            
            if not username:
                return JsonResponse({"error": "Username required."}, status=400)
            
            if not is_advisor:
                # Handle student
                try:
                    student = Student.objects.get(username=username)
                except Student.DoesNotExist:
                    return JsonResponse({"error": "Student not found."}, status=404)

                # Update read status for messages in the incoming inbox
                if incoming_inbox:
                    for message in incoming_inbox:
                        # Find the corresponding message in the database by question text
                        try:
                            question_obj = AdvisorQuestion.objects.get(
                                student=student,
                                question=message.get("question", "")
                            )
                            
                            # Update student_read status
                            message_read_status = message.get("read", False)
                            if question_obj.student_read != message_read_status:
                                question_obj.student_read = message_read_status
                                question_obj.save()
                                
                        except AdvisorQuestion.DoesNotExist:
                            # Message not found in database, skip
                            continue
                        except AdvisorQuestion.MultipleObjectsReturned:
                            # Multiple messages with same content, update all of them
                            question_objs = AdvisorQuestion.objects.filter(
                                student=student,
                                question=message.get("question", "")
                            )
                            message_read_status = message.get("read", False)
                            for q_obj in question_objs:
                                if q_obj.student_read != message_read_status:
                                    q_obj.student_read = message_read_status
                                    q_obj.save()

                # Get updated advisor questions for this student
                questions = AdvisorQuestion.objects.filter(student=student).order_by('-q_timestamp')
                red_dot = any(not q.student_read for q in questions)
                
                inbox = [
                    {
                        "category": q.category if q.category else "",
                        "subject": q.subject if q.subject else "",
                        "question": q.question,
                        "answer": q.answer if q.answer else "",
                        "student": q.student.username,
                        "advisor": q.advisor.username if q.advisor else "",
                        "read": q.student_read,
                        "question_timestamp": q.q_timestamp.isoformat() if q.q_timestamp else None,
                        **({} if not q.a_timestamp or not q.answer or not q.answer.strip() or q.a_timestamp == q.q_timestamp else {"answer_timestamp": q.a_timestamp.isoformat()})
                    }
                    for q in questions
                ]
                
                return JsonResponse({
                    "data": {
                        "is_advisor": False,
                        "username": student.username,
                        "red_dot": red_dot,
                        "inbox": inbox
                    }
                })
                
            else:
                # Handle advisor
                try:
                    advisor = Advisor.objects.get(username=username)
                except Advisor.DoesNotExist:
                    return JsonResponse({"error": "Advisor not found."}, status=404)

                # Update read status for messages in the incoming inbox
                if incoming_inbox:
                    for message in incoming_inbox:
                        # Find the corresponding message in the database by question text
                        try:
                            question_obj = AdvisorQuestion.objects.get(
                                advisor=advisor,
                                question=message.get("question", "")
                            )
                            
                            # Update advisor_read status
                            message_read_status = message.get("read", False)
                            if question_obj.advisor_read != message_read_status:
                                question_obj.advisor_read = message_read_status
                                question_obj.save()
                                
                        except AdvisorQuestion.DoesNotExist:
                            # Message not found in database, skip
                            continue
                        except AdvisorQuestion.MultipleObjectsReturned:
                            # Multiple messages with same content, update all of them
                            question_objs = AdvisorQuestion.objects.filter(
                                advisor=advisor,
                                question=message.get("question", "")
                            )
                            message_read_status = message.get("read", False)
                            for q_obj in question_objs:
                                if q_obj.advisor_read != message_read_status:
                                    q_obj.advisor_read = message_read_status
                                    q_obj.save()

                # Get updated advisor questions for this advisor
                questions = AdvisorQuestion.objects.filter(advisor=advisor).order_by('-q_timestamp')
                red_dot = any(not q.advisor_read for q in questions)
                
                inbox = [
                    {
                        "category": q.category if q.category else "",
                        "subject": q.subject if q.subject else "",
                        "question": q.question,
                        "answer": q.answer if q.answer else "",
                        "student": q.student.username if q.student else "",
                        "advisor": q.advisor.username if q.advisor else "",
                        "read": q.advisor_read,
                        "question_timestamp": q.q_timestamp.isoformat() if q.q_timestamp else None,
                        **({} if not q.a_timestamp or not q.answer or not q.answer.strip() or q.a_timestamp == q.q_timestamp else {"answer_timestamp": q.a_timestamp.isoformat()})
                    }
                    for q in questions
                ]
                
                return JsonResponse({
                    "data": {
                        "is_advisor": True,
                        "username": advisor.username,
                        "red_dot": red_dot,
                        "inbox": inbox
                    }
                })
            
        except json.JSONDecodeError:
            return JsonResponse({"error": "Invalid JSON in request body."}, status=400)
        except Exception as e:
            return JsonResponse({"error": str(e)}, status=500)
    
    return JsonResponse({"error": "Only POST allowed"}, status=405)
