from django.db import models
from django.contrib.auth.hashers import make_password, check_password
from django.db import models
from django.contrib.auth.models import AbstractBaseUser, BaseUserManager, PermissionsMixin
from django.utils import timezone


class AdvisorManager(BaseUserManager):
    def create_user(self, username, email, password=None, **extra_fields):
        if not email:
            raise ValueError('Advisors must have an email address')
        if not username:
            raise ValueError('Advisors must have a username')

        email = self.normalize_email(email)
        user = self.model(username=username, email=email, **extra_fields)
        user.set_password(password)
        user.save(using=self._db)
        return user

    def create_superuser(self, username, email, password=None, **extra_fields):
        extra_fields.setdefault('is_staff', True)
        extra_fields.setdefault('is_superuser', True)

        if extra_fields.get('is_staff') is not True:
            raise ValueError('Superuser must have is_staff=True.')
        if extra_fields.get('is_superuser') is not True:
            raise ValueError('Superuser must have is_superuser=True.')

        return self.create_user(username, email, password, **extra_fields)


class Advisor(AbstractBaseUser, PermissionsMixin):
    advisor_id = models.AutoField(primary_key=True)
    username = models.CharField(max_length=255, unique=True)
    password = models.CharField(max_length=255)
    first_name = models.CharField(max_length=255, blank=True, null=True)
    last_name = models.CharField(max_length=255, blank=True, null=True)
    email = models.EmailField(max_length=255, unique=True, blank=True, null=True)
    title = models.CharField(max_length=255, null=True, blank=True)
    phone_number = models.CharField(max_length=255, blank=True, null=True)
    building = models.CharField(max_length=255, blank=True, null=True)
    office_number = models.CharField(max_length=255, blank=True, null=True)

    # Relationships
    faculties = models.ManyToManyField("handbook.Faculty", blank=True)
    departments = models.ManyToManyField("handbook.Department", blank=True)
    programs = models.ManyToManyField("handbook.Program", blank=True)
    degrees = models.ManyToManyField("handbook.Degree", blank=True)
    majors = models.ManyToManyField("handbook.Major", blank=True)

    # Optional fields
    profile_photo = models.BinaryField(blank=True, null=True)
    notes = models.TextField(blank=True, null=True)
    is_active = models.BooleanField(default=True)
    is_staff = models.BooleanField(default=False)
    date_joined = models.DateTimeField(default=timezone.now)
    last_login = models.DateTimeField(blank=True, null=True)

    # Permissions
    groups = models.ManyToManyField(
        'auth.Group',
        related_name='advisor_set',
        blank=True,
        help_text='The groups this advisor belongs to.',
        verbose_name='groups'
    )
    user_permissions = models.ManyToManyField(
        'auth.Permission',
        related_name='advisor_set',
        blank=True,
        help_text='Specific permissions for this advisor.',
        verbose_name='user permissions'
    )

    objects = AdvisorManager()

    USERNAME_FIELD = 'username'
    REQUIRED_FIELDS = ['email']

    class Meta:
        db_table = 'advisor'

    def __str__(self):
        if self.first_name:
            if self.title:
                return f"{self.title} {self.first_name or ''} {self.last_name or ''}"
            return f"{self.first_name or ''} {self.last_name or ''}"
        return self.username
        

class StudentManager(BaseUserManager):
    def create_user(self, username, email, password=None, **extra_fields):
        if not email:
            raise ValueError('Users must have an email address')
        if not username:
            raise ValueError('Users must have a username')
        
        email = self.normalize_email(email)
        user = self.model(username=username, email=email, **extra_fields)
        user.set_password(password)
        user.save(using=self._db)
        return user

    def create_superuser(self, username, email, password=None, **extra_fields):
        extra_fields.setdefault('is_staff', True)
        extra_fields.setdefault('is_superuser', True)

        if extra_fields.get('is_staff') is not True:
            raise ValueError('Superuser must have is_staff=True.')
        if extra_fields.get('is_superuser') is not True:
            raise ValueError('Superuser must have is_superuser=True.')

        return self.create_user(username, email, password, **extra_fields)


class Student(AbstractBaseUser, PermissionsMixin):
    student_id = models.AutoField(primary_key=True)
    username = models.CharField(max_length=255, unique=True)
    password = models.CharField(max_length=255)
    first_name = models.CharField(max_length=255, blank=True, null=True)
    last_name = models.CharField(max_length=255, blank=True, null=True)
    email = models.EmailField(max_length=255, unique=True, blank=True, null=True)

    # Optional fields
    is_active = models.BooleanField(default=True)
    is_staff = models.BooleanField(default=False)
    date_joined = models.DateTimeField(default=timezone.now)
    last_login = models.DateTimeField(blank=True, null=True)

    # Override the related_name to avoid clashes with auth.User
    groups = models.ManyToManyField(
        'auth.Group',
        related_name='student_set',
        blank=True,
        help_text='The groups this user belongs to.',
        verbose_name='groups'
    )
    user_permissions = models.ManyToManyField(
        'auth.Permission',
        related_name='student_set',
        blank=True,
        help_text='Specific permissions for this user.',
        verbose_name='user permissions'
    )

    objects = StudentManager()

    USERNAME_FIELD = 'username'
    REQUIRED_FIELDS = ['email']

    class Meta:
        db_table = 'student'

    def __str__(self):
        return self.username


class AdvisorQuestion(models.Model):
    question_id = models.AutoField(primary_key=True)
    category = models.CharField(max_length=255, blank=True, null=True, help_text="Can only be 1 of 'faculty', 'department', 'program', 'degree', or 'major'")
    subject = models.CharField(max_length=255, blank=True, null=True)
    question = models.TextField()
    answer = models.TextField(blank=True, null=True)
    q_timestamp = models.DateTimeField(auto_now_add=True, blank=True, null=True)
    a_timestamp = models.DateTimeField(blank=True, null=True)
    student = models.ForeignKey(Student, models.CASCADE, db_column='student', to_field='student_id', null=True, blank=True)
    advisor = models.ForeignKey(Advisor, models.CASCADE, db_column='advisor', to_field='advisor_id', null=True, blank=True)
    student_read = models.BooleanField(default=True)
    advisor_read = models.BooleanField(default=False)

    def __str__(self):
        return f"{self.question}"

    class Meta:
        db_table = 'advisor_question'


class FaqSection(models.Model):
    section_id = models.IntegerField(primary_key=True)
    section_name = models.CharField(max_length=255)

    def __str__(self):
        return self.section_name

    class Meta:
        db_table = 'faq_section'
        verbose_name_plural = "FAQ Sections"


class Faq(models.Model):
    faq_id = models.AutoField(primary_key=True)
    question = models.TextField()
    answer = models.TextField()
    section = models.ForeignKey(FaqSection, models.CASCADE)

    def __str__(self):
        return str(f"{self.section} - {self.question}")

    class Meta:
        db_table = 'faq'
        verbose_name_plural = "FAQs"


class Glossary(models.Model):
    term_id = models.AutoField(primary_key=True)
    term = models.CharField(max_length=255, unique=True)
    definition = models.TextField()

    def __str__(self):
        return f"{self.term}"

    class Meta:
        db_table = 'glossary'
        verbose_name = 'term'
        verbose_name_plural = 'Glossary'