-module(gmysql).
-compile([no_auto_import, nowarn_unused_vars, nowarn_unused_function, nowarn_nomatch]).

-export([default_config/0, connect/1, with_connection/2, exec_with_timeout/3, exec/2, to_param/1, null_param/0, to_pid/1, from_pid/1, query_with_timeout/5, 'query'/4, with_transaction/3, disconnect/1]).
-export_type([connection/0, timeout_/0, connection_mode/0, connection_option/0, config/0, error/0, param/0, transaction_error/1]).

-type connection() :: any().

-type timeout_() :: infinity | {ms, integer()}.

-type connection_mode() :: synchronous | asynchronous | lazy.

-type connection_option() :: {host, gleam@erlang@charlist:charlist()} |
    {port, integer()} |
    {user, gleam@erlang@charlist:charlist()} |
    {password, gleam@erlang@charlist:charlist()} |
    {database, gleam@erlang@charlist:charlist()} |
    {connect_mode, connection_mode()} |
    {connect_timeout, integer()} |
    {keep_alive, integer()}.

-type config() :: {config,
        binary(),
        integer(),
        gleam@option:option(binary()),
        gleam@option:option(binary()),
        binary(),
        connection_mode(),
        timeout_(),
        integer()}.

-type error() :: {server_error, integer(), bitstring()} |
    {unknown_error, gleam@dynamic:dynamic_()} |
    {decode_error, list(gleam@dynamic:decode_error())}.

-type param() :: any().

-type transaction_error(HNC) :: {function_error, HNC} |
    {other_error, gleam@dynamic:dynamic_()}.

-spec default_config() -> config().
default_config() ->
    {config,
        <<"localhost"/utf8>>,
        3306,
        none,
        none,
        <<"db"/utf8>>,
        asynchronous,
        infinity,
        1000}.

-spec config_to_connection_options(config()) -> list(connection_option()).
config_to_connection_options(Config) ->
    _pipe@5 = [{some,
            {host,
                begin
                    _pipe = erlang:element(2, Config),
                    unicode:characters_to_list(_pipe)
                end}},
        {some, {port, erlang:element(3, Config)}},
        begin
            _pipe@1 = gleam@option:map(
                erlang:element(4, Config),
                fun unicode:characters_to_list/1
            ),
            gleam@option:map(_pipe@1, fun(Field@0) -> {user, Field@0} end)
        end,
        begin
            _pipe@2 = gleam@option:map(
                erlang:element(5, Config),
                fun unicode:characters_to_list/1
            ),
            gleam@option:map(_pipe@2, fun(Field@0) -> {password, Field@0} end)
        end,
        {some,
            {database,
                begin
                    _pipe@3 = erlang:element(6, Config),
                    unicode:characters_to_list(_pipe@3)
                end}},
        {some, {connect_mode, erlang:element(7, Config)}},
        {some,
            {connect_timeout,
                begin
                    _pipe@4 = erlang:element(8, Config),
                    gmysql_ffi:from_timeout(_pipe@4)
                end}},
        {some, {keep_alive, erlang:element(9, Config)}}],
    gleam@option:values(_pipe@5).

-spec connect(config()) -> {ok, connection()} |
    {error, gleam@dynamic:dynamic_()}.
connect(Config) ->
    _pipe = config_to_connection_options(Config),
    gmysql_ffi:connect(_pipe).

-spec with_connection(config(), fun((connection()) -> HNM)) -> {ok, HNM} |
    {error, gleam@dynamic:dynamic_()}.
with_connection(Config, Function) ->
    _pipe = config_to_connection_options(Config),
    gmysql_ffi:with_connection(_pipe, Function).

-spec exec_with_timeout(binary(), connection(), timeout_()) -> {ok, nil} |
    {error, error()}.
exec_with_timeout(Sql, Connection, Timeout) ->
    gmysql_ffi:exec(Connection, Sql, Timeout).

-spec exec(binary(), connection()) -> {ok, nil} | {error, error()}.
exec(Sql, Connection) ->
    exec_with_timeout(Sql, Connection, infinity).

-spec to_param(any()) -> param().
to_param(Param) ->
    gmysql_ffi:to_param(Param).

-spec null_param() -> param().
null_param() ->
    gmysql_ffi:null_param().

-spec to_pid(connection()) -> gleam@erlang@process:pid_().
to_pid(Connection) ->
    gmysql_ffi:to_pid(Connection).

-spec from_pid(gleam@erlang@process:pid_()) -> connection().
from_pid(Connection) ->
    gmysql_ffi:from_pid(Connection).

-spec query_with_timeout(
    binary(),
    connection(),
    list(param()),
    fun((gleam@dynamic:dynamic_()) -> {ok, HOJ} |
        {error, list(gleam@dynamic:decode_error())}),
    timeout_()
) -> {ok, list(HOJ)} | {error, error()}.
query_with_timeout(Sql, Connection, Arguments, Decoder, Timeout) ->
    case gmysql_ffi:'query'(Connection, Sql, Arguments, Timeout) of
        {error, Int} ->
            {error, Int};

        {ok, Dyn} ->
            case (gleam@dynamic:list(Decoder))(Dyn) of
                {ok, Decoded} ->
                    {ok, Decoded};

                {error, Decode_errors} ->
                    {error, {decode_error, Decode_errors}}
            end
    end.

-spec 'query'(
    binary(),
    connection(),
    list(param()),
    fun((gleam@dynamic:dynamic_()) -> {ok, HOB} |
        {error, list(gleam@dynamic:decode_error())})
) -> {ok, list(HOB)} | {error, error()}.
'query'(Sql, Connection, Arguments, Decoder) ->
    query_with_timeout(Sql, Connection, Arguments, Decoder, infinity).

-spec with_transaction(
    connection(),
    integer(),
    fun((connection()) -> {ok, HOQ} | {error, HOR})
) -> {ok, HOQ} | {error, transaction_error(HOR)}.
with_transaction(Connection, Retries, Function) ->
    gmysql_ffi:with_transaction(Connection, Retries, Function).

-spec disconnect(connection()) -> nil.
disconnect(Connection) ->
    gmysql_ffi:close(Connection).
