-module(based@testing).
-compile([no_auto_import, nowarn_unused_vars, nowarn_unused_function, nowarn_nomatch]).

-export([new_state/0, empty_returns_for/1, add/3, mock_service/2, insert/2, with_connection/2, mock_adapter/1]).
-export_type([connection/0, state/0, message/0]).

-type connection() :: {connection, gleam@erlang@process:subject(message())}.

-type state() :: {state,
        gleam@dict:dict(binary(), list(gleam@dynamic:dynamic_()))}.

-type message() :: shutdown |
    {insert, state()} |
    {get,
        gleam@erlang@process:subject({ok, list(gleam@dynamic:dynamic_())} |
            {error, based:based_error()}),
        binary()}.

-spec new_state() -> state().
new_state() ->
    {state, gleam@dict:new()}.

-spec empty_returns_for(list(based:'query'())) -> state().
empty_returns_for(Queries) ->
    _pipe = Queries,
    _pipe@1 = gleam@list:map(
        _pipe,
        fun(Query) -> {erlang:element(2, Query), []} end
    ),
    _pipe@2 = maps:from_list(_pipe@1),
    {state, _pipe@2}.

-spec add(state(), binary(), list(gleam@dynamic:dynamic_())) -> state().
add(State, Key, Value) ->
    New_state = begin
        _pipe = erlang:element(2, State),
        gleam@dict:insert(_pipe, Key, Value)
    end,
    {state, New_state}.

-spec mock_service(based:'query'(), connection()) -> {ok,
        list(gleam@dynamic:dynamic_())} |
    {error, based:based_error()}.
mock_service(Query, Conn) ->
    {connection, Subject} = Conn,
    gleam@erlang@process:call(
        Subject,
        fun(_capture) -> {get, _capture, erlang:element(2, Query)} end,
        10
    ).

-spec insert(connection(), state()) -> connection().
insert(Conn, State) ->
    {connection, Subject} = Conn,
    gleam@erlang@process:send(Subject, {insert, State}),
    Conn.

-spec handle_message(
    message(),
    gleam@dict:dict(binary(), list(gleam@dynamic:dynamic_()))
) -> gleam@otp@actor:next(message(), gleam@dict:dict(binary(), list(gleam@dynamic:dynamic_()))).
handle_message(Message, Store) ->
    case Message of
        shutdown ->
            {stop, normal};

        {insert, State} ->
            {state, Init} = State,
            _pipe = Store,
            _pipe@1 = gleam@dict:merge(_pipe, Init),
            gleam@otp@actor:continue(_pipe@1);

        {get, Client, Key} ->
            Value = begin
                _pipe@2 = Store,
                _pipe@3 = gleam@dict:get(_pipe@2, Key),
                gleam@result:replace_error(
                    _pipe@3,
                    {based_error,
                        <<""/utf8>>,
                        <<"query_error"/utf8>>,
                        <<"Query could not be completed"/utf8>>}
                )
            end,
            gleam@erlang@process:send(Client, Value),
            gleam@otp@actor:continue(Store)
    end.

-spec with_connection(state(), fun((connection()) -> HKG)) -> HKG.
with_connection(State, Callback) ->
    _assert_subject = gleam@otp@actor:start(
        erlang:element(2, State),
        fun handle_message/2
    ),
    {ok, Conn} = case _assert_subject of
        {ok, _} -> _assert_subject;
        _assert_fail ->
            erlang:error(#{gleam_error => let_assert,
                        message => <<"Assertion pattern match failed"/utf8>>,
                        value => _assert_fail,
                        module => <<"based/testing"/utf8>>,
                        function => <<"with_connection"/utf8>>,
                        line => 49})
    end,
    Result = begin
        _pipe = Conn,
        _pipe@1 = {connection, _pipe},
        Callback(_pipe@1)
    end,
    gleam@erlang@process:send(Conn, shutdown),
    Result.

-spec mock_adapter(state()) -> based:based_adapter(state(), connection(), any()).
mock_adapter(State) ->
    {based_adapter, fun with_connection/2, State, fun mock_service/2}.
