# This class contains auxiliary methods used by different parts of the programme

from flair.visual.training_curves import Plotter
from flair.data import Dictionary

def get_tags(sentence):
    """Takes a Flair Sentence object and returns the tag for each token as a list

    Parameters
    ----------
    sentence : flair.data.Sentence
        A Sentence object

    Returns
    -------
    [string]
        A list of strings of the tag/tags for each token in the sentence
    """
    return [tag.value for tag in sentence]

def get_abstract_token(sentence):
    """Gets a tree2label representation of a sentence: (token, part of speech)

    Parameters
    ----------
    sentence :flair.data.Sentence
        A Sentence object

    Returns
    -------
    [tuple]
        List of (token, POS) tuples
    """
    return [(word.text, "XX") for word in sentence]

def get_surface_token(sentence):
    return [word.text for word in sentence]

def plot_loss_curves(path):
    """Plots loss curves for a given model

    Parameters
    ----------
    path : String
        Path to model folder 
    """
    plotter = Plotter()
    print("Plotting loss curves now...")
    plotter.plot_training_curves(f'{path}/loss.tsv')

def plot_learning_rate_curves(path):
    """Plots learning rate surves for a given model

    Parameters
    ----------
    path : String
        Path to model folder
    """
    plotter = Plotter()
    plotter.plot_learning_rate(f"{path}/learning_rate.tsv", skip_first=0, skip_last=0)

def check_space(item):
    """Checks if input it a space or not

    Parameters
    ----------
    item : String
        Any string

    Returns
    -------
    Boolean
        Returns False is item is a space, True otherwise
    """
    if item == " ":
        return False
    else:
        return True

def remove_semi_colon(item):
    """Removes leading/trailing semi-colon from string

    Parameters
    ----------
    item : String
        Any string

    Returns
    -------
    String
        String without ;
    """
    return item.strip(";")

def get_numeric_value_for_text_labels(label_list, ner_dictionary : Dictionary):
    """Creates a numeric representation of list of labels

    Parameters
    ----------
    label_list : [Strings]
        List of token labels
    ner_dictionary : Dictionary
        Dictionary for a given label type

    Returns
    -------
    _type_
        _description_
    """
    result = ner_dictionary.get_idx_for_items(label_list)
    return result  

def strip_bio_tags(labelled_string):
    """Removes BIO tags from a string

    Parameters
    ----------
    labelled_string : String
        Removes A/X/D from input

    Returns
    -------
    String
        String without BIO tags
    """
    if "A-" in labelled_string:
        return labelled_string.replace("A-","")
    elif "X-" in labelled_string:
        return labelled_string.replace("X-","")
    elif "D-" in labelled_string:
        return labelled_string.replace("D-","")
    else:
        return labelled_string       


abs_gold = ['8:8 XX comp;11:11 XX compound', '1:1 XX q;2:2 XX q;15:15 XX superl;18:18 XX q;22:22 XX q;23:23 XX q;0:25 XX compound', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;8:8 XX d;7:8 XX compound;6:8 XX compound;5:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:19 XX compound;18:19 XX compound;0:19 XX focus', '3:4 XX compound;5:6 XX proper;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;0:18 XX appos', '0:16 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;0:9 XX focus', '0:0 XX nonsp;0:1 XX loc;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;6:15 XX udef;16:16 XX d;17:17 XX d;18:18 XX d;20:20 XX q;20:21 XX udef;22:22 XX d;20:22 XX compound;23:23 XX d;24:24 XX d;0:24 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;13:13 XX q;13:13 XX compound;12:13 XX udef;14:14 XX d;12:14 XX udef;15:15 XX d;16:17 XX proper;18:18 XX q;19:19 XX q;21:21 XX q;20:21 XX compound;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX proper;29:29 XX q;32:32 XX compound;26:32 XX udef;24:32 XX udef;20:32 XX proper;18:32 XX udef;0:32 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX nonsp;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;0:17 XX focus', '11:12 XX proper;19:19 XX q;18:19 XX compound;18:20 XX proper;21:21 XX q;23:23 XX q;22:23 XX proper;24:24 XX q;25:25 XX q;25:25 XX proper;22:25 XX udef;18:25 XX udef;17:25 XX compound', '0:22 XX subord', '11:11 XX q;0:19 XX compound', '11:11 XX q;12:12 XX q;21:21 XX q;21:21 XX compound;0:27 XX appos', '3:4 XX compound;10:10 XX q', '1:1 XX d;2:2 XX q;2:2 XX compound;5:5 XX q;4:5 XX compound;2:5 XX udef;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX q;16:16 XX q;17:17 XX q;17:17 XX compound;16:17 XX udef;15:17 XX udef;18:18 XX d;15:18 XX udef;7:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX q;29:29 XX d;30:30 XX d;32:32 XX d;33:33 XX q;33:33 XX nominalization;0:33 XX def', '11:12 XX proper;21:21 XX q', '', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;10:10 XX d;8:10 XX compound;11:11 XX d;13:13 XX d;14:14 XX p;15:15 XX p;16:16 XX p;17:17 XX d;31:31 XX q;32:32 XX q;32:32 XX udef;30:32 XX udef;0:32 XX focus', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;0:13 XX udef', '0:0 XX q;1:1 XX q;3:3 XX d;2:3 XX udef;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX q;17:17 XX d;18:18 XX q;18:18 XX nominalization;20:20 XX q;20:20 XX nominalization;0:21 XX udef', '0:0 XX q;1:1 XX q;0:7 XX udef', '1:1 XX d;3:3 XX d;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;10:10 XX q;21:21 XX q;0:21 XX focus', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX q;11:11 XX q;10:11 XX compound;9:11 XX udef;12:12 XX q;13:13 XX udef;0:13 XX focus', '9:9 XX q;9:9 XX compound;8:9 XX udef;10:11 XX proper;10:12 XX compound', '0:0 XX superl', '4:4 XX q;8:8 XX q;7:8 XX compound;12:12 XX q;12:12 XX compound;11:12 XX proper;11:13 XX udef;25:25 XX q;26:26 XX q;0:26 XX udef', '8:9 XX compound', '3:3 XX q;17:17 XX q;17:17 XX nominalization', '3:3 XX q;0:4 XX compound', '4:4 XX q;5:5 XX q;6:6 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;12:12 XX q;12:12 XX compound;11:12 XX udef;11:12 XX proper', '10:10 XX q;12:12 XX q;14:14 XX d;17:17 XX d;18:18 XX nonsp;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX superl;25:26 XX compound;28:28 XX q;27:28 XX udef;0:28 XX appos', '', '1:1 XX d;2:2 XX d;3:3 XX d;4:5 XX proper;11:12 XX compound;9:12 XX compound;21:21 XX udef;19:21 XX udef;18:21 XX udef;0:21 XX proper', '13:13 XX nonsp;15:15 XX nonsp', '1:2 XX compound;3:4 XX compound;7:8 XX compound;9:9 XX q;15:15 XX generic;10:15 XX appos', '0:0 XX conj;1:1 XX conj;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX conj;7:7 XX conj;8:8 XX d;10:10 XX q;10:11 XX proper;14:14 XX d;13:14 XX compound;15:15 XX d;16:16 XX q;17:17 XX q;19:19 XX q;18:20 XX udef;22:22 XX q;23:23 XX q;26:27 XX proper;26:27 XX compound;24:27 XX compound;16:27 XX udef;0:27 XX implicit', '10:10 XX q;10:11 XX udef;16:16 XX q;17:17 XX q;33:33 XX q;34:34 XX q', '13:13 XX q;0:17 XX appos', '0:0 XX neg;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX compound;16:17 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;20:20 XX d;0:21 XX focus', '11:11 XX thing', '0:0 XX q;1:1 XX q;16:16 XX q;21:21 XX udef;0:21 XX proper', '6:6 XX q;14:14 XX q;15:16 XX compound', '14:14 XX q;15:15 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX proper;8:8 XX d;6:8 XX proper;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX q;18:18 XX d;19:19 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:27 XX focus', '', '3:3 XX q;0:5 XX proper', '0:0 XX d;1:1 XX d;3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX p;17:17 XX p;18:18 XX p;16:19 XX with;5:19 XX udef;0:20 XX focus', '6:6 XX q', '8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;15:15 XX q;15:15 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;20:20 XX q;20:20 XX nominalization;22:22 XX q;22:22 XX nominalization;24:24 XX q;24:24 XX nominalization', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;6:6 XX q;7:7 XX udef;0:7 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;17:17 XX d;16:17 XX compound;18:18 XX d;20:20 XX d;23:23 XX d;22:23 XX compound;24:24 XX d;27:27 XX d;0:27 XX focus', '4:4 XX q;7:7 XX q;8:8 XX q', '0:0 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;0:24 XX proper', '0:0 XX neg', '0:0 XX d;4:5 XX proper;7:7 XX q;4:7 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;16:16 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX d;22:25 XX udef;26:26 XX d;27:27 XX q;28:28 XX q;0:28 XX focus', '1:1 XX neg', '10:10 XX comp;13:13 XX comp;17:17 XX q;18:18 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;5:5 XX q;4:5 XX compound;6:6 XX q;8:8 XX q;8:8 XX compound;7:8 XX proper;9:9 XX d;7:9 XX proper;10:10 XX d;11:11 XX d;12:12 XX nonsp;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;0:18 XX focus', '7:7 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;12:13 XX udef;18:19 XX udef;20:20 XX q;20:21 XX udef;23:23 XX q;23:24 XX udef;20:24 XX udef;18:24 XX udef;17:24 XX udef;27:27 XX q;28:28 XX q;28:28 XX udef', '0:15 XX compound', '1:1 XX q;2:2 XX q;3:4 XX loc;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:14 XX compound;3:14 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX q;15:15 XX d;16:16 XX d;17:17 XX q;0:17 XX udef', '6:7 XX compound;10:10 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;20:23 XX udef;16:23 XX udef', '0:0 XX q;0:0 XX compound;1:1 XX q;1:1 XX compound;2:2 XX q;2:2 XX compound;10:10 XX q;11:11 XX q;0:11 XX udef', '5:7 XX generic', '5:5 XX q;25:27 XX proper;28:29 XX compound', '4:4 XX q;6:6 XX nonsp;7:8 XX proper;5:8 XX loc;15:15 XX q;16:16 XX q', '', '3:3 XX q;4:4 XX q;4:4 XX compound;3:4 XX udef;2:4 XX udef;5:5 XX conj;2:5 XX udef;6:6 XX conj;7:7 XX conj', '4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX proper;8:8 XX q;6:9 XX udef;13:13 XX q;15:15 XX conj;17:17 XX conj;18:18 XX conj;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX compound;19:23 XX udef', '1:1 XX neg;5:5 XX q;7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;14:14 XX nonsp;13:14 XX compound;15:15 XX nonsp;17:17 XX nonsp;19:19 XX nonsp;20:20 XX nonsp;21:21 XX nonsp', '6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:15 XX udef;11:15 XX udef;9:15 XX udef;17:18 XX udef;19:19 XX neg;29:29 XX nominalization', '5:5 XX q;6:6 XX q;6:6 XX compound;5:6 XX udef;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;12:12 XX compound;12:14 XX udef;15:15 XX q;12:15 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;23:23 XX q;22:23 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;27:27 XX udef;25:27 XX udef;22:27 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;6:7 XX compound;8:8 XX d;0:17 XX focus', '0:1 XX proper;0:12 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;17:18 XX subord;10:18 XX udef;35:35 XX q;0:35 XX proper', '5:6 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper;19:20 XX proper', '', '11:12 XX compound;10:12 XX compound', '0:0 XX q;0:0 XX compound;13:13 XX p;14:14 XX p;0:15 XX udef', '4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:7 XX proper;8:8 XX q;8:8 XX nominalization;6:8 XX compound;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;18:18 XX q;18:18 XX nominalization;17:18 XX compound;16:18 XX compound;19:19 XX q;19:19 XX nominalization;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;26:26 XX q;20:26 XX udef', '2:3 XX compound;5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;9:12 XX udef;19:19 XX q', '0:1 XX proper;6:6 XX q;9:9 XX q;8:9 XX compound;7:9 XX compound;10:10 XX q;11:12 XX udef;18:18 XX q;22:22 XX nominalization;0:24 XX compound', '3:3 XX q;3:3 XX compound;2:4 XX udef;5:5 XX q;2:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q', '13:13 XX nonsp;14:14 XX nonsp;15:15 XX nonsp;16:16 XX nonsp;17:17 XX nonsp;18:18 XX nonsp;20:20 XX compound;19:20 XX proper', '10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;15:26 XX udef;12:26 XX udef;0:26 XX compound', '0:1 XX udef;7:7 XX neg;0:21 XX compound', '4:4 XX q;5:5 XX q;6:6 XX loc', '0:0 XX nominalization;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:13 XX compound;11:13 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;18:19 XX udef;20:20 XX q;21:21 XX udef;18:21 XX udef', '4:4 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:14 XX compound', '6:7 XX compound;5:7 XX proper;0:12 XX compound', '5:5 XX q;9:9 XX neg;11:11 XX q;13:13 XX q;15:15 XX q;14:15 XX udef;16:16 XX p;17:17 XX p;19:19 XX p;20:20 XX p;21:22 XX compound;16:22 XX with;30:30 XX q;31:32 XX compound;31:32 XX compound', '', '0:0 XX q;0:0 XX compound;8:8 XX neg;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;0:27 XX proper', '0:12 XX measure', '3:3 XX q;4:4 XX q;4:4 XX compound;3:4 XX udef;6:6 XX neg;17:17 XX q;17:17 XX compound;16:17 XX proper', '7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;12:20 XX udef', '10:10 XX q;11:11 XX q;12:12 XX q;12:13 XX proper;14:14 XX q;12:14 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX udef;22:23 XX udef;19:23 XX udef;12:23 XX udef', '5:5 XX q;6:6 XX q;7:8 XX proper;17:17 XX d;17:18 XX parg', '', '15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;23:24 XX udef;25:25 XX q;26:26 XX q;20:26 XX udef', '0:0 XX q;0:0 XX compound;0:20 XX proper', '3:3 XX q;3:3 XX nominalization;4:4 XX q;5:5 XX q;7:7 XX q;4:7 XX udef;3:7 XX udef', '0:18 XX compound', '6:6 XX q;15:15 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX compound;14:14 XX d;12:14 XX udef;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:21 XX udef;16:21 XX udef;22:22 XX q;23:23 XX q;22:24 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;31:31 XX q;32:32 XX q;32:33 XX udef;34:34 XX compound;32:34 XX compound;26:34 XX udef;22:34 XX udef;0:34 XX focus', '0:0 XX q;1:1 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;18:18 XX q;0:18 XX udef', '2:2 XX q;9:9 XX q;10:10 XX q;12:12 XX q;11:12 XX proper;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;23:23 XX q;24:24 XX times;22:24 XX udef;26:26 XX q;22:26 XX compound;27:27 XX q;29:29 XX q;28:29 XX proper', '5:5 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q', '3:3 XX q;3:4 XX udef;6:6 XX q;6:7 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;20:20 XX compound;21:21 XX q;21:21 XX compound;22:22 XX q;22:22 XX compound;20:23 XX udef;33:33 XX superl', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:6 XX compound;7:7 XX d;10:10 XX d;0:10 XX focus', '0:0 XX q;0:0 XX compound;1:1 XX q;1:1 XX compound;5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;12:12 XX q;13:13 XX times;14:14 XX q;14:14 XX nominalization;15:15 XX q;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;33:33 XX q;33:33 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX q;36:36 XX nominalization;37:37 XX q;37:37 XX nominalization;38:38 XX q;38:38 XX nominalization;0:38 XX udef', '3:3 XX comp;4:4 XX q;4:4 XX compound;5:5 XX q;5:5 XX compound;6:6 XX q;6:6 XX compound;8:8 XX q;8:9 XX udef;16:16 XX udef', '2:2 XX q;4:4 XX q;3:4 XX compound;3:5 XX proper;10:11 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:20 XX compound;19:21 XX compound;22:23 XX proper;25:25 XX q;25:26 XX udef;25:27 XX compound;30:31 XX compound', '4:4 XX q;5:5 XX q;6:6 XX q;9:9 XX q;10:10 XX q;9:10 XX udef;7:10 XX eventuality;11:11 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;15:15 XX udef;12:15 XX udef', '0:0 XX comp', '2:2 XX q;2:2 XX compound;3:3 XX q;3:3 XX compound;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;0:16 XX appos', '1:1 XX q;0:1 XX proper;2:2 XX q;3:4 XX proper;7:9 XX compound;3:9 XX appos;13:13 XX q;14:14 XX q;16:16 XX q;18:18 XX q;17:19 XX udef;30:30 XX q;31:31 XX q;32:32 XX times;0:35 XX udef', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:9 XX compound;0:21 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX compound;20:20 XX q;21:21 XX q;0:27 XX udef', '0:0 XX q;1:1 XX q;4:4 XX q;5:6 XX compound;5:7 XX udef;2:7 XX udef;16:16 XX q;16:16 XX compound;15:16 XX proper;15:17 XX proper;0:22 XX udef', '1:1 XX nominalization;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;27:30 XX udef;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;9:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;14:19 XX udef;21:21 XX neg', '0:0 XX nonsp;1:1 XX nonsp;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;0:16 XX loc', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;22:22 XX compound;21:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;33:33 XX d;35:35 XX d;36:36 XX d;37:37 XX d;39:39 XX d;0:41 XX focus', '15:15 XX compound', '', '1:1 XX q;0:1 XX compound;2:2 XX q;3:3 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;17:17 XX nominalization;16:17 XX udef;12:17 XX udef;7:17 XX udef;4:17 XX eventuality;19:19 XX neg;0:33 XX proper', '9:9 XX q;8:10 XX udef;13:13 XX q;14:15 XX compound;16:16 XX q;14:16 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;25:25 XX q;24:25 XX proper;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;27:27 XX q;27:27 XX compound;26:27 XX udef;0:27 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;34:34 XX q;35:35 XX q;37:37 XX q;38:38 XX q;33:38 XX udef;0:38 XX focus', '7:8 XX udef;7:9 XX compound;10:11 XX proper;13:13 XX q;12:13 XX compound;14:14 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;17:17 XX compound;17:18 XX udef;15:18 XX proper;12:18 XX udef;10:18 XX appos;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q', '3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX d;6:7 XX udef;9:9 XX d;11:11 XX d;12:12 XX d', '', '2:2 XX neg;12:12 XX q;12:13 XX udef;15:15 XX neg;34:34 XX comp;40:40 XX q;40:40 XX compound;41:41 XX q;41:41 XX compound;42:42 XX q;42:42 XX compound', '0:1 XX compound;7:7 XX q;6:7 XX proper;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:28 XX compound', '0:0 XX q;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;8:8 XX nominalization;0:11 XX udef', '8:8 XX q;10:10 XX q;11:11 XX q;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q', '3:4 XX udef;9:10 XX proper;12:12 XX q;15:15 XX q;14:15 XX compound;13:15 XX compound;12:15 XX udef;11:15 XX id;16:16 XX q;17:18 XX compound;20:20 XX q;19:20 XX compound;22:22 XX q;23:23 XX q;17:23 XX proper;11:23 XX udef', '8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;14:14 XX q;12:14 XX compound', '9:9 XX q;11:11 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX udef', '0:0 XX q;1:1 XX q;1:1 XX compound;0:5 XX proper', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q', '10:10 XX q;11:11 XX q', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX times;14:16 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;0:9 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;13:13 XX q;14:14 XX q;15:15 XX times;27:27 XX q;28:28 XX q;29:29 XX times;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;0:37 XX udef', '0:0 XX q;1:1 XX q;0:16 XX udef', '0:0 XX q;1:1 XX q;3:3 XX q;8:8 XX q;9:9 XX q;0:19 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;6:10 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;12:14 XX udef;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;18:20 XX udef;16:20 XX udef;22:22 XX q;24:24 XX q;23:24 XX compound;25:25 XX q;26:26 XX q;27:27 XX q;26:28 XX udef;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef;30:33 XX udef;23:33 XX udef;0:33 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:9 XX proper;0:9 XX focus', '0:0 XX conj;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;3:4 XX udef;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX conj;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;17:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;5:25 XX q;0:25 XX implicit', '0:0 XX q;0:5 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;9:9 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;16:16 XX conj;18:18 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX conj;23:23 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX q;0:29 XX implicit', '', '0:-1 XX EMPTY-LABEL', '', '5:5 XX comp', '4:4 XX q;5:5 XX q', '', '0:23 XX subord', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;3:18 XX udef', '0:0 XX d;1:1 XX q;1:1 XX nominalization;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX q;10:10 XX q;11:11 XX so;11:11 XX not;12:12 XX q;11:12 XX comp;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX udef;21:21 XX q;21:22 XX udef;25:25 XX q;26:26 XX q;26:26 XX compound;25:26 XX udef;24:26 XX udef;27:27 XX compound;24:27 XX udef;21:27 XX udef;19:27 XX udef;0:27 XX focus', '1:1 XX q;1:1 XX nominalization;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;9:9 XX conj;10:10 XX conj;19:19 XX conj;21:21 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX conj;30:30 XX conj;32:32 XX conj;33:33 XX conj;35:35 XX conj;36:36 XX conj;37:37 XX conj;38:38 XX conj;0:38 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX compound', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;0:14 XX implicit', '0:0 XX q;0:8 XX udef', '4:4 XX q', '0:1 XX proper;2:2 XX q;0:2 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX proper;11:12 XX proper;13:13 XX d;11:13 XX compound;6:13 XX udef;15:15 XX d;16:16 XX d;0:16 XX udef', '', '11:10 XX very;11:11 XX XX', '', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX conj;8:8 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:12 XX you;13:13 XX conj;13:13 XX XX;15:15 XX conj;0:16 XX implicit', '0:1 XX proper;6:6 XX q;7:7 XX q;8:8 XX q;16:16 XX neg;23:23 XX d;0:28 XX compound', '0:19 XX proper', '', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX compound', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;0:13 XX implicit', '', '1:1 XX d;3:3 XX d;5:5 XX q;4:5 XX compound;6:6 XX q;0:18 XX focus', '10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q', '3:4 XX proper', '0:0 XX q;6:6 XX q;7:7 XX q;8:8 XX q;15:16 XX proper;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;33:33 XX q;34:34 XX q;35:35 XX q;0:36 XX udef', '15:17 XX compound;21:21 XX q;21:21 XX nominalization;14:21 XX appos;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;31:31 XX q;32:32 XX q;31:33 XX udef;29:33 XX udef;28:33 XX udef;34:34 XX q;34:34 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX compound', '0:15 XX subord', '0:0 XX q;27:27 XX q;27:27 XX compound;26:27 XX compound;0:27 XX udef', '', '11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;14:15 XX proper;22:22 XX q;21:23 XX udef;26:26 XX q;26:26 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;32:32 XX q;31:32 XX udef;25:32 XX udef;16:32 XX udef', '2:2 XX q;3:3 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX proper;8:8 XX q;6:8 XX proper;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX compound', '13:13 XX q;14:14 XX q;19:20 XX compound;23:23 XX q;24:24 XX q', '1:1 XX neg;8:8 XX q;8:9 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;17:17 XX q;19:19 XX q;20:20 XX q;22:22 XX d;21:22 XX udef;23:23 XX d;24:24 XX d;0:24 XX udef', '21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q', '0:0 XX q;0:1 XX udef;0:11 XX compound', '25:25 XX q;26:26 XX q;27:27 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX d;0:18 XX focus', '3:3 XX q;4:5 XX proper;2:5 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;10:17 XX udef;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;20:24 XX udef', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;4:6 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;8:11 XX udef;4:11 XX udef;24:24 XX q;25:25 XX q;24:26 XX udef;27:27 XX q;28:28 XX q;29:29 XX q;28:31 XX udef;24:31 XX udef;32:32 XX nonsp;33:33 XX nonsp;32:33 XX loc', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX times;6:8 XX udef;10:10 XX q;6:10 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX compound;13:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX times;18:20 XX udef;22:22 XX q;18:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX compound;24:25 XX udef;24:25 XX udef', '0:0 XX q;1:1 XX q;7:7 XX q;8:8 XX q;0:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;11:14 XX udef;7:14 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;23:26 XX udef;19:26 XX udef;27:27 XX nonsp;28:28 XX nonsp;27:28 XX loc;17:28 XX udef;0:28 XX focus', '3:3 XX q;4:4 XX q', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;0:12 XX focus', '7:7 XX q', '0:1 XX proper;0:8 XX compound', '2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;23:23 XX number', '1:1 XX q;1:1 XX pron;2:2 XX q;2:2 XX pron;3:3 XX q;3:3 XX pron;4:4 XX q;4:4 XX pron;5:5 XX q;5:5 XX pron;6:6 XX conj;8:8 XX conj;9:9 XX conj;11:11 XX q;12:12 XX q;0:13 XX focus', '23:23 XX q;0:27 XX proper', '3:3 XX neg;12:12 XX neg', '0:0 XX q;0:0 XX compound;0:0 XX unknown;0:0 XX proper', '7:7 XX neg', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX d;13:14 XX compound;9:14 XX udef;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX compound;34:35 XX udef;0:35 XX udef', '5:5 XX q;5:5 XX nominalization;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;15:15 XX udef;13:15 XX udef;6:15 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX compound;13:21 XX udef;0:21 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;14:14 XX q;15:15 XX compound;0:15 XX focus', '7:7 XX comp', '0:0 XX neg;10:10 XX d;12:12 XX d;11:12 XX udef', '11:11 XX q;0:14 XX measure', '3:3 XX superl;17:17 XX q;17:18 XX udef', '0:0 XX nonsp;0:13 XX loc', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:27 XX focus', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;10:10 XX conj;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;15:17 XX compound;11:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;30:30 XX comp;22:30 XX appos;19:30 XX udef;0:30 XX implicit', '13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;0:25 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;7:7 XX conj;6:7 XX udef;8:8 XX conj;9:9 XX conj;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;23:23 XX conj;24:24 XX conj;25:25 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX conj;30:30 XX conj;31:31 XX conj;32:32 XX conj;34:34 XX conj;35:35 XX q;36:36 XX q;37:37 XX q;39:39 XX q;0:39 XX implicit', '12:12 XX q;11:12 XX udef;13:13 XX q;13:14 XX udef;16:16 XX q;16:16 XX udef;13:16 XX udef', '10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;18:18 XX q;19:19 XX q;19:19 XX nominalization;18:19 XX udef;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization', '0:6 XX proper', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;11:11 XX conj;12:12 XX conj;14:14 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;0:21 XX implicit', '9:9 XX q;8:9 XX udef;10:11 XX udef;13:14 XX udef;10:14 XX udef', '6:7 XX proper;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX compound;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX compound;19:19 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX conj;18:18 XX d;19:19 XX d;20:20 XX d;23:23 XX d;26:26 XX d;0:26 XX focus', '0:0 XX conj;1:1 XX conj;2:2 XX conj;4:4 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX d;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX d;12:16 XX udef;17:17 XX q;17:17 XX pron;19:19 XX q;19:19 XX pron;17:19 XX pronoun;0:19 XX implicit', '9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;14:16 XX udef;17:18 XX udef;19:20 XX udef;17:20 XX udef', '1:1 XX conj;11:11 XX entity;13:13 XX nonsp;0:13 XX unknown', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;0:9 XX udef', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;0:11 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX nonsp;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;0:23 XX focus', '3:3 XX q;4:4 XX q;7:7 XX q', '7:8 XX compound;9:10 XX proper;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;19:19 XX compound;18:19 XX proper;20:20 XX q;20:20 XX compound;18:20 XX proper;18:21 XX proper;18:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:12 XX compound;11:13 XX compound;0:24 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX neg;0:7 XX udef', '3:3 XX q;4:4 XX q;4:5 XX proper;16:16 XX q;16:16 XX compound;17:18 XX udef;19:19 XX q;19:20 XX udef;17:20 XX udef;25:25 XX compound;0:25 XX unknown', '0:0 XX neg', '0:7 XX compound', '2:3 XX compound;1:3 XX udef;4:4 XX q;5:5 XX q;6:6 XX udef;4:6 XX udef', '1:1 XX q;1:1 XX compound;0:1 XX proper;0:2 XX proper;3:3 XX conj;3:3 XX unknown;4:4 XX conj;4:4 XX unknown;5:5 XX conj;5:5 XX unknown;6:6 XX conj;6:6 XX unknown;7:7 XX q;9:9 XX conj;9:9 XX unknown;11:11 XX conj;11:11 XX unknown;12:12 XX conj;12:12 XX unknown;13:13 XX conj;13:13 XX unknown;14:14 XX conj;14:14 XX unknown;15:15 XX conj;15:15 XX unknown;16:16 XX conj;16:16 XX unknown;17:17 XX conj;17:17 XX unknown;19:19 XX q;18:19 XX udef;20:20 XX q;0:20 XX compound', '16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX nominalization;20:21 XX udef;24:24 XX q;26:26 XX q;27:27 XX q', '0:0 XX nominalization', '0:0 XX neg;7:7 XX q;8:8 XX nominalization;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX udef', '6:6 XX q;6:7 XX udef;8:8 XX q;10:10 XX q;14:14 XX q', '7:7 XX q;8:8 XX nominalization;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX udef', '3:3 XX q;4:4 XX nominalization;19:19 XX neg', '8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '8:8 XX q;8:8 XX nominalization;9:9 XX q;10:10 XX q;9:11 XX udef', '1:1 XX neg;3:3 XX q;4:4 XX q;6:6 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;14:14 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;18:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;0:21 XX focus', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX q;0:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;16:17 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:27 XX focus', '0:0 XX q;4:4 XX q;4:4 XX nominalization;5:5 XX q;7:7 XX q;7:7 XX nominalization;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX udef;0:11 XX udef', '5:5 XX q;6:6 XX nominalization;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;14:17 XX udef;10:17 XX udef;18:18 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX compound', '8:8 XX q;9:9 XX nominalization;13:13 XX neg;21:21 XX q;21:21 XX compound;0:22 XX compound', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:6 XX udef;7:7 XX d;8:8 XX d;0:8 XX focus', '3:3 XX q;5:5 XX q;7:7 XX q;7:7 XX named;8:8 XX q;8:8 XX named;9:9 XX q;9:9 XX named;7:10 XX proper;11:12 XX compound;13:13 XX q;13:13 XX compound;11:13 XX udef;11:14 XX udef;11:15 XX compound;7:15 XX compound;0:16 XX compound', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:17 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;9:9 XX q;11:11 XX q;12:12 XX q;14:14 XX neg;0:16 XX udef', '5:5 XX neg;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;10:12 XX udef;0:12 XX compound', '0:0 XX d;3:3 XX nonsp;4:4 XX d;5:5 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:14 XX focus', '2:2 XX neg', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;3:4 XX loc;5:6 XX proper;8:8 XX q;9:9 XX q;10:11 XX generic;8:11 XX udef;12:12 XX d;8:12 XX compound;5:12 XX appos;13:13 XX d;15:15 XX q;14:15 XX compound;16:16 XX d;14:16 XX udef;17:17 XX q;17:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;0:21 XX focus', '1:1 XX thing', '2:2 XX q', '0:1 XX compound;7:7 XX q;9:9 XX q;10:10 XX q;0:11 XX compound', '', '0:0 XX q;0:18 XX proper', '0:0 XX q;0:18 XX proper', '1:1 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:8 XX proper;13:13 XX q;12:13 XX compound;14:14 XX q;14:14 XX compound;14:15 XX udef;12:15 XX proper;17:18 XX udef;17:18 XX compound;12:18 XX appos;0:18 XX focus', '0:0 XX q;0:6 XX proper', '4:5 XX udef;9:9 XX q;7:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX compound', '', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX proper', '6:6 XX q;5:7 XX udef', '0:0 XX q;1:1 XX q;0:21 XX proper', '', '7:7 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;9:9 XX compound;11:11 XX compound;9:11 XX udef', '', '0:0 XX neg', '', '4:4 XX q;4:4 XX nominalization;10:10 XX q;0:26 XX subord', '2:2 XX q;3:3 XX q;2:3 XX udef;4:4 XX q;7:8 XX proper;9:9 XX d;7:9 XX compound;5:9 XX appos;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:17 XX focus', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;14:14 XX mod;16:16 XX mod;17:17 XX mod;18:18 XX mod;19:19 XX mod;20:20 XX mod;14:21 XX relative;0:21 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;7:7 XX q;0:8 XX udef', '', '', '10:10 XX neg', '0:0 XX d;1:1 XX d;2:3 XX proper;7:7 XX d;6:7 XX compound;5:7 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;0:15 XX focus', '6:6 XX q', '8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX compound;11:12 XX udef', '0:13 XX compound', '11:12 XX proper;16:16 XX q;17:18 XX proper;19:19 XX q;17:19 XX compound;20:20 XX q;22:22 XX q;24:24 XX q;24:24 XX compound;23:24 XX proper;23:24 XX proper', '6:6 XX nonsp;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q', '4:4 XX q;4:4 XX nominalization', '0:0 XX d;2:2 XX q;3:3 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:18 XX focus', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX conj;0:22 XX implicit', '6:6 XX q;6:6 XX compound;9:9 XX q;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;0:18 XX subord', '2:2 XX q;1:2 XX compound;3:3 XX q;17:17 XX q;19:19 XX q;0:22 XX appos', '0:0 XX q;1:2 XX compound;0:4 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;7:8 XX compound;6:8 XX proper;0:22 XX appos', '2:2 XX q;3:3 XX q;5:5 XX q;4:5 XX compound;4:6 XX proper;11:11 XX q;10:12 XX udef;15:15 XX neg;0:20 XX appos', '0:0 XX d;1:1 XX d;3:3 XX d;0:12 XX focus', '0:0 XX q;0:0 XX unknown;0:0 XX udef', '', '10:10 XX q;10:11 XX measure;10:11 XX udef;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;29:30 XX proper;29:31 XX compound;33:33 XX compound', '', '1:1 XX q;1:1 XX nominalization;2:2 XX q;0:5 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;6:6 XX conj;8:8 XX conj;10:10 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX q;14:14 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX compound;16:20 XX udef;0:20 XX implicit', '', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;4:5 XX compound;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;0:22 XX focus', '5:5 XX q;6:6 XX q;5:9 XX udef', '8:9 XX compound', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;0:18 XX focus', '5:5 XX comp;21:21 XX proper;0:21 XX compound', '0:0 XX d;2:2 XX d;0:2 XX focus', '0:0 XX conj;1:1 XX conj;2:2 XX conj;4:4 XX conj;6:6 XX conj;9:9 XX nonsp;10:10 XX nonsp;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:15 XX card;18:18 XX conj;23:24 XX compound;26:27 XX proper;29:30 XX proper;0:32 XX implicit', '1:1 XX neg', '5:5 XX comp', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX q;5:5 XX q;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;13:14 XX udef;12:14 XX udef;0:14 XX implicit', '0:28 XX compound', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX compound;22:25 XX udef;27:27 XX conj;26:27 XX compound;28:28 XX conj;0:28 XX implicit', '2:2 XX neg', '9:9 XX nonsp;10:10 XX nonsp;11:11 XX nonsp;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:20 XX compound;25:25 XX q;26:26 XX q;27:27 XX q', '3:3 XX superl;5:5 XX superl', '9:9 XX person', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;7:7 XX conj;9:9 XX q;10:10 XX q;11:11 XX conj;10:11 XX udef;8:11 XX udef;0:11 XX implicit', '18:19 XX proper;0:19 XX proper', '0:0 XX conj;1:1 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX conj;17:17 XX conj;18:18 XX conj;20:20 XX conj;21:21 XX q;22:22 XX q;21:23 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;24:27 XX udef;0:28 XX implicit', '18:18 XX q;17:18 XX proper;19:19 XX q;20:20 XX proper;0:20 XX proper', '', '', '1:1 XX q;2:2 XX q;4:4 XX q;6:6 XX q;5:6 XX udef;1:6 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;13:13 XX q;14:14 XX q;13:14 XX udef;15:17 XX eventuality;12:17 XX udef;8:17 XX udef;24:24 XX q;25:25 XX q;26:26 XX times;24:26 XX udef', '', '0:0 XX d;1:1 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;0:17 XX focus', '6:6 XX q;7:7 XX times', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX proper;9:13 XX udef', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX conj;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX implicit', '', '', '6:6 XX q;7:7 XX q;9:9 XX q;6:10 XX udef;12:12 XX q;13:13 XX q;12:14 XX udef;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;16:20 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q', '6:7 XX compound;9:9 XX neg;32:32 XX q', '1:1 XX q;1:1 XX nominalization;0:1 XX udef', '', '4:4 XX q;4:4 XX nominalization', '1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:11 XX focus', '2:2 XX q;3:3 XX q;2:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX times;6:8 XX udef;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization', '0:0 XX verb;6:6 XX q;7:7 XX times;10:10 XX q;10:10 XX nominalization;11:11 XX q;12:12 XX q;12:12 XX nominalization;11:12 XX udef;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;16:16 XX q;18:18 XX q;18:18 XX compound;19:19 XX q;19:19 XX nominalization;17:19 XX udef;15:19 XX udef;23:23 XX nonsp;27:28 XX udef;27:29 XX compound;24:31 XX subord;21:31 XX loc;20:31 XX loc;0:31 XX generic', '6:7 XX proper', '0:0 XX q;0:6 XX udef', '2:3 XX loc;10:11 XX compound;12:12 XX proper', '0:0 XX neg;11:12 XX compound', '1:1 XX d;0:1 XX loc;3:3 XX d;2:3 XX proper;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX conj;12:12 XX conj;13:13 XX conj;14:15 XX compound;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;21:21 XX q;22:22 XX q;0:23 XX focus', '2:2 XX q;3:3 XX q;7:7 XX superl;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;11:14 XX udef;4:14 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '', '0:0 XX d;1:1 XX q;2:2 XX q;1:2 XX udef;3:3 XX q;6:6 XX q;5:6 XX compound;7:7 XX d;4:7 XX udef;9:9 XX q;12:12 XX d;11:12 XX compound;13:13 XX d;14:14 XX d;15:15 XX d;0:16 XX focus', '17:17 XX q;28:28 XX compound', '', '0:9 XX appos', '1:1 XX q;1:2 XX udef;0:2 XX udef;10:10 XX comp;15:15 XX q;0:15 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX conj;12:12 XX q;12:12 XX compound;11:12 XX proper;20:21 XX proper;23:24 XX compound;23:24 XX compound;0:24 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX q;20:20 XX q;21:22 XX compound;20:22 XX udef;24:24 XX q;24:24 XX compound;23:24 XX proper;23:24 XX proper;17:24 XX appos;0:24 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:20 XX compound;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX q;31:31 XX q;0:31 XX focus', '2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;19:19 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX focus', '0:0 XX neg', '19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;34:34 XX q;36:36 XX q;35:36 XX compound;38:38 XX q;39:39 XX proper;29:39 XX udef', '16:16 XX neg;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;31:31 XX q;31:31 XX nominalization', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;25:25 XX q;24:25 XX compound;23:25 XX compound;26:26 XX q;27:27 XX proper;0:27 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;12:12 XX d;15:15 XX q;0:19 XX focus', '', '', '0:0 XX d;2:2 XX d;3:3 XX q;3:3 XX nominalization;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:17 XX proper;0:17 XX focus', '', '', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:17 XX proper;20:20 XX superl;23:23 XX q;23:23 XX nominalization;21:23 XX compound;16:23 XX appos;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;28:28 XX d;29:29 XX d;30:30 XX d;32:32 XX d;33:33 XX q;33:33 XX nominalization;34:34 XX q;34:34 XX nominalization;36:36 XX d;0:36 XX proper', '', '5:5 XX q;6:6 XX q;7:7 XX times;5:7 XX udef;8:9 XX compound;8:10 XX compound;14:14 XX q;15:15 XX q;16:16 XX times;14:16 XX udef;20:20 XX q;20:20 XX compound;20:22 XX udef;0:24 XX proper', '2:2 XX q', '1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;15:15 XX q;16:16 XX conj;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:24 XX loc', '7:7 XX q;8:8 XX q;9:9 XX times;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;14:14 XX q;14:14 XX nominalization;17:17 XX q;17:17 XX nominalization', '15:15 XX q;14:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;20:22 XX udef', '0:0 XX q;1:0 XX some;1:1 XX q;2:2 XX q;3:3 XX q;1:4 XX XX;6:6 XX q;7:7 XX q;8:8 XX q;0:21 XX udef', '10:10 XX q;18:18 XX q', '0:0 XX q;1:1 XX conj;2:2 XX conj;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;16:16 XX conj;19:19 XX conj;21:21 XX neg;0:23 XX udef', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:10 XX proper;14:14 XX q;15:15 XX q;15:15 XX compound;15:16 XX udef;14:16 XX proper;18:19 XX proper;18:19 XX compound;14:19 XX appos;0:19 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;11:11 XX q;12:12 XX q;13:13 XX times;14:14 XX neg', '8:8 XX q;7:8 XX proper;9:9 XX q;10:11 XX proper;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;34:34 XX q', '4:4 XX q;8:8 XX q;8:8 XX compound;14:14 XX q;14:14 XX named;18:18 XX q;21:21 XX q;22:22 XX q;22:22 XX compound;0:23 XX appos', '0:23 XX proper', '17:17 XX q;16:18 XX udef;16:19 XX proper', '0:0 XX nonsp;2:2 XX nonsp;4:4 XX nonsp;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX q;0:14 XX loc', '', '3:3 XX q;5:5 XX q;6:6 XX q;7:8 XX compound;16:16 XX q', '4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX udef', '1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;13:14 XX proper;13:15 XX compound;18:18 XX d;19:19 XX d;0:19 XX proper', '', '0:0 XX d;1:1 XX d;3:3 XX d;0:4 XX focus', '3:3 XX comp;7:7 XX nominalization;19:19 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;7:7 XX q;8:8 XX q;0:11 XX proper', '0:0 XX d;1:1 XX d;3:3 XX d;5:5 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;11:12 XX proper;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;10:19 XX udef;0:19 XX focus', '16:16 XX q;23:23 XX q;23:23 XX compound;0:23 XX appos', '', '3:3 XX q;3:3 XX nominalization;14:14 XX q;14:14 XX named;15:15 XX q;14:15 XX proper;16:16 XX q;17:17 XX q;18:18 XX proper', '8:8 XX q;8:8 XX named', '17:17 XX comp;20:20 XX q', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX named;11:11 XX q;11:11 XX compound;10:11 XX proper;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;0:5 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX times;15:17 XX udef;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:25 XX focus', '3:3 XX q;4:4 XX q;4:4 XX compound;3:4 XX udef;5:6 XX proper;5:7 XX compound;12:12 XX q;12:12 XX named;14:14 XX neg', '0:0 XX q;2:2 XX d;7:7 XX d;3:7 XX loc;9:9 XX d;8:9 XX proper;10:10 XX d;14:14 XX q;13:15 XX udef;16:16 XX d;13:16 XX compound;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:26 XX proper', '0:0 XX conj;1:1 XX conj;2:2 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX q;13:14 XX udef;15:15 XX d;16:16 XX d;17:17 XX d;21:21 XX neg;24:24 XX q;25:25 XX q;24:26 XX udef;30:30 XX subord;32:32 XX q;34:34 XX q;32:34 XX udef;18:34 XX subord;0:34 XX implicit', '2:2 XX q;7:8 XX compound;26:26 XX comp;28:28 XX q;28:28 XX udef', '3:3 XX neg', '16:16 XX neg;18:18 XX q;19:19 XX q;21:21 XX q;20:21 XX proper;23:23 XX q;0:23 XX compound', '1:1 XX d;2:2 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:12 XX proper;17:18 XX compound;17:19 XX compound;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;33:33 XX q;34:34 XX q;36:36 XX q;37:37 XX q;38:38 XX q;39:39 XX q;42:42 XX q;43:43 XX q;42:44 XX udef;40:44 XX generic;45:45 XX q;47:47 XX q;48:48 XX q;38:48 XX udef;0:48 XX focus', '3:3 XX q', '8:8 XX q;17:18 XX compound', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX q;3:3 XX pron;5:5 XX q;5:5 XX pron;7:7 XX q;7:7 XX pron;0:7 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX q;6:6 XX conj;5:6 XX compound;7:7 XX conj;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX q;14:14 XX conj;13:14 XX udef;10:14 XX udef;15:15 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX q;24:24 XX pron;0:25 XX implicit', '3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;24:24 XX q;25:25 XX q;26:26 XX q;26:26 XX named', '', '', '8:8 XX q;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX proper;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q', '1:1 XX q;2:2 XX q;3:3 XX q', '2:2 XX times;10:10 XX q;12:12 XX q;11:12 XX compound;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;20:20 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q', '5:5 XX q;4:5 XX compound;8:9 XX proper;0:17 XX appos', '', '1:1 XX d;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;0:22 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d', '1:1 XX neg;4:4 XX nonsp;5:5 XX nonsp;6:6 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX q;11:11 XX q;12:12 XX times;10:12 XX udef;14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization', '', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization', '0:0 XX q;0:0 XX compound;18:18 XX q;0:18 XX proper', '0:0 XX q;1:1 XX q;0:18 XX proper', '0:0 XX d;3:3 XX d;5:5 XX nonsp;6:6 XX nonsp;7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;12:12 XX d;8:12 XX loc;0:13 XX focus', '12:12 XX q;12:12 XX nominalization;13:13 XX q;14:14 XX q', '0:0 XX q;5:5 XX conj;5:5 XX unknown;6:6 XX conj;6:6 XX unknown;8:8 XX conj;8:8 XX unknown;9:9 XX conj;9:9 XX unknown;10:10 XX conj;10:10 XX unknown;12:12 XX conj;12:12 XX unknown;13:13 XX conj;13:13 XX unknown;14:14 XX conj;14:14 XX unknown;15:15 XX q;16:16 XX conj;16:16 XX unknown;17:17 XX conj;17:17 XX unknown;19:19 XX conj;19:19 XX unknown;20:20 XX conj;20:20 XX unknown;21:21 XX conj;21:21 XX unknown;22:22 XX conj;22:22 XX unknown;25:25 XX conj;25:25 XX unknown;26:26 XX conj;26:26 XX unknown;28:28 XX conj;28:28 XX unknown;29:29 XX conj;29:29 XX unknown;31:31 XX conj;31:31 XX unknown;33:33 XX conj;33:33 XX unknown;0:34 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;5:5 XX q;10:10 XX comp;12:12 XX q;15:15 XX q;15:15 XX named;0:15 XX compound;0:15 XX udef', '9:9 XX q;8:9 XX named;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX proper;19:19 XX q;15:20 XX udef', '', '3:3 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;0:11 XX udef', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;7:8 XX udef;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;17:17 XX q;17:17 XX nominalization;16:18 XX udef;10:18 XX udef;20:20 XX d;21:21 XX d;22:22 XX d;0:22 XX focus', '1:1 XX neg;4:4 XX neg;10:10 XX neg;13:13 XX q;13:14 XX udef', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;16:16 XX q;17:17 XX d;16:17 XX udef;18:18 XX d;0:28 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX person;22:22 XX q;0:23 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX person;22:22 XX q;0:23 XX proper', '0:0 XX d;3:3 XX d;2:3 XX compound;6:6 XX d;7:8 XX proper;13:13 XX d;7:13 XX appos;14:14 XX d;15:15 XX d;26:26 XX neg;28:28 XX compound;16:28 XX appos;0:28 XX focus', '2:2 XX q;5:5 XX q;5:6 XX measure;5:6 XX udef', '', '5:5 XX q;6:6 XX q;7:7 XX q', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX nonsp;13:13 XX d;0:14 XX focus', '', '', '0:0 XX q;0:0 XX compound;2:2 XX q;2:2 XX compound;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;0:16 XX udef', '11:11 XX q;11:12 XX udef', '0:0 XX neg;1:1 XX q;6:6 XX neg', '8:8 XX q;8:8 XX named;9:9 XX q;8:9 XX proper;10:10 XX q;11:11 XX proper', '4:4 XX q;6:6 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:13 XX focus', '12:12 XX discourse', '0:0 XX d;1:2 XX compound;3:3 XX d;1:3 XX udef;5:5 XX d;4:5 XX proper;7:7 XX d;8:8 XX d;0:8 XX focus', '17:17 XX q;18:18 XX q;19:19 XX q;28:28 XX compound;20:28 XX generic', '0:0 XX q;1:1 XX q;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX d;11:11 XX d;19:20 XX proper;0:23 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX conj;5:5 XX conj;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX q;0:14 XX udef', '', '3:3 XX q', '14:14 XX superl', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;0:35 XX focus', '0:0 XX q;0:0 XX compound;8:8 XX q;9:9 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;0:38 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;0:13 XX focus', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;10:10 XX q;7:11 XX udef;12:12 XX q;13:13 XX q;14:15 XX compound;13:15 XX udef;18:18 XX neg;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization', '1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;14:14 XX neg;0:20 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;11:11 XX q;0:26 XX proper', '14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX q;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:19 XX compound;0:19 XX focus', '1:1 XX d;3:3 XX q;2:3 XX proper;4:4 XX q;6:6 XX d;2:6 XX udef;7:7 XX d;8:8 XX d;0:8 XX focus', '2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;5:5 XX q;5:5 XX compound;6:6 XX q;5:6 XX udef;7:7 XX q;8:9 XX proper;4:9 XX proper;19:19 XX q;19:19 XX compound;18:19 XX udef;20:21 XX compound;0:24 XX appos', '14:14 XX neg', '', '1:1 XX q;0:1 XX proper;2:2 XX q;4:4 XX d;3:4 XX udef;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX d;12:12 XX q;12:12 XX compound;11:12 XX proper;16:17 XX compound;0:18 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;0:6 XX focus', '', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;0:29 XX focus', '', '', '0:0 XX q;0:0 XX pron;1:1 XX q;1:1 XX pron;2:2 XX q;2:2 XX pron;3:3 XX q;3:3 XX pron;4:4 XX q;4:4 XX pron;0:6 XX pronoun', '8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d', '0:0 XX neg;3:3 XX q;4:4 XX udef', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;0:12 XX udef', '16:16 XX q;17:17 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;0:5 XX udef', '3:3 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX compound', '0:0 XX q;0:7 XX udef', '2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;19:20 XX udef;17:20 XX udef;0:20 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;11:11 XX nominalization;12:12 XX q;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;17:20 XX udef;13:20 XX eventuality;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;0:26 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;11:11 XX nominalization;12:12 XX q;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;17:20 XX udef;13:20 XX eventuality;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;0:26 XX proper', '2:2 XX q;2:2 XX compound;1:2 XX proper;1:3 XX proper;1:4 XX compound;13:14 XX proper;22:22 XX q;22:22 XX compound;21:22 XX proper', '15:15 XX q', '0:0 XX q;0:0 XX compound;5:5 XX q;18:18 XX q;0:26 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;13:13 XX d;15:15 XX d;16:16 XX d;0:16 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;19:19 XX udef;17:19 XX udef;13:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX q;1:2 XX proper;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX p;10:10 XX p;11:11 XX q;12:12 XX q;12:12 XX compound;11:12 XX udef;11:12 XX udef;9:12 XX with;0:12 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;0:7 XX udef', '', '1:1 XX d;4:4 XX d;6:6 XX d;7:7 XX conj;9:9 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;14:14 XX q;14:14 XX compound;15:15 XX conj;16:16 XX conj;0:17 XX focus', '', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;13:14 XX compound;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;0:21 XX proper', '0:0 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;18:18 XX q;19:19 XX q;19:19 XX compound;0:19 XX proper', '0:1 XX udef;5:5 XX q;7:7 XX q;8:8 XX q;0:8 XX compound', '3:3 XX q;2:3 XX compound;4:4 XX q;4:4 XX compound;4:5 XX udef;13:13 XX q;13:13 XX compound;12:14 XX udef', '10:10 XX q;13:13 XX q', '4:4 XX q;5:5 XX q;7:8 XX loc', '10:10 XX q;11:11 XX q;14:14 XX nonsp;13:14 XX loc;0:14 XX compound', '0:1 XX udef;6:6 XX q;9:9 XX q;0:15 XX compound', '0:0 XX q;0:0 XX compound;6:6 XX q;7:7 XX q;0:10 XX udef', '6:6 XX q;7:7 XX q;10:10 XX nonsp;9:10 XX loc;0:10 XX compound', '0:0 XX q;5:5 XX q;5:5 XX compound;28:28 XX q;30:30 XX q;33:33 XX q;32:33 XX q;0:33 XX proper', '9:10 XX proper;11:12 XX udef;11:13 XX compound;15:15 XX q;16:16 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;5:5 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;11:13 XX udef;15:15 XX d;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;18:22 XX udef;0:22 XX focus', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;9:9 XX compound;6:9 XX udef;10:10 XX q;6:10 XX udef;11:11 XX q;12:12 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;12:12 XX q;13:13 XX q;14:14 XX q;0:19 XX udef', '6:6 XX q;6:6 XX compound;5:6 XX number;7:8 XX compound;7:9 XX compound', '0:1 XX proper;2:2 XX q;3:3 XX q;6:6 XX q;6:6 XX compound;5:6 XX proper;7:7 XX q;7:7 XX compound;5:7 XX proper;5:8 XX proper;13:13 XX q;14:14 XX q;15:15 XX q;18:18 XX q;17:19 XX udef;22:22 XX q;17:22 XX compound;23:23 XX q;28:28 XX q;29:29 XX q;31:31 XX q;31:31 XX compound;0:31 XX compound', '2:2 XX q;3:4 XX compound;7:7 XX q;8:8 XX q;9:9 XX times;14:14 XX q;15:15 XX q', '7:7 XX q;8:8 XX q;10:10 XX q;0:10 XX appos', '0:0 XX q;0:0 XX compound;5:5 XX q;7:8 XX compound;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;0:19 XX proper', '0:0 XX d;2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;3:4 XX measure;3:4 XX udef;5:5 XX d;6:6 XX d;10:10 XX q;10:10 XX compound;9:10 XX proper;9:11 XX proper;9:12 XX appos;0:17 XX focus', '3:3 XX q;5:5 XX compound', '15:15 XX q', '1:1 XX q;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX times;19:19 XX q;20:20 XX q;0:23 XX compound', '4:4 XX q;5:5 XX q;6:6 XX times', '0:0 XX q;6:6 XX q;10:10 XX q;11:11 XX q;12:12 XX times;17:17 XX q;17:17 XX compound;0:18 XX proper', '0:0 XX q;3:3 XX q;3:3 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;29:29 XX nonsp;30:30 XX nonsp;0:30 XX proper', '0:0 XX superl;5:5 XX poss;7:7 XX q;8:8 XX times;10:10 XX q;11:11 XX q', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '10:10 XX q;10:10 XX udef', '0:1 XX number;4:4 XX q;3:4 XX compound;5:5 XX q;7:7 XX q;5:8 XX proper;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:23 XX compound', '1:1 XX q;0:1 XX compound;11:11 XX q;12:12 XX q;11:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;15:18 XX udef;20:20 XX q;21:21 XX q;20:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;24:26 XX udef;0:26 XX proper', '0:0 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:13 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX nonsp;5:5 XX nonsp;6:6 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;11:11 XX nonsp;12:12 XX d;13:13 XX d;14:15 XX proper;17:17 XX q;16:17 XX compound;18:18 XX q;20:22 XX udef;19:22 XX udef;23:23 XX d;19:23 XX proper;16:23 XX udef;24:24 XX d;25:25 XX d;26:26 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX q;32:32 XX q;0:33 XX focus', '2:3 XX number;4:5 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;24:24 XX udef;21:24 XX udef;14:24 XX udef', '0:1 XX udef;8:8 XX q;9:9 XX q;12:12 XX q;13:13 XX q;0:14 XX compound', '20:20 XX superl', '1:1 XX q;1:1 XX compound;0:1 XX proper;0:2 XX proper;4:4 XX q;6:7 XX compound;5:7 XX compound;4:7 XX udef;3:7 XX udef;0:15 XX appos', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX udef', '4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX q;12:13 XX udef;11:13 XX udef;14:14 XX q;11:14 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;19:20 XX compound;19:20 XX compound;18:20 XX compound;10:20 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX q;0:14 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;0:4 XX unknown;0:4 XX udef', '2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;3:4 XX compound;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;15:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;18:19 XX udef;15:19 XX udef', '10:11 XX proper;15:16 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:4 XX compound;5:5 XX neg;0:6 XX udef', '0:0 XX q;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX udef', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;14:14 XX q;13:14 XX compound;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX times;21:23 XX udef;24:24 XX compound;21:24 XX compound;16:24 XX proper;0:24 XX compound', '13:13 XX q;14:14 XX q;13:14 XX udef;16:16 XX q;16:17 XX udef;13:17 XX udef;13:18 XX compound', '13:13 XX q;14:14 XX q;16:18 XX compound;19:19 XX q;19:20 XX udef;21:21 XX q;19:21 XX compound;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;35:35 XX q;34:35 XX udef;28:35 XX udef;19:35 XX udef', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;9:9 XX q;12:12 XX q;13:13 XX d;2:13 XX udef;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;0:23 XX udef', '0:0 XX q;1:1 XX q;11:12 XX compound;13:13 XX q;11:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;0:25 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:14 XX proper;15:15 XX q;16:16 XX q;18:18 XX q;18:19 XX udef;18:20 XX compound;22:22 XX q;22:22 XX compound;21:22 XX proper;21:23 XX proper;24:24 XX q;21:24 XX compound;25:25 XX q;21:25 XX proper;17:25 XX appos;15:25 XX udef;0:25 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;7:7 XX d;9:9 XX d;0:9 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;5:5 XX q;6:6 XX q;8:8 XX q;8:8 XX compound;7:8 XX proper;9:9 XX d;7:9 XX proper;5:9 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;15:17 XX udef;18:18 XX q;20:20 XX d;19:20 XX compound;18:20 XX udef;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:27 XX focus', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;8:8 XX compound;7:8 XX proper;7:9 XX proper;12:12 XX compound;11:12 XX compound;0:12 XX focus', '', '9:9 XX q;19:19 XX q;23:23 XX q', '2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX d;10:10 XX d;14:14 XX d;16:16 XX q;17:17 XX q;19:19 XX q;18:19 XX compound;21:21 XX q;22:23 XX proper;26:26 XX q;26:27 XX udef;25:27 XX id;31:31 XX q;31:32 XX udef;42:42 XX q;43:43 XX compound;42:43 XX udef;22:43 XX appos;16:43 XX udef;0:43 XX focus', '2:2 XX q;4:4 XX q;6:6 XX q', '0:12 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX d;18:18 XX d;19:19 XX d;0:20 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;0:18 XX udef', '0:0 XX q;3:3 XX d;4:4 XX d;6:6 XX q;6:6 XX compound;5:6 XX proper;7:8 XX compound;5:8 XX udef;10:10 XX q;9:10 XX compound;11:11 XX q;13:13 XX q;13:13 XX compound;12:13 XX proper;0:13 XX udef', '4:4 XX q;4:4 XX compound;3:4 XX proper;5:5 XX q;3:5 XX proper;6:6 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX proper;23:23 XX q;23:23 XX named;24:24 XX q;24:24 XX named;23:25 XX proper;27:28 XX compound;23:28 XX udef;21:28 XX udef;19:28 XX udef;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;32:32 XX udef;29:32 XX udef', '2:2 XX q;3:3 XX q', '4:4 XX nominalization', '0:1 XX udef;0:4 XX compound', '1:1 XX q;0:1 XX proper;2:3 XX proper;6:6 XX d;5:6 XX proper;2:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;16:16 XX d;17:17 XX d;0:17 XX udef', '16:16 XX q;17:17 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX q;5:6 XX compound;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;9:10 XX proper;9:11 XX proper;12:12 XX q;9:12 XX compound;13:13 XX q;14:14 XX proper;9:14 XX proper;0:14 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;0:5 XX focus', '0:0 XX neg', '6:6 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:13 XX proper;15:15 XX q;14:15 XX compound;16:16 XX q;17:17 XX proper;14:17 XX udef;0:17 XX focus', '11:11 XX q;11:12 XX udef;10:12 XX udef;10:13 XX compound;14:14 XX p;15:15 XX p;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;16:22 XX udef;14:22 XX with;24:24 XX neg;29:29 XX q;30:30 XX q', '2:2 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;5:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;15:15 XX compound;14:15 XX udef;14:15 XX udef;11:15 XX udef;5:15 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q', '6:6 XX q;7:7 XX nominalization', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:17 XX focus', '', '0:0 XX q;1:1 XX q;3:3 XX d;2:3 XX compound;4:4 XX d;5:5 XX d;6:6 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;25:25 XX q;0:25 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;18:19 XX compound;17:19 XX udef;16:19 XX udef;20:20 XX d;16:20 XX compound;22:22 XX q;21:22 XX udef;23:23 XX q;0:24 XX focus', '0:0 XX q;6:6 XX q;0:14 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX mod;4:4 XX mod;5:5 XX mod;6:6 XX mod;7:7 XX mod;9:9 XX d;10:11 XX proper;13:14 XX proper;15:16 XX compound;13:16 XX compound;19:19 XX q;19:19 XX compound;18:19 XX udef;20:20 XX q;20:20 XX compound;21:21 XX q;21:21 XX compound;21:21 XX udef;18:21 XX proper;0:21 XX focus', '0:0 XX neg;13:13 XX q;13:13 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:20 XX udef', '0:0 XX nonsp;1:1 XX d;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;25:25 XX d;0:25 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;0:4 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:18 XX udef', '', '8:8 XX person', '', '1:1 XX q;1:1 XX compound;0:1 XX proper;2:2 XX d;4:4 XX q;3:4 XX proper;5:5 XX q;6:7 XX compound;5:7 XX udef;5:8 XX udef;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX d;18:18 XX d;0:20 XX proper', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:11 XX udef', '', '6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;13:13 XX udef;10:13 XX generic', '13:13 XX q;12:13 XX compound;14:14 XX q;14:14 XX compound;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX number;21:21 XX compound;19:21 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;4:5 XX udef;7:8 XX compound;14:14 XX q;15:15 XX compound;0:15 XX udef', '0:0 XX d;2:2 XX q;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;7:7 XX d;6:7 XX udef;8:8 XX d;0:10 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;8:8 XX comp;0:8 XX udef', '1:1 XX comp', '1:1 XX d;4:4 XX d;3:4 XX compound;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;0:23 XX focus', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX d;22:22 XX d;24:24 XX q;24:24 XX compound;23:24 XX proper;27:27 XX q;30:30 XX q;29:30 XX compound;28:30 XX compound;27:30 XX udef;26:30 XX id;31:31 XX q;33:33 XX q;32:33 XX compound;32:34 XX proper;36:36 XX compound;32:36 XX appos;0:36 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;6:6 XX nominalization;6:7 XX udef;11:11 XX comp;0:16 XX udef', '17:18 XX udef;28:28 XX q;30:30 XX q;31:31 XX q;33:33 XX q;34:34 XX q;36:36 XX q;35:37 XX udef;38:38 XX q;35:38 XX compound;39:39 XX q;31:39 XX udef', '0:0 XX q;2:2 XX d;3:3 XX mod;6:6 XX mod;10:10 XX mod;11:11 XX mod;12:12 XX mod;13:13 XX mod;15:15 XX d;18:18 XX q;19:19 XX q;20:20 XX q;20:21 XX proper;23:24 XX compound;23:25 XX compound;28:28 XX udef;20:28 XX appos;0:28 XX udef', '', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;22:22 XX comp', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;22:22 XX comp', '7:7 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;13:14 XX proper;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;18:19 XX compound;20:20 XX q', '0:1 XX proper;9:9 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;23:23 XX q;23:23 XX compound;22:23 XX proper;22:24 XX proper;22:24 XX compound;0:24 XX compound', '6:6 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper;22:22 XX q;21:22 XX compound;19:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX udef;19:25 XX udef', '8:8 XX q;8:8 XX compound;7:8 XX proper;7:9 XX proper', '3:3 XX q;3:3 XX compound;2:3 XX proper;0:20 XX appos', '', '0:0 XX d;1:1 XX d;2:2 XX q;2:2 XX nominalization;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;9:9 XX compound;8:9 XX proper;8:10 XX proper;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;0:23 XX focus', '0:0 XX q;1:1 XX q;12:12 XX q;12:12 XX named;13:13 XX q;13:13 XX named;14:14 XX q;14:14 XX named;21:21 XX q;22:22 XX q;23:23 XX q;0:24 XX udef', '1:1 XX q;2:2 XX q;2:2 XX compound;1:2 XX udef;1:3 XX udef;1:4 XX compound;16:16 XX q;17:17 XX q;0:18 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX d;6:7 XX udef;3:7 XX udef;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;28:28 XX q;28:28 XX compound;28:29 XX udef;0:29 XX udef', '3:5 XX udef;15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper;19:19 XX q;19:19 XX nominalization;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;30:30 XX q;24:31 XX udef;20:31 XX udef', '2:2 XX q;12:12 XX q', '6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX udef;10:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;19:19 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;15:27 XX udef;0:27 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;22:22 XX d;0:22 XX focus', '3:3 XX q;3:3 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization', '6:6 XX q;8:8 XX q;9:9 XX q;12:12 XX q;11:12 XX compound;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX q;22:22 XX q;23:23 XX q;23:23 XX nominalization;25:25 XX q;25:25 XX nominalization;24:25 XX udef;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX compound;27:28 XX udef;23:28 XX udef;15:28 XX udef', '17:17 XX q;17:17 XX compound;19:19 XX q;17:19 XX udef;21:21 XX q;22:22 XX q;22:22 XX compound;21:22 XX udef;21:23 XX udef;21:24 XX udef;20:24 XX udef', '', '5:5 XX q;6:6 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX times;11:13 XX udef;14:15 XX compound;25:25 XX q;26:26 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;9:9 XX d;10:10 XX d;0:10 XX focus', '9:9 XX compound;0:9 XX compound', '0:0 XX q;0:0 XX compound;1:1 XX q;1:1 XX compound;11:11 XX q;11:11 XX named;12:12 XX q;12:12 XX named;13:13 XX q;13:13 XX named;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX proper', '1:1 XX q;3:3 XX q;3:3 XX compound;2:4 XX udef;2:5 XX compound;1:5 XX udef;0:5 XX udef;18:18 XX q;19:19 XX q;0:20 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX d;6:7 XX udef;3:7 XX udef;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX conj;17:17 XX q;18:18 XX q;20:20 XX conj;19:20 XX udef;17:20 XX udef;21:21 XX conj;22:22 XX conj;25:25 XX q;24:25 XX compound;26:26 XX q;27:27 XX q;27:27 XX udef;23:27 XX udef;0:27 XX udef', '8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX compound;22:22 XX times;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;5:5 XX compound;7:8 XX compound;5:8 XX udef;12:12 XX mod;14:14 XX mod;16:16 XX mod;17:17 XX mod;19:19 XX mod;20:20 XX mod;21:21 XX mod;22:22 XX mod;23:23 XX mod;24:24 XX mod;25:25 XX mod;27:27 XX mod;28:28 XX mod;29:29 XX mod;30:30 XX mod;32:32 XX mod;33:33 XX mod;35:35 XX mod;36:36 XX mod;37:37 XX mod;38:39 XX udef;0:39 XX proper', '0:0 XX q;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:16 XX udef;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;23:23 XX q;0:25 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;0:7 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX udef;6:10 XX udef', '0:0 XX q;0:0 XX compound;4:4 XX q;5:6 XX compound;17:17 XX q;18:18 XX entity;20:20 XX p;21:21 XX p;22:22 XX p;24:24 XX p;25:25 XX p;26:26 XX p;0:28 XX proper', '8:8 XX q;7:9 XX udef;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;20:20 XX compound;22:23 XX compound;25:25 XX q;25:25 XX udef;20:25 XX udef;16:25 XX udef', '0:0 XX q;1:1 XX q;1:1 XX compound;7:7 XX q;11:11 XX q;17:17 XX q;16:17 XX compound;23:23 XX q;0:24 XX proper', '0:0 XX neg', '0:0 XX q;1:1 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:8 XX ;4:8 XX udef;0:16 XX udef', '', '0:0 XX q;0:0 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;32:32 XX q;32:32 XX compound;31:32 XX proper;33:33 XX q;34:34 XX q;0:35 XX udef', '0:0 XX q;0:0 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;32:32 XX q;32:32 XX compound;31:32 XX proper;33:33 XX q;34:34 XX q;0:35 XX udef', '0:1 XX compound;10:10 XX q;10:10 XX compound;9:10 XX proper;0:14 XX compound', '9:9 XX q;9:9 XX compound;8:9 XX proper;10:10 XX q;10:10 XX compound;8:10 XX proper;8:11 XX proper;8:11 XX compound', '0:0 XX nonsp;3:3 XX d;4:4 XX q;8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;14:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;21:22 XX udef;0:22 XX loc', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX nonsp;19:19 XX nonsp;20:20 XX d;21:21 XX d;24:24 XX q;26:26 XX d;0:26 XX compound', '11:11 XX nonsp;12:12 XX nonsp', '2:2 XX q;3:3 XX q;4:4 XX q;11:11 XX q;12:12 XX q;13:13 XX nonsp;14:14 XX nonsp;15:15 XX q;13:15 XX loc;11:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;17:20 XX udef;11:20 XX udef', '0:1 XX udef;0:22 XX compound', '0:0 XX q;0:0 XX compound;0:10 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;16:16 XX q;0:17 XX proper', '3:3 XX q;2:3 XX udef;4:4 XX q;6:6 XX q;9:10 XX proper;11:11 XX q;9:11 XX compound;15:15 XX compound', '0:0 XX q;1:1 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;12:12 XX q;11:13 XX udef;14:14 XX q;14:14 XX compound;11:14 XX proper;15:15 XX q;15:15 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX udef', '0:0 XX neg', '2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;7:7 XX compound;6:7 XX udef;5:7 XX udef;8:8 XX q;8:8 XX compound;5:8 XX udef;5:8 XX udef', '0:0 XX q;0:0 XX compound;6:6 XX q;0:10 XX proper', '1:1 XX q;2:2 XX q;4:4 XX q;9:9 XX q;11:11 XX q;0:11 XX udef', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;12:12 XX q;10:12 XX compound;10:12 XX compound;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q', '2:2 XX q;4:4 XX q;3:4 XX compound;6:7 XX compound;8:8 XX q;8:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;10:12 XX udef;8:12 XX udef;5:12 XX udef;2:12 XX udef;13:13 XX q;14:20 XX generic', '', '0:4 XX compound', '3:3 XX q;4:4 XX q;3:5 XX number;6:6 XX q;8:8 XX q;9:9 XX q;9:10 XX proper;13:13 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;20:21 XX udef;22:22 XX q;23:23 XX udef;0:23 XX appos', '', '0:0 XX d;2:2 XX d;4:4 XX d;5:5 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;0:14 XX focus', '1:1 XX q;0:1 XX compound;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;9:10 XX compound;19:19 XX q;20:20 XX q;21:21 XX times;19:21 XX udef;25:25 XX q;25:25 XX nominalization;27:27 XX q;27:27 XX nominalization;29:30 XX compound;36:36 XX q;38:38 XX q;37:38 XX compound;39:39 XX q;41:41 XX compound;40:41 XX udef;36:41 XX udef;28:41 XX appos;0:41 XX proper', '5:5 XX q;3:5 XX compound;6:6 XX q;8:8 XX q;9:9 XX q;7:9 XX udef', '0:0 XX d;1:1 XX d;3:3 XX q;3:3 XX compound;2:3 XX proper;2:4 XX proper;5:5 XX d;6:6 XX d;7:8 XX proper;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:26 XX compound;27:27 XX q;27:27 XX nominalization;25:27 XX compound;33:33 XX q;35:35 XX q;36:36 XX q;33:38 XX udef;40:40 XX generic;28:40 XX subord;16:40 XX udef;0:40 XX focus', '19:20 XX proper;17:20 XX udef;24:24 XX q;26:26 XX q;27:27 XX q', '0:0 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;19:19 XX q;19:19 XX compound;18:19 XX proper;18:20 XX proper;21:21 XX q;18:21 XX compound;23:23 XX q;26:26 XX d;28:28 XX d;0:28 XX focus', '', '3:4 XX proper;21:21 XX q;21:21 XX compound;20:21 XX proper', '4:4 XX q;4:5 XX udef;21:21 XX q', '0:6 XX compound', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;9:9 XX q;9:9 XX nominalization;11:11 XX q;10:12 XX udef;13:13 XX conj;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX conj;16:17 XX udef;14:17 XX udef;18:18 XX conj;19:19 XX conj;20:20 XX conj;22:22 XX conj;23:23 XX q;23:23 XX udef;13:23 XX implicit', '0:0 XX q;2:2 XX q;2:2 XX compound;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;0:32 XX udef', '0:0 XX q;1:1 XX q;0:10 XX udef', '5:6 XX compound;25:25 XX q', '0:0 XX q;0:0 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;14:13 XX Business;14:14 XX q;14:14 XX XX;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;30:30 XX q;0:30 XX proper', '1:1 XX q;10:10 XX q', '5:5 XX q;6:6 XX q;7:7 XX q;17:17 XX q;0:17 XX appos', '0:19 XX appos', '19:19 XX q;20:20 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;23:23 XX q;21:23 XX compound;24:24 XX q;25:25 XX q;27:27 XX q;26:27 XX compound;29:29 XX q;30:30 XX q;31:31 XX d;30:31 XX udef;33:33 XX d;0:33 XX focus', '5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;11:11 XX compound;10:11 XX udef;13:13 XX q;13:13 XX nominalization;12:13 XX compound;10:13 XX udef;8:13 XX udef;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX udef', '3:3 XX q;4:4 XX q;6:6 XX q;10:10 XX q;11:11 XX q;10:12 XX udef;9:12 XX udef;9:13 XX compound', '5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;14:14 XX d;15:15 XX q;17:17 XX d;18:18 XX d;20:20 XX d;0:20 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;11:12 XX compound;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX d;17:18 XX compound;13:18 XX udef;9:18 XX appos;19:19 XX d;20:21 XX proper;22:22 XX d;0:22 XX focus', '8:9 XX compound;0:13 XX compound', '1:1 XX q;2:2 XX q;3:3 XX q;6:6 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;12:12 XX q;13:13 XX q;31:31 XX q;31:31 XX compound;32:32 XX q;32:32 XX compound;0:35 XX compound', '1:1 XX q;7:7 XX q;8:8 XX q;11:11 XX q;12:12 XX q;0:13 XX compound', '5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;4:6 XX proper;7:8 XX compound;10:10 XX q', '5:5 XX q', '4:4 XX q;5:5 XX q;6:6 XX q;7:8 XX udef;7:8 XX compound', '1:1 XX q;1:1 XX compound;0:1 XX proper;2:2 XX q;2:2 XX compound;0:3 XX udef;4:5 XX compound;4:6 XX compound;7:7 XX neg;0:8 XX compound', '2:2 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:12 XX proper;13:13 XX q;11:13 XX compound;14:14 XX q;15:15 XX q;0:15 XX focus', '9:9 XX nonsp;13:13 XX q;15:15 XX neg;0:15 XX compound', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;11:12 XX udef', '0:0 XX q;10:10 XX nonsp;13:13 XX q;0:15 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;12:15 XX udef;16:16 XX d;17:17 XX d;18:18 XX conj;19:19 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;0:33 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX d;12:12 XX d;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX proper;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;21:21 XX q;23:23 XX q;22:23 XX compound;22:23 XX proper;19:23 XX udef;16:23 XX udef;0:23 XX focus', '3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;5:6 XX udef;7:7 XX neg;10:10 XX q', '12:12 XX q;11:12 XX compound;11:13 XX proper;15:15 XX q;14:15 XX compound;14:16 XX proper;21:21 XX q;21:21 XX compound;20:21 XX compound;19:23 XX proper;14:23 XX udef;11:23 XX udef;10:23 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;12:12 XX d;14:14 XX d;15:15 XX conj;0:20 XX implicit', '1:1 XX q;0:1 XX compound;2:3 XX compound;15:15 XX q;16:16 XX q;0:19 XX udef', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;14:15 XX udef;0:15 XX focus', '4:4 XX d;5:5 XX d;7:7 XX d;0:8 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;9:9 XX compound;8:9 XX proper;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX proper;13:15 XX udef;0:15 XX focus', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX q;13:13 XX q;0:14 XX implicit', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:9 XX udef;10:10 XX q;8:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;18:18 XX q;19:20 XX compound;0:20 XX focus', '0:0 XX q;0:18 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;16:16 XX p;17:17 XX p;18:18 XX p;19:19 XX p;21:21 XX p;22:22 XX p;23:23 XX nonsp;24:24 XX nonsp;25:25 XX nonsp;26:26 XX nonsp;0:26 XX udef', '3:4 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;14:14 XX q', '2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX compound', '0:0 XX superl;9:9 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX proper', '0:2 XX compound;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX q;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;0:13 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;17:17 XX nonsp;0:23 XX proper', '6:6 XX q;7:7 XX q;8:8 XX loc', '', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;0:19 XX focus', '0:0 XX q;13:14 XX compound;0:23 XX udef', '3:3 XX q', '0:18 XX appos', '', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;0:11 XX focus', '0:0 XX q;2:2 XX q;10:10 XX q;11:11 XX q;0:11 XX proper', '3:3 XX entity;4:4 XX entity;5:5 XX entity;6:6 XX entity;7:8 XX compound;10:10 XX q;10:10 XX nominalization', '0:0 XX q;3:3 XX q;4:4 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;19:19 XX q;20:20 XX q;20:20 XX udef;0:20 XX udef', '9:9 XX q;8:9 XX compound;11:11 XX q;12:12 XX q;13:14 XX proper;0:24 XX appos', '0:0 XX q;2:2 XX q;3:3 XX q;11:11 XX q;14:14 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;26:27 XX udef;28:28 XX q;29:29 XX q;0:29 XX udef', '10:10 XX q;11:11 XX q;10:12 XX udef', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;7:11 XX udef;18:18 XX q;17:18 XX compound;20:20 XX q;21:21 XX nonsp;21:21 XX loc', '', '0:0 XX d;2:3 XX compound;5:5 XX q;4:5 XX proper;6:6 XX q;8:8 XX d;7:8 XX proper;4:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX focus', '6:6 XX q;6:7 XX udef;0:17 XX appos', '2:2 XX comp', '', '2:2 XX conj;1:2 XX compound;3:3 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:16 XX compound;17:17 XX conj;18:18 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX conj;0:22 XX proper', '1:1 XX q;2:2 XX q;3:3 XX q;8:8 XX q;0:15 XX compound', '', '11:11 XX q;0:13 XX appos', '0:0 XX q;1:1 XX q;1:2 XX udef;11:11 XX neg;0:17 XX udef', '5:5 XX q;6:6 XX q;8:8 XX q;9:10 XX udef;19:19 XX q;18:19 XX compound;22:22 XX q;23:23 XX q;24:24 XX q;23:25 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:32 XX compound;31:32 XX udef;28:32 XX udef;23:32 XX udef;18:32 XX proper', '0:0 XX q;1:1 XX q;10:10 XX q;11:11 XX q;12:12 XX times;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX times;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;0:28 XX udef', '9:10 XX udef', '0:0 XX q;0:1 XX udef;4:4 XX q;0:6 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX d;12:17 XX udef;18:18 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:25 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;29:31 XX udef;0:31 XX focus', '0:0 XX d;3:3 XX d;5:5 XX d;0:5 XX focus', '', '', '5:5 XX nonsp;6:6 XX nonsp', '0:0 XX d;1:1 XX d;3:4 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;11:12 XX proper;6:12 XX udef;14:14 XX superl;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX d;22:22 XX d;23:23 XX q;24:24 XX q;0:24 XX focus', '8:8 XX q;0:8 XX compound', '0:0 XX superl;7:7 XX nonsp;8:8 XX nonsp', '0:0 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;0:18 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:10 XX focus', '5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q', '1:3 XX udef;0:3 XX udef;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX d;17:17 XX d;0:18 XX proper', '', '', '', '0:0 XX q;1:1 XX q;1:1 XX compound;0:1 XX udef;3:3 XX neg;0:21 XX udef', '2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;22:22 XX d;23:23 XX d;24:24 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:32 XX udef;0:32 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;13:15 XX measure;20:20 XX compound;19:20 XX compound;0:20 XX focus', '6:6 XX q;7:7 XX q;8:9 XX compound;16:16 XX q;17:17 XX q;18:18 XX times;16:18 XX udef;23:23 XX q;25:25 XX q;24:25 XX compound;26:26 XX q;27:27 XX q;27:28 XX proper;29:29 XX q;30:30 XX q;31:31 XX q;30:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX q;8:8 XX compound;7:8 XX udef;9:9 XX d;11:11 XX d;0:11 XX focus', '0:0 XX d;1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;0:15 XX focus', '', '0:0 XX q;1:1 XX q;2:2 XX times;0:2 XX udef;4:4 XX q;4:4 XX compound;3:4 XX proper;5:5 XX q;5:5 XX compound;3:5 XX proper;6:7 XX compound;3:7 XX proper;8:8 XX d;3:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;21:21 XX q;21:21 XX compound;20:21 XX proper;20:22 XX proper;23:23 XX q;24:24 XX q;26:26 XX q;26:26 XX compound;25:26 XX proper;25:27 XX proper;28:28 XX q;25:28 XX compound;25:28 XX proper;23:28 XX udef;0:28 XX compound', '0:0 XX d;2:2 XX d;3:3 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:12 XX focus', '', '', '0:0 XX q;7:7 XX q;8:8 XX q;9:9 XX q;0:13 XX proper', '0:0 XX q;4:5 XX compound;3:5 XX compound;9:9 XX d;8:9 XX proper;10:10 XX d;11:11 XX d;13:13 XX q;14:14 XX q;15:15 XX q;21:21 XX d;20:21 XX compound;0:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;8:8 XX q;9:9 XX q;0:9 XX focus', '3:4 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;0:9 XX appos', '3:3 XX q;4:4 XX q', '2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;19:20 XX udef;17:20 XX udef;0:20 XX focus', '3:3 XX q;4:4 XX nominalization;13:13 XX q;14:14 XX q;0:15 XX compound', '1:1 XX q;0:2 XX udef;4:4 XX nonsp;8:8 XX q;0:15 XX compound', '0:1 XX compound;4:4 XX comp;10:10 XX q;10:10 XX nominalization;13:13 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX compound', '0:0 XX q;9:10 XX proper;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;0:21 XX udef', '0:5 XX appos', '8:9 XX udef', '', '0:0 XX q;3:3 XX d;4:4 XX d;6:6 XX q;7:7 XX q;9:9 XX q;11:11 XX q;10:12 XX udef;16:16 XX q;17:17 XX q;0:17 XX proper', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:11 XX focus', '3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;8:8 XX q', '', '5:5 XX q;6:6 XX q;14:14 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;6:6 XX d;7:7 XX conj;8:8 XX q;9:9 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX q;0:14 XX focus', '0:0 XX d;1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX q;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;0:23 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:13 XX part', '2:3 XX udef;2:4 XX compound;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX compound;0:13 XX subord', '9:9 XX q', '0:1 XX compound;2:3 XX compound;7:7 XX comp;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:13 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX q;6:6 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:19 XX focus', '1:1 XX q;2:2 XX q;5:5 XX q;17:17 XX q;20:20 XX q;21:21 XX q;22:22 XX udef;20:22 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX proper;16:16 XX q;17:17 XX q;18:18 XX q;18:19 XX proper;21:23 XX measure;24:25 XX compound;18:32 XX appos;16:32 XX udef;0:32 XX focus', '0:0 XX nonsp;1:1 XX d;2:2 XX q;5:6 XX proper;7:8 XX compound;9:9 XX d;7:9 XX compound;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;0:24 XX loc', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX q;20:20 XX q;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;7:8 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;15:16 XX udef;0:16 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;17:18 XX proper;0:20 XX focus', '3:3 XX neg', '0:0 XX d;2:2 XX d;3:3 XX d;15:15 XX q;16:16 XX q;17:17 XX times;0:18 XX focus', '3:3 XX nonsp;12:12 XX q;14:14 XX q', '0:0 XX d;1:2 XX proper;3:3 XX d;1:3 XX compound;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX nonsp;12:12 XX loc', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:5 XX compound;6:6 XX compound', '0:0 XX q;1:1 XX q;1:1 XX compound;1:2 XX udef;6:7 XX udef;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;0:20 XX udef', '13:13 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:8 XX focus', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:6 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;13:13 XX q;14:14 XX q;17:17 XX q;18:18 XX q;19:19 XX q;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;18:18 XX d;19:19 XX d;21:21 XX q;21:21 XX compound;20:21 XX proper;20:22 XX proper;23:23 XX q;24:24 XX q;25:26 XX proper;29:29 XX q;29:29 XX compound;28:29 XX proper;28:30 XX proper;31:32 XX udef;31:32 XX compound;28:32 XX compound;25:32 XX appos;23:32 XX udef;0:32 XX focus', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper;21:22 XX proper;23:23 XX q;21:23 XX compound;19:23 XX proper;24:24 XX q;25:25 XX q;26:26 XX q;27:28 XX proper;29:29 XX q;27:29 XX compound;30:30 XX q;0:30 XX udef', '9:9 XX q;10:10 XX q;9:10 XX udef;12:12 XX q;12:13 XX udef;9:13 XX udef;17:17 XX q;16:17 XX compound;18:18 XX q', '', '6:6 XX q;7:7 XX q;8:8 XX q;11:11 XX nonsp;21:21 XX q;22:22 XX q', '1:2 XX proper;1:3 XX appos;4:4 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:13 XX focus', '3:3 XX d;6:7 XX compound;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:19 XX subord', '3:4 XX compound;17:17 XX nonsp;0:23 XX appos', '', '0:0 XX q;13:13 XX nominalization;0:22 XX proper', '0:8 XX appos', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX q;22:22 XX compound;21:22 XX proper;25:25 XX q;24:25 XX compound;26:26 XX q;30:31 XX proper;27:31 XX appos;0:31 XX focus', '0:0 XX q;0:0 XX compound;2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;19:19 XX q;0:19 XX proper', '7:7 XX q', '2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;4:6 XX proper;13:13 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX appos', '3:4 XX compound;14:16 XX compound', '', '1:1 XX q;2:2 XX q;3:4 XX proper;5:5 XX q;3:5 XX compound;6:6 XX q;6:6 XX compound;6:7 XX udef;3:7 XX proper;9:10 XX proper;11:12 XX compound;9:12 XX compound;14:14 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;26:26 XX compound;0:26 XX appos', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;7:7 XX q;8:8 XX q;10:10 XX d;9:10 XX udef;6:10 XX udef;0:11 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;9:9 XX d;12:12 XX nonsp;13:13 XX nonsp;14:14 XX nonsp;15:15 XX nonsp;16:16 XX nonsp;0:16 XX focus', '0:0 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;8:9 XX udef;8:10 XX compound;13:13 XX comp;33:33 XX neg;0:35 XX proper', '4:4 XX subord;10:10 XX q;11:11 XX q;12:12 XX q;13:14 XX udef;23:23 XX q;24:24 XX q;25:25 XX compound;24:25 XX udef', '15:15 XX q;16:16 XX q;15:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;19:22 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;26:27 XX udef', '0:0 XX nonsp;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX loc', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;8:10 XX udef;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;12:15 XX udef;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;0:20 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;8:10 XX udef;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;12:15 XX udef;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;0:20 XX focus', '3:3 XX q;4:5 XX compound;4:6 XX compound;15:15 XX q;16:16 XX q;21:21 XX q;22:22 XX q', '0:0 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;0:15 XX udef', '0:0 XX nonsp;4:4 XX q;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX q;0:16 XX loc', '1:1 XX q;2:2 XX q;3:3 XX q;19:19 XX q;20:20 XX q;22:22 XX nonsp;21:22 XX loc;0:22 XX compound', '0:0 XX conj;1:1 XX q;3:3 XX q;4:4 XX q;0:4 XX implicit', '3:3 XX q;7:7 XX q;7:7 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q', '1:1 XX q;2:2 XX q;3:3 XX q;2:3 XX udef;1:3 XX compound;1:3 XX udef;6:6 XX q;6:6 XX compound;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;25:25 XX q;28:28 XX nonsp;29:29 XX nonsp;0:29 XX appos', '', '0:0 XX q;1:1 XX q;8:8 XX q;9:9 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX proper;18:18 XX q;19:20 XX proper;16:20 XX udef;28:28 XX q;29:29 XX q;33:33 XX q;34:34 XX q;35:35 XX q;37:37 XX compound;36:37 XX compound;0:37 XX udef', '0:0 XX superl;6:6 XX q;7:7 XX q', '0:0 XX q;1:1 XX q;7:7 XX q;8:8 XX q;9:9 XX times;11:11 XX q;12:12 XX q;13:13 XX times;0:20 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:19 XX udef;20:20 XX d;18:20 XX compound;17:20 XX udef;0:21 XX focus', '6:6 XX q;7:7 XX q;8:8 XX times;13:13 XX q;14:14 XX q;15:15 XX times;16:16 XX nonsp;17:17 XX nonsp', '6:6 XX q;7:7 XX q;8:9 XX udef;6:9 XX udef;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX q;21:21 XX q;22:22 XX q;21:22 XX udef;23:23 XX q;24:25 XX compound;23:25 XX udef;21:25 XX udef;26:26 XX q;21:26 XX compound;27:27 XX q;28:29 XX compound;28:29 XX udef;20:29 XX udef;0:29 XX subord', '0:0 XX q;2:2 XX d;3:3 XX d;4:4 XX d;0:5 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;0:7 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX d;11:14 XX udef;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX d;23:26 XX udef;27:27 XX nonsp;28:28 XX nonsp;0:28 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '', '13:14 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;29:30 XX udef;31:32 XX compound;31:39 XX udef;29:39 XX udef', '0:2 XX appos;7:8 XX proper;17:19 XX udef;16:19 XX udef;23:23 XX q;0:26 XX appos', '12:12 XX entity;13:13 XX entity', '', '', '4:4 XX nonsp;6:6 XX nonsp;5:6 XX compound;7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;11:11 XX nonsp;4:11 XX loc', '7:8 XX udef;6:8 XX implicit', '0:0 XX q;0:0 XX nominalization;0:10 XX udef', '', '0:0 XX unknown', '0:1 XX unknown', '10:10 XX neg', '12:12 XX nonsp', '5:5 XX q;13:13 XX d;16:16 XX d;17:17 XX d;20:20 XX nonsp;21:21 XX nonsp;22:22 XX nonsp;21:22 XX loc;18:23 XX loc;25:25 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d', '8:8 XX q;10:10 XX q;10:10 XX udef', '29:29 XX q;28:29 XX compound;30:30 XX q;31:31 XX q;33:33 XX q;32:33 XX compound;35:35 XX q;37:37 XX q;38:38 XX q', '19:19 XX q;19:20 XX udef;31:31 XX q;30:31 XX proper;32:32 XX q;33:33 XX proper;0:33 XX subord', '', '5:5 XX q;5:5 XX nominalization', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;21:21 XX d;22:22 XX d;23:23 XX d;0:24 XX focus', '0:0 XX neg', '6:6 XX discourse', '', '', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;12:12 XX conj;13:13 XX conj;15:15 XX conj;16:16 XX conj;0:16 XX implicit', '2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;10:10 XX q', '3:3 XX q;4:4 XX q', '13:13 XX q;13:13 XX compound;15:15 XX q;23:23 XX q;23:23 XX compound;22:23 XX proper;24:24 XX q;24:24 XX compound', '1:1 XX neg', '12:12 XX q;12:13 XX udef', '2:2 XX d;3:3 XX d;6:6 XX q;7:7 XX q;6:8 XX udef;9:9 XX d;10:10 XX q;11:11 XX q;11:11 XX compound;10:11 XX udef;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX q;18:18 XX q;17:19 XX udef;20:20 XX d;21:21 XX d;0:23 XX focus', '2:2 XX q;3:3 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;0:19 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;0:17 XX focus', '3:3 XX q;2:3 XX compound;2:4 XX proper;8:8 XX nominalization;7:8 XX udef;7:9 XX compound', '0:1 XX proper;4:5 XX compound;21:21 XX nonsp;22:22 XX proper;0:22 XX appos', '0:0 XX d;0:3 XX focus', '0:12 XX appos', '16:16 XX compound', '9:10 XX compound;15:16 XX proper', '', '0:-1 XX EMPTY-LABEL', '2:3 XX compound;9:9 XX nonsp;10:10 XX proper;0:10 XX compound', '4:4 XX q;4:4 XX nominalization;6:6 XX q;6:6 XX nominalization;5:6 XX udef;7:7 XX q;7:7 XX nominalization;12:12 XX q;12:12 XX nominalization;13:14 XX compound;15:15 XX compound;13:15 XX proper', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX q;13:13 XX udef', '5:5 XX q;6:6 XX q;8:8 XX q;11:11 XX q;12:12 XX q;13:13 XX q', '8:8 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;13:14 XX udef', '0:0 XX q;1:1 XX q;2:2 XX d;3:3 XX d;5:5 XX d;0:6 XX udef', '', '', '2:2 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:7 XX focus', '', '13:13 XX q;13:13 XX compound', '2:2 XX q;3:3 XX times;8:9 XX proper;13:13 XX superl;14:15 XX compound;18:19 XX compound;23:23 XX q;22:23 XX compound;24:24 XX q', '9:9 XX q;10:11 XX compound;18:18 XX q;19:19 XX compound;18:19 XX udef;16:19 XX compound;0:19 XX compound', '2:2 XX superl', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX nonsp;9:9 XX q;11:11 XX q;12:12 XX q;0:20 XX udef', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;24:24 XX q;26:26 XX q;26:26 XX compound;25:26 XX proper;25:27 XX proper;30:30 XX q;30:30 XX compound;29:30 XX proper;29:30 XX proper;0:30 XX udef', '5:6 XX compound;8:9 XX compound;7:9 XX udef', '0:0 XX neg', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;17:17 XX d;18:18 XX d;0:21 XX focus', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;10:10 XX q;11:11 XX q;12:12 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;0:24 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;13:14 XX proper;13:15 XX compound', '', '0:0 XX d;1:1 XX q;2:2 XX q;3:4 XX compound;5:5 XX q;3:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;11:12 XX compound;10:12 XX udef;13:13 XX p;14:14 XX p;16:16 XX d;15:16 XX udef;13:16 XX with;17:17 XX q;19:19 XX d;20:20 XX d;22:22 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX q;30:30 XX q;32:32 XX d;33:33 XX q;34:34 XX q;38:38 XX proper;36:38 XX udef;35:38 XX udef;0:38 XX focus', '9:9 XX q;9:9 XX udef;0:9 XX compound', '', '1:1 XX nominalization;13:13 XX q;12:14 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;23:23 XX compound;21:23 XX udef;24:24 XX compound;21:24 XX proper', '0:2 XX compound', '', '12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX proper', '0:0 XX q;2:2 XX q;2:2 XX compound;1:2 XX udef;3:3 XX q;4:4 XX q;5:5 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;27:27 XX q;29:29 XX q;31:31 XX q;0:32 XX udef', '0:0 XX d;1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:14 XX compound;12:14 XX proper;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;26:26 XX q;27:28 XX compound;31:31 XX q;0:31 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;18:18 XX compound;0:18 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:10 XX proper', '13:13 XX q;12:13 XX proper;14:14 XX q', '5:5 XX neg', '13:13 XX q;16:16 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:25 XX proper;24:25 XX compound', '0:0 XX q;1:1 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX compound;24:24 XX q;27:27 XX q;0:28 XX udef', '0:0 XX superl;6:6 XX q;7:7 XX q;7:7 XX compound;6:7 XX udef;8:8 XX q;8:8 XX compound;9:9 XX q;9:9 XX compound;10:10 XX q;10:10 XX compound;12:12 XX q;13:13 XX q;13:14 XX udef;13:17 XX compound', '0:0 XX q;10:12 XX compound;13:13 XX nonsp;0:24 XX proper', '15:15 XX q;15:15 XX nominalization;16:17 XX compound;16:17 XX udef', '14:14 XX q;15:15 XX q;16:16 XX q;19:19 XX q;24:24 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;30:30 XX q;31:31 XX q;0:31 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX q;5:5 XX d;4:5 XX udef;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;13:13 XX d;0:13 XX focus', '', '0:4 XX unknown;0:4 XX appos', '0:0 XX q;0:1 XX proper;2:2 XX q;3:3 XX q;3:4 XX proper;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;0:15 XX udef', '2:3 XX compound;8:9 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;0:24 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;0:24 XX udef', '10:11 XX compound;10:12 XX udef;10:14 XX compound;19:19 XX superl;20:20 XX superl', '', '2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX nominalization;16:16 XX q;17:17 XX q;19:19 XX q;18:19 XX udef;20:20 XX q;21:21 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;4:6 XX compound;4:6 XX udef;13:13 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:19 XX proper;20:21 XX udef;23:23 XX q;22:24 XX udef;20:24 XX appos;28:28 XX q;28:28 XX compound;27:28 XX proper;29:29 XX q;27:29 XX proper;30:30 XX q;27:31 XX udef;27:31 XX compound;0:31 XX focus', '0:0 XX neg;2:2 XX q;7:7 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;20:20 XX d;21:21 XX q;22:22 XX d;0:24 XX focus', '1:1 XX neg', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:8 XX focus', '4:4 XX comp;8:8 XX q;9:9 XX q;9:9 XX udef', '1:1 XX conj;3:3 XX conj;5:5 XX conj;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;0:10 XX unknown', '0:0 XX q;0:0 XX pron;1:1 XX q;1:1 XX pron;2:2 XX q;3:3 XX q;3:3 XX pron;2:3 XX udef;5:5 XX q;5:5 XX pron;6:6 XX q;6:6 XX pron;7:7 XX q;7:7 XX pron;8:8 XX q;8:8 XX pron;9:9 XX q;9:9 XX pron;10:10 XX q;10:10 XX pron;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX q;19:20 XX udef;24:24 XX compound;19:24 XX appos;11:24 XX implicit;0:24 XX pronoun', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:13 XX proper;17:17 XX q;17:17 XX proper;0:17 XX compound', '11:11 XX neg', '5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;16:16 XX q;14:16 XX compound;18:18 XX q;20:20 XX q;19:20 XX udef', '2:2 XX neg', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;0:6 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX d;23:23 XX d;26:27 XX udef;33:33 XX q;34:35 XX compound;33:35 XX udef;32:35 XX udef;32:36 XX compound;39:39 XX proper;0:39 XX focus', '1:1 XX q;2:2 XX q;4:4 XX neg;19:19 XX nonsp', '0:0 XX d;1:1 XX d;2:2 XX d;5:6 XX compound;8:8 XX d;7:8 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;0:19 XX focus', '5:6 XX proper;10:11 XX compound;14:14 XX q;14:14 XX compound;13:14 XX proper', '6:6 XX comp;11:12 XX udef;26:26 XX q', '3:3 XX q;4:4 XX q;3:5 XX udef;7:7 XX q;8:8 XX q;8:8 XX compound;7:8 XX udef;6:8 XX udef;9:9 XX q;9:9 XX compound;6:10 XX udef;25:25 XX comp;32:32 XX q;33:33 XX q;34:34 XX q;33:34 XX udef;32:34 XX compound;32:34 XX udef', '6:6 XX q;11:11 XX q;10:11 XX proper;12:12 XX q;10:13 XX udef;29:29 XX comp;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;35:35 XX q;35:35 XX nominalization;37:37 XX q;37:37 XX nominalization;36:37 XX compound;38:38 XX q;38:38 XX nominalization', '4:4 XX q', '0:0 XX q;1:2 XX compound;13:13 XX superl;0:26 XX udef', '', '0:0 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;26:26 XX q;27:27 XX q;28:28 XX q;0:28 XX proper', '3:3 XX q;4:4 XX q', '4:4 XX q', '6:6 XX q;5:6 XX compound;7:7 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:9 XX focus', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;10:10 XX q;8:10 XX compound;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;13:22 XX udef;8:22 XX udef', '1:2 XX compound;6:6 XX q;13:13 XX q;13:13 XX nominalization;15:15 XX q;14:15 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;24:25 XX udef;26:26 XX q;27:27 XX udef;24:27 XX udef;19:27 XX udef;14:27 XX udef', '5:5 XX comp;8:10 XX udef;11:12 XX compound', '0:0 XX d;1:1 XX d;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX nominalization;7:8 XX compound;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX d;14:14 XX d;15:16 XX proper;20:20 XX udef;0:20 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:7 XX def', '1:1 XX neg;9:9 XX q;10:10 XX q;12:12 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;0:4 XX proper', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;19:19 XX q;20:20 XX q;21:22 XX compound;21:23 XX udef', '0:4 XX unknown;0:4 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;14:14 XX nominalization;0:18 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q;7:7 XX compound;7:9 XX udef;11:11 XX q;10:11 XX proper;12:12 XX q;13:14 XX proper;10:14 XX udef;10:14 XX compound;7:14 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;0:13 XX focus', '', '4:4 XX proper;3:4 XX udef;2:4 XX udef;1:4 XX udef;0:4 XX udef', '4:4 XX proper;3:4 XX udef;2:4 XX udef;1:4 XX udef;0:4 XX udef', '0:-1 XX EMPTY-LABEL', '8:8 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:23 XX appos', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;7:10 XX udef;3:10 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;20:20 XX q;21:21 XX udef;0:21 XX udef', '14:14 XX q;15:15 XX q;14:16 XX udef;18:18 XX q;19:19 XX q;20:20 XX times;18:20 XX udef;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;25:25 XX compound', '3:3 XX superl;15:15 XX times', '0:0 XX superl;2:3 XX compound;11:11 XX q;10:11 XX compound;17:17 XX q;18:18 XX q;19:19 XX times;22:22 XX q;21:22 XX compound;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:32 XX udef;31:33 XX udef;31:33 XX compound', '7:7 XX q;8:8 XX q', '', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX compound;7:8 XX proper;7:8 XX proper', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:10 XX proper;15:15 XX q;14:15 XX compound;16:16 XX q;16:16 XX compound;16:16 XX udef;14:16 XX proper;0:16 XX focus', '0:1 XX proper;5:6 XX proper;7:7 XX q;5:7 XX compound;5:8 XX proper;20:20 XX nonsp;19:21 XX loc;18:21 XX loc;0:43 XX appos', '9:9 XX q;11:11 XX q;12:13 XX proper;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX proper;18:20 XX proper;0:20 XX compound', '', '0:0 XX ellipsis', '7:7 XX q;6:7 XX compound;8:8 XX q;9:10 XX proper;11:11 XX q;9:11 XX compound;9:12 XX proper;6:12 XX proper;13:13 XX q;15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper;18:19 XX compound;22:22 XX q;21:22 XX compound;23:23 XX q;23:23 XX compound;23:24 XX udef;21:24 XX proper;14:24 XX appos;27:28 XX proper;0:40 XX appos', '2:2 XX q;3:3 XX q;6:7 XX compound;8:9 XX proper;10:10 XX q;8:10 XX compound;11:11 XX q;13:13 XX q;12:14 XX udef;8:14 XX proper', '0:0 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;14:14 XX d;15:15 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX d;0:24 XX focus', '0:1 XX proper;3:4 XX compound;7:7 XX q;7:7 XX compound;6:7 XX proper;6:8 XX proper;9:9 XX q;6:9 XX compound;10:10 XX q;6:11 XX proper;0:29 XX appos', '', '11:11 XX q;18:18 XX neg', '9:9 XX comp;16:16 XX q;20:20 XX neg;28:28 XX comp', '5:5 XX q;5:5 XX compound;4:5 XX proper;4:6 XX proper', '8:8 XX q;8:8 XX compound;7:8 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX q;7:8 XX compound;7:9 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;0:17 XX focus', '5:5 XX q', '1:1 XX neg', '0:0 XX conj;0:0 XX unknown;1:1 XX conj;1:1 XX unknown;2:2 XX conj;2:2 XX unknown;3:3 XX conj;3:3 XX unknown;4:4 XX conj;4:4 XX unknown;5:5 XX conj;5:5 XX unknown;6:6 XX q;7:7 XX q;8:8 XX q;12:12 XX conj;12:12 XX unknown;11:12 XX compound;9:12 XX compound;8:12 XX udef;13:13 XX conj;13:13 XX unknown;14:14 XX conj;14:14 XX unknown;15:15 XX conj;15:15 XX unknown;16:16 XX conj;16:16 XX unknown;17:17 XX conj;17:17 XX unknown;18:18 XX conj;18:18 XX unknown;19:19 XX conj;19:19 XX unknown;20:20 XX conj;20:20 XX unknown;21:21 XX conj;21:21 XX unknown;24:24 XX conj;24:24 XX unknown;25:25 XX conj;25:25 XX unknown;27:27 XX conj;27:27 XX unknown;28:28 XX q;0:28 XX implicit', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;0:29 XX proper', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:14 XX q;15:15 XX d;14:15 XX udef;16:16 XX d;18:18 XX d;20:20 XX d;0:20 XX focus', '2:2 XX q;3:3 XX q;2:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;6:9 XX udef;19:19 XX q;20:20 XX q', '0:0 XX mod;1:1 XX mod;2:2 XX mod;3:3 XX mod;4:4 XX mod;5:5 XX q;6:6 XX q;5:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX mod;8:12 XX udef;14:14 XX mod;15:15 XX mod;17:17 XX mod;18:18 XX mod;19:22 XX ;23:23 XX mod;24:24 XX mod;25:25 XX mod;26:26 XX mod;27:27 XX mod;28:28 XX mod;30:30 XX q;30:30 XX compound;29:30 XX proper;29:31 XX udef;32:32 XX mod;29:32 XX compound;33:33 XX mod;34:34 XX mod;35:35 XX mod;36:36 XX mod;37:37 XX mod;38:38 XX mod;39:39 XX mod;40:40 XX mod;41:41 XX mod;42:42 XX q;42:43 XX proper;44:44 XX mod;45:45 XX mod;46:46 XX mod;47:47 XX compound;0:47 XX relative', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:2 XX compound', '0:27 XX compound', '6:6 XX q;7:7 XX q;10:10 XX q;10:11 XX udef;8:11 XX udef', '1:1 XX neg;3:3 XX thing;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX udef', '', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX proper;0:11 XX focus', '0:0 XX q;1:1 XX q;12:12 XX q;15:15 XX q;16:16 XX q;17:17 XX q;21:21 XX q;24:24 XX q;35:35 XX comp;0:38 XX proper', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;9:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;13:16 XX udef;9:16 XX udef;17:17 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q', '', '4:4 XX mod;5:5 XX mod;7:7 XX q;8:8 XX q;8:8 XX compound;6:8 XX udef;10:10 XX mod;9:10 XX compound;6:10 XX udef;11:11 XX mod;12:12 XX mod;13:13 XX mod;14:14 XX mod;15:15 XX mod;16:16 XX mod;17:17 XX mod;18:18 XX mod;19:19 XX mod;20:20 XX mod;21:21 XX mod;22:22 XX mod;24:24 XX mod;23:24 XX compound;25:25 XX mod;26:26 XX q;26:26 XX udef', '4:4 XX q;5:5 XX q;10:10 XX q;13:14 XX compound;15:15 XX q;16:16 XX q;17:18 XX proper;19:20 XX compound;17:20 XX compound', '4:4 XX q;3:4 XX compound;5:5 XX q', '', '2:2 XX q;1:2 XX compound;3:3 XX q;5:5 XX d;4:5 XX proper;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;0:12 XX focus', '', '', '7:7 XX q;7:7 XX compound;7:9 XX udef', '3:3 XX q;2:3 XX compound;4:4 XX q;6:6 XX q;7:7 XX q;7:8 XX proper;11:12 XX compound;7:12 XX appos;27:27 XX superl;0:36 XX appos', '', '6:6 XX mod;7:7 XX mod;8:8 XX q;9:9 XX mod;8:9 XX measure;8:9 XX udef;10:10 XX mod;11:11 XX mod;12:12 XX mod;14:14 XX mod;15:15 XX mod;16:16 XX mod;17:17 XX mod;24:24 XX q;25:25 XX q;26:26 XX q', '17:17 XX loc', '13:13 XX nonsp', '0:9 XX compound', '6:7 XX compound;0:22 XX appos', '3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX q;22:22 XX q;24:24 XX q;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX q;23:31 XX udef;0:31 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;2:3 XX udef;4:4 XX q;5:5 XX q;0:9 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX q;5:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX d;9:12 XX udef;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:26 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:0 XX q;0:0 XX compound;5:5 XX q;6:6 XX q;7:7 XX q;0:25 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;11:13 XX udef;0:13 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;11:13 XX udef;0:13 XX focus', '', '1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;0:8 XX focus', '0:0 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:15 XX compound;16:16 XX q;18:18 XX q;18:18 XX compound;17:18 XX compound;16:18 XX udef;14:18 XX udef;0:18 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX q;0:18 XX udef', '2:2 XX q;2:2 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;17:17 XX q;19:19 XX q', '3:3 XX comp;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;11:15 XX udef', '7:7 XX q;9:9 XX comp', '4:4 XX q;4:4 XX compound;4:6 XX udef;7:7 XX q;4:7 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX d;17:17 XX d;21:21 XX comp', '6:6 XX q;0:6 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;19:19 XX d;0:19 XX udef', '2:3 XX compound;10:10 XX q;10:10 XX compound;0:21 XX subord', '0:1 XX compound;0:4 XX compound', '2:2 XX q;1:2 XX proper;3:3 XX q;4:5 XX proper;18:18 XX q;0:20 XX subord', '1:1 XX q;2:2 XX q;11:11 XX entity;0:25 XX compound', '4:4 XX q;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization', '1:1 XX q;2:2 XX q;3:4 XX proper;5:5 XX q;3:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX compound;14:16 XX udef;17:17 XX q;14:17 XX compound;10:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:23 XX udef;10:23 XX udef;0:26 XX compound', '20:20 XX d;20:21 XX parg', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX q;11:11 XX compound;10:11 XX proper;12:12 XX d;14:14 XX d;0:14 XX focus', '1:1 XX conj;2:2 XX conj;3:3 XX conj;5:5 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;11:11 XX conj;0:11 XX implicit', '11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper', '4:4 XX q;5:5 XX q;6:6 XX q', '3:4 XX proper;6:7 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;13:17 XX udef;11:17 XX udef;27:27 XX q;26:27 XX compound;0:27 XX compound', '', '9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;13:13 XX compound;14:14 XX q;14:14 XX compound;12:14 XX udef;12:14 XX udef;10:14 XX udef', '6:6 XX q;7:7 XX q;8:8 XX q', '9:9 XX neg', '0:0 XX q;0:0 XX compound;2:2 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX named;15:15 XX q;15:15 XX compound;16:16 XX q;16:16 XX compound;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX proper', '9:9 XX q;10:10 XX q', '4:5 XX proper;6:6 XX q;12:12 XX q;13:13 XX q;14:14 XX times', '15:15 XX neg', '11:11 XX nonsp;14:14 XX q;15:15 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;28:28 XX q;29:29 XX q;29:29 XX udef;24:29 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:12 XX compound;13:13 XX d;0:13 XX focus', '', '12:12 XX q;12:12 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX comp;8:10 XX subord;11:11 XX d;12:13 XX compound;14:14 XX d;0:14 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;14:14 XX d;17:17 XX superl;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;28:28 XX q;28:28 XX nominalization;31:31 XX q;31:31 XX nominalization;30:31 XX compound;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;35:35 XX d;34:35 XX compound;23:35 XX udef;18:35 XX udef;36:36 XX d;38:38 XX q;38:38 XX compound;37:38 XX proper;39:39 XX q;39:39 XX compound;41:41 XX q;41:41 XX compound;40:41 XX compound;37:41 XX udef;42:43 XX proper;42:43 XX compound;0:43 XX focus', '17:17 XX q;19:19 XX q;18:19 XX udef;20:20 XX q;20:21 XX udef;18:21 XX udef;22:22 XX q;18:22 XX compound;23:23 XX q;25:25 XX q;24:25 XX proper;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;28:29 XX udef;24:29 XX udef', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;11:11 XX conj;13:13 XX conj;14:14 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;22:22 XX q;23:23 XX q;24:24 XX times;22:24 XX udef;0:24 XX implicit', '2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX q;6:6 XX conj;7:7 XX conj;11:11 XX q;10:11 XX compound;9:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:17 XX compound;18:18 XX q;16:18 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX conj;23:24 XX udef;16:24 XX udef;13:24 XX udef;25:25 XX conj;27:27 XX conj;28:29 XX compound;30:30 XX conj;31:31 XX q;33:33 XX q;34:34 XX q;36:36 XX conj;35:36 XX proper;0:36 XX implicit', '3:3 XX comp;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;23:24 XX udef;26:26 XX nonsp;23:26 XX appos;27:27 XX nonsp;28:28 XX nonsp;30:30 XX nonsp;29:30 XX compound;31:31 XX nonsp;32:32 XX nonsp;33:33 XX nonsp;35:35 XX q;36:36 XX nonsp;35:36 XX udef;37:37 XX nonsp;38:38 XX q;39:39 XX q;40:40 XX q;42:42 XX q;38:42 XX udef', '1:1 XX q;0:1 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;10:11 XX loc;13:13 XX q;15:15 XX q;16:16 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;21:21 XX proper;17:21 XX udef;0:21 XX udef', '0:0 XX q;1:1 XX q;2:3 XX proper;4:4 XX q;5:5 XX q;5:5 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX d;17:18 XX compound;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;0:28 XX proper', '', '13:13 XX q;14:14 XX q;13:15 XX udef', '0:0 XX superl;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX proper;27:27 XX q;26:27 XX compound;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;35:35 XX udef', '1:1 XX comp;8:8 XX d;7:8 XX loc;6:8 XX loc;9:9 XX d;10:10 XX d;11:11 XX d;12:13 XX compound;14:14 XX q;14:14 XX compound;12:14 XX udef;12:15 XX udef;16:16 XX d;18:18 XX d;19:19 XX d;20:21 XX proper;22:22 XX d;20:22 XX compound;24:24 XX d;25:25 XX d;26:27 XX compound;26:27 XX udef', '4:4 XX nonsp;5:5 XX nonsp;6:6 XX nonsp;14:15 XX compound;13:15 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;19:24 XX udef', '0:0 XX q;0:0 XX compound;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;24:24 XX d;25:25 XX d;0:25 XX proper', '4:4 XX neg;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;33:33 XX q;34:34 XX q;34:34 XX nominalization;41:41 XX superl', '5:5 XX q;5:6 XX proper;9:9 XX nominalization;8:9 XX udef;8:10 XX compound;22:22 XX q;21:22 XX compound;23:23 XX q;21:24 XX proper;29:29 XX q;28:29 XX compound;28:30 XX proper;35:36 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX q;4:4 XX q;5:5 XX q;8:8 XX q;11:11 XX q;12:12 XX q;2:13 XX udef;14:14 XX q;15:15 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;23:24 XX compound;18:25 XX udef;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX q;33:33 XX q;33:33 XX compound;32:33 XX udef;34:34 XX d;0:34 XX focus', '0:0 XX q;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX d;0:11 XX udef', '13:13 XX q;12:14 XX udef', '8:8 XX q;9:9 XX q;10:10 XX q;11:12 XX proper;13:13 XX q;11:13 XX compound;15:15 XX q;14:16 XX udef;14:17 XX proper;18:18 XX q;19:19 XX q;20:20 XX q;18:21 XX proper;18:23 XX udef;14:23 XX udef;10:23 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX compound;10:28 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;9:9 XX d;0:9 XX focus', '1:1 XX d;2:2 XX d;3:4 XX udef;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;12:12 XX d;13:13 XX d;17:17 XX d;18:18 XX d;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;22:23 XX udef;24:24 XX q;24:24 XX nominalization;22:24 XX compound;25:25 XX q;25:25 XX nominalization;26:27 XX proper;29:29 XX compound;28:29 XX udef;26:29 XX appos;0:29 XX focus', '1:1 XX neg', '6:7 XX udef;5:7 XX implicit;4:7 XX udef;11:11 XX q;10:12 XX udef;20:20 XX times;25:25 XX neg', '1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX d;6:7 XX udef;8:8 XX d;9:9 XX d;10:11 XX proper;12:12 XX d;14:14 XX d;16:16 XX d;0:16 XX compound', '9:9 XX q;10:10 XX q;10:10 XX compound;9:10 XX udef;8:10 XX udef;23:23 XX neg', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '0:0 XX q;2:2 XX q;1:2 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;7:8 XX compound;9:9 XX q;10:11 XX udef;15:15 XX q;16:16 XX q;17:17 XX times;0:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:10 XX proper;13:13 XX q;15:15 XX compound;14:15 XX compound;13:15 XX udef;12:15 XX id;11:15 XX udef;0:15 XX focus', '9:9 XX q;10:10 XX q;11:11 XX times', '1:1 XX q;0:2 XX udef;7:7 XX q;6:7 XX compound;10:10 XX q;11:11 XX q;12:12 XX times;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX q;21:21 XX q;20:22 XX udef;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX q;27:27 XX q;26:28 XX udef;29:29 XX conj;30:30 XX q;31:31 XX q;30:32 XX udef;33:33 XX conj;34:34 XX conj;36:36 XX conj;35:36 XX proper;38:38 XX conj;23:38 XX implicit;0:38 XX udef', '1:1 XX neg', '5:5 XX conj;7:7 XX conj;8:8 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj', '1:2 XX compound;23:23 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:8 XX focus', '10:10 XX q;9:10 XX compound', '0:0 XX nonsp;2:2 XX q;1:2 XX udef;3:3 XX q;5:5 XX d;4:5 XX udef;7:7 XX d;9:9 XX nonsp;0:9 XX loc', '0:0 XX q;3:3 XX q;5:5 XX d;4:5 XX subord;6:6 XX d;8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;0:25 XX udef', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX q;12:12 XX compound;12:14 XX udef;15:15 XX neg;0:15 XX focus', '4:4 XX thing', '2:2 XX q;8:8 XX q', '', '', '0:0 XX q;0:0 XX compound;12:12 XX q;0:20 XX proper', '3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;8:9 XX udef;7:9 XX udef;12:12 XX nonsp;13:13 XX nonsp;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization', '15:15 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX conj;7:7 XX conj;8:8 XX conj;11:11 XX conj;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX d;16:17 XX udef;12:17 XX udef;18:18 XX d;19:20 XX proper;22:23 XX compound;26:27 XX proper;26:28 XX compound;31:31 XX q;31:31 XX compound;30:31 XX proper;30:32 XX proper;36:36 XX loc;0:36 XX focus', '', '0:1 XX proper;3:5 XX measure;13:14 XX proper;13:15 XX compound;19:19 XX q;20:20 XX q;0:21 XX appos', '', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;7:7 XX d;8:8 XX d;10:10 XX q;10:10 XX compound;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;0:17 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;0:6 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;0:10 XX focus', '', '0:0 XX d;1:1 XX d;3:3 XX q;3:4 XX udef;5:5 XX d;6:6 XX d;7:7 XX q;11:12 XX compound;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:20 XX focus', '0:6 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;11:11 XX d;10:11 XX loc;0:12 XX focus', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;2:3 XX udef;5:6 XX compound;11:11 XX d;10:11 XX proper;4:11 XX appos;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;0:20 XX focus', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;12:12 XX conj;13:13 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;0:25 XX implicit', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;14:14 XX d;0:15 XX focus', '0:10 XX subord', '', '0:12 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX proper;16:16 XX q;17:17 XX q;17:18 XX proper;14:18 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;20:21 XX udef;22:22 XX q;23:23 XX q;23:23 XX nominalization;25:25 XX q;25:25 XX nominalization;23:25 XX udef;0:25 XX focus', '4:4 XX q;5:5 XX q;6:6 XX q;4:6 XX udef', '0:0 XX conj;2:2 XX conj;3:3 XX q;3:3 XX pron;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;0:8 XX implicit', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:6 XX focus', '7:7 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;20:20 XX q;20:20 XX nominalization;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;0:29 XX appos', '', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX d;11:11 XX d;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;0:19 XX focus', '', '4:4 XX q;5:5 XX q;8:9 XX proper;11:13 XX measure;14:15 XX proper;14:16 XX compound;23:24 XX compound;20:24 XX eventuality;0:24 XX compound', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;14:14 XX conj;15:15 XX q;16:16 XX conj;15:16 XX udef;17:17 XX conj;18:18 XX q;19:19 XX d;18:19 XX udef;0:20 XX focus', '', '', '0:0 XX q;0:0 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;0:23 XX udef', '2:2 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;6:7 XX udef;8:8 XX conj;9:9 XX conj;11:11 XX conj;12:12 XX conj;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;13:18 XX udef', '0:23 XX appos', '3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;0:11 XX loc', '', '0:0 XX nonsp;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;9:9 XX named;10:10 XX q;10:10 XX named;11:11 XX q;11:11 XX compound;9:11 XX proper;9:12 XX udef;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX q;22:22 XX times;20:22 XX udef;24:24 XX d;23:24 XX compound;25:25 XX d;0:26 XX focus', '', '3:3 XX comp', '3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;6:6 XX q;5:6 XX udef;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;17:17 XX q;18:20 XX compound;17:20 XX udef;25:25 XX nonsp;25:25 XX loc;0:25 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;3:3 XX compound;5:5 XX q;6:6 XX q;11:11 XX q;12:12 XX q;15:15 XX comp;17:17 XX thing;18:18 XX q;0:28 XX proper', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;7:7 XX d;0:12 XX udef', '1:1 XX thing;3:3 XX q;4:4 XX q', '1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;17:17 XX q;17:17 XX nominalization;15:17 XX compound;18:18 XX q;18:18 XX nominalization;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX q;29:29 XX times;27:29 XX udef;30:30 XX d;31:31 XX d;32:32 XX d;33:34 XX udef;0:34 XX udef', '4:4 XX q;3:4 XX compound;15:15 XX q;14:15 XX compound;14:16 XX proper;19:19 XX compound;18:19 XX compound', '0:0 XX q;1:1 XX d;0:1 XX udef;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;8:9 XX compound;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX q;18:18 XX q;17:18 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;16:21 XX udef;0:21 XX focus', '0:0 XX q;1:1 XX q;0:7 XX udef', '0:0 XX d;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d;34:34 XX d;0:35 XX focus', '', '5:5 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q', '', '4:5 XX proper;11:11 XX q;12:12 XX q;17:17 XX q;16:17 XX compound;14:17 XX compound;18:18 XX q;19:19 XX q;19:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX q;7:8 XX proper;7:9 XX compound;13:14 XX proper;15:15 XX q;13:15 XX compound;6:15 XX appos;2:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:27 XX proper;31:31 XX d;30:31 XX compound;29:31 XX compound;26:31 XX appos;23:31 XX udef;32:32 XX d;33:33 XX d;0:33 XX focus', '', '13:14 XX compound;15:16 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:7 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:7 XX compound;8:8 XX q;9:9 XX q;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;0:29 XX focus', '7:7 XX q;9:9 XX q;7:10 XX udef;13:13 XX nonsp;15:15 XX nonsp;11:15 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;8:8 XX d;7:8 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX q;23:23 XX q;24:24 XX d;22:24 XX udef;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX d;27:28 XX udef;29:29 XX d;30:30 XX d;31:32 XX compound;33:33 XX d;31:33 XX compound;34:34 XX d;35:35 XX d;36:36 XX d;37:37 XX d;38:38 XX q;39:39 XX q;40:40 XX q;41:41 XX q;38:41 XX udef;0:41 XX focus', '1:1 XX q;2:2 XX q;11:11 XX thing;0:16 XX udef', '7:7 XX q;16:17 XX proper;0:17 XX compound', '', '0:0 XX q;7:7 XX q;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX times;0:30 XX proper', '11:11 XX q;12:12 XX q;15:16 XX compound;14:16 XX compound;19:19 XX comp;26:27 XX compound;28:28 XX q;26:28 XX compound', '0:0 XX q;2:2 XX q;18:18 XX q;0:28 XX proper', '1:2 XX compound;0:2 XX udef;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX d;12:12 XX q;12:12 XX compound;11:12 XX proper;13:13 XX d;11:13 XX proper;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX compound', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;7:7 XX q;7:8 XX udef;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;0:16 XX udef', '13:13 XX q;17:17 XX q', '11:11 XX q;10:11 XX proper;12:12 XX q;13:14 XX proper;15:15 XX q;13:15 XX compound;16:16 XX q;17:17 XX q;13:18 XX proper;20:20 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef;24:33 XX udef', '1:1 XX q;2:2 XX q;4:4 XX q;0:10 XX measure', '4:4 XX q;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX proper', '0:0 XX d;1:1 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;14:17 XX udef;10:17 XX udef;18:18 XX q;19:19 XX q;20:21 XX compound;23:23 XX q;20:23 XX compound;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX times;27:29 XX udef;25:29 XX udef;0:37 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;12:13 XX proper;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;19:21 XX compound;22:22 XX q;26:26 XX q;25:26 XX compound;27:27 XX q;28:28 XX q;29:29 XX nonsp;30:30 XX q;29:30 XX loc;28:30 XX udef;31:31 XX q;32:32 XX q;33:33 XX q;33:34 XX udef;35:35 XX q;33:35 XX compound;36:36 XX loc;18:36 XX udef;0:36 XX focus', '', '', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX q;13:14 XX compound;15:15 XX q;16:17 XX compound;18:18 XX q;16:18 XX compound;16:18 XX proper;0:18 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;0:18 XX focus', '0:0 XX q;1:1 XX q;5:5 XX comp;0:9 XX proper', '2:2 XX neg;16:16 XX q;17:17 XX q;0:17 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX compound;12:13 XX udef;0:13 XX udef', '5:5 XX q;7:7 XX q;8:8 XX q;7:9 XX udef;6:9 XX udef;10:10 XX q;6:10 XX compound;5:12 XX udef;13:13 XX q;14:15 XX compound;16:16 XX q;14:16 XX compound;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX udef;21:21 XX q;22:22 XX udef;19:22 XX udef;14:22 XX udef', '4:4 XX q;4:4 XX compound;3:4 XX proper;5:5 XX q;5:5 XX compound;3:5 XX proper;3:6 XX udef;9:9 XX q;9:9 XX compound;8:9 XX proper;10:11 XX udef;10:11 XX compound;8:11 XX udef;3:11 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX q;0:18 XX udef', '6:6 XX q;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q', '0:1 XX udef;10:10 XX q;10:10 XX compound;0:11 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;4:4 XX d;5:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;0:15 XX udef', '0:0 XX q;0:0 XX compound;2:2 XX q;3:3 XX q;0:20 XX proper', '0:0 XX d;1:1 XX d;3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:18 XX compound;19:19 XX q;17:19 XX udef;20:20 XX q;21:21 XX q;23:23 XX d;22:23 XX compound;21:23 XX udef;24:24 XX d;25:25 XX d;0:25 XX focus', '0:-1 XX EMPTY-LABEL', '0:0 XX q;1:1 XX q;6:6 XX q;4:6 XX compound;15:15 XX q;15:15 XX compound;14:15 XX proper;0:21 XX udef', '0:0 XX nonsp;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX nonsp;10:10 XX q;11:11 XX d;0:15 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX compound;8:9 XX proper;5:9 XX udef;0:9 XX focus', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization', '4:4 XX neg', '', '0:0 XX q;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX entity;9:9 XX entity;10:10 XX conj;11:11 XX conj;12:12 XX conj;14:14 XX conj;15:15 XX conj;17:17 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;26:26 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;35:35 XX q;35:35 XX compound;34:35 XX proper;37:37 XX q;37:37 XX compound;36:37 XX compound;0:37 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:17 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d;34:34 XX conj;36:36 XX conj;37:37 XX conj;0:37 XX focus', '', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;0:11 XX focus', '2:2 XX d;3:3 XX d;0:3 XX focus', '10:11 XX proper;12:12 XX q;0:29 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX entity;8:8 XX d;0:9 XX focus', '0:0 XX conj;2:2 XX conj;5:5 XX q;5:5 XX compound;4:5 XX udef;6:6 XX nonsp;4:6 XX udef;7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;11:12 XX proper;14:14 XX nonsp;13:14 XX compound;11:14 XX compound;15:15 XX nonsp;17:17 XX d;3:17 XX loc;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX q;24:24 XX q;26:26 XX d;23:26 XX udef;27:27 XX d;28:28 XX d;29:29 XX q;30:30 XX d;29:30 XX udef;31:31 XX d;33:33 XX d;34:34 XX d;36:36 XX d;37:37 XX d;38:38 XX d;39:39 XX q;40:40 XX d;39:40 XX udef;41:41 XX d;42:42 XX q;42:42 XX udef;0:42 XX implicit', '2:3 XX proper;4:5 XX compound', '', '2:2 XX q', '21:21 XX q;21:22 XX udef;29:29 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;19:19 XX d;18:19 XX proper;0:19 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;23:23 XX d;0:24 XX focus', '7:7 XX q', '7:7 XX q;7:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX q;0:13 XX focus', '3:3 XX q;4:4 XX q;5:5 XX q', '5:5 XX q;15:15 XX q;15:15 XX named;16:16 XX q;16:16 XX named;15:17 XX proper;18:19 XX compound;22:22 XX q;21:22 XX compound', '3:3 XX q;4:5 XX compound;0:11 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;0:3 XX focus', '', '0:0 XX q;0:0 XX compound;0:11 XX proper', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;0:22 XX compound', '0:-1 XX EMPTY-LABEL', '2:3 XX proper;7:7 XX q;6:7 XX compound;8:8 XX q;6:9 XX proper;10:10 XX neg;15:15 XX q;15:15 XX compound;14:15 XX udef;17:17 XX nonsp;16:17 XX compound;14:17 XX proper;18:18 XX nonsp;19:19 XX nonsp;20:20 XX nonsp;21:21 XX nonsp', '5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX proper;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;0:7 XX implicit', '2:2 XX q;1:2 XX proper;3:3 XX q;5:6 XX compound;7:7 XX proper;4:7 XX appos', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;0:19 XX focus', '0:12 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;11:11 XX neg;16:16 XX compound;0:16 XX focus', '1:1 XX q;0:1 XX proper;2:2 XX q;8:8 XX q;8:8 XX nominalization;0:24 XX udef', '12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d', '', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:13 XX udef', '0:0 XX q;1:1 XX q;8:8 XX neg;14:14 XX person;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;23:23 XX conj;24:24 XX conj;0:25 XX proper', '7:7 XX neg;14:14 XX thing', '7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization', '', '1:1 XX neg', '0:0 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;7:8 XX proper;9:9 XX q;7:9 XX compound;11:11 XX q;12:12 XX q;12:12 XX named;13:13 XX q;13:13 XX named;14:14 XX q;14:14 XX named;15:15 XX q;15:15 XX compound;12:15 XX udef;16:16 XX q;16:16 XX compound;12:16 XX number;17:18 XX compound;17:19 XX compound;12:19 XX udef;12:20 XX compound;23:23 XX q;23:23 XX compound;23:25 XX udef;31:31 XX q;31:31 XX proper;0:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;9:9 XX compound;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;19:19 XX d;0:20 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;6:6 XX d;7:8 XX proper;10:11 XX compound;14:14 XX q;13:14 XX compound;16:16 XX q;13:16 XX proper;0:16 XX focus', '0:1 XX number;2:3 XX compound;7:7 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;9:11 XX udef;13:13 XX q;31:31 XX comp;32:32 XX nonsp;33:33 XX nonsp;34:34 XX nonsp;35:35 XX nonsp;0:35 XX compound', '4:4 XX q;4:5 XX udef;9:10 XX number;0:11 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;17:18 XX number;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:26 XX focus', '1:1 XX q;2:2 XX q;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:17 XX udef;18:18 XX q;18:18 XX nominalization;16:18 XX compound;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;20:21 XX proper;22:22 XX q;23:23 XX q;23:23 XX nominalization;22:23 XX udef;0:23 XX udef', '0:0 XX d;1:1 XX d;4:4 XX d;3:4 XX compound;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:14 XX number;15:15 XX q;13:15 XX compound;16:16 XX q;18:18 XX q;22:22 XX q;21:22 XX compound;20:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX q;0:30 XX focus', '0:12 XX generic', '3:3 XX q', '0:0 XX conj;0:0 XX unknown;1:1 XX conj;1:1 XX unknown;3:3 XX conj;3:3 XX unknown;4:4 XX conj;4:4 XX unknown;5:5 XX conj;5:5 XX unknown;6:6 XX conj;6:6 XX unknown;7:7 XX conj;7:7 XX unknown;9:9 XX conj;9:9 XX unknown;10:10 XX conj;10:10 XX unknown;12:12 XX conj;12:12 XX unknown;14:14 XX conj;14:14 XX unknown;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX compound;0:23 XX implicit', '0:0 XX q;0:11 XX udef', '12:12 XX q;11:12 XX fraction;16:17 XX compound;20:20 XX comp;15:20 XX loc;25:25 XX subord;25:25 XX comp;27:27 XX neg', '0:1 XX number;14:14 XX q;15:15 XX q;16:16 XX q;0:17 XX compound', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;0:14 XX implicit', '0:0 XX d;2:2 XX d;6:6 XX d;7:7 XX d;9:9 XX q;8:9 XX compound;8:10 XX proper;11:11 XX q;12:12 XX q;12:13 XX proper;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;0:18 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX neg;0:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX number;14:14 XX d;12:14 XX compound;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;23:23 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;30:30 XX d;32:32 XX d;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX q;40:40 XX q;39:40 XX compound;41:41 XX q;35:41 XX udef;0:41 XX focus', '2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;6:6 XX compound;8:9 XX compound;6:9 XX udef;18:18 XX q;18:18 XX compound;17:18 XX proper;17:19 XX proper', '7:8 XX proper;7:9 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX q;10:10 XX compound;9:10 XX udef;11:11 XX d;0:11 XX focus', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX q;23:23 XX q;23:23 XX compound;22:23 XX udef;21:23 XX udef;24:24 XX d;25:25 XX d;26:26 XX d;0:26 XX focus', '', '0:1 XX proper;2:2 XX q;0:2 XX compound;0:3 XX proper;9:9 XX d;8:9 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;25:26 XX udef;0:26 XX appos', '', '15:15 XX neg;21:21 XX neg', '5:6 XX proper;8:8 XX q;7:9 XX udef;10:10 XX q;7:10 XX compound;5:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:15 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;21:22 XX udef;24:24 XX q;23:24 XX compound;21:24 XX compound;17:24 XX udef;14:24 XX loc;25:25 XX q;27:27 XX q;26:27 XX number;28:28 XX q;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX number;29:31 XX udef', '1:1 XX q;6:7 XX proper;4:7 XX udef;3:7 XX udef;10:10 XX q;19:19 XX q;0:20 XX udef', '3:4 XX udef;10:10 XX q;13:13 XX q;11:13 XX compound;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;28:28 XX q;26:28 XX compound;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;31:32 XX udef;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX q;41:41 XX q;42:42 XX q;39:42 XX udef;31:42 XX udef;43:43 XX q;44:44 XX q;45:45 XX q;46:46 XX nonsp;46:46 XX loc;26:46 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;11:12 XX compound;13:13 XX q;0:13 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX q;5:6 XX compound;7:7 XX q;0:7 XX focus', '', '9:9 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:24 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper;15:15 XX q;12:15 XX compound;16:16 XX q;16:16 XX compound;16:16 XX udef;12:16 XX proper;0:16 XX focus', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX compound;5:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;15:19 XX udef', '0:0 XX q;1:1 XX q;1:1 XX named;2:2 XX q;2:2 XX named;3:3 XX q;3:3 XX named;6:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;0:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '5:5 XX q;5:6 XX udef;7:8 XX compound;10:10 XX q;17:17 XX q', '2:2 XX neg', '4:4 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;19:19 XX q;0:20 XX focus', '2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;25:25 XX d;0:25 XX focus', '8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX nonsp;19:19 XX nonsp;18:19 XX loc', '1:1 XX q;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX compound;12:12 XX d;10:12 XX udef;14:14 XX d;0:14 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;0:7 XX focus', '4:4 XX comp', '2:2 XX q;3:4 XX compound', '5:5 XX q', '', '11:12 XX compound', '11:12 XX compound', '6:6 XX q;9:9 XX q;7:9 XX compound;10:10 XX q;11:11 XX q;13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper;15:15 XX q;12:15 XX compound;16:16 XX q;19:21 XX compound;0:33 XX subord', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;2:3 XX udef;4:5 XX udef;8:8 XX d;7:8 XX udef;4:8 XX udef;10:10 XX d;0:21 XX focus', '3:3 XX neg;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q', '2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization', '0:0 XX q;2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;12:12 XX q;25:25 XX q;24:25 XX compound;26:26 XX q;28:28 XX q;27:28 XX compound;27:28 XX proper;0:28 XX udef', '4:4 XX q;4:4 XX compound;14:14 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;23:23 XX d;0:23 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;21:21 XX compound;20:21 XX proper;0:21 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX d;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;6:8 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX d;19:20 XX udef;14:20 XX udef;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;5:25 XX focus;26:26 XX d;28:28 XX d;27:28 XX compound;0:28 XX implicit', '4:4 XX q;4:4 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;27:27 XX nonsp;26:28 XX loc;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX udef;11:35 XX udef', '0:0 XX conj;2:2 XX conj;4:4 XX conj;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX conj;19:19 XX q;19:19 XX compound;18:19 XX udef;20:20 XX q;18:20 XX udef;21:21 XX q;22:22 XX udef;18:22 XX udef;0:22 XX implicit', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:17 XX proper;20:20 XX q;20:20 XX compound;19:20 XX proper;19:21 XX proper;19:22 XX compound;0:22 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;18:18 XX q;19:19 XX d;18:19 XX udef;15:19 XX generic;20:20 XX d;21:22 XX proper;23:23 XX q;24:24 XX q;25:26 XX proper;27:27 XX q;25:27 XX compound;25:27 XX proper;23:27 XX udef;21:27 XX appos;0:27 XX focus', '4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;11:11 XX q;9:11 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX compound;14:15 XX udef;8:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;16:16 XX compound;0:17 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:7 XX proper;9:10 XX compound;13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper;16:17 XX proper;18:18 XX compound;16:18 XX compound;12:18 XX appos;0:18 XX focus', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;6:6 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;0:19 XX implicit', '2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization', '0:0 XX d;1:1 XX d;2:2 XX q;2:2 XX nominalization;4:4 XX d;3:4 XX udef;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:30 XX proper;31:31 XX q;32:32 XX q;34:34 XX q;34:34 XX compound;33:34 XX proper;35:35 XX q;33:35 XX proper;38:38 XX q;37:39 XX udef;37:40 XX compound;31:40 XX udef;0:40 XX focus', '0:13 XX subord', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX q;5:5 XX q;4:5 XX udef;6:6 XX q;8:8 XX d;9:9 XX q;10:10 XX d;11:11 XX d;14:14 XX d;0:14 XX focus', '1:1 XX q;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;0:23 XX focus', '', '0:0 XX d;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;0:19 XX focus', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;0:6 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;7:8 XX udef;9:9 XX q;10:10 XX q;11:11 XX d;10:11 XX udef;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX d;18:18 XX d;20:20 XX d;21:21 XX d;22:22 XX q;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX d;28:28 XX d;29:29 XX q;30:30 XX q;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;34:34 XX q;34:34 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX q;36:36 XX nominalization;37:37 XX d;31:37 XX udef;38:38 XX d;0:39 XX focus', '5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;18:18 XX nominalization;19:19 XX q;20:20 XX q;20:20 XX nominalization;19:20 XX udef;22:22 XX q;22:22 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX d;18:25 XX udef;15:25 XX udef;9:26 XX focus', '6:8 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;28:28 XX superl;31:31 XX nonsp;0:31 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:12 XX focus', '6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;9:10 XX udef;9:11 XX udef;0:22 XX proper', '0:0 XX q;1:2 XX proper;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX compound;13:14 XX udef;15:15 XX q;13:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX d;0:27 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;9:9 XX compound;8:9 XX udef;10:10 XX d;8:10 XX udef;0:11 XX focus', '9:9 XX superl', '5:5 XX nonsp;7:7 XX nonsp;8:8 XX nonsp;10:10 XX nonsp;9:10 XX compound;11:11 XX nonsp;12:12 XX nonsp;13:13 XX nonsp', '1:1 XX q;1:1 XX pron;3:3 XX q;3:3 XX pron;7:7 XX d;0:8 XX pronoun', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;14:15 XX compound;16:16 XX nominalization;14:16 XX udef;14:17 XX compound;19:19 XX q;19:19 XX nominalization;19:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX nonsp;10:10 XX q;11:11 XX d;10:11 XX udef;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;0:21 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;6:7 XX proper;9:9 XX d;8:9 XX compound;6:9 XX compound;10:10 XX d;0:10 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;0:19 XX focus', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX d;11:13 XX udef;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;25:25 XX d;24:25 XX loc;26:26 XX d;27:27 XX d;28:28 XX d;10:28 XX focus;30:30 XX d;29:30 XX proper;0:30 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;0:7 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;14:14 XX d;0:15 XX focus', '', '0:0 XX q;0:0 XX compound;2:2 XX q;2:2 XX nominalization;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX udef;6:9 XX udef;0:9 XX udef', '1:1 XX neg;0:1 XX unknown', '2:2 XX q;1:2 XX compound', '1:1 XX q;1:1 XX compound;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;12:12 XX compound;11:12 XX proper;14:14 XX q;15:15 XX q;15:15 XX compound;0:15 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;9:10 XX proper;11:11 XX q;11:11 XX compound;9:11 XX udef;12:12 XX q;12:12 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;20:20 XX q;21:21 XX q;26:26 XX q;27:27 XX q;28:28 XX times;0:28 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;10:10 XX q;10:11 XX proper;7:11 XX appos;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;27:27 XX q;29:29 XX q;26:29 XX udef;30:30 XX q;31:31 XX q;32:32 XX q;31:32 XX udef;26:32 XX udef;0:32 XX udef', '1:1 XX comp;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:22 XX proper', '8:8 XX conj;10:10 XX conj;9:10 XX udef;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;16:16 XX conj', '0:0 XX q;2:2 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;13:13 XX q;14:14 XX q;24:24 XX q;25:25 XX q;26:27 XX proper;28:28 XX compound;0:28 XX udef', '4:4 XX q;7:7 XX q;8:8 XX q;9:9 XX q;9:11 XX udef;16:16 XX q;19:19 XX q;22:22 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX udef;8:8 XX d;6:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;12:13 XX udef;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX q;18:18 XX compound;17:18 XX proper;17:19 XX proper;20:20 XX d;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:28 XX compound;29:29 XX d;30:30 XX d;32:32 XX d;0:32 XX focus', '11:11 XX p;12:12 XX p;13:13 XX p;14:14 XX p;15:15 XX p;16:16 XX p;17:17 XX p;18:18 XX p;19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX p;23:23 XX p;25:25 XX p;26:26 XX p;27:27 XX p;28:28 XX compound', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;2:3 XX compound;4:4 XX q;4:4 XX nominalization;5:5 XX q;6:6 XX q;6:6 XX nominalization;5:6 XX udef;7:7 XX nonsp;8:8 XX nonsp;10:10 XX nonsp;9:10 XX proper;11:11 XX nonsp;13:13 XX nonsp;15:15 XX d;14:15 XX loc;9:15 XX loc;7:15 XX loc;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX q;25:25 XX d;0:26 XX udef', '0:1 XX udef;15:15 XX q;15:15 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;23:23 XX nominalization;22:23 XX compound;21:23 XX udef;18:23 XX udef;16:23 XX udef;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;30:31 XX udef;0:31 XX compound', '0:0 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;9:9 XX q;9:10 XX udef;11:11 XX d;9:11 XX compound;12:12 XX d;13:14 XX udef;16:16 XX q;15:16 XX compound;13:16 XX compound;17:17 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;20:20 XX compound;20:20 XX udef;18:20 XX proper;0:20 XX focus', '', '5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;4:7 XX proper', '', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;9:10 XX udef;11:11 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;24:24 XX comp;29:29 XX q;29:29 XX compound;28:29 XX udef;31:31 XX q;30:31 XX compound;28:31 XX proper;22:31 XX appos;32:32 XX q;21:32 XX udef;14:32 XX udef;12:32 XX udef', '10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX p;24:24 XX p;26:26 XX q;26:26 XX compound;25:26 XX udef;23:26 XX with;12:26 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;25:25 XX udef;0:25 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX nonsp;31:31 XX nonsp;32:32 XX d;30:32 XX loc;34:34 XX d;35:35 XX d;36:36 XX q;37:37 XX q;0:37 XX focus', '6:6 XX neg;9:9 XX q;10:10 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;9:14 XX udef;7:14 XX generic;27:27 XX q', '1:1 XX q;3:3 XX q;4:4 XX q;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;11:12 XX udef;13:13 XX q;14:14 XX q;14:14 XX udef;0:14 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;0:24 XX focus', '4:4 XX q;16:16 XX q', '8:8 XX q;9:9 XX q;15:15 XX neg;36:36 XX q;0:36 XX subord', '3:3 XX q;3:3 XX compound;2:3 XX udef;4:5 XX compound;13:14 XX udef;21:21 XX superl;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX udef', '5:5 XX q;4:5 XX compound;8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX q;28:28 XX comp', '0:0 XX q;1:1 XX q;3:3 XX q;4:4 XX q;4:4 XX compound;0:17 XX proper', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:12 XX proper;14:15 XX compound;17:18 XX proper;19:19 XX q;17:19 XX compound;17:19 XX proper;0:19 XX focus', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;0:10 XX udef', '4:4 XX q', '1:1 XX q;3:3 XX q;2:3 XX compound;2:4 XX proper;0:4 XX udef;7:8 XX compound;13:13 XX d;11:13 XX compound;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;0:33 XX appos', '0:0 XX q;1:1 XX q;2:2 XX q;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX q;0:29 XX udef', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;14:14 XX d;0:14 XX udef', '6:6 XX q;8:8 XX neg;10:10 XX q;11:11 XX q', '0:0 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;19:19 XX q;0:19 XX udef', '1:1 XX q;3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX udef', '0:0 XX q;8:8 XX q;9:10 XX compound;0:17 XX udef', '0:20 XX unknown', '9:9 XX q;10:11 XX compound;12:12 XX nonsp;13:13 XX nonsp;14:14 XX nonsp', '2:2 XX q;2:3 XX udef', '13:13 XX neg', '0:0 XX q;0:0 XX compound;3:3 XX q;5:5 XX q;5:5 XX compound;6:6 XX q;6:6 XX compound;7:7 XX q;7:7 XX compound;9:9 XX q;10:10 XX q;12:12 XX q;16:16 XX neg;18:18 XX q;19:19 XX q;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX compound;23:23 XX q;23:23 XX compound;24:24 XX q;24:24 XX compound;26:26 XX q;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;0:31 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;4:6 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX proper;20:20 XX d;7:20 XX udef;21:21 XX d;22:22 XX d;0:23 XX focus', '11:12 XX udef', '0:10 XX compound', '11:11 XX q', '5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;12:12 XX q;12:13 XX udef;14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization', '10:10 XX q;11:11 XX q;10:13 XX udef;16:16 XX neg;22:22 XX comp;25:25 XX q', '2:2 XX q;2:2 XX compound;1:2 XX udef;4:4 XX q;3:4 XX compound;1:4 XX proper;5:5 XX q;6:6 XX q;7:8 XX compound;6:8 XX udef;32:32 XX q', '']

abs_glove_pred = ['', '15:15 XX superl;18:18 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX compound;22:25 XX udef;0:25 XX compound', '1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;5:6 XX proper;8:8 XX d;7:8 XX compound;5:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:19 XX compound;17:19 XX udef;15:19 XX udef;4:19 XX udef;0:19 XX loc', '12:12 XX q;16:16 XX q', '9:9 XX q;10:11 XX udef;0:16 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d', '3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;21:22 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;14:14 XX d;13:14 XX compound;12:14 XX udef;15:15 XX d;16:17 XX proper;18:18 XX q;19:19 XX q;21:21 XX q;20:21 XX compound;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;32:32 XX compound;26:32 XX udef;24:32 XX udef;20:32 XX udef;18:32 XX udef;16:32 XX appos;0:32 XX focus', '6:6 XX nonsp;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;16:16 XX q;17:17 XX udef', '17:18 XX proper;19:19 XX q;17:19 XX compound;17:20 XX proper;23:23 XX q;22:23 XX proper;24:24 XX q;25:25 XX q;25:25 XX proper;22:25 XX udef;17:25 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:14 XX compound;15:15 XX q;16:16 XX d;18:18 XX q;20:21 XX proper;0:22 XX focus', '7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;0:19 XX compound', '9:9 XX q;12:12 XX q;0:27 XX appos', '10:10 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX q;15:15 XX q;10:15 XX udef;16:16 XX q;18:18 XX d;17:18 XX udef;7:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX q;29:29 XX d;27:29 XX udef;6:29 XX udef;30:30 XX d;32:32 XX d;33:33 XX q;33:33 XX nominalization;33:33 XX udef;5:33 XX udef;4:33 XX udef;0:33 XX focus', '14:14 XX q;21:21 XX q', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;6:6 XX q;7:7 XX q;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX q;31:31 XX q;32:32 XX q;0:32 XX focus', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:13 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;2:3 XX udef;4:4 XX d;5:5 XX d;8:8 XX neg;15:15 XX q;18:18 XX q;18:18 XX nominalization;20:20 XX q;20:20 XX nominalization;18:21 XX udef;14:21 XX udef;0:21 XX udef', '1:1 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;18:18 XX q;21:21 XX q;21:21 XX udef;18:21 XX udef;10:21 XX udef;7:21 XX udef', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;13:13 XX udef;12:13 XX udef;10:13 XX udef;0:13 XX focus', '9:9 XX q;9:9 XX compound;8:9 XX udef;10:11 XX proper;10:12 XX compound', '0:0 XX superl', '0:0 XX q;4:4 XX q;6:6 XX q;8:8 XX q;12:12 XX q;12:12 XX compound;25:25 XX q;26:26 XX q;0:26 XX udef', '8:8 XX q;8:8 XX compound;10:10 XX q;11:11 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;9:9 XX q;17:17 XX q;17:17 XX nominalization', '4:4 XX comp;0:4 XX compound', '4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;11:11 XX compound;11:12 XX udef;9:12 XX udef', '11:11 XX q;12:12 XX q;18:18 XX nonsp;24:24 XX superl;0:28 XX appos', '', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;9:11 XX compound;12:12 XX d;20:20 XX q;21:21 XX compound;18:21 XX udef;13:21 XX subord;0:21 XX proper', '', '8:8 XX nominalization;15:15 XX superl', '2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX q;9:9 XX nominalization;13:14 XX proper;15:15 XX q;13:15 XX compound;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;26:27 XX proper;26:27 XX compound;24:27 XX compound;20:27 XX udef;18:27 XX udef;10:27 XX proper;9:27 XX udef;8:27 XX focus', '15:15 XX q;16:16 XX q;17:17 XX q;33:33 XX q;34:34 XX q', '15:15 XX q;16:16 XX q;0:17 XX appos', '0:0 XX neg;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;0:21 XX focus', '11:11 XX thing', '0:0 XX q;1:1 XX q;13:13 XX q;20:20 XX q;21:21 XX q;0:21 XX proper', '6:6 XX q;8:8 XX q', '14:14 XX q;15:15 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;6:7 XX proper;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:27 XX focus', '6:6 XX q', '4:4 XX comp;5:5 XX q;0:5 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX compound;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;0:20 XX focus', '6:6 XX q;7:7 XX q;8:9 XX compound;16:16 XX q;17:17 XX q;18:18 XX q', '8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;20:27 XX compound;16:27 XX udef', '5:5 XX q;6:6 XX q;7:7 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX d;18:18 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;27:27 XX d;26:27 XX proper;0:27 XX focus', '4:4 XX q', '0:0 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:24 XX proper', '0:0 XX neg', '0:0 XX d;1:1 XX d;2:2 XX d;7:7 XX d;8:8 XX q;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;20:20 XX d;21:21 XX d;22:22 XX q;23:23 XX q;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX d;0:28 XX focus', '1:1 XX neg', '10:10 XX comp;17:17 XX q;18:18 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;5:5 XX q;4:5 XX compound;6:6 XX q;8:8 XX q;8:8 XX compound;7:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX nonsp;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;0:18 XX focus', '9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;20:21 XX udef;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;27:27 XX q;28:28 XX q;28:28 XX udef;26:28 XX udef;23:28 XX udef;20:28 XX udef;11:28 XX udef;10:28 XX udef;9:28 XX udef', '0:15 XX compound', '1:1 XX q;2:2 XX q;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;14:14 XX q;14:14 XX nominalization;13:14 XX compound;15:15 XX q;15:15 XX nominalization;17:17 XX q;17:17 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;3:21 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX q;15:15 XX d;16:16 XX d;17:17 XX q;0:17 XX udef', '6:7 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;10:10 XX q;11:11 XX q;0:11 XX udef', '', '5:5 XX q;8:8 XX q;9:9 XX q;12:13 XX compound;20:22 XX compound;26:26 XX q;26:26 XX compound;25:26 XX proper;27:28 XX proper;27:29 XX compound;25:29 XX proper;32:32 XX proper;9:32 XX udef', '5:5 XX q;4:5 XX compound;6:6 XX q;7:8 XX proper;15:15 XX q;16:16 XX q', '', '3:3 XX q;5:5 XX q;4:5 XX udef;7:7 XX q;7:8 XX udef;6:8 XX udef;3:8 XX udef;2:8 XX udef', '4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX udef;8:8 XX q;13:13 XX q;16:16 XX q;17:17 XX q;20:20 XX q;21:21 XX q;22:22 XX q', '5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q', '6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:15 XX udef;11:15 XX udef;17:18 XX udef;16:18 XX subord;19:19 XX q;32:32 XX compound;9:32 XX udef', '5:5 XX q;6:7 XX proper;13:14 XX compound;16:17 XX udef;18:18 XX q;19:19 XX q', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX q;6:6 XX compound;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;0:17 XX focus', '0:1 XX proper;0:12 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;12:13 XX udef;10:13 XX udef;35:35 XX q;35:35 XX udef;0:35 XX proper', '10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper', '', '8:9 XX compound;10:11 XX compound', '0:0 XX q;0:0 XX compound;0:15 XX udef', '4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:7 XX compound;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;21:21 XX q;21:21 XX nominalization;23:23 XX q;24:24 XX q;26:26 XX q;26:26 XX nominalization', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;9:12 XX udef;19:19 XX q;19:19 XX udef', '0:1 XX proper;6:6 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;18:18 XX q;0:24 XX compound', '3:3 XX q;3:3 XX compound;2:3 XX proper;5:5 XX q;4:5 XX compound;2:5 XX udef;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;18:18 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:28 XX compound', '19:19 XX q;20:20 XX q;20:20 XX compound', '10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;15:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;12:26 XX udef;0:26 XX compound', '0:1 XX udef;7:7 XX neg;0:21 XX compound', '4:4 XX q', '5:5 XX q;7:7 XX q;6:7 XX compound;8:8 XX q;9:9 XX q;10:11 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;18:19 XX udef;20:20 XX q;21:21 XX udef', '4:4 XX q;9:9 XX q;13:14 XX compound', '0:12 XX compound', '5:5 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:22 XX compound;18:22 XX udef;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;30:32 XX udef', '6:6 XX q;10:10 XX q;10:10 XX udef', '0:0 XX q;0:0 XX compound;8:8 XX neg;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;0:27 XX proper', '0:12 XX measure', '4:4 XX q;4:4 XX compound;3:4 XX udef;17:18 XX compound', '7:7 XX q;8:8 XX q;11:11 XX q;12:12 XX q;13:13 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;12:20 XX udef', '10:10 XX q;11:11 XX q;12:13 XX proper;14:14 XX q;12:14 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX udef;22:23 XX udef;19:23 XX udef;0:23 XX compound', '5:5 XX q;7:7 XX q;10:10 XX q;7:11 XX udef', '', '15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q', '0:0 XX q;0:0 XX compound;19:19 XX q;20:20 XX q;0:20 XX proper', '3:3 XX q;3:3 XX nominalization;4:4 XX q;5:5 XX q;5:5 XX nominalization;4:5 XX udef;7:7 XX q', '4:5 XX proper', '6:6 XX q;15:15 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX compound;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;22:24 XX udef;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;32:33 XX udef;34:34 XX compound;32:34 XX compound;29:34 XX udef;26:34 XX udef;22:34 XX udef;21:34 XX udef;0:34 XX focus', '0:0 XX q;1:1 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;0:18 XX udef', '2:2 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX times;22:24 XX udef;26:26 XX q;27:27 XX q;29:29 XX q;28:29 XX proper', '5:5 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;9:16 XX udef', '3:3 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;6:7 XX udef;11:12 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;20:20 XX compound;21:21 XX q;21:21 XX compound;22:22 XX q;22:22 XX compound;23:24 XX compound;20:24 XX udef;25:25 XX q;26:26 XX q;28:29 XX compound;33:33 XX superl;14:34 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;6:7 XX compound;10:10 XX d;9:10 XX compound;5:10 XX compound;0:10 XX focus', '0:0 XX q;0:0 XX compound;1:1 XX q;1:1 XX compound;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX times;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;33:33 XX q;33:33 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX q;36:36 XX nominalization;37:37 XX q;37:37 XX nominalization;38:38 XX q;38:38 XX nominalization;0:38 XX udef', '4:4 XX q;4:4 XX compound;5:5 XX q;5:5 XX compound;6:6 XX q;6:6 XX compound;16:16 XX udef', '2:2 XX q;4:4 XX q;3:4 XX compound;3:5 XX proper;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:20 XX compound;25:25 XX q;25:26 XX udef', '9:9 XX q;9:10 XX udef;11:11 XX q;12:13 XX udef;15:15 XX q;15:15 XX udef', '', '2:2 XX q;2:2 XX compound;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;0:16 XX appos', '1:1 XX q;0:1 XX proper;2:2 XX q;3:4 XX proper;7:9 XX compound;3:9 XX appos;13:13 XX q;14:14 XX q;18:18 XX q;19:19 XX q;20:20 XX q;24:25 XX udef;30:30 XX q;31:31 XX q;33:33 XX q;34:34 XX q;35:35 XX q;30:35 XX udef;19:35 XX udef;17:35 XX udef;13:35 XX udef;0:35 XX udef', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;11:11 XX q;12:12 XX q;16:16 XX q;17:18 XX compound;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;26:26 XX q;25:26 XX udef;0:27 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;2:3 XX udef;4:4 XX q;5:6 XX compound;5:7 XX udef;16:16 XX q;16:16 XX compound;15:16 XX proper;15:17 XX proper;0:22 XX udef', '1:1 XX nominalization;10:10 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;27:27 XX q;28:28 XX q;30:30 XX q;27:30 XX udef;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef', '0:0 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX neg;14:23 XX udef;0:23 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX d;14:14 XX q;15:15 XX q;16:16 XX q;0:16 XX focus', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;12:12 XX q;20:20 XX q;22:22 XX compound;21:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;33:33 XX d;35:35 XX d;36:36 XX d;37:37 XX d;39:39 XX d;41:41 XX loc;0:41 XX focus', '15:15 XX compound', '', '1:1 XX q;0:1 XX compound;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:17 XX udef;12:17 XX udef;7:33 XX udef;0:33 XX proper', '9:9 XX q;8:10 XX udef;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;27:27 XX compound;26:27 XX udef;0:27 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;34:34 XX q;34:34 XX nominalization;35:35 XX q;35:35 XX nominalization;37:37 XX q;37:37 XX nominalization;38:38 XX q;38:38 XX nominalization;0:38 XX focus', '9:9 XX q;9:9 XX compound;10:11 XX proper;13:13 XX q;12:13 XX compound;14:14 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;17:17 XX compound;17:18 XX udef;15:18 XX proper;12:18 XX udef;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX d;6:7 XX udef;9:9 XX d;11:11 XX d;12:12 XX d', '', '2:2 XX neg;12:12 XX q;12:13 XX udef;34:34 XX comp;40:40 XX q;40:40 XX compound;41:41 XX q;41:41 XX compound;42:42 XX q;42:42 XX compound;43:43 XX compound', '7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;23:23 XX q;0:28 XX compound', '0:0 XX q;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;8:8 XX nominalization;11:11 XX q;11:11 XX nominalization;0:11 XX udef', '8:8 XX q;10:10 XX q;11:11 XX q;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX udef;18:18 XX q;19:19 XX q;21:21 XX q;22:22 XX q;21:23 XX udef', '3:4 XX proper;9:10 XX proper;12:12 XX q;15:15 XX q;14:15 XX compound;13:15 XX compound;12:15 XX udef;11:15 XX udef;16:16 XX q;17:18 XX compound;20:20 XX q;19:20 XX compound;22:22 XX q;23:23 XX q;19:23 XX proper;17:23 XX proper', '0:0 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;14:14 XX compound;12:14 XX compound;0:14 XX focus', '8:8 XX q;9:9 XX q;11:11 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;0:21 XX udef', '0:0 XX q;1:1 XX q;1:1 XX compound;0:5 XX proper', '5:5 XX q;7:7 XX q;8:8 XX q', '10:10 XX q;11:11 XX q', '14:14 XX q;15:15 XX q;16:16 XX times;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;0:9 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;13:13 XX q;14:14 XX q;15:15 XX times;27:27 XX q;28:28 XX q;29:29 XX times;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX compound;0:37 XX udef', '0:0 XX q;1:1 XX q;9:9 XX q;10:10 XX q;13:13 XX q;14:14 XX q;15:15 XX q;0:16 XX udef', '0:0 XX q;1:1 XX q;3:3 XX q;7:7 XX q;8:8 XX q;9:9 XX q;17:17 XX q;0:19 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;6:10 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;12:14 XX udef;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;18:20 XX udef;24:24 XX q;23:24 XX compound;25:25 XX q;26:26 XX q;27:27 XX q;26:28 XX udef;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef;30:33 XX udef;23:33 XX udef;21:33 XX subord;16:33 XX udef;0:33 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:9 XX proper', '1:1 XX q;2:2 XX q;3:3 XX q;6:6 XX q;8:8 XX q;18:18 XX q;19:19 XX q;22:22 XX q;23:23 XX q;25:25 XX q;17:25 XX udef;5:25 XX q;4:25 XX compound;3:25 XX udef', '0:0 XX q;0:5 XX udef', '1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;9:9 XX conj;10:10 XX thing;11:11 XX conj;12:12 XX conj;27:27 XX q;29:29 XX q;29:29 XX udef', '', '0:-1 XX EMPTY-LABEL', '1:1 XX d;3:3 XX q', '5:5 XX comp;13:13 XX q', '4:4 XX q;5:5 XX q', '', '10:10 XX comp', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q', '1:1 XX q;1:1 XX nominalization;2:2 XX d;3:3 XX q;3:3 XX nominalization;4:4 XX d;5:5 XX d;6:6 XX q;6:6 XX nominalization;8:8 XX d;9:9 XX q;10:10 XX q;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;21:21 XX q;21:22 XX udef;25:25 XX q;26:26 XX q;27:27 XX compound;26:27 XX udef;24:27 XX udef;21:27 XX udef;19:27 XX udef;9:27 XX udef', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;21:21 XX q;25:25 XX q;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;31:32 XX proper;33:33 XX q;0:38 XX implicit', '7:7 XX neg', '2:2 XX neg', '6:6 XX q', '', '0:0 XX q;1:2 XX compound;3:3 XX d;4:4 XX d;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX proper;11:12 XX compound;13:13 XX d;6:13 XX udef;15:15 XX d;14:15 XX proper;16:16 XX d;0:16 XX udef', '', '11:10 XX very;11:11 XX XX', '', '1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;14:14 XX conj;13:14 XX conj;17:17 XX comp', '6:6 XX q;16:16 XX neg;23:24 XX compound;27:28 XX compound', '0:19 XX proper', '', '7:7 XX q', '', '', '1:1 XX d;3:3 XX d;5:5 XX q;4:5 XX compound;6:6 XX q;8:8 XX q;9:9 XX d;10:11 XX proper;14:14 XX d;16:16 XX d;18:18 XX compound;0:18 XX focus', '9:9 XX q;11:11 XX q', '', '1:1 XX q;2:2 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;29:29 XX q;32:33 XX compound;0:36 XX loc', '2:2 XX q;14:14 XX q;14:14 XX nominalization;15:16 XX compound;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;28:28 XX q;29:29 XX q;28:29 XX udef;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX compound;33:36 XX udef;31:36 XX udef;13:36 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;0:15 XX focus', '0:0 XX q;0:27 XX udef', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;9:9 XX q;10:10 XX q;9:11 XX udef;12:12 XX nonsp;20:21 XX proper;12:22 XX loc', '14:14 XX q;15:15 XX q;22:22 XX q;21:23 XX udef;24:24 XX q;26:26 XX d;27:27 XX q;28:28 XX q;29:29 XX d;30:30 XX d;31:31 XX q;32:32 XX q;31:32 XX udef;27:32 XX udef;26:32 XX focus;25:32 XX udef;16:32 XX udef', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX proper;8:8 XX q;6:8 XX proper;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX compound;10:13 XX udef;9:13 XX udef', '13:13 XX q;14:14 XX q;16:16 XX q;23:23 XX q;24:24 XX q', '1:1 XX neg;8:8 XX q;8:9 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX d;5:5 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;17:17 XX q;19:19 XX q;20:20 XX q;22:22 XX d;21:22 XX udef;17:22 XX udef;23:23 XX d;24:24 XX d;0:24 XX udef', '21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;28:28 XX q', '0:0 XX q;0:1 XX udef;8:8 XX q;9:9 XX q;0:11 XX compound', '25:25 XX q;26:26 XX q;27:27 XX q;27:27 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;0:18 XX focus', '3:3 XX q;3:4 XX udef;2:4 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX nominalization;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;22:22 XX q;23:23 XX q;24:24 XX q', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;4:6 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;8:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;24:24 XX q;25:25 XX q;24:26 XX udef;27:27 XX q;28:28 XX q;29:29 XX q;28:31 XX udef;24:31 XX udef;32:32 XX nonsp;33:33 XX nonsp;32:33 XX loc', '6:6 XX q;7:7 XX q;8:8 XX times;6:8 XX udef;9:10 XX compound;13:13 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX times;18:20 XX udef;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX compound;24:25 XX udef;24:25 XX udef;18:25 XX compound', '0:0 XX q;1:1 XX q;7:7 XX q;8:8 XX q;0:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX d;11:14 XX udef;7:14 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX d;23:26 XX udef;19:26 XX udef;27:27 XX nonsp;28:28 XX nonsp;27:28 XX loc;17:28 XX udef;15:28 XX subord;0:28 XX focus', '3:3 XX q;4:4 XX q', '3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d', '7:7 XX q', '8:8 XX q', '4:4 XX q;11:11 XX neg;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;22:22 XX q;23:23 XX number', '10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q', '11:12 XX compound;15:15 XX q;14:15 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;23:24 XX udef;27:27 XX generic;14:27 XX udef;10:27 XX compound;0:27 XX proper', '3:3 XX neg;12:12 XX neg', '0:0 XX q;0:0 XX compound;0:0 XX unknown;0:0 XX proper', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;34:35 XX udef;32:35 XX udef;0:35 XX udef', '5:5 XX q;5:5 XX nominalization;7:7 XX q;8:8 XX q;7:8 XX udef;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;15:15 XX udef;13:15 XX udef;10:15 XX udef;6:15 XX compound', '13:13 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX compound;13:21 XX udef;7:21 XX loc;0:21 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX compound;14:15 XX udef;0:15 XX focus', '7:7 XX comp', '0:0 XX neg;12:12 XX q', '14:14 XX q;0:14 XX measure', '3:3 XX superl;17:17 XX q;17:18 XX udef', '0:0 XX nonsp;1:1 XX d;0:1 XX loc;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX compound;0:13 XX focus', '13:13 XX discourse;14:14 XX q;15:15 XX q;15:16 XX udef', '7:7 XX conj;10:10 XX conj;11:11 XX q;13:13 XX q;14:14 XX q;19:19 XX q;20:20 XX q;21:21 XX q;30:30 XX comp', '13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:25 XX udef', '3:3 XX q;5:5 XX q;6:6 XX q;10:10 XX q;13:13 XX q;14:14 XX q;25:25 XX q;35:35 XX q;35:39 XX udef', '12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q', '10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;26:33 XX udef;25:33 XX udef', '0:6 XX proper', '3:3 XX q;10:10 XX q', '9:9 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;13:14 XX udef;12:14 XX udef', '7:8 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:14 XX compound;16:16 XX q;17:17 XX q;13:18 XX compound;19:19 XX q;10:20 XX udef;6:20 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:26 XX focus', '3:4 XX compound;6:7 XX loc;0:19 XX compound', '10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q', '4:4 XX q;5:6 XX compound;12:12 XX q;13:13 XX nonsp;13:13 XX loc;12:13 XX number', '4:4 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX d;0:11 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;23:23 XX d;0:23 XX focus', '3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;7:7 XX udef', '8:8 XX q;8:8 XX compound;7:8 XX compound;9:10 XX compound;19:19 XX q;19:19 XX compound;18:19 XX proper;22:22 XX q;21:22 XX udef;20:22 XX proper;18:22 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;24:24 XX q;23:24 XX udef;8:24 XX udef;0:24 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX neg;0:7 XX udef', '3:3 XX q;2:4 XX udef;5:5 XX q;5:5 XX compound;15:15 XX q;16:16 XX q;19:19 XX q;20:21 XX udef;19:21 XX udef;25:25 XX compound;24:25 XX proper;23:25 XX compound;18:25 XX udef', '0:0 XX neg', '0:7 XX compound', '4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX udef;2:6 XX compound;1:6 XX udef', '2:2 XX nominalization;7:7 XX q;8:8 XX nominalization;9:9 XX q;19:19 XX q;0:20 XX compound', '16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX nominalization;20:21 XX udef;24:24 XX q;26:26 XX q;27:27 XX q;24:27 XX udef', '0:0 XX nominalization;8:8 XX q;10:10 XX q', '0:0 XX neg;7:7 XX q;8:8 XX nominalization;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX udef', '6:6 XX q;14:14 XX q;14:15 XX udef;17:17 XX compound', '7:7 XX q;8:8 XX nominalization;21:21 XX q;22:22 XX q;23:23 XX q', '3:3 XX q;4:4 XX nominalization;6:6 XX q;7:7 XX q;8:8 XX q;19:19 XX neg', '8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q', '8:8 XX q;8:8 XX nominalization;9:9 XX q;10:10 XX q;10:10 XX nominalization;9:11 XX udef', '1:1 XX neg;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX nominalization;8:8 XX q;9:9 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;18:19 XX udef;23:23 XX q;24:24 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX q;10:10 XX q;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;0:21 XX focus', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;19:19 XX q;0:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX d;6:7 XX udef;8:8 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;11:17 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX d;23:24 XX proper;25:25 XX d;26:26 XX d;27:27 XX compound;0:27 XX focus', '0:0 XX q;4:4 XX q;4:4 XX nominalization;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX udef;8:11 XX udef;0:11 XX udef', '5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;18:18 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX compound;17:28 XX udef;10:28 XX udef;8:28 XX udef', '8:8 XX q;9:9 XX nominalization;13:13 XX neg;21:22 XX compound;0:22 XX compound', '2:2 XX d;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:6 XX udef;7:7 XX d;8:8 XX d', '3:3 XX q;5:5 XX q;8:8 XX q;7:9 XX udef;11:12 XX compound;14:15 XX compound;13:16 XX compound;0:16 XX compound', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;0:17 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;9:9 XX q;15:15 XX q;0:16 XX udef', '5:5 XX neg;10:10 XX q;11:11 XX q;12:12 XX q;10:12 XX udef;0:12 XX compound', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;0:15 XX focus', '2:2 XX neg', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;3:4 XX loc;5:6 XX proper;8:8 XX d;12:12 XX d;13:13 XX d;15:15 XX d;14:15 XX udef;16:16 XX d;17:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;5:21 XX appos;0:21 XX focus', '1:1 XX thing', '2:2 XX q', '7:7 XX q;9:9 XX q;10:10 XX q;0:11 XX compound', '7:7 XX q;8:8 XX q', '0:0 XX q;0:18 XX proper', '0:0 XX q;0:18 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;0:23 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:8 XX proper;13:13 XX q;12:13 XX proper;14:14 XX q;12:15 XX udef;17:18 XX udef;17:18 XX compound;0:18 XX focus', '0:0 XX q;0:6 XX proper', '3:3 XX q;8:8 XX q', '', '1:1 XX q;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX proper', '6:6 XX q;7:8 XX udef', '0:0 XX q;1:1 XX q;11:11 XX d;0:21 XX proper', '', '2:2 XX q;6:7 XX udef;9:9 XX q;9:9 XX compound;11:11 XX compound;9:11 XX udef;6:11 XX udef', '', '0:0 XX neg', '', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;0:26 XX subord', '2:2 XX q;3:3 XX q;4:4 XX d;7:8 XX proper;9:9 XX d;7:9 XX compound;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX compound;5:17 XX appos;3:17 XX udef;0:17 XX focus', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;0:21 XX focus', '0:0 XX q;1:1 XX q;1:1 XX compound;5:5 XX q;6:6 XX q;7:7 XX q;0:8 XX udef', '', '', '10:10 XX neg', '0:0 XX d;1:1 XX d;2:3 XX proper;6:7 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;5:15 XX compound;0:15 XX focus', '6:6 XX q', '8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX compound;11:12 XX udef', '12:13 XX compound;0:13 XX compound', '19:19 XX q;20:20 XX q;22:22 XX q', '3:3 XX q;6:6 XX nonsp;6:7 XX loc;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;16:22 XX udef', '', '2:2 XX q;1:2 XX compound;3:3 XX q;5:5 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d', '5:5 XX comp;15:15 XX comp;0:22 XX compound', '9:9 XX q;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization', '3:3 XX q', '0:0 XX q;1:2 XX compound;0:4 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX proper;6:8 XX proper;0:22 XX appos', '2:2 XX q;3:3 XX q;5:5 XX q;4:5 XX compound;4:6 XX proper;11:11 XX q;10:12 XX udef;15:15 XX neg;0:20 XX appos', '0:0 XX d;1:1 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;0:12 XX focus', '', '0:23 XX compound', '10:10 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;24:24 XX q;29:30 XX proper;29:31 XX compound;33:33 XX compound', '', '1:1 XX d;2:2 XX q;4:4 XX d', '14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;18:18 XX q;20:20 XX compound;17:20 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;4:5 XX compound;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;18:18 XX d;20:20 XX d;21:21 XX d;0:22 XX focus', '5:5 XX q;6:6 XX q;5:9 XX udef', '8:9 XX compound', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;0:18 XX focus', '5:5 XX comp;0:21 XX compound', '', '1:1 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;15:15 XX conj;16:16 XX q;17:17 XX times;14:17 XX generic;18:18 XX conj;23:24 XX compound;26:26 XX q;26:27 XX udef;30:30 XX q;31:31 XX q;32:32 XX q;29:32 XX proper', '1:1 XX neg', '5:5 XX comp', '4:4 XX q;7:7 XX q;8:8 XX q;14:14 XX q;13:14 XX udef;12:14 XX udef;11:14 XX udef', '', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX conj;22:22 XX q;23:23 XX q;24:24 XX q;26:27 XX compound;25:28 XX proper', '2:2 XX neg', '9:9 XX nonsp;15:15 XX q;17:17 XX q;18:18 XX q', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q', '9:9 XX person;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization', '2:2 XX neg;9:9 XX q;10:10 XX q;10:11 XX udef', '0:19 XX proper', '5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q', '18:18 XX q;19:19 XX q;0:20 XX proper', '', '', '1:1 XX q;2:2 XX q;4:4 XX q;6:6 XX q;5:6 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;13:13 XX q;12:14 XX udef;24:24 XX q;25:25 XX q;26:26 XX times;24:26 XX udef;24:27 XX compound;29:30 XX proper;8:36 XX udef', '', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d', '1:1 XX d;0:17 XX focus', '11:11 XX q;12:12 XX q', '8:8 XX q', '', '', '6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;16:16 XX nominalization;17:17 XX q;18:18 XX q;19:19 XX q;19:19 XX nominalization;20:20 XX q;17:20 XX udef;16:20 XX udef;12:20 XX udef;10:20 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q', '9:9 XX neg;32:32 XX q;39:39 XX q', '', '', '4:4 XX q;4:4 XX nominalization', '1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX compound;0:11 XX focus', '2:2 XX q;3:3 XX q;2:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX nonsp;9:10 XX loc;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;12:16 XX udef;6:16 XX udef', '1:1 XX neg;10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;11:12 XX udef;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;16:16 XX q;17:18 XX proper;17:19 XX compound;15:19 XX udef;28:31 XX compound;21:31 XX loc;20:31 XX loc', '7:8 XX compound;9:10 XX loc;0:15 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;0:6 XX udef', '2:2 XX nonsp;11:11 XX q;11:11 XX compound;10:11 XX compound;12:12 XX compound;2:12 XX loc', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:12 XX compound;0:13 XX focus', '1:1 XX d;0:1 XX loc;3:3 XX d;2:3 XX proper;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;16:16 XX d;14:16 XX compound;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX q;20:21 XX udef;22:22 XX q;23:23 XX q;23:23 XX udef;0:23 XX focus', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;11:14 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '', '0:0 XX q;0:0 XX nominalization;1:1 XX q;2:2 XX q;3:3 XX q;6:6 XX q;5:6 XX compound;7:7 XX q;9:9 XX q;11:12 XX compound;10:12 XX udef;8:16 XX udef;0:16 XX udef', '15:15 XX q;16:16 XX q;17:17 XX q', '', '0:9 XX appos', '1:1 XX q;1:2 XX udef;0:2 XX udef;15:15 XX q;0:15 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:24 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX q;18:18 XX q;20:20 XX q;22:22 XX q;22:22 XX compound;21:22 XX compound;24:24 XX q;24:24 XX compound;23:24 XX proper;23:24 XX proper;20:24 XX udef;19:24 XX id;17:24 XX appos;0:24 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;7:8 XX udef;9:9 XX d;11:11 XX d;10:11 XX proper;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX q;31:31 XX q;30:31 XX udef;17:31 XX udef;0:31 XX focus', '2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;19:19 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;15:25 XX udef;0:25 XX focus', '0:0 XX neg', '27:27 XX q;29:29 XX q;34:34 XX q;35:36 XX compound;39:39 XX proper', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;14:14 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;21:21 XX conj;22:22 XX conj;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX neg', '3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX q;22:22 XX d;25:25 XX q;24:25 XX compound;26:26 XX d;27:27 XX proper;23:27 XX proper;0:27 XX subord', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;12:12 XX d;15:15 XX q;0:19 XX focus', '', '', '0:0 XX d;2:2 XX d;3:3 XX q;3:3 XX nominalization;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;15:15 XX d;0:17 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d', '', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;8:8 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;20:20 XX superl;23:23 XX d;24:24 XX d;25:25 XX d;28:28 XX d;29:29 XX d;30:30 XX d;32:32 XX d;33:33 XX q;33:33 XX nominalization;34:34 XX q;34:34 XX nominalization;36:36 XX q;36:36 XX nominalization;0:36 XX proper', '', '5:5 XX q;6:6 XX q;7:7 XX times;5:7 XX udef;8:8 XX q;8:8 XX compound;14:14 XX q;15:15 XX q;16:16 XX times;0:24 XX proper', '2:2 XX q', '1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;10:10 XX nominalization;11:11 XX q;12:12 XX q;16:16 XX d;14:16 XX compound;13:16 XX appos;17:24 XX unknown;11:24 XX udef;0:24 XX loc', '7:7 XX q;8:8 XX q;9:9 XX times;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;14:14 XX q;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization', '11:11 XX superl;15:15 XX q;17:17 XX neg;20:20 XX q;21:21 XX q;22:22 XX q', '2:2 XX q;4:4 XX q;5:5 XX q;1:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;16:22 XX udef;0:22 XX compound', '10:10 XX q;12:12 XX q;18:18 XX q', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;17:19 XX compound;21:21 XX neg', '1:1 XX d;3:3 XX d;4:4 XX d;7:7 XX d;8:8 XX d;9:10 XX proper;14:14 XX q;15:15 XX q;15:15 XX compound;15:16 XX udef;14:16 XX proper;18:19 XX proper;18:19 XX compound;14:19 XX appos;0:19 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;11:11 XX q;12:12 XX q;13:13 XX times', '8:8 XX q;7:8 XX proper;9:9 XX q;10:11 XX proper;15:15 XX q;16:16 XX q;17:17 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;35:35 XX q;35:35 XX nominalization', '4:4 XX q;18:18 XX q;18:19 XX proper;21:21 XX q;22:22 XX q;22:22 XX compound;21:22 XX udef;23:23 XX compound;0:23 XX appos', '0:23 XX proper', '17:17 XX q;18:19 XX proper;22:22 XX proper;18:22 XX compound;16:22 XX udef', '0:0 XX d;2:2 XX d;4:4 XX d;6:6 XX d;5:6 XX compound;8:8 XX d;7:8 XX proper;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX udef;11:14 XX udef;0:14 XX focus', '0:6 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;7:8 XX compound;16:16 XX q;16:16 XX udef;3:16 XX udef;0:16 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX udef', '1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:14 XX proper;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;15:19 XX compound;0:19 XX proper', '3:3 XX q', '0:0 XX d;1:1 XX d;3:3 XX d;0:4 XX focus', '3:3 XX comp;20:20 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;11:11 XX q;10:11 XX udef;0:11 XX proper', '0:0 XX d;1:1 XX d;3:3 XX d;5:5 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;11:12 XX proper;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;10:19 XX udef;0:19 XX focus', '6:6 XX q;5:6 XX proper;8:8 XX q;15:17 XX udef;23:23 XX compound;22:23 XX proper;0:23 XX appos', '', '3:3 XX q;3:3 XX nominalization;5:5 XX q;5:5 XX nominalization;14:14 XX q;14:14 XX named;14:15 XX proper;18:18 XX compound', '', '20:20 XX q', '7:7 XX q;8:8 XX q;9:9 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX q;11:11 XX q;12:12 XX conj;13:13 XX d;15:15 XX q;16:16 XX q;17:17 XX times;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:25 XX focus', '3:3 XX q;4:4 XX q;4:4 XX compound;3:4 XX udef;6:7 XX compound;5:7 XX proper;14:14 XX neg', '0:0 XX q;2:2 XX d;3:3 XX q;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX q;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:26 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:7 XX proper;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;21:21 XX neg;0:34 XX focus', '2:2 XX q;28:28 XX q', '', '16:16 XX neg;0:23 XX compound', '1:1 XX d;2:2 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:12 XX proper;17:18 XX compound;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX d;30:30 XX d;31:31 XX q;33:33 XX d;34:34 XX d;36:36 XX d;37:37 XX d;38:38 XX q;39:39 XX q;42:42 XX q;43:43 XX q;45:45 XX q;47:47 XX q;48:48 XX q;42:48 XX udef;40:48 XX generic;38:48 XX udef;0:48 XX focus', '3:3 XX q', '8:8 XX q', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX q;3:3 XX pron;4:4 XX d;5:5 XX d;7:7 XX d;0:7 XX udef', '4:4 XX q;5:6 XX compound;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;22:22 XX q;22:22 XX nominalization', '3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX q;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX q;25:25 XX q;26:26 XX generic;24:26 XX udef', '9:9 XX q', '', '8:8 XX q;9:10 XX compound;12:12 XX q;15:15 XX q;18:18 XX q', '1:1 XX q;2:2 XX q;3:3 XX q', '0:0 XX q;1:1 XX q;2:2 XX times;0:2 XX udef;11:12 XX compound;25:25 XX q;26:26 XX q;25:27 XX udef;0:27 XX compound', '5:5 XX q;0:17 XX appos', '', '0:0 XX d;1:1 XX d;3:3 XX q;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;0:22 XX focus', '', '1:1 XX neg;10:10 XX q;11:11 XX q;12:12 XX times', '', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization', '0:0 XX q;0:0 XX compound;11:11 XX q;11:11 XX compound;0:18 XX proper', '1:1 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;0:13 XX focus', '12:12 XX q;12:12 XX nominalization;14:14 XX q', '0:0 XX q;27:27 XX superl;0:34 XX proper', '0:0 XX q;1:1 XX q;3:3 XX neg;10:10 XX comp;12:12 XX q;0:15 XX udef', '5:6 XX compound;7:7 XX q;8:11 XX generic;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX proper;19:19 XX q;15:20 XX udef', '3:3 XX q;3:3 XX nominalization;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;5:10 XX udef', '3:3 XX neg', '0:0 XX q;4:4 XX q;5:5 XX q;0:11 XX udef', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;7:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX q;17:17 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;16:22 XX udef;0:22 XX focus', '1:1 XX neg;4:4 XX neg;10:10 XX neg', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX d;22:22 XX d;21:22 XX proper;23:23 XX d;24:24 XX d;25:25 XX d;27:27 XX d;28:28 XX q;0:28 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX person;22:22 XX q;0:23 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX person;22:22 XX q;0:23 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:28 XX focus', '2:2 XX q;5:5 XX q;6:6 XX q;5:6 XX udef', '', '7:7 XX q', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX nonsp;13:13 XX d;0:14 XX focus', '', '', '2:2 XX q;2:2 XX compound;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;0:16 XX appos', '11:11 XX q;11:12 XX udef', '1:1 XX q;3:3 XX q;6:6 XX neg', '8:8 XX q;8:8 XX named;10:10 XX q', '4:4 XX neg', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:13 XX focus', '', '0:0 XX d;1:1 XX d;3:3 XX d;2:3 XX compound;5:5 XX d;4:5 XX proper;7:7 XX d;8:8 XX d;0:8 XX focus', '23:23 XX q;24:24 XX q;26:26 XX q;27:27 XX q;28:28 XX compound;20:28 XX generic', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:23 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;9:9 XX conj;14:14 XX q;0:14 XX udef', '', '', '14:14 XX superl', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;0:35 XX focus', '0:0 XX q;0:0 XX compound;8:8 XX q;9:9 XX q;17:17 XX q;18:18 XX q;19:19 XX q;33:33 XX q;34:34 XX q;35:35 XX q;0:38 XX proper', '', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;0:13 XX focus', '8:8 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:15 XX compound;13:15 XX udef;18:18 XX neg;26:26 XX q;26:26 XX nominalization;28:28 XX q;28:28 XX nominalization;11:28 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;0:20 XX focus', '1:1 XX q;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;11:11 XX q;11:12 XX udef', '14:14 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX compound;18:19 XX compound', '3:3 XX q;2:3 XX proper;4:4 XX q;6:6 XX d;2:6 XX udef;7:7 XX d;8:8 XX d;0:8 XX focus', '2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;5:5 XX q;5:5 XX compound;7:7 XX q;8:9 XX proper;5:9 XX udef;12:12 XX neg;19:19 XX q;19:19 XX compound;18:19 XX proper;20:21 XX compound;18:24 XX proper;4:24 XX proper;0:24 XX appos', '10:10 XX discourse;14:14 XX neg', '', '1:1 XX d;0:1 XX proper;2:2 XX q;4:4 XX d;3:4 XX udef;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:8 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX d;10:10 XX d;12:12 XX q;12:12 XX compound;0:18 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d', '', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;17:19 XX udef;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX d;28:28 XX d;29:29 XX d;14:29 XX proper;0:29 XX focus', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;0:6 XX focus', '', '1:1 XX q;3:3 XX q;4:4 XX udef', '3:3 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;0:12 XX udef', '16:16 XX q;17:17 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;0:5 XX udef', '16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX compound', '0:0 XX q;0:7 XX udef', '2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;19:20 XX udef;17:20 XX udef;0:20 XX focus', '11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q', '11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q', '2:2 XX q;2:2 XX compound;15:15 XX q;21:21 XX q;22:22 XX q', '15:15 XX q;14:15 XX udef', '0:0 XX q;0:0 XX compound;5:5 XX q;18:18 XX q;0:26 XX proper', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;11:13 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;19:19 XX udef;13:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX q;1:2 XX proper;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX compound;0:12 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;0:7 XX udef', '', '1:1 XX d;4:4 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;0:17 XX focus', '', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:14 XX d;13:14 XX compound;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;0:21 XX proper', '0:0 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX compound;0:19 XX proper', '0:1 XX udef;5:5 XX q;7:7 XX q;8:8 XX q;0:8 XX compound', '3:3 XX q;2:3 XX compound;4:4 XX q;4:4 XX compound;4:5 XX udef;13:14 XX compound;13:14 XX compound;12:14 XX compound', '10:10 XX q;13:13 XX q', '4:4 XX q;7:7 XX q', '10:10 XX q;11:11 XX q;14:14 XX nonsp;13:14 XX loc;0:14 XX compound', '6:6 XX q;9:9 XX q;0:15 XX compound', '6:6 XX q;7:7 XX q;9:9 XX q;9:10 XX udef;0:10 XX compound', '6:6 XX q;7:7 XX q;10:10 XX nonsp', '0:0 XX q;28:28 XX q;30:30 XX q;33:33 XX q;32:33 XX q;0:33 XX proper', '15:15 XX q;16:16 XX q', '4:4 XX q;5:5 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;18:22 XX udef;11:22 XX udef;0:22 XX focus', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;9:9 XX compound;8:9 XX udef;12:12 XX udef;11:12 XX udef;10:12 XX udef;6:12 XX compound;6:12 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;0:19 XX udef', '6:6 XX q;8:9 XX compound;7:10 XX compound;5:10 XX number', '0:1 XX proper;2:2 XX q;3:3 XX q;6:6 XX q;6:6 XX compound;5:6 XX proper;5:7 XX proper;5:8 XX compound;13:13 XX q;14:14 XX q;15:15 XX q;18:18 XX q;17:19 XX udef;17:22 XX compound;30:30 XX q;31:31 XX q;0:31 XX compound', '2:2 XX q;3:4 XX compound;7:7 XX q;8:8 XX q;9:9 XX times;7:9 XX udef;14:14 XX q;15:15 XX q;14:15 XX udef', '3:3 XX q;3:3 XX compound;7:7 XX q;8:8 XX q;10:10 XX q;0:10 XX appos', '0:0 XX q;0:0 XX compound;5:5 XX q;7:8 XX compound;13:13 XX q;14:14 XX q;15:15 XX q;19:19 XX q;19:19 XX udef;17:19 XX udef;0:19 XX proper', '2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;3:4 XX measure;3:4 XX udef;5:5 XX d;1:5 XX proper;6:6 XX d;8:8 XX d;9:10 XX proper;9:11 XX compound;9:12 XX compound;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;7:17 XX appos;0:17 XX compound', '3:3 XX q;5:5 XX compound;4:5 XX compound', '15:15 XX q', '1:1 XX q;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX times;19:19 XX q;20:20 XX q;22:22 XX q;23:23 XX q;0:23 XX compound', '4:4 XX q;5:5 XX q;6:6 XX times', '0:0 XX q;6:6 XX q;10:10 XX q;11:11 XX q;12:12 XX times;0:18 XX proper', '0:0 XX q;3:3 XX q;3:3 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;29:29 XX nonsp;30:30 XX nonsp;0:30 XX proper', '0:0 XX superl;10:10 XX q;11:11 XX q;10:11 XX udef', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '10:10 XX q', '0:1 XX number;5:5 XX q;7:7 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:23 XX compound', '1:1 XX q;0:1 XX compound;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:26 XX proper', '0:0 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;11:13 XX udef;0:13 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:15 XX proper;17:17 XX q;16:17 XX compound;18:18 XX q;20:20 XX q;19:20 XX proper;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;31:33 XX udef;22:33 XX proper;19:33 XX udef;16:33 XX udef;14:33 XX appos;0:33 XX focus', '2:3 XX number;4:5 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;24:24 XX udef;21:24 XX udef;14:24 XX udef;11:24 XX udef', '1:2 XX compound;5:6 XX loc;8:8 XX q;12:12 XX q;13:13 XX q;12:14 XX udef;8:14 XX udef;0:14 XX compound', '', '1:1 XX q;1:1 XX compound;0:1 XX proper;0:2 XX proper;4:4 XX q;6:7 XX compound;5:7 XX compound;4:7 XX udef;3:7 XX udef;0:15 XX appos', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX udef', '4:4 XX d;5:5 XX d;6:6 XX superl;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX q;14:14 XX q;13:14 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:20 XX compound;19:20 XX compound;13:20 XX udef;11:20 XX udef;10:20 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX q;0:14 XX udef', '', '3:3 XX q;3:3 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '10:11 XX proper;15:16 XX compound;18:18 XX q;17:23 XX subord', '5:5 XX neg', '0:0 XX q;0:9 XX udef', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX q;13:14 XX compound;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX times;21:23 XX udef;24:24 XX compound;21:24 XX compound;16:24 XX proper;0:24 XX compound', '13:13 XX q;13:14 XX udef;16:16 XX q;16:17 XX udef;16:18 XX compound', '10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;16:16 XX q;18:18 XX conj;15:18 XX appos;23:23 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;35:35 XX q;34:35 XX udef;31:35 XX udef', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;2:8 XX udef;9:9 XX q;9:9 XX nominalization;12:12 XX d;13:13 XX d;15:15 XX d;14:15 XX proper;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;0:23 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;9:9 XX d;8:9 XX compound;10:10 XX d;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX d;19:19 XX q;20:20 XX q;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;25:25 XX q;21:25 XX udef;11:25 XX compound;0:25 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:14 XX proper;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;15:19 XX udef;22:22 XX q;22:22 XX compound;21:22 XX proper;21:23 XX proper;24:24 XX q;25:25 XX q;0:25 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;6:6 XX d;4:6 XX number;7:7 XX d;8:9 XX generic;0:9 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;5:5 XX q;6:6 XX q;8:8 XX q;8:8 XX compound;7:8 XX proper;7:9 XX proper;5:9 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;18:27 XX udef;0:27 XX focus', '', '', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;8:8 XX compound;7:8 XX proper;7:9 XX proper;12:12 XX compound;11:12 XX compound;0:12 XX focus', '', '9:9 XX q;19:19 XX q;21:21 XX q;23:23 XX q;22:23 XX udef', '2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX d;10:10 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;19:19 XX q;18:19 XX compound;21:21 XX q;22:23 XX proper;26:26 XX q;27:27 XX q;26:27 XX udef;30:30 XX q;31:31 XX q;32:32 XX q;42:42 XX q;43:43 XX compound;42:43 XX udef;37:43 XX eventuality;32:43 XX proper;25:43 XX id;22:43 XX appos;17:43 XX udef;16:43 XX udef;0:43 XX focus', '', '12:12 XX q;0:12 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX d;18:18 XX d;19:19 XX d;14:20 XX udef;0:20 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;11:12 XX udef;13:13 XX neg;18:18 XX udef;5:18 XX udef;0:18 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX q;6:6 XX compound;5:6 XX proper;8:8 XX q;8:8 XX compound;7:8 XX compound;10:10 XX q;11:11 XX q;13:13 XX q;13:13 XX compound;0:13 XX focus', '3:3 XX q;5:5 XX q;4:5 XX compound;6:6 XX q;19:19 XX q;20:20 XX q;23:25 XX udef;26:26 XX q;27:28 XX compound;32:32 XX q;32:32 XX udef;22:32 XX udef', '', '', '0:4 XX compound', '1:1 XX q;1:1 XX compound;0:1 XX proper;2:2 XX q;2:2 XX compound;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;16:16 XX d;17:17 XX q;0:17 XX proper', '16:16 XX q;17:17 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX q;5:6 XX compound;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;9:10 XX proper;9:11 XX proper;12:12 XX q;9:12 XX compound;13:13 XX q;14:14 XX proper;0:14 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;0:5 XX focus', '0:0 XX neg', '12:12 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:13 XX proper;15:15 XX q;14:15 XX compound;16:16 XX q;17:17 XX proper;14:17 XX udef;0:17 XX focus', '7:7 XX q;11:11 XX q;11:12 XX udef;10:12 XX udef;10:13 XX compound;16:17 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;29:30 XX udef;22:30 XX udef', '', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;10:10 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;15:15 XX compound;14:15 XX udef;14:15 XX udef;5:15 XX udef', '7:7 XX q', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:17 XX focus', '6:6 XX q', '0:0 XX q;1:1 XX q;3:3 XX d;2:3 XX compound;4:4 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;15:16 XX udef;25:25 XX q;0:25 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;20:20 XX q;19:20 XX udef;22:24 XX udef;21:24 XX udef;19:24 XX udef;18:24 XX compound;17:24 XX udef;16:24 XX udef;0:24 XX focus', '0:0 XX q;6:6 XX q;0:14 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;13:14 XX proper;15:16 XX compound;18:19 XX proper;20:20 XX q;18:20 XX compound;21:21 XX q;21:21 XX compound;21:21 XX udef;18:21 XX proper;13:21 XX compound;0:21 XX focus', '0:0 XX neg;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;23:23 XX q;23:23 XX udef;19:23 XX udef', '0:0 XX nonsp;1:1 XX d;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;22:22 XX d;23:23 XX d;25:25 XX d;0:25 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;0:4 XX focus', '0:0 XX d;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX proper;0:18 XX focus', '', '8:8 XX person', '', '1:1 XX q;1:1 XX compound;0:1 XX proper;2:2 XX q;2:2 XX compound;0:2 XX proper;3:4 XX proper;5:5 XX q;8:8 XX d;7:8 XX compound;6:8 XX compound;9:9 XX d;10:10 XX d;12:12 XX d;11:12 XX udef;13:13 XX d;14:14 XX d;17:17 XX d;16:17 XX compound;18:18 XX d;5:20 XX udef;3:20 XX appos;0:20 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:11 XX focus', '', '', '10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;14:14 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX number;21:21 XX compound;19:21 XX udef;14:21 XX udef;12:21 XX proper', '4:4 XX q;14:14 XX q;15:15 XX compound;0:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;5:5 XX d;6:6 XX q;7:7 XX d;6:7 XX udef;8:8 XX d;10:10 XX compound;4:10 XX subord;0:10 XX focus', '8:8 XX comp', '1:1 XX comp', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX q;17:18 XX compound;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX q;18:18 XX d;21:21 XX d;22:22 XX d;24:24 XX q;24:24 XX compound;23:24 XX proper;23:25 XX proper;27:27 XX q;30:30 XX q;29:30 XX compound;28:30 XX compound;27:30 XX udef;26:30 XX udef;31:31 XX q;33:33 XX q;32:33 XX compound;32:34 XX proper;36:36 XX compound;32:36 XX appos;0:36 XX focus', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX nominalization;11:11 XX comp;6:16 XX udef', '23:23 XX q;24:24 XX q;28:28 XX q;30:30 XX q;31:31 XX q;33:33 XX q;34:34 XX q;36:36 XX q;35:37 XX udef;38:38 XX q;31:38 XX udef;39:39 XX q;28:39 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;16:17 XX proper;18:18 XX q;19:19 XX q;20:20 XX q;20:21 XX proper;23:24 XX compound;23:25 XX compound;28:28 XX udef;20:28 XX appos;18:28 XX udef;0:28 XX focus', '5:5 XX d;6:6 XX d', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;22:22 XX comp', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;22:22 XX comp', '7:7 XX q;6:7 XX compound;8:8 XX q;10:10 XX q;9:10 XX compound;9:11 XX proper;15:15 XX q;13:15 XX compound;16:16 XX q;19:19 XX q;20:20 XX q;18:20 XX compound;17:20 XX udef', '1:1 XX q;1:1 XX compound;0:1 XX proper;12:12 XX q;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;0:24 XX proper', '6:6 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper;21:22 XX compound;19:22 XX udef;25:25 XX q;25:25 XX udef', '7:8 XX proper', '2:3 XX proper;0:20 XX appos', '', '0:0 XX d;1:1 XX d;2:2 XX q;2:2 XX nominalization;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;9:9 XX compound;8:9 XX proper;8:10 XX proper;15:15 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;8:23 XX appos;0:23 XX focus', '0:0 XX q;1:1 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;13:13 XX compound;14:14 XX q;14:14 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;0:24 XX udef', '1:1 XX q;1:2 XX udef;1:3 XX compound;16:16 XX q;17:17 XX q;0:18 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX d;6:7 XX udef;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;29:29 XX compound;28:29 XX compound;0:29 XX udef', '15:15 XX q;15:15 XX compound;19:19 XX q;19:19 XX nominalization;20:20 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;30:30 XX q;31:31 XX q', '2:2 XX q;12:12 XX q', '6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX times;15:17 XX udef;12:17 XX udef;21:21 XX neg;10:27 XX udef;0:27 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX q;19:19 XX q;20:20 XX q;21:22 XX compound;21:22 XX compound;0:22 XX focus', '', '6:6 XX q;8:8 XX neg;11:12 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;23:23 XX q;23:23 XX nominalization;25:25 XX q;26:26 XX q;26:26 XX nominalization;28:28 XX compound;27:28 XX compound;24:28 XX udef;23:28 XX udef', '19:19 XX q;17:19 XX compound;20:20 XX q;17:20 XX udef;21:21 XX q;23:24 XX udef;22:24 XX compound;21:24 XX udef', '', '5:5 XX q;6:6 XX q;7:7 XX times;11:11 XX q;12:12 XX q;13:13 XX times;25:25 XX q;26:26 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX d;0:10 XX focus', '8:8 XX q;9:9 XX compound;0:9 XX compound', '0:0 XX q;0:0 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;12:12 XX compound;13:13 XX q;13:13 XX compound;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX proper', '1:1 XX q;1:2 XX udef;0:2 XX udef;3:3 XX q;3:4 XX udef;3:5 XX compound;18:18 XX q;19:19 XX q;0:20 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX d;6:7 XX udef;3:7 XX udef;8:8 XX d;10:10 XX d;9:10 XX compound;11:11 XX d;13:13 XX d;14:14 XX d;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX udef;21:21 XX q;22:22 XX q;25:25 XX q;26:26 XX q;27:27 XX q;27:27 XX udef;24:27 XX compound;23:27 XX udef;22:27 XX udef;0:27 XX udef', '8:8 XX q;13:13 XX q;14:15 XX compound;13:16 XX udef;21:21 XX q;22:22 XX times;24:25 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;7:8 XX compound;5:8 XX compound;19:19 XX q;22:22 XX q;26:26 XX superl;38:39 XX udef;38:39 XX compound;34:39 XX compound;23:39 XX appos;0:39 XX proper', '0:0 XX q;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;23:23 XX q;0:25 XX udef', '1:1 XX q;2:2 XX q', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q', '0:0 XX q;0:0 XX compound;4:4 XX q;5:6 XX compound;10:10 XX neg;14:14 XX q;23:23 XX nominalization;0:28 XX proper', '8:8 XX q;7:9 XX udef;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;20:20 XX compound;22:22 XX q;24:24 XX q;20:24 XX udef;25:25 XX q', '0:0 XX q;1:1 XX q;1:1 XX compound;7:7 XX q;11:11 XX q;17:17 XX q;16:17 XX compound;23:23 XX q;0:24 XX proper', '0:0 XX neg', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;16:16 XX q;0:16 XX udef', '', '0:0 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;28:28 XX q;29:29 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;0:35 XX proper', '0:0 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;28:28 XX q;29:29 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;0:35 XX proper', '9:11 XX proper;0:14 XX compound', '8:9 XX proper;8:10 XX compound;11:11 XX compound', '0:0 XX nonsp;3:3 XX d;4:4 XX q;8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX d;14:17 XX udef;10:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;21:22 XX udef;19:22 XX udef;0:22 XX loc', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;24:24 XX q;24:25 XX udef;23:25 XX udef;26:26 XX d;23:26 XX compound;22:26 XX proper;0:26 XX compound', '0:0 XX d;11:11 XX nonsp;12:12 XX nonsp;0:12 XX focus', '2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:8 XX loc;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;11:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;17:20 XX udef;11:20 XX udef;9:20 XX subord', '11:11 XX q;12:12 XX q;12:12 XX nominalization;15:15 XX q;15:15 XX nominalization;14:15 XX compound;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;11:22 XX udef', '0:0 XX q;0:0 XX compound;0:10 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;16:16 XX q;0:17 XX proper', '4:4 XX q;7:7 XX q;5:7 XX compound;8:8 XX q;9:10 XX proper;11:11 XX q;9:11 XX compound;9:12 XX proper;14:15 XX compound', '0:0 XX q;1:1 XX q;4:5 XX udef;12:12 XX q;13:13 XX q;13:13 XX compound;12:13 XX udef;14:14 XX q;14:14 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX udef', '0:0 XX neg', '4:4 XX q;6:6 XX q;8:8 XX compound;7:8 XX udef;5:8 XX udef', '0:0 XX q;0:0 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;0:10 XX udef', '0:0 XX d;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX d;5:6 XX proper;3:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX q;11:11 XX udef;10:11 XX id;0:11 XX focus', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;12:12 XX q;10:12 XX measure;13:13 XX q', '6:7 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;10:12 XX udef;13:13 XX q;18:18 XX q;19:19 XX q;14:20 XX generic;9:20 XX udef;8:20 XX compound', '', '0:4 XX compound', '3:3 XX q;6:6 XX q;8:8 XX q;9:9 XX q;9:10 XX proper;5:10 XX udef;13:13 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q', '', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;0:14 XX focus', '1:1 XX q;0:1 XX compound;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;9:10 XX compound;19:19 XX q;20:20 XX q;21:21 XX times;19:21 XX udef;25:25 XX q;25:25 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;29:30 XX compound;31:31 XX q;31:31 XX nominalization;33:33 XX q;33:33 XX nominalization;32:35 XX appos;40:41 XX proper;0:41 XX proper', '5:5 XX q;3:5 XX compound;6:6 XX q;8:8 XX q;9:9 XX q;7:9 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;3:5 XX proper;6:6 XX d;7:8 XX proper;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX q;15:15 XX d;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;27:27 XX q;27:27 XX nominalization;16:27 XX udef;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;35:35 XX q;36:36 XX q;38:38 XX q;38:38 XX nominalization;33:38 XX udef;39:39 XX q;39:39 XX nominalization;40:40 XX generic;0:40 XX focus', '18:18 XX q;19:20 XX proper;17:20 XX udef;24:24 XX q;26:26 XX q;27:27 XX q;24:29 XX udef', '0:0 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX q;17:17 XX q;19:19 XX q;19:19 XX compound;18:19 XX proper;18:20 XX proper;21:21 XX q;23:23 XX d;26:26 XX q;26:26 XX compound;28:28 XX d;0:28 XX focus', '12:12 XX q', '10:10 XX q;21:21 XX q;21:21 XX compound', '4:4 XX q;4:5 XX udef;21:21 XX q', '0:6 XX compound', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;11:11 XX q;12:12 XX q;13:13 XX q;13:13 XX nominalization;14:14 XX q;15:15 XX q;16:16 XX q;16:17 XX udef;14:17 XX udef;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX udef;12:23 XX udef;10:23 XX udef', '0:0 XX q;0:0 XX compound;1:1 XX q;1:1 XX compound;2:2 XX q;2:2 XX compound;13:13 XX q;15:15 XX q;17:17 XX q;30:30 XX q;31:31 XX q;0:32 XX proper', '1:1 XX q', '', '8:8 XX q;9:9 XX q;10:10 XX q;14:13 XX Business;14:14 XX XX;18:18 XX q;20:20 XX q;22:22 XX q;23:23 XX q;30:30 XX q;30:30 XX udef;0:30 XX compound', '10:10 XX q', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;17:17 XX q', '0:19 XX appos', '17:17 XX q;19:19 XX q;20:20 XX q;23:23 XX q;24:24 XX q;25:25 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX q;21:21 XX compound;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;29:29 XX d;30:30 XX q;31:31 XX d;33:33 XX d;0:33 XX focus', '5:5 XX q;5:5 XX nominalization;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;11:11 XX compound;10:11 XX udef;13:13 XX q;12:13 XX compound;10:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;16:16 XX udef;8:16 XX udef;6:16 XX udef', '3:3 XX q;4:4 XX q;6:6 XX q;10:10 XX q;11:13 XX compound;10:13 XX udef;20:20 XX q;9:20 XX udef', '5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX d;15:16 XX proper;13:16 XX udef;17:17 XX d;12:20 XX udef;0:20 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;13:13 XX d;14:14 XX q;15:15 XX q;16:16 XX d;18:18 XX d;19:19 XX d;20:21 XX proper;22:22 XX d;20:22 XX compound;0:22 XX focus', '7:8 XX compound;0:13 XX compound', '1:1 XX q;2:2 XX q;3:3 XX q;6:6 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;25:25 XX q;26:26 XX q;31:31 XX q;31:31 XX compound;32:32 XX q;32:32 XX compound;35:35 XX q;0:35 XX compound', '1:1 XX q;7:7 XX q;8:8 XX q;11:11 XX q;12:12 XX q;0:13 XX compound', '5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;4:6 XX proper;7:8 XX proper;10:10 XX q', '', '2:2 XX q;3:3 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:8 XX udef;7:8 XX compound', '1:1 XX q;0:1 XX compound;2:2 XX q;2:2 XX compound;2:3 XX udef;0:3 XX proper;6:8 XX compound;5:8 XX udef', '2:2 XX d;4:4 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;12:12 XX compound;11:12 XX proper;13:13 XX d;14:14 XX d;15:15 XX d;0:15 XX focus', '9:9 XX nonsp;0:15 XX compound', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;8:8 XX nominalization;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;11:12 XX udef;8:12 XX udef;6:12 XX generic', '0:0 XX q;0:0 XX compound;10:10 XX nonsp;13:13 XX q;0:15 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;0:33 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX d;10:11 XX udef;12:12 XX d;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX proper;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;21:21 XX q;23:23 XX q;22:23 XX compound;22:23 XX proper;16:23 XX udef;13:23 XX appos;0:23 XX focus', '10:10 XX q', '12:12 XX q;11:12 XX compound;11:13 XX proper;15:15 XX q;14:15 XX compound;14:16 XX proper;17:17 XX q;18:18 XX q;21:21 XX q;20:21 XX compound;20:23 XX proper;19:23 XX proper;27:27 XX udef;14:27 XX udef;11:27 XX udef;10:27 XX proper', '3:3 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;12:12 XX d;13:13 XX d;14:14 XX nonsp;15:15 XX d;16:16 XX d;18:18 XX q;19:19 XX q;20:20 XX d', '15:15 XX q;16:16 XX q;0:19 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX q;14:14 XX q;15:15 XX d;14:15 XX udef;0:15 XX focus', '0:0 XX nonsp;1:1 XX nonsp;4:4 XX d;5:5 XX d;7:7 XX d;0:8 XX loc', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:9 XX proper;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX proper;13:15 XX udef;10:15 XX compound;0:15 XX focus', '14:14 XX comp', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:9 XX proper;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;0:20 XX focus', '0:0 XX q;0:18 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q', '2:2 XX q;3:4 XX compound;9:9 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;14:14 XX q;9:14 XX udef', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q', '0:0 XX superl;11:11 XX q;10:11 XX proper;12:12 XX q;13:13 XX proper', '1:2 XX compound;6:6 XX q;7:7 XX q;13:13 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;17:17 XX nonsp;0:23 XX proper', '6:6 XX q', '', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;6:6 XX nominalization;7:7 XX d;9:9 XX d;8:9 XX compound;11:11 XX d;12:12 XX d;14:14 XX d;13:14 XX compound;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;19:19 XX udef;0:19 XX focus', '0:0 XX q;12:13 XX compound;17:17 XX q;18:19 XX compound;0:23 XX udef', '', '', '', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;0:11 XX focus', '0:0 XX q;2:2 XX q;6:6 XX q;10:10 XX q;11:11 XX q;0:11 XX proper', '2:2 XX superl;10:10 XX q', '0:0 XX q;2:2 XX q;3:3 XX q;4:4 XX q;4:5 XX udef;12:12 XX q;13:13 XX q;12:13 XX udef;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX udef;19:19 XX q;20:20 XX q;20:20 XX udef;15:20 XX udef;0:20 XX udef', '9:9 XX q;11:11 XX q;12:12 XX q;0:24 XX appos', '0:0 XX q;2:2 XX q;11:11 XX q;14:14 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;19:21 XX compound;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;26:27 XX udef;28:28 XX q;26:28 XX compound;29:29 XX q;0:29 XX udef', '10:10 XX q;11:11 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;7:11 XX udef;17:18 XX udef;21:21 XX nonsp;21:21 XX loc', '', '0:0 XX d;1:1 XX d;3:3 XX d;6:6 XX q;7:7 XX q;8:8 XX d;7:8 XX udef;5:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;12:13 XX udef;14:14 XX d;15:15 XX d;16:16 XX d;5:16 XX focus;4:16 XX proper;0:16 XX focus', '6:6 XX q;6:7 XX udef;0:17 XX appos', '2:2 XX comp;7:7 XX q', '', '2:2 XX d;1:2 XX compound;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;12:12 XX q;12:12 XX nominalization;14:14 XX q;14:14 XX nominalization;18:18 XX q;18:18 XX nominalization;19:20 XX compound;15:22 XX compound;12:22 XX udef;9:22 XX eventuality;0:22 XX proper', '1:1 XX q;2:2 XX q;3:3 XX q;8:8 XX q;0:15 XX compound', '', '11:11 XX q;13:13 XX compound;0:13 XX appos', '0:0 XX q;1:1 XX q;11:11 XX neg;0:17 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:10 XX udef;7:10 XX udef;18:18 XX q;23:23 XX q;24:24 XX q;23:25 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:32 XX compound;31:32 XX udef;28:32 XX udef;23:32 XX udef;18:32 XX udef;16:32 XX subord', '0:0 XX q;1:1 XX q;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX times;19:21 XX udef;23:23 XX q;24:24 XX q;28:28 XX compound;22:28 XX compound;0:28 XX udef', '9:10 XX udef', '0:0 XX q;0:6 XX udef', '4:4 XX q;8:8 XX q;8:8 XX nominalization;9:10 XX udef;8:12 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:25 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;29:31 XX udef;0:31 XX focus', '2:2 XX neg;3:3 XX d;4:4 XX d;5:5 XX d', '', '', '3:3 XX comp;5:5 XX nonsp', '0:0 XX d;1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;14:14 XX superl;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX d;22:22 XX d;23:23 XX q;24:24 XX d;0:24 XX focus', '8:8 XX q;0:8 XX compound', '0:0 XX superl', '0:0 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;0:18 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:10 XX focus', '3:3 XX d;4:4 XX q;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;11:11 XX d', '1:1 XX q;0:1 XX proper;2:2 XX q;0:3 XX udef;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;16:16 XX d;17:17 XX d;18:18 XX q;0:18 XX proper', '', '', '', '0:0 XX q;1:2 XX compound;3:3 XX neg;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;16:16 XX q;0:21 XX udef', '2:2 XX d;3:3 XX d;5:5 XX q;5:5 XX compound;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;22:22 XX d;23:23 XX d;24:24 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:32 XX udef;0:32 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;13:15 XX measure;19:20 XX udef;19:20 XX compound;0:20 XX focus', '6:6 XX q;7:7 XX q;8:9 XX compound;16:16 XX q;17:17 XX q;18:18 XX times;16:18 XX udef;27:27 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX q;8:8 XX compound;7:8 XX udef;9:9 XX q;9:9 XX compound;0:11 XX focus', '0:0 XX d;1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX udef;0:15 XX focus', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization', '0:0 XX q;1:1 XX q;2:2 XX times;0:2 XX udef;3:4 XX proper;6:6 XX q;8:8 XX d;7:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;17:18 XX compound;19:19 XX d;21:21 XX q;21:21 XX compound;20:21 XX proper;20:22 XX proper;23:23 XX q;24:24 XX q;26:26 XX q;26:26 XX compound;25:26 XX proper;25:27 XX proper;28:28 XX q;25:28 XX compound;23:28 XX udef;20:28 XX appos;6:28 XX udef;5:28 XX compound;3:28 XX compound;0:28 XX compound', '2:2 XX d;3:3 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d', '', '', '0:0 XX q;7:7 XX q;8:8 XX q;9:9 XX q;0:13 XX proper', '0:0 XX q;2:2 XX q;5:5 XX q;7:7 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;14:14 XX q;12:14 XX udef;15:15 XX d;16:16 XX q;21:21 XX d;20:21 XX compound;22:22 XX d;17:22 XX compound;16:22 XX udef;0:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;8:8 XX q;9:9 XX q;0:9 XX focus', '3:4 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;8:9 XX udef;6:9 XX udef', '3:3 XX q;4:4 XX q', '2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;19:20 XX udef;17:20 XX udef;0:20 XX focus', '0:0 XX q;3:3 XX q;5:5 XX q;0:15 XX udef', '1:1 XX q;1:2 XX udef;0:3 XX udef;8:8 XX q;7:9 XX udef', '4:4 XX comp;13:13 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX compound', '0:0 XX q;9:9 XX q;11:11 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;0:21 XX udef', '0:5 XX appos', '8:8 XX q;11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;0:14 XX compound', '', '3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;0:17 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;0:11 XX focus', '2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q', '', '5:5 XX q;6:6 XX q;14:14 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX q', '0:0 XX d;1:1 XX d;3:3 XX q;3:4 XX udef;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX d;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;20:20 XX q;21:21 XX d;22:22 XX q;23:23 XX q;0:23 XX focus', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX times;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:13 XX part', '3:4 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX compound;12:13 XX udef;8:13 XX udef;3:13 XX compound', '9:9 XX q', '9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX q;18:18 XX d;0:19 XX focus', '0:0 XX d;1:1 XX q;2:2 XX q;3:3 XX d;1:3 XX udef;5:5 XX q;6:6 XX d;4:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;12:12 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX udef;20:22 XX udef;17:22 XX udef;12:22 XX udef;0:22 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:14 XX proper;13:15 XX appos;16:16 XX q;17:17 XX q;18:18 XX q;18:19 XX proper;21:23 XX measure;24:25 XX compound;18:32 XX appos;16:32 XX udef;0:32 XX focus', '0:0 XX nonsp;1:1 XX d;2:2 XX q;5:6 XX proper;7:8 XX compound;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX q;16:16 XX d;15:16 XX compound;17:17 XX d;18:18 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;14:24 XX udef;0:24 XX loc', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX d;22:22 XX d;23:23 XX d;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;7:8 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;15:16 XX udef;0:16 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;17:18 XX proper;0:20 XX focus', '3:3 XX neg', '0:0 XX d;2:2 XX d;3:3 XX d;6:6 XX q;9:9 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX times;15:17 XX udef;0:18 XX focus', '12:12 XX q;14:14 XX q;14:15 XX udef;12:21 XX udef', '0:0 XX d;1:2 XX proper;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX nonsp;12:12 XX loc', '', '0:0 XX q;1:1 XX q;1:1 XX compound;6:7 XX udef;16:16 XX q;16:16 XX nominalization;0:20 XX proper', '0:0 XX d;13:13 XX q;0:13 XX focus', '0:0 XX d;1:1 XX q;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:8 XX focus', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:6 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX q;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;18:18 XX q;19:19 XX q;17:19 XX udef;15:19 XX udef;13:19 XX udef;0:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;18:18 XX d;19:19 XX d;21:21 XX q;20:21 XX compound;20:22 XX proper;23:23 XX q;24:24 XX q;25:26 XX proper;29:29 XX q;29:29 XX compound;28:29 XX proper;28:30 XX proper;28:31 XX compound;25:32 XX appos;23:32 XX udef;0:32 XX focus', '', '0:0 XX q;1:1 XX q;2:2 XX d;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:12 XX proper;13:13 XX q;14:14 XX q;13:14 XX udef;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper;21:22 XX proper;23:23 XX q;21:23 XX compound;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;19:30 XX proper;0:30 XX udef', '9:9 XX q;9:10 XX udef;12:12 XX q;12:13 XX udef;12:14 XX compound;16:17 XX udef', '', '6:6 XX q;7:7 XX q;8:8 XX q;11:11 XX nonsp;21:21 XX q;22:22 XX q', '1:2 XX proper;1:3 XX appos;12:13 XX compound;0:13 XX focus', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;4:5 XX proper;6:7 XX compound;8:8 XX d;6:8 XX compound;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;0:19 XX focus', '3:4 XX compound;17:17 XX nonsp;23:23 XX q;0:23 XX appos', '', '0:0 XX q;0:0 XX compound;15:15 XX q;17:17 XX q;18:18 XX q;0:22 XX proper', '0:8 XX appos', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX q;22:22 XX compound;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;31:31 XX q;31:31 XX compound;30:31 XX proper;0:31 XX focus', '0:0 XX q;0:0 XX compound;2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;19:19 XX q;0:19 XX proper', '6:6 XX q;6:6 XX nominalization;7:7 XX q', '2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;4:6 XX proper;13:13 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX appos', '0:0 XX d;1:1 XX d;4:4 XX d;3:4 XX compound;5:5 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:15 XX compound;16:16 XX d;14:16 XX compound;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;13:20 XX udef;12:20 XX udef;0:20 XX focus', '', '1:1 XX q;2:2 XX q;3:4 XX proper;5:5 XX q;3:5 XX compound;6:6 XX q;6:6 XX compound;6:7 XX udef;9:10 XX proper;11:12 XX compound;14:14 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX compound;25:26 XX udef;24:26 XX udef;21:26 XX udef;0:26 XX appos', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;10:10 XX d;0:11 XX focus', '6:6 XX q;16:16 XX d', '0:0 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;9:9 XX compound;13:13 XX comp;33:33 XX neg;0:35 XX proper', '11:11 XX q;12:12 XX q;13:14 XX udef;23:23 XX q;24:24 XX q;25:25 XX compound;24:25 XX udef;22:25 XX udef', '15:15 XX q;16:16 XX q;15:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;19:22 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;26:27 XX udef', '0:0 XX nonsp;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX loc', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;8:10 XX udef;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;12:15 XX udef;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;0:20 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;8:10 XX udef;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;12:15 XX udef;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;0:20 XX focus', '3:3 XX q;4:4 XX q;6:6 XX q;21:21 XX q;22:22 XX q', '0:0 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;13:13 XX q;14:14 XX q;0:15 XX udef', '0:0 XX nonsp;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX q;0:16 XX loc', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;7:7 XX q;8:8 XX q;9:10 XX proper;9:11 XX compound;20:20 XX q;22:22 XX nonsp;21:22 XX loc;0:22 XX compound', '0:0 XX conj;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;0:4 XX implicit', '3:3 XX q;7:7 XX q;7:7 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:23 XX compound', '2:2 XX q;2:3 XX udef;1:3 XX udef;1:4 XX compound;0:4 XX udef;6:6 XX q;6:6 XX compound;6:8 XX udef;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;14:17 XX udef;10:17 XX udef;20:20 XX q;21:21 XX q;20:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;24:27 XX udef;20:27 XX udef;28:28 XX nonsp;29:29 XX nonsp;28:29 XX loc', '', '0:0 XX q;1:1 XX q;8:8 XX q;9:9 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;20:21 XX compound;19:21 XX proper;28:28 XX q;29:29 XX q;28:31 XX udef;33:33 XX q;34:34 XX q;35:35 XX q;37:37 XX compound;36:37 XX udef;35:37 XX udef;33:37 XX udef;0:37 XX udef', '0:0 XX superl;6:6 XX q;7:7 XX q;22:23 XX loc', '0:0 XX q;1:1 XX q;7:7 XX q;8:8 XX q;9:9 XX times;11:11 XX q;12:12 XX q;13:13 XX times;0:20 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:19 XX compound;20:20 XX d;0:21 XX focus', '6:6 XX q;7:7 XX q;8:8 XX times;6:8 XX udef;13:13 XX q;14:14 XX q;16:16 XX nonsp;17:17 XX nonsp;16:17 XX loc;13:17 XX udef', '6:6 XX q;7:7 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX q;9:15 XX compound;17:18 XX loc;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;24:25 XX udef;26:26 XX q;24:26 XX udef;27:27 XX q;29:29 XX q;28:29 XX compound;23:29 XX udef;20:29 XX udef;19:29 XX subord;8:29 XX udef', '0:0 XX q;1:1 XX q;1:1 XX compound;2:2 XX d;3:3 XX d;4:4 XX d;0:5 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;0:7 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX d;11:14 XX udef;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX d;23:26 XX udef;27:27 XX nonsp;28:28 XX nonsp;0:28 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '5:5 XX neg;25:25 XX q;26:26 XX q', '13:13 XX q;13:14 XX udef;29:29 XX q;31:31 XX q;32:32 XX q;32:32 XX nominalization;39:39 XX q;32:39 XX udef;29:39 XX udef;20:39 XX compound', '0:0 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;23:23 XX q;24:24 XX q;0:26 XX focus', '6:6 XX q', '', '', '2:2 XX nonsp;5:6 XX compound', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q', '', '', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;12:12 XX p;13:13 XX d;14:14 XX d;0:16 XX focus', '', '5:5 XX q;6:6 XX q;10:10 XX q;20:20 XX conj;21:21 XX nonsp;25:25 XX conj;23:33 XX focus;21:33 XX loc', '8:8 XX q;10:10 XX q;9:10 XX udef', '17:17 XX q;18:18 XX q;19:19 XX q;29:29 XX q', '0:0 XX q;0:0 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX d;7:7 XX conj;8:8 XX d;9:9 XX q;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX d;19:20 XX udef;31:31 XX q;30:31 XX proper;32:32 XX q;33:33 XX proper;30:33 XX udef;9:33 XX udef;0:33 XX udef', '7:7 XX q', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX d;7:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX q;19:19 XX q;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;0:24 XX focus', '0:0 XX neg', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX q;11:11 XX q;11:11 XX nominalization;7:11 XX udef;0:11 XX implicit', '', '', '2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj', '10:10 XX q;11:11 XX q;12:12 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX proper', '15:15 XX q;17:17 XX q;19:20 XX proper;16:20 XX udef;23:23 XX q;23:23 XX compound;22:23 XX proper;24:24 XX q;24:24 XX compound;22:24 XX proper;22:24 XX proper;14:24 XX udef', '1:1 XX neg', '12:12 XX q;13:14 XX compound;20:20 XX compound;12:20 XX udef', '2:2 XX d;3:3 XX d;6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;12:12 XX d;11:12 XX compound;10:12 XX udef;13:13 XX d;14:14 XX d;17:17 XX q;18:18 XX q;19:19 XX times;17:19 XX udef;15:19 XX generic;20:20 XX d;21:21 XX d;23:23 XX compound;6:23 XX udef;0:23 XX focus', '3:3 XX q;3:4 XX udef;2:4 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;0:19 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;17:17 XX q;17:17 XX nominalization;16:17 XX udef;0:17 XX focus', '3:3 XX q;2:3 XX compound;2:4 XX proper;6:7 XX proper;8:8 XX nominalization;6:8 XX compound;6:9 XX compound;18:19 XX compound;20:20 XX d;21:21 XX d;22:22 XX compound;0:22 XX focus', '4:5 XX compound;8:9 XX udef;22:22 XX proper;0:22 XX proper', '0:0 XX d;0:3 XX focus', '0:12 XX appos', '11:11 XX q;16:16 XX compound', '', '', '0:-1 XX EMPTY-LABEL', '2:3 XX compound;10:10 XX proper', '4:4 XX q;4:4 XX nominalization;5:5 XX q;6:6 XX q;6:6 XX nominalization;5:6 XX udef;7:7 XX q;7:7 XX nominalization;9:9 XX q;12:12 XX q;12:12 XX nominalization;13:13 XX q;15:15 XX compound;13:15 XX udef;12:15 XX udef', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q', '5:5 XX q', '13:13 XX q;14:14 XX d', '0:0 XX q;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:6 XX udef', '', '', '2:2 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:7 XX focus', '', '13:13 XX q;13:13 XX compound', '2:2 XX q;6:6 XX q;7:7 XX q;8:9 XX proper;13:13 XX superl;14:15 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q', '17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX udef;18:19 XX udef', '2:2 XX superl', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX nonsp;9:9 XX q;11:11 XX q;12:12 XX q;14:14 XX q;0:20 XX udef', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;26:27 XX proper;30:30 XX q;30:30 XX compound;29:30 XX proper;26:30 XX udef;0:30 XX udef', '7:7 XX q;8:9 XX udef;7:9 XX udef;11:11 XX q;12:14 XX subord;11:14 XX udef', '', '0:0 XX d;1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;17:17 XX q;18:18 XX d;0:21 XX focus', '5:5 XX q;6:6 XX q;7:7 XX q', '1:1 XX q;2:2 XX q;10:10 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;15:16 XX udef;11:16 XX compound;18:18 XX q;21:21 XX q;20:22 XX compound;0:24 XX compound', '5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;13:14 XX proper;13:15 XX compound;22:23 XX compound;26:28 XX compound', '', '1:1 XX q;2:2 XX q;3:4 XX compound;5:5 XX q;3:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;11:12 XX compound;16:17 XX compound;21:21 XX q;22:22 XX q;28:28 XX q;29:29 XX q;28:29 XX udef;30:30 XX q;33:33 XX q;34:34 XX q;38:38 XX proper;36:38 XX udef;35:38 XX udef;33:38 XX udef;21:38 XX udef;15:38 XX udef;10:38 XX udef;0:38 XX subord', '0:9 XX compound', '1:1 XX q', '13:13 XX q;14:15 XX proper;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:22 XX compound;21:22 XX udef;23:23 XX q;23:23 XX compound;21:23 XX udef;24:24 XX compound', '0:2 XX compound', '', '12:12 XX q;13:13 XX q;15:15 XX q', '0:0 XX q;3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;9:10 XX compound;14:15 XX proper;14:16 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;27:27 XX q;29:29 XX q;31:31 XX q;0:32 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;23:23 XX d;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX d;31:31 XX q;0:31 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX d;8:8 XX d;10:10 XX d;9:10 XX proper;11:11 XX d;13:13 XX d;14:14 XX q;15:15 XX q;18:18 XX compound;17:18 XX compound;14:18 XX udef;0:18 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;10:10 XX compound;0:10 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX d;17:17 XX d;0:17 XX focus', '', '13:13 XX q;16:16 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:25 XX proper', '0:0 XX q;1:1 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX compound;24:24 XX q;27:27 XX q;0:28 XX udef', '0:0 XX superl;6:6 XX q;6:7 XX udef;11:11 XX q;9:11 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;13:17 XX udef', '0:0 XX q;4:4 XX q;3:4 XX compound;6:6 XX q;10:11 XX compound;13:13 XX nonsp;0:24 XX proper', '15:15 XX q;15:15 XX nominalization;16:17 XX compound;16:17 XX udef', '1:2 XX proper;19:19 XX q;19:21 XX udef;24:24 XX q;26:26 XX q;26:28 XX udef;24:31 XX udef;22:31 XX subord', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX d;10:10 XX d;11:11 XX d;0:13 XX focus', '', '1:1 XX d;2:2 XX d;3:3 XX d;0:4 XX focus', '0:0 XX q;2:2 XX q;3:3 XX q;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;0:15 XX proper', '2:3 XX compound;8:9 XX compound;7:13 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;10:10 XX q;11:11 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;0:24 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;10:10 XX q;11:11 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;0:24 XX udef', '19:19 XX superl;20:20 XX superl', '', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;17:17 XX q;20:20 XX q;21:21 XX udef;19:21 XX udef;18:21 XX udef', '5:5 XX q;6:6 XX q;6:7 XX udef;5:7 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:19 XX proper;21:21 XX q;20:21 XX compound;22:22 XX q;20:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:28 XX proper;29:29 XX q;27:29 XX compound;30:30 XX q;30:31 XX udef;0:31 XX focus', '2:2 XX q;7:7 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;12:14 XX udef;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX d;23:24 XX udef;21:24 XX udef;16:24 XX udef;0:24 XX focus', '1:1 XX neg', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:8 XX focus', '5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;9:9 XX udef;7:9 XX udef', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q', '1:1 XX q;1:1 XX nominalization;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX nominalization;7:7 XX q;9:9 XX conj;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;24:24 XX compound;11:24 XX udef;2:24 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:13 XX proper;17:17 XX q;17:17 XX proper;0:17 XX compound', '11:11 XX neg', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;14:16 XX compound;18:18 XX q;20:20 XX q;19:20 XX udef', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;0:6 XX focus', '0:0 XX nonsp;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX d;23:23 XX d;27:27 XX q;26:27 XX compound;28:28 XX q;29:29 XX q;33:33 XX q;34:35 XX compound;36:36 XX q;34:36 XX compound;39:39 XX proper;33:39 XX udef;32:39 XX udef;0:39 XX loc', '0:0 XX d;1:1 XX q;2:2 XX q;1:3 XX udef;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX q;10:11 XX udef;13:13 XX d;14:14 XX d;16:16 XX d;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;6:6 XX q;6:6 XX compound;5:6 XX compound;8:8 XX d;7:8 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX compound;0:19 XX focus', '6:6 XX q;14:14 XX q;14:14 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q', '6:6 XX comp;26:26 XX q;26:26 XX udef', '0:0 XX q;0:0 XX nominalization;1:1 XX q;2:2 XX q;2:2 XX nominalization;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:10 XX compound;7:10 XX udef;6:10 XX udef;15:15 XX q;16:16 XX q;25:25 XX comp;33:33 XX q;34:34 XX udef;33:34 XX udef;32:34 XX udef;19:34 XX compound;15:34 XX udef;0:34 XX udef', '6:6 XX q;12:12 XX q;12:13 XX udef;11:13 XX udef;11:14 XX compound;10:14 XX proper;22:23 XX udef;29:29 XX comp;30:30 XX q;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;35:35 XX q;35:35 XX nominalization;37:37 XX q;37:37 XX nominalization;36:37 XX compound;38:38 XX q;38:38 XX nominalization;34:38 XX udef;32:38 XX udef', '4:4 XX q', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX p;25:25 XX d;26:26 XX d;0:26 XX udef', '', '0:0 XX q;5:5 XX q;11:11 XX q;13:13 XX q;14:14 XX q;17:17 XX q;20:20 XX q;23:23 XX q;24:24 XX q;25:25 XX q;27:27 XX q;28:28 XX udef;0:28 XX proper', '4:4 XX q', '4:4 XX q', '', '1:1 XX d;2:2 XX d;4:4 XX superl;5:5 XX d;6:6 XX d;8:8 XX d', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;15:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;13:22 XX udef;8:22 XX compound', '6:6 XX q;9:9 XX q;10:10 XX q;13:13 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;27:27 XX udef;25:27 XX udef', '5:5 XX comp', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX d;14:14 XX d;15:16 XX proper;20:20 XX proper;0:20 XX focus', '2:2 XX person;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d', '1:1 XX neg', '0:0 XX q;1:1 XX q;2:2 XX q;0:4 XX proper', '2:2 XX q;6:6 XX q;7:7 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:22 XX udef;19:23 XX udef', '', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;18:18 XX q;0:18 XX udef', '5:5 XX q;6:6 XX q;8:8 XX q;7:8 XX compound;9:9 XX q;7:9 XX udef;10:11 XX proper;13:13 XX q;14:14 XX q;13:14 XX udef;10:14 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:13 XX focus', '', '4:4 XX q', '4:4 XX q', '0:-1 XX EMPTY-LABEL', '8:8 XX q;13:13 XX superl;20:20 XX q;21:21 XX q;22:22 XX q', '0:0 XX q;1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;7:8 XX idiom;9:9 XX q;3:10 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;20:20 XX q;21:21 XX udef;19:21 XX udef;0:21 XX udef', '14:14 XX q;15:15 XX q;16:16 XX times;18:18 XX q;19:19 XX q;20:20 XX times;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization', '3:3 XX superl;13:13 XX q;14:14 XX q;15:15 XX times;13:15 XX udef', '0:0 XX superl;11:11 XX q;17:17 XX q;18:18 XX q;19:19 XX times;22:22 XX q;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization', '7:7 XX q;8:8 XX q', '', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;5:5 XX q;6:6 XX q;8:8 XX compound;7:8 XX proper;4:8 XX compound', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:10 XX proper;15:15 XX q;14:15 XX compound;16:16 XX q;16:16 XX compound;16:16 XX udef;14:16 XX proper;0:16 XX focus', '0:1 XX proper;5:6 XX proper;7:7 XX q;5:7 XX compound;5:8 XX proper;0:43 XX appos', '9:9 XX q;11:11 XX q;12:13 XX proper;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX proper;18:20 XX proper;0:20 XX compound', '13:14 XX udef', '', '7:7 XX q;6:7 XX compound;8:8 XX q;9:10 XX proper;11:11 XX q;9:11 XX compound;9:12 XX proper;13:13 XX q;15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper;18:19 XX compound;22:22 XX q;21:22 XX compound;23:23 XX q;23:23 XX compound;23:24 XX udef;21:24 XX proper;14:24 XX appos;28:29 XX compound;9:29 XX udef;39:39 XX q;40:40 XX compound', '2:2 XX q;3:3 XX q;7:7 XX q;7:7 XX compound;6:7 XX compound;8:9 XX proper;10:10 XX q;8:10 XX compound;11:11 XX q;13:13 XX q;12:14 XX udef;8:14 XX proper', '0:0 XX d;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;14:14 XX q;13:14 XX compound;15:15 XX q;11:16 XX udef;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX d;23:24 XX proper;7:24 XX udef;0:24 XX focus', '0:1 XX proper;3:4 XX compound;7:7 XX q;7:7 XX compound;6:7 XX proper;6:8 XX proper;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;11:17 XX udef;9:29 XX udef;6:29 XX compound;0:29 XX appos', '', '11:11 XX q;18:18 XX neg;11:20 XX udef', '9:9 XX comp;16:16 XX q;20:20 XX neg;28:28 XX comp', '4:5 XX udef', '2:2 XX q;1:2 XX proper;3:3 XX q;8:8 XX q;8:8 XX compound', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;9:9 XX d;8:9 XX compound;7:9 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX proper;7:17 XX focus;0:17 XX focus', '5:5 XX q', '1:1 XX neg', '6:6 XX q;7:7 XX q;8:8 XX q;11:12 XX compound;9:12 XX compound;19:19 XX q;20:20 XX q;21:21 XX q;24:24 XX q;25:25 XX q;27:27 XX q;28:28 XX q;28:28 XX udef;19:28 XX udef;8:28 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX q;22:22 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;26:29 XX udef;0:29 XX proper', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:14 XX q;14:15 XX udef;16:16 XX d;0:20 XX focus', '2:2 XX q;3:3 XX q;2:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;6:9 XX udef;16:16 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:21 XX udef', '5:5 XX q;6:6 XX q;5:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;9:13 XX udef;5:13 XX udef;19:19 XX q;19:19 XX named;20:20 XX q;20:20 XX named;21:21 XX q;21:21 XX compound;19:23 XX proper;30:30 XX q;30:30 XX compound;29:30 XX compound;36:36 XX q;42:42 XX q;0:47 XX subord', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:2 XX compound', '0:27 XX compound', '9:9 XX q;10:10 XX q', '1:1 XX neg;3:3 XX thing;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX udef', '', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX proper;0:11 XX focus', '0:0 XX q;1:1 XX q;12:12 XX q;15:15 XX q;16:16 XX q;17:17 XX q;21:21 XX q;24:24 XX q;25:25 XX q;26:26 XX q;35:35 XX comp;0:38 XX proper', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;9:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;13:16 XX udef;9:16 XX udef;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q', '5:5 XX q;5:5 XX named', '6:7 XX udef;9:10 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;22:22 XX q;24:24 XX q;23:24 XX compound;25:25 XX q;26:26 XX q;26:26 XX udef', '4:4 XX q;5:5 XX q;10:10 XX q;13:14 XX compound;15:15 XX q;16:16 XX q;17:18 XX proper;19:20 XX compound;17:20 XX compound', '5:5 XX q;6:7 XX compound;5:14 XX udef', '3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;13:13 XX d', '0:0 XX q;0:0 XX nominalization;2:2 XX q;1:2 XX compound;3:3 XX d;5:5 XX d;4:5 XX proper;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;1:12 XX proper;0:12 XX udef', '', '', '3:3 XX neg;9:10 XX compound', '3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;7:8 XX proper;11:12 XX compound;7:12 XX appos;19:20 XX udef;27:27 XX superl;32:33 XX compound;5:36 XX udef;0:36 XX appos', '', '8:8 XX q;15:15 XX q;19:19 XX q;19:20 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;24:27 XX udef', '2:2 XX q;3:3 XX q;4:4 XX q;5:6 XX compound;5:7 XX compound;17:17 XX loc;16:17 XX loc', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;13:13 XX nonsp', '0:9 XX compound', '', '3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;12:13 XX udef;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;16:17 XX udef;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX q;28:31 XX udef;20:31 XX udef;0:31 XX focus', '2:2 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX q;5:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX d;9:12 XX udef;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX q;20:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;24:26 XX udef;0:26 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:0 XX q;0:0 XX compound;5:5 XX q;6:6 XX q;7:7 XX q;0:25 XX proper', '2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX d;11:11 XX q;12:12 XX q;11:13 XX udef', '8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;11:13 XX udef;0:13 XX focus', '', '1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;0:8 XX focus', '0:0 XX q;5:5 XX q;6:6 XX q;16:16 XX q;18:18 XX compound;17:18 XX compound;16:18 XX udef;14:18 XX compound;0:18 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX q;0:18 XX udef', '2:2 XX q;2:2 XX compound;9:9 XX q;12:12 XX q;13:13 XX q;15:15 XX q;17:17 XX q', '3:3 XX comp;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;21:21 XX comp', '7:7 XX q;9:9 XX comp', '4:4 XX q;4:4 XX compound;21:21 XX comp', '6:6 XX q;0:6 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX d;5:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;12:13 XX udef;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;16:17 XX udef;19:19 XX d;18:19 XX proper;0:19 XX compound', '10:10 XX q;10:10 XX compound;9:10 XX proper;16:16 XX q;17:17 XX q;18:18 XX q;0:21 XX subord', '0:4 XX compound', '2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;10:10 XX q;18:18 XX q;0:20 XX subord', '1:1 XX q;2:2 XX q;7:7 XX q;0:25 XX compound', '4:4 XX q;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;9:13 XX udef', '1:1 XX q;2:2 XX q;5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;17:17 XX q;16:17 XX compound;14:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:23 XX udef;10:23 XX udef;26:26 XX q;26:26 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX q;11:11 XX compound;10:11 XX proper;12:12 XX d;14:14 XX d;0:14 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;10:10 XX q;11:11 XX q;10:11 XX udef;0:11 XX focus', '11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;15:15 XX compound;14:15 XX proper', '4:4 XX q;5:5 XX q;6:6 XX q', '3:4 XX proper;6:7 XX compound;11:11 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;11:17 XX udef;3:17 XX appos;23:24 XX compound;27:27 XX q;26:27 XX compound;26:27 XX proper;0:27 XX compound', '16:16 XX q;17:18 XX proper;24:24 XX compound', '9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;13:13 XX compound;12:13 XX compound;14:14 XX compound;12:14 XX udef;10:14 XX udef', '6:6 XX q;7:7 XX q;8:8 XX q', '9:9 XX neg', '0:0 XX q;0:0 XX compound;2:2 XX q;11:11 XX q;12:12 XX q;14:14 XX q;16:16 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX proper', '9:9 XX q;10:10 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX nonsp', '4:5 XX proper;6:6 XX q;12:12 XX q;13:13 XX q;14:14 XX times', '15:15 XX neg', '11:11 XX nonsp;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;28:28 XX q;29:29 XX q;29:29 XX udef;24:29 XX udef;22:29 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;0:13 XX focus', '', '12:13 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;13:14 XX compound;0:14 XX focus', '14:17 XX eventuality', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;23:23 XX nominalization;24:24 XX d;25:25 XX d;26:26 XX d;28:28 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d;34:34 XX nonsp;35:35 XX d;36:36 XX d;37:38 XX proper;39:39 XX q;41:41 XX q;41:41 XX compound;40:41 XX compound;39:41 XX udef;42:43 XX proper;42:43 XX compound;39:43 XX proper;37:43 XX appos;23:43 XX udef;0:43 XX focus', '5:5 XX q;17:17 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;22:22 XX q;21:22 XX udef;23:23 XX q;24:25 XX proper;20:25 XX udef;28:28 XX q;29:29 XX q;28:29 XX udef', '0:0 XX conj;1:1 XX q;1:1 XX compound;22:22 XX q;23:23 XX q;24:24 XX times;0:24 XX implicit', '3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;13:14 XX udef;15:15 XX q;16:17 XX compound;18:18 XX q;16:18 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX d;23:24 XX udef;9:24 XX udef;25:25 XX d;27:27 XX d;28:28 XX q;31:31 XX q;30:31 XX compound;33:33 XX q;34:34 XX q;36:36 XX d;35:36 XX proper;28:36 XX udef;8:36 XX udef;5:36 XX udef;0:36 XX focus', '5:5 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;29:30 XX compound;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;22:34 XX udef;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX q;39:39 XX q;40:40 XX q;42:42 XX q;38:42 XX udef;36:42 XX udef', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;16:16 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;21:21 XX proper;17:21 XX udef;0:21 XX udef', '0:0 XX q;1:1 XX q;2:3 XX proper;4:4 XX q;5:5 XX q;5:5 XX compound;9:9 XX d;14:14 XX q;15:15 XX q;16:16 XX q;0:28 XX proper', '', '13:13 XX q;14:14 XX q', '0:0 XX superl;13:14 XX proper;26:26 XX q;35:35 XX q', '0:0 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:13 XX compound;14:15 XX compound;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:21 XX proper;22:22 XX q;20:22 XX compound;24:24 XX q;25:25 XX q;26:27 XX compound;26:27 XX udef;17:27 XX udef;12:27 XX udef;7:27 XX loc;0:27 XX udef', '19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q', '0:0 XX q;0:0 XX compound;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX comp;24:24 XX d;25:25 XX d;0:25 XX proper', '4:4 XX neg;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;34:34 XX q;34:34 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX q;36:36 XX nominalization;38:38 XX q;38:38 XX nominalization;39:39 XX q;39:39 XX nominalization;41:41 XX superl', '5:5 XX q;5:6 XX proper;9:9 XX nominalization;8:9 XX compound;22:22 XX q;23:23 XX q;29:29 XX q;28:29 XX compound;35:35 XX q', '', '0:0 XX d;1:1 XX d;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX d;8:8 XX d;11:11 XX d;12:12 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX q;20:20 XX d;21:21 XX d;22:22 XX q;24:24 XX d;23:24 XX compound;26:26 XX d;27:27 XX d;29:29 XX d;30:30 XX d;32:32 XX q;32:32 XX compound;31:32 XX udef;33:33 XX q;33:33 XX compound;6:33 XX udef;34:34 XX d;0:34 XX focus', '0:0 XX q;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX d;0:11 XX udef', '1:1 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;12:14 XX udef', '8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;12:12 XX compound;11:12 XX proper;13:13 XX q;11:13 XX proper;14:14 XX q;15:15 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;9:9 XX d;0:9 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;12:12 XX d;13:13 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;27:27 XX q;27:27 XX compound;26:27 XX proper;29:29 XX compound;28:29 XX proper;26:29 XX proper;23:29 XX compound;0:29 XX focus', '1:1 XX neg', '5:5 XX q;6:6 XX q;11:11 XX q;19:19 XX q;20:20 XX times;22:22 XX q;25:25 XX neg;31:31 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;7:7 XX d;6:7 XX compound;8:8 XX d;9:9 XX d;10:11 XX proper;12:12 XX d;10:12 XX compound;13:13 XX d;14:14 XX d;16:16 XX d;15:16 XX proper;0:16 XX compound', '9:9 XX q;10:10 XX q;10:10 XX compound;22:22 XX q;23:23 XX neg;0:26 XX compound', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '0:0 XX q;2:2 XX q;1:2 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;7:8 XX compound;9:9 XX q;10:11 XX udef;15:15 XX q;16:16 XX q;17:17 XX times;15:17 XX udef;0:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:10 XX proper;13:13 XX q;15:15 XX compound;14:15 XX compound;13:15 XX udef;12:15 XX udef;11:15 XX udef;0:15 XX focus', '9:9 XX q;10:10 XX q;11:11 XX times', '1:1 XX q;7:7 XX q;10:10 XX q;11:11 XX q;12:12 XX times;20:20 XX q;21:21 XX q;22:22 XX times;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX q;31:31 XX q;33:33 XX q;36:36 XX q;38:38 XX q;0:38 XX udef', '1:1 XX neg', '', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;20:20 XX d;22:22 XX d;23:23 XX q;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:8 XX focus', '10:10 XX q;9:10 XX compound;9:11 XX proper', '2:2 XX q;1:2 XX udef;3:3 XX q;5:6 XX compound;4:6 XX udef;9:9 XX nonsp;9:9 XX loc', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:25 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX q;12:12 XX compound;13:13 XX q;12:14 XX udef;15:15 XX neg;0:15 XX focus', '4:4 XX thing', '8:8 XX q', '', '', '0:0 XX q;0:0 XX compound;12:12 XX q;0:20 XX proper', '3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;18:18 XX q;19:19 XX q', '3:3 XX q;3:3 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;15:15 XX q;14:16 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;26:27 XX proper;30:31 XX proper;32:32 XX q;30:32 XX compound;34:34 XX q;35:35 XX q;36:36 XX loc;33:36 XX loc;0:36 XX focus', '', '14:15 XX compound;19:19 XX q', '4:4 XX q;5:5 XX q;6:6 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX udef;0:17 XX focus', '0:0 XX d;1:1 XX d;5:6 XX compound;0:6 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:6 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;0:10 XX focus', '', '0:0 XX d;1:1 XX d;3:3 XX q;3:4 XX udef;5:5 XX d;6:6 XX d;7:7 XX q;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:20 XX focus', '0:6 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX d;10:10 XX d;11:11 XX d;0:12 XX focus', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;2:3 XX udef;4:4 XX d;6:6 XX d;5:6 XX compound;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;10:11 XX compound;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;0:20 XX focus', '', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;14:14 XX d;0:15 XX focus', '', '5:5 XX q;6:6 XX q;7:7 XX q;7:7 XX udef', '0:12 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX proper;16:16 XX q;17:17 XX q;14:17 XX proper;19:19 XX q;20:20 XX q;21:21 XX q;20:21 XX udef;22:22 XX q;23:23 XX q;23:23 XX nominalization;25:25 XX q;25:25 XX nominalization;23:25 XX udef;0:25 XX focus', '6:6 XX q', '3:3 XX q;3:3 XX pron', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:6 XX focus', '7:7 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;16:16 XX q;17:17 XX q;17:18 XX udef;16:18 XX udef;19:19 XX q;20:20 XX q;20:20 XX nominalization;22:22 XX q;23:23 XX q;23:24 XX udef;21:24 XX udef;25:25 XX q;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;29:29 XX q;20:29 XX udef;0:29 XX appos', '8:8 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;0:19 XX focus', '', '4:5 XX compound;8:9 XX proper;11:13 XX measure;14:15 XX proper;14:16 XX compound;23:24 XX compound;23:24 XX compound;20:24 XX eventuality', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX q;19:19 XX d;0:20 XX focus', '', '', '9:9 XX q;23:23 XX q;23:23 XX nominalization;0:23 XX subord', '13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization', '11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;0:23 XX appos', '10:10 XX conj;0:11 XX loc', '', '0:0 XX nonsp;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX d;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX q;22:22 XX times;20:22 XX udef;24:24 XX q;23:24 XX compound;25:25 XX d;20:26 XX compound;0:26 XX focus', '', '3:3 XX comp', '3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX q;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX q;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;18:25 XX compound;0:25 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;15:15 XX comp;17:17 XX thing;0:28 XX proper', '0:0 XX d;1:1 XX q;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;0:12 XX focus', '1:1 XX thing;3:3 XX q;4:4 XX q', '1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX d;14:14 XX q;14:14 XX nominalization;15:16 XX compound;17:17 XX d;18:18 XX d;20:20 XX d;23:23 XX d;22:23 XX compound;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX q;29:29 XX times;27:29 XX udef;30:30 XX d;27:30 XX compound;31:31 XX d;32:32 XX d;33:34 XX proper;33:34 XX compound;21:34 XX proper;0:34 XX udef', '4:4 XX q;3:4 XX compound;15:15 XX q;14:15 XX compound;14:16 XX proper;18:19 XX compound', '0:0 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;18:21 XX udef;0:21 XX udef', '0:0 XX q;1:1 XX q;0:7 XX udef', '0:0 XX d;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX d;8:8 XX d;6:8 XX focus;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX q;20:21 XX compound;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;28:28 XX d;29:29 XX d;35:35 XX compound;15:35 XX udef;0:35 XX focus', '', '5:5 XX q;12:12 XX q;14:14 XX q;15:15 XX q;17:17 XX conj;14:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX times;18:20 XX udef;12:24 XX udef', '', '16:17 XX udef;19:19 XX q;19:19 XX udef;16:19 XX udef;14:19 XX compound', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;13:14 XX proper;15:15 XX q;13:15 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;23:24 XX udef;25:25 XX q;26:27 XX proper;31:31 XX d;30:31 XX compound;29:31 XX compound;33:33 XX d;10:33 XX udef;9:33 XX udef;0:33 XX focus', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:7 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:7 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX d;13:13 XX d;14:14 XX q;16:16 XX d;15:16 XX proper;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX q;22:22 XX d;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;26:29 XX udef;25:29 XX udef;23:29 XX udef;19:29 XX udef;0:29 XX focus', '9:9 XX q;9:10 XX udef;11:15 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;8:8 XX d;7:8 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX q;23:23 XX q;24:24 XX d;22:24 XX udef;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:32 XX compound;33:33 XX d;31:33 XX compound;34:34 XX d;36:36 XX d;37:37 XX d;38:38 XX q;39:39 XX q;40:40 XX q;41:41 XX q;38:41 XX udef;27:41 XX udef;7:41 XX focus;0:41 XX focus', '1:1 XX q;2:2 XX q;11:11 XX thing;0:16 XX udef', '5:5 XX q;7:7 XX q;7:8 XX udef;14:15 XX compound;16:17 XX proper;16:17 XX compound;0:17 XX compound', '', '0:0 XX q;5:5 XX q;7:7 XX q;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX times;0:30 XX proper', '11:11 XX q;12:12 XX q;15:16 XX compound;14:16 XX compound;19:19 XX comp;28:28 XX q;27:28 XX compound;27:28 XX proper;26:28 XX proper', '0:0 XX q;2:2 XX q;18:18 XX q;0:28 XX proper', '0:0 XX d;3:3 XX d;2:3 XX compound;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:12 XX proper;13:13 XX d;11:13 XX compound;1:13 XX compound;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX focus', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;4:5 XX udef;7:7 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX compound;15:15 XX q;16:16 XX q', '13:13 XX q;17:17 XX q', '11:11 XX q;10:11 XX proper;12:12 XX q;13:14 XX compound;16:16 XX q;17:17 XX q;13:18 XX compound;23:23 XX q;24:24 XX q;23:25 XX udef;22:25 XX udef;32:32 XX q;33:33 XX q;32:33 XX udef', '0:10 XX measure', '3:3 XX q;4:4 XX q;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX proper', '0:0 XX d;1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX d;14:17 XX udef;10:17 XX udef;18:18 XX d;19:19 XX d;22:22 XX d;20:22 XX compound;23:23 XX d;24:24 XX d;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;27:29 XX udef;25:29 XX udef;30:30 XX q;32:32 XX d;34:34 XX d;35:35 XX d;36:36 XX d;37:37 XX d;0:37 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;19:21 XX compound;22:22 XX q;26:26 XX q;25:26 XX compound;27:27 XX q;28:28 XX q;29:29 XX nonsp;30:30 XX q;29:30 XX loc;31:31 XX q;32:32 XX q;33:33 XX q;33:34 XX udef;35:35 XX q;33:35 XX compound;36:36 XX loc;28:36 XX udef;12:36 XX proper;0:36 XX focus', '', '', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX q;13:14 XX compound;15:15 XX q;16:17 XX compound;18:18 XX q;16:18 XX compound;16:18 XX proper;0:18 XX focus', '', '0:0 XX q;1:1 XX q;5:5 XX comp;0:9 XX proper', '0:0 XX q;0:0 XX compound;2:2 XX neg;8:8 XX q;8:8 XX nominalization;16:16 XX q;17:17 XX q;0:17 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX compound;12:13 XX udef;0:13 XX proper', '5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;13:13 XX q;14:14 XX q;14:14 XX compound;16:16 XX q;14:16 XX udef;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX udef;22:22 XX q;22:22 XX udef;19:22 XX udef;12:22 XX udef;8:22 XX udef;7:22 XX udef;6:22 XX udef;5:22 XX udef', '6:6 XX q;5:6 XX compound;7:7 XX q;8:8 XX q;9:9 XX q;10:11 XX udef;10:11 XX compound;8:11 XX udef;7:11 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX q;0:18 XX udef', '6:6 XX q;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q', '0:1 XX proper;10:10 XX q;10:10 XX compound;0:11 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;4:4 XX d;5:6 XX proper;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX q;15:15 XX udef;0:15 XX udef', '0:0 XX q;0:0 XX compound;2:2 XX q;3:3 XX q;4:4 XX q;0:20 XX proper', '0:0 XX d;1:1 XX d;3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX d;15:15 XX q;16:16 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;21:21 XX q;23:23 XX d;22:23 XX compound;21:23 XX udef;24:24 XX d;25:25 XX d;17:25 XX compound;0:25 XX focus', '0:-1 XX EMPTY-LABEL', '15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper', '3:3 XX d;5:5 XX d;7:7 XX d;10:10 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;6:6 XX q;5:6 XX compound;7:7 XX q;9:9 XX q;8:9 XX compound;8:9 XX proper;5:9 XX udef;0:9 XX focus', '', '4:4 XX neg', '', '0:0 XX q;14:14 XX q;24:24 XX conj;26:26 XX q;30:30 XX q;31:31 XX q;32:32 XX q;35:35 XX q;35:35 XX compound;34:35 XX proper;37:37 XX q;37:37 XX compound;36:37 XX compound;0:37 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;0:17 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:14 XX compound;19:19 XX d;25:25 XX superl;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d;34:34 XX d;36:36 XX d;37:37 XX d;0:37 XX focus', '3:3 XX q', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;0:11 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;0:3 XX focus', '10:11 XX proper;12:12 XX q;16:16 XX d;0:29 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;11:12 XX proper;13:14 XX compound;23:23 XX q;24:24 XX q;29:29 XX q;30:30 XX q;32:32 XX q;39:39 XX q;39:40 XX udef;42:42 XX q;42:42 XX udef', '2:3 XX proper;4:5 XX compound', '', '2:2 XX q', '5:5 XX q;5:5 XX compound;21:21 XX q;29:29 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX d;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX d;18:19 XX proper;0:19 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:5 XX compound;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;23:23 XX d;0:24 XX focus', '7:7 XX q', '7:7 XX q;7:8 XX udef;13:13 XX q;15:15 XX q;16:16 XX compound;14:16 XX subord;13:16 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;13:13 XX q;13:13 XX udef;11:13 XX generic;0:13 XX focus', '', '2:2 XX q;4:4 XX q;5:5 XX q;5:6 XX udef;14:14 XX q;15:15 XX q;15:15 XX named;16:16 XX q;16:16 XX named;15:17 XX proper;22:22 XX q;21:22 XX compound;21:22 XX proper;18:22 XX compound;15:22 XX compound', '3:3 XX q;4:5 XX compound;0:11 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;0:3 XX focus', '', '0:0 XX q;0:0 XX compound;0:11 XX proper', '0:0 XX q;0:0 XX compound;0:22 XX proper', '0:-1 XX EMPTY-LABEL', '7:7 XX q;15:15 XX q;15:15 XX compound;14:15 XX compound;16:17 XX compound;12:17 XX appos', '5:5 XX q;7:7 XX q;13:13 XX q', '', '2:2 XX q;1:2 XX proper;3:3 XX q;6:6 XX q;6:6 XX compound;5:6 XX compound;7:7 XX compound;4:7 XX appos', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;0:19 XX focus', '0:12 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;0:16 XX focus', '1:1 XX q;0:1 XX proper;2:2 XX q;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;11:11 XX q;11:11 XX nominalization;10:11 XX proper;21:22 XX proper;0:24 XX udef', '5:5 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX d;29:29 XX d;30:30 XX q;31:31 XX d;32:32 XX d', '', '0:0 XX q;1:1 XX q;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;0:13 XX udef', '1:1 XX q;8:8 XX neg;14:14 XX person;15:15 XX person;22:22 XX neg', '7:7 XX neg;14:14 XX thing', '7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization', '', '1:1 XX neg;3:3 XX q;3:3 XX nominalization;10:11 XX proper', '0:0 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;7:8 XX proper;7:9 XX compound;13:13 XX q;17:18 XX compound;19:20 XX compound;17:20 XX udef;31:31 XX q;31:31 XX proper;24:31 XX compound;23:31 XX compound;22:31 XX subord;12:31 XX udef;0:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX d;5:6 XX udef;7:7 XX q;8:8 XX d;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;19:19 XX d;9:19 XX compound;9:20 XX focus;0:20 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:8 XX proper;10:11 XX compound;14:14 XX q;13:14 XX compound;16:16 XX q;13:16 XX proper;0:16 XX focus', '0:1 XX number;7:7 XX q;8:8 XX q;9:9 XX q;12:12 XX q;13:13 XX q;18:18 XX q;18:19 XX udef;31:31 XX comp;22:35 XX loc;0:35 XX compound', '4:4 XX q;4:5 XX udef;9:10 XX number;0:11 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:18 XX number;19:19 XX d;17:19 XX compound;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX compound;0:26 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;22:22 XX q;23:23 XX q;23:23 XX nominalization;0:23 XX udef', '0:0 XX d;3:4 XX compound;5:5 XX neg;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:14 XX number;15:15 XX q;13:15 XX compound;16:16 XX q;18:18 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX q;20:30 XX compound;10:30 XX udef;7:30 XX udef;0:30 XX focus', '0:12 XX generic', '', '15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX p;21:21 XX p;22:22 XX p;23:23 XX compound;20:23 XX with', '0:0 XX q;8:8 XX q;9:11 XX compound;0:11 XX udef', '11:11 XX q;15:15 XX q;16:17 XX compound;18:18 XX comp;20:20 XX comp;25:25 XX comp;15:25 XX udef;27:27 XX neg', '0:1 XX number;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;0:17 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:14 XX focus', '0:0 XX q;2:2 XX q;4:4 XX superl;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX compound;8:10 XX proper;11:11 XX q;12:12 XX q;12:13 XX proper;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:18 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '4:4 XX neg', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX number;14:14 XX d;12:14 XX compound;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:23 XX compound;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX q;40:40 XX q;39:40 XX compound;41:41 XX q;33:41 XX udef;0:41 XX focus', '2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;6:6 XX compound;8:9 XX compound;6:9 XX udef;18:18 XX q;18:18 XX compound;17:18 XX proper;17:19 XX proper;17:20 XX compound', '7:8 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;8:8 XX compound;7:8 XX udef;10:10 XX q;10:10 XX compound;9:10 XX udef;11:11 XX d;9:11 XX udef;7:11 XX udef;0:11 XX compound', '2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX q;25:25 XX q;26:26 XX q;0:26 XX focus', '', '0:1 XX proper;2:2 XX q;0:2 XX compound;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;25:26 XX udef;0:26 XX proper', '', '0:0 XX d;1:1 XX d;3:3 XX d;2:3 XX compound;4:4 XX d;5:5 XX d;6:6 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;21:21 XX neg;0:23 XX focus', '6:6 XX q;6:6 XX compound;5:6 XX proper;8:8 XX q;7:9 XX udef;10:10 XX q;7:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;21:22 XX udef;24:24 XX q;23:24 XX compound;25:25 XX q;26:27 XX number;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX number;29:31 XX udef;21:31 XX compound;17:31 XX udef', '1:1 XX q;5:5 XX q;6:7 XX proper;4:7 XX udef;3:7 XX udef;10:10 XX q;10:11 XX udef;20:20 XX loc;0:20 XX udef', '3:4 XX udef;20:20 XX q;22:22 XX q;24:24 XX q;25:25 XX q;27:27 XX q;28:28 XX q;26:28 XX compound;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX q;41:41 XX q;46:46 XX nonsp', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;11:12 XX compound;13:13 XX q;0:13 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX q;5:6 XX compound;7:7 XX q;0:7 XX focus', '6:7 XX compound', '9:9 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:24 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper;15:15 XX q;12:15 XX compound;16:16 XX q;16:16 XX compound;16:16 XX udef;0:16 XX focus', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q', '0:0 XX q;2:2 XX q;6:7 XX udef;0:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '5:5 XX q;5:6 XX udef;7:8 XX compound;10:10 XX q;10:11 XX udef;17:17 XX q;17:18 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q', '2:2 XX neg', '', '0:0 XX q;1:1 XX q;2:2 XX d;3:3 XX d;4:4 XX d;8:8 XX q;10:11 XX compound;20:20 XX loc;9:20 XX udef;0:20 XX udef', '2:2 XX d;5:5 XX d;4:5 XX compound;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;13:13 XX q;13:13 XX nominalization;15:15 XX q;16:16 XX q;17:18 XX compound;19:19 XX q;19:19 XX nominalization;17:19 XX compound;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;25:25 XX q;25:25 XX nominalization;23:25 XX udef;11:25 XX udef;3:25 XX compound;0:25 XX focus', '10:10 XX q;14:14 XX q;16:16 XX q;18:18 XX nonsp', '1:1 XX q;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;12:12 XX d;10:12 XX compound;13:13 XX d;14:14 XX d;0:14 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;0:7 XX focus', '4:4 XX comp', '2:2 XX q;3:4 XX compound;9:9 XX compound', '5:5 XX q', '', '10:10 XX q', '10:10 XX q', '13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper', '2:2 XX q;3:3 XX q;6:6 XX q;7:8 XX udef;6:8 XX udef;18:19 XX compound;5:21 XX udef;4:21 XX udef', '3:3 XX neg;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q', '2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:7 XX compound', '12:12 XX q;25:25 XX q;24:25 XX compound;26:26 XX q;28:28 XX q;27:28 XX compound;27:28 XX proper', '4:4 XX q;4:4 XX compound;7:7 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:22 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;0:9 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;23:23 XX d;0:23 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;9:9 XX d;8:9 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX d;19:20 XX udef;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;28:28 XX d;27:28 XX compound;16:28 XX udef;6:28 XX udef;0:28 XX focus', '4:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;35:35 XX udef;28:35 XX udef;26:35 XX proper;14:35 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;19:19 XX compound;18:19 XX udef;20:20 XX q;18:20 XX udef;21:21 XX q;22:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:17 XX proper;20:20 XX q;20:20 XX compound;19:20 XX proper;19:21 XX proper;19:22 XX compound;0:22 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;18:18 XX q;19:19 XX d;18:19 XX udef;20:20 XX d;21:22 XX proper;23:23 XX q;24:24 XX q;25:26 XX proper;27:27 XX q;25:27 XX compound;25:27 XX proper;23:27 XX udef;21:27 XX appos;15:27 XX generic;0:27 XX focus', '4:4 XX neg;8:8 XX q;11:11 XX q;9:11 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX compound;14:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;16:16 XX compound;0:17 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:7 XX proper;9:10 XX compound;13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper;16:17 XX proper;18:18 XX compound;16:18 XX compound;12:18 XX appos;0:18 XX focus', '', '2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX nominalization;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q', '0:0 XX d;1:1 XX d;2:2 XX q;2:2 XX nominalization;4:4 XX q;3:4 XX udef;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:30 XX proper;31:31 XX q;32:32 XX q;34:34 XX q;34:34 XX compound;33:34 XX proper;35:35 XX q;33:35 XX proper;38:38 XX q;38:39 XX udef;37:39 XX udef;37:40 XX compound;31:40 XX udef;29:40 XX appos;18:40 XX udef;7:40 XX udef;0:40 XX focus', '12:12 XX q', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX d;9:10 XX udef;11:11 XX d;12:12 XX d;14:14 XX d;13:14 XX proper;7:14 XX udef;0:14 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;18:19 XX compound;0:23 XX focus', '', '2:2 XX q;1:3 XX udef;8:9 XX compound;0:19 XX udef', '3:3 XX superl', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;7:8 XX udef;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX q;23:23 XX d;22:23 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX d;26:27 XX udef;28:28 XX d;29:29 XX q;30:30 XX q;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;34:34 XX q;34:34 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX q;36:36 XX nominalization;37:37 XX d;31:37 XX udef;38:38 XX d;39:39 XX proper;29:39 XX udef;10:39 XX udef;7:39 XX udef;0:39 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX q;5:5 XX nominalization;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;18:18 XX nominalization;19:19 XX q;20:20 XX q;22:22 XX q;22:22 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX d;19:25 XX udef;18:26 XX udef;15:26 XX udef;6:26 XX udef;0:26 XX focus', '6:8 XX udef;16:16 XX q;15:17 XX udef;28:28 XX superl;31:31 XX nonsp;31:31 XX loc;0:31 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:12 XX focus', '6:6 XX q;8:8 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:22 XX proper', '0:0 XX q;1:2 XX proper;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;14:15 XX compound;13:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX d;16:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;23:27 XX udef;11:27 XX udef;0:27 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX q;10:10 XX d;9:10 XX compound;8:10 XX udef;0:11 XX focus', '9:9 XX superl', '5:5 XX nonsp;20:21 XX compound;9:27 XX compound;6:27 XX loc', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;7:7 XX d;5:7 XX compound;0:8 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX q;12:12 XX nominalization;13:13 XX d;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;19:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX nonsp;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX d;15:15 XX d;14:15 XX compound;16:16 XX d;17:17 XX d;21:21 XX d;0:21 XX focus', '6:7 XX proper;8:9 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;0:19 XX focus', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX q;25:25 XX d;24:25 XX udef;26:26 XX d;27:27 XX d;28:28 XX d;30:30 XX d;29:30 XX proper;0:30 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;0:7 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;12:12 XX d;14:14 XX d;0:15 XX focus', '', '0:0 XX q;2:2 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX udef;6:9 XX udef;0:9 XX udef', '1:1 XX neg', '2:2 XX q;1:2 XX compound', '1:1 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;12:12 XX compound;14:14 XX q;15:15 XX q;15:15 XX compound;0:15 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;9:10 XX proper;12:12 XX q;26:26 XX q;27:27 XX q;28:28 XX times;0:28 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX q;10:11 XX proper;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;14:15 XX udef;16:16 XX q;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;27:27 XX q;29:29 XX q;26:29 XX udef;30:30 XX q;31:31 XX q;32:32 XX q;31:32 XX udef;26:32 XX udef;17:32 XX udef;0:32 XX focus', '1:1 XX comp;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;0:22 XX proper', '17:17 XX comp', '0:0 XX q;2:2 XX q;3:3 XX q;5:5 XX q;7:7 XX q;8:8 XX q;15:16 XX compound;22:23 XX compound;24:24 XX q;25:25 XX q;26:27 XX proper;28:28 XX compound;26:28 XX compound;22:28 XX udef;3:28 XX proper;0:28 XX udef', '4:4 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;9:11 XX udef;12:12 XX q;16:16 XX q;16:17 XX udef;19:19 XX q;19:20 XX udef;22:22 XX q;22:24 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX q;8:8 XX d;7:8 XX compound;6:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;12:13 XX udef;14:14 XX d;12:14 XX compound;15:15 XX d;16:16 XX d;18:18 XX q;18:18 XX compound;17:18 XX proper;17:19 XX proper;21:21 XX d;22:22 XX d;23:23 XX q;24:24 XX d;25:25 XX d;26:26 XX d;27:28 XX compound;29:29 XX d;27:29 XX compound;30:30 XX d;32:32 XX nonsp;23:32 XX udef;17:32 XX compound;0:32 XX focus', '19:19 XX q;20:20 XX q;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;24:24 XX q;26:26 XX q;27:27 XX q;28:28 XX compound;24:28 XX udef;21:28 XX udef', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;6:6 XX q;7:7 XX nonsp;8:8 XX d;7:8 XX loc;10:10 XX d;9:10 XX proper;11:11 XX d;13:13 XX d;15:15 XX d;14:15 XX loc;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX q;25:25 XX d;24:25 XX udef;26:26 XX loc;5:26 XX udef;0:26 XX udef', '1:2 XX compound;7:8 XX proper;15:15 XX q;15:15 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;22:23 XX compound;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;30:31 XX udef;18:31 XX udef;16:31 XX udef;15:31 XX udef;0:31 XX compound', '0:0 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX q;14:14 XX compound;13:14 XX udef;16:16 XX q;15:16 XX compound;17:17 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;20:20 XX compound;20:20 XX udef;0:20 XX focus', '2:2 XX nonsp', '', '', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;15:15 XX q;16:16 XX q;14:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;29:29 XX q;29:29 XX compound;28:29 XX udef;31:31 XX q;30:31 XX compound;32:32 XX q;28:32 XX proper;12:32 XX udef;9:32 XX udef', '10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;26:26 XX q;25:26 XX udef;12:26 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;25:25 XX udef;0:25 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;13:13 XX d;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX d;29:29 XX q;30:30 XX nonsp;31:31 XX q;36:36 XX q;37:37 XX q;0:37 XX focus', '6:6 XX neg', '3:3 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;9:17 XX udef;23:23 XX q;22:25 XX loc;27:27 XX q;27:27 XX udef', '1:1 XX q;3:3 XX q;4:4 XX q;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;11:12 XX udef;13:13 XX q;14:14 XX q;14:14 XX udef;0:14 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;0:24 XX focus', '4:4 XX q;16:16 XX q', '8:8 XX q;9:9 XX q;36:36 XX q;0:36 XX subord', '3:3 XX q;3:3 XX compound;2:3 XX udef;4:5 XX compound;21:21 XX superl;33:33 XX q;34:34 XX q;35:35 XX q;37:37 XX q;37:38 XX udef', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;20:20 XX neg;28:28 XX comp;25:28 XX loc', '0:0 XX q;1:1 XX q;3:3 XX q;4:4 XX q;4:4 XX compound;9:9 XX neg;0:17 XX proper', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:12 XX proper;14:15 XX compound;17:18 XX proper;19:19 XX q;17:19 XX compound;17:19 XX proper;0:19 XX focus', '', '3:3 XX q;4:4 XX q', '1:1 XX q;3:3 XX q;2:3 XX compound;2:4 XX proper;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;11:12 XX compound;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;16:17 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;0:33 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;8:8 XX q;8:8 XX nominalization;12:12 XX q;13:13 XX q;0:29 XX udef', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;14:14 XX q;0:14 XX udef', '6:6 XX q;10:10 XX q;11:11 XX q', '0:0 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;19:19 XX q;0:19 XX udef', '1:1 XX q;3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;8:14 XX udef;0:14 XX udef', '0:0 XX q;7:7 XX q;8:8 XX q;9:10 XX compound;0:17 XX udef', '', '', '', '', '0:0 XX q;0:0 XX compound;1:1 XX q;1:1 XX compound;3:3 XX q;5:5 XX q;5:5 XX compound;6:6 XX q;6:6 XX compound;7:7 XX q;7:7 XX compound;9:9 XX q;10:10 XX q;12:12 XX q;18:18 XX q;19:19 XX q;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX compound;23:23 XX q;23:23 XX compound;24:24 XX q;24:24 XX compound;25:25 XX q;25:25 XX nominalization;22:25 XX udef;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX loc;0:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;6:6 XX q;6:6 XX compound;5:6 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;0:23 XX focus', '', '0:10 XX compound', '11:11 XX q', '5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX q;13:13 XX nominalization;12:13 XX udef', '1:1 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX q;9:9 XX nominalization;10:10 XX q;11:11 XX q;13:13 XX d;10:13 XX udef;14:14 XX q;14:14 XX nominalization;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;26:26 XX d;25:26 XX udef;27:27 XX q;36:36 XX compound;9:36 XX udef;0:36 XX focus', '2:2 XX q;2:2 XX compound;1:2 XX udef;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;7:8 XX compound;32:32 XX q', '']

abs_bert_pred = ['8:8 XX comp;11:11 XX compound', '1:1 XX q;2:2 XX q;15:15 XX superl;18:18 XX q;22:22 XX q;23:23 XX q;0:25 XX compound', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;5:6 XX udef;8:8 XX d;7:8 XX compound;5:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:19 XX compound;18:19 XX compound;0:19 XX focus', '3:4 XX compound;5:6 XX proper;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;0:18 XX appos', '0:0 XX q;0:0 XX compound;0:16 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;0:9 XX focus', '0:0 XX nonsp;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;6:15 XX udef;16:16 XX d;17:17 XX d;18:18 XX d;20:20 XX q;20:21 XX udef;22:22 XX d;20:22 XX compound;23:23 XX d;24:24 XX d;0:24 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;9:9 XX d;8:9 XX proper;13:13 XX q;13:13 XX compound;12:13 XX udef;14:14 XX d;12:14 XX udef;15:15 XX d;16:17 XX proper;18:18 XX q;19:19 XX q;21:21 XX q;20:21 XX compound;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX proper;29:29 XX q;32:32 XX compound;26:32 XX udef;24:32 XX udef;20:32 XX proper;18:32 XX udef;0:32 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX nonsp;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;0:17 XX focus', '19:19 XX q;18:19 XX compound;18:20 XX proper;21:21 XX q;23:23 XX q;22:23 XX proper;24:24 XX q;25:25 XX q;25:25 XX proper;22:25 XX udef;18:25 XX udef;17:25 XX compound', '0:22 XX subord', '8:8 XX q;8:8 XX compound;11:11 XX q;0:19 XX compound', '11:11 XX q;12:12 XX q;21:21 XX q;21:21 XX compound;20:22 XX compound;0:27 XX appos', '10:10 XX q', '1:1 XX d;6:6 XX q;7:7 XX q;6:7 XX udef;4:7 XX compound;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX q;16:16 XX q;17:17 XX q;17:17 XX compound;16:17 XX udef;15:17 XX udef;18:18 XX d;15:18 XX udef;2:18 XX compound;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;27:27 XX q;28:28 XX q;33:33 XX q;33:33 XX nominalization;0:33 XX def', '12:12 XX q;11:12 XX proper;13:13 XX q;14:14 XX q;21:21 XX q', '', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;10:10 XX d;8:10 XX compound;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;31:31 XX q;32:32 XX q;32:32 XX udef;30:32 XX udef;0:32 XX focus', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;0:13 XX udef', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX q;17:17 XX d;18:18 XX q;18:18 XX nominalization;20:20 XX q;20:20 XX nominalization;0:21 XX udef', '0:0 XX q;1:1 XX q;0:7 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;10:10 XX q;21:21 XX q', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;13:13 XX compound;12:13 XX udef;0:13 XX focus', '9:9 XX q;9:9 XX compound;8:9 XX udef;10:11 XX proper;10:12 XX compound', '0:0 XX superl', '4:4 XX q;8:8 XX q;7:8 XX compound;12:12 XX q;12:12 XX compound;11:12 XX proper;11:13 XX udef;25:25 XX q;26:26 XX q;0:26 XX udef', '8:8 XX q;8:8 XX compound', '3:3 XX q;17:17 XX q;17:17 XX nominalization', '3:3 XX q;0:4 XX compound', '4:4 XX q;5:5 XX q;6:6 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;12:12 XX q;12:12 XX compound;11:12 XX udef;11:12 XX udef', '10:10 XX q;12:12 XX q;18:18 XX nonsp;24:24 XX superl;0:28 XX appos', '', '1:1 XX d;2:2 XX d;3:3 XX d;4:5 XX compound;11:12 XX compound;9:12 XX compound;21:21 XX udef;19:21 XX udef;18:21 XX udef;0:21 XX proper', '10:11 XX proper', '7:8 XX compound;9:9 XX q;15:15 XX superl;10:15 XX appos', '0:0 XX conj;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;8:8 XX d;11:11 XX d;10:11 XX proper;14:14 XX d;13:14 XX compound;15:15 XX d;16:16 XX q;17:17 XX q;19:19 XX q;18:20 XX udef;16:20 XX udef;22:22 XX conj;23:23 XX conj;26:27 XX proper;26:27 XX compound;24:27 XX compound;21:27 XX subord;0:27 XX implicit', '10:10 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;15:18 XX udef;33:33 XX q;34:34 XX q;33:34 XX udef', '13:13 XX q;0:17 XX appos', '0:0 XX neg;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX compound;16:17 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;20:20 XX d;0:21 XX focus', '11:11 XX thing', '0:0 XX q;1:1 XX q;16:16 XX q;21:21 XX udef;0:21 XX proper', '6:6 XX q;15:15 XX q;15:15 XX compound;14:15 XX proper', '14:14 XX q;15:15 XX q;15:15 XX compound;14:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX proper;8:8 XX d;6:8 XX proper;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX q;18:18 XX d;19:19 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:27 XX focus', '', '3:3 XX q;0:5 XX proper', '0:0 XX d;1:1 XX d;3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;5:19 XX udef;0:20 XX focus', '16:16 XX q;17:17 XX q;18:18 XX q', '8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;6:6 XX q;7:7 XX udef;0:7 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;17:17 XX d;16:17 XX compound;18:18 XX d;20:20 XX d;23:23 XX d;21:23 XX compound;24:24 XX d;27:27 XX d;26:27 XX proper;21:27 XX focus;15:27 XX compound;0:27 XX focus', '4:4 XX q;7:7 XX q;8:8 XX q', '0:0 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;0:24 XX proper', '0:0 XX neg', '0:0 XX d;4:5 XX proper;7:7 XX q;4:7 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;16:16 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX d;22:25 XX udef;26:26 XX d;27:27 XX q;28:28 XX q;0:28 XX focus', '1:1 XX neg', '10:10 XX comp;13:13 XX comp;17:17 XX q;18:18 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;5:5 XX q;4:5 XX compound;6:6 XX q;8:8 XX q;8:8 XX compound;7:8 XX proper;9:9 XX d;7:9 XX proper;10:10 XX d;11:11 XX d;12:12 XX nonsp;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;0:18 XX focus', '11:11 XX q;12:12 XX q;18:19 XX udef;20:20 XX q;20:21 XX udef;23:23 XX q;23:24 XX udef;20:24 XX udef;18:24 XX udef;17:24 XX udef;27:27 XX q;28:28 XX q;26:28 XX udef;16:28 XX udef', '0:15 XX compound', '1:1 XX q;2:2 XX q;3:4 XX udef;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:14 XX compound', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX q;15:15 XX d;16:16 XX d;17:17 XX q;0:17 XX udef', '6:7 XX compound;10:10 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;20:23 XX udef;16:23 XX udef', '0:0 XX q;0:0 XX compound;1:1 XX q;2:2 XX q;10:10 XX q;11:11 XX q;0:11 XX udef', '2:2 XX q;5:7 XX generic', '5:5 XX q;25:27 XX proper;28:29 XX compound', '4:4 XX q;6:6 XX nonsp;7:8 XX proper;5:8 XX loc;15:15 XX q;16:16 XX q', '', '0:0 XX conj;1:1 XX conj;3:3 XX q;4:4 XX q;4:4 XX compound;3:4 XX udef;2:4 XX udef;5:5 XX conj;2:5 XX udef;6:6 XX conj;7:7 XX conj;0:8 XX implicit', '4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX proper;8:8 XX q;6:9 XX udef;13:13 XX q;12:14 XX udef;15:15 XX conj;17:17 XX conj;18:18 XX conj;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX compound;19:23 XX udef;15:23 XX implicit', '1:1 XX neg;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;14:14 XX q;15:15 XX nonsp;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q', '6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:15 XX udef;11:15 XX udef;9:15 XX udef;17:18 XX udef;19:19 XX neg;29:29 XX nominalization;16:32 XX subord', '6:6 XX q;6:6 XX compound;5:6 XX udef;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:14 XX compound;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;6:7 XX compound;8:8 XX d;10:10 XX conj;10:10 XX unknown;11:11 XX conj;11:11 XX unknown;12:12 XX conj;12:12 XX unknown;13:13 XX conj;13:13 XX unknown;15:15 XX conj;15:15 XX unknown;0:17 XX focus', '0:1 XX proper;0:12 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;17:17 XX q;35:35 XX q;0:35 XX proper', '5:6 XX udef;5:7 XX appos;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper;19:20 XX proper;10:20 XX udef', '', '11:12 XX compound;10:12 XX compound', '0:0 XX q;0:0 XX compound;0:15 XX udef', '4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:7 XX proper;8:8 XX q;8:8 XX nominalization;6:8 XX compound;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;18:18 XX q;18:18 XX nominalization;17:18 XX compound;16:18 XX compound;19:19 XX q;19:19 XX nominalization;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;26:26 XX q;20:26 XX udef', '2:3 XX compound;5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;9:12 XX udef;19:19 XX q', '0:1 XX proper;6:6 XX q;9:9 XX q;8:9 XX compound;7:9 XX compound;10:10 XX q;11:12 XX udef;18:18 XX q;22:22 XX nominalization;0:24 XX compound', '3:3 XX q;2:3 XX compound;2:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q', '20:20 XX compound', '10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;15:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;12:26 XX udef;0:26 XX compound', '0:1 XX udef;7:7 XX neg;0:21 XX compound', '4:4 XX q', '0:0 XX nominalization;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:12 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q', '4:4 XX q', '6:7 XX compound;5:7 XX udef;0:12 XX compound', '5:5 XX q;9:9 XX neg;11:11 XX q;13:13 XX q;15:15 XX q;14:15 XX proper;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;21:22 XX compound;30:30 XX q;32:32 XX q;31:32 XX compound', '', '0:0 XX q;0:0 XX compound;8:8 XX neg;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;0:27 XX proper', '0:12 XX measure', '4:4 XX q;4:4 XX compound;3:4 XX udef;6:6 XX neg;17:17 XX q;17:17 XX compound;16:17 XX proper', '7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;12:20 XX udef', '10:10 XX q;11:11 XX q;12:12 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX udef;22:23 XX udef', '5:5 XX q;6:6 XX q;7:8 XX proper;17:17 XX d;17:18 XX parg', '', '15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;24:24 XX q;23:24 XX compound;25:25 XX q;26:26 XX q;23:26 XX udef;20:26 XX udef', '0:0 XX q;0:0 XX compound;20:20 XX udef;0:20 XX proper', '3:3 XX q;3:3 XX nominalization;4:4 XX q;5:5 XX q;7:7 XX q;4:7 XX udef;3:7 XX udef', '4:4 XX q;4:5 XX udef;0:18 XX compound', '6:6 XX q;15:15 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX compound;14:14 XX d;12:14 XX udef;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;23:23 XX q;22:24 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;31:31 XX q;32:32 XX q;32:33 XX udef;34:34 XX compound;32:34 XX compound;26:34 XX udef;19:34 XX udef;16:34 XX udef;0:34 XX focus', '0:0 XX q;1:1 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;18:18 XX q;0:18 XX udef', '2:2 XX q;9:9 XX q;10:10 XX q;12:12 XX q;11:12 XX proper;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;23:23 XX q;24:24 XX times;22:24 XX udef;26:26 XX q;22:26 XX compound;27:27 XX q;29:29 XX q;28:29 XX proper', '5:5 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q', '3:3 XX q;3:4 XX udef;6:6 XX q;6:7 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;20:20 XX compound;21:21 XX q;21:21 XX compound;22:22 XX q;22:22 XX compound;20:23 XX udef;33:33 XX superl', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:6 XX compound;7:7 XX d;10:10 XX d;0:10 XX focus', '0:0 XX q;0:0 XX compound;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX times;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;33:33 XX q;33:33 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX q;36:36 XX nominalization;37:37 XX q;37:37 XX nominalization;38:38 XX q;38:38 XX nominalization;0:38 XX udef', '3:3 XX comp;4:4 XX q;4:4 XX compound;5:5 XX q;5:5 XX compound;6:6 XX q;6:6 XX compound;8:8 XX q;8:9 XX udef;16:16 XX udef', '2:2 XX q;4:4 XX q;3:4 XX compound;3:5 XX proper;10:11 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:20 XX compound;19:21 XX compound;22:23 XX proper;25:25 XX q;25:26 XX udef;25:27 XX compound;30:31 XX compound', '4:4 XX q;5:5 XX q;6:6 XX q;9:9 XX q;10:10 XX q;9:10 XX udef;7:10 XX eventuality;11:11 XX q;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;15:15 XX udef;12:15 XX udef', '0:0 XX comp', '2:2 XX q;2:2 XX compound;1:2 XX udef;4:5 XX compound;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;0:16 XX appos', '1:1 XX q;0:1 XX proper;2:2 XX q;3:4 XX appos;7:9 XX compound;0:9 XX udef;13:13 XX q;14:14 XX q;16:16 XX q;18:18 XX q;19:19 XX q;18:19 XX udef;20:20 XX q;21:21 XX q;22:22 XX q;17:23 XX udef;30:30 XX q;31:31 XX q;32:32 XX times', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:9 XX compound;0:21 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX compound;20:20 XX q;21:21 XX q;0:27 XX udef', '0:0 XX q;1:1 XX q;4:4 XX q;5:6 XX compound;5:7 XX udef;2:7 XX udef;16:16 XX q;16:16 XX compound;15:16 XX proper;15:17 XX proper;0:22 XX udef', '1:1 XX nominalization;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;27:30 XX udef;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef', '0:0 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX neg;0:23 XX proper', '0:0 XX nonsp;1:1 XX nonsp;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:16 XX loc', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;22:22 XX q;22:22 XX compound;21:22 XX udef;21:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;33:33 XX d;35:35 XX d;36:36 XX d;37:37 XX d;39:39 XX d;0:41 XX focus', '15:15 XX compound', '', '1:1 XX q;0:1 XX compound;2:2 XX q;3:3 XX q;7:7 XX q;8:8 XX q;12:12 XX q;13:13 XX q;15:15 XX q;17:17 XX q;17:17 XX q;16:17 XX udef;12:17 XX udef;7:17 XX udef;4:17 XX eventuality;0:17 XX proper', '9:9 XX q;8:10 XX udef;13:13 XX q;14:15 XX compound;16:16 XX q;14:16 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;24:25 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;27:27 XX compound;0:27 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;34:34 XX q;35:35 XX q;35:35 XX nominalization;37:37 XX q;37:37 XX nominalization;38:38 XX q;38:38 XX nominalization;33:38 XX udef;0:38 XX focus', '8:9 XX compound;7:9 XX udef;10:11 XX proper;13:13 XX q;12:13 XX compound;14:14 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;17:17 XX compound;17:18 XX udef;15:18 XX proper;12:18 XX udef;7:18 XX appos;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;22:29 XX udef', '3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX d;6:7 XX udef;9:9 XX d;11:11 XX d;12:12 XX d', '', '2:2 XX neg;12:12 XX q;12:13 XX udef;15:15 XX neg;34:34 XX comp;40:40 XX q;40:40 XX compound;41:41 XX q;41:41 XX compound;42:42 XX q;42:42 XX compound', '0:1 XX compound;7:7 XX q;6:7 XX proper;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:28 XX compound', '0:0 XX q;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;8:8 XX nominalization;0:11 XX udef', '8:8 XX q;10:10 XX q;11:11 XX q;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q', '3:4 XX udef;9:10 XX proper;12:12 XX q;15:15 XX q;14:15 XX compound;13:15 XX compound;12:15 XX udef;11:15 XX id;16:16 XX q;17:18 XX compound;20:20 XX q;19:20 XX compound;22:22 XX q;23:23 XX q;17:23 XX proper;11:23 XX udef', '8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;14:14 XX q;12:14 XX compound', '9:9 XX q;11:11 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX udef', '0:0 XX q;1:1 XX q;1:1 XX compound;0:5 XX proper', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q', '10:10 XX q;11:11 XX q', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q;18:23 XX udef;14:23 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;0:9 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;13:13 XX q;14:14 XX q;15:15 XX times;27:27 XX q;28:28 XX q;29:29 XX times;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;0:37 XX udef', '0:0 XX q;1:1 XX q;0:16 XX udef', '0:0 XX q;1:1 XX q;3:3 XX q;9:9 XX q;10:10 XX q;10:10 XX compound;9:10 XX udef;7:10 XX udef;16:17 XX compound;0:19 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;6:10 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX conj;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;18:20 XX udef;16:20 XX udef;12:20 XX udef;21:21 XX conj;22:22 XX conj;24:24 XX q;23:24 XX compound;25:25 XX q;26:26 XX q;27:27 XX q;26:28 XX udef;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef;30:33 XX udef;23:33 XX udef;9:33 XX udef;0:33 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:9 XX proper;0:9 XX focus', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;3:4 XX udef;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX d;17:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;5:25 XX q', '0:0 XX q;0:5 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;9:9 XX conj;10:10 XX thing;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;16:16 XX conj;18:18 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX conj;23:23 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX q;0:29 XX implicit', '', '0:-1 XX EMPTY-LABEL', '', '5:5 XX comp', '4:4 XX q;5:5 XX q', '', '10:10 XX comp;0:23 XX subord', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;3:18 XX udef', '0:0 XX d;1:1 XX q;1:1 XX nominalization;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX q;10:10 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:21 XX udef;22:22 XX q;20:22 XX compound;19:22 XX udef;23:23 XX q;25:25 XX q;26:26 XX q;26:26 XX compound;25:26 XX udef;24:26 XX udef;27:27 XX compound;24:27 XX udef;19:27 XX udef;0:27 XX focus', '1:1 XX q;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;9:9 XX conj;10:10 XX conj;19:19 XX conj;21:21 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX conj;30:30 XX conj;32:32 XX conj;33:33 XX conj;35:35 XX conj;36:36 XX conj;37:37 XX conj;38:38 XX conj;0:38 XX proper', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX compound', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;0:14 XX implicit', '0:0 XX q;6:6 XX conj;7:7 XX conj;8:8 XX conj;0:8 XX udef', '4:4 XX q', '0:1 XX proper;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX proper;11:12 XX proper;13:13 XX d;11:13 XX compound;6:13 XX udef;2:13 XX udef;15:15 XX d;16:16 XX d;0:16 XX appos', '2:2 XX comp', '11:10 XX very;11:11 XX XX', '', '3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:12 XX you;13:13 XX d;13:13 XX XX;15:15 XX d', '0:1 XX proper;6:6 XX q;7:7 XX q;8:8 XX q;16:16 XX neg;0:28 XX compound', '0:19 XX proper', '', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;0:13 XX implicit', '', '1:1 XX d;3:3 XX d;5:5 XX q;4:5 XX compound;6:6 XX q;0:18 XX focus', '9:10 XX compound', '', '3:3 XX times;2:3 XX udef;1:3 XX implicit;8:8 XX q;9:10 XX proper;15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper;17:17 XX q;6:17 XX appos;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:36 XX udef', '15:17 XX compound;31:31 XX q;30:31 XX udef;29:31 XX udef;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX compound;33:36 XX udef', '0:15 XX subord', '0:0 XX q;17:17 XX q;0:27 XX udef', '9:9 XX q;10:10 XX q', '11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;14:15 XX udef;22:22 XX q;21:23 XX udef;26:26 XX q;26:26 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;32:32 XX q;31:32 XX udef;25:32 XX udef;16:32 XX udef', '2:2 XX q;3:3 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX proper;8:8 XX q;6:8 XX proper;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX compound', '13:13 XX q;14:14 XX q;19:20 XX compound;23:23 XX q;24:24 XX q', '1:1 XX neg;8:8 XX q;8:9 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;17:17 XX q;22:22 XX d;21:22 XX udef;23:23 XX d;24:24 XX d;0:24 XX udef', '21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;29:29 XX measure;28:29 XX compound', '0:0 XX q;0:1 XX udef;0:11 XX compound', '25:25 XX q;26:26 XX q;27:27 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX d;0:18 XX focus', '3:3 XX q;4:5 XX proper;3:5 XX udef;2:6 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;10:17 XX udef;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;20:24 XX udef;18:24 XX subord;8:24 XX udef', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;4:6 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;8:11 XX udef;4:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;18:18 XX q;17:18 XX loc;24:24 XX q;25:25 XX q;24:26 XX udef;27:27 XX q;28:28 XX q;29:29 XX q;28:31 XX udef;24:31 XX udef;32:32 XX nonsp;33:33 XX nonsp;32:33 XX loc;19:33 XX subord', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX times;6:8 XX udef;10:10 XX q;6:10 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX compound;13:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX times;18:20 XX udef;22:22 XX q;18:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX compound;24:25 XX udef;24:25 XX udef', '0:0 XX q;1:1 XX q;7:7 XX q;8:8 XX q;0:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX conj;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX conj;11:14 XX udef;7:14 XX udef;15:15 XX conj;16:16 XX conj;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;23:26 XX udef;19:26 XX udef;27:27 XX nonsp;28:28 XX nonsp;27:28 XX loc;17:28 XX udef;5:28 XX udef;0:28 XX focus', '3:3 XX q;4:4 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;0:12 XX focus', '7:7 XX q', '0:1 XX proper;8:8 XX q;0:8 XX compound', '2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;22:22 XX q;23:23 XX number', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX q;10:11 XX udef;12:12 XX q;13:13 XX udef;0:13 XX focus', '23:23 XX q;0:27 XX proper', '3:3 XX neg;12:12 XX neg', '0:0 XX q;0:0 XX compound;0:0 XX unknown;0:0 XX proper', '7:7 XX neg', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX d;13:14 XX compound;9:14 XX udef;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX q;33:33 XX q;35:35 XX q;35:35 XX compound;34:35 XX proper;34:35 XX udef;0:35 XX udef', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;5:5 XX nominalization;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;11:11 XX nominalization;10:11 XX udef;6:11 XX udef;12:12 XX q;12:12 XX nominalization;13:13 XX q;14:14 XX q;15:15 XX q;15:15 XX udef;13:15 XX udef;0:15 XX focus', '13:13 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;14:14 XX q;15:15 XX compound;0:15 XX focus', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;7:7 XX comp;0:8 XX udef', '0:0 XX neg;10:10 XX d;12:12 XX d;11:12 XX udef', '11:11 XX q;12:14 XX udef;0:14 XX measure', '3:3 XX superl;17:17 XX q;17:18 XX udef', '0:0 XX nonsp;1:1 XX d;0:1 XX loc;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX compound;9:13 XX focus;0:13 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX conj;14:14 XX conj;0:27 XX focus', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;10:10 XX conj;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;15:17 XX compound;11:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;30:30 XX comp;22:30 XX appos;19:30 XX udef;0:30 XX implicit', '13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;0:25 XX udef', '4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;6:7 XX udef;10:10 XX q;23:23 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX conj;30:30 XX conj;31:31 XX conj;32:32 XX conj;35:35 XX q;36:36 XX q;37:37 XX q;39:39 XX q', '1:1 XX q;12:12 XX q;11:12 XX udef;13:13 XX q;13:14 XX udef;16:16 XX q;16:16 XX udef;13:16 XX udef;0:16 XX proper', '10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;18:18 XX q;19:19 XX q;19:19 XX nominalization;18:19 XX udef;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization', '0:6 XX proper', '6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;21:21 XX compound;10:21 XX udef', '9:9 XX q;8:9 XX udef;10:11 XX udef;13:14 XX udef;10:14 XX udef', '6:7 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;23:23 XX d;26:26 XX d;0:26 XX focus', '12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;17:17 XX pron;19:19 XX q;19:19 XX pron', '9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;14:16 XX udef;17:18 XX udef;19:20 XX udef;17:20 XX udef', '13:13 XX nonsp', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;0:9 XX udef', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX d;0:11 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX nonsp;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;0:23 XX focus', '3:3 XX q;4:4 XX q;7:7 XX q', '7:8 XX compound;9:10 XX proper;19:19 XX q;19:19 XX compound;18:19 XX proper;20:20 XX q;20:20 XX compound;18:20 XX proper;22:22 XX q;21:22 XX udef;18:22 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:12 XX compound;11:13 XX compound;0:24 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX neg;0:7 XX udef', '4:4 XX q;2:4 XX udef;15:15 XX q;17:17 XX q;17:17 XX compound;16:17 XX udef;19:19 XX q;20:21 XX compound;19:21 XX udef;18:21 XX implicit;24:24 XX q;24:24 XX compound;23:24 XX proper;25:25 XX q;25:25 XX compound;23:25 XX proper;23:25 XX proper;16:25 XX udef;0:25 XX unknown', '0:0 XX neg', '0:7 XX compound', '2:3 XX compound;1:3 XX udef;4:4 XX q;5:5 XX q;6:6 XX udef;4:6 XX udef', '1:1 XX q;1:1 XX compound;0:1 XX proper;0:2 XX proper;3:3 XX conj;3:3 XX unknown;4:4 XX conj;4:4 XX unknown;5:5 XX conj;5:5 XX unknown;6:6 XX conj;6:6 XX unknown;7:7 XX q;9:9 XX q;11:11 XX conj;11:11 XX unknown;10:11 XX udef;12:12 XX conj;12:12 XX unknown;13:13 XX conj;13:13 XX unknown;14:14 XX conj;14:14 XX unknown;15:15 XX conj;15:15 XX unknown;16:16 XX conj;16:16 XX unknown;17:17 XX conj;17:17 XX unknown;20:20 XX q;20:20 XX udef;0:20 XX compound', '16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX nominalization;20:21 XX udef;24:24 XX q;26:26 XX q;27:27 XX q', '0:0 XX nominalization', '0:0 XX neg;7:7 XX q;8:8 XX nominalization;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX udef', '5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;14:14 XX q;14:15 XX udef;17:17 XX compound', '7:7 XX q;8:8 XX nominalization;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX udef', '3:3 XX q;4:4 XX nominalization;6:6 XX q;7:7 XX q;8:8 XX q;19:19 XX neg', '8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '8:8 XX q;8:8 XX nominalization;9:9 XX q;10:10 XX q;9:11 XX udef', '1:1 XX neg;3:3 XX q;4:4 XX q;6:6 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;14:14 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;18:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;0:21 XX focus', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;14:14 XX neg;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX q;0:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX q;17:17 XX q;16:17 XX udef;15:17 XX implicit;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:27 XX focus', '0:0 XX q;4:4 XX q;4:4 XX nominalization;5:5 XX q;7:7 XX q;7:7 XX nominalization;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX udef;0:11 XX udef', '5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;18:18 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX compound;14:28 XX udef;10:28 XX udef;8:28 XX udef', '8:8 XX q;9:9 XX nominalization;13:13 XX neg;21:21 XX q;21:21 XX compound;0:22 XX compound', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:6 XX udef;7:7 XX d;8:8 XX d;0:8 XX focus', '3:3 XX q;5:5 XX q;8:8 XX q;7:9 XX udef;11:11 XX q;11:11 XX compound;13:13 XX q;13:13 XX compound;11:13 XX udef;11:14 XX udef;15:15 XX q;16:16 XX q;0:16 XX compound', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:17 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;9:9 XX q;11:11 XX q;12:12 XX q;14:14 XX neg;0:16 XX udef', '5:5 XX neg;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;10:12 XX udef;0:12 XX compound', '0:0 XX d;3:3 XX nonsp;4:4 XX d;6:6 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:15 XX focus', '2:2 XX neg', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;3:4 XX loc;5:6 XX proper;12:12 XX d;5:12 XX appos;13:13 XX d;15:15 XX d;14:15 XX udef;16:16 XX d;17:17 XX q;17:18 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;0:21 XX focus', '1:1 XX thing', '2:2 XX q', '0:1 XX compound;7:7 XX q;0:11 XX compound', '', '0:0 XX q;0:18 XX proper', '0:0 XX q;0:18 XX proper', '1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:8 XX proper;13:13 XX q;12:13 XX compound;14:14 XX q;12:15 XX udef;17:18 XX udef;17:18 XX compound;0:18 XX focus', '0:0 XX q;0:6 XX proper', '1:1 XX q;2:2 XX q;3:3 XX q;4:5 XX udef;8:8 XX q', '', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX proper', '6:6 XX q;5:7 XX udef', '0:0 XX q;1:1 XX q;0:21 XX proper', '', '6:7 XX udef;9:9 XX q;9:9 XX compound;11:11 XX compound;9:11 XX udef', '', '0:0 XX neg', '', '2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;4:4 XX nominalization;4:5 XX udef;10:10 XX q;12:12 XX comp;0:26 XX subord', '2:2 XX q;3:3 XX q;2:3 XX udef;4:4 XX q;7:8 XX proper;9:9 XX d;7:9 XX compound;5:9 XX appos;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX compound;0:17 XX focus', '1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;18:19 XX compound;0:21 XX focus', '0:0 XX q;1:1 XX q;5:5 XX q;6:6 XX q;7:7 XX q;0:8 XX udef', '', '', '10:10 XX neg', '0:0 XX d;1:1 XX d;2:3 XX proper;7:7 XX d;6:7 XX compound;5:7 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;0:15 XX focus', '6:6 XX q', '8:8 XX q;11:11 XX q;12:12 XX compound', '0:13 XX compound', '12:13 XX compound;16:16 XX q;17:18 XX proper;19:19 XX q;17:19 XX compound;20:20 XX q;22:22 XX q;24:24 XX q;24:24 XX compound;23:24 XX proper;23:24 XX proper', '6:6 XX nonsp;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q', '4:4 XX q;4:4 XX nominalization', '0:0 XX d;2:2 XX q;3:3 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:18 XX focus', '5:5 XX comp;15:15 XX comp;16:16 XX comp', '9:9 XX q;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;0:18 XX subord', '2:2 XX q;1:2 XX compound;3:3 XX q;17:17 XX q;0:22 XX appos', '0:0 XX q;1:2 XX compound;0:4 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q;5:5 XX q;7:8 XX compound;6:8 XX proper;0:22 XX appos', '2:2 XX q;3:3 XX q;5:5 XX q;4:5 XX compound;4:6 XX proper;11:11 XX q;10:12 XX udef;15:15 XX neg;0:20 XX appos', '0:0 XX d;1:1 XX d;3:3 XX d;0:12 XX focus', '0:0 XX q;0:0 XX unknown;0:0 XX udef', '', '10:10 XX q;10:11 XX measure;10:11 XX udef;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;29:30 XX proper;29:31 XX compound;33:33 XX compound', '', '1:1 XX q;1:1 XX nominalization;2:2 XX q;0:5 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;6:6 XX conj;8:8 XX conj;10:10 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX q;14:14 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX compound;16:20 XX udef;14:20 XX udef;0:20 XX implicit', '', '0:0 XX d;2:2 XX d;5:5 XX d;4:5 XX compound;3:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;13:14 XX compound;15:15 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;0:22 XX focus', '5:5 XX q;6:6 XX q;5:9 XX udef', '8:9 XX compound', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;0:18 XX focus', '5:5 XX comp;21:21 XX proper', '0:0 XX d;2:2 XX d;0:2 XX focus', '0:0 XX conj;1:1 XX conj;2:2 XX conj;4:4 XX conj;6:6 XX conj;9:9 XX nonsp;10:10 XX nonsp;11:11 XX conj;12:12 XX conj;13:13 XX q;14:14 XX q;18:18 XX conj;0:32 XX implicit', '1:1 XX neg', '5:5 XX comp', '4:4 XX q;5:5 XX q;13:14 XX udef;12:14 XX udef', '0:28 XX compound', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX compound;22:25 XX udef;27:27 XX conj;26:27 XX compound;28:28 XX conj;0:28 XX implicit', '2:2 XX neg', '9:9 XX nonsp;10:10 XX nonsp;11:11 XX nonsp;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:20 XX compound;27:27 XX q', '3:3 XX superl;5:5 XX superl', '9:9 XX person', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;7:7 XX conj;9:9 XX q;10:10 XX q;10:11 XX udef;0:11 XX implicit', '18:18 XX q;18:19 XX udef;0:19 XX proper', '0:0 XX conj;1:1 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;14:14 XX conj;15:15 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;20:20 XX conj;21:21 XX q;22:22 XX q;21:23 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;25:27 XX udef;0:28 XX implicit', '18:18 XX q;17:18 XX proper;19:19 XX q;20:20 XX proper;0:20 XX proper', '', '', '1:1 XX q;2:2 XX q;4:4 XX q;6:6 XX q;5:6 XX udef;1:6 XX udef;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;13:13 XX q;12:14 XX udef;24:24 XX q;25:25 XX q;26:26 XX times;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q', '', '0:0 XX d;1:1 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;0:17 XX focus', '6:6 XX q;7:7 XX times;17:17 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;13:13 XX q', '0:0 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX conj;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX implicit', '', '', '6:6 XX q;7:7 XX q;9:9 XX q;12:12 XX q;13:13 XX q;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization', '1:1 XX q', '6:7 XX compound;9:9 XX neg;32:32 XX q;39:39 XX q;40:40 XX q;41:41 XX q;42:42 XX q;43:43 XX q;44:44 XX q;45:45 XX q', '', '', '4:4 XX q;4:4 XX nominalization', '1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:11 XX focus', '2:2 XX q;3:3 XX q;2:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;6:10 XX udef;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;12:16 XX udef', '0:0 XX verb;6:6 XX q;7:7 XX times;10:10 XX q;10:10 XX nominalization;11:11 XX q;12:12 XX q;12:12 XX nominalization;11:12 XX udef;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;16:16 XX q;18:18 XX q;18:18 XX compound;19:19 XX q;19:19 XX nominalization;17:19 XX udef;15:19 XX udef;23:23 XX q;23:23 XX nominalization;21:23 XX loc;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:28 XX udef;29:29 XX q;29:29 XX nominalization;27:29 XX compound;20:31 XX loc;0:31 XX generic', '', '0:0 XX q;0:6 XX udef', '2:3 XX loc;10:11 XX compound;12:12 XX proper', '0:0 XX neg;10:11 XX proper', '1:1 XX d;0:1 XX loc;3:3 XX d;2:3 XX proper;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX conj;12:12 XX conj;13:13 XX conj;16:16 XX conj;14:16 XX compound;17:17 XX conj;18:18 XX conj;19:19 XX conj;21:21 XX q;22:22 XX q;20:23 XX udef;0:23 XX focus', '2:2 XX q;3:3 XX q;7:7 XX superl;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;11:14 XX udef;4:14 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '', '1:1 XX q;2:2 XX q;1:2 XX udef;3:3 XX q;6:6 XX q;5:6 XX compound;4:6 XX udef;9:9 XX q', '17:17 XX q', '', '0:9 XX appos', '0:0 XX q;1:1 XX q;1:2 XX udef;10:10 XX comp;15:15 XX q;0:15 XX compound;0:15 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX conj;12:12 XX q;12:12 XX compound;11:12 XX proper;20:21 XX proper;23:24 XX compound;23:24 XX compound;0:24 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX q;20:20 XX q;21:22 XX compound;20:22 XX udef;19:22 XX id;24:24 XX q;24:24 XX compound;23:24 XX proper;23:24 XX proper;17:24 XX appos;0:24 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:20 XX compound;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX q;31:31 XX q;0:31 XX focus', '2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;19:19 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX focus', '0:0 XX neg', '19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX conj;24:24 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;34:34 XX q;36:36 XX q;35:36 XX compound;38:38 XX q;39:39 XX proper;29:39 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;21:21 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;31:31 XX q;31:31 XX nominalization;0:32 XX implicit', '21:21 XX d;23:24 XX proper;25:25 XX q;26:26 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;12:12 XX d;15:15 XX q;16:16 XX q;17:17 XX p;18:18 XX p;19:19 XX p;17:19 XX with;0:19 XX focus', '', '', '0:0 XX d;2:2 XX d;3:3 XX q;3:3 XX nominalization;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:17 XX proper;0:17 XX focus', '', '', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:17 XX proper;20:20 XX superl;23:23 XX q;23:23 XX nominalization;21:23 XX compound;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;28:28 XX d;29:29 XX d;30:30 XX d;32:32 XX d;33:33 XX q;33:33 XX nominalization;34:34 XX q;34:34 XX nominalization;36:36 XX d;0:36 XX proper', '', '5:5 XX q;6:6 XX q;7:7 XX times;5:7 XX udef;8:9 XX compound;8:10 XX compound;14:14 XX q;15:15 XX q;16:16 XX times;14:16 XX udef;22:23 XX compound;0:24 XX proper', '2:2 XX q', '1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;15:15 XX q;16:16 XX conj;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:24 XX loc', '7:7 XX q;8:8 XX q;9:9 XX times;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;14:14 XX q;14:14 XX nominalization;17:17 XX q;17:17 XX nominalization', '15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q', '0:0 XX q;2:2 XX q;3:3 XX q;4:4 XX q;1:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:22 XX udef', '10:10 XX q;11:11 XX q;12:12 XX q;18:18 XX q', '0:0 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;21:21 XX neg;0:23 XX udef', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:10 XX proper;14:14 XX q;15:15 XX q;15:15 XX compound;15:16 XX udef;14:16 XX proper;18:19 XX proper;18:19 XX compound;14:19 XX appos;0:19 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;11:11 XX q;12:12 XX q;13:13 XX times;14:14 XX neg', '8:8 XX q;7:8 XX proper;9:9 XX q;10:11 XX proper;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX udef', '4:4 XX q;14:14 XX q;14:14 XX named;18:18 XX q;21:21 XX q;22:22 XX q;22:22 XX compound;0:23 XX appos', '0:23 XX proper', '11:11 XX q;11:11 XX named;11:12 XX proper;17:17 XX q;18:19 XX proper;16:20 XX udef;22:22 XX proper', '0:0 XX nonsp;2:2 XX nonsp;4:4 XX nonsp;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX q;0:14 XX loc', '', '3:3 XX q;5:5 XX q;6:6 XX q;7:8 XX compound;16:16 XX q', '4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX udef', '1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;13:14 XX proper;13:15 XX compound;18:18 XX d;19:19 XX d;0:19 XX proper', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization', '0:0 XX d;1:1 XX d;3:3 XX d;0:4 XX focus', '3:3 XX comp;7:7 XX nominalization;15:15 XX superl;19:19 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;7:7 XX q;8:8 XX q;0:11 XX proper', '0:0 XX d;1:1 XX d;3:3 XX d;5:5 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;10:10 XX nominalization;12:12 XX q;12:12 XX nominalization;11:12 XX proper;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX d;10:16 XX udef;18:18 XX d;19:19 XX d;0:19 XX focus', '6:6 XX q;6:6 XX compound;16:16 XX q;23:23 XX q;23:23 XX compound;0:23 XX appos', '', '3:3 XX q;3:3 XX nominalization;15:15 XX q;14:15 XX generic;16:16 XX q;17:17 XX q;18:18 XX proper', '8:8 XX q;8:8 XX named', '17:17 XX comp;20:20 XX q', '7:7 XX q;8:8 XX q;9:9 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;0:5 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX times;15:17 XX udef;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:25 XX focus', '3:3 XX q;4:4 XX q;4:4 XX compound;3:4 XX udef;5:6 XX proper;5:7 XX compound;12:12 XX q;12:12 XX named;14:14 XX neg', '0:0 XX q;2:2 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX q;15:15 XX q;15:16 XX proper;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;13:26 XX udef;0:26 XX proper', '0:0 XX conj;1:1 XX conj;2:2 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;15:15 XX d;16:16 XX conj;17:17 XX conj;21:21 XX neg;24:24 XX q;25:25 XX q;24:26 XX udef;32:32 XX q;34:34 XX q;32:34 XX udef;0:34 XX implicit', '2:2 XX q;7:8 XX compound;26:26 XX comp;28:28 XX q;28:28 XX udef', '3:3 XX neg', '7:7 XX entity;8:8 XX entity;16:16 XX neg;18:18 XX q;19:19 XX q;21:21 XX q;20:21 XX proper;23:23 XX q;0:23 XX compound', '1:1 XX d;2:2 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:12 XX proper;17:18 XX compound;17:19 XX compound;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;33:33 XX q;34:34 XX q;36:36 XX d;37:37 XX d;38:38 XX q;39:39 XX q;42:42 XX q;43:43 XX q;42:44 XX udef;40:44 XX generic;45:45 XX q;47:47 XX q;48:48 XX q;38:48 XX udef;0:48 XX focus', '3:3 XX q', '8:8 XX q;17:18 XX compound', '0:0 XX q;1:1 XX conj;2:2 XX conj;3:3 XX q;3:3 XX pron;5:5 XX q;5:5 XX pron;7:7 XX q;7:7 XX pron;0:7 XX udef', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX q;6:6 XX conj;5:6 XX compound;7:7 XX conj;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX q;14:14 XX conj;13:14 XX udef;10:14 XX udef;15:15 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;22:22 XX conj;21:22 XX proper;23:23 XX conj;24:24 XX q;24:24 XX pron;24:25 XX pronoun;0:25 XX implicit', '3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;24:24 XX q;25:25 XX q;26:26 XX q;26:26 XX named;26:26 XX proper', '', '', '8:8 XX q;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;14:14 XX q;13:14 XX proper;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;8:21 XX udef', '1:1 XX q;2:2 XX q;3:3 XX q', '2:2 XX times;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q', '5:5 XX q;4:5 XX compound;8:9 XX proper;0:17 XX appos', '', '0:0 XX d;1:1 XX d;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;0:22 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d', '1:1 XX neg;10:10 XX q;11:11 XX q;12:12 XX times', '', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization', '0:0 XX q;0:0 XX compound;18:18 XX q;0:18 XX proper', '0:0 XX q;1:1 XX q;0:18 XX proper', '0:0 XX d;3:3 XX d;5:5 XX nonsp;6:6 XX nonsp;7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;12:12 XX d;8:12 XX loc;0:13 XX focus', '12:12 XX q;12:12 XX nominalization;13:13 XX q;14:14 XX q', '0:0 XX q;13:13 XX conj;13:13 XX unknown;14:14 XX conj;14:14 XX unknown;16:16 XX conj;16:16 XX unknown;17:17 XX conj;17:17 XX unknown;19:19 XX conj;19:19 XX unknown;20:20 XX conj;20:20 XX unknown;21:21 XX conj;21:21 XX unknown;22:22 XX conj;22:22 XX unknown;25:25 XX conj;25:25 XX unknown;26:26 XX conj;26:26 XX unknown;28:28 XX conj;28:28 XX unknown;29:29 XX conj;29:29 XX unknown;31:31 XX conj;31:31 XX unknown;33:33 XX conj;33:33 XX unknown;0:34 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;5:5 XX q;10:10 XX comp;12:12 XX q;15:15 XX q;15:15 XX named;0:15 XX compound;0:15 XX udef', '8:8 XX q;8:8 XX named;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX proper;19:19 XX q', '', '3:3 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;0:11 XX udef', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;7:8 XX udef;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;17:17 XX nominalization;16:18 XX udef;13:18 XX udef;20:20 XX d;21:21 XX d;22:22 XX d;10:22 XX udef;0:22 XX focus', '1:1 XX neg;4:4 XX neg;10:10 XX neg', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;16:16 XX q;17:17 XX d;16:17 XX udef;18:18 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;27:27 XX d;28:28 XX d;0:28 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX person;22:22 XX q;0:23 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX person;22:22 XX q;0:23 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:8 XX proper;14:14 XX d;15:15 XX d;26:26 XX neg;28:28 XX compound;0:28 XX focus', '2:2 XX q;5:5 XX q', '', '7:7 XX q', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX nonsp;13:13 XX d;0:14 XX focus', '', '', '15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;0:16 XX appos', '11:11 XX q;11:12 XX udef', '0:0 XX neg;1:1 XX q;6:6 XX neg', '8:9 XX generic;10:10 XX q;11:11 XX proper', '4:4 XX q;6:6 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:13 XX focus', '', '0:0 XX d;1:2 XX compound;3:3 XX d;1:3 XX udef;5:5 XX d;4:5 XX proper;7:7 XX d;8:8 XX d;0:8 XX focus', '17:17 XX q;18:18 XX q;19:19 XX q;28:28 XX compound;20:28 XX generic', '0:0 XX q;1:1 XX q;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX d;11:11 XX d;0:23 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX conj;5:5 XX conj;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX q;0:14 XX udef', '', '3:3 XX q', '14:14 XX superl', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;0:35 XX focus', '0:0 XX q;0:0 XX compound;8:8 XX q;9:9 XX q;14:14 XX q;15:15 XX q;16:16 XX q;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;0:38 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;0:13 XX focus', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;10:10 XX q;7:11 XX udef;12:12 XX q;13:13 XX q;14:15 XX compound;13:15 XX udef;18:18 XX neg;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization', '1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX d;7:7 XX d;8:8 XX d;14:14 XX neg;0:20 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;11:11 XX q;0:26 XX proper', '14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX p;6:6 XX d;8:8 XX q;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:19 XX compound;0:19 XX focus', '1:1 XX d;3:3 XX q;2:3 XX proper;4:4 XX q;6:6 XX d;2:6 XX udef;7:7 XX d;8:8 XX d;0:8 XX focus', '2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;5:5 XX q;5:5 XX compound;6:6 XX q;5:6 XX udef;7:7 XX q;8:9 XX proper;4:9 XX proper;19:19 XX q;19:19 XX compound;18:19 XX proper;20:21 XX compound;0:24 XX appos', '12:12 XX q;12:12 XX named;14:14 XX neg', '', '1:1 XX q;0:1 XX proper;2:2 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;12:12 XX q;12:12 XX compound;0:18 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;0:6 XX focus', '', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;0:29 XX focus', '', '', '0:0 XX q;0:0 XX pron;1:1 XX q;1:1 XX pron;2:2 XX q;2:2 XX pron;3:3 XX q;3:3 XX pron;4:4 XX q;4:4 XX pron;0:6 XX pronoun', '10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d', '0:0 XX neg;3:3 XX q;4:4 XX udef', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;0:12 XX udef', '16:16 XX q;17:17 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;0:5 XX udef', '3:3 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX compound', '0:0 XX q;0:7 XX udef', '2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;19:20 XX udef;17:20 XX udef;0:20 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:26 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:26 XX udef', '2:2 XX q;2:2 XX compound;1:2 XX proper;1:3 XX proper;1:4 XX compound;13:14 XX proper;22:22 XX q;21:22 XX proper', '15:15 XX q', '0:0 XX q;0:0 XX compound;5:5 XX q;18:18 XX q;0:26 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;13:13 XX d;15:15 XX d;16:16 XX d;0:16 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;19:19 XX udef;17:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX q;1:2 XX proper;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX compound;0:12 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;0:7 XX udef', '', '1:1 XX d;4:4 XX d;6:6 XX d;7:7 XX conj;9:9 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;15:15 XX q;16:16 XX q;0:17 XX focus', '', '0:0 XX q;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;13:14 XX compound;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;0:21 XX proper', '0:0 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;18:18 XX q;19:19 XX q;19:19 XX compound;0:19 XX proper', '0:1 XX udef;5:5 XX q;6:6 XX times;0:8 XX compound', '3:3 XX q;2:3 XX compound;4:4 XX q;4:4 XX compound;4:5 XX udef;13:13 XX q;13:13 XX compound;12:14 XX udef', '10:10 XX q;13:13 XX q', '4:4 XX q;5:5 XX q', '10:10 XX q;11:11 XX q;14:14 XX nonsp;13:14 XX loc;0:14 XX compound', '0:1 XX udef;6:6 XX q;9:9 XX q;0:15 XX compound', '0:0 XX q;0:0 XX compound;6:6 XX q;7:7 XX q;0:10 XX udef', '6:6 XX q;7:7 XX q;10:10 XX nonsp;9:10 XX loc;0:10 XX compound', '0:0 XX q;5:5 XX q;5:5 XX compound;28:28 XX q;30:30 XX q;33:33 XX q;32:33 XX q;0:33 XX proper', '11:12 XX udef;10:13 XX compound;15:15 XX q;16:16 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;5:5 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;11:13 XX udef;15:15 XX d;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;18:22 XX udef;0:22 XX focus', '6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX udef;6:12 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;12:12 XX q;13:13 XX q;14:14 XX q;0:19 XX udef', '6:6 XX q;6:6 XX compound;5:6 XX number;7:8 XX compound;7:9 XX compound', '0:1 XX proper;2:2 XX q;3:3 XX q;6:6 XX q;6:6 XX compound;5:6 XX proper;7:7 XX q;7:7 XX compound;5:7 XX proper;5:8 XX proper;13:13 XX q;14:14 XX q;15:15 XX q;18:18 XX q;17:19 XX udef;22:22 XX q;17:22 XX compound;23:23 XX q;28:28 XX q;29:29 XX q;31:31 XX q;31:31 XX compound;0:31 XX compound', '2:2 XX q;3:4 XX compound;7:7 XX q;8:8 XX q;9:9 XX times;14:14 XX q;15:15 XX q', '7:7 XX q;8:8 XX q;10:10 XX q;0:10 XX appos', '0:0 XX q;0:0 XX compound;5:5 XX q;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;0:19 XX proper', '0:0 XX d;2:2 XX q;1:2 XX compound;3:3 XX q;4:4 XX q;3:4 XX measure;3:4 XX udef;5:5 XX d;6:6 XX d;10:10 XX q;10:10 XX compound;9:10 XX proper;9:11 XX proper;9:12 XX appos;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:17 XX focus', '3:3 XX q;5:5 XX compound;4:5 XX compound', '15:15 XX q', '1:1 XX q;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX times;16:16 XX p;17:17 XX p;19:19 XX q;20:20 XX q;22:22 XX p;23:23 XX p;0:23 XX compound', '4:4 XX q;5:5 XX q;6:6 XX times', '0:0 XX q;6:6 XX q;10:10 XX q;11:11 XX q;12:12 XX times;17:17 XX q;17:17 XX compound;0:18 XX proper', '0:0 XX q;3:3 XX q;3:3 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;29:29 XX nonsp;30:30 XX nonsp;0:30 XX proper', '0:0 XX superl;7:7 XX q;8:8 XX times;10:10 XX q;11:11 XX q', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '10:10 XX q;10:10 XX udef', '0:1 XX number;3:4 XX compound;5:5 XX q;7:7 XX q;17:17 XX q;18:18 XX q;17:19 XX udef;20:20 XX q;21:21 XX q;22:22 XX q;21:23 XX udef;0:23 XX compound', '1:1 XX q;0:1 XX compound;11:11 XX q;12:12 XX q;11:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;15:18 XX udef;20:20 XX q;21:21 XX q;20:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;24:26 XX udef;0:26 XX proper', '0:0 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:13 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:15 XX proper;17:17 XX q;16:17 XX compound;18:18 XX q;20:20 XX q;19:20 XX proper;21:21 XX q;19:22 XX udef;23:23 XX q;19:23 XX proper;16:23 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;0:33 XX focus', '2:3 XX number;4:5 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;24:24 XX udef;21:24 XX udef;14:24 XX udef', '0:1 XX udef;12:12 XX q;13:13 XX q;0:14 XX compound', '20:20 XX superl', '1:1 XX q;1:1 XX compound;0:1 XX proper;0:2 XX proper;4:4 XX q;6:7 XX compound;5:7 XX compound;4:7 XX udef;3:7 XX udef;0:15 XX appos', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX udef', '4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX q;12:12 XX q;13:13 XX d;12:13 XX udef;10:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;20:20 XX q;19:20 XX compound;18:20 XX compound', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX d;13:13 XX d;14:14 XX q;0:14 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;0:4 XX unknown;0:4 XX udef', '2:2 XX q;2:2 XX nominalization;3:4 XX compound;5:5 XX q;5:5 XX nominalization;3:5 XX compound;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;16:16 XX q;15:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;18:19 XX udef', '10:11 XX proper;15:16 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:4 XX compound;5:5 XX neg;0:6 XX udef', '0:0 XX q;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX udef', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;14:14 XX q;13:14 XX compound;15:15 XX q;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX times;21:23 XX udef;24:24 XX compound;21:24 XX compound;16:24 XX proper;0:24 XX compound', '13:13 XX q;14:14 XX q;13:14 XX udef;16:16 XX q;16:17 XX udef;13:17 XX udef;13:18 XX compound;19:19 XX loc', '13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;17:17 XX compound;16:18 XX udef;19:19 XX q;19:19 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;35:35 XX q', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;6:6 XX nominalization;2:6 XX udef;7:7 XX q;7:7 XX nominalization;9:9 XX q;9:9 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;0:23 XX udef', '0:0 XX q;1:1 XX q;11:12 XX compound;13:13 XX q;11:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;0:25 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:14 XX proper;15:15 XX q;16:16 XX q;18:18 XX q;18:19 XX udef;18:20 XX compound;22:22 XX q;22:22 XX compound;21:22 XX proper;23:23 XX q;23:23 XX compound;21:23 XX proper;24:24 XX q;21:24 XX proper;25:25 XX q;17:25 XX appos;15:25 XX udef;0:25 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:9 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;5:5 XX q;6:6 XX q;8:8 XX q;8:8 XX compound;7:8 XX proper;9:9 XX d;7:9 XX proper;5:9 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX d;19:20 XX compound;18:20 XX udef;21:21 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX conj;0:27 XX focus', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;8:8 XX compound;7:8 XX proper;7:9 XX proper;12:12 XX compound;11:12 XX compound;0:12 XX focus', '', '9:9 XX q;9:10 XX udef;19:19 XX q;23:23 XX q', '2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;14:14 XX d;12:14 XX compound;16:16 XX q;17:17 XX q;19:19 XX q;18:19 XX compound;21:21 XX q;22:23 XX proper;26:26 XX q;26:27 XX udef;25:27 XX id;31:31 XX q;31:32 XX udef;42:42 XX q;43:43 XX compound;42:43 XX udef;22:43 XX appos;16:43 XX udef;0:43 XX focus', '2:2 XX q;4:4 XX q;6:6 XX q', '0:12 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX d;18:18 XX d;19:19 XX d;0:20 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;0:18 XX udef', '0:0 XX q;3:3 XX d;4:4 XX d;6:6 XX q;6:6 XX compound;5:6 XX proper;7:8 XX compound;5:8 XX udef;10:10 XX q;9:10 XX compound;11:11 XX q;13:13 XX q;13:13 XX compound;12:13 XX proper;0:13 XX udef', '4:4 XX q;4:4 XX compound;3:4 XX proper;5:5 XX q;3:5 XX proper;6:6 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX udef;24:24 XX q;23:25 XX udef;27:28 XX compound;19:28 XX udef;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;32:32 XX udef;29:32 XX udef', '2:2 XX q;3:3 XX q', '4:4 XX nominalization', '0:1 XX udef;0:4 XX compound', '1:1 XX q;0:1 XX proper;2:3 XX proper;6:6 XX d;5:6 XX proper;2:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:14 XX q;16:16 XX q;17:17 XX q;0:17 XX udef', '16:16 XX q;17:17 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX q;5:6 XX compound;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;9:10 XX proper;12:12 XX q;11:12 XX compound;13:13 XX q;14:14 XX proper;9:14 XX proper;0:14 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;19:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;0:5 XX focus', '0:0 XX neg', '6:6 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX compound;7:12 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:13 XX proper;15:15 XX q;14:15 XX compound;16:16 XX q;17:17 XX proper;14:17 XX udef;0:17 XX focus', '11:11 XX q;11:12 XX udef;10:12 XX udef;10:13 XX compound;14:14 XX p;15:15 XX p;16:17 XX compound;18:18 XX p;19:19 XX p;20:20 XX p;21:21 XX p;16:22 XX udef;24:24 XX neg;26:27 XX generic;29:29 XX q;30:30 XX q;29:30 XX udef;14:30 XX with', '2:2 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;5:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;15:15 XX compound;14:15 XX udef;14:15 XX udef;11:15 XX udef;5:15 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q', '6:6 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:17 XX focus', '', '0:0 XX q;1:1 XX q;3:3 XX d;2:3 XX compound;4:4 XX d;5:5 XX d;6:6 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;25:25 XX q;0:25 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;14:14 XX q;17:17 XX q;18:19 XX compound;17:19 XX udef;16:19 XX udef;20:20 XX d;16:20 XX compound;22:22 XX q;21:22 XX udef;23:23 XX q;0:24 XX focus', '0:0 XX q;6:6 XX q;0:14 XX udef', '', '0:0 XX mod;1:1 XX mod;2:2 XX mod;4:4 XX mod;5:5 XX mod;6:6 XX mod;7:7 XX mod;9:9 XX d;10:11 XX proper;13:14 XX proper;15:16 XX compound;13:16 XX compound;18:19 XX udef;20:20 XX q;18:20 XX compound;21:21 XX q;21:21 XX compound;21:21 XX udef;10:21 XX appos;1:21 XX relative;0:21 XX relative', '0:0 XX neg;13:13 XX q;13:13 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:20 XX udef', '0:0 XX nonsp;1:1 XX d;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;25:25 XX d;0:25 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;0:4 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:18 XX udef', '7:7 XX comp', '8:8 XX person', '', '1:1 XX q;1:1 XX compound;0:1 XX proper;2:2 XX d;3:4 XX proper;5:5 XX q;6:7 XX compound;5:7 XX udef;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX d;18:18 XX d;0:20 XX proper', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:11 XX udef', '', '13:13 XX udef', '13:13 XX q;12:13 XX compound;14:14 XX q;14:14 XX compound;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX number;21:21 XX compound;19:21 XX udef', '1:1 XX q;3:3 XX q;4:4 XX q;5:5 XX q;4:5 XX udef;8:8 XX d;7:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX compound;0:15 XX udef', '0:0 XX d;2:2 XX q;3:3 XX d;6:6 XX q;7:7 XX d;6:7 XX udef;8:8 XX d;0:10 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;8:8 XX comp;0:8 XX udef', '1:1 XX comp', '1:1 XX d;4:4 XX d;3:4 XX compound;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;0:23 XX focus', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX d;22:22 XX d;24:24 XX q;24:24 XX compound;23:24 XX proper;27:27 XX q;30:30 XX q;29:30 XX compound;28:30 XX compound;27:30 XX udef;26:30 XX id;31:31 XX q;33:33 XX q;32:33 XX compound;32:34 XX proper;36:36 XX compound;32:36 XX appos;0:36 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;6:6 XX nominalization;6:7 XX udef;11:11 XX comp;0:16 XX udef', '17:18 XX udef;28:28 XX q;30:30 XX q;31:31 XX q;33:33 XX q;34:34 XX q;36:36 XX q;35:37 XX udef;38:38 XX q;35:38 XX compound;39:39 XX q;31:39 XX udef', '0:0 XX q;2:2 XX d;3:3 XX mod;6:6 XX mod;8:8 XX mod;10:10 XX mod;9:10 XX loc;11:11 XX mod;12:12 XX mod;13:13 XX mod;15:15 XX d;16:17 XX proper;18:18 XX q;19:19 XX q;20:20 XX q;20:21 XX proper;23:24 XX compound;23:25 XX compound;28:28 XX q;28:28 XX nominalization;28:28 XX udef;20:28 XX appos;18:28 XX udef;16:28 XX appos;0:28 XX udef', '', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;22:22 XX comp', '7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;22:22 XX comp', '7:7 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;13:14 XX proper;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;18:19 XX compound;20:20 XX q', '0:1 XX proper;9:9 XX q;12:12 XX nonsp;13:13 XX nonsp;14:14 XX nonsp;15:15 XX nonsp;16:16 XX nonsp;17:17 XX nonsp;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;23:23 XX q;23:23 XX compound;22:23 XX proper;22:24 XX proper;22:24 XX compound;18:24 XX udef;0:24 XX compound', '6:6 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;17:17 XX q;18:18 XX q;20:20 XX q;20:20 XX compound;19:20 XX proper;22:22 XX q;21:22 XX compound;19:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX udef;19:25 XX udef', '8:8 XX q;8:8 XX compound;7:8 XX proper;7:9 XX proper', '3:3 XX q;3:3 XX compound;2:3 XX proper;0:20 XX appos', '', '0:0 XX d;1:1 XX d;2:2 XX q;2:2 XX nominalization;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;9:9 XX compound;8:9 XX proper;8:10 XX proper;0:23 XX focus', '0:0 XX q;1:1 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;12:12 XX named;13:13 XX q;13:13 XX named;14:14 XX q;14:14 XX named;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;0:24 XX udef', '1:1 XX q;2:2 XX q;2:2 XX compound;1:2 XX udef;1:3 XX udef;1:4 XX compound;16:16 XX q;17:17 XX q;0:18 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX d;6:7 XX udef;3:7 XX udef;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;28:28 XX q;28:28 XX compound;28:29 XX udef;0:29 XX udef', '3:5 XX number;15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:21 XX udef;22:22 XX q;20:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;30:30 XX q;24:31 XX udef', '2:2 XX q;12:12 XX q', '6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;12:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;19:19 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;15:27 XX udef;10:27 XX udef;0:27 XX udef', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;22:22 XX d;0:22 XX focus', '', '6:6 XX q;8:8 XX q;12:12 XX q;11:12 XX compound;13:13 XX q;11:13 XX udef;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX q;22:22 XX q;23:23 XX q;23:23 XX nominalization;25:25 XX q;25:25 XX nominalization;24:25 XX udef;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX compound;27:28 XX udef;23:28 XX udef;15:28 XX udef', '17:17 XX q;17:17 XX compound;19:19 XX q;17:19 XX udef;21:21 XX q;22:23 XX compound;21:23 XX udef;21:24 XX udef;20:24 XX udef', '', '5:5 XX q;6:6 XX q;7:7 XX times;11:11 XX q;12:12 XX q;13:13 XX times;25:25 XX q;26:26 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;9:9 XX d;10:10 XX d;0:10 XX focus', '9:9 XX compound;0:9 XX compound', '0:0 XX q;0:0 XX compound;1:1 XX q;1:1 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;11:11 XX named;12:12 XX q;12:12 XX named;13:13 XX q;13:13 XX named;15:15 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX proper', '1:1 XX q;2:2 XX q;2:2 XX compound;1:2 XX udef;0:2 XX udef;3:3 XX q;3:4 XX udef;0:5 XX udef;18:18 XX q;19:19 XX q;0:20 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX d;6:7 XX udef;3:7 XX udef;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;17:17 XX conj;16:17 XX udef;18:18 XX conj;20:20 XX conj;19:20 XX udef;21:21 XX conj;22:22 XX conj;25:25 XX q;24:25 XX compound;26:26 XX q;27:27 XX q;27:27 XX udef;23:27 XX udef;0:27 XX udef', '8:8 XX q;9:9 XX q;10:10 XX q;13:13 XX q;14:15 XX compound;13:15 XX udef;12:15 XX udef;22:22 XX times;25:25 XX q;24:25 XX compound;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;5:5 XX compound;7:8 XX compound;5:8 XX udef;4:8 XX udef;26:26 XX superl;34:35 XX compound;38:39 XX udef;38:39 XX compound;3:39 XX udef;0:39 XX proper', '0:0 XX q;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:16 XX udef;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;23:23 XX q;0:25 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;0:7 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;4:8 XX udef;9:9 XX q;10:10 XX q;10:10 XX udef', '0:0 XX q;0:0 XX compound;4:4 XX q;5:6 XX compound;17:17 XX q;23:23 XX nominalization;0:28 XX proper', '8:8 XX q;7:9 XX udef;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;20:20 XX compound;22:23 XX compound;24:24 XX q;20:24 XX udef;25:25 XX q;25:25 XX udef', '0:0 XX q;1:1 XX q;1:1 XX compound;7:7 XX q;11:11 XX q;17:17 XX q;16:17 XX compound;23:23 XX q;24:24 XX q;24:24 XX compound;0:24 XX proper', '0:0 XX neg', '0:0 XX q;1:1 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;4:8 XX udef;0:16 XX udef', '', '0:0 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;31:31 XX q;32:32 XX q;32:32 XX compound;33:33 XX q;34:34 XX q;0:35 XX proper', '0:0 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;31:31 XX q;32:32 XX q;32:32 XX compound;33:33 XX q;34:34 XX q;0:35 XX proper', '0:1 XX compound;10:10 XX q;10:10 XX compound;9:10 XX proper;0:14 XX compound', '9:9 XX q;9:9 XX compound;8:9 XX proper;8:10 XX proper;11:11 XX compound;8:11 XX compound', '0:0 XX nonsp;3:3 XX d;4:4 XX q;8:8 XX q;7:8 XX compound;5:8 XX compound;9:9 XX q;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX d;14:17 XX udef;10:17 XX udef;4:17 XX udef;18:18 XX d;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;21:22 XX udef;19:22 XX udef;0:22 XX loc', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;18:18 XX nonsp;19:19 XX nonsp;20:20 XX d;21:21 XX d;24:24 XX q;26:26 XX d;0:26 XX compound', '11:11 XX nonsp;12:12 XX nonsp', '3:3 XX q;2:3 XX compound;4:4 XX q;6:6 XX q;8:8 XX q;7:8 XX loc;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX nonsp;14:14 XX nonsp;15:15 XX q;13:15 XX loc;11:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;17:20 XX udef;11:20 XX udef', '0:1 XX udef;0:22 XX compound', '0:0 XX q;0:0 XX compound;0:10 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;16:16 XX q;0:17 XX proper', '3:3 XX q;2:3 XX udef;4:4 XX q;9:10 XX proper;11:11 XX q;9:11 XX compound;15:15 XX compound', '0:0 XX q;1:1 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;12:12 XX q;13:13 XX q;13:13 XX compound;12:13 XX udef;11:13 XX udef;14:14 XX q;14:14 XX compound;11:14 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX udef', '0:0 XX neg', '2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;7:7 XX compound;6:7 XX udef;5:7 XX udef;8:8 XX compound;5:8 XX udef', '0:0 XX q;0:0 XX compound;6:6 XX q;0:10 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:6 XX proper;9:9 XX q;11:11 XX q;11:11 XX udef;0:11 XX proper', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;12:12 XX q;10:12 XX measure;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q', '2:2 XX q;4:4 XX q;3:4 XX compound;6:7 XX compound;8:8 XX q;8:9 XX udef;10:10 XX q;11:11 XX q;10:12 XX udef;8:12 XX udef;18:18 XX q;14:20 XX generic;5:20 XX udef', '', '0:4 XX compound', '4:4 XX q;6:6 XX q;8:8 XX q;9:9 XX q;9:10 XX proper;13:13 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;20:21 XX udef;22:22 XX q;23:23 XX q;23:23 XX udef;0:23 XX appos', '', '0:0 XX d;2:2 XX d;4:4 XX d;8:8 XX nonsp;10:10 XX nonsp;11:11 XX nonsp;12:12 XX d;14:14 XX d;0:14 XX focus', '1:1 XX q;0:1 XX compound;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;9:10 XX compound;19:19 XX q;20:20 XX q;21:21 XX times;19:21 XX udef;25:25 XX q;25:25 XX nominalization;27:27 XX q;27:27 XX nominalization;30:30 XX q;30:30 XX nominalization;29:30 XX compound;31:31 XX q;31:31 XX nominalization;36:36 XX q;38:38 XX q;37:38 XX compound;39:39 XX q;41:41 XX compound;40:41 XX proper;36:41 XX udef;32:41 XX appos;28:41 XX appos;0:41 XX proper', '5:5 XX q;3:5 XX compound;6:6 XX q;8:8 XX q;9:9 XX q;7:9 XX udef', '0:0 XX d;1:1 XX d;3:3 XX q;3:3 XX compound;2:3 XX proper;5:5 XX d;4:5 XX compound;6:6 XX d;8:8 XX d;7:8 XX compound;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX q;12:14 XX udef;15:15 XX d;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;16:26 XX udef;27:27 XX d;33:33 XX q;35:35 XX q;36:36 XX q;33:38 XX udef;40:40 XX generic;0:40 XX focus', '19:20 XX proper;17:20 XX udef;24:24 XX q;26:26 XX q;27:27 XX q', '0:0 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;19:19 XX q;19:19 XX compound;18:19 XX proper;18:20 XX proper;21:21 XX q;18:21 XX compound;23:23 XX q;26:26 XX d;28:28 XX d;0:28 XX focus', '12:12 XX q', '3:4 XX proper;16:17 XX compound;21:21 XX q;21:21 XX compound;20:21 XX proper;20:22 XX udef', '4:4 XX q;4:5 XX udef;21:21 XX q', '0:6 XX compound', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;9:9 XX q;9:9 XX nominalization;11:11 XX q;10:12 XX udef;13:13 XX conj;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX conj;16:17 XX udef;14:17 XX udef;18:18 XX conj;19:19 XX conj;20:20 XX conj;22:22 XX conj;23:23 XX q;23:23 XX udef;13:23 XX implicit', '0:0 XX q;2:2 XX q;2:2 XX compound;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;0:32 XX udef', '0:0 XX q;1:1 XX q;0:10 XX udef', '', '8:8 XX q;9:9 XX q;14:13 XX Business;14:14 XX q;14:14 XX XX;14:14 XX udef;13:14 XX parenthetical;15:19 XX eventuality;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;24:25 XX compound;30:30 XX q;30:30 XX udef;26:30 XX eventuality;0:30 XX compound', '1:1 XX q;10:10 XX q', '5:5 XX q;6:6 XX q;12:12 XX q;13:14 XX compound;12:15 XX udef;17:17 XX q;17:17 XX udef;11:17 XX appos', '0:19 XX appos', '19:19 XX q;20:20 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;23:23 XX q;21:23 XX compound;24:24 XX q;25:25 XX q;27:27 XX q;26:27 XX compound;29:29 XX q;30:30 XX q;31:31 XX d;30:31 XX udef;33:33 XX d;0:33 XX focus', '5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;11:11 XX compound;10:11 XX udef;13:13 XX q;13:13 XX nominalization;12:13 XX compound;10:13 XX udef;8:13 XX udef;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX udef', '3:3 XX q;4:4 XX q;6:6 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;10:12 XX udef;9:12 XX udef;9:13 XX udef', '5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;10:11 XX udef;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;0:20 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;11:12 XX proper;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX d;17:18 XX compound;11:18 XX compound;19:19 XX d;22:22 XX d;0:22 XX focus', '8:9 XX compound;12:12 XX q;13:13 XX q;0:13 XX compound', '1:1 XX q;2:2 XX q;3:3 XX q;6:6 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;32:32 XX q;0:35 XX compound', '1:1 XX q;7:7 XX q;8:8 XX q;11:11 XX q;12:12 XX q;0:13 XX compound', '5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;4:6 XX proper;7:8 XX compound;10:10 XX q', '5:5 XX q', '4:4 XX q;5:5 XX q;6:6 XX q;7:8 XX udef;7:8 XX compound', '1:1 XX q;1:1 XX compound;0:1 XX proper;2:2 XX q;2:2 XX compound;0:3 XX udef;4:5 XX compound;4:6 XX compound;7:7 XX neg;0:8 XX compound', '2:2 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:12 XX proper;13:13 XX q;11:13 XX compound;14:14 XX q;15:15 XX q;0:15 XX focus', '9:9 XX nonsp;13:13 XX q;15:15 XX neg;0:15 XX compound', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;11:12 XX udef', '0:0 XX q;10:10 XX nonsp;13:13 XX q;0:15 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;12:15 XX udef;16:16 XX d;17:17 XX d;18:18 XX conj;19:19 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;28:28 XX q;29:29 XX q;30:30 XX q;32:32 XX q;33:33 XX q;0:33 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX d;12:12 XX d;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX proper;17:17 XX q;16:17 XX compound;18:18 XX q;19:19 XX q;21:21 XX q;23:23 XX q;22:23 XX compound;22:23 XX proper;19:23 XX udef;16:23 XX udef;0:23 XX focus', '3:3 XX q;4:4 XX q;5:5 XX q;5:6 XX udef;7:7 XX neg;10:10 XX q', '10:11 XX compound;12:12 XX q;10:12 XX compound;10:13 XX proper;15:15 XX q;14:15 XX compound;14:16 XX proper;19:19 XX q;21:21 XX q;21:21 XX compound;20:21 XX compound;19:22 XX proper;14:23 XX udef;10:23 XX udef', '3:3 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;12:12 XX d;14:14 XX d;15:15 XX conj', '0:0 XX q;1:1 XX q;2:3 XX compound;15:15 XX q;16:16 XX q;0:19 XX udef', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX q;15:15 XX d;0:15 XX focus', '4:4 XX d;5:5 XX d;7:7 XX d;0:8 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;9:9 XX compound;8:9 XX proper;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX proper;0:15 XX focus', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX q;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX q;13:13 XX q;0:14 XX implicit', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:9 XX proper;10:10 XX q;8:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;15:16 XX compound;17:17 XX q;18:18 XX q;19:20 XX compound;0:20 XX focus', '0:0 XX q;0:18 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;16:16 XX p;17:17 XX p;18:18 XX p;19:19 XX p;21:21 XX p;22:22 XX p;23:23 XX p;24:24 XX p;25:25 XX p;26:26 XX p;0:26 XX udef', '3:4 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;14:14 XX q', '2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX compound', '0:0 XX superl;9:9 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX proper', '0:2 XX compound;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX q;7:7 XX conj;6:7 XX udef;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;13:13 XX udef;0:13 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;17:17 XX nonsp;0:23 XX proper', '6:6 XX q', '', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX q;0:19 XX focus', '0:0 XX q;13:14 XX compound;0:23 XX udef', '3:3 XX q', '0:18 XX appos', '', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;0:11 XX focus', '0:0 XX q;2:2 XX q;10:10 XX q;11:11 XX q;0:11 XX proper', '10:10 XX q;10:10 XX nominalization', '0:0 XX q;3:3 XX q;4:4 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;0:20 XX udef', '9:9 XX q;8:9 XX compound;8:10 XX proper;11:11 XX q;12:12 XX q;13:14 XX proper;13:15 XX compound;0:24 XX appos', '0:0 XX q;2:2 XX q;3:3 XX q;11:11 XX q;14:14 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;26:27 XX udef;28:28 XX q;29:29 XX q;0:29 XX udef', '10:10 XX q;11:11 XX q', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;7:11 XX udef;18:18 XX q;17:18 XX compound;20:20 XX q;21:21 XX nonsp;21:21 XX loc', '', '0:0 XX d;2:3 XX compound;5:5 XX q;4:5 XX proper;6:6 XX q;8:8 XX d;7:8 XX proper;4:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX focus', '6:6 XX q;6:7 XX udef;0:17 XX appos', '2:2 XX comp', '', '0:0 XX q;2:2 XX conj;1:2 XX compound;3:3 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;17:17 XX conj;18:18 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX conj;0:22 XX proper', '1:1 XX q;2:2 XX q;3:3 XX q;8:8 XX q;0:15 XX compound', '', '11:11 XX q;13:13 XX compound;0:13 XX appos', '0:0 XX q;1:1 XX q;1:2 XX udef;11:11 XX neg;0:17 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:10 XX udef;7:10 XX udef;18:18 XX q;19:19 XX q;18:19 XX proper;22:22 XX q;23:23 XX q;24:24 XX q;23:25 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:32 XX compound;31:32 XX udef;28:32 XX udef;23:32 XX udef', '0:0 XX q;1:1 XX q;10:10 XX q;11:11 XX q;12:12 XX times;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX times;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;0:28 XX udef', '9:10 XX udef', '0:0 XX q;0:1 XX udef;4:4 XX q;0:6 XX compound', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX d;12:17 XX udef;18:18 XX d;21:21 XX q;22:22 XX q;23:23 XX q;24:25 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;30:30 XX q;29:31 XX udef;0:31 XX focus', '0:0 XX d;3:3 XX d;5:5 XX d;0:5 XX focus', '', '', '5:5 XX nonsp;6:6 XX nonsp', '0:0 XX d;1:1 XX d;4:4 XX d;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;11:11 XX q;11:12 XX proper;14:14 XX superl;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX d;22:22 XX d;23:23 XX q;24:24 XX q;0:24 XX focus', '8:8 XX q;0:8 XX compound', '0:0 XX superl;7:7 XX nonsp;8:8 XX nonsp', '0:0 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;0:18 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;0:10 XX focus', '5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q', '1:3 XX udef;4:4 XX d;0:4 XX udef;6:6 XX d;5:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX q;10:11 XX measure;10:11 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;18:18 XX proper;10:18 XX udef;0:18 XX focus', '', '', '', '0:0 XX q;1:1 XX q;1:1 XX compound;0:1 XX udef;3:3 XX neg;0:21 XX udef', '2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;22:22 XX d;23:23 XX d;24:24 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:32 XX udef;0:32 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX p;5:5 XX p;6:6 XX p;8:8 XX p;9:9 XX d;10:11 XX proper;13:15 XX measure;20:20 XX compound;19:20 XX compound;10:20 XX appos;0:20 XX focus', '6:6 XX q;7:7 XX q;8:9 XX compound;10:10 XX nonsp;12:12 XX nonsp;16:16 XX q;17:17 XX q;18:18 XX times;16:18 XX udef;23:23 XX q;25:25 XX q;24:25 XX compound;26:26 XX q;27:27 XX q;27:28 XX proper;29:29 XX q;30:30 XX q;31:31 XX q;30:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX q;8:8 XX compound;7:8 XX udef;9:9 XX d;11:11 XX d;0:11 XX focus', '0:0 XX d;1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;0:15 XX focus', '1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization', '0:0 XX q;1:1 XX q;2:2 XX times;0:2 XX udef;4:4 XX q;4:4 XX compound;3:4 XX proper;5:5 XX q;5:5 XX compound;3:5 XX proper;6:7 XX proper;8:8 XX d;6:8 XX compound;3:8 XX proper;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;17:18 XX compound;19:19 XX d;21:21 XX q;21:21 XX compound;20:21 XX proper;20:22 XX proper;23:23 XX q;24:24 XX q;26:26 XX q;26:26 XX compound;25:26 XX proper;25:27 XX proper;28:28 XX q;25:28 XX compound;25:28 XX proper;23:28 XX udef;20:28 XX appos;3:28 XX focus;0:28 XX compound', '0:0 XX d;2:2 XX d;3:3 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:12 XX focus', '', '', '0:0 XX q;7:7 XX q;8:8 XX q;9:9 XX q;0:13 XX proper', '0:0 XX q;4:5 XX compound;3:5 XX compound;9:9 XX d;8:9 XX proper;10:10 XX d;11:11 XX d;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;21:21 XX d;20:21 XX compound;0:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;8:8 XX q;9:9 XX q;0:9 XX focus', '3:4 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;0:9 XX appos', '3:3 XX q;4:4 XX q', '2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX compound;19:20 XX udef;17:20 XX udef;0:20 XX focus', '3:3 XX q;13:13 XX q;14:14 XX q;0:15 XX compound', '1:1 XX q;0:2 XX udef;4:4 XX nonsp;8:8 XX q;0:15 XX compound', '0:1 XX compound;4:4 XX comp;10:10 XX q;10:10 XX nominalization;13:13 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX compound', '0:0 XX q;9:10 XX proper;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;0:21 XX udef', '0:5 XX appos', '8:9 XX udef', '', '0:0 XX q;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;11:11 XX q;10:12 XX udef;16:16 XX q;17:17 XX q;0:17 XX proper', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;11:11 XX d;0:11 XX focus', '3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;8:8 XX proper', '', '5:5 XX q;6:6 XX q;14:14 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;6:6 XX d;7:7 XX conj;8:8 XX q;9:9 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX q;0:14 XX focus', '0:0 XX d;1:1 XX d;3:3 XX q;4:5 XX proper;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX q;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;0:23 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:13 XX generic', '2:3 XX udef;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX compound;0:13 XX subord', '9:9 XX q', '0:1 XX compound;2:3 XX compound;7:7 XX comp;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;0:13 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX q;6:6 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:19 XX focus', '1:1 XX q;2:2 XX q;5:5 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX udef;20:22 XX udef;17:22 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX q;13:14 XX compound;13:15 XX proper;16:16 XX q;17:17 XX q;18:18 XX q;18:19 XX proper;21:23 XX measure;24:25 XX compound;18:32 XX appos;16:32 XX udef;0:32 XX focus', '0:0 XX nonsp;1:1 XX d;2:2 XX q;5:6 XX proper;7:8 XX compound;9:9 XX d;7:9 XX compound;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX d;0:24 XX loc', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX q;20:20 XX q;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;7:8 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;15:16 XX udef;0:16 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX d;16:16 XX d;17:18 XX proper;0:20 XX focus', '3:3 XX neg', '0:0 XX d;2:2 XX d;3:3 XX d;15:15 XX q;16:16 XX q;17:17 XX times;0:18 XX focus', '3:3 XX nonsp;4:5 XX loc;12:12 XX q;14:14 XX q;12:15 XX udef', '0:0 XX d;1:2 XX proper;3:3 XX d;1:3 XX compound;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '3:3 XX nominalization;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX nonsp;12:12 XX loc', '', '0:0 XX q;1:1 XX q;1:1 XX compound;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;0:20 XX proper', '13:13 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:8 XX focus', '0:0 XX d;1:1 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:6 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;18:18 XX d;19:19 XX d;21:21 XX q;21:21 XX compound;20:21 XX proper;20:22 XX proper;23:23 XX q;24:24 XX q;25:26 XX proper;29:29 XX q;29:29 XX compound;28:29 XX proper;28:30 XX proper;31:32 XX udef;31:32 XX compound;28:32 XX compound;25:32 XX appos;23:32 XX udef;0:32 XX focus', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;13:13 XX q;14:14 XX q;15:15 XX q;19:20 XX proper;21:22 XX proper;23:23 XX q;21:23 XX compound;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;29:29 XX q;28:29 XX compound;30:30 XX q;21:30 XX proper;16:30 XX appos;0:30 XX udef', '9:9 XX q;10:10 XX q;9:10 XX udef;12:12 XX q;12:13 XX udef;9:13 XX udef', '', '6:6 XX q;7:7 XX q;8:8 XX q;11:11 XX nonsp;21:21 XX q;22:22 XX q', '1:2 XX proper;1:3 XX appos;4:4 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:13 XX focus', '2:2 XX d;3:3 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:19 XX subord', '3:4 XX compound;17:17 XX nonsp;0:23 XX appos', '', '0:0 XX q;13:13 XX nominalization;0:22 XX proper', '0:8 XX appos', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:5 XX proper;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX q;22:22 XX compound;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;31:31 XX q;31:31 XX compound;30:31 XX proper;0:31 XX focus', '0:0 XX q;0:0 XX compound;2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;19:19 XX q;0:19 XX proper', '7:7 XX q', '2:2 XX q;3:3 XX q;5:5 XX q;5:5 XX compound;4:5 XX proper;6:6 XX q;6:6 XX compound;4:6 XX proper;13:13 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:21 XX appos', '3:4 XX compound;14:16 XX compound', '', '1:1 XX q;2:2 XX q;3:4 XX proper;5:5 XX q;3:5 XX compound;6:6 XX q;6:6 XX compound;6:7 XX udef;3:7 XX proper;9:10 XX proper;11:12 XX compound;9:12 XX compound;14:14 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX compound;0:26 XX appos', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;7:7 XX q;8:8 XX q;10:10 XX d;9:10 XX udef;6:10 XX udef;0:11 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;9:9 XX d;12:12 XX nonsp;13:13 XX nonsp;14:14 XX nonsp;15:15 XX nonsp;16:16 XX nonsp;0:16 XX focus', '0:0 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;8:9 XX udef;8:10 XX compound;13:13 XX comp;33:33 XX neg;0:35 XX proper', '10:10 XX q;11:11 XX q;12:12 XX q;13:14 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;25:25 XX compound;23:25 XX udef;22:25 XX udef', '15:15 XX q;16:16 XX q;15:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;19:22 XX udef;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;26:27 XX udef', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;0:20 XX loc', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;8:10 XX udef;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;12:15 XX udef;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;0:20 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;8:10 XX udef;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX d;12:15 XX udef;16:16 XX d;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;19:20 XX udef;0:20 XX focus', '3:3 XX q;4:5 XX compound;4:6 XX compound;15:15 XX q;16:16 XX q;21:21 XX q;22:22 XX q', '0:0 XX q;2:2 XX q;3:3 XX q;4:4 XX q;6:6 XX q;7:7 XX q;8:8 XX q;0:15 XX udef', '0:0 XX nonsp;4:4 XX q;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX q;0:16 XX loc', '1:1 XX q;2:2 XX q;3:3 XX q;19:19 XX q;20:20 XX q;22:22 XX nonsp;21:22 XX loc;0:22 XX compound', '0:0 XX conj;1:1 XX q;3:3 XX q;4:4 XX q;4:4 XX udef;0:4 XX implicit', '3:3 XX q;7:7 XX q;7:7 XX compound;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q', '2:3 XX udef;1:3 XX implicit;0:3 XX udef;6:6 XX q;6:6 XX compound;6:8 XX udef;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;14:17 XX udef;10:17 XX udef;20:20 XX q;21:21 XX q;20:22 XX udef;23:23 XX q;24:24 XX q;25:25 XX q;24:27 XX udef;20:27 XX udef;28:28 XX nonsp;29:29 XX nonsp;28:29 XX loc;0:29 XX compound', '', '0:0 XX q;1:1 XX q;8:8 XX q;9:9 XX q;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX q;16:17 XX proper;18:18 XX q;19:20 XX proper;16:20 XX udef;28:28 XX q;29:29 XX q;33:33 XX q;34:34 XX q;35:35 XX q;37:37 XX nominalization;36:37 XX udef;36:37 XX compound;0:37 XX udef', '0:0 XX superl;6:6 XX q;7:7 XX q', '0:0 XX q;1:1 XX q;7:7 XX q;8:8 XX q;9:9 XX times;11:11 XX q;12:12 XX q;13:13 XX times;0:20 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX udef;16:16 XX q;17:17 XX q;18:19 XX udef;20:20 XX d;18:20 XX compound;17:20 XX udef;0:21 XX focus', '6:6 XX q;7:7 XX q;8:8 XX times;13:13 XX q;14:14 XX q;15:15 XX times;16:16 XX nonsp;17:17 XX nonsp', '6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX udef;6:9 XX udef;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX q;10:15 XX udef;21:21 XX q;22:22 XX q;21:22 XX udef;23:23 XX q;24:25 XX compound;23:25 XX udef;21:25 XX udef;26:26 XX q;21:26 XX compound;27:27 XX q;28:29 XX compound;28:29 XX udef;20:29 XX udef;0:29 XX subord', '0:0 XX q;1:1 XX q;1:1 XX compound;2:2 XX d;3:3 XX d;4:4 XX d;0:5 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;0:7 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;7:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX d;11:14 XX udef;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;19:19 XX q;20:20 XX q;19:21 XX udef;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX d;23:26 XX udef;27:27 XX nonsp;28:28 XX nonsp;0:28 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '25:25 XX q;27:28 XX appos', '13:14 XX udef;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;29:30 XX udef;31:32 XX compound;31:39 XX udef;29:39 XX udef', '16:19 XX udef;23:23 XX q', '13:13 XX entity', '', '', '2:2 XX nonsp;3:3 XX nonsp;4:4 XX nonsp;6:6 XX nonsp;5:6 XX compound;7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;11:11 XX nonsp;4:11 XX loc;2:11 XX loc', '5:5 XX q;7:7 XX q;7:8 XX udef', '0:0 XX q;0:0 XX nominalization;0:10 XX udef', '', '0:0 XX unknown', '0:1 XX unknown', '10:10 XX neg', '12:12 XX nonsp', '5:5 XX q;13:13 XX d;16:16 XX d;17:17 XX d;20:20 XX nonsp;21:21 XX nonsp;22:22 XX nonsp;25:25 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d;18:33 XX loc', '8:8 XX q;10:10 XX q;10:10 XX udef', '', '19:19 XX q;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;28:28 XX conj;29:29 XX conj;31:31 XX q;32:32 XX q;0:33 XX subord', '', '5:5 XX q;5:5 XX nominalization', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;21:21 XX d;22:22 XX d;23:23 XX d;0:24 XX focus', '0:0 XX neg', '0:0 XX conj;2:2 XX d;3:3 XX d;4:4 XX conj;0:11 XX implicit', '', '', '0:0 XX conj;1:1 XX conj;2:2 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;12:12 XX conj;13:13 XX conj;15:15 XX conj;16:16 XX conj;0:16 XX implicit', '10:10 XX q', '2:2 XX q;3:3 XX q;4:4 XX q', '12:13 XX udef;15:15 XX q;17:17 XX q;18:18 XX q;23:23 XX q;23:23 XX compound;22:23 XX proper;24:24 XX q;24:24 XX compound', '1:1 XX neg', '12:12 XX q;12:13 XX udef', '2:2 XX d;3:3 XX d;6:6 XX q;7:7 XX q;6:8 XX udef;9:9 XX d;10:10 XX q;11:11 XX q;11:11 XX compound;10:11 XX udef;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX q;18:18 XX q;17:19 XX udef;20:20 XX d;21:21 XX d;0:23 XX focus', '2:2 XX q;3:3 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;0:19 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;17:17 XX d;16:17 XX udef;0:17 XX focus', '3:3 XX q;2:3 XX compound;2:4 XX proper;6:6 XX q;8:8 XX nominalization;7:8 XX compound;6:9 XX udef;6:12 XX appos;18:19 XX compound;22:22 XX compound', '0:1 XX proper;4:5 XX compound;0:22 XX appos', '0:0 XX d;0:3 XX focus', '0:12 XX appos', '16:16 XX compound', '15:16 XX proper', '', '0:-1 XX EMPTY-LABEL', '0:1 XX compound;2:3 XX compound;9:9 XX nonsp;0:10 XX compound', '4:4 XX q;4:4 XX nominalization;6:6 XX q;6:6 XX nominalization;5:6 XX udef;7:7 XX q;7:7 XX nominalization;12:12 XX q;12:12 XX nominalization;13:14 XX compound;15:15 XX compound;13:15 XX udef', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;10:11 XX compound;12:12 XX q;13:13 XX q;13:13 XX udef', '5:5 XX q;6:6 XX q;8:8 XX q', '8:8 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX q;13:14 XX udef', '0:0 XX q;1:1 XX q;2:2 XX d;3:3 XX d;5:5 XX d;0:6 XX udef', '', '', '2:2 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:7 XX focus', '', '13:13 XX q;13:13 XX compound', '2:2 XX q;3:3 XX times;8:9 XX proper;13:13 XX superl;14:15 XX compound;18:19 XX compound;23:23 XX q;24:24 XX q', '10:11 XX compound;9:11 XX udef;18:18 XX q;19:19 XX compound;18:19 XX udef;16:19 XX udef;13:19 XX generic;8:19 XX implicit;7:19 XX compound;0:19 XX compound', '2:2 XX superl', '0:0 XX q;1:1 XX q;3:3 XX q;4:4 XX q;6:6 XX nonsp;9:9 XX q;11:11 XX q;12:12 XX q;0:20 XX udef', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;24:24 XX q;26:26 XX q;26:26 XX compound;25:26 XX proper;25:27 XX proper;30:30 XX q;30:30 XX compound;29:30 XX proper;29:30 XX proper;0:30 XX udef', '6:6 XX conj;5:6 XX compound;7:9 XX udef;5:9 XX implicit;12:14 XX subord', '0:0 XX neg', '0:0 XX d;1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;16:16 XX d;17:17 XX d;18:18 XX d;0:21 XX focus', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;10:10 XX q;11:11 XX q;12:12 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;0:24 XX udef', '5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;10:10 XX q;14:15 XX compound;13:15 XX proper;20:21 XX compound;20:22 XX compound', '', '0:0 XX d;1:1 XX q;2:2 XX q;3:4 XX compound;5:5 XX q;3:5 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;11:12 XX compound;10:12 XX udef;13:13 XX p;14:14 XX p;16:16 XX d;15:16 XX udef;13:16 XX with;17:17 XX q;19:19 XX d;20:20 XX d;22:22 XX d;25:25 XX d;26:26 XX d;27:27 XX p;28:28 XX p;29:29 XX q;30:30 XX q;32:32 XX p;33:33 XX q;34:34 XX q;38:38 XX proper;36:38 XX udef;35:38 XX udef;33:38 XX udef;29:38 XX udef;0:38 XX focus', '9:9 XX q;9:9 XX udef;0:9 XX compound', '', '13:13 XX q;14:15 XX proper;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;23:23 XX compound;21:23 XX udef;24:24 XX compound;21:24 XX proper', '0:2 XX compound', '', '12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX proper', '0:0 XX q;3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;27:27 XX q;29:29 XX q;31:31 XX q;0:32 XX udef', '0:0 XX d;1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:14 XX compound;12:14 XX proper;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;26:26 XX q;27:28 XX compound;27:29 XX compound;31:31 XX q;0:31 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;18:18 XX compound;0:18 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:10 XX proper', '13:13 XX q;12:13 XX proper;14:14 XX q', '5:5 XX neg', '13:13 XX q;16:16 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q', '0:0 XX q;1:1 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX compound;24:24 XX q;27:27 XX q;28:28 XX q;0:28 XX udef', '0:0 XX superl;6:6 XX q;6:7 XX udef;12:12 XX q;13:13 XX q;13:14 XX udef;15:15 XX q;16:16 XX q;15:17 XX udef', '0:0 XX q;13:13 XX nonsp;0:24 XX proper', '15:15 XX q;15:15 XX nominalization;16:17 XX compound;16:17 XX udef', '19:19 XX q;24:24 XX q;26:26 XX q;28:28 XX q;27:28 XX compound;29:29 XX q;30:30 XX q;31:31 XX q;0:31 XX appos', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX d;13:13 XX d;0:13 XX focus', '', '1:1 XX conj;1:1 XX unknown;2:2 XX conj;2:2 XX unknown;3:3 XX conj;3:3 XX unknown;0:4 XX implicit', '0:0 XX q;0:1 XX proper;2:2 XX q;3:3 XX q;3:4 XX proper;5:5 XX d;6:6 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;0:15 XX udef', '2:3 XX compound;8:9 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;0:24 XX udef', '0:0 XX q;1:1 XX q;2:2 XX q;11:11 XX q;21:21 XX q;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;0:24 XX udef', '10:11 XX compound;10:12 XX udef;10:14 XX compound;19:19 XX superl;20:20 XX superl', '', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;18:19 XX udef;20:20 XX q;21:21 XX udef;18:21 XX udef', '5:5 XX q;6:6 XX q;13:13 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:19 XX proper;20:21 XX compound;23:23 XX q;24:24 XX q;23:24 XX udef;22:24 XX udef;25:25 XX q;26:26 XX q;28:28 XX q;28:28 XX compound;27:28 XX proper;29:29 XX q;27:29 XX proper;30:30 XX q;27:31 XX udef;20:31 XX appos;0:31 XX focus', '0:0 XX neg;2:2 XX q;7:7 XX q', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;20:20 XX d;21:21 XX q;22:22 XX q;23:23 XX q;23:24 XX udef;0:24 XX focus', '1:1 XX neg', '4:4 XX p;5:5 XX p;6:6 XX p;7:7 XX p;8:8 XX p;9:9 XX p', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:8 XX focus', '4:4 XX comp;8:8 XX q;9:9 XX q;9:9 XX udef', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX udef;0:10 XX unknown', '0:0 XX q;0:0 XX pron;1:1 XX q;1:1 XX pron;2:2 XX q;3:3 XX q;3:3 XX pron;2:3 XX udef;4:4 XX q;4:4 XX pron;5:5 XX q;5:5 XX pron;6:6 XX q;6:6 XX pron;7:7 XX q;7:7 XX pron;8:8 XX q;8:8 XX pron;9:9 XX q;9:9 XX pron;10:10 XX q;10:10 XX pron;19:19 XX q;19:20 XX udef;24:24 XX compound;19:24 XX appos;11:24 XX subord;0:24 XX pronoun', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:13 XX proper;17:17 XX q;17:17 XX proper;0:17 XX compound', '11:11 XX neg', '5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;16:16 XX q;14:16 XX compound;18:18 XX q;20:20 XX q;19:20 XX udef', '2:2 XX neg', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;0:6 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX q;22:22 XX d;23:23 XX d;26:27 XX udef;28:28 XX q;33:33 XX q;34:35 XX compound;33:35 XX udef;32:35 XX udef;32:36 XX compound;39:39 XX proper;30:39 XX appos;28:39 XX udef;0:39 XX focus', '0:0 XX d;1:1 XX q;2:2 XX q;1:3 XX udef;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;5:6 XX compound;8:8 XX d;7:8 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;0:19 XX focus', '5:6 XX proper;10:11 XX compound;14:14 XX q;14:14 XX compound;13:14 XX proper', '6:6 XX comp;11:12 XX udef;26:26 XX q', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;4:4 XX q;3:5 XX udef;7:7 XX q;8:8 XX q;8:8 XX compound;7:8 XX udef;6:8 XX udef;9:9 XX q;9:9 XX compound;6:10 XX udef;25:25 XX comp;32:32 XX q;33:33 XX q;34:34 XX q;33:34 XX udef;32:34 XX compound;32:34 XX udef;0:34 XX udef', '6:7 XX compound;11:11 XX q;10:11 XX proper;12:12 XX q;10:13 XX udef;29:29 XX comp;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;35:35 XX q;35:35 XX nominalization;37:37 XX q;37:37 XX nominalization;36:37 XX compound;38:38 XX q;38:38 XX nominalization', '4:4 XX q', '0:0 XX q;0:0 XX nominalization;1:2 XX compound;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;6:6 XX d;5:6 XX compound;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX d;19:19 XX p;20:20 XX p;22:22 XX p;21:22 XX udef;23:23 XX p;24:24 XX p;25:25 XX p;26:26 XX p;0:26 XX udef', '', '0:0 XX q;4:4 XX q;11:11 XX q;13:13 XX q;14:14 XX q;26:26 XX q;27:27 XX q;28:28 XX q;0:28 XX proper', '3:3 XX q;4:4 XX q', '4:4 XX q', '6:6 XX q;5:6 XX compound;7:7 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:9 XX focus', '6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;10:10 XX q;8:10 XX compound;11:11 XX q;12:12 XX q;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;13:22 XX udef;8:22 XX udef', '1:2 XX compound;6:6 XX q;13:13 XX q;13:13 XX nominalization;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;24:25 XX udef;26:26 XX q;27:27 XX udef;24:27 XX udef;19:27 XX udef;14:27 XX udef', '5:5 XX comp;8:10 XX udef;11:12 XX compound', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:16 XX proper;20:20 XX proper;0:20 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:7 XX def', '1:1 XX neg', '0:0 XX q;1:1 XX q;2:2 XX q;0:4 XX proper', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;6:7 XX udef;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;19:19 XX q;20:20 XX q;21:22 XX udef', '', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;14:14 XX nominalization;0:18 XX udef', '5:5 XX q;6:6 XX q;9:9 XX q;7:9 XX compound;11:11 XX q;10:11 XX proper;14:14 XX q;14:14 XX compound;13:14 XX udef;10:14 XX udef;7:14 XX udef', '4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;0:13 XX focus', '', '0:0 XX q;0:0 XX compound;2:2 XX q;2:2 XX compound;1:2 XX proper;3:3 XX q;3:3 XX compound;0:4 XX unknown;0:4 XX proper', '0:0 XX q;0:0 XX compound;2:2 XX q;2:2 XX compound;1:2 XX proper;3:3 XX q;3:3 XX compound;0:4 XX unknown;0:4 XX proper', '0:-1 XX EMPTY-LABEL', '2:3 XX compound;8:8 XX q;7:8 XX compound;20:20 XX q;21:21 XX q;22:22 XX q', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;9:9 XX q;7:10 XX idiom;3:10 XX udef;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;20:20 XX q;21:21 XX udef;19:21 XX udef;14:21 XX subord;0:21 XX udef', '14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX times;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization', '3:3 XX superl;15:15 XX times', '0:0 XX superl;1:2 XX udef;11:11 XX q;10:11 XX compound;17:17 XX q;18:18 XX q;19:19 XX times;22:22 XX q;21:22 XX compound;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;32:32 XX nominalization;31:32 XX udef;33:33 XX compound;31:33 XX compound', '7:7 XX q;8:8 XX q', '', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX q;8:8 XX compound;7:8 XX proper;7:8 XX proper', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:10 XX proper;15:15 XX q;14:15 XX compound;16:16 XX q;16:16 XX compound;16:16 XX udef;14:16 XX proper;0:16 XX focus', '0:1 XX proper;5:6 XX proper;7:7 XX q;5:7 XX compound;5:8 XX proper;20:20 XX nonsp;0:43 XX appos', '9:9 XX q;11:11 XX q;12:13 XX proper;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX proper;18:20 XX proper;0:20 XX compound', '5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;8:8 XX q;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization', '', '7:7 XX q;6:7 XX compound;8:8 XX q;9:10 XX proper;11:11 XX q;9:11 XX compound;9:12 XX proper;6:12 XX proper;13:13 XX q;15:15 XX q;15:15 XX compound;14:15 XX proper;14:16 XX proper;18:19 XX compound;22:22 XX q;21:22 XX compound;23:23 XX q;23:23 XX compound;23:24 XX udef;21:24 XX proper;14:24 XX appos;27:28 XX proper;27:29 XX compound;0:40 XX appos', '2:2 XX q;3:3 XX q;6:7 XX compound;8:9 XX proper;10:10 XX q;8:10 XX compound;11:11 XX q;13:13 XX q;12:14 XX udef;8:14 XX proper', '0:0 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;14:14 XX d;15:15 XX d;17:17 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;24:24 XX d;23:24 XX proper;0:24 XX focus', '0:1 XX proper;3:4 XX compound;7:7 XX q;7:7 XX compound;6:7 XX proper;6:8 XX proper;9:9 XX q;6:9 XX compound;10:10 XX q;0:29 XX appos', '', '11:11 XX q;18:18 XX neg', '9:9 XX comp;16:16 XX q;20:20 XX neg;28:28 XX comp', '5:6 XX udef;5:7 XX compound', '8:8 XX q;8:8 XX compound;7:8 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;8:8 XX q;8:8 XX compound;7:8 XX proper;9:9 XX d;7:9 XX proper;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;0:17 XX focus', '5:5 XX q', '1:1 XX neg', '6:6 XX q;7:7 XX q;8:8 XX q;11:12 XX compound;9:12 XX compound;8:12 XX udef;26:26 XX times;6:26 XX udef;28:28 XX q;28:28 XX udef;6:28 XX appos', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;0:29 XX proper', '2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;14:14 XX q;15:15 XX d;14:15 XX udef;16:16 XX d;18:18 XX d;20:20 XX d;0:20 XX focus', '2:2 XX q;3:3 XX q;2:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;6:9 XX udef;12:13 XX udef;19:19 XX q;20:20 XX q;19:21 XX udef', '5:5 XX q;6:6 XX q;7:7 XX times;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;19:19 XX q;19:19 XX named;20:20 XX q;20:20 XX compound;21:21 XX q;21:21 XX compound;19:22 XX proper;30:30 XX q;30:30 XX compound;29:30 XX proper;29:31 XX udef;29:32 XX compound;42:42 XX q;42:43 XX proper;47:47 XX compound', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:2 XX compound', '0:27 XX compound', '6:6 XX q;7:7 XX q;10:10 XX q;10:11 XX udef;8:11 XX udef', '1:1 XX neg;3:3 XX thing;21:21 XX q', '', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX proper;0:11 XX focus', '0:0 XX q;1:1 XX q;12:12 XX q;15:15 XX q;16:16 XX q;17:17 XX q;21:21 XX q;24:24 XX q;35:35 XX comp;0:38 XX proper', '3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;6:7 XX compound;8:8 XX q;9:9 XX q;10:10 XX q;9:11 XX udef;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;13:16 XX udef;9:16 XX udef;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;23:24 XX compound;26:26 XX q', '', '3:3 XX neg;4:4 XX mod;5:5 XX mod;7:7 XX q;8:8 XX q;8:8 XX compound;6:8 XX udef;10:10 XX mod;9:10 XX compound;6:10 XX udef;11:11 XX mod;12:12 XX mod;13:13 XX mod;14:14 XX mod;15:15 XX mod;16:16 XX mod;17:17 XX mod;18:18 XX mod;19:19 XX mod;20:20 XX mod;21:21 XX mod;22:22 XX mod;24:24 XX mod;23:24 XX compound;25:25 XX mod;26:26 XX q;26:26 XX udef', '4:4 XX q;5:5 XX q;10:10 XX q;13:14 XX compound;15:15 XX q;16:16 XX q;17:18 XX proper;19:20 XX compound;17:20 XX compound', '3:4 XX compound;5:5 XX q;6:7 XX compound;5:14 XX udef', '', '2:2 XX q;1:2 XX compound;3:3 XX q;5:5 XX d;4:5 XX proper;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;0:12 XX focus', '', '', '9:10 XX compound', '3:3 XX q;2:3 XX compound;4:4 XX q;6:6 XX q;7:7 XX q;7:8 XX proper;11:12 XX compound;7:12 XX appos;27:27 XX superl;0:36 XX appos', '', '2:2 XX mod;3:3 XX mod;4:4 XX mod;5:5 XX mod;6:6 XX mod;7:7 XX mod;8:8 XX q;9:9 XX mod;8:9 XX measure;8:9 XX udef;10:10 XX mod;11:11 XX mod;12:12 XX mod;14:14 XX mod;15:15 XX mod;16:16 XX mod;17:17 XX mod;18:18 XX mod;19:19 XX mod;21:21 XX mod;22:22 XX mod;23:23 XX mod;24:24 XX q;25:25 XX q;26:26 XX q;24:27 XX udef', '17:17 XX loc', '13:13 XX nonsp', '0:9 XX compound', '6:7 XX compound;0:22 XX appos', '3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;24:24 XX d;26:26 XX d;27:27 XX d;29:29 XX q;30:30 XX q;0:31 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;2:3 XX udef;4:4 XX q;5:5 XX q;0:9 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX q;5:7 XX udef;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX d;9:12 XX udef;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX q;21:21 XX q;23:23 XX q;24:24 XX q;25:25 XX q;0:26 XX focus', '4:4 XX q;5:5 XX q;6:6 XX times;8:8 XX q;9:9 XX q', '0:0 XX q;0:0 XX compound;5:5 XX q;6:6 XX q;7:7 XX q;0:25 XX proper', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;11:13 XX udef;0:13 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;0:13 XX focus', '', '1:1 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;0:8 XX focus', '0:0 XX q;5:5 XX q;6:6 XX q;14:15 XX compound;16:16 XX q;17:17 XX q;17:17 XX compound;16:17 XX udef;18:18 XX q;18:18 XX compound;0:18 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX q;0:18 XX udef', '2:2 XX q;2:2 XX compound;12:12 XX q;13:13 XX q;15:15 XX q;17:17 XX q;19:19 XX q', '3:3 XX comp;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;11:15 XX udef', '7:7 XX q;9:9 XX comp', '4:4 XX q;4:4 XX compound;21:21 XX comp', '6:6 XX q;0:6 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;19:19 XX d;0:19 XX udef', '2:3 XX compound;8:9 XX compound;10:10 XX q;10:10 XX compound;16:16 XX q;20:21 XX udef;0:21 XX subord', '0:1 XX compound;0:4 XX compound', '2:2 XX q;1:2 XX proper;3:3 XX q;4:5 XX proper;1:5 XX udef;18:18 XX q;18:19 XX udef;20:20 XX compound;0:20 XX subord', '1:1 XX q;2:2 XX q;11:11 XX entity;0:25 XX compound', '4:4 XX q;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization', '3:4 XX proper;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX compound;14:16 XX udef;17:17 XX q;14:17 XX compound;10:17 XX udef;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;22:23 XX udef;0:26 XX compound', '20:20 XX d;20:21 XX parg', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX q;11:11 XX compound;10:11 XX proper;12:12 XX d;14:14 XX d;0:14 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX conj;9:9 XX d;11:11 XX d;0:11 XX focus', '15:15 XX q;15:15 XX compound;14:15 XX proper', '4:4 XX q;5:5 XX q;6:6 XX q', '3:4 XX proper;6:7 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX compound;16:16 XX q;13:17 XX udef;11:17 XX udef;3:17 XX appos;23:24 XX compound;27:27 XX q;26:27 XX compound;26:27 XX proper;0:27 XX compound', '', '9:9 XX q;10:10 XX q;11:11 XX q;13:13 XX q;13:13 XX compound;14:14 XX q;14:14 XX compound;12:14 XX udef;12:14 XX udef;10:14 XX udef', '6:6 XX q;7:7 XX q;8:8 XX q', '9:9 XX neg', '2:2 XX q;11:11 XX q;12:12 XX q;14:14 XX q;14:14 XX named;22:22 XX q;23:23 XX q;24:24 XX q;0:25 XX compound', '9:9 XX q;10:10 XX q', '4:5 XX proper;6:6 XX q;12:12 XX q;13:13 XX q;14:14 XX times', '15:15 XX neg', '11:11 XX nonsp;14:14 XX q;15:15 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;26:26 XX q;28:28 XX q;29:29 XX q;29:29 XX udef;24:29 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:12 XX compound;13:13 XX d;0:13 XX focus', '', '12:12 XX q;12:12 XX compound', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX comp;8:10 XX subord;11:11 XX d;12:13 XX compound;14:14 XX d;0:14 XX focus', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;14:14 XX d;15:15 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;28:28 XX q;28:28 XX nominalization;31:31 XX q;31:31 XX nominalization;30:31 XX compound;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;36:36 XX d;38:38 XX q;38:38 XX compound;37:38 XX proper;39:39 XX q;39:39 XX compound;41:41 XX q;41:41 XX compound;40:41 XX compound;42:43 XX proper;0:43 XX focus', '17:17 XX q;18:18 XX q;18:18 XX compound;17:18 XX udef;20:21 XX udef;19:21 XX implicit;22:22 XX q;17:22 XX udef;23:23 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q', '0:1 XX compound;0:2 XX udef;12:13 XX compound;22:22 XX q;23:23 XX q;24:24 XX times;22:24 XX udef;22:24 XX compound;21:24 XX compound', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX d;7:7 XX d;10:10 XX q;10:10 XX compound;9:10 XX udef;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:17 XX compound;18:18 XX q;16:18 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX d;23:24 XX udef;16:24 XX udef;25:25 XX d;27:27 XX d;30:30 XX d;31:31 XX q;33:33 XX q;34:34 XX q;36:36 XX d;0:36 XX focus', '3:3 XX comp;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;23:24 XX udef;30:30 XX q;29:30 XX compound;35:35 XX q;36:36 XX q;35:36 XX udef;38:38 XX q;39:39 XX q;40:40 XX q;42:42 XX q;38:42 XX udef', '1:1 XX q;0:1 XX compound;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;10:11 XX udef;13:13 XX q;16:16 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;21:21 XX proper;17:21 XX udef;0:21 XX udef', '0:0 XX q;1:1 XX q;2:3 XX proper;4:4 XX q;5:5 XX q;5:5 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX conj;27:27 XX conj;0:28 XX proper', '', '13:13 XX q;14:14 XX q;13:15 XX udef', '0:0 XX superl;14:14 XX q;14:14 XX compound;13:14 XX proper;13:15 XX proper;27:27 XX q;26:27 XX compound;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX q;35:35 XX udef', '1:1 XX comp;8:8 XX d;7:8 XX loc;6:8 XX loc;9:9 XX d;10:10 XX d;11:11 XX d;12:13 XX compound;14:14 XX q;14:14 XX compound;12:14 XX udef;15:15 XX nominalization;12:15 XX udef;16:16 XX d;18:18 XX d;19:19 XX d;20:21 XX proper;22:22 XX d;20:22 XX compound;24:24 XX d;25:25 XX d;26:27 XX compound;26:27 XX udef', '7:7 XX comp;14:15 XX compound;13:15 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;19:24 XX udef', '0:0 XX q;0:0 XX compound;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;24:24 XX d;25:25 XX d;0:25 XX proper', '4:4 XX neg;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;34:34 XX q;34:34 XX nominalization;41:41 XX superl', '5:5 XX q;5:6 XX proper;9:9 XX nominalization;8:9 XX udef;8:10 XX compound;22:22 XX q;21:22 XX compound;23:23 XX q;21:24 XX proper;29:29 XX q;28:29 XX compound;28:30 XX proper;35:36 XX udef', '', '0:0 XX d;1:1 XX d;2:2 XX q;4:4 XX q;5:5 XX q;8:8 XX q;11:11 XX q;12:12 XX q;14:14 XX d;15:15 XX q;17:17 XX d;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;26:26 XX d;27:27 XX d;29:29 XX d;30:30 XX d;32:32 XX d;34:34 XX d;0:34 XX focus', '0:0 XX q;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX d;0:11 XX udef', '7:7 XX discourse;13:13 XX q', '8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;12:12 XX compound;11:12 XX proper;11:13 XX proper;15:15 XX q;14:16 XX udef;14:17 XX proper;18:18 XX q;10:18 XX udef;19:19 XX q;20:20 XX q;22:22 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;9:9 XX d;0:9 XX focus', '1:1 XX d;2:2 XX d;3:4 XX udef;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:11 XX proper;12:12 XX d;13:13 XX d;17:17 XX d;18:18 XX d;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;22:23 XX udef;24:24 XX q;24:24 XX nominalization;22:24 XX compound;25:25 XX q;25:25 XX nominalization;26:27 XX proper;29:29 XX udef;26:29 XX appos;0:29 XX focus', '1:1 XX neg', '5:5 XX q;4:5 XX udef;6:6 XX q;6:7 XX udef;4:7 XX udef;11:11 XX q;10:12 XX udef;20:20 XX times;25:25 XX neg', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;7:7 XX d;6:7 XX compound;8:8 XX d;9:9 XX d;10:11 XX proper;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;0:16 XX compound', '9:9 XX q;10:10 XX q;10:10 XX compound;9:10 XX udef;8:10 XX udef;23:23 XX neg', '0:0 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '0:0 XX q;2:2 XX q;1:2 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;7:8 XX compound;9:9 XX q;10:11 XX udef;15:15 XX q;16:16 XX q;17:17 XX times;0:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:10 XX proper;13:13 XX q;15:15 XX compound;14:15 XX compound;13:15 XX udef;12:15 XX udef;11:15 XX udef;0:15 XX focus', '9:9 XX q;10:10 XX q;11:11 XX times', '1:1 XX q;0:2 XX udef;7:7 XX q;6:7 XX compound;10:10 XX q;11:11 XX q;12:12 XX times;16:16 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX q;21:21 XX q;20:22 XX udef;23:23 XX conj;24:24 XX conj;25:25 XX conj;26:26 XX q;27:27 XX q;26:28 XX udef;29:29 XX conj;30:30 XX q;31:31 XX q;30:32 XX udef;33:33 XX conj;34:34 XX conj;36:36 XX conj;35:36 XX proper;38:38 XX conj;23:38 XX implicit;0:38 XX udef', '1:1 XX neg', '5:5 XX conj;7:7 XX conj;8:8 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;13:13 XX conj', '1:2 XX compound;23:23 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;0:8 XX focus', '10:10 XX q;9:10 XX compound;9:11 XX proper;12:17 XX loc', '0:0 XX nonsp;2:2 XX q;1:2 XX udef;3:3 XX q;5:5 XX d;4:5 XX udef;7:7 XX d;9:9 XX nonsp;0:9 XX loc', '0:0 XX q;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;0:25 XX udef', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX q;12:12 XX compound;12:14 XX udef;15:15 XX neg;0:15 XX focus', '4:4 XX thing', '2:2 XX q;8:8 XX q', '', '', '0:0 XX q;0:0 XX compound;12:12 XX q;0:20 XX proper', '3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;8:9 XX udef;7:9 XX udef;12:12 XX nonsp;13:13 XX nonsp;14:14 XX nonsp;15:15 XX nonsp;16:16 XX q;16:16 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization', '15:15 XX q;28:28 XX p;29:29 XX p', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX conj;7:7 XX conj;8:8 XX conj;11:11 XX conj;12:12 XX q;13:13 XX q;14:14 XX q;16:16 XX q;17:17 XX d;16:17 XX udef;12:17 XX udef;18:18 XX d;19:20 XX proper;22:23 XX compound;26:27 XX proper;26:28 XX compound;30:31 XX proper;32:32 XX q;30:32 XX compound;35:35 XX nonsp;36:36 XX loc;33:36 XX loc;30:36 XX proper;0:36 XX focus', '', '0:1 XX proper;3:5 XX measure;13:14 XX proper;13:15 XX compound;0:21 XX appos', '4:4 XX q;5:5 XX q;6:6 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;7:7 XX d;8:8 XX d;10:10 XX q;10:10 XX compound;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;0:17 XX focus', '', '0:0 XX d;1:1 XX d;0:6 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;0:10 XX focus', '', '0:0 XX d;1:1 XX d;3:3 XX q;3:4 XX udef;5:5 XX d;6:6 XX d;7:7 XX q;11:12 XX compound;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;0:20 XX focus', '0:6 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX q;8:8 XX q;9:9 XX q;11:11 XX d;10:11 XX loc;0:12 XX focus', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX d;2:3 XX udef;5:6 XX compound;11:11 XX d;10:11 XX proper;4:11 XX appos;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;19:19 XX d;20:20 XX d;0:20 XX focus', '0:0 XX conj;1:1 XX conj;3:3 XX conj;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;12:12 XX conj;13:13 XX conj;16:16 XX conj;17:17 XX conj;18:18 XX conj;22:22 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;0:25 XX implicit', '1:1 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;12:13 XX udef;14:14 XX d;0:15 XX focus', '0:10 XX subord', '5:5 XX q;6:6 XX q;7:7 XX q;5:7 XX udef', '0:12 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;14:15 XX proper;16:16 XX q;17:17 XX q;17:18 XX proper;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;23:23 XX nominalization;25:25 XX q;25:25 XX nominalization;23:25 XX udef;20:25 XX udef;14:25 XX udef;0:25 XX focus', '4:4 XX q;4:5 XX udef', '3:3 XX q;3:3 XX pron', '2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;0:6 XX focus', '7:7 XX q;8:8 XX q;10:10 XX q;9:10 XX compound;15:15 XX q;16:16 XX q;17:17 XX q;20:20 XX q;20:20 XX nominalization;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;0:29 XX appos', '', '0:0 XX d;2:2 XX d;4:4 XX conj;5:5 XX conj;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX d;11:11 XX d;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;0:19 XX focus', '', '4:4 XX q;5:5 XX q;8:9 XX proper;11:13 XX measure;14:15 XX proper;14:16 XX compound;23:24 XX compound;20:24 XX eventuality', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX conj;10:10 XX conj;11:11 XX conj;12:12 XX conj;14:14 XX conj;15:15 XX q;16:16 XX conj;15:16 XX udef;17:17 XX conj;18:18 XX q;19:19 XX d;18:19 XX udef;0:20 XX focus', '', '', '0:0 XX q;0:0 XX nominalization;2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;12:12 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;16:16 XX q;16:16 XX nominalization;22:22 XX q;22:22 XX nominalization;23:23 XX q;23:23 XX nominalization;0:23 XX udef', '2:2 XX conj;4:4 XX conj;5:5 XX conj;7:7 XX conj;6:7 XX udef;8:8 XX conj;9:9 XX conj;11:11 XX conj;12:12 XX conj;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;13:18 XX udef', '0:23 XX appos', '3:3 XX d;4:4 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:11 XX loc', '', '0:0 XX nonsp;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;15:15 XX d;16:16 XX d;0:16 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;9:11 XX measure;13:13 XX q;12:13 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX times;20:22 XX udef;24:24 XX q;23:24 XX compound;20:24 XX compound;25:25 XX q;0:26 XX focus', '', '3:3 XX comp', '3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX q;4:4 XX q;6:6 XX q;5:6 XX udef;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;17:17 XX q;18:20 XX compound;17:20 XX udef;0:25 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;3:3 XX compound;5:5 XX q;6:6 XX q;11:11 XX q;12:12 XX q;15:15 XX comp;17:17 XX thing;18:18 XX q;0:28 XX proper', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;7:7 XX d;0:12 XX udef', '1:1 XX thing', '1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;13:13 XX q;13:13 XX nominalization;14:14 XX q;14:14 XX nominalization;17:17 XX q;17:17 XX nominalization;15:17 XX compound;18:18 XX q;18:18 XX nominalization;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX q;29:29 XX times;27:29 XX udef;30:30 XX d;31:31 XX d;32:32 XX d;34:34 XX compound;0:34 XX udef', '4:4 XX q;3:4 XX compound;15:15 XX q;14:15 XX compound;14:16 XX proper;19:19 XX compound;18:19 XX compound', '0:0 XX q;1:1 XX d;0:1 XX udef;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;8:9 XX compound;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX q;18:18 XX q;17:18 XX udef;19:19 XX q;20:20 XX q;21:21 XX q;16:21 XX udef;0:21 XX focus', '0:0 XX q;1:1 XX q;0:7 XX udef', '0:0 XX d;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d;34:34 XX d;0:35 XX focus', '', '0:0 XX conj;1:1 XX conj;2:2 XX nominalization;3:3 XX conj;4:4 XX conj;5:5 XX q;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX conj;15:17 XX udef;18:18 XX q;19:19 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;0:24 XX implicit', '', '4:5 XX proper;11:11 XX q;12:13 XX compound;17:17 XX q;16:17 XX compound;14:17 XX compound;11:17 XX udef;18:18 XX q;19:19 XX q;19:19 XX udef', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;5:5 XX q;7:7 XX q;8:8 XX q;13:14 XX proper;15:15 XX q;13:15 XX compound;9:15 XX appos;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;23:24 XX udef;25:25 XX q;26:27 XX proper;31:31 XX d;30:31 XX compound;29:31 XX compound;26:31 XX appos;6:31 XX appos;32:32 XX d;33:33 XX d;2:33 XX udef;0:33 XX focus', '', '13:14 XX compound;15:16 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;0:7 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:7 XX compound;8:8 XX q;9:9 XX q;11:11 XX d;13:13 XX d;14:14 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX q;0:29 XX focus', '7:7 XX q;9:9 XX q;7:10 XX udef;13:13 XX nonsp;15:15 XX nonsp;11:15 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX d;9:9 XX d;8:9 XX compound;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX q;23:23 XX q;24:24 XX d;22:24 XX udef;25:25 XX d;26:26 XX d;27:27 XX q;28:28 XX d;27:28 XX udef;29:29 XX d;30:30 XX d;31:32 XX compound;33:33 XX d;31:33 XX compound;34:34 XX d;35:35 XX d;36:36 XX d;37:37 XX d;38:38 XX q;39:39 XX q;40:40 XX q;41:41 XX q;38:41 XX udef;7:41 XX proper;0:41 XX focus', '1:1 XX q;2:2 XX q;11:11 XX thing;0:16 XX udef', '7:7 XX q;17:17 XX compound;16:17 XX proper;0:17 XX compound', '', '0:0 XX q;7:7 XX q;10:10 XX q;11:11 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX times;0:30 XX proper', '11:11 XX q;12:12 XX q;15:16 XX compound;14:16 XX compound;19:19 XX comp;26:27 XX compound;28:28 XX q;26:28 XX compound', '0:0 XX q;2:2 XX q;18:18 XX q;0:28 XX proper', '1:2 XX compound;3:3 XX d;0:3 XX udef;4:4 XX d;5:5 XX q;6:6 XX d;12:12 XX q;12:12 XX compound;11:12 XX proper;13:13 XX d;11:13 XX proper;14:14 XX d;15:15 XX d;16:16 XX d;0:16 XX focus', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;7:7 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;0:16 XX udef', '13:13 XX q;17:17 XX q', '11:11 XX q;10:11 XX proper;12:12 XX q;14:14 XX q;13:14 XX compound;16:16 XX q;17:17 XX q;13:18 XX proper;20:20 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;24:25 XX udef;26:26 XX q;28:28 XX q;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;32:33 XX udef;21:33 XX udef', '1:1 XX q;2:2 XX q;4:4 XX q;0:10 XX measure', '4:4 XX q;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX proper', '0:0 XX d;1:1 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;10:12 XX udef;13:13 XX q;14:14 XX q;15:15 XX q;17:17 XX d;14:17 XX udef;10:17 XX udef;18:18 XX d;19:19 XX d;20:21 XX compound;23:23 XX d;24:24 XX d;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX q;29:29 XX times;27:29 XX udef;0:37 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:12 XX q;12:13 XX proper;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;19:20 XX udef;21:21 XX q;19:21 XX compound;22:22 XX q;26:26 XX q;25:26 XX compound;27:27 XX q;29:29 XX nonsp;30:30 XX q;29:30 XX loc;31:31 XX q;32:32 XX q;33:33 XX q;33:34 XX udef;35:35 XX q;33:35 XX compound;36:36 XX loc;18:36 XX udef;0:36 XX focus', '', '', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX q;13:14 XX compound;15:15 XX q;16:17 XX compound;18:18 XX q;16:18 XX compound;16:18 XX proper;0:18 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;10:10 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;0:18 XX focus', '0:0 XX q;1:1 XX q;5:5 XX comp;0:9 XX proper', '2:2 XX neg;16:16 XX q;17:17 XX q;0:17 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX compound;12:13 XX udef;0:13 XX proper', '5:5 XX q;7:7 XX q;8:8 XX q;7:9 XX udef;6:9 XX udef;10:10 XX q;6:10 XX compound;5:12 XX udef;13:13 XX q;14:15 XX compound;16:16 XX q;14:16 XX compound;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX udef;21:21 XX q;22:22 XX q;22:22 XX udef;19:22 XX udef;14:22 XX udef', '3:3 XX conj;5:6 XX compound;3:6 XX implicit;9:9 XX q;9:9 XX compound;8:9 XX proper;10:11 XX udef;10:11 XX compound;8:11 XX udef', '0:1 XX compound;2:2 XX q;3:3 XX q;4:5 XX proper;7:7 XX d;6:7 XX compound;4:7 XX compound;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX q;14:14 XX q;16:16 XX d;17:17 XX d;18:18 XX q;0:18 XX udef', '6:6 XX q;7:7 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q', '0:1 XX proper;10:10 XX q;10:10 XX compound;0:11 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX d;4:4 XX d;5:6 XX udef;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;0:15 XX udef', '0:0 XX q;0:0 XX compound;2:2 XX q;3:3 XX q;4:4 XX q;4:4 XX compound;0:20 XX proper', '0:0 XX d;1:1 XX d;3:3 XX q;2:3 XX compound;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX compound;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;17:17 XX compound;19:19 XX q;17:19 XX udef;20:20 XX q;21:21 XX q;23:23 XX d;22:23 XX compound;21:23 XX udef;24:24 XX d;25:25 XX d;0:25 XX focus', '0:-1 XX EMPTY-LABEL', '0:0 XX q;1:1 XX q;6:6 XX q;4:6 XX compound;15:15 XX q;15:15 XX compound;14:15 XX proper;0:21 XX udef', '3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX nonsp;10:10 XX q;11:11 XX d;0:15 XX loc', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX compound;8:9 XX proper;5:9 XX udef;0:9 XX focus', '2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization', '4:4 XX neg', '', '0:0 XX q;26:26 XX q;35:35 XX q;35:35 XX compound;37:37 XX q;37:37 XX compound;0:37 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;0:17 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;0:5 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;33:33 XX d;34:34 XX d;36:36 XX d;37:37 XX d;0:37 XX focus', '', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;0:11 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;0:3 XX focus', '10:11 XX proper;12:12 XX q;0:29 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '0:0 XX conj;2:2 XX conj;3:3 XX conj;5:5 XX q;5:5 XX compound;4:5 XX udef;6:6 XX conj;7:7 XX conj;8:8 XX conj;9:9 XX conj;10:10 XX conj;11:12 XX proper;14:14 XX conj;13:14 XX compound;15:15 XX conj;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;22:22 XX conj;23:23 XX q;24:24 XX q;26:26 XX conj;23:26 XX udef;27:27 XX conj;28:28 XX q;29:29 XX q;30:30 XX conj;29:30 XX udef;31:31 XX conj;33:33 XX d;34:34 XX d;36:36 XX d;37:37 XX d;38:38 XX d;39:39 XX q;40:40 XX q;41:41 XX q;42:42 XX q;42:42 XX udef;39:42 XX udef;0:42 XX implicit', '2:3 XX proper;4:5 XX compound', '', '2:2 XX q', '21:21 XX q;21:22 XX udef;29:29 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX q;19:19 XX d;18:19 XX proper;0:19 XX focus', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX q;4:5 XX compound;6:6 XX q;7:7 XX q;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;23:23 XX d;0:24 XX focus', '7:7 XX q', '7:7 XX q;7:8 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX q;0:13 XX focus', '3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;5:6 XX udef;15:15 XX q;15:15 XX named;16:16 XX q;16:16 XX named;15:17 XX proper;18:19 XX compound;22:22 XX q;21:22 XX compound', '3:3 XX q;4:5 XX compound;0:11 XX compound', '', '0:0 XX nonsp;1:1 XX nonsp;2:2 XX d;0:3 XX loc', '', '0:0 XX q;0:0 XX compound;0:11 XX proper', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;0:22 XX compound', '0:-1 XX EMPTY-LABEL', '2:3 XX proper;7:7 XX q;6:7 XX compound;8:8 XX q;6:9 XX proper;10:10 XX neg;15:15 XX q;15:15 XX compound;14:15 XX udef;16:17 XX compound', '5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX proper;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q', '', '2:2 XX q;1:2 XX proper;3:3 XX q;5:6 XX compound;7:7 XX proper;4:7 XX appos', '0:0 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX q;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;0:19 XX focus', '0:12 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;11:11 XX neg;16:16 XX compound;0:16 XX focus', '1:1 XX q;0:1 XX proper;2:2 XX q;8:8 XX q;8:8 XX nominalization;0:24 XX udef', '12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:22 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d', '', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;0:13 XX udef', '0:0 XX q;1:1 XX q;8:8 XX neg;14:14 XX person;17:17 XX conj;18:18 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;23:23 XX conj;24:24 XX conj;25:25 XX conj;0:25 XX proper', '7:7 XX neg;14:14 XX thing', '7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization', '', '1:1 XX neg', '0:0 XX q;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;7:8 XX proper;9:9 XX q;7:9 XX compound;11:11 XX q;13:13 XX q;12:14 XX udef;17:18 XX compound;17:19 XX udef;17:20 XX compound;16:20 XX compound;23:23 XX q;24:24 XX q;23:25 XX udef;31:31 XX q;31:31 XX proper;0:31 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX q;9:9 XX compound;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;17:17 XX d;19:19 XX d;0:20 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;6:6 XX d;7:8 XX proper;10:11 XX compound;14:14 XX q;13:14 XX compound;16:16 XX q;13:16 XX proper;0:16 XX focus', '0:1 XX number;2:3 XX compound;7:7 XX q;8:8 XX q;9:9 XX q;10:11 XX compound;13:13 XX q;18:18 XX q;31:31 XX comp;32:32 XX nonsp;33:33 XX nonsp;34:34 XX nonsp;35:35 XX nonsp;0:35 XX compound', '4:4 XX q;4:5 XX udef;9:10 XX number;0:11 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;8:8 XX d;9:9 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;17:18 XX number;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;0:26 XX focus', '1:1 XX q;2:2 XX q;14:14 XX q;14:14 XX nominalization;15:15 XX q;15:15 XX nominalization;16:17 XX udef;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;23:23 XX q;23:23 XX nominalization;0:23 XX udef', '0:0 XX d;1:1 XX d;4:4 XX d;3:4 XX compound;6:6 XX d;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX q;11:11 XX q;12:12 XX q;13:14 XX number;15:15 XX q;13:15 XX compound;16:16 XX q;18:18 XX q;20:21 XX udef;22:22 XX q;20:22 XX compound;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;29:29 XX q;30:30 XX q;0:30 XX focus', '0:12 XX generic', '3:3 XX q;13:15 XX udef', '11:12 XX compound;15:15 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX compound;15:23 XX udef', '0:0 XX q;0:11 XX udef', '11:11 XX q;12:12 XX q;15:15 XX q;16:17 XX compound;15:17 XX udef;18:18 XX comp;20:20 XX comp;25:25 XX subord;25:25 XX comp;27:27 XX neg', '0:1 XX number;14:14 XX q;15:15 XX q;16:16 XX q;0:17 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;0:14 XX focus', '0:0 XX d;2:2 XX d;6:6 XX d;7:7 XX d;9:9 XX q;8:9 XX compound;8:10 XX proper;11:11 XX q;12:12 XX q;12:13 XX proper;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;0:18 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;9:9 XX d;11:11 XX d;0:12 XX focus', '4:4 XX neg', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX d;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:13 XX number;14:14 XX d;12:14 XX compound;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;0:23 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;23:23 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;30:30 XX d;32:32 XX d;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX q;40:40 XX q;39:40 XX compound;41:41 XX q;0:41 XX focus', '2:2 XX q;3:3 XX q;5:5 XX q;6:6 XX q;6:6 XX compound;8:9 XX compound;6:9 XX udef;18:18 XX q;18:18 XX compound;17:18 XX proper;17:19 XX proper', '7:8 XX proper', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX q;10:10 XX compound;11:11 XX d;0:11 XX focus', '2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;11:11 XX q;22:22 XX q;23:23 XX q;23:23 XX compound;0:26 XX focus', '', '2:2 XX q;1:2 XX compound;0:2 XX proper;0:3 XX proper;9:9 XX d;8:9 XX udef;0:9 XX appos;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;21:21 XX d;22:23 XX proper;25:26 XX udef;25:26 XX compound;0:26 XX focus', '', '15:15 XX neg;21:21 XX neg', '5:6 XX proper;8:8 XX q;7:9 XX udef;10:10 XX q;7:10 XX compound;5:10 XX compound;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:15 XX udef;21:21 XX q;21:22 XX udef;24:24 XX q;23:24 XX compound;21:24 XX compound;17:24 XX loc;25:25 XX q;27:27 XX q;26:27 XX number;28:28 XX q;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX number;29:31 XX udef;14:31 XX loc', '1:1 XX q;3:3 XX q;3:3 XX compound;2:3 XX proper;4:4 XX q;2:4 XX proper;5:5 XX q;6:7 XX proper;10:10 XX q;19:19 XX q;0:20 XX udef', '3:4 XX udef;10:10 XX q;13:13 XX q;11:13 XX compound;14:14 XX q;15:15 XX q;17:17 XX q;18:18 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX q;25:25 XX q;28:28 XX q;26:28 XX compound;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;31:32 XX udef;33:33 XX q;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX q;41:41 XX q;42:42 XX q;39:42 XX udef;31:42 XX udef;26:42 XX udef;43:43 XX q;44:44 XX q;45:45 XX q;46:46 XX nonsp;46:46 XX loc', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX q;13:13 XX q;0:13 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;6:6 XX q;5:6 XX compound;7:7 XX q;0:7 XX focus', '', '9:9 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:24 XX appos', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;10:10 XX q;9:10 XX compound;11:11 XX q;12:13 XX proper;15:15 XX q;14:15 XX compound;12:15 XX compound;16:16 XX q;16:16 XX compound;16:16 XX udef;12:16 XX proper;0:16 XX focus', '1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;8:9 XX udef;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;15:19 XX udef;5:19 XX udef', '0:0 XX q;1:4 XX udef;6:7 XX udef;8:8 XX q;6:8 XX compound;9:9 XX q;10:10 XX q;10:12 XX udef;0:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '5:5 XX q;5:6 XX udef;7:8 XX compound;10:10 XX q;17:17 XX q', '2:2 XX neg', '4:4 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;6:6 XX d;0:20 XX focus', '2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;13:13 XX d;15:15 XX d;16:16 XX d;19:19 XX d;18:19 XX compound;20:20 XX d;21:21 XX d;25:25 XX d;0:25 XX focus', '8:8 XX q;7:8 XX compound;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX nonsp;19:19 XX nonsp;18:19 XX loc', '1:1 XX q;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;10:10 XX compound;12:12 XX d;10:12 XX udef;13:13 XX d;14:14 XX d;0:14 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;0:7 XX focus', '4:4 XX comp', '2:2 XX q;3:4 XX compound', '5:5 XX q', '', '11:12 XX compound', '11:12 XX compound', '6:6 XX q;9:9 XX q;7:9 XX compound;10:10 XX q;11:11 XX q;13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper;15:15 XX q;12:15 XX compound;16:16 XX q;19:21 XX compound;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;0:33 XX subord', '0:0 XX d;1:1 XX d;2:2 XX q;3:3 XX q;2:3 XX udef;4:5 XX udef;8:8 XX d;7:8 XX udef;4:8 XX udef;10:10 XX entity;11:11 XX conj;13:13 XX conj;14:14 XX conj;16:16 XX conj;17:17 XX conj;19:19 XX conj;20:20 XX conj;21:21 XX conj;0:21 XX focus', '3:3 XX neg;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q', '2:2 XX q;2:2 XX nominalization;4:4 XX q;4:4 XX nominalization;5:5 XX q;5:5 XX nominalization;6:6 XX q;6:6 XX nominalization', '12:12 XX q;22:23 XX proper;25:25 XX q;24:25 XX compound;26:26 XX q;28:28 XX q;27:28 XX compound;27:28 XX proper;24:28 XX udef', '4:4 XX q;4:4 XX compound;14:14 XX q', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;0:9 XX focus', '2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;15:15 XX nominalization;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;23:23 XX d;0:23 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;21:21 XX q;21:21 XX compound;20:21 XX proper;0:21 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;6:8 XX udef;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX d;19:25 XX udef;14:25 XX udef;26:26 XX d;28:28 XX d;27:28 XX compound;0:28 XX focus', '4:4 XX q;4:4 XX nominalization;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;25:25 XX q;27:27 XX nonsp;26:28 XX loc;29:29 XX q;30:30 XX q;31:31 XX q;32:32 XX q;33:33 XX q;34:34 XX q;35:35 XX udef;11:35 XX udef;5:35 XX udef', '1:1 XX neg;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;19:19 XX q;19:19 XX compound;18:19 XX udef;18:20 XX udef;22:22 XX udef;18:22 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX nonsp;11:11 XX nonsp;12:12 XX nonsp;13:13 XX nonsp;14:14 XX d;15:15 XX d;16:17 XX proper;20:20 XX q;20:20 XX compound;19:20 XX proper;19:21 XX proper;19:22 XX compound;0:22 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;14:14 XX d;18:18 XX q;19:19 XX d;18:19 XX udef;15:19 XX generic;20:20 XX d;21:22 XX proper;23:23 XX q;24:24 XX q;25:26 XX proper;27:27 XX q;25:27 XX compound;25:27 XX proper;23:27 XX udef;21:27 XX appos;0:27 XX focus', '4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;11:11 XX q;9:11 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX compound;14:15 XX udef;8:15 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX q;16:16 XX compound;0:17 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:7 XX proper;9:10 XX compound;13:13 XX q;13:13 XX compound;12:13 XX proper;12:14 XX proper;16:17 XX proper;18:18 XX compound;16:18 XX compound;12:18 XX appos;0:18 XX focus', '', '2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;6:6 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization', '0:0 XX d;1:1 XX d;2:2 XX q;2:2 XX nominalization;4:4 XX d;3:4 XX udef;5:5 XX d;6:6 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:30 XX proper;31:31 XX q;32:32 XX q;34:34 XX q;34:34 XX compound;33:34 XX proper;35:35 XX q;33:35 XX proper;38:38 XX q;37:39 XX udef;31:40 XX udef;0:40 XX focus', '0:0 XX q;0:0 XX nominalization;2:2 XX q;2:2 XX nominalization;3:3 XX q;3:3 XX nominalization;11:11 XX q;12:12 XX q;0:13 XX udef', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX q;5:5 XX d;6:6 XX q;7:7 XX d;8:8 XX d;9:9 XX q;10:10 XX d;11:11 XX d;14:14 XX d;0:14 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;13:13 XX d;14:14 XX d;0:23 XX focus', '', '2:2 XX q;0:19 XX udef', '3:3 XX superl', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;7:8 XX udef;9:9 XX q;10:10 XX q;11:11 XX d;10:11 XX udef;12:12 XX d;13:13 XX d;14:14 XX d;17:17 XX d;18:18 XX d;20:20 XX d;21:21 XX d;22:22 XX q;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX q;27:27 XX d;28:28 XX d;29:29 XX q;30:30 XX q;31:31 XX q;31:31 XX nominalization;32:32 XX q;32:32 XX nominalization;33:33 XX q;33:33 XX nominalization;34:34 XX q;34:34 XX nominalization;35:35 XX q;35:35 XX nominalization;36:36 XX q;36:36 XX nominalization;37:37 XX d;38:38 XX d;0:39 XX focus', '1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX q;5:5 XX nominalization;7:7 XX d;8:8 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX d;15:16 XX udef;17:17 XX d;18:18 XX q;18:18 XX nominalization;19:19 XX q;20:20 XX q;20:20 XX nominalization;19:20 XX udef;25:25 XX d;18:25 XX udef;5:25 XX udef;0:26 XX focus', '6:8 XX udef;15:15 XX q;15:15 XX named;17:17 XX entity;15:17 XX proper;15:18 XX generic;28:28 XX superl;31:31 XX nonsp;0:31 XX compound', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;10:10 XX d;11:11 XX d;0:12 XX focus', '6:6 XX q;7:7 XX q;8:8 XX q;9:10 XX udef;11:11 XX q;9:11 XX compound;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;22:22 XX q;0:22 XX proper', '0:0 XX q;1:2 XX proper;4:4 XX q;3:4 XX compound;5:5 XX q;6:6 XX q;9:9 XX q;10:10 XX q;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;14:14 XX compound;13:14 XX udef;15:15 XX q;13:15 XX udef;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX d;0:27 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;9:9 XX compound;8:9 XX udef;10:10 XX d;8:10 XX udef;0:11 XX focus', '9:9 XX superl', '5:5 XX nonsp;7:7 XX nonsp;8:8 XX nonsp;10:10 XX nonsp;9:10 XX compound;11:11 XX nonsp;12:12 XX nonsp;13:13 XX nonsp', '1:1 XX q;1:1 XX pron;2:2 XX q;3:3 XX q;7:7 XX d;5:7 XX compound;0:8 XX pronoun', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;11:11 XX d;14:15 XX compound;16:16 XX nominalization;14:16 XX udef;14:17 XX compound;19:19 XX q;19:19 XX nominalization;19:19 XX udef;0:19 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX nonsp;10:10 XX q;11:11 XX d;10:11 XX udef;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;0:21 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:4 XX proper;6:7 XX proper;9:9 XX d;8:9 XX compound;6:9 XX compound;10:10 XX d;0:10 XX focus', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;6:6 XX d;8:8 XX d;9:9 XX d;10:10 XX d;12:12 XX d;13:13 XX d;15:15 XX d;16:16 XX d;18:18 XX d;17:18 XX udef;0:19 XX focus', '0:0 XX d;2:2 XX d;5:5 XX d;6:6 XX d;7:7 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX d;11:13 XX udef;15:15 XX d;16:16 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;25:25 XX d;24:25 XX loc;26:26 XX d;27:27 XX d;30:30 XX d;29:30 XX proper;0:30 XX focus', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;7:7 XX d;0:7 XX focus', '0:0 XX d;1:1 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;14:14 XX d;0:15 XX focus', '7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;3:4 XX udef;5:5 XX q;6:6 XX q;7:7 XX q;9:9 XX q;8:9 XX udef;6:9 XX udef;0:9 XX unknown;0:9 XX udef', '0:0 XX q;0:1 XX unknown;0:1 XX udef', '2:2 XX q;1:2 XX compound', '1:1 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;12:12 XX compound;11:12 XX proper;14:14 XX q;15:15 XX q;15:15 XX compound;0:15 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;7:7 XX q;8:8 XX q;10:10 XX q;10:10 XX compound;9:10 XX proper;11:11 XX q;11:11 XX compound;9:11 XX udef;12:12 XX q;12:12 XX compound;14:14 XX q;15:15 XX q;16:16 XX q;17:18 XX proper;20:20 XX q;21:21 XX q;22:22 XX q;26:26 XX q;27:27 XX q;28:28 XX times;0:28 XX proper', '0:0 XX q;1:1 XX q;2:2 XX q;3:3 XX q;4:4 XX q;5:5 XX q;6:6 XX q;10:10 XX q;10:11 XX proper;7:11 XX appos;12:12 XX d;13:13 XX d;14:14 XX q;15:15 XX d;16:16 XX d;17:17 XX q;17:17 XX nominalization;18:18 XX q;18:18 XX nominalization;19:19 XX q;19:19 XX nominalization;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;23:23 XX q;23:23 XX nominalization;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;29:29 XX q;26:29 XX udef;30:30 XX q;31:31 XX q;32:32 XX q;31:32 XX udef;26:32 XX udef;24:32 XX udef;0:32 XX udef', '1:1 XX comp;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX compound', '0:0 XX q;1:1 XX q;2:2 XX q;4:4 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;0:22 XX proper', '8:8 XX conj;10:10 XX conj;9:10 XX udef;11:11 XX conj;12:12 XX conj;13:13 XX conj;14:14 XX conj;15:15 XX conj;16:16 XX conj', '0:0 XX q;2:2 XX q;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX q;10:10 XX q;13:13 XX q;14:14 XX q;24:24 XX q;25:25 XX q;26:27 XX proper;28:28 XX compound;0:28 XX udef', '4:4 XX q;7:7 XX q;8:8 XX q;9:9 XX q;9:11 XX udef;16:16 XX q;19:19 XX q;22:22 XX q', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX q;7:7 XX compound;6:7 XX udef;8:8 XX d;6:8 XX udef;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX q;12:13 XX udef;14:14 XX d;15:15 XX d;16:16 XX d;18:18 XX q;18:18 XX compound;17:18 XX proper;17:19 XX proper;20:20 XX d;27:28 XX compound;27:29 XX compound;0:32 XX focus', '19:19 XX q;20:20 XX q', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;3:3 XX q;3:3 XX nominalization;2:3 XX compound;4:4 XX q;4:4 XX nominalization;5:5 XX q;6:6 XX q;6:6 XX nominalization;5:6 XX udef;7:7 XX nonsp;8:8 XX nonsp;10:10 XX nonsp;9:10 XX proper;11:11 XX nonsp;13:13 XX nonsp;7:13 XX loc;15:15 XX d;14:15 XX loc;18:18 XX d;19:19 XX d;20:20 XX d;22:22 XX d;23:23 XX d;24:24 XX q;25:25 XX d;0:26 XX udef', '0:1 XX udef;15:15 XX q;15:15 XX nominalization;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;23:23 XX q;23:23 XX nominalization;22:23 XX compound;21:23 XX udef;18:23 XX udef;24:24 XX q;24:24 XX nominalization;25:25 XX q;25:25 XX nominalization;26:26 XX q;26:26 XX nominalization;27:27 XX q;27:27 XX nominalization;28:28 XX q;28:28 XX nominalization;29:29 XX q;29:29 XX nominalization;30:30 XX q;30:30 XX nominalization;31:31 XX q;31:31 XX nominalization;30:31 XX udef;16:31 XX udef;0:31 XX compound', '0:0 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX q;7:7 XX compound;9:9 XX q;9:10 XX udef;11:11 XX d;9:11 XX compound;12:12 XX d;13:14 XX udef;16:16 XX q;15:16 XX compound;13:16 XX compound;17:17 XX q;19:19 XX q;18:19 XX compound;20:20 XX q;20:20 XX compound;20:20 XX udef;18:20 XX proper;13:20 XX proper;7:20 XX udef;0:20 XX focus', '6:6 XX nonsp;7:7 XX nonsp;8:8 XX nonsp;9:9 XX nonsp;10:10 XX nonsp;11:11 XX nonsp;12:12 XX nonsp;13:13 XX nonsp;14:14 XX nonsp;15:15 XX nonsp;16:16 XX nonsp;17:17 XX nonsp;18:18 XX nonsp', '', '', '6:6 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;9:10 XX udef;11:11 XX q;13:13 XX q;15:15 XX q;16:16 XX q;14:16 XX udef;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;21:21 XX q;24:24 XX comp;29:29 XX q;29:29 XX compound;28:29 XX udef;30:31 XX compound;28:31 XX proper;23:32 XX udef;22:32 XX udef;21:32 XX udef;12:32 XX udef;9:32 XX udef', '10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;19:19 XX q;20:20 XX q;22:22 XX q;21:22 XX compound;23:23 XX q;24:24 XX q;26:26 XX q;26:26 XX compound;25:26 XX udef;12:26 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;25:25 XX udef;0:25 XX proper', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;4:4 XX d;5:5 XX q;6:6 XX q;7:7 XX q;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX d;13:13 XX d;14:14 XX d;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;20:20 XX q;19:20 XX compound;21:21 XX q;22:22 XX q;23:23 XX q;24:24 XX q;25:25 XX q;26:26 XX q;27:27 XX q;28:28 XX d;29:29 XX d;30:30 XX nonsp;31:31 XX nonsp;32:32 XX d;34:34 XX d;35:35 XX d;36:36 XX q;37:37 XX q;0:37 XX focus', '6:6 XX neg;9:9 XX q;10:10 XX q', '3:3 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;9:14 XX udef;27:27 XX q', '1:1 XX q;3:3 XX q;4:4 XX q;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;13:13 XX q;14:14 XX q;0:14 XX udef', '0:0 XX d;1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX d;7:7 XX d;9:9 XX d;10:10 XX d;12:12 XX d;14:14 XX d;15:15 XX d;17:17 XX d;18:18 XX d;19:19 XX d;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX q;23:23 XX nominalization;24:24 XX q;24:24 XX nominalization;0:24 XX focus', '4:4 XX q;16:16 XX q', '8:8 XX q;9:9 XX q;15:15 XX neg;36:36 XX q;0:36 XX subord', '3:3 XX q;3:3 XX compound;2:3 XX udef;4:5 XX compound;13:14 XX udef;21:21 XX superl;34:34 XX q;35:35 XX q;36:36 XX q;37:37 XX q;38:38 XX udef', '5:5 XX q;4:5 XX compound;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;14:14 XX q;15:15 XX q;16:16 XX q;17:17 XX q;18:18 XX q;21:21 XX q;28:28 XX comp', '0:0 XX q;1:1 XX q;3:3 XX q;4:4 XX q;4:4 XX compound;0:17 XX proper', '', '', '0:0 XX d;1:1 XX d;2:2 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;10:10 XX d;11:12 XX proper;14:15 XX compound;17:18 XX proper;19:19 XX q;17:19 XX compound;17:19 XX proper;0:19 XX focus', '0:0 XX q;0:0 XX nominalization;1:1 XX q;1:1 XX nominalization;0:10 XX udef', '4:4 XX q', '1:1 XX q;3:3 XX q;2:3 XX compound;2:4 XX proper;7:8 XX compound;13:13 XX d;11:13 XX compound;0:13 XX udef;14:14 XX d;15:15 XX d;16:16 XX q;17:17 XX d;16:17 XX udef;19:19 XX d;20:20 XX d;21:21 XX d;23:23 XX d;24:24 XX d;25:25 XX d;26:26 XX d;27:27 XX d;28:28 XX d;29:29 XX d;30:30 XX d;31:31 XX d;32:32 XX d;0:33 XX focus', '0:0 XX q;1:1 XX q;2:2 XX q;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;11:11 XX q;11:11 XX nominalization;12:12 XX q;13:13 XX q;0:29 XX udef', '0:0 XX q;1:1 XX q;3:3 XX d;4:4 XX d;5:5 XX d;6:6 XX d;7:7 XX d;8:8 XX d;9:9 XX d;10:10 XX d;11:11 XX q;12:12 XX q;14:14 XX d;0:14 XX udef', '6:6 XX q;8:8 XX neg;10:10 XX q;11:11 XX q', '0:0 XX q;6:6 XX q;7:7 XX q;8:8 XX times;16:16 XX q;16:16 XX nominalization;17:17 XX q;17:17 XX nominalization;18:18 XX q;19:19 XX q;0:19 XX udef', '1:1 XX q;3:3 XX d;4:4 XX q;5:5 XX q;7:7 XX q;9:9 XX d;10:10 XX d;11:11 XX d;12:12 XX d;13:13 XX d;14:14 XX d;0:14 XX udef', '0:0 XX q;5:5 XX q;6:7 XX compound;8:8 XX q;9:10 XX compound;0:17 XX udef', '0:20 XX unknown', '9:9 XX q;10:11 XX compound;12:12 XX nonsp;13:13 XX nonsp;14:14 XX nonsp', '2:2 XX q;2:3 XX udef', '13:13 XX neg', '0:0 XX q;0:0 XX compound;3:3 XX q;5:5 XX q;5:5 XX compound;6:6 XX q;6:6 XX compound;7:7 XX q;7:7 XX compound;9:9 XX q;10:10 XX q;12:12 XX q;16:16 XX neg;18:18 XX q;19:19 XX q;20:20 XX q;20:20 XX nominalization;21:21 XX q;21:21 XX nominalization;22:22 XX q;22:22 XX compound;23:23 XX q;23:23 XX compound;24:24 XX q;24:24 XX compound;26:26 XX q;28:28 XX q;28:28 XX nominalization;30:30 XX q;30:30 XX nominalization;0:31 XX udef', '1:1 XX d;2:2 XX d;3:3 XX d;5:5 XX q;7:7 XX q;8:8 XX q;9:9 XX q;10:10 XX q;12:12 XX q;13:13 XX q;14:14 XX q;15:15 XX q;16:16 XX q;18:18 XX q;17:18 XX proper;20:20 XX d;21:21 XX d;22:22 XX d;23:23 XX d;0:23 XX focus', '11:12 XX udef', '8:8 XX unknown;0:10 XX compound', '11:11 XX q', '5:5 XX q;5:5 XX nominalization;7:7 XX q;7:7 XX nominalization;8:8 XX q;8:8 XX nominalization;9:9 XX q;9:9 XX nominalization;10:10 XX q;10:10 XX nominalization;12:12 XX q;12:13 XX udef', '10:10 XX q;11:11 XX q;10:13 XX udef;16:16 XX neg;22:22 XX comp;25:25 XX q', '2:2 XX q;2:2 XX compound;1:2 XX udef;4:4 XX q;3:4 XX compound;1:4 XX proper;5:5 XX q;6:6 XX q;7:8 XX compound;6:8 XX udef;32:32 XX q', '']

surf_gold = ['0:0 Consumers _n_of;1:1 may _v_modal;2:2 want _v_1;4:4 move _v_cause;6:6 telephones _n_1;7:8 a little _x_deg;9:9 closer _a_to;10:10 to _p;11:11 the _q;12:12 TV _n_1;0:0 Consumers _n_of', '0:1 _n_1 Couch- potato;2:2 jocks _n_unknown;3:3 watching _v_1;4:4 ABC named;6:6 "Monday dofw;7:7 Night _n_of;8:8 Football" named;9:9 can _v_modal;10:10 now _a_1;11:11 vote _v_for;12:12 during _p;13:13 halftime _n_unknown;15:15 the _q;16:16 greatest _a_for;17:17 play _n_1;18:18 in _p;19:19 20 card;20:20 years _n_1;21:22 _p from among;23:23 four card;24:24 or _c;25:25 five card;26:26 filmed _v_1;0:0 Couch- _n_1', '0:0 Two card;1:1 weeks _n_1;2:2 ago _p;3:3 viewers _n_of;5:5 several _a_1;6:6 NBC named;7:7 daytime _n_1;8:8 consumer _n_of;9:9 segments _n_1;10:11 _v_1 started calling;12:12 a _q;13:13 900 card;14:14 number _n_of;15:15 for _p;16:16 advice _n_1;17:17 on _p;18:18 various _a_1;19:20 _n_of life- style;0:0 Two card', '0:0 And _c;1:1 the _q;2:2 new _a_1;3:3 syndicated _v_1;4:4 "reality" _n_1;5:5 show _n_of;6:7 named "Hard Copy";8:8 records _v_1;9:11 _n_of viewers opinions;12:12 for _p;13:13 possible _a_for;14:14 airing _v_1;15:15 on _p;16:16 the _q;17:17 next _a_1;18:20 _n_of day show', '0:0 Interactive _a_1;1:2 _n_1 telephone technology;4:4 taken _v_1;5:5 a _q;6:6 new _a_1;7:7 leap _n_1;8:8 in _p;9:9 sophistication _n_1;10:10 and _c;11:11 television _n_1;12:12 programmers _n_of;14:14 racing _v_1;15:15 to _x;16:16 exploit _v_1;17:17 the _q;0:0 Interactive _a_1', '0:0 Eventually _a_1;1:1 viewers _n_of;2:2 may _v_modal;3:3 grow _v_to;4:4 bored _a_1;5:5 with _p;6:6 the _q;7:7 technology _n_1;8:8 and _c;9:9 resent _v_1;10:10 the _q;0:0 Eventually _a_1', '0:0 But _c;1:1 right _x_deg;2:2 now _a_1;3:3 programmers _n_of;5:5 figuring _v_1;7:7 viewers _n_of;10:10 busy _a_1;11:11 dialing _v_1;12:12 up _p;13:13 a _q;14:14 range _n_of;16:16 services _n_1;17:17 may _v_modal;18:18 put _v_down;21:21 remote _a_1;22:22 control _n_of;23:23 zappers _n_unknown;24:24 and _c;25:25 stay _v_prd;0:0 But _c', '3:3 spending _v_1;4:4 a _q;5:7 _n_of lot time;8:8 in _p;9:10 named Los Angeles;11:11 talking _v_about;12:12 to _p;13:13 TV _n_1;14:15 _n_of production people,";16:16 says _v_to;17:18 named Mike Parks;19:19 president _n_of;21:22 named Call Interactive;24:24 supplied _v_1;25:25 technology _n_1;26:26 for _p;28:29 named ABC Sports;30:30 and _c;31:31 NBC named;33:33 consumer _n_of;3:3 spending _v_1', '0:0 "With _p;1:1 the _q;2:2 competitiveness _n_unknown;3:3 of _p;4:4 the _q;5:6 _n_1 television market;7:7 these _q_dem;8:8 days _n_of;11:11 looking _v_for;13:13 a _q;14:14 way _n_of;16:16 get _v_state;17:17 viewers _n_of;0:0 "With _p', '0:0 One card;2:2 the _q;3:3 leaders _n_of;4:4 behind _p;5:5 the _q;6:6 expanded _v_cause;7:7 use _n_of;9:9 900 yofc;10:10 numbers _n_of;11:11 is _v_id;12:13 named Call Interactive;14:14 a _q;15:15 joint _a_1;16:16 venture _n_1;17:17 of _p;18:18 giants _n_1;19:20 named American Express;21:21 Co _n_1;22:22 and _c;23:24 named American Telephone;25:25 & _c;26:26 Telegraph named;0:0 One card', '0:0 Formed _v_cause;1:1 in _p_temp;2:2 August mofy;3:3 the _q;4:4 venture _n_1;5:5 weds _v_1;6:6 AT&T named;8:8 newly _a_1;9:9 expanded _v_cause;10:10 900 yofc;11:11 service _n_1;12:12 with _p;13:13 200 card;14:14 voice- _n_1;15:15 activated _v_1;16:16 computers _n_1;17:17 in _p;18:19 named American Express;21:22 named Omaha Neb;23:23 service _n_1;0:0 Formed _v_cause', '0:0 Other _a_1;2:2 distance _n_1;2:2 distance _long_a_1;3:3 carriers _n_of;5:5 also _a_1;6:7 _v_1 begun marketing;8:8 enhanced _v_1;9:9 900 yofc;10:10 service _n_1;11:11 and _c;12:12 special _a_1;13:13 consultants _n_1;15:15 springing _v_up;17:17 to _x;18:18 exploit _v_1;19:19 the _q;20:20 new _a_1;0:0 Other _a_1', '0:1 named Blair Entertainment;2:2 a _q;3:4 named New York;5:5 firm _n_1;7:7 advises _v_1;8:9 _n_1 TV stations;10:10 and _c;11:11 sells _v_1;12:12 ads _n_1;13:13 for _p;16:16 just _a_1;17:17 formed _v_cause;18:18 a _q;19:19 subsidiary _n_1;21:21 900 yofc;22:22 Blair named;24:24 to _x;25:25 apply _v_to;26:26 the _q;27:29 _n_1 technology television', '0:0 The _q;1:1 use _n_of;3:3 900 yofc;4:4 toll _n_1;5:5 numbers _n_of;8:8 expanding _v_1;9:9 rapidly _a_1;10:10 in _p_temp;11:11 recent _a_1;0:0 The _q', '0:0 For _p;1:2 a while _awhile_n_1;4:4 cost _n_1;4:4 cost _high_a_1;5:5 pornography _n_unknown;6:6 lines _n_of;7:7 and _c;8:8 services _n_1;10:10 tempt _v_1;11:11 children _n_1;13:13 dial _v_1;14:14 (and _c;15:15 redial) _v_1;15:15 redial) _re-_a_again;16:16 movie _n_of;17:17 or _c;18:18 music _n_1;19:19 information _n_on-about;20:20 earned _v_1;21:21 the _q;22:22 service _n_1;23:23 a _q;24:24 somewhat _x_deg;25:25 sleazy _a_unknown;26:26 image _n_of;27:27 but _c;28:29 _a_1 new legal;30:30 restrictions _n_of;32:32 aimed _v_at;34:34 trimming _v_1;0:0 For _p', '0:0 The _q;1:1 cost _n_1;2:2 of _p;3:3 a _q;4:4 900 yofc;5:5 call _n_1;7:7 set _v_1;9:9 the _q;10:10 originator _n_of;12:13 named ABC Sports;17:17 with _x_subord;18:18 the _q;19:19 cheapest _a_1;20:20 starting _v_1;21:21 at _p;22:22 75 card;0:0 The _q', '0:0 Billing _n_1;2:2 included _v_1;3:3 in _p;4:4 a _q;5:5 caller _n_of;7:7 regular _a_1;8:8 phone _n_1;0:0 Billing _n_1', '1:1 the _q;2:2 fee _n_1;3:3 the _q;4:4 local _a_1;5:5 phone _n_1;6:6 company _n_of;7:7 and _c;8:8 the _q;10:10 distance _n_1;10:10 distance _long_a_1;11:11 carrier _n_of;12:12 extract _v_from;14:14 costs _n_1;16:16 carry _v_1;17:17 the _q;18:18 call _n_1;19:19 passing _v_1;20:20 the _q;21:21 rest _n_of;23:23 the _q;24:24 money _n_1;26:26 the _q;27:27 originator _n_of;29:29 must _v_modal;30:30 cover _v_1;31:31 advertising _n_1;32:32 and _c;33:33 other _a_1;1:1 the _q', '0:0 In _p_temp;1:1 recent _a_1;2:2 months _n_1;3:3 the _q;4:4 technology _n_1;6:6 become _v_id;8:8 flexible _a_1;9:9 and _c;10:10 able _a_1;12:12 handle _v_1;13:13 much _x_deg;0:0 In _p_temp', '0:0 Before _p;1:1 callers _n_of;3:3 900 yofc;4:4 numbers _n_of;5:5 would _v_modal;6:6 just _a_1;7:7 listen _v_to;8:9 and not _c;10:10 talk _v_about;11:11 or _c;13:13 \'d _v_modal;14:14 vote _v_1;15:15 "yes" _a_1;16:16 or _c;17:17 "no" _a_1;18:18 by _p_means;19:19 calling _v_1;20:22 card one two;0:0 Before _p', '0:0 (People _n_of;1:1 in _p;2:2 the _q;3:4 _n_1 phone business;5:5 call _v_1;6:6 this _q_dem;7:7 technology _n_1;8:8 "900 yofc;0:0 (People _n_of', '0:0 Now _a_1;1:1 callers _n_of;3:3 led _v_1;4:4 through _p;5:5 complex _a_1;6:6 menus _n_1;7:7 of _p;8:8 choices _n_of;9:9 to _x;10:10 retrieve _v_1;11:11 information _n_on-about;13:13 want _v_1;14:14 and _c;15:15 the _q;16:16 hardware _n_1;17:17 can _v_modal;18:18 process _v_1;19:19 10,000 card;20:20 calls _n_1;21:21 in _p;22:22 90 card;0:0 Now _a_1', '0:1 Up to _p;2:2 now _a_1;3:3 900 yofc;4:4 numbers _n_of;6:6 mainly _a_1;8:8 used _v_1;9:9 on _p;10:10 local _a_1;11:12 _n_1 TV stations;13:13 and _c;14:15 _n_1 cable channels', '0:0 MTV named;1:1 used _v_1;2:2 one card;4:4 give _v_away;6:6 the _q;7:7 house _n_of;9:10 _n_1 rock star;11:12 named Jon Bon;13:13 Jovi named;14:14 grew _v_up;0:0 MTV named', '0:0 The _q;1:1 newest _a_1;2:2 uses _n_of;4:4 the _q;5:5 900- yofc;6:6 interactive _a_1;7:7 technology _n_1;8:8 demonstrate _v_to;9:9 the _q;10:10 growing _v_1;11:11 variety _n_of-n;0:0 The _q', '0:1 named Capital Cities;3:3 ABC named;4:4 Inc _n_1;5:5 CBS named;6:6 Inc _n_1;7:7 and _c;8:9 named General Electric;10:10 Co _n_1;12:13 named National Broadcasting;14:14 Co _n_1;15:15 unit _n_of;17:17 expected _v_1;19:19 announce _v_to;20:20 soon _p;21:21 a _q;22:22 joint _a_1;23:23 campaign _n_1;25:25 raise _v_cause;26:26 awareness _n_1;27:27 about _p;0:0 Capital named', '0:0 The _q;1:1 subject _n_of;4:4 written _v_to;5:5 into _p;6:6 the _q;7:7 plots _n_of;9:9 prime- _n_1;10:11 _n_of time shows;12:12 and _c;13:13 viewers _n_of;16:16 given _v_1;17:17 a _q;18:18 900 yofc;19:19 number _n_of;0:0 The _q', '0:0 Callers _n_of;3:3 sent _v_1;4:4 educational _a_1;5:5 booklets _n_of;6:6 and _c;7:7 the _q;8:8 call _n_1;10:10 modest _a_1;11:11 cost _n_1;13:13 be _v_id;14:14 an _q;15:15 immediate _a_1;16:16 method _n_of;18:18 raising _v_cause;0:0 Callers _n_of', '0:0 Other _a_1;1:2 _n_of network applications;3:3 have _v_1;4:4 very _x_deg;5:5 different _a_than-from;0:0 Other _a_1', '0:1 named ABC Sports;3:3 looking _v_for;5:5 ways _n_of;7:8 _v_cause lift deflated;9:9 halftime _n_unknown;10:10 ratings _n_1;11:11 for _p;12:12 "Monday dofw;13:13 Night _n_of;0:0 ABC named', '0:1 named Kurt Sanger;2:3 named ABC Sports;5:5 marketing _v_1;6:6 director _n_of;7:7 says _v_to;9:9 now _a_1;10:12 card "tens thousands";14:14 fans _n_1;15:15 call _v_1;17:17 900 yofc;18:18 number _n_of;19:19 each _q;20:20 week _n_1;21:21 to _x;22:22 vote _v_for;24:24 the _q;25:25 best _a_at-for-of;26:26 punt _n_1;27:27 return _n_of;28:28 quarterback _n_unknown;29:29 sack _n_1;0:0 Kurt named', '0:0 Jackets _n_1;1:1 may _v_modal;3:3 sold _v_1;0:0 Jackets _n_1', '0:0 Meanwhile _p;1:2 named NBC Sports;3:3 recently _a_1;4:4 began _v_1;5:6 named "Scores Plus,";7:7 a _q;8:9 year- round _a_1;11:11 hour _n_1;11:11 hour card;12:12 900 yofc;13:13 line _n_of;14:14 providing _v_1;15:15 a _q;16:16 complex _a_1;17:17 array _n_1;18:18 of _p;19:20 _n_of scores analysis;21:21 and _c;22:22 fan _n_1;0:0 Meanwhile _p', '0:0 A _q;1:1 spokesman _n_1;2:2 said _v_to;4:4 purpose _n_1;6:6 "to _x;7:7 bolster _v_1;8:8 the _q;9:9 impression _n_of;11:12 named NBC Sports;14:14 always _x_deg;15:15 there _a_1;16:16 for _p;0:0 A _q', '0:2 named NBC "On-;3:3 Line" named;4:4 consumer _n_of;5:5 minutes _n_1;7:7 increased _v_cause;8:8 advertiser _n_unknown;9:9 spending _v_1;10:10 during _p;11:11 the _q;12:12 day _n_of;13:13 the _q;14:14 network _n_of;16:17 _a_1 weakest period', '0:0 Each _q;1:1 weekday _n_1;2:2 matches _v_cause;3:3 a _q;4:4 sponsor _n_of;5:5 and _c;6:6 a _q;7:7 topic _n_of;8:8 : _p_namely;9:9 On _p_temp;10:10 Mondays dofw;11:11 Unilever named;12:12 N.V _n_1;14:15 named Lever Bros;16:16 sponsors _v_1;17:17 tips _n_1;18:18 on _p;19:19 diet _n_1;20:20 and _c;21:21 exercise _n_1;22:22 followed _v_1;24:24 a _q;26:26 second _n_1;26:26 second card;27:28 named Lever Bros;0:0 Each _q', '0:0 If _x_then;1:1 the _q;2:2 caller _n_of;3:3 stays _v_prd;4:4 on _p;5:5 the _q;6:6 line _n_of;7:7 and _c;8:8 leaves _v_1;9:9 a _q;10:10 name _n_of;11:11 and _c;12:12 address _n_1;13:13 for _p;14:14 the _q;15:15 sponsor _n_of;16:16 coupons _n_1;17:17 and _c;18:18 a _q;19:19 newsletter _n_of;22:22 mailed _v_1;23:23 and _c;24:24 the _q;25:25 sponsor _n_of;28:28 able _a_1;30:30 gather _v_coll;31:31 a _q;32:32 list _n_of;34:34 desirable _a_to;35:35 potential _a_1;0:0 If _x_then', '0:1 named Diane Seaman;2:2 an _q;3:4 named NBC- TV;5:5 vice _n_1;6:6 president _n_of;7:7 says _v_to;8:8 NBC named;11:11 able _a_1;13:13 charge _v_1;14:14 premium _a_1;15:15 rates _n_of;16:16 for _p;17:17 this _q_dem;18:18 ad _n_1;0:0 Diane named', "1:1 wouldn't _v_modal;2:2 say _v_to;4:4 the _q;5:5 premium _n_1;6:6 is _v_id;7:7 but _c;10:10 believed _v_1;12:12 be _v_id;13:13 about _x_deg;14:14 40 card;15:15 % _n_of;16:16 above _p;17:17 regular _a_1;18:18 daytime _n_1;1:1 wouldn't _v_modal", '2:2 able _a_1;4:4 get _v_cause;5:5 advertisers _n_unknown;7:7 use _v_1;9:9 promotion _n_of;10:10 budget _n_1;11:11 for _p;12:12 this _q_dem;13:13 because _x;15:15 get _v_1;16:16 a _q;17:17 chance _n_of;19:19 do _v_1;20:20 couponing," _n_unknown;21:21 says _v_to;22:22 Ms _n_1;2:2 able _a_1', '0:0 "And _c;3:3 able _a_1;5:5 attract _v_1;6:6 some _q;7:7 new _a_1;8:8 advertisers _n_unknown;9:9 because _x;10:10 this _q_dem;11:11 is _v_id;12:12 something _q;0:0 "And _c', '0:0 Mr _n_1;1:1 Parks named;2:2 of _p;3:4 named Call Interactive;5:5 says _v_to;6:7 _n_1 TV executives;9:9 considering _v_1;10:10 the _q;11:11 use _n_of;13:13 900 yofc;14:14 numbers _n_of;15:15 for _p;16:16 "talk _n_of-on;17:17 shows _n_of;18:18 game _n_1;19:20 _n_of shows news;21:21 and _c;22:22 opinion _n_of;0:0 Mr _n_1', '0:0 Experts _n_1;2:2 predicting _v_1;3:3 a _q;4:4 big _a_1;5:5 influx _n_1;6:6 of _p;7:7 new _a_1;8:8 shows _n_of;9:9 in _p_temp;10:10 1990 yofc;11:11 when _x_subord;12:12 a _q;13:13 service _n_1;14:14 called _v_name;15:15 "automatic _a_1;16:16 number _n_of;17:17 information" _n_on-about;19:19 become _v_id;20:21 _a_1 widely available', '0:0 This _q_dem;1:1 service _n_1;2:2 identifies _v_1;3:3 each _q;4:4 caller _n_of;6:6 phone _n_1;7:7 number _n_of;8:8 and _c;10:10 can _v_modal;12:14 _v_1 used generate;15:15 instant _a_1;16:16 mailing _n_1;0:0 This _q_dem', '0:0 "For _p;1:1 a _q;2:2 piece _n_of;3:3 on _p;4:4 local _a_1;5:5 heroes _n_1;6:6 of _p;7:8 named World War;9:9 II named;11:11 can _v_modal;12:12 ask _v_1;13:13 people _n_of;15:15 leave _v_1;16:16 the _q;17:17 name _n_of;18:18 and _c;19:19 number _n_of;20:20 of _p;21:21 anyone _q;23:25 _v_1 know won;26:26 a _q;27:27 medal," _n_for;0:0 "For _p', '0:0 "That _q_dem;2:2 save _v_1;4:4 time _n_of;5:5 and _c;6:6 get _v_state;7:7 people _n_of;0:0 "That _q_dem', '0:0 But _c;1:1 Mr _n_1;2:2 Monsky named;3:3 sees _v_1;4:4 much _x_deg;5:5 bigger _a_1;6:6 changes _n_of;0:0 But _c', '0:0 "These _q_dem;1:1 are _v_id;2:2 just _x_deg;3:4 _n_1 baby steps;5:5 toward _p;6:7 _a_1 real interactive;8:8 video _n_of;11:11 believe _v_1;13:13 be _v_id;14:14 the _q;15:15 biggest _a_1;16:16 thing _n_of-about;17:17 yet _a_1;19:19 affect _v_1;20:20 television," _n_1;0:0 "These _q_dem', '0:0 Although _x;2:2 would _v_modal;4:4 costly _a_1;5:5 to _x;6:6 shoot _v_1;7:7 multiple _a_1;8:8 versions _n_of;9:9 TV _n_1;10:10 programmers _n_of;11:11 could _v_modal;12:12 let _v_1;13:13 audiences _n_1;14:14 vote _v_1;15:15 on _p;16:16 different _a_than-from;17:17 endings _n_1;18:18 for _p;19:19 a _q;0:0 Although _x', '0:1 named Fox Broadcasting;2:2 experimented _v_with;4:4 this _q_dem;5:5 concept _n_1;6:6 last _a_1;7:7 year _n_1;8:8 when _x_subord;9:9 viewers _n_of;11:11 "Married named;13:13 With _p;14:14 Children" _n_1;15:15 voted _v_1;16:16 on _p;18:18 Al named;19:19 should _v_modal;20:20 say _v_to;22:22 love _v_1;24:24 to _p;25:25 Peg named;26:26 on _p_temp;27:27 Valentine named;0:0 Fox named', '0:0 Someday _q;1:1 viewers _n_of;2:2 may _v_modal;3:3 also _a_1;4:4 choose _v_from;5:5 different _a_than-from;6:8 _n_of depths news;0:0 Someday _q', '0:0 "A _q;1:1 menu _n_1;2:2 by _p_means;3:3 phone _n_1;4:4 could _v_modal;5:7 _v_1 let decide;10:10 interested _a_in;12:12 just _x_deg;13:13 the _q;14:14 beginning _v_1;16:17 _n_of story No;18:18 1 card;19:19 and _c;21:21 want _v_1;22:23 _n_of story No;24:24 2 card;25:25 in _p;26:26 depth," _n_of;27:27 Mr _n_1;28:28 Monsky named;0:0 "A _q', '2:4 _v_1 start see;5:7 _n_of shows viewers;8:8 program _v_1;9:9 the _q;2:2 start _v_1', '0:1 named Integrated Resources;2:2 Inc _n_1;3:3 the _q;4:4 troubled _a_1;6:6 services _n_1;6:6 services _financial_a_1;7:7 company _n_of;11:13 _v_1 trying sell;15:15 core _n_1;16:16 companies _n_of;17:17 to _x;18:18 restructure _v_cause;19:19 debt _n_1;20:20 said _v_to;21:21 talks _n_of-on;22:22 with _p;23:23 a _q;24:24 potential _a_1;25:25 buyer _n_of;0:0 Integrated named', '0:0 Integrated named;2:2 identify _v_1;3:3 the _q;4:4 party _n_of-to;5:5 or _c;6:6 say _v_to;7:7 why _p;8:8 the _q;9:9 talks _n_of-on;0:0 Integrated named', '0:0 Last _a_1;1:1 week _n_1;2:2 another _q;3:3 potential _a_1;4:4 buyer _n_of;5:6 named Whitehall Financial;7:7 Group _n_of;11:11 agreed _v_1;12:12 in _p_temp;13:13 August mofy;15:15 purchase _v_1;16:16 most _q;18:18 Integrated named;20:20 core _n_1;21:21 companies _n_of;22:22 for _p;23:23 $ _n_1;24:25 card 310 million;27:27 ended _v_cause;28:28 talks _n_of-on;29:29 with _p;0:0 Last _a_1', '0:0 A _q;1:1 price _n_of;0:0 A _q', '0:0 Integrated named;1:1 also _a_1;2:2 said _v_to;4:4 expects _v_1;6:6 report _v_to;7:7 a _q;9:9 quarter _n_temp;9:9 quarter ord;10:10 loss _n_of;11:11 wider _a_1;13:13 the _q;14:14 earlier _a_1;15:15 estimate _n_of;17:17 about _x_deg;18:18 $ _n_1;19:20 card 600 million', '0:0 Meanwhile _p;1:1 a _q;2:2 number _n_of;4:4 top _a_1;5:6 _n_of sales producers;7:7 from _p;8:9 named Integrated Resources;10:10 Equity named;12:12 meet _v_1;13:13 this _q_dem;14:14 afternoon _n_of;15:15 in _p;16:16 Chicago named;17:17 to _x;18:18 discuss _v_1;0:0 Meanwhile _p', '0:0 The _q;1:1 unit _n_of;2:2 is _v_id;3:3 a _q;4:4 loosely _a_1;5:5 constructed _v_1;6:6 group _n_of;8:8 about _x_deg;9:9 3,900 card;10:10 independent _a_of;11:11 brokers _n_1;12:12 and _c;13:13 financial _a_1;14:14 planners _n_1;16:16 sell _v_1;17:17 insurance _n_1;18:18 annuities _n_unknown;19:19 limited _v_1;20:20 partnerships _n_1;21:21 mutual _a_1;22:22 funds _n_1;23:23 and _c;24:24 other _a_1;25:25 investments _n_1;26:26 for _p;27:27 Integrated named;28:28 and _c;29:29 other _a_1;0:0 The _q', '0:0 The _q;1:1 sales _n_of;2:2 force _n_1;4:4 viewed _v_as;6:6 a _q;7:7 critical _a_of;8:8 asset _n_1;9:9 in _p;10:10 Integrated named;12:12 attempt _n_1;14:14 sell _v_1;16:16 core _n_1;0:0 The _q', '0:0 Whitehall named;1:1 cited _v_as;2:2 concerns _n_1;3:3 about _p;5:5 long _a_1;6:6 Integrated named;7:7 would _v_modal;9:9 able _a_1;11:11 hold _v_1;12:12 together _p;13:13 the _q;14:14 sales _n_of;15:15 force _n_1;17:17 one card;18:18 reason _n_for;20:20 talks _n_of-on;21:21 with _p;22:22 Integrated named;0:0 Whitehall named', '0:0 In _p_state;1:1 composite _n_1;2:2 trading _v_1;3:3 on _p;4:4 the _q;5:6 named New York;7:8 _n_1 Stock Exchange;9:9 yesterday _a_1;10:10 Integrated named;11:11 closed _v_1;12:12 at _p;13:13 $ _n_1;14:14 1.25 card;15:15 a _p_per;16:16 share _n_of;17:17 down _p;18:18 25 card;0:0 In _p_state', '0:0 Integrated named;3:5 _v_1 struggling avoid;6:6 a _q;7:8 _n_1 bankruptcy- law;9:9 filing _n_1;10:10 since _p;11:11 June mofy;14:16 _v_1 failed make;17:17 interest _n_in;18:18 payments _n_of;19:19 on _p;20:20 nearly _x_deg;21:21 $ _n_1;22:23 card 1 billion;24:24 of _p;0:0 Integrated named', '0:0 Integrated named;1:1 senior _a_1;2:2 and _c;3:3 junior _a_1;4:4 creditors _n_1;6:6 owed _v_1;7:7 a _q;8:8 total _n_of;10:10 about _x_deg;11:11 $ _n_1;12:13 card 1.8 billion', '0:0 AN _q;1:1 EARTHQUAKE _n_1;2:2 STRUCK _v_1;3:4 named Northern California;5:5 killing _v_1;8:8 50 card;0:0 AN _q', '0:0 The _q;1:1 violent _a_1;2:2 temblor _n_1;4:4 lasted _v_1;5:5 about _x_deg;6:6 15 card;7:7 seconds _n_1;8:8 and _c;9:9 registered _v_1;10:10 6.9 card;11:11 on _p;12:12 the _q;13:13 Richter named;14:14 scale _n_1;15:15 also _a_1;16:16 caused _v_1;17:17 the _q;18:18 collapse _n_1;19:19 of _p;20:20 a _q;22:22 foot _n_1;22:22 foot card;23:23 section _n_of;25:25 the _q;26:27 named San Francisco-;28:29 named Oakland Bay;30:30 Bridge named;31:31 and _c;32:32 shook _v_1;33:34 named Candlestick Park', '0:0 The _q;1:1 tremor _n_1;3:3 centered _v_1;4:4 near _p;5:5 Hollister named;6:6 southeast _a_1;8:9 named San Francisco;10:10 and _c;12:12 felt _v_1;14:14 far _a_1;16:16 200 card;17:17 miles _n_1;0:0 The _q', '0:0 Numerous _a_1;1:1 injuries _n_1;0:0 Numerous _a_1', '0:0 Some _q;1:1 buildings _n_1;2:2 collapsed _v_1;3:3 gas _n_1;4:4 and _c;5:5 water _n_1;6:6 lines _n_of;7:7 ruptured _v_unknown;8:8 and _c;9:9 fires _n_1;0:0 Some _q', '0:0 The _q;1:1 quake _n_1;3:3 also _a_1;4:4 caused _v_1;5:5 damage _n_to;6:6 in _p;7:8 named San Jose;9:9 and _c;10:10 Berkeley named;11:11 knocked _v_out;13:13 electricity _n_1;14:14 and _c;15:15 telephones _n_1;16:16 cracked _v_cause;17:17 roadways _n_unknown;18:18 and _c;19:19 disrupted _v_1;20:21 _n_1 subway service;22:22 in _p;23:23 the _q;24:25 named Bay Area', '0:0 Major _a_1;1:1 injuries _n_1;3:3 reported _v_to;4:4 at _p;5:6 named Candlestick Park;8:8 the _q;9:9 third ord;10:10 game _n_1;11:11 of _p;12:12 baseball _n_1;14:15 named World Series;17:17 canceled _v_1;18:18 and _c;19:19 fans _n_1;20:20 evacuated _v_1;21:21 from _p;22:22 the _q;0:0 Major _a_1', '0:0 Bush named;1:1 vowed _v_to;3:3 veto _v_1;4:4 a _q;5:5 bill _n_of;6:6 allowing _v_1;7:7 federal _a_1;8:8 financing _v_1;9:9 for _p;10:10 abortions _n_of;11:11 in _p;12:12 cases _n_of;14:14 rape _n_1;15:15 and _c;16:16 incest _n_unknown;17:17 saying _v_to;18:19 _n_1 tax dollars;20:20 shouldn\'t _v_modal;22:24 _v_1 used "compound;25:25 a _q;26:26 violent _a_1;27:27 act _n_of;28:28 with _p;29:29 the _q;30:30 taking _v_1;32:32 an _q;33:33 unborn _a_1;0:0 Bush named', '1:1 pledge _n_1;2:2 in _p;3:3 a _q;4:4 letter _n_of;5:5 to _p;6:6 Democratic _a_1;7:7 Sen _n_1;8:8 Byrd named;9:9 came _v_1;10:11 ahead of _p;12:12 an _q;13:13 expected _v_1;14:14 Senate named_n;15:15 vote _n_1;16:16 on _p;17:17 spending _v_1;18:18 legislation _n_1;19:19 containing _v_1;20:20 the _q;1:1 pledge _n_1', '0:1 named East Germany;3:3 Politburo named;4:4 met _v_1;5:5 amid _p;6:6 speculation _n_about;8:8 the _q;9:9 ruling _v_1;10:10 body _n_1;11:11 would _v_modal;12:12 oust _v_1;14:14 line _n_of;14:14 line _hard_a_for;15:15 leader _n_of;16:16 Honecker named;18:18 rule _n_of;21:21 challenged _v_to;23:23 mass _n_of;24:24 emigration _n_unknown;25:25 and _c;26:26 calls _n_1;27:27 for _p;28:28 democratic _a_1;0:0 East named', '0:0 Meanwhile _p;1:1 about _x_deg;2:2 125 card;3:3 refugees _n_1;4:4 flew _v_1;5:5 to _p;6:7 named Duesseldorf West;8:8 Germany named;9:9 from _p;10:10 Warsaw named;11:11 the _q;12:12 first ord;13:13 airlift _n_unknown;14:14 in _p;15:16 named East Germany;18:19 _n_1 refugee exodus', '0:0 The _q;1:2 named World Psychiatric;3:3 Association named;4:4 voted _v_1;5:5 at _p;6:6 an _q;7:7 Athens named;8:8 parley _n_1;10:10 conditionally _a_unknown;11:11 readmit _v_to;11:11 readmit _re-_a_again;12:12 the _q;13:14 named Soviet Union', '0:0 Moscow named;2:2 left _v_1;3:3 the _q;4:4 group _n_of;5:5 in _p_temp;6:6 1983 yofc;7:7 to _x;8:8 avoid _v_1;9:9 explusion _n_unknown;10:10 over _p;11:11 allegations _n_1;13:13 political _a_1;14:14 dissidents _n_unknown;17:17 certified _v_1;19:19 insane _a_1;20:20 could _v_modal;22:22 suspended _v_1;23:23 if _x_then;24:24 the _q;25:25 misuse _n_of;27:27 psychiatry _n_1;28:28 against _p;29:29 dissenters _n_unknown;31:31 discovered _v_1;32:32 during _p;33:33 a _q;34:34 review _n_of;35:35 within _p;36:36 a card;0:0 Moscow named', '0:0 NASA named;1:1 postponed _v_1;2:2 the _q;3:3 liftoff _n_unknown;4:4 of _p;5:5 the _q;6:6 space _n_of;7:7 shuttle _n_1;8:8 Atlantis named;9:10 because of _p;11:11 rain _n_1;12:12 near _p;13:13 the _q;14:14 site _n_1;15:15 of _p;16:16 the _q;17:18 _n_1 launch pad;19:19 in _p;20:21 named Cape Canaveral;0:0 NASA named', '0:0 The _q;1:1 flight _n_1;3:3 rescheduled _v_1;3:3 rescheduled _re-_a_again;4:4 for _p;0:0 The _q', '0:0 The _q;1:1 spacecraft _n_1;3:3 five card;4:4 astronauts _n_1;7:7 dispatch _v_1;8:8 the _q;10:10 powered _n_1;10:10 powered _nuclear_a_1;11:11 Galileo named;12:12 space _n_of;13:13 probe _n_1;14:14 on _p;15:15 an _q;16:16 exploratory _a_unknown;17:17 mission _n_1;18:18 to _p;0:0 The _q', '0:0 Senate named_n;1:1 Democratic _a_1;2:2 leaders _n_of;3:3 said _v_to;5:5 had _v_1;6:6 enough _q;7:7 votes _n_1;8:8 to _x;9:9 defeat _v_1;10:10 a _q;11:11 proposed _v_to;12:12 constitutional _a_1;13:13 amendment _n_of;15:15 ban _v_from;16:16 flag _n_1;0:0 Senate named_n', '0:0 The _q;1:1 amendment _n_of;3:3 aimed _v_at;5:5 skirting _v_1;6:6 a _q;7:8 named Supreme Court;9:9 ruling _n_1;11:11 threw _v_out;13:13 the _q;14:14 conviction _n_of;16:16 a _q;17:17 Texas named;18:19 _n_1 flag- burner;20:20 on _p;21:24 _n_1 grounds freedom;25:25 of _p;26:26 speech _n_1;0:0 The _q', '0:0 Federal _a_1;1:1 researchers _n_of;2:2 said _v_to;3:4 _n_1 lung- cancer;5:5 mortality _n_1;6:6 rates _n_of;7:7 for _p;8:8 people _n_of;9:9 under _p;10:10 45 card;11:11 years _n_1;12:12 of _p;13:13 age _n_1;15:17 _v_1 begun decline;18:18 particularly _x_deg;19:19 for _p;20:20 white _a_1;0:0 Federal _a_1', '0:0 The _q;1:2 named National Cancer;3:3 Institute named;4:4 also _a_1;5:5 projected _v_1;7:7 overall _a_1;8:8 U.S named_n;9:9 mortality _n_1;10:10 rates _n_of;11:11 from _p;12:13 _n_1 lung cancer;14:14 should _v_modal;15:17 _v_1 begin drop;18:18 in _p;19:19 several _a_1;20:20 years _n_1;21:21 if _x_then;22:22 cigarette _n_1;23:23 smoking _v_1;24:24 continues _v_2;0:0 The _q', '0:0 Bush named;1:1 met _v_1;2:2 with _p;3:3 South named;4:4 Korean _a_1;5:5 President _n_of;6:6 Roh named;8:8 indicated _v_1;10:10 Seoul named;11:11 plans _v_1;13:13 further _a_1;14:14 ease _v_1;15:16 _n_of trade rules;17:17 to _x;18:18 ensure _v_1;21:21 economy _n_1;22:22 becomes _v_id;24:24 open _a_1;26:26 the _q;27:28 _a_1 other industrialized;29:29 nations _n_of;30:30 by _p;31:31 the _q;33:33 1990s year_range;0:0 Bush named', '0:0 Bush named;1:1 assured _v_of;2:2 Roh named;4:4 the _q;5:5 U.S named_n;6:6 would _v_modal;7:7 stand _v_1;8:8 by _p;10:10 security _n_1;11:11 commitments _n_to;13:13 long _a_1;16:16 is _v_there;17:17 a _q;18:18 threat" _n_1;19:19 from _p;20:21 named Communist North;0:0 Bush named', '0:0 The _q;1:1 Bush named;2:2 administration _n_of;4:4 seeking _v_1;5:5 an _q;6:6 understanding _n_of;7:7 with _p;8:8 Congress named;9:9 to _x;10:10 ease _v_1;11:11 restrictions _n_of;12:12 on _p;13:13 U.S named_n;14:14 involvement _n_in;16:16 foreign _a_1;17:17 coups _n_1;19:19 might _v_modal;20:20 result _v_in;22:22 the _q;23:23 death _n_1;24:24 of _p;25:25 a _q;26:28 _n_of country leader', '0:0 A _q;1:1 White named;2:2 House named_n;3:3 spokesman _n_1;4:4 said _v_to;6:6 while _x;7:7 Bush named;8:8 wouldn\'t _v_modal;9:9 alter _v_1;10:10 a _q;11:11 longstanding _a_unknown;12:12 ban _n_1;13:13 on _p;14:14 such _q;15:15 involvement _n_in;17:17 \'s _v_there;18:18 a _q;19:19 clarification _n_of;20:20 needed" _v_1;21:21 on _p;0:0 A _q', '0:2 named India Gandhi;3:3 called _v_for;5:5 parliamentary _a_1;6:6 elections _n_of-for;7:7 next _a_1;0:0 India named', '0:0 The _q;1:2 _v_1 balloting considered;3:3 a _q;4:4 test _n_of;5:5 for _p;6:6 the _q;7:8 prime minister _n_of;9:9 and _c;10:10 the _q;11:11 ruling _v_1;12:13 named Congress (I);14:14 Party _n_of-to;15:15 comes _v_1;16:16 amid _p;17:17 charges _n_of;19:19 inept _a_1;20:20 leadership _n_1;21:21 and _c;22:22 government _n_of;0:0 The _q', '0:0 Gandhi named;2:2 family _n_of;4:4 ruled _v_1;5:5 independent _a_of;6:6 India named;7:7 for _p;8:8 all _q;9:9 but _p_except;10:10 five card;11:11 years _n_1;12:12 of _p;15:15 year _n_1;15:15 year card;0:0 Gandhi named', '0:0 The _q;1:2 named Soviet Union;3:3 abstained _v_from;5:5 a _q;6:6 U.N named_n;7:8 named General Assembly;9:9 vote _n_1;11:11 reject _v_as;12:12 Israel named;0:0 The _q', '1:1 was _v_id;2:2 the _q;3:3 first ord;4:4 time _n_of;5:5 in _p;6:6 seven card;7:7 years _n_1;9:9 Moscow named;11:11 joined _v_1;12:12 efforts _n_1;13:13 led _v_1;15:15 Moslem named;16:16 nations _n_of;18:18 expel _v_1;19:19 Israel named;20:20 from _p;21:21 the _q;22:22 world _n_of;23:23 body _n_1;24:24 and _c;26:26 viewed _v_as;28:28 a _q;29:29 sign _n_of;31:31 improving _v_1;32:33 named Soviet- Israeli;1:1 was _v_id', '0:0 Israel named;2:2 seated _v_cause;3:3 by _p;4:4 a _q;5:5 vote _n_1;6:6 of _p;7:9 card 95 37;10:10 with _p;11:11 15 card;0:0 Israel named', "0:0 Black _a_1;1:1 activist _n_1;2:3 named Walter Sisulu;4:4 said _v_to;5:5 the _q;6:7 named African National;8:8 Congress named;9:9 wouldn't _v_modal;10:10 reject _v_as;11:11 violence _n_1;12:12 as _p;13:13 a _q;14:14 way _n_of;16:16 pressure _v_1;17:17 the _q;18:18 South named;19:19 African _a_1;20:20 government _n_of;21:21 into _p;22:22 concessions _n_1;24:24 might _v_modal;25:25 lead _v_to;27:27 negotiations _n_of;28:28 over _p;0:0 Black _a_1", '0:0 The _q;2:2 year- _n_1;2:2 year- card;3:3 old _a_1;4:4 Sisulu named;6:6 among _p;7:7 eight card;8:9 _a_1 black political;10:10 activists _n_1;11:11 freed _v_from-of;12:12 Sunday dofw;0:0 The _q', '0:0 London named;2:2 concluded _v_1;4:4 Austrian _a_1;5:5 President _n_of;6:6 Waldheim named;8:8 responsible _a_for;10:10 the _q;11:11 execution _n_1;12:12 of _p;13:13 six card;14:14 British _a_1;15:15 commandos _n_unknown;16:16 in _p;17:18 named World War;19:19 II named;20:20 although _x;22:22 probably _a_1;24:24 aware _a_of;26:26 the _q;0:0 London named', '0:0 The _q;1:1 report _n_of;2:2 by _p;3:3 the _q;4:5 named Defense Ministry;6:6 also _a_1;7:7 rejected _v_as;8:8 allegations _n_1;10:10 Britain named;11:11 covered _v_up;13:13 evidence _n_of;15:15 Waldheim named;17:17 activities _n_1;18:18 as _p;19:19 a _q;20:20 German _a_1;21:21 army _n_of;0:0 The _q', '0:0 An _q;1:1 international _a_1;2:2 group _n_of;3:3 approved _v_1;4:4 a _q;5:5 formal _a_1;6:6 ban _n_1;7:7 on _p;8:8 ivory _n_1;9:9 trade _n_of;10:10 despite _p;11:11 objections _n_to;12:12 from _p;13:13 southern _a_1;14:14 African named;15:15 governments _n_of;17:19 _v_1 threatened find;20:20 alternative _a_1;21:21 channels _n_1;22:22 for _p;23:23 selling _v_1;24:25 _n_1 elephant tusks', '0:0 The _q;1:1 move _n_1;2:2 by _p;3:3 the _q;4:4 Convention named;5:5 on _p;6:6 Trade _n_of;7:7 in _p;8:9 named Endangered Species;10:10 meeting _v_1;11:11 in _p;12:12 Switzerland named;13:13 places _v_1;14:14 the _q;15:15 elephant _n_1;16:16 on _p;17:17 the _q;19:19 species _n_of-n;19:19 species _endanger_v_1;0:0 The _q', '0:0 An _q;1:1 assassin _n_1;2:2 in _p;3:3 Colombia named;4:4 killed _v_1;5:5 a _q;6:6 federal _a_1;7:7 judge _n_of;8:8 on _p;9:9 a _q;10:10 Medellin named;0:0 An _q', '0:0 An _q;1:1 anonymous _a_1;2:2 caller _n_of;3:3 to _p;4:4 a _q;5:5 local _a_1;6:7 _n_1 radio station;8:8 said _v_to;9:9 cocaine _n_1;10:10 traffickers _n_unknown;12:12 slain _v_unknown;13:13 the _q;14:14 magistrate _n_1;15:15 in _p;16:16 retaliation _n_unknown;17:17 for _p;18:18 the _q;19:19 extraditions _n_unknown;20:20 of _p;21:21 Colombians _n_1;22:22 wanted _a_1;23:23 on _p;24:24 drug _n_1;25:25 charges _n_of;26:26 in _p;27:27 the _q;0:0 An _q', '0:0 Libyan _a_unknown;1:1 leader _n_of;2:2 Gadhafi named;3:3 met _v_1;4:4 with _p;5:5 Egypt named;7:7 President _n_of;8:8 Mubarak named;9:9 and _c;10:10 the _q;11:11 two card;12:12 officials _n_1;13:15 _v_1 pledged respect;16:16 each _q;17:19 _n_1 other laws;20:20 security _n_1;21:21 and _c;0:0 Libyan _a_unknown', '1:1 stopped _v_1;2:2 short _a_of;4:4 resuming _v_1;5:5 diplomatic _a_1;6:6 ties _n_1;7:7 severed _v_1;8:8 in _p_temp;1:1 stopped _v_1', '0:0 The _q;1:1 reconciliation _n_unknown;2:2 talks _n_of-on;3:3 in _p;4:4 the _q;5:5 Libyan _a_unknown;6:7 _n_1 desert town;8:8 of _p;9:9 Tobruk named;10:10 followed _v_1;11:11 a _q;12:12 meeting _n_of;13:13 Monday dofw;14:14 in _p;15:15 the _q;16:16 Egyptian _a_1;17:17 resort _n_1;18:18 of _p;19:20 named Mersa Metruh', '0:0 The _q;1:1 new _a_1;2:2 notes _n_1;4:4 bear _v_1;5:5 interest _n_in;6:6 at _p;7:7 5.5 card;8:8 % _n_of;9:9 through _p;10:10 July mofy;11:11 31 dofm;12:12 1991 yofc;13:13 and _c;14:15 _p thereafter at;16:16 10 card;0:0 The _q', '0:0 Under _p_state;1:1 the _q;2:2 original _a_1;3:3 proposal _n_1;4:4 the _q;5:5 maker _n_of;7:7 specialty _n_1;8:8 coatings _n_of;9:9 and _c;10:10 a _q;11:11 developer _n_of;13:13 information- _n_on-about;14:14 display _n_of;15:15 technologies _n_1;16:16 offered _v_1;17:17 $ _n_1;18:18 400 card;19:19 of _p;20:20 notes _n_1;21:21 due _a_1;22:23 card 1996 10;24:24 common _a_for;25:25 shares _n_of;26:26 and _c;27:27 $ _n_1;28:28 175 card;29:29 in _p;30:30 cash _n_1;31:31 for _p;32:32 each _q;33:33 $ _n_1;34:34 1,000 card;35:35 face _n_1;0:0 Under _p_state', '0:0 Completion _n_of;2:2 the _q;3:4 _n_1 exchange offer;6:6 subject _a_to;8:8 the _q;9:9 tender _n_1;10:10 of _p;11:12 at least _x_deg;13:13 80 card;14:14 % _n_of;15:15 of _p;16:16 the _q;17:17 debt _n_1;18:18 among _p;19:19 other _a_1;0:0 Completion _n_of', '0:0 The _q;1:1 stock _n_1;2:2 of _p;3:3 UAL named;4:4 Corp _n_1;5:5 continued _v_2;8:8 pounded _v_1;9:9 amid _p;10:10 signs _n_of;12:13 named British Airways;14:14 may _v_modal;15:15 balk _v_at;17:17 any _q;18:18 hasty _a_1;19:19 reformulation _n_unknown;20:20 of _p;21:21 the _q;22:22 aborted _v_1;23:23 $ _n_1;24:25 card 6.79 billion;26:27 buy- out _n_1;28:28 of _p;29:30 named United Airlines;0:0 The _q', '0:0 The _q;1:1 plunge _n_of;2:2 followed _v_1;3:3 a _q;4:4 drop _n_of;6:6 $ _n_1;7:7 56.875 card;8:8 Monday dofw;9:9 amid _p;10:10 indications _n_of;11:11 the _q;12:12 takeover _n_of;13:13 may _v_modal;14:14 take _v_2;15:15 weeks _n_1;0:0 The _q', '0:0 The _q;1:1 stock _n_1;3:3 fallen _v_1;4:4 $ _n_1;5:5 87.25 card;6:6 or _c;7:7 31 card;8:8 % _n_of;9:9 in _p;10:10 the _q;11:11 three card;12:12 trading _v_1;13:13 days _n_of;14:14 since _x_subord;15:15 announcement _n_of;17:17 the _q;18:18 collapse _n_1;19:19 of _p;20:20 the _q;21:21 $ _n_1;22:22 300- card;24:24 share _n_of;24:24 share _a_p_per;25:25 takeover _n_of;26:26 jolted _v_1;27:27 the _q;28:28 entire _a_1;29:30 _n_1 stock market;31:31 into _p;34:34 worst _a_1;34:34 worst ord;0:0 The _q', '0:0 "This _q_dem;1:1 is _v_id;2:2 a _q;3:3 total _a_1;4:4 bloodbath" _n_unknown;5:5 for _p;6:6 takeover- _n_of;7:8 _n_1 stock traders;9:9 one card;10:11 _n_1 investment banker;0:0 "This _q_dem', '0:1 named Los Angeles;2:2 financier _n_unknown;3:4 named Marvin Davis;6:6 put _v_1;7:7 United named;8:8 in _p;9:9 play _n_1;10:10 with _p;11:11 a _q;12:12 $ _n_1;13:14 card 5.4 billion;15:15 bid _n_1;16:16 two card;17:17 months _n_1;18:18 ago _p;19:19 last _a_1;20:20 night _n_of;21:21 proffered _v_unknown;23:23 a _q;24:24 ray _n_1;25:25 of _p;26:26 hope _n_1;27:27 and _c;28:28 an _q;29:29 extra _a_1;30:30 element _n_1;31:31 of _p;32:32 uncertainty _n_1;33:33 by _p_means;34:34 saying _v_to;36:36 remains _v_1;37:37 interested _a_in;39:39 acquiring _v_1;0:0 Los named', '0:0 But _c;2:2 dropped _v_cause;4:4 earlier _a_1;5:5 $ _n_1;6:6 300- card;8:8 share _n_of;8:8 share _a_p_per;9:10 back- up _backup_n_1;11:11 bid _n_1;12:12 saying _v_to;14:14 must _v_modal;15:15 first _a_1;16:16 explore _v_1;17:17 bank _n_of;0:0 But _c', '0:0 Even _x_deg;1:1 as _x_subord;2:2 Citicorp named;3:3 and _c;4:5 named Chase Manhattan;6:6 Corp _n_1;7:7 scrambled _v_1;8:8 to _x;9:9 line _v_up;11:11 bank _n_of;12:12 financing _v_1;13:13 for _p;14:14 a _q;15:15 revised _v_1;16:16 version _n_of;18:18 the _q;19:19 lapsed _v_1;20:21 _n_1 labor- management;22:22 bid _n_1;23:24 named British Airways;25:25 a _q;26:26 15 card;27:27 % _n_of;28:28 partner _n_1;29:29 in _p;30:30 the _q;31:31 buying _v_1;32:32 group _n_of;33:35 _v_1 indicated wants;37:37 start _v_1;38:38 from _p;0:0 Even _x_deg', '1:1 partners _n_1;2:2 are _v_id;3:3 United named;5:5 pilots _n_of;9:9 own _v_1;10:10 75 card;11:11 % _n_of;12:12 and _c;13:13 UAL named;14:14 management _n_1;15:15 at _p;16:16 10 card;1:1 partners _n_1', '0:0 A _q;1:1 similar _a_to;2:2 demand _n_1;4:4 made _v_1;6:6 a _q;7:7 group _n_of;9:9 represents _v_1;10:10 some _q;12:12 United named;14:14 26,000 card;15:15 noncontract _a_unknown;0:0 A _q', '0:1 named John Peterpaul;2:2 Machinists named;3:3 union _n_of;4:4 general _a_1;5:5 vice _n_1;6:6 president _n_of;7:7 attacked _v_1;8:8 Mr _n_1;9:9 Wolf named;11:11 "greedy _a_1;12:12 and _c;13:13 irresponsible" _a_unknown;14:14 for _p;15:15 pursuing _v_1;16:16 the _q;0:0 John named', '0:0 Although _x;1:1 Mr _n_1;2:2 Wolf named;3:3 and _c;4:5 named John Pope;6:6 UAL named;8:8 chief _n_1;9:9 financial _a_1;10:10 officer _n_1;11:13 _v_1 stood pocket;14:14 $ _n_1;15:16 card 114.3 million;17:17 for _p;18:18 stock _n_1;19:19 and _c;20:20 options _n_1;21:21 in _p;22:22 the _q;23:24 buy- out _n_1;25:25 UAL named;26:26 executives _n_1;27:27 planned _v_1;29:29 reinvest _v_in;29:29 reinvest _re-_a_again;30:30 only _x_deg;31:31 $ _n_1;32:33 card 15 million;35:35 the _q;36:36 new _a_1;0:0 Although _x', '0:0 The _q;2:2 collar _n_1;2:2 collar _blue_a_1;3:3 machinists _n_1;4:4 longtime _a_unknown;5:5 rivals _n_1;6:6 of _p;7:7 the _q;9:9 collar _n_1;9:9 collar _white_a_1;10:10 pilots _n_of;11:11 say _v_to;12:12 the _q;13:13 buyout _n_unknown;14:14 would _v_modal;15:15 load _v_1;16:16 the _q;17:17 company _n_of;18:18 with _p;19:19 debt _n_1;20:20 and _c;21:21 weaken _v_cause;0:0 The _q', '0:0 Confusion _n_1;1:1 about _p;2:2 the _q;3:3 two card;4:4 banks _n_of;6:6 hurried _v_1;7:7 efforts _n_1;9:9 round _v_up;11:11 financing _v_1;12:12 for _p;13:13 a _q;14:14 new _a_1;15:15 bid _n_1;17:17 the _q;18:18 UAL named;19:19 board _n_of;21:21 even _a_1;22:22 seen _v_1;23:23 yet _a_1;24:24 helped _v_to;25:25 send _v_1;26:26 UAL named;27:27 stock _n_1;28:28 spiraling _v_1;0:0 Confusion _n_1', '0:0 And _c;1:1 rumors _n_1;2:2 of _p;3:4 _v_1 forced selling;5:5 by _p;6:6 takeover- _n_of;7:8 _n_1 stock traders;9:9 triggered _v_1;10:10 a _q;12:12 point _n_of;12:12 point card;13:13 downdraft _n_unknown;14:14 in _p;15:15 the _q;16:17 named Dow Jones;18:19 named Industrial Average;20:20 around _p;21:21 11:15 numbered_hour;22:22 a.m _x;23:23 EDT timezone_p;0:0 And _c', '0:0 Yesterday _a_1;2:3 _v_1 selling began;4:4 after _x_h;5:5 a _q;6:6 Japanese _a_1;7:7 news _n_of;8:8 agency _n_1;9:9 reported _v_to;11:11 Japanese _a_1;12:12 banks _n_of;14:14 balked _v_at;16:16 the _q;17:17 first ord;18:18 bid _n_1;20:20 ready _a_1;22:22 reject _v_as;23:23 a _q;24:24 revised _v_1;25:25 version _n_of;26:26 at _p;27:27 around _x_deg;28:28 $ _n_1;29:29 250 card;30:30 a _p_per;31:31 share _n_of;32:32 or _c;33:33 $ _n_1;34:35 card 5.65 billion', '0:0 Citicorp named;3:3 said _v_to;4:4 only _a_1;7:7 had _v_1;8:8 "expressions _n_1;9:9 of _p;10:10 interest _n_in;11:11 of _p;12:12 a _q;13:13 transaction _n_1;14:14 from _p;16:16 the _q;17:17 borrowers _n_unknown;18:18 and _c;19:19 the _q;20:20 banks," _n_of;21:21 but _c;23:23 have _v_1;24:24 an _q;0:0 Citicorp named', '0:0 Late _p;1:1 in _p_temp;2:2 the _q;3:3 day _n_of;4:4 Mr _n_1;5:5 Wolf named;6:6 issued _v_1;7:7 a _q;8:8 onepage _a_unknown;9:9 statement _n_of;10:10 calling _v_1;11:11 Mr _n_1;12:12 Peterpaul named;14:14 blast _n_1;15:15 "divisive _a_unknown;16:16 and _c;17:17 uncalled _v_for;0:0 Late _p', '0:0 But _c;2:2 gave _v_1;4:4 details _n_1;5:5 on _p;6:6 the _q;7:7 progress _n_1;8:8 toward _p;9:9 a _q;10:10 new _a_1;11:11 bid _n_1;12:12 saying _v_to;13:13 only _a_1;16:16 working _v_1;17:17 toward _p;18:18 a _q;19:19 revised _v_1;20:20 proposal _n_1;21:21 for _p;22:23 _n_of majority employee;0:0 But _c', '0:1 _p Meanwhile in;2:2 another _q;3:3 sign _n_of;5:5 a _q;6:6 new _a_1;7:7 bid _n_1;9:9 imminent _a_1;12:12 learned _v_1;14:14 the _q;15:15 UAL named;16:16 board _n_of;17:17 held _v_1;18:18 a _q;19:19 telephone _n_1;20:20 meeting _n_of;21:21 Monday dofw;22:22 to _x;23:23 hear _v_1;24:24 an _q;25:25 update _n_of;26:26 on _p;27:27 the _q;28:28 situation _n_1;29:29 but _c;31:31 a _q;32:32 formal _a_1;33:34 _n_of board meeting;36:36 likely _a_1;39:39 convened _v_cause;40:40 until _p;41:42 _a_1 early next;0:0 Meanwhile _p', '0:0 The _q;1:1 effort _n_1;3:3 revive _v_cause;4:4 the _q;5:5 bid _n_1;7:7 complicated _v_1;9:9 the _q;10:10 unwieldy _a_for;11:11 nature _n_1;12:12 of _p;13:13 the _q;15:15 party _n_of-to;15:15 party card;16:16 buying _v_1;0:0 The _q', '0:0 The _q;1:1 pilots _n_of;3:3 meeting _v_1;4:4 outside _p;5:5 Chicago named;0:0 The _q', '0:0 But _c;1:2 named British Air;7:7 supplied _v_1;8:8 $ _n_1;9:10 card 750 million;11:12 out of _p;13:13 $ _n_1;14:15 card 965 million;16:16 in _p;17:17 equity _n_1;18:18 financing _v_1;19:19 apparently _a_1;21:21 involved _v_in;23:23 the _q;24:24 second ord;25:25 proposal _n_1;26:26 and _c;27:27 could _v_modal;28:28 well _a_1;29:29 reject _v_as;31:31 even _x_deg;32:32 if _x_then;33:33 banks _n_of;34:35 _v_1 obtain financing', '0:0 The _q;1:1 machinists _n_1;2:2 also _a_1;3:3 asked _v_for;5:5 an _q;6:6 investigation _n_1;7:7 by _p;8:8 the _q;9:9 Securities named;10:10 and _c;11:12 named Exchange Commission;13:13 into _p;14:14 possible _a_for;15:16 _n_1 securities- law;17:17 violations _n_of;18:18 in _p;19:19 the _q;20:20 original _a_1;21:21 bid _n_1;22:22 for _p;23:23 UAL named;24:24 by _p;25:25 Mr _n_1;26:26 Davis named;27:29 as well as _c;30:30 in _p;31:31 the _q;32:32 response _n_to;33:33 by _p;0:0 The _q', '0:0 Last _a_1;1:1 week _n_1;2:2 just _x_deg;3:3 before _x_h;4:4 the _q;5:5 bank _n_of;6:6 commitments _n_to;8:8 due _a_1;9:9 the _q;10:10 union _n_of;11:11 asked _v_1;12:12 the _q;13:13 U.S named_n;14:15 named Labor Department;17:17 study _v_1;19:19 the _q;20:20 bid _n_1;21:21 violated _v_1;22:22 legal _a_1;23:23 standards _n_1;24:24 of _p;25:25 fairness _n_1;26:26 governing _v_1;27:27 employee _n_of;28:29 _n_1 investment funds', '0:0 In _p_state;2:2 statement _n_of;3:3 Mr _n_1;4:4 Wolf named;5:5 said _v_to;7:7 continue _v_2;9:9 believe _v_1;11:11 approach _n_to;13:13 sound _a_1;14:14 and _c;18:18 far _x_deg;19:19 better _a_at-for-of;20:20 for _p;21:21 all _q;22:22 employees _n_of;24:24 the _q;25:25 alternative _n_to;26:26 of _p;27:27 having _v_cause;28:28 an _q;29:29 outsider _n_1;30:30 own _v_1;31:31 the _q;32:32 company _n_of;33:33 with _p;34:34 employees _n_of;35:35 paying _v_for;38:38 just _x_deg;0:0 In _p_state', '0:0 The _q;1:1 huge _a_1;2:2 drop _n_of;3:3 in _p;4:4 UAL named;5:5 stock _n_1;6:6 prompted _v_1;7:7 one card;8:8 takeover _n_of;9:10 _n_1 stock trader;11:12 named George Kellner;13:13 managing _v_1;14:14 partner _n_1;15:15 of _p;16:17 named Kellner DiLeo;18:19 & Co _n_1;21:21 deny _v_to;22:22 publicly _a_1;23:26 _n_1 rumors firm;28:28 going _v_1;29:30 out of _p_dir;0:0 The _q', '0:0 Mr _n_1;1:1 Kellner named;2:2 said _v_to;4:4 despite _p;5:5 losses _n_of;6:6 on _p;7:7 UAL named;8:10 _n_1 stock firm;12:12 health _n_1;0:0 Mr _n_1', '0:0 The _q;1:3 _n_1 stock decline;4:4 also _a_1;6:6 left _v_1;7:7 the _q;8:8 UAL named;9:9 board _n_of;10:10 in _p;11:11 a _q;0:0 The _q', "0:0 Although _x;2:2 may _v_modal;5:5 legally _a_1;6:8 _v_1 obligated sell;9:9 the _q;10:10 company _n_of;11:11 if _x_then;12:12 the _q;13:14 buy- out _n_1;15:15 group _n_of;16:16 can't _v_modal;17:17 revive _v_cause;19:19 bid _n_1;21:21 may _v_modal;22:22 have _v_qmodal;24:24 explore _v_1;25:25 alternatives _n_to;26:26 if _x_then;27:27 the _q;28:28 buyers _n_of;29:29 come _v_back;31:31 with _p;32:32 a _q;33:33 bid _n_1;34:34 much _x_deg;35:35 lower _a_1;37:37 the _q;38:38 group _n_of;40:40 original _a_1;41:41 $ _n_1;42:42 300- card;44:44 share _n_of;44:44 share _a_p_per;0:0 Although _x", '0:0 The _q;1:1 takeover- _n_of;2:3 _n_1 stock traders;5:5 hoping _v_1;7:7 Mr _n_1;8:8 Davis named;9:9 or _c;10:10 one card;12:12 the _q;13:13 other _a_1;14:14 interested _a_in;15:15 parties _n_of-to;16:16 might _v_modal;17:17 re-emerge _v_1;17:17 re-emerge _re-_a_again;18:18 with _p;19:19 the _q;20:20 situation _n_1;21:21 in _p;22:22 disarray _n_unknown;23:23 or _c;25:25 the _q;26:26 board _n_of;27:27 might _v_modal;28:28 consider _v_1;29:29 a _q;0:0 The _q', '0:0 Meanwhile _p;1:1 Japanese _a_1;2:2 bankers _n_1;3:3 said _v_to;6:7 _a_1 still hesitant;8:8 about _p;9:9 accepting _v_1;10:10 Citicorp named;12:12 latest _a_for;0:0 Meanwhile _p', '0:0 Macmillan named;1:1 Inc _n_1;2:2 said _v_to;4:4 plans _v_1;5:5 a _q;6:6 public _a_1;7:7 offering _n_of;9:10 card 8.4 million;11:11 shares _n_of;12:12 of _p;14:15 named Berlitz International;16:16 Inc _n_1;17:17 unit _n_of;18:18 at _p;19:19 $ _n_1;20:20 19 card;21:21 to _p;22:22 $ _n_1;23:23 21 card;24:24 a _p_per;0:0 Macmillan named', '0:0 The _q;1:1 offering _n_of;2:2 for _p;3:3 the _q;4:5 _n_1 language school;6:6 unit _n_of;8:8 announced _v_to;10:11 named Robert Maxwell;12:12 chairman _n_of;13:13 and _c;14:15 _n_1 chief executive;16:16 officer _n_1;17:17 of _p;18:18 London- named;19:19 based _v_1;20:21 named Maxwell Communication;22:22 Corp _n_1;24:24 owns _v_1;0:0 The _q', '0:0 After _x_h;1:1 the _q;2:2 offering _n_of;4:4 completed _v_2;5:5 Macmillan named;7:7 own _v_1;8:8 about _x_deg;9:9 56 card;10:10 % _n_of;11:11 of _p;12:12 the _q;13:13 Berlitz named;14:14 common _a_for;15:15 stock _n_1;0:0 After _x_h', '0:1 card Five million;2:2 shares _n_of;5:5 offered _v_1;6:6 in _p;7:7 the _q;8:8 U.S named_n;9:9 and _c;10:11 card 3.4 million;12:12 additional _a_1;13:13 shares _n_of;16:16 offered _v_1;17:17 in _p;18:19 _a_1 concurrent international;20:20 offerings _n_of;21:21 outside _p;22:22 the _q;0:0 Five card', '0:1 named Goldman Sachs;2:3 & Co _n_1;5:5 manage _v_1;6:6 the _q;0:0 Goldman named', '0:0 Macmillan named;1:1 said _v_to;2:2 Berlitz named;3:5 _v_for intends pay;6:6 quarterly _a_1;7:7 dividends _n_1;8:8 on _p;9:9 the _q;0:0 Macmillan named', '0:0 The _q;1:1 company _n_of;2:2 said _v_to;4:4 expects _v_1;6:6 pay _v_for;7:7 the _q;8:8 first ord;9:9 dividend _n_1;10:10 of _p;11:11 12.5 card;12:12 cents _n_1;13:13 a _p_per;14:14 share _n_of;15:15 in _p_temp;16:16 the _q;17:17 1990 yofc;18:18 first ord;0:0 The _q', '0:0 Berlitz named;2:2 borrow _v_from;3:3 an _q;4:4 amount _n_of;5:5 equal _a_to;8:8 expected _v_1;9:10 _n_1 net proceeds;11:11 from _p;12:12 the _q;13:13 offerings _n_of;14:14 plus _c;15:15 $ _n_1;16:17 card 50 million;18:18 in _p;19:19 connection _n_of;20:20 with _p;21:21 a _q;22:23 _n_1 credit agreement;24:24 with _p;0:0 Berlitz named', '0:0 The _q;1:1 total _a_1;2:2 borrowing _v_from;4:4 be _v_id;5:5 about _x_deg;6:6 $ _n_1;7:8 card 208 million;9:9 the _q;10:10 company _n_of;0:0 The _q', '0:0 Proceeds _n_1;1:1 from _p;2:2 the _q;3:3 borrowings _n_1;4:4 under _p;5:5 the _q;6:7 _n_1 credit agreement;10:10 used _v_1;12:12 pay _v_for;13:13 an _q;14:14 $ _n_1;15:16 card 80 million;17:18 _n_1 cash dividend;20:20 Macmillan named;21:21 and _c;23:23 lend _v_to;24:24 the _q;25:25 remainder _n_of;27:27 about _x_deg;28:28 $ _n_1;29:30 card 128 million;32:33 named Maxwell Communications;34:34 in _p;35:35 connection _n_of;36:36 with _p;37:37 a _q;38:38 promissory _a_1;0:0 Proceeds _n_1', '0:0 Proceeds _n_1;1:1 from _p;2:2 the _q;3:3 offering _n_of;6:8 _v_1 used repay;9:9 borrowings _n_1;10:10 under _p;11:11 the _q;13:13 term _n_of;13:13 term _short_a_of;14:14 parts _n_1;15:15 of _p;16:16 a _q;17:18 _n_1 credit agreement', '0:0 Berlitz named;3:3 based _v_1;4:4 in _p;5:6 named Princeton N.J;7:7 provides _v_1;8:8 language _n_1;9:9 instruction _n_of;10:10 and _c;11:12 _n_1 translation services;13:13 through _p;16:16 260 card;17:17 language _n_1;18:18 centers _n_of;19:19 in _p;20:20 25 card;0:0 Berlitz named', '0:0 In _p_state;1:1 the _q;2:2 past _a_1;3:3 five card;4:4 years _n_1;7:7 68 card;8:8 % _n_of;9:9 of _p;11:11 sales _n_of;14:14 outside _p;15:15 the _q;0:0 In _p_state', '0:0 Macmillan named;2:2 owned _v_1;3:3 Berlitz named;4:4 since _p;0:0 Macmillan named', '0:0 In _p_state;1:1 the _q;2:2 first ord;3:3 six card;4:4 months _n_1;5:5 of _p;6:6 this _q_dem;7:7 year _n_1;8:8 Berlitz named;9:9 posted _v_1;10:11 _n_1 net income;12:12 of _p;13:13 $ _n_1;14:15 card 7.6 million;16:16 on _p;17:17 sales _n_of;19:19 $ _n_1;20:21 card 106.2 million;22:22 compared _v_with;24:25 _n_1 net income;26:26 of _p;27:27 $ _n_1;28:29 card 8.2 million;30:30 on _p;31:31 sales _n_of;33:33 $ _n_1;34:35 card 90.6 million', '0:1 Right away _p;3:3 notice _v_1;4:4 the _q;5:5 following _a_1;6:6 things _n_of-about;7:7 about _p;8:8 a _q;9:10 named Philip Glass;0:1 Right away _p', '1:1 attracts _v_1;2:2 people _n_of;3:3 with _p;4:4 funny _a_1;5:5 hair _n_1;6:6 (or _c;7:7 with _p;8:8 no _q;9:9 hair _n_1;11:13 in front of _p_state;15:15 a _q;16:16 girl _n_1;17:17 with _p;18:18 spiked _v_1;19:19 locks _n_1;20:20 sat _v_1;21:21 beside _p;22:22 a _q;23:23 boy _n_1;1:1 attracts _v_1', '0:0 People _n_of;1:1 in _p;2:2 Glass named;3:3 houses _n_of;4:6 _v_1 tend look;0:0 People _n_of', '0:0 And _c;1:1 if _x_then;2:2 still _a_1;3:3 conscious _a_of;4:4 at _p;5:5 the _q;6:8 _n_of evening end;10:10 notice _v_1;11:11 something _q;12:12 else _a_1;13:13 : _p_namely;14:14 The _q;15:15 audience _n_1;16:17 at first _a_1;18:18 entranced _v_1;19:19 and _c;20:20 hypnotized _v_unknown;21:21 by _p;22:22 the _q;23:23 music _n_1;24:24 releases _v_1;26:27 pent- up _a_1;28:28 feelings _n_of;29:29 in _p;30:30 collective _a_1;0:0 And _c', '1:1 sits _v_down;3:3 at _p;4:4 the _q;5:5 piano _n_1;6:6 and _c;1:1 sits _v_down', '0:0 And _c;0:0 And _c', '1:1 one _n_1;2:2 likes _v_1;4:4 or _c;1:1 one _n_1', '0:0 The _q;1:1 typical _a_of;2:2 Glass named;3:3 audience _n_1;7:7 likely _a_1;10:10 composed _a_of;12:12 music _n_1;13:16 _n_of students teachers;0:0 The _q', '0:0 The _q;1:1 work _n_1;3:3 sounds _v_seem-to;4:4 like _p;5:5 Muzak named;6:6 for _p;0:0 The _q', '1:1 success _n_1;3:3 easy _a_for;1:1 success _n_1', '0:0 Softly _a_1;1:1 introducing _v_to;2:2 and _c;3:3 explaining _v_to;5:5 pieces _n_of;6:6 Mr _n_1;7:7 Glass named;8:8 looks _v_1;9:9 and _c;10:10 sounds _v_cause;12:12 like _p;13:13 a _q;14:14 shaggy _a_1;15:15 poet _n_1;16:16 describing _v_to;18:18 work _n_1;20:20 a _q;21:21 classical _a_1;22:22 pianist _n_unknown;23:23 playing _v_1;24:24 a _q;0:0 Softly _a_1', '1:1 is _v_id;2:2 music _n_1;3:3 for _p;4:4 people _n_of;6:8 _v_1 want hear;9:9 something _q;10:10 different _a_than-from;11:11 but _c;13:15 _v_1 want work;16:16 especially _x_deg;17:17 hard _a_for;18:18 at _p;19:19 the _q;1:1 is _v_id', '0:0 Far _x_deg;1:1 from _p_state;3:3 minimalist _a_1;4:4 the _q;5:5 music _n_1;6:6 unabatingly _a_unknown;7:7 torments _v_1;9:9 with _p;10:10 apparent _a_to;11:11 novelties _n_1;14:14 cleverly _a_1;15:15 disguised _v_1;16:16 in _p;17:17 the _q;18:18 simplicities _n_unknown;19:19 of _p;20:20 4/4 dofm;21:21 time _n_of;22:22 octave _a_unknown;23:23 intervals _n_of;24:24 and _c;25:25 ragtime _n_1;26:26 or _c;27:28 _n_1 gospel chord;0:0 Far _x_deg', '0:0 "Opening" _v_cause;1:1 (1981) yofc;2:2 from _p;3:3 Glassworks named;4:4 introduces _v_to;5:5 the _q;6:6 audience _n_1;8:8 the _q;9:9 Glass _n_of;10:10 technique _n_1;11:11 : _p_namely;12:12 Never _a_1;13:13 straying _v_1;15:15 far _a_1;16:16 from _p;17:17 the _q;18:18 piano _n_1;20:20 center _n_of;21:21 Mr _n_1;22:22 Glass named;23:23 works _v_1;24:24 in _p;25:25 the _q;26:26 two card;27:27 octaves _n_unknown;28:28 on _p;29:29 either _q;30:30 side _n_1;31:31 of _p;32:32 middle _n_of;33:33 C named;34:34 and _c;36:36 fingers _n_1;37:37 seldom _a_1;38:38 leave _v_1;39:39 the _q;0:0 "Opening" _v_cause', '1:1 is _v_there;2:2 a _q;3:4 _a_1 recognizable musical;5:5 style _n_of;6:6 here _a_1;7:8 but not _c;9:9 a _q;10:10 particular _a_1;11:11 performance _n_1;1:1 is _v_there', '0:0 The _q;1:1 music _n_1;4:4 especially _a_1;5:5 pianistic; _a_unknown;6:6 indeed _a_1;9:9 hard _a_for;11:11 imagine _v_1;12:12 a _q;13:13 bad _a_at;14:14 performance _n_1;0:0 The _q', '0:0 Nothing _n_1;1:1 bravura _n_unknown;2:2 no _q;3:3 arpeggios _n_unknown;4:4 no _q;5:5 ticklish _a_unknown;6:6 fingering _v_1;7:7 problems _n_of;8:8 challenge _v_to;9:9 the _q;0:0 Nothing _n_1', '1:1 hear _v_1;3:3 may _v_modal;4:4 think _v_1;5:5 inner _a_1;6:6 voices _n_1;7:7 but _c;9:9 all _a_1;10:10 seem _v_to;13:13 saying _v_1;14:14 the _q;15:15 same _a_as;1:1 hear _v_1', '0:0 With _p;1:2 named "Planet News,";3:3 music _n_1;4:6 _v_1 meant accompany;7:7 readings _n_of;9:10 named Allen Ginsberg;12:13 named "Wichita Vortex;14:14 Sutra," named;15:15 Mr _n_1;16:16 Glass named;17:18 _v_1 gets going', '1:1 hands _n_1;2:2 sit _v_1;3:4 _a_1 farther apart;5:5 on _p;6:6 the _q;1:1 hands _n_1', '0:0 Seventh ord;1:1 chords _n_1;2:2 make _v_cause;4:4 feel _v_1;8:8 may _v_modal;9:9 break _v_into;11:11 a _q;12:12 (very _x_deg;13:13 slow) _a_1;14:14 improvisatory _a_unknown;0:0 Seventh ord', '0:0 The _q;1:1 chords _n_1;2:2 modulate _v_1;3:3 but _c;5:5 is _v_there;7:7 filigree _n_1;8:8 even _x_deg;9:9 though _x;11:11 fingers _n_1;12:14 _v_1 begin wander;15:15 over _p_dir;18:18 the _q;0:0 The _q', '0:0 Contrasts _n_1;1:1 predictably _a_1;2:2 accumulate _v_1;3:3 : _p_namely;4:4 First _a_1;5:5 the _q;6:6 music _n_1;8:8 loud _a_1;9:9 then _c;11:11 becomes _v_id;12:12 soft _a_1;13:13 then _c;15:15 realize) _v_1;17:17 becomes _v_id;18:19 _a_1 louder again', '0:0 "The _q;1:2 named Fourth Knee;3:3 Play," named;4:4 an _q;5:5 interlude _n_1;6:6 from _p;7:7 "Einstein named;8:8 on _p;9:9 the _q;10:10 Beach," named;12:12 like _p;13:13 a _q;14:14 toccata _n_unknown;15:15 but _c;18:18 seem _v_to;20:20 move _v_1;21:21 much _x_deg;22:22 beyond _p_dir;25:25 hand _n_1;25:25 hand _leave_v_1;26:26 ground _n_1;27:27 in _p;28:28 "Three card;29:29 Blind named;0:0 "The _q', '0:0 When _x_subord;1:1 Mr _n_1;2:2 Glass named;3:3 decides _v_1;5:5 get _v_state;6:6 really _x_deg;7:7 fancy _a_1;9:9 crosses _v_1;11:11 hands _n_1;12:12 and _c;13:13 hits _v_1;14:14 a _q;15:15 resonant _a_unknown;16:17 _n_1 bass note;18:18 with _p;20:20 right _a_1;0:0 When _x_subord', '1:1 does _v_1;2:2 this _q_dem;3:3 in _p;4:5 at least _a_1;6:6 three card;9:9 solo _n_1;1:1 does _v_1', '1:1 might _v_modal;2:2 call _v_name;4:4 a _q;5:5 leitmotif _n_unknown;6:6 or _c;7:7 a _q;8:8 virtuoso _n_unknown;1:1 might _v_modal', '3:3 the _q;4:4 typical _a_of;5:5 Glasswork named;7:7 ended _n_of;7:7 ended _open_a_1;10:11 _a_1 also often;12:12 multiple _a_1;13:13 in _p;3:3 the _q', '0:1 named "Mad Rush";2:2 began _v_1;4:4 life _n_of;5:5 as _p;6:6 the _q;7:7 accompaniment _n_1;8:8 to _p;9:9 the _q;10:11 named Dalai Lama;13:13 first ord;14:14 public _a_1;15:15 address _n_1;16:16 in _p;17:17 the _q;18:18 U.S named_n;19:19 when _x_subord;20:20 Mr _n_1;21:21 Glass named;22:22 played _v_1;24:24 on _p;25:25 the _q;26:26 organ _n_1;27:27 at _p;28:29 named New York;31:31 Cathedral named;32:32 of _p;33:33 St _n_1;34:34 John named;35:35 the _q;0:0 "Mad named', '0:0 Later _a_for;3:3 performed _v_1;4:4 on _p;5:6 named Radio Bremen;7:7 in _p;8:8 Germany named;9:10 and then _c;11:12 named Lucinda Childs;13:13 took _v_1;15:15 for _p;16:16 one card;19:19 dance _n_1;0:0 Later _a_for', '0:0 The _q;1:1 point _n_of;2:2 is _v_nv;4:4 any _q;5:5 piece _n_of;6:6 can _v_modal;8:8 used _v_as;10:11 _n_1 background music;12:12 for _p;13:13 virtually _x_deg;0:0 The _q', '0:0 The _q;1:1 evening _n_of;2:2 ended _v_1;3:3 with _p;4:4 Mr _n_1;5:7 named Glass "Metamorphosis,";8:8 another _q;9:9 multiple _a_1;0:0 The _q', '0:0 Parts named;1:2 card 1 2;3:3 and _c;4:4 5 card;5:5 come _v_1;6:6 from _p_dir;7:7 the _q;8:8 soundtrack _n_unknown;9:9 of _p;10:11 named Errol Morris;13:13 acclaimed _a_unknown;14:14 film _n_1;15:15 "The _q;16:17 named Thin Blue;18:18 Line," named;19:19 and _c;20:20 the _q;21:21 two card;22:22 other _a_1;23:23 parts _n_1;24:24 from _p;25:25 incidental _a_1;26:26 music _n_1;27:27 to _p;28:28 two card;29:29 separate _a_from;30:30 dramatizations _n_unknown;31:31 of _p;32:32 the _q;33:33 Kafka named;34:34 story _n_of;35:35 of _p;36:36 the _q;37:37 same _a_as;0:0 Parts named', '0:0 When _x_subord;1:1 used _v_as;3:3 background _n_1;4:4 in _p;5:5 this _q_dem;6:6 way _n_of;7:7 the _q;8:8 music _n_1;9:9 has _v_1;10:10 an _q;11:11 appropriate _a_to;12:12 eeriness _n_unknown;13:13 as _p;15:15 a _q;17:17 note _n_1;17:17 note card;18:18 phrase _n_1;19:19 a _q;20:20 descending _v_1;21:21 minor _a_1;22:22 third _n_of;23:23 accompanies _v_1;24:24 the _q;25:26 _a_1 seemingly endless;27:27 litany _n_unknown;28:28 of _p;29:29 reports _n_of;30:30 interviews _n_1;31:31 and _c;32:32 confessions _n_of;34:34 witnesses _n_1;35:35 in _p;36:36 the _q;37:37 Morris named;0:0 When _x_subord', '0:0 Served _v_as;1:1 up _p;3:3 a _q;4:4 solo _n_1;6:6 the _q;7:7 music _n_1;8:8 lacks _v_1;9:9 the _q;10:10 resonance _n_1;11:11 provided _v_1;13:13 a _q;14:14 context _n_1;15:15 within _p;16:16 another _q;0:0 Served _v_as', '0:0 Admirers _n_unknown;1:1 of _p;2:2 Mr _n_1;3:3 Glass named;4:4 may _v_modal;5:5 agree _v_on;6:6 with _p;7:7 the _q;8:8 critic _n_of;9:10 named Richard Kostelanetz;12:12 sense _n_of;14:14 the _q;15:15 1974 yofc;16:16 "Music _n_1;17:17 in _p;18:18 Twelve card;19:19 Parts" _n_1;22:22 encyclopedic _a_unknown;23:23 and _c;24:24 weighty _a_1;25:25 as _p;26:27 named "The Well-;28:29 named Tempered Clavier."', '0:0 But _c;1:1 while _x;2:2 making _v_1;3:3 the _q;4:4 obvious _a_to;5:5 point _n_of;7:7 both _q;7:7 both card;8:8 composers _n_of;9:9 develop _v_cause;10:10 variations _n_of;11:11 from _p;12:12 themes _n_of;13:13 this _q_dem;14:14 comparison _n_1;15:15 ignores _v_1;16:16 the _q;17:17 intensely _a_1;18:18 claustrophobic _a_unknown;19:19 nature _n_1;20:20 of _p;21:21 Mr _n_1;22:22 Glass named;0:0 But _c', '1:2 _a_1 supposedly austere;3:3 minimalism _n_unknown;4:4 overlays _v_1;5:5 a _q;6:6 bombast _n_unknown;8:8 makes _v_cause;9:9 one _n_1;10:10 yearn _v_for;12:12 the _q;13:13 astringency _n_unknown;14:14 of _p;15:15 neoclassical _a_unknown;16:16 Stravinsky named;17:17 the _q;18:19 _a_1 genuinely radical;20:20 minimalism _n_unknown;21:21 of _p;22:22 Berg named;23:23 and _c;24:24 Webern named;25:25 and _c;27:27 in _p;28:28 retrospect _n_unknown;29:29 even _a_1;30:30 seems _v_to;31:31 like _p;32:32 concision _n_unknown;33:33 in _p;1:1 supposedly _a_1', '0:0 Mr _n_1;1:1 Spiegelman named;2:2 is _v_id;3:3 professor _n_of;4:4 of _p;5:5 English named;6:6 at _p;7:8 named Southern Methodist;9:9 University named;10:10 and _c;11:11 editor _n_1;12:12 of _p;13:13 the _q;14:15 named Southwest Review', '0:0 Honeywell named;1:1 Inc _n_1;2:2 said _v_to;4:4 hopes _v_1;6:6 complete _v_2;7:7 shortly _a_1;8:8 the _q;9:9 first ord;11:11 two card;12:14 _n_of sales shares;15:15 in _p;17:18 _a_1 Japanese joint;19:19 venture _n_1;20:21 named Yamatake- Honeywell;22:22 for _p;23:23 about _x_deg;24:24 $ _n_1;25:26 card 280 million', "0:0 The _q;1:1 company _n_of;2:2 wouldn't _v_modal;3:3 disclose _v_to;4:4 the _q;5:5 buyer _n_of;7:7 the _q;8:8 initial _a_1;9:9 16 card;10:10 % _n_of;0:0 The _q", '0:0 Proceeds _n_1;1:1 of _p;2:2 the _q;3:3 sale _n_of;4:4 expected _v_1;7:7 completed _v_2;8:8 next _a_1;9:9 week _n_1;10:10 would _v_modal;12:14 _v_1 used repurchase;14:14 repurchase _re-_a_again;18:19 card 10 million;20:20 shares _n_of;21:21 of _p;22:22 Honeywell named;23:23 stock _n_1;24:24 the _q;25:25 company _n_of;0:0 Proceeds _n_1', '0:0 Honeywell named;1:1 said _v_to;4:4 negotiating _v_1;5:5 the _q;6:6 sale _n_of;8:8 a _q;9:9 second ord;10:10 stake _n_1;11:11 in _p;12:13 named Yamatake- Honeywell;14:14 but _c;15:15 indicated _v_1;17:17 intends _v_for;19:19 hold _v_1;20:21 at least _x_deg;22:22 20 card;23:23 % _n_of;24:24 of _p;25:25 the _q;26:26 joint _a_1;27:29 _n_1 venture stock;0:0 Honeywell named', '0:0 A _q;1:1 20 card;2:2 % _n_of;3:3 stake _n_1;4:4 would _v_modal;5:5 allow _v_1;6:6 Honeywell named;8:8 include _v_1;9:9 Yamatake named;10:10 earnings _n_1;11:11 in _p;0:0 A _q', '0:0 Honeywell named;1:1 previously _p;2:2 said _v_to;4:4 intended _v_for;6:6 reduce _v_1;8:8 holding _n_1;9:9 in _p;10:10 the _q;11:11 Japanese _a_1;12:12 concern _n_1;13:13 as _p;14:16 _q part a;17:17 restructuring _v_1;18:18 plan _n_1;20:20 also _a_1;21:21 calls _v_for;23:23 a _q;24:24 reduction _n_of;26:26 dependence _n_1;27:27 on _p;28:28 weapons _n_1;0:0 Honeywell named', '0:0 Yesterday _a_1;1:1 a _q;2:2 spokeswoman _n_unknown;3:3 said _v_to;4:4 the _q;5:5 company _n_of;7:7 "pleased _a_with;10:10 progress" _n_1;11:11 in _p;12:12 that _q_dem;13:13 regard _n_1;14:14 and _c;15:17 _v_1 "hopes provide;18:18 additional _a_1;19:19 details _n_1;0:0 Yesterday _a_1', '0:0 Honeywell named;1:1 said _v_to;3:3 Defense named;4:4 and _c;5:6 named Marine Systems;7:7 group _n_of;8:8 incurred _v_1;9:9 delays _n_1;10:10 in _p;11:11 shipping _v_cause;12:12 some _q;13:13 undisclosed _v_to;13:13 undisclosed _un-_a_rvrs;14:14 contracts _n_1;15:15 during _p;16:16 the _q;17:17 third ord;18:18 quarter _n_temp;19:19 resulting _v_in;21:21 lower _a_1;22:22 operating _v_1;23:23 profit _n_1;24:24 for _p;25:25 that _q_dem;0:0 Honeywell named', '0:0 Overall _a_1;1:1 Honeywell named;2:2 reported _v_to;3:3 earnings _n_1;4:4 of _p;5:5 $ _n_1;6:7 card 74.4 million;8:8 or _c;9:9 $ _n_1;10:10 1.73 card;11:11 a _p_per;12:12 share _n_of;13:13 for _p;14:14 the _q;15:15 three card;16:16 months _n_1;17:17 ended _v_cause;18:18 Oct mofy;19:19 1 dofm;20:20 compared _v_with;22:22 a _q;23:23 loss _n_of;25:25 $ _n_1;26:27 card 41.4 million;28:28 or _c;29:29 98 card;30:30 cents _n_1;31:31 a _p_per;32:32 share _n_of;33:33 a _q;34:34 year _n_1;0:0 Overall _a_1', '0:0 The _q;1:1 previous _a_1;2:4 _n_of period results;5:5 included _v_1;6:6 a _q;7:7 $ _n_1;8:9 card 108 million;10:10 pretax _n_1;10:10 pretax _pre-_a_ante;11:11 charge _n_of;12:12 related _v_to;14:14 unrecoverable _a_unknown;15:16 _n_1 contract costs;17:17 and _c;18:18 a _q;19:19 $ _n_1;20:21 card 12.3 million;22:22 pretax _n_1;22:22 pretax _pre-_a_ante;23:23 gain _n_1;24:24 on _p;25:26 real estate _n_1;0:0 The _q', '0:0 Sales _n_of;1:1 for _p;2:2 the _q;3:3 latest _a_for;4:4 quarter _n_temp;6:6 flat _a_1;7:7 at _p;8:8 $ _n_1;9:10 card 1.72 billion', '0:0 For _p;1:1 the _q;2:2 nine card;3:3 months _n_1;4:4 Honeywell named;5:5 reported _v_to;6:6 earnings _n_1;7:7 of _p;8:8 $ _n_1;9:10 card 212.1 million;11:11 or _c;12:12 $ _n_1;13:13 4.92 card;14:14 a _p_per;15:15 share _n_of;16:16 compared _v_with;18:18 earnings _n_1;19:19 of _p;20:20 $ _n_1;21:22 card 47.9 million;23:23 or _c;24:24 $ _n_1;25:25 1.13 card;26:26 a _p_per;27:27 share _n_of;28:28 a _q;29:29 year _n_1;0:0 For _p', '0:0 Sales _n_of;1:1 declined _v_1;2:2 slightly _a_1;3:3 to _p;4:4 $ _n_1;5:6 card 5.17 billion', '0:1 Once again _a_1;3:4 _n_1 editorial page;5:5 misstates _v_to;5:5 misstates _mis-_a_error;6:6 the _q;7:7 law _n_1;8:8 to _x;9:9 conform _v_to;12:12 almost _a_1;13:13 beatific _a_unknown;0:1 Once again _a_1', '0:0 This _q_dem;1:1 statement _n_of;2:2 surely _a_1;3:3 buttresses _v_1;5:5 editorial _a_1;6:6 viewpoint _n_1;8:8 environmental _a_1;9:9 protection _n_of;11:12 _a_1 generally silly;13:13 or _c;14:14 excessive _a_1;15:15 but _c;18:18 simply _a_1;0:0 This _q_dem', '0:0 The _q;1:2 named Clean Water;3:3 Act named;4:4 contains _v_1;5:5 no _q;6:6 "legal _a_1;7:7 standard" _n_1;8:8 of _p;9:9 zero _n_1;0:0 The _q', '1:1 requires _v_of;3:3 "discharges _n_of;5:5 pollutants" _n_unknown;6:6 into _p;7:7 the _q;8:8 "waters _n_1;9:9 of _p;10:10 the _q;11:12 named United States";14:14 authorized _v_1;16:16 permits _n_1;18:18 reflect _v_1;19:19 the _q;20:20 effluent _a_unknown;21:21 limitations _n_1;22:22 developed _v_cause;23:23 under _p;24:24 section _n_of;1:1 requires _v_of', '0:0 Whatever _p;1:1 may mofy;2:2 be _v_id;3:3 the _q;4:4 problems _n_of;5:5 with _p;6:6 this _q_dem;7:7 system _n_of;9:9 scarcely _a_1;10:10 reflects _v_1;11:11 "zero _n_1;12:12 risk" _n_of;13:13 or _c;14:14 "zero _n_1;0:0 Whatever _p', '0:0 Perhaps _a_1;1:1 Mr _n_1;2:2 Greve named;4:4 confused _v_1;6:6 Congress named;8:8 meaningless _a_1;9:9 statement _n_of;11:11 "the _q;12:12 national _a_1;13:13 goal" _n_1;14:14 in _p_temp;15:15 section _n_of;16:16 101 yofc;18:18 indeed _a_1;19:19 calls _v_for;21:21 the _q;22:22 elimination _n_1;23:23 of _p;24:24 discharges _n_of;26:26 by _p_temp;27:27 1985 yofc;0:0 Perhaps _a_1', '0:0 This _q_dem;1:1 fatuous _a_unknown;2:2 statement _n_of;5:5 taken _v_1;6:6 seriously _a_1;7:7 when _x_subord;8:8 enacted _v_1;9:9 in _p_temp;10:10 1972 yofc;11:11 and _c;12:12 should _v_modal;14:14 now _a_1;16:16 confused _v_with;18:18 the _q;19:19 operative _a_1;20:20 provisions _n_1;21:21 of _p;22:22 the _q;0:0 This _q_dem', '0:1 named Robert J;0:0 Robert named', '0:0 To _x;1:1 hear _v_1;2:2 advertisers _n_unknown;3:3 tell _v_1;5:5 the _q;6:6 magazine _n_1;7:7 just _a_1;9:9 kept _v_up;11:11 with _p;12:12 the _q;0:0 To _x', '0:0 Despite _p;1:1 renewed _v_1;2:2 interest _n_in;3:3 by _p;4:4 the _q;5:5 public _n_1;7:7 such _q;8:8 topics _n_of;9:9 as _p;10:10 the _q;11:11 environment _n_1;12:12 and _c;13:13 the _q;14:15 named Third World;19:19 able _a_1;21:21 shake _v_1;23:23 reputation _n_1;24:24 as _p;25:25 a _q;26:27 _n_1 magazine boys;28:30 _v_1 like flip;31:31 through _p_dir;32:32 in _p;33:33 search _n_of;35:35 topless _a_unknown;36:37 _n_1 tribe women', '0:0 Worse _a_1;2:2 lagged _v_1;3:3 behind _p;4:4 competitors _n_1;5:5 in _p;6:6 offering _v_1;8:8 standard _a_1;8:8 standard _now_a_1;9:9 gimmicks _n_unknown;10:10 from _p_time;11:11 regional _a_1;12:12 editions _n_1;13:13 to _p;14:14 discounts _n_1;15:15 for _p;16:16 frequent _a_1;0:0 Worse _a_1', '0:0 But _c;1:1 now _a_1;2:2 the _q;3:3 magazine _n_1;5:5 attempting _v_1;7:7 fight _v_back;9:9 with _p;10:10 an _q;11:11 ambitious _a_1;12:12 plan _n_1;13:13 including _v_1;14:14 a _q;15:15 revamped _v_unknown;16:17 _n_of sales strategy;18:18 and _c;19:19 a _q;20:21 _a_1 surprisingly aggressive;22:23 _n_1 ad campaign', '0:0 Advertisers _n_unknown;2:2 think _v_of;4:4 the _q;5:5 magazine _n_1;6:6 first _a_1;7:7 says _v_to;8:9 named Joan McCraw;11:11 joined _v_1;12:12 in _p_temp;13:13 April mofy;14:14 as _p_nbar;15:15 national _a_1;16:16 advertising _n_1;0:0 Advertisers _n_unknown', '0:0 "What _q;2:2 want _v_1;4:4 do _v_be;5:5 is _v_do;6:6 take _v_1;7:7 a _q;9:9 aggressive _a_1;0:0 "What _q', '0:0 People _n_of;2:2 believe _v_1;5:5 in _p;6:6 tune _n_1;7:7 with _p;8:8 the _q;9:9 marketplace _n_of;10:10 and _c;11:11 in _p_state;0:0 People _n_of', '0:0 The _q;2:2 year- _n_1;2:2 year- card;3:3 old _a_1;4:4 magazine _n_1;6:6 never _a_1;7:7 had _v_qmodal;9:9 woo _v_unknown;10:10 advertisers _n_unknown;11:11 with _p;12:12 quite _x;15:15 fervor _n_unknown;0:0 The _q', '0:0 The _q;1:1 magazine _n_1;2:2 had _v_1;4:4 best _a_at-for-of;5:5 year _n_1;6:6 yet _a_1;7:7 in _p_temp;8:8 1988 yofc;9:9 when _x_subord;11:11 celebrated _v_1;13:13 centennial _n_1;14:14 and _c;15:15 racked _v_up;17:17 a _q;18:18 17 card;19:19 % _n_of;20:20 gain _n_1;21:21 in _p;22:23 _n_1 ad pages;24:24 to _p;0:0 The _q', '0:0 But _x;1:1 this _q_dem;2:2 year _n_1;3:3 when _x_subord;4:4 the _q;5:5 hullabaloo _n_unknown;6:6 surrounding _v_1;8:8 centennial _n_1;9:9 died _v_1;11:11 too _a_also;13:13 some _q;14:14 advertiser _n_unknown;0:0 But _x', '0:0 The _q;1:1 reason _n_for;2:3 _n_1 ad executives;4:4 say _v_to;5:5 is _v_nv;7:7 the _q;8:8 entire _a_1;9:10 _n_1 magazine business;13:13 soft _a_1;15:15 and _c;16:17 named National Geographic;18:18 has _v_1;19:19 some _q;20:20 quirks _n_unknown;22:22 make _v_cause;24:25 _a_1 especially unattractive;26:26 during _p;27:27 a _q;28:28 soft _a_1;0:0 The _q', '0:0 Perhaps _x_deg;1:1 the _q;2:2 biggest _a_1;4:4 those _q_dem;5:5 factors _n_1;6:6 is _v_id;8:8 high _a_1;9:9 ad _n_1;10:10 prices _n_of;12:12 $ _n_1;13:13 130,000 card;14:14 for _p;15:15 a _q;17:17 color _n_1;17:17 color card;18:18 page _n_1;19:19 vs _c;20:20 $ _n_1;21:21 47,000 card;22:22 for _p;23:23 the _q;24:24 Smithsonian named;25:25 a _q;26:26 comparable _a_1;27:27 publication _n_1;28:28 with _p;29:29 a _q;30:30 far _x_deg;31:31 smaller _a_1;0:0 Perhaps _x_deg', '0:0 When _x_subord;1:2 _n_1 ad dollars;4:4 tight _a_1;5:5 the _q;6:6 high _a_1;7:8 _n_1 page cost;9:9 is _v_id;10:10 a _q;11:11 major _a_1;12:12 deterrent _n_unknown;13:13 for _p;14:14 advertisers _n_unknown;16:16 generally _a_1;17:19 _v_1 want appear;20:20 regularly _a_1;21:21 in _p;22:22 a _q;23:23 publication _n_1;24:24 or _c;26:26 at _p;0:0 When _x_subord', '0:0 To _x;1:1 combat _v_1;2:2 that _q_dem;3:3 problem _n_of;4:5 named National Geographic;6:6 like _p;7:7 other _a_1;8:8 magazines _n_1;9:10 _v_1 began offering;11:11 regional _a_1;12:12 editions _n_1;13:13 allowing _v_1;14:14 advertisers _n_unknown;16:16 appear _v_1;17:17 in _p;18:18 only _x_deg;19:19 a _q;20:20 portion _n_of;23:23 magazines _n_1;25:26 for example _a_1;27:27 ads _n_1;28:28 can _v_modal;29:29 run _v_1;30:30 only _x_deg;31:31 in _p;32:32 the _q;33:33 magazines _n_1;34:34 sent _v_1;35:35 to _p;36:36 subscribers _n_to;37:37 in _p;38:38 the _q;39:39 largest _a_1;40:40 25 card;0:0 To _x', '0:0 Time named;1:1 magazine _n_1;4:4 has _v_1;7:7 100 card;8:8 separate _a_from;9:9 editions _n_1;10:10 going _v_1;11:11 to _p;12:12 different _a_than-from;13:13 regions _n_of;14:14 top _a_1;15:15 management _n_1;16:16 and _c;17:17 other _a_1;0:0 Time named', '0:0 Another _q;1:1 sticking _v_1;2:2 point _n_of;3:3 for _p;4:4 advertisers _n_unknown;5:5 was _v_id;6:7 named National Geographic;9:9 tradition _n_1;10:10 of _p;11:11 lumping _v_1;13:13 ads _n_1;14:14 together _p;15:15 usually _a_1;16:16 at _p;17:17 the _q;18:18 beginning _n_1;19:19 or _c;20:20 end _n_of;21:21 of _p;22:22 the _q;23:23 magazine _n_1;24:25 rather than _c;26:26 spreading _v_out;27:27 ads _n_1;29:29 among _p;31:31 articles _n_of;32:32 as _x_subord;33:33 most _q;0:0 Another _q', '0:0 But _c;1:1 Ms _n_1;2:2 McCraw named;3:3 says _v_to;4:4 the _q;5:5 magazine _n_1;0:0 But _c', '1:1 now _a_1;2:2 offers _v_1;3:3 30 card;4:4 regional _a_1;5:5 editions _n_1;7:7 very _x_deg;8:8 recently _a_1;9:10 _v_1 began running;11:11 ads _n_1;12:13 adjacent to _p;14:14 articles _n_of;15:15 and _c;19:19 beefing _v_up;22:22 sales _n_of;1:1 now _a_1', '0:0 And _c;2:2 just _a_1;3:3 launched _v_cause;4:4 a _q;5:5 promotional _a_1;6:6 campaign _n_1;8:8 tell _v_1;9:10 _n_1 chief executives;11:11 marketing _v_1;12:12 directors _n_of;13:13 and _c;14:15 _n_1 media executives;16:16 just _x_deg;0:0 And _c', '0:0 The _q;1:1 campaign _n_1;2:2 created _v_1;4:5 named Omnicom Group;7:8 named DDB Needham;9:9 agency _n_1;10:10 takes _v_1;11:11 advantage _n_1;12:12 of _p;13:13 the _q;14:14 eye- _n_1;15:15 catching _v_1;16:16 photography _n_1;17:17 that _q_dem;18:19 named National Geographic;21:21 known _v_1;0:0 The _q', '0:0 In _p_state;1:1 one card;2:2 ad _n_1;3:3 a _q;4:4 photo _n_1;5:5 of _p;6:6 the _q;7:7 interior _n_1;8:8 of _p;9:9 the _q;10:11 named Sainte- Chapelle;12:12 in _p;13:13 Paris named;15:15 paired _v_1;16:16 with _p;17:17 the _q;18:18 headline _n_1;19:19 "The _q;20:20 only _a_1;21:21 book _n_of;23:23 respected _v_1;24:24 than _p_ind;27:28 _v_1 accept advertising."', '0:0 Another _q;1:1 ad _n_1;2:2 pictures _v_1;3:3 a _q;4:4 tree _n_of;5:5 ant _n_1;6:6 magnified _v_1;7:7 80 card;8:8 times _n_of;9:9 with _p;10:10 the _q;11:11 headline _n_1;12:12 "For _p;13:13 impact _n_1;14:14 far _x_deg;15:15 beyond _p;17:17 size _n_1;18:18 consider _v_1;20:20 regional _a_1;0:0 Another _q', '0:0 Ms _n_1;1:1 McCraw named;2:2 says _v_to;4:4 wants _v_1;5:5 the _q;6:6 campaign _n_1;8:8 help _v_to;9:9 attract _v_1;10:10 advertisers _n_unknown;11:11 in _p;12:12 10 card;13:13 categories _n_of;14:14 including _v_1;15:16 _a_1 corporate financial;17:17 services _n_1;18:18 consumer _n_of;19:20 _n_1 electronics insurance;21:21 and _c;0:0 Ms _n_1', '1:1 goal _n_1;2:2 : _p_namely;3:3 to _x;4:4 top _v_1;5:5 300 card;6:7 _n_1 ad pages;8:8 in _p_temp;9:9 1990 yofc;10:11 _p up from;12:12 about _x_deg;13:13 274 card;14:14 this _q_dem;1:1 goal _n_1', '2:2 can _v_modal;3:3 meet _v_1;4:4 that _q_dem;5:5 ambitious _a_1;6:6 goal _n_1;8:8 still _a_1;9:10 far from _x_deg;2:2 can _v_modal', '0:0 "The _q;1:2 _n_1 ad campaign;4:4 meant _v_1;6:6 contemporize _v_unknown;7:7 the _q;8:8 thought _n_of;10:11 named National Geographic,";0:0 "The _q', "1:1 want _v_1;4:4 be _v_id;5:5 a _q;6:6 '90s _a_unknown;7:7 kind _n_of-n;1:1 want _v_1", '0:1 named WCRS Group;2:2 hopes _v_1;4:4 announce _v_to;5:5 perhaps _x_deg;6:6 today _a_1;7:7 an _q;8:8 agreement _n_1;10:10 sell _v_1;11:11 the _q;12:12 majority _n_of;15:15 ad _n_1;16:16 unit _n_of;18:18 Paris- named;19:19 based _v_1;20:20 Eurocom named;21:21 a _q;22:22 European _a_1;23:24 _n_1 ad executive;0:0 WCRS named', '0:0 WCRS named;3:3 in _p;4:4 discussions _n_1;5:5 with _p;6:6 Eurocom named;7:7 for _p;8:8 several _a_1;0:0 WCRS named', '0:0 However _a_1;1:1 when _x_subord;2:2 negotiations _n_of;3:3 bogged _v_down;5:5 recently _a_1;6:6 WCRS named;8:9 _n_1 chief executive;10:11 named Peter Scott;12:12 met _v_1;13:13 in _p;14:14 Paris named;15:15 with _p;16:16 another _q;17:17 French _a_1;18:18 firm _n_1;19:20 named Boulet Dru;21:22 named Dupuy Petit;23:23 or _c;0:0 However _a_1', '0:1 According to _p;2:2 the _q;3:3 executive _n_1;4:4 BDDP named;6:6 involvement _n_in;7:8 _v_1 prompted renewed;9:9 vigor _n_1;10:10 in _p;11:11 the _q;12:13 named WCRS- Eurocom;14:14 talks _n_of-on;15:15 and _c;16:16 the _q;17:17 two card;18:18 agencies _n_1;20:20 hoping _v_1;22:22 hammer _v_out;24:24 details _n_1;25:25 by _p_temp;0:1 According to _p', "0:0 Executives _n_1;1:1 of _p;2:2 the _q;3:3 two card;4:4 agencies _n_1;5:5 couldn't _v_modal;7:7 reached _v_1;8:8 last _a_1;0:0 Executives _n_1", '0:0 NEW _a_1;1:1 ACCOUNT _n_of;2:2 : _p_namely;3:3 Procter named;4:4 & _c;5:5 Gamble named;6:6 Co _n_1;7:7 Cincinnati named;8:8 awarded _v_1;9:9 the _q;10:10 ad _n_1;11:11 accounts _n_of;12:12 for _p;14:14 line _n_of;16:17 named Professional Crisco;18:18 vegetable _n_1;19:19 shortening _v_1;20:20 and _c;21:22 _n_1 oil products;24:25 named Northlich Stolley;26:27 named LaWarre Cincinnati', '0:0 Billings _n_1;0:0 Billings _n_1', '0:0 Professional _a_1;1:1 Crisco named;2:2 products _n_1;4:4 specially _a_1;5:5 made _v_1;6:6 for _p;7:7 the _q;8:8 foodservice _n_unknown;0:0 Professional _a_1', '1:1 was _v_id;2:3 _n_1 executive vice;4:5 _n_of president director;7:7 broadcast _n_1;1:1 was _v_id', '0:0 The _q;1:2 named Commodity Futures;3:4 named Trading Commission;5:5 plans _v_1;7:7 restrict _v_to;8:8 dual _a_1;9:9 trading _v_1;10:10 on _p;11:12 _n_1 commodity exchanges;13:13 a _q;14:14 move _n_1;15:15 almost _a_1;16:16 certain _a_of;18:18 infuriate _v_1;19:20 _n_1 exchange officials;21:21 and _c;0:0 The _q', '0:0 The _q;1:1 CFTC named;2:5 _v_to said propose;6:6 the _q;7:7 restrictions _n_of;8:8 after _p;9:9 the _q;10:10 release _n_of;12:12 a _q;13:13 study _n_of;15:15 shows _v_1;17:17 economic _a_1;18:18 benefit _n_of;19:19 resulting _v_from;21:21 dual _a_1;22:22 trading _v_1;23:23 and _c;24:24 cites _v_for;25:25 "problems" _n_of;26:26 associated _v_with;28:28 the _q;0:0 The _q', '0:0 Dual _a_1;1:2 _v_1 trading gives;3:3 an _q;4:5 _n_1 exchange trader;6:6 the _q;7:7 right _n_1;9:9 trade _v_1;11:11 for _p;13:13 own _a_1;14:14 account _n_of;15:15 and _c;16:16 for _p;0:0 Dual _a_1', '0:0 While _x;2:2 specifically _a_to;3:3 mentioned _v_to;4:4 in _p;5:5 the _q;6:6 FBI named;7:7 charges _n_of;8:8 dual _a_1;9:9 trading _v_1;10:10 became _v_id;11:11 a _q;12:12 focus _n_1;13:13 of _p;14:14 attempts _n_1;16:16 tighten _v_cause;17:18 _n_1 industry regulations', '0:0 Critics _n_of;1:1 contend _v_1;3:3 traders _n_1;5:6 _v_1 putting buying;7:7 or _c;8:8 selling _v_1;9:9 for _p;11:11 own _a_1;12:12 accounts _n_of;13:14 ahead of _p;15:15 other _a_1;16:16 traders _n_1;18:19 _n_of customer orders', '0:0 Traders _n_1;2:2 likely _a_1;4:4 oppose _v_1;5:5 such _q;6:6 restrictions _n_of;7:7 because _x;8:8 dual _a_1;9:10 _v_1 trading provides;11:11 a _q;12:12 way _n_of;14:14 make _v_1;15:15 money _n_1;16:16 in _p;17:17 slower _a_1;18:18 markets _n_1;21:21 is _v_there;22:22 a _q;23:25 _n_of shortage customer;0:0 Traders _n_1', '0:0 The _q;1:1 exchanges _n_1;2:2 contend _v_1;4:4 dual _a_1;5:5 trading _v_1;6:6 improves _v_cause;7:7 liquidity _n_1;8:8 in _p;9:9 the _q;10:10 markets _n_1;11:11 because _x;12:12 traders _n_1;13:13 can _v_modal;14:14 buy _v_1;15:15 or _c;16:16 sell _v_1;17:17 even _x_deg;18:18 when _x_subord;21:21 have _v_1;22:22 a _q;23:24 _n_of customer order;25:25 in _p;0:0 The _q', '0:0 The _q;1:1 exchanges _n_1;2:2 say _v_to;3:3 liquidity _n_1;4:4 becomes _v_id;5:5 a _q;6:6 severe _a_1;7:7 problem _n_of;8:8 for _p;9:9 thinly _a_unknown;10:10 traded _v_1;11:11 contracts _n_1;12:13 such as _p;14:14 those _q_dem;15:15 with _p;16:16 a _q;17:17 long _a_1;18:18 time _n_of;19:19 remaining _v_1;20:20 before _p;0:0 The _q', '0:0 The _q;1:1 CFTC named;2:2 may _v_modal;3:3 take _v_into;4:4 those _q_dem;5:5 arguments _n_1;7:7 account _n_i;8:8 by _p_means;9:9 allowing _v_1;10:10 exceptions _n_to;0:0 The _q', '0:0 The _q;1:1 agency _n_1;3:3 cite _v_for;4:4 specific _a_to;5:5 situations _n_1;7:7 dual _a_1;8:8 trading _v_1;9:9 might _v_modal;11:11 allowed _v_1;12:12 but _c;13:13 smaller _a_1;14:14 exchanges _n_1;15:15 or _c;16:16 contracts _n_1;18:18 need _v_1;19:19 additional _a_1;20:20 liquidity _n_1;22:22 expected _v_1;0:0 The _q', '0:0 The _q;1:1 study _n_of;2:2 by _p;3:3 the _q;4:4 CFTC named;6:6 division _n_of;8:8 economic _a_1;9:9 analysis _n_of;10:10 shows _v_1;12:12 "a _q;13:13 trade _n_of;14:14 is _v_id;15:15 a _q;16:16 trade," _n_of;17:17 a _q;18:18 member _n_of;20:20 the _q;21:22 _n_of study team;0:0 The _q', '1:1 a _q;2:2 trade _n_of;4:4 done _v_1;5:5 on _p;6:6 a _q;7:7 dual _a_1;8:8 or _c;10:10 dual _a_1;11:11 basis _n_of;12:12 the _q;13:13 member _n_of;14:16 _v_to said seem;18:18 have _v_1;20:20 economic _a_1;1:1 a _q', '0:0 "The _q;1:1 tests _n_of;2:2 indicate _v_1;4:4 dual _a_1;5:5 and _c;7:7 dual _a_1;8:8 traders _n_1;10:10 similar _a_to;11:11 in _p;12:12 terms _n_of;14:14 the _q;15:15 trade _n_of;16:16 executions _n_1;17:17 and _c;18:18 liquidity _n_1;20:20 provide _v_1;22:22 the _q;23:24 _n_1 market," Mrs;25:25 Gramm named;26:26 told _v_1;27:27 the _q;28:28 Senate named_n;0:0 "The _q', '0:0 Members _n_of;2:2 Congress named;4:5 _v_to proposed restricting;6:6 dual _a_1;7:7 trading _v_1;8:8 in _p;9:9 bills _n_of;11:11 reauthorize _v_1;11:11 reauthorize _re-_a_again;12:12 CFTC named;0:0 Members _n_of', '0:0 The _q;1:1 House named_n;3:3 bill _n_of;4:4 would _v_modal;5:5 prohibit _v_1;6:6 dual _a_1;7:7 trading _v_1;8:8 in _p;9:9 markets _n_1;10:10 with _p;11:12 _a_1 daily average;13:13 volume _n_of;15:15 7,000 card;16:16 contracts _n_1;17:17 or _c;19:19 comprising _v_1;20:20 those _q_dem;21:21 considered _v_1;23:23 difficult _a_for;25:25 track _v_1;26:26 without _p;27:27 a _q;28:28 sophisticated _a_1;29:29 computer _n_1;0:0 The _q', "0:0 The _q;1:1 Senate named_n;2:2 bill _n_of;3:3 would _v_modal;4:4 force _v_1;5:5 the _q;6:6 CFTC named;8:8 suspend _v_1;9:9 dual _a_1;10:10 trading _v_1;11:11 if _x_then;12:12 an _q;13:13 exchange _n_1;14:14 can't _v_modal;15:15 show _v_1;18:18 oversight _n_1;19:19 system _n_of;20:20 can _v_modal;21:23 _v_1 detect trading;23:23 trading _dual_a_1;0:0 The _q", '1:1 far _a_1;2:2 one card;3:3 test _n_of;5:5 restricting _v_to;6:6 dual _a_1;7:9 _v_1 trading worked;1:1 far _a_1', '0:0 The _q;1:2 named Chicago Merc;3:3 banned _v_from;4:4 dual _a_1;5:5 trading _v_1;6:6 in _p;12:12 500- yofc;13:14 _n_1 stock index;15:16 _n_1 futures pit;17:17 in _p_temp;0:0 The _q', '0:0 Under _p_state;1:1 the _q;2:2 rules _n_of;3:3 traders _n_1;4:4 decide _v_1;5:5 before _x_h;6:6 a _q;7:7 session _n_1;8:12 _v_1 begins trade;13:13 for _p;15:15 own _a_1;16:16 account _n_of;17:17 or _c;18:18 for _p;0:0 Under _p_state', "0:0 Traders _n_1;2:2 stand _v_1;3:3 on _p;4:4 the _q;5:5 pit _n_1;7:7 top _a_1;8:8 step _n_1;10:10 most _q;11:12 _n_of customer orders;14:14 executed _v_1;15:15 can't _v_modal;16:16 trade _v_1;0:0 Traders _n_1", '0:0 A _q;1:1 Merc named;2:2 spokesman _n_1;3:3 said _v_to;4:4 the _q;5:5 plan _n_1;7:7 made _v_1;9:9 difference _n_1;10:10 in _p;11:11 liquidity _n_1;12:12 in _p;13:13 the _q;0:0 A _q', '3:3 soon _p;5:5 tell _v_1;7:7 but _c;8:8 people _n_of;10:10 seem _v_to;13:13 unhappy _a_1;14:14 with _p;3:3 soon _p', "1:1 said _v_to;3:3 wouldn't _v_modal;4:4 comment _v_on;6:6 the _q;7:7 CFTC named;8:8 plan _n_1;9:9 until _x_h;10:10 the _q;11:11 exchange _n_1;13:13 seen _v_1;14:14 the _q;15:15 full _a_of;1:1 said _v_to", '0:0 But _c;1:1 at _p;2:2 a _q;3:3 meeting _n_of;4:4 last _a_1;5:5 week _n_1;6:7 named Tom Donovan;8:8 the _q;9:9 Board _n_of;11:11 Trade named;13:13 president _n_of;14:14 told _v_1;15:16 _n_1 commodity lawyers;17:17 : _p_namely;18:18 "Dual _a_1;19:19 trading _v_1;21:22 _a_1 definitely worth;0:0 But _c', '1:1 adds _v_to;2:4 _q something the;1:1 adds _v_to', '0:0 Japanese _a_1;1:1 Firms _n_1;2:2 Push _v_1;3:3 Posh _a_1;4:5 _n_1 Car Showrooms', '0:0 JAPANESE _a_1;1:2 _n_1 luxury- car;3:3 makers _n_of;5:7 _v_1 trying set;8:8 strict _a_1;9:9 design _n_of;10:10 standards _n_1;11:11 for _p;0:0 JAPANESE _a_1', '0:0 But _c;1:1 some _q;2:2 dealers _n_1;4:4 negotiating _v_1;5:5 looser _a_1;6:6 terms _n_of;7:7 while _x;8:8 others _n_1;9:11 _v_1 decline deal;0:0 But _c', '0:1 named Nissan Motor;2:2 Co _n_1;4:4 Infiniti named;5:5 division _n_of;6:8 _v_1 likes insist;10:10 every _q;11:11 dealer _n_1;12:12 construct _v_1;13:13 and _c;14:14 furnish _v_to;15:15 a _q;16:16 building _n_1;17:17 in _p;18:18 a _q;19:19 Japanese _a_1;0:0 Nissan named', '0:1 named Nissan Motor;2:2 Co _n_1;4:4 Infiniti named;5:5 division _n_of;6:8 _v_1 likes insist;10:10 every _q;11:11 dealer _n_1;12:12 construct _v_1;13:13 and _c;14:14 furnish _v_to;15:15 a _q;16:16 building _n_1;17:17 in _p;18:18 a _q;19:19 Japanese _a_1;0:0 Nissan named', '0:0 Specifications _n_for;1:1 include _v_1;2:2 a _q;3:3 polished _v_1;4:4 bronze _a_1;5:5 sculpture _n_1;6:6 at _p;7:7 the _q;8:8 center _n_of;10:10 each _q;11:11 showroom _n_1;12:12 and _c;13:13 a _q;14:15 _n_1 tile bridge;16:16 spanning _v_1;17:17 a _q;18:18 stream _n_of;20:20 flows _v_1;21:21 into _p;22:22 the _q;23:23 building _n_1;24:25 _p from outside', '0:0 "Infiniti named;1:1 has _v_prd;3:4 _p down to;5:5 the _q;6:6 ashtrays," _n_unknown;7:7 says _v_to;8:9 named Jay Ferron;10:10 a _q;11:11 partner _n_1;12:12 at _p;13:14 named J.D Power;15:16 & Associates _n_1;17:17 an _q;18:19 _n_1 auto research;0:0 "Infiniti named', '0:1 named Toyota Motor;2:2 Corp _n_1;4:4 Lexus named;5:5 division _n_of;6:6 also _a_1;7:7 provides _v_1;0:0 Toyota named', '0:0 But _c;1:1 only _a_1;3:3 thirds _n_of;3:3 thirds card;4:4 of _p;5:5 Lexus named;6:6 dealers _n_1;8:8 constructing _v_1;9:9 new _a_1;10:10 buildings _n_1;11:12 according to _p;13:13 the _q;14:14 Lexus named;0:0 But _c', '0:0 Some _q;2:2 even _a_1;3:3 coming _v_up;5:5 with _p;7:8 _a_1 own novel;0:0 Some _q', '0:0 In _p_state;1:2 named Louisville Ky;5:6 named David Peterson;8:8 built _v_1;9:9 a _q;10:10 Lexus named;11:11 dealership _n_1;12:12 with _p;13:13 the _q;14:14 showroom _n_1;15:15 on _p;16:16 the _q;17:17 second ord;0:0 In _p_state', '0:0 Yet _c;1:1 some _q;2:2 dealers _n_1;4:4 turned _v_down;6:6 Infiniti named;7:7 or _c;8:8 Lexus named;9:9 franchises _n_1;10:10 because _x;13:13 unwilling _a_1;14:14 or _c;15:15 unable _a_1;17:17 meet _v_1;18:18 the _q;19:19 design _n_of;0:0 Yet _c', '0:1 named Lee Seidman;2:2 of _p;3:3 Cleveland named;4:4 says _v_to;5:5 Infiniti named;6:6 "was _v_id;7:7 a _q;8:8 bear _n_1;9:9 on _p;10:10 interiors" _n_1;11:11 but _c;12:13 at least _a_1;14:14 let _v_1;16:16 retrofit _v_unknown;17:17 an _q;18:18 existing _v_1;19:19 building _n_1;21:21 without _p;22:22 the _q;0:0 Lee named', '0:0 Mr _n_1;1:1 Seidman named;2:2 says _v_to;4:4 turned _v_down;6:6 a _q;7:7 Lexus named;8:8 franchise _n_1;9:9 in _p;10:10 part _n_1;11:11 because _x;12:12 "the _q;13:13 building _n_1;15:15 gorgeous _a_1;16:16 but _c;17:17 very _x_deg;0:0 Mr _n_1', '0:0 To _x;1:1 head _v_off;3:3 arguments _n_1;4:4 Infiniti named;5:5 offers _v_1;6:7 _n_1 dealers cash;8:8 bonuses _n_unknown;9:9 and _c;11:11 interest _n_in;11:11 interest _low_a_on;12:12 construction _n_of;0:0 To _x', '0:1 named Dictation Device;3:3 Saga named;4:4 Plays _v_1;5:5 Back _p;6:6 a _q;0:0 Dictation named', '0:0 PRODUCTS _n_1;2:2 have _v_qmodal;4:4 be _v_id;5:5 first ord;6:6 to _x;7:7 be _v_id;0:0 PRODUCTS _n_1', "0:0 That _q_dem;1:1 's _v_id;2:2 the _q;3:3 lesson _n_1;4:4 offered _v_1;5:5 through _p;6:6 one card;7:8 _n_of case study;9:9 featured _v_cause;10:10 in _p;11:11 a _q;12:12 design _n_of;0:0 That _q_dem", '0:0 Blocked _v_1;2:3 _n_of patent protection;4:4 from _p;5:5 following _v_1;6:6 suit _n_i;7:7 Dictaphone named;8:10 _v_1 decided go;11:11 a card;12:12 step _n_meas;13:13 further _a_1;14:14 and _c;15:15 cut _v_1;16:16 the _q;17:17 cassette _n_1;18:18 in _p;19:19 half _q;20:20 again _a_1;22:23 _p down to;24:24 the _q;25:25 length _n_of;27:27 a _q;0:0 Blocked _v_1', '0:0 By _p_temp;1:1 1979 yofc;2:2 designers _n_of;3:3 and _c;4:4 engineers _n_1;5:5 at _p;6:6 Dictaphone named;7:7 a _q;8:9 named Pitney Bowes;10:10 subsidiary _n_1;12:12 produced _v_1;13:13 a _q;14:14 working _v_1;15:15 model _n_of;17:17 a _q;18:18 "picocassette" _n_unknown;0:0 By _p_temp', '0:0 By _p_temp;1:1 1982 yofc;2:2 however _a_1;3:3 the _q;4:4 patent _n_of;5:5 status _n_of-as;7:7 the _q;8:8 Lanier named;9:9 microcassette _n_unknown;11:12 _v_1 changed permitting;13:13 Dictaphone named;15:15 develop _v_cause;17:18 _a_1 own competitive;19:19 micro _a_1;20:20 system _n_of;0:0 By _p_temp', '0:0 Marketing _v_1;1:1 and _c;2:2 sales _n_of;3:3 departments _n_1;4:4 then _c;5:5 urged _v_1;6:6 abandonment _n_1;7:7 of _p;8:8 the _q;9:9 pico _n_unknown;0:0 Marketing _v_1', '0:0 But _c;1:1 others _n_1;2:2 said _v_to;3:3 pico _n_unknown;4:4 should _v_modal;0:0 But _c', '0:0 Both _q;0:0 Both card;0:0 Both _q', '0:0 Dictaphone named;1:1 went _v_ahead;3:3 and _c;4:4 introduced _v_to;5:5 the _q;6:6 pico _n_unknown;7:7 in _p_temp;8:8 1985 yofc;9:9 but _c;12:12 sold _v_1;0:0 Dictaphone named', '0:1 To date _p;2:2 says _v_to;3:4 named Emil Jachmann;5:5 a _q;6:6 Dictaphone named;7:7 vice _n_1;8:8 president _n_of;11:11 "broken _v_even;13:13 or _c;14:14 shown _v_1;15:15 a _q;16:16 small _a_1;0:1 To date _p', '0:0 Nevertheless _a_1;1:1 the _q;2:2 device _n_1;5:5 successful _a_1;6:6 in _p;7:7 other _a_1;0:0 Nevertheless _a_1', '1:1 helped _v_to;2:2 Dictaphone named;3:3 attract _v_1;4:4 better _a_at-for-of;5:5 engineers _n_1;6:6 and _c;8:8 provided _v_1;9:9 new _a_1;10:10 technology _n_1;11:11 for _p;12:12 other _a_1;13:13 company _n_of;1:1 helped _v_to', '0:0 The _q;1:1 picocassette _n_unknown;2:2 recorder _n_of;3:3 also _a_1;4:4 helped _v_to;5:5 transform _v_into;6:6 the _q;7:7 company _n_of;9:9 reputation _n_1;10:10 from _p_time;11:11 follower _n_1;12:12 to _p_nbar;14:14 edge _n_of;14:14 edge _lead_v_1;0:0 The _q', '0:0 Dictaphone named;2:2 picocassette _n_unknown;3:3 recorder _n_of;4:4 is _v_id;5:7 card one 13;8:9 _n_of case studies;10:10 in _p;11:11 the _q;12:13 named TRIAD Design;14:14 Project named;15:15 sponsored _v_1;17:17 the _q;18:19 named Design Management;20:20 Institute named;21:21 of _p;22:22 Boston named;23:23 and _c;24:25 named Harvard Business;0:0 Dictaphone named', '0:0 The _q;1:1 studies _n_of;3:3 on _p;4:4 exhibit _n_1;5:5 at _p;6:6 Harvard named;7:7 this _q_dem;8:8 month _n_1;9:9 and _c;11:11 travel _v_1;12:12 to _p;13:13 Chicago named;15:15 Institute _n_of;17:17 Design named;18:18 and _c;19:19 the _q;20:20 University _n_of;22:22 California named;23:23 at _p;0:0 The _q', '0:0 A _q;1:3 _n_1 Rake Progress;4:4 Means _v_1;0:0 A _q', '0:0 ONE card;1:1 DAY _n_of;2:3 named Carl Barrett;4:4 of _p;5:6 named Mobile Ala;8:8 raking _v_1;9:9 some _q;10:11 _n_1 sycamore leaves;12:12 but _c;13:13 the _q;14:14 rake _n_1;15:15 kept _v_on;16:16 riding _v_1;17:17 up _p_dir;18:18 over _p;19:19 the _q;0:0 ONE card', '1:1 crude _a_1;2:2 device _n_1;3:3 worked _v_1;4:4 : _p_namely;5:5 The _q;6:6 lower _a_1;7:7 teeth _n_1;8:8 gathered _v_coll;9:9 the _q;10:10 leaves _n_1;11:11 into _p;12:12 a _q;13:13 pile _n_of;14:14 while _x;15:15 the _q;16:16 higher _a_1;17:17 harder _a_for;18:18 teeth _n_1;19:19 moved _v_cause;20:20 the _q;21:21 top _n_1;22:22 of _p;23:23 the _q;1:1 crude _a_1', '0:0 Now _a_1;1:1 incorporated _v_1;2:2 into _p;3:3 a _q;4:4 polypropylene _n_unknown;5:5 rake _n_1;6:6 the _q;8:8 inch _n_1;8:8 inch card;9:9 prongs _n_1;10:10 or _c;11:11 "wonderbars," _n_unknown;12:14 _a_1 also supposed;16:16 aid _v_1;17:17 in _p;18:18 picking _v_up;0:0 Now _a_1', '0:0 One card;1:1 customer _n_of;2:3 named Donald Blaggs;4:4 of _p;5:5 Mobile named;6:6 says _v_to;7:7 the _q;8:9 named Barrett Rake;10:13 _v_1 allowed do;15:15 lawn _n_1;16:16 in _p;17:17 2 card;18:18 1/2 fraction;19:19 hours _n_1;20:20 two card;21:21 hours _n_1;22:23 less than _x;0:0 One card', '0:0 But _c;1:1 other _a_1;2:2 rake _n_1;3:3 makers _n_of;4:4 have _v_1;0:0 But _c', '0:1 named Richard Mason;2:2 president _n_of;4:4 Ames named;5:5 Co _n_1;6:6 in _p;7:8 named Parkersburg W;9:9 Va named;10:10 says _v_to;11:11 the _q;12:12 Barrett named;13:13 rake _n_1;14:14 "makes _v_1;15:15 sense," _n_of;16:16 but _c;18:18 would _v_modal;20:20 "tough" _a_for;22:22 explain _v_to;0:0 Richard named', '0:1 named John Stoner;2:2 marketing _v_1;3:3 director _n_of;4:4 for _p;5:6 named True Temper;7:7 Corp _n_1;8:8 a _q;9:9 subsidiary _n_1;10:10 of _p;11:11 Black named;12:12 & _c;13:13 Decker named;14:14 says _v_to;15:15 people _n_of;17:17 want _v_1;19:19 move _v_cause;20:20 a _q;21:21 leaf _n_1;0:0 John named', '1:1 either _a_also;2:2 pick _v_up;6:6 says _v_to;7:7 "or _c;9:10 _v_1 start pulling;11:11 from _p;12:12 a _q;13:13 fresh _a_1;1:1 either _a_also', '0:0 Odds _n_1;1:1 and _c;0:0 Odds _n_1', '0:0 Four card;1:1 tubular _a_1;2:2 steel _n_1;3:3 "Bedfellows," _n_unknown;4:4 each _p;5:5 roughly _x_deg;6:6 in _p;7:7 the _q;8:8 shape _n_1;9:9 of _p;10:10 a _q;11:11 "W," named;13:13 attached _v_1;14:14 to _p;15:15 the _q;16:16 bottom _n_1;17:17 of _p;18:18 the _q;19:19 box _n_of;20:20 spring _n_1;21:21 in _p;22:22 a _q;23:23 recessed _v_1;0:0 Four card', '0:0 Nearly _x_deg;1:1 half _q;3:3 U.S named_n;4:4 consumers _n_of;5:5 say _v_to;8:8 pay _v_for;9:10 up to _x_deg;11:11 5 card;12:12 % _n_of;15:15 packaging _v_1;17:17 can _v_modal;19:19 recycled _v_1;20:20 or _c;22:22 biodegradable _a_unknown;23:24 according to _p;25:25 a _q;26:26 survey _n_1;27:27 commissioned _v_1;29:29 the _q;30:31 named Michael Peters;32:32 Group named;33:33 a _q;34:34 design _n_of;0:0 Nearly _x_deg', '0:0 The _q;1:1 Pentagon named;2:2 is _v_id;3:3 a _q;4:4 haunted _v_1;0:0 The _q', '0:0 Living _v_1;1:1 there _a_1;2:2 for _p;3:3 six card;4:4 years _n_1;6:6 really _x_deg;0:0 Living _v_1', '0:0 The _q;1:1 ghosts _n_1;2:2 of _p;3:3 the _q;4:4 past _n_1;5:5 are _v_id;7:7 : _p_namely;10:10 kept _v_prd;11:11 at _p;12:12 bay _n_1;13:13 only _x_deg;14:14 by _p_means;15:15 feeding _v_1;17:17 vast _a_1;18:18 quantities _n_of;21:22 _n_1 defense budget', '0:0 Some _q;1:1 can _v_modal;3:3 bought _v_off;5:5 relatively _x_deg;0:0 Some _q', '0:0 During _p;1:1 the _q;2:2 Korean _a_1;3:3 War named;4:4 Gen _n_1;5:6 named Douglas MacArthur;7:7 demanded _v_of;8:8 and _c;9:9 got _v_1;10:12 in addition to _p;14:14 U.N named_n;15:15 command _n_1;16:16 in _p;17:17 Korea named;19:20 _a_1 own naval;21:21 command _n_1;22:22 in _p;23:24 named Japan NavforJapan', '0:0 Those _q_dem;1:1 obsolete _a_1;2:2 operations _n_of;3:3 cost _v_1;6:6 $ _n_1;7:8 card 2 billion;9:9 a _p_per;10:10 year _n_1;11:11 and _c;12:12 keep _v_prd;13:13 Mac named;15:15 ghost _n_1;0:0 Those _q_dem', "0:0 That _q_dem;1:1 's _v_id;2:2 about _x_deg;3:3 all _q;5:5 costs _v_x;7:7 appease _v_unknown;8:8 Adm _n_1;9:10 named Erich Raeder;0:0 That _q_dem", '0:0 In _p_temp;1:1 1941 yofc;2:2 Raeder named;3:3 and _c;4:4 the _q;5:5 German _a_1;6:6 navy _n_1;7:9 _v_1 threatened attack;10:10 the _q;11:12 named Panama Canal;13:13 so _x;15:15 created _v_1;16:16 the _q;17:18 named Southern Command;19:19 in _p;0:0 In _p_temp', '0:0 The _q;1:2 named Southern Command;4:4 grown _v_to;5:5 even _x_deg;6:6 bigger _a_1;7:7 since _p;8:8 the _q;9:9 war _n_1;10:10 because _x;11:11 Raeder named;13:13 ghost _n_1;14:14 sometimes _a_1;15:15 runs _v_1;16:16 through _p;17:17 the _q;18:18 E named;19:19 ring _n_1;20:20 dressed _v_in;21:21 like _p;22:22 Gen _n_1;0:0 The _q', '1:1 far _a_1;1:1 far _a_1', "0:0 The _q;1:1 ghost _n_1;2:2 of _p;3:3 the _q;4:4 Soviet named;5:5 brigade _n_of;6:6 discovered _v_1;7:7 in _p;8:8 Cuba named;9:10 _p back in;11:11 the _q;12:12 '70s _n_unknown;13:13 costs _v_1;14:14 just _x_deg;17:18 card hundred million;19:19 : _p_namely;20:20 the _q;21:21 price _n_of;23:23 the _q;24:24 Caribbean named_n;25:25 Command named;26:26 in _p;27:28 named Key West;30:30 President _n_of;31:31 Carter named;32:32 created _v_1;33:33 in _p_temp;0:0 The _q", '0:0 The _q;1:1 brigade _n_of;4:4 heard _v_1;5:6 _p from since;7:7 but _c;9:9 keep _v_prd;10:10 the _q;11:11 staff _n_of;12:12 around _p;13:13 just _x_deg;14:14 in _p;0:0 The _q', '0:1 named George Marshall;3:3 ghost _n_1;5:5 much _x_deg;7:7 difficult _a_for;9:9 keep _v_prd;0:0 George named', '1:1 keep _v_unspec;2:2 a _q;3:3 lot _n_of;5:5 shrines _n_1;6:8 _p to around;9:9 the _q;10:10 Pentagon named;11:11 : _p_namely;12:13 _n_1 statues busts;14:14 relics _n_1;15:15 and _c;1:1 keep _v_unspec', '0:0 The _q;1:1 Army named;2:2 headquarters _n_1;3:3 on _p;4:4 the _q;5:5 third ord;6:6 deck _n_1;7:7 of _p;8:8 the _q;9:9 Pentagon named;10:10 used _v_qmodal;12:12 burn _v_cause;13:13 a _q;14:14 lot _n_of;16:16 incense _n_1;17:17 to _p;19:19 but _c;20:20 the _q;21:21 Navy named;22:22 headquarters _n_1;23:23 on _p;24:24 the _q;25:25 fourth ord;26:26 deck _n_1;27:27 made _v_cause;0:0 The _q', '1:1 see _v_1;2:2 Marshall named;3:3 had _v_1;4:4 this _q_dem;5:5 thing _n_of-about;7:7 the _q;8:8 Navy named;9:9 and _c;10:10 the _q;11:11 Marines named;14:16 _v_1 wanted make;18:20 _q part the;21:21 Army named;22:22 but _c;23:23 Secretary _n_of;25:25 the _q;26:27 named Navy James;28:28 Forrestal named;1:1 see _v_1', '0:0 Now _a_1;2:2 ghost _n_1;4:4 let _v_up;6:6 till _x_h;0:0 Now _a_1', '0:0 To _x;1:1 keep _v_prd;3:3 quiet _a_1;5:5 invent _v_1;6:6 a _q;7:7 new _a_1;8:8 unified _v_cause;9:9 command _n_1;10:10 every _q;11:11 year _n_1;14:14 run _v_1;16:16 the _q;17:17 Army named;18:18 or _c;19:19 the _q;20:21 named Air Force;22:22 and _c;23:23 put _v_1;26:26 the _q;27:27 Navy named;28:28 and _c;29:29 Marines named;0:0 To _x', '1:1 brings _v_up;3:3 the _q;4:4 worst _a_1;5:5 and _c;6:6 meanest _a_1;7:7 ghost _n_1;8:8 of _p;9:11 _q all the;12:12 ghost _n_1;13:13 of _p;14:14 the _q;15:15 shah _n_1;16:16 of _p;1:1 brings _v_up', '0:0 Mr _n_1;1:1 Carter named;2:2 said _v_to;4:4 would _v_modal;5:5 go _v_1;6:6 to _p;7:7 war _n_1;8:8 to _x;9:9 stop _v_from;10:10 anyone _q;12:14 _v_1 trying grab;0:0 Mr _n_1', "0:0 But _c;1:1 that _q_dem;2:2 ghost _n_1;3:3 wouldn't _v_modal;4:4 settle _v_1;5:5 for _p;6:6 words _n_of;8:8 wanted _v_1;9:9 money _n_1;10:10 and _c;11:11 people _n_of;0:0 But _c", '0:0 So _a_thus;1:1 Mr _n_1;2:2 Carter named;3:3 formed _v_cause;4:4 three card;5:5 new _a_1;6:6 Army named;7:7 divisions _n_of;8:8 and _c;9:9 gave _v_1;12:12 a _q;13:13 new _a_1;14:14 bureaucracy _n_1;15:15 in _p;16:16 Tampa named;17:17 called _v_name;18:18 the _q;19:20 named Rapid Deployment;0:0 So _a_thus', '0:0 But _c;1:1 that _q_dem;2:2 ghost _n_1;4:6 _v_1 fooled; knew;7:7 the _q;8:8 RDF named;9:9 was _v_id;10:10 neither _q;11:11 rapid _a_1;12:12 nor _c;13:13 deployable _a_unknown;14:14 nor _c;15:15 a _q;16:16 force _n_1;18:18 even _x_deg;19:19 though _x;21:21 cost _v_1;22:22 $ _n_1;23:24 card 8 billion;25:25 or _c;26:26 $ _n_1;27:28 card 10 billion;29:29 a _p_per;0:0 But _c', '0:0 After _x_h;1:1 Mr _n_1;2:2 Carter named;4:4 defeated _v_1;5:5 in _p_temp;6:6 1980 yofc;7:7 the _q;8:10 _n_1 shah ghost;11:11 claimed _v_1;12:12 the _q;13:13 credit _n_1;14:15 and then _c;16:16 went _v_1;17:17 after _p;18:18 President _n_of;19:19 Reagan named;20:20 and _c;21:22 named Cap Weinberger', '1:4 _v_1 saw did;5:5 to _p;1:1 saw _v_1', '1:1 made _v_cause;3:3 shoelaces _n_1;4:4 dance _v_1;5:5 with _p;1:1 made _v_cause', '1:1 bought _v_1;2:2 $ _n_1;3:4 card 4 billion;5:5 in _p;6:6 prepositioning _v_1;6:6 prepositioning _pre-_a_ante;7:7 ships _n_of;8:8 and _c;9:9 $ _n_1;10:11 card 7 billion;12:12 in _p;13:13 ammo _n_unknown;14:14 and _c;15:15 equipment _n_1;17:17 fill _v_cause;19:19 and _c;20:20 parked _v_1;22:22 at _p;23:23 a _q;24:24 new _a_1;25:25 $ _n_1;26:27 card 6 billion;28:28 base _n_1;29:29 at _p;30:31 named Diego Garcia;32:32 in _p;33:33 the _q;34:34 middle _n_of;36:36 the _q;37:37 Indian _a_1;1:1 bought _v_1', '1:1 dedicated _v_to;2:2 all _q;3:3 these _q_dem;4:4 new _a_1;5:5 forces _n_1;7:7 the _q;8:8 Persian _a_1;1:1 dedicated _v_to', '0:0 One card;1:1 night _n_of;3:3 Marshall named;5:5 ghost _n_1;6:6 and _c;7:7 the _q;8:10 _n_1 shah ghost;11:11 together _p;12:12 caught _v_1;13:13 Cap named;14:14 and _c;15:15 threw _v_1;17:17 to _p;18:18 the _q;0:0 One card', '0:0 Before _x_h;2:4 _v_1 let go;6:6 added _v_to;7:8 card a thousand;9:9 bureaucrats _n_unknown;11:11 the _q;12:12 RDF named;13:13 in _p;14:14 Tampa named;15:15 and _c;16:16 renamed _v_1;16:16 renamed _re-_a_again;18:19 named Central Command', '1:1 gave _v_1;2:2 those _q_dem;3:3 bureaucrats _n_unknown;4:4 charge _n_of;6:6 all _q;7:7 naval _a_1;8:8 operations _n_of;9:9 in _p;10:10 the _q;11:11 Persian _a_1;12:12 Gulf named;13:13 and _c;14:14 Indian _n_1;1:1 gave _v_1', '0:0 Marshall named;1:1 figured _v_1;3:3 would _v_modal;4:4 be _v_id;5:5 good _a_at-for-of;6:6 training _v_1;7:7 for _p;8:8 those _q_dem;9:9 soldiers _n_1;11:11 someday _q;12:12 maybe _a_1;14:14 would _v_modal;15:15 get _v_1;16:16 the _q;17:17 whole _a_1;0:0 Marshall named', '1:1 had _v_1;2:2 fun _n_1;3:3 moving _v_cause;4:4 the _q;5:5 carriers _n_of;6:6 around _p;7:7 but _c;9:9 turned _v_out;14:14 forgotten _v_1;15:15 all _q;16:16 about _p;17:17 mine _n_of;1:1 had _v_1', "1:1 'd _v_modal;2:2 get _v_state;4:4 asses _n_1;1:1 'd _v_modal", '0:0 So _a_thus;1:1 the _q;2:2 U.S named_n;3:3 found _v_1;5:5 paying _v_for;6:6 about _x_deg;7:7 $ _n_1;8:9 card 2 billion;10:10 in _p;11:11 baksheesh _n_unknown;13:14 _a_1 various Arab;15:15 potentates _n_unknown;17:17 basing _v_1;18:18 rights _n_1;19:19 around _p;20:20 the _q;21:21 Indian _a_1;0:0 So _a_thus', '1:1 had _v_1;2:2 great _a_for;3:3 success _n_1;4:4 in _p;1:1 had _v_1', '0:0 But _c;1:1 then _a_1;3:3 turned _v_out;6:6 President _n_of;7:8 named Siad Barrah;9:9 was _v_id;11:12 at all _a_1;13:13 a _q;14:14 nice _a_of-for;15:15 person _n_1;16:16 and _c;17:17 the _q;18:18 Navy named;19:19 pointed _v_out-to;22:22 the _q;23:23 base _n_1;25:25 promised _v_1;27:27 in _p;28:28 Berbera named;30:30 silted _v_up;32:32 about _x_deg;33:34 card a hundred;35:35 years _n_1;36:36 ago _p;37:37 and _c;38:38 anyway _a_1;39:39 was _v_id;40:40 1,244 card;41:41 miles _n_1;42:42 from _p;43:43 the _q;44:44 mouth _n_1;45:45 of _p;46:46 the _q;0:0 But _c', '0:0 (But _p_except;0:0 (But _p_except', '0:0 Still _a_disc;1:1 Berbera named;2:2 was _v_id;3:3 the _q;4:4 best _a_at-for-of;6:6 could _v_modal;7:7 get _v_1;8:8 so _x;10:10 stay _v_prd;11:11 in _p;12:12 bed _n_1;13:13 with _p;14:14 President _n_of;0:0 Still _a_disc', '0:0 All _q;1:1 these _q_dem;2:2 reports _n_of;3:3 about _p;5:5 committing _v_to;6:6 genocide _n_unknown;8:8 probably _a_1;9:9 exaggerated _v_1;0:0 All _q', '0:0 Meanwhile _p;1:1 Congress named;3:3 cutting _v_1;4:4 huge _a_1;5:5 chunks _n_of;6:7 out of _p;8:8 the _q;9:9 rest _n_of;11:11 the _q;12:13 _n_1 defense budget', "0:0 Couldn't _v_modal;2:2 save _v_1;3:3 $ _n_1;4:5 card 20 billion;6:6 or _c;7:7 $ _n_1;8:9 card 30 billion;10:10 a _p_per;11:11 year _n_1;12:12 by _p_means;13:13 shifting _v_1;14:14 that _q_dem;15:15 stuff _n_1;16:16 to _p;17:17 the _q;0:0 Couldn't _v_modal", '0:0 And _c;1:2 why not _x;3:3 save _v_1;4:4 the _q;5:5 costs _n_1;6:6 of _p;7:8 card a thousand;9:9 bureaucrats _n_unknown;10:10 by _p_means;11:11 abolishing _v_1;12:12 Central _a_1;13:13 Command _n_1;14:14 and _c;15:15 putting _v_1;16:16 responsibility _n_1;17:17 for _p;18:18 Gulf named;19:19 naval _a_1;20:20 operations _n_of;21:21 back _p;24:24 belongs _v_to;25:25 afloat _a_1;26:26 with _p;27:27 the _q;28:28 task _n_of;29:29 force _n_1;30:30 commander _n_of;31:31 in _p;32:32 the _q;0:0 And _c', '0:0 And _c;3:3 all _q;5:5 handsomely _a_unknown;6:6 paid _v_for;7:7 Indian _n_1;8:8 Ocean named;9:9 allies _n_1;10:10 last _a_1;11:11 year _n_1;12:12 when _x_subord;14:14 convoys _n_1;0:0 And _c', '0:0 Questions _n_about;1:1 like _p;2:2 that _q_dem;3:3 really _a_1;4:4 stir _v_up;6:6 Marshall named;0:0 Questions _n_about', '1:1 appeared _v_1;2:2 late _p;3:3 one card;4:4 night _n_of;5:5 in _p;6:6 the _q;7:7 bedroom _n_1;8:8 of _p;9:9 the _q;10:10 new _a_1;11:11 defense _n_1;12:12 secretary _n_of;13:14 named Dick Cheney', "1:1 wouldn't _v_modal;2:2 leave _v_1;3:3 until _x_h;4:4 Mr _n_1;5:5 Cheney named;6:8 _v_1 promised do;10:10 the _q;11:11 Pentagon named;12:12 systems _n_of;13:13 analysts _n_1;1:1 wouldn't _v_modal", '0:0 So _a_thus;1:1 next _a_1;2:2 day _n_of;3:3 Mr _n_1;4:4 Cheney named;5:5 went _v_1;6:6 out _p_dir;7:7 and _c;8:8 did _v_1;9:9 just _x_deg;10:10 that _q_dem;11:11 : _p_namely;13:13 canceled _v_1;14:14 the _q;15:15 600- yofc;16:16 ship _n_of;17:17 Navy _n_1;18:18 and _c;19:19 cut _v_back;21:21 one card;22:22 carrier _n_of;23:23 and _c;24:24 20 card;0:0 So _a_thus', '0:0 Then _a_1;2:2 canceled _v_1;3:3 production _n_of;5:5 the _q;6:6 Navy named;9:9 important _a_for;10:10 carrier _n_of;11:11 aircraft _n_1;12:12 the _q;13:13 F-14 named;14:14 and _c;15:15 the _q;0:0 Then _a_1', '0:0 On _p_state;1:1 the _q;2:2 other _a_1;3:4 _n_1 hand Mr;5:5 Cheney named;6:6 retained _v_1;7:7 all _q;8:8 those _q_dem;9:9 new _a_1;10:10 land _n_of;0:0 On _p_state', '0:0 Marshall named;2:2 ghost _n_1;4:4 satisfied _a_1;5:5 for _p;6:6 now _a_1;7:7 but _c;0:0 Marshall named', '0:1 What with _p;2:2 Halloween holiday;3:3 coming _v_1;4:4 and _c;5:5 bigger _a_1;6:7 _n_1 defense cuts;8:8 looming _v_1;10:10 and _c;12:12 Pentagon named;13:13 bureaucrats _n_unknown;15:15 crawling _v_dir;16:16 under _p_dir;0:1 What with _p', '0:0 Then _a_1;1:1 the _q;2:2 whole _a_1;3:3 thing _n_of-about;5:7 _v_1 start collapse;8:8 just _x_deg;9:9 as _x_subord;12:12 in _p;13:13 the _q;14:14 1970s year_range;15:15 and _c;16:16 the _q;17:17 ghosts _n_1;18:18 and _c;19:19 banshees _n_unknown;22:22 howling _v_1;23:23 through _p;24:24 the _q;25:25 place _n_of;26:26 turning _v_1;27:27 people _n_of;29:29 hair _n_1;0:0 Then _a_1', '0:0 Gives _v_1;2:2 the _q;3:3 willies _n_1;4:4 just _a_1;0:0 Gives _v_1', '0:0 Mr _n_1;1:1 Lehman named;2:2 a _q;3:4 named Reagan Navy;5:5 secretary _n_of;6:6 is _v_id;7:7 a _q;8:8 managing _v_1;9:9 director _n_of;0:0 Mr _n_1', '0:0 The _q;1:1 metal _n_1;2:2 and _c;3:4 _n_1 marble lobby;5:5 of _p;6:7 named CenTrust Bank;9:9 headquarters _n_1;11:14 _a_1 grander average;15:15 savings _n_1;16:16 and _c;0:0 The _q', '0:0 For _p;1:1 one card;2:2 thing _n_of-about;4:4 is _v_there;5:5 an _q;6:6 old _a_1;7:7 master _n_1;8:8 on _p;9:9 the _q;10:10 wall _n_of;12:13 named "Samuel Anointing;14:14 David," named;15:15 a _q;16:17 _a_1 big baroque;18:18 canvas _n_1;19:19 painted _v_1;21:22 named Mattia Preti;23:23 a _q;24:24 17th- _n_unknown;25:25 century _n_1;0:0 For _p', '0:0 At _p_temp;1:1 the _q;2:2 moment _n_1;4:4 the _q;5:5 painting _n_of;6:6 is _v_id;7:7 a _q;8:8 nagging _v_1;9:9 reminder _n_1;10:10 of _p;11:11 the _q;12:12 problems _n_of;15:15 engulfed _v_unknown;16:16 CenTrust named;17:17 and _c;19:19 flamboyant _a_unknown;20:20 chairman _n_of;21:21 and _c;22:23 _n_1 chief executive;24:25 named David L;0:0 At _p_temp', '0:0 In _p_state;1:1 an _q;2:2 international _a_1;3:3 buying _v_1;4:4 spree _n_unknown;6:6 began _v_1;7:7 barely _a_1;8:8 two card;9:9 years _n_1;10:10 ago _p;11:11 Mr _n_1;12:12 Paul named;13:13 amassed _v_1;14:14 a _q;15:15 collection _n_of;17:17 about _x_deg;18:18 30 card;20:20 18th- _n_unknown;21:21 century _n_1;21:21 century _pre-_a_ante;22:22 works _n_1;23:23 including _v_1;24:24 the _q;25:25 Preti named;26:26 at _p;27:27 a _q;28:28 total _a_1;29:29 cost _n_1;30:30 of _p;31:31 $ _n_1;32:33 card 28 million', '0:0 By _p_temp;1:1 midnight numbered_hour;2:2 Oct mofy;3:3 6 dofm;4:6 _q all the;7:7 paintings _n_of;9:9 supposed _a_1;13:13 sold _v_off;15:15 under _p;16:16 orders _n_from;18:18 Florida named;20:20 comptroller _n_unknown;22:22 office _n_of;23:23 regulates _v_1;24:24 the _q;25:25 state _n_of;0:0 By _p_temp', '0:0 CenTrust named;2:2 meet _v_1;3:3 the _q;0:0 CenTrust named', '0:0 The _q;1:1 collection _n_of;3:3 at _p;4:4 the _q;5:5 heart _n_1;6:6 of _p;7:7 a _q;8:8 grandiose _a_1;9:10 _n_1 plan Mr;11:11 Paul named;12:12 had _v_1;13:13 in _p;15:15 the _q;16:16 art _n_1;18:18 to _x;19:19 do _v_1;20:20 double _a_1;21:21 duty _n_1;23:23 as _p;24:24 an _q;25:25 investment _n_1;26:26 for _p;27:27 CenTrust named;28:28 and _c;29:29 as _p;30:30 decoration _n_of;31:31 for _p;32:32 the _q;33:33 S&L _n_unknown;35:35 new _a_1;36:37 _n_of office tower;38:38 designed _v_1;40:41 named I.M Pei', '0:0 Mr _n_1;1:1 Paul named;2:2 had _v_1;3:3 no _q;4:4 right _n_1;6:6 buy _v_1;7:7 art _n_1;8:8 for _p;9:9 the _q;10:10 S&L _n_unknown;11:11 in _p;12:12 the _q;13:13 first ord;14:14 place _n_of;18:18 on _p;19:19 the _q;20:20 comptroller _n_unknown;22:22 "permissible" _a_1;23:23 list _n_of;25:25 without _p;26:26 seeking _v_1;27:27 a _q;28:28 special _a_1;29:29 dispensation _n_1;0:0 Mr _n_1', '0:0 Besides _p;1:1 that _q_dem;2:4 _q some the;5:5 paintings _n_of;9:9 grace _v_1;10:10 the _q;11:11 walls _n_of;13:13 CenTrust named;14:14 actually _a_1;15:15 ended _v_up;17:17 hanging _v_1;18:18 in _p;19:19 the _q;20:20 chairman _n_of;22:22 estate _n_1;23:23 on _p;24:25 named La Gorce;26:26 Isle named;27:27 off _p;28:29 named Miami Beach', '0:0 Last _a_1;1:1 spring season;2:2 the _q;3:3 comptroller _n_unknown;5:5 office _n_of;6:6 called _v_1;7:7 a _q;8:8 halt _n_1;9:9 to _p;10:10 Mr _n_1;11:11 Paul named;13:13 fling _n_1;14:14 giving _v_1;16:16 six card;17:17 months _n_1;18:18 to _x;19:19 sell _v_1;20:20 the _q;0:0 Last _a_1', '0:0 The _q;1:1 comptroller _n_unknown;3:3 office _n_of;4:4 says _v_to;7:7 "monitoring _v_1;8:8 the _q;0:0 The _q', '0:0 Though _x;1:1 the _q;2:2 agency _n_1;3:3 could _v_modal;4:4 remove _v_1;5:5 Mr _n_1;6:6 Paul named;8:8 has _v_1;9:9 no _q;10:10 current _a_1;11:11 intention _n_1;13:13 do _v_1;0:0 Though _x', '3:3 like _p;4:4 selling _v_1;5:5 Chevrolets," named;6:6 Mr _n_1;7:7 Paul named;8:8 says _v_to;9:9 as _x_subord;11:11 takes _v_1;12:12 a _q;13:13 drag _n_1;14:14 on _p;15:15 a _q;16:16 goldbanded _v_unknown;17:17 St _n_1;18:18 Moritz named;3:3 like _p', '0:0 "The _q;1:1 last _a_1;2:2 six card;3:3 months _n_1;5:5 established _v_1;6:6 the _q;7:7 quality _n_1;8:8 of _p;9:9 the _q;0:0 "The _q', "1:1 's _v_there;2:2 no _q;3:3 fire _n_1;4:4 sale _n_of;1:1 's _v_there", '0:0 Despite _p;1:1 Mr _n_1;2:2 Paul named;4:4 characteristic _a_of;5:5 hauteur _n_unknown;6:6 the _q;8:8 year- _n_1;8:8 year- card;9:9 old _a_1;10:10 chain- _n_of;11:11 smoking _v_1;12:12 dynamo _n_unknown;14:14 finding _v_1;16:16 getting _v_state;17:19 named CenTrust Florida;21:21 largest _a_1;22:22 thrift _n_unknown;23:23 institution _n_1;25:26 out of _p;28:28 riskiest _a_for;29:29 investments _n_1;31:31 much _x_deg;32:32 tougher _a_for;34:34 getting _v_state;0:0 Despite _p', '0:0 Paintings _n_of;1:1 are _v_id;2:2 just _x_deg;3:5 _q part the;0:0 Paintings _n_of', '0:0 Although _x;1:1 Mr _n_1;2:2 Paul named;4:4 pared _v_1;5:5 a _q;6:6 $ _n_1;7:8 card 1.35 billion;9:10 _n_1 junk- bond;11:11 portfolio _n_1;12:12 to _p;15:15 $ _n_1;16:17 card 900 million;18:18 since _p;19:19 April mofy;20:20 the _q;22:22 yield _n_1;22:22 yield _high_a_1;23:24 _n_1 debt market;0:0 Although _x', '0:0 And _c;1:1 CenTrust named;2:2 has _v_1;3:3 other _a_1;0:0 And _c', '0:0 Late _p;1:1 last _a_1;2:2 week _n_1;3:3 federal _a_1;4:4 regulators _n_1;5:5 ordered _v_1;6:6 the _q;7:7 thrift _n_unknown;8:8 institution _n_1;10:10 stop _v_prd;11:11 paying _v_for;12:12 dividends _n_1;13:13 on _p;15:15 preferred _v_to;16:16 stock _n_1;18:18 a _q;19:19 move _n_1;21:21 suggests _v_to;22:22 deep _a_1;23:23 concern _n_1;24:24 about _p;25:25 an _q;0:0 Late _p', '0:0 Mr _n_1;1:1 Paul named;2:2 has _v_1;3:3 a _q;4:4 plan _n_1;6:6 bring _v_in;8:8 $ _n_1;9:10 card 150 million;11:11 by _p_means;12:12 selling _v_off;14:14 63 card;16:16 CenTrust named;18:18 71 card;19:19 branches _n_of;20:20 but _c;22:22 has _v_yet;26:26 approved _v_1;0:0 Mr _n_1', '1:1 is _v_itcleft;2:2 Mr _n_1;3:3 Paul named;5:6 _n_1 art venture;7:7 however _a_1;10:10 drawn _v_1;11:12 the most _q;13:13 attention _n_to;14:14 from _p;15:15 investors _n_1;16:16 and _c;17:17 regulators _n_1;18:20 not to mention _c;21:21 galleries _n_of;22:22 throughout _p;23:23 the _q;1:1 is _v_itcleft', '0:0 Embittered _a_unknown;1:1 shareholders _n_1;2:2 (some _q;6:6 suing) _v_1;7:7 say _v_to;8:8 the _q;9:9 chairman _n_of;10:10 and _c;12:12 collection _n_of;13:13 epitomize _v_unknown;14:14 the _q;15:15 excesses _n_1;16:16 of _p;17:17 speculation _n_about;19:19 set _v_off;21:21 the _q;22:22 national _a_1;23:23 S&L _n_unknown;0:0 Embittered _a_unknown', '0:0 (CenTrust named;1:1 shares _n_of;3:3 fallen _v_1;4:4 sharply _a_1;5:5 in _p;6:6 price _n_of;7:7 from _p;8:8 a _q;9:9 high _n_1;10:10 of _p;11:11 $ _n_1;12:12 15.125 card;13:13 in _p_temp;14:14 1986 yofc;15:15 to _x;16:16 close _v_1;17:17 yesterday _a_1;18:18 at _p;19:19 $ _n_1;0:0 (CenTrust named', '0:1 _n_of Gallery directors;2:2 meanwhile _p;3:3 say _v_to;4:4 Mr _n_1;5:5 Paul named;6:6 and _c;7:7 others _n_of;10:10 ilk _n_unknown;12:12 left _v_1;13:13 an _q;14:14 indelible _a_unknown;15:15 mark _n_of;16:16 on _p;17:17 the _q;18:18 art _n_1;19:19 world _n_of;21:21 and _c;23:23 for _p;24:24 the _q;0:0 Gallery _n_of', '0:0 Collectors _n_1;2:2 say _v_to;4:4 \'s _v_id;5:5 a _q;6:7 named van Gogh";8:8 anymore _a_1;9:9 laments _v_to;10:11 named Harry Brooks;12:12 the _q;13:13 president _n_of;15:15 Wildenstein named;16:17 & Co _n_1;18:18 a _q;19:20 named New York;0:0 Collectors _n_1', "1:1 say _v_to;2:3 named `Johnny Payson;4:4 got _v_1;5:5 $ _n_1;6:7 card 53 million;8:8 for _p;10:10 so _x;11:11 certainly _a_1;12:12 $ _n_1;13:14 card 10 million;15:15 isn't _v_id;1:1 say _v_to", '0:0 The _q;1:1 great _a_for;2:2 collectors _n_1;4:4 depended _v_on-upon;6:7 such as _p;8:9 named Paul Mellon;10:10 or _c;11:12 named Norton Simon;14:14 stopped _v_prd;15:15 buying _v_1;16:16 and _c;17:17 the _q;18:18 new _a_1;19:19 buyers _n_of;20:20 are _v_id;21:21 brilliant _a_1;22:22 men _n_1;24:24 made _v_1;25:25 money _n_1;26:26 in _p;27:27 the _q;28:29 _n_1 stock market;30:30 or _c;31:31 in _p;32:32 takeovers _n_of;33:33 and _c;34:34 rushed _v_1;35:35 into _p;0:0 The _q', '0:0 Mr _n_1;1:1 Payson named;2:2 an _q;3:4 _n_1 art dealer;5:5 and _c;6:6 collector _n_1;7:7 sold _v_1;8:9 named Vincent van;10:12 named Gogh "Irises";13:13 at _p;14:14 a _q;17:17 auction _n_of;18:18 in _p_temp;19:19 November mofy;20:20 1987 yofc;22:22 Australian _a_1;23:23 businessman _n_1;24:25 named Alan Bond', '0:0 When _x_subord;1:1 Mr _n_1;2:2 Paul named;3:3 moved _v_in;5:5 on _p;6:6 the _q;7:8 _n_1 art market;10:13 _v_1 let known;15:15 virtually _x_deg;16:16 no _q;17:17 piece _n_of;20:20 costly _a_1;23:23 considered _v_1;24:24 by _p;0:0 When _x_subord', '1:1 established _v_as;3:3 reputation _n_1;5:5 a _q;6:6 freespender _n_unknown;7:7 in _p_temp;8:8 January mofy;9:9 last _a_1;10:10 year _n_1;11:11 at _p;12:12 Sotheby named;14:14 auction _n_of;16:16 the _q;17:17 Linda named;18:18 and _c;19:20 named Gerald Guterman;21:21 collection _n_of;22:22 in _p;23:24 named New York', '0:0 There _a_1;1:1 on _p;2:2 one card;5:5 first ord;6:6 shopping _v_buy;7:7 trips _n_of;8:8 Mr _n_1;9:9 Paul named;10:10 picked _v_up;12:12 several _a_1;13:13 paintings _n_of;14:14 at _p;15:15 stunning _v_1;0:0 There _a_1', '0:0 The _q;1:1 price _n_of;2:2 paid _v_for;3:3 was _v_id;4:4 a _q;5:5 record _n_of;6:6 for _p;7:7 the _q;0:0 The _q', '0:0 (Some _q;1:1 64 card;2:4 _n_of % items;5:5 offered _v_1;6:6 at _p;7:7 the _q;8:8 Guterman named;9:9 auction _n_of;11:11 sold _v_1;12:12 at _p;13:13 an _q;14:14 average _a_1;15:15 price _n_of;17:17 $ _n_1;0:0 (Some _q', '0:0 The _q;1:1 rest _n_of;3:3 withdrawn _v_1;4:4 for _p;5:5 lack _n_1;6:6 of _p;7:7 acceptable _a_for;0:0 The _q', '0:0 Afterward _p;1:1 Mr _n_1;2:2 Paul named;4:4 said _v_1;6:6 Mr _n_1;7:7 Guterman named;10:10 phoned _v_1;11:11 Mr _n_1;12:12 Guterman named;13:13 the _q;14:15 named New York;16:16 developer _n_of;17:17 selling _v_1;18:18 the _q;19:19 collection _n_of;20:20 and _c;0:0 Afterward _p', '0:0 Mr _n_1;1:1 Paul named;2:2 denies _v_to;3:3 phoning _v_1;4:4 and _c;0:0 Mr _n_1', '2:2 just _x_deg;4:4 true," _a_of;2:2 just _x_deg', '0:0 Mr _n_1;1:1 Paul named;2:2 quickly _a_1;3:3 became _v_id;5:5 aggressive _a_1;6:6 in _p;8:8 collecting _v_1;9:9 with _p;10:10 the _q;11:11 help _n_1;12:12 of _p;13:14 named George Wachter;15:15 a _q;16:16 Sotheby named;18:18 expert _n_1;19:19 in _p;20:20 old _a_1;21:21 masters _n_1;24:24 met _v_1;25:25 at _p;26:26 an _q;27:27 exhibition _n_of;29:29 the _q;30:30 Guterman named;0:0 Mr _n_1', '0:0 Mr _n_1;1:1 Wachter named;3:3 became _v_id;5:5 principal _a_1;6:6 adviser _n_1;7:7 searched _v_1;8:8 galleries _n_of;9:9 in _p;10:11 named London Paris;12:12 and _c;0:0 Mr _n_1', '0:0 And _c;1:2 according to _p;3:3 one card;4:5 _n_1 dealer Mr;6:6 Wachter named;7:7 had _v_1;8:8 a _q;9:9 penchant _n_unknown;10:10 for _p;11:11 introducing _v_to;12:12 Mr _n_1;13:13 Paul named;14:14 with _p;15:15 the _q;16:16 phrase _n_1;17:17 : _p_namely;19:19 can _v_modal;20:20 buy _v_1;0:0 And _c', '0:1 named Nicholas Hall;2:2 the _q;3:3 president _n_of;5:5 the _q;6:6 Colnaghi named;7:7 U.S.A named_n;8:8 Ltd _n_1;9:9 gallery _n_of;10:10 in _p;11:12 named New York;13:13 sold _v_1;14:14 Mr _n_1;15:16 named Paul "Abraham;17:17 and _c;18:18 Sarah named;19:19 in _p;20:20 the _q;21:21 Wilderness" named;22:22 by _p;23:24 named Giovanni Battista;0:0 Nicholas named', '0:0 Mr _n_1;1:1 Hall named;2:2 says _v_to;3:3 Mr _n_1;4:4 Paul named;6:8 _v_1 known spend;9:9 a _q;10:10 lot _n_of;0:0 Mr _n_1', '0:0 People _n_of;2:2 interested _a_in;4:4 seeing _v_1;6:6 but _c;9:9 recognized _v_1;11:11 the _q;12:12 route _n_1;14:14 through _p;17:17 and _c;18:18 particularly _x_deg;19:20 named George Wachter."', '0:0 Mr _n_1;1:1 Paul named;2:2 thus _a_1;3:3 developed _v_cause;4:4 a _q;5:5 close _a_to;6:6 symbiotic _a_1;7:7 relationship _n_of;0:0 Mr _n_1', '0:0 Mr _n_1;1:1 Paul named;3:3 eager _a_1;5:5 assemble _v_cause;6:6 a _q;7:7 collection _n_of;8:8 for _p;9:9 the _q;10:10 headquarters _n_1;11:11 CenTrust named;14:14 moving _v_1;15:16 _p into for;17:17 the _q;18:18 greater _a_for;19:19 part _n_1;20:20 of _p;21:21 a card;0:0 Mr _n_1', '0:0 European _a_1;1:1 dealers _n_1;2:2 continued _v_2;4:4 dominate _v_1;5:5 the _q;6:6 action _n_1;7:7 in _p;8:8 old _a_1;9:9 masters _n_1;13:14 named North America;16:16 lately _a;18:18 touting _v_to;19:19 in _p;20:20 this _q_dem;0:0 European _a_1', '0:0 For _p;1:1 several _a_1;2:2 months _n_1;4:4 was _v_there;5:5 optimism _n_1;6:6 all _x_deg;0:0 For _p', '0:0 Last _a_1;1:1 October mofy;2:2 Mr _n_1;3:3 Paul named;4:4 paid _v_out;6:6 $ _n_1;7:8 card 12 million;9:9 of _p;10:10 CenTrust named;12:12 cash _n_1;14:14 plus _c;15:15 a _q;16:16 $ _n_1;17:18 card 1.2 million;19:19 commission _n_of;21:21 for _p;22:22 "Portrait named;23:23 of _p;24:24 a _q;25:25 Man _n_1;26:26 as _p;0:0 Last _a_1', '0:0 The _q;1:1 painting _n_of;2:2 attributed _v_to;4:4 Flemish _a_1;5:5 artist _n_1;6:7 named Peter Paul;8:8 Rubens named;10:10 purchased _v_1;11:11 privately _a_1;12:16 _p through at;0:0 The _q', '0:0 In _p_temp;1:1 March mofy;2:2 1989 yofc;3:3 just _x_deg;4:4 15 card;5:5 months _n_1;6:6 into _p;8:9 _n_1 campaign Mr;10:10 Paul named;12:12 named _v_1;13:13 by _p_means;14:14 Art named;15:15 & _c;16:16 Antiques named;17:17 magazine _n_1;18:18 as _p;19:19 one card;21:21 the _q;22:22 top _a_1;23:23 100 card;24:24 individual _a_1;25:25 collectors _n_1;26:26 in _p;27:27 the _q;0:0 In _p_temp', '0:0 "An _q;1:1 unknown _a_1;2:2 quantity _n_of;3:3 to _p;4:6 _q most the;7:7 art _n_1;8:8 world _n_of;9:9 Paul named;10:10 is _v_id;11:11 no _q;12:12 stranger _n_1;13:13 to _p;14:14 lavish _a_1;15:15 spending," _v_1;16:16 the _q;17:17 magazine _n_1;18:19 _v_to said noting;23:23 stop _v_1;24:24 at _p;25:25 paint _n_1;26:26 on _p;27:27 canvas _n_1;28:29 but also _c;30:30 spends _v_1;31:31 big _a_1;32:32 on _p;33:33 art _n_1;35:35 can _v_modal;0:0 "An _q', '1:1 recently _a_1;2:2 bid _v_1;3:3 $ _n_1;4:4 30,000 card;5:5 at _p;6:6 a _q;7:7 Paris named;8:8 charity _n_1;9:9 auction _n_of;10:10 for _p;11:11 a _q;12:12 dinner _n_1;13:13 cooked _v_1;15:15 six card;17:17 the _q;18:18 world _n_of;20:20 great _a_for;21:21 chefs _n_1;22:22 but _c;23:23 the _q;24:24 final _a_1;25:25 party _n_of-to;26:26 cost _v_1;27:27 closer _a_to;28:28 to _p;29:29 $ _n_1;1:1 recently _a_1', '0:0 (Mr _n_1;1:1 Paul named;2:2 says _v_to;5:5 that _x_deg;0:0 (Mr _n_1', '0:0 The _q;1:1 art _n_1;2:2 collection _n_of;3:3 might _v_modal;5:7 _v_1 come rival;8:8 the _q;11:11 had _x_then;12:12 the _q;13:13 Florida named;14:14 comptroller _n_unknown;16:16 office _n_of;18:18 got _v_1;19:19 wind _n_1;20:20 of _p;21:21 Mr _n_1;22:22 Paul named;24:24 aesthetic _a_1;0:0 The _q', '0:0 In _p_state;2:2 letter _n_of;3:3 to _p;5:5 dated _v_1;6:6 March mofy;7:7 2 dofm;8:8 and _c;9:9 shared _v_1;10:10 with _p;11:11 reporters _n_of;12:13 named Alex Hager;14:14 the _q;15:15 chief _n_1;16:16 of _p;17:17 the _q;18:18 thrift- _n_unknown;19:19 institution _n_1;20:20 bureau _n_of;21:21 in _p;22:22 the _q;23:23 comptroller _n_unknown;25:25 office _n_of;26:26 expressed _v_to;27:27 puzzlement _n_1;29:29 the _q;30:30 S&L _n_unknown;31:31 could _v_modal;34:34 profligate _a_unknown;35:35 when _x_subord;38:38 reported _v_to;39:39 losses _n_of;43:43 $ _n_1;44:45 card 13 million;46:46 in _p_temp;48:48 two card;49:49 preceding _v_1;0:0 In _p_state', '0:0 The _q;1:1 state _n_of;2:2 gave _v_1;3:3 CenTrust named;4:4 30 card;5:5 days _n_of;6:6 to _x;7:7 sell _v_1;8:8 the _q;0:0 The _q', '0:0 The _q;1:1 comptroller _n_unknown;3:3 office _n_of;4:4 eventually _a_1;5:5 extended _v_1;6:6 the _q;7:7 deadline _n_of;8:8 to _p;9:9 six card;10:10 months _n_1;11:11 but _c;12:12 broadened _v_cause;14:14 demands _n_1;15:15 ordering _v_1;17:17 the _q;18:18 "book _n_of;19:19 value _n_1;20:20 of _p;21:21 the _q;22:22 collection _n_of;24:24 reduced _v_1;25:25 to _p;0:0 The _q', '0:0 In _p_state;1:1 other _a_1;2:2 words _n_of;3:3 : _p_namely;4:4 Get _v_state;5:5 rid _v_of;7:8 _q all the;0:0 In _p_state', '0:0 The _q;1:1 state _n_of;2:2 obliquely _a_1;3:3 noted _v_to;5:5 unsafe _a_1;6:6 banking _n_1;7:7 practices _n_of;8:8 are _v_id;9:9 grounds _n_1;10:10 for _p;11:11 removing _v_1;12:12 an _q;13:13 officer _n_1;14:14 or _c;15:15 director _n_of;16:16 and _c;17:17 closed _v_1;18:18 with _p;19:19 the _q;20:20 admonition _n_1;21:21 to _p;22:22 Mr _n_1;23:23 Paul named;24:24 : _p_namely;25:25 "Govern _v_1;0:0 The _q', '0:0 Mr _n_1;1:1 Paul named;2:2 says _v_to;4:4 at _p;5:5 one card;6:6 point _n_of;9:9 indeed _a_1;10:10 have _v_1;11:11 eight card;12:12 or _c;13:13 nine card;15:15 the _q;16:16 paintings _n_of;17:18 _p at home;19:19 and _c;21:21 the _q;22:22 rest _n_of;24:24 in _p;25:25 storage _n_of;0:0 Mr _n_1', '1:1 explains _v_to;5:5 "merely _a_1;6:6 storing _v_cause;7:7 the _q;8:8 paintings _n_of;9:10 _p at home;12:12 with _p;13:13 some _q;14:14 display _n_of;16:17 because of _p;18:18 the _q;19:19 special _a_1;20:20 dehumidified _a_unknown;21:21 environment" _n_1;22:22 required _v_of;23:23 for _p;25:25 safekeeping _n_unknown;26:26 until _x_h;27:27 CenTrust named;29:29 new _a_1;30:30 building _n_1;32:32 ready _a_1;1:1 explains _v_to', '0:0 Still _a_disc;1:1 the _q;2:2 incident _n_1;0:0 Still _a_disc', '1:1 came _v_1;2:2 on _p;3:3 the _q;4:4 heels _n_1;5:5 of _p;6:6 a _q;7:7 number _n_of;9:9 local _a_1;10:11 _n_of newspaper articles;12:12 suggesting _v_to;14:14 Mr _n_1;15:15 Paul named;17:17 benefited _v_from;18:18 handsomely _a_unknown;21:21 association _n_of;22:22 with _p;1:1 came _v_1', '1:1 owns _v_1;2:2 43 card;3:3 % _n_of;4:4 of _p;5:5 CenTrust named;1:1 owns _v_1', '0:0 The _q;1:1 $ _n_1;2:3 card 12 million;4:4 lavished _v_on;6:6 the _q;7:7 Rubens named;10:10 was _v_id;11:11 a _q;12:13 _n_of record price;14:14 for _p;15:15 the _q;16:16 artist _n_1;17:17 and _c;18:18 maybe _x_deg;19:19 twice _q;21:21 value _n_1;22:22 given _v_1;23:23 a _q;24:24 dispute _n_1;25:25 among _p;26:26 scholars _n_1;27:27 about _p;0:0 The _q', '0:1 named David Tunick;2:2 the _q;3:3 president _n_of;5:6 named David Tunick;7:7 Inc _n_1;8:8 a _q;9:10 named New York;11:11 gallery _n_of;12:12 says _v_to;13:13 scholars _n_1;14:14 question _v_1;15:15 the _q;16:16 authenticity _n_1;17:17 of _p;18:18 the _q;0:0 David named', '1:1 may _v_modal;4:4 painted _v_1;5:6 _p instead by;7:7 a _q;8:8 Rubens named;1:1 may _v_modal', '0:0 "The _q;1:1 feeling _v_1;2:2 among _p;4:4 experts _n_1;5:5 on _p;6:6 the _q;7:7 commercial _a_1;8:8 side _n_1;9:9 is _v_nv;11:11 the _q;12:12 price _n_of;13:13 paid _v_for;14:14 at _p_temp;15:15 the _q;16:16 time _n_of;18:18 excessive _a_1;19:19 in _p;20:20 any _q;21:21 event," _n_item;22:22 Mr _n_1;23:23 Tunick named;0:0 "The _q', '1:1 sounds _v_seem-to;3:3 with _p;4:4 the _q;5:5 Rubens named;7:7 got _v_state;8:8 absolutely _a_1;9:9 taken _v_1;10:10 to _p;11:11 the _q;1:1 sounds _v_seem-to', '1:1 \'s _v_itcleft;3:3 beyond _p;4:4 credibility _n_1;5:5 the _q;6:6 Rubens named;8:8 someday _q;10:10 worth _a_1;11:11 $ _n_1;12:13 card 12 million;14:14 but _c;17:17 could _v_modal;19:19 sold _v_1;20:20 for _p;21:21 that _q_dem;22:22 amount _n_of;23:23 tomorrow _a_1;24:27 _v_1 remains seen."', '0:0 Still _a_disc;1:1 predicting _v_1;0:0 Still _a_disc', '2:2 forever _a_1;3:3 dumbfounded _a_unknown;4:4 by _p;7:8 _v_1 see making;9:9 these _q_dem;10:10 high _a_1;2:2 forever _a_1', '0:1 named Jonathan H;2:2 Kress named;3:3 the _q;4:4 son _n_of;6:6 the _q;7:7 painting _n_of;9:9 former _a_1;10:10 owner _n_of;11:11 Mrs _n_1;12:13 named Rush Kress;14:14 dismisses _v_as;15:15 the _q;16:16 price _n_of;17:17 talk _n_of-on;19:19 "sour _a_1;0:0 Jonathan named', '0:0 Mr _n_1;1:1 Paul named;2:2 for _p;4:4 part _n_1;5:5 defends _v_from;6:6 the _q;7:7 Rubens named;8:8 price _n_of;9:9 saying _v_to;10:10 a _q;11:11 lot _n_of;13:13 the _q;14:14 experts _n_1;16:16 never _a_1;17:17 seen _v_1;18:18 the _q;0:0 Mr _n_1', '0:0 "Most _q;4:4 even _a_1;5:5 born _v_2;6:6 the _q;7:7 last _a_1;8:8 time _n_of;9:9 the _q;10:10 painting _n_of;12:12 displayed _v_to;13:13 publicly," _a_1;0:0 "Most _q', '0:0 Art _n_1;1:1 prices _n_of;3:3 skyrocketing _v_unknown;4:4 but _c;5:5 a _q;6:6 good _a_at-for-of;7:7 deal _n_1;8:8 of _p;9:9 legerdemain _n_unknown;11:11 involved _v_in;13:13 compiling _v_1;14:14 statistics _n_1;15:15 on _p;0:0 Art _n_1', '0:1 named Salomon Brothers;2:2 Inc _n_1;3:3 the _q;4:5 _n_1 investment- banking;6:6 firm _n_1;7:7 in _p;9:9 annual _a_1;10:10 tally _n_1;11:11 of _p;12:12 investment _n_1;13:13 returns _n_of;14:14 reported _v_to;16:16 old _a_1;17:17 masters _n_1;18:18 appreciated _v_2;19:19 51 card;20:20 % _n_of;21:21 in _p_temp;22:22 the _q;23:23 year _n_1;24:24 ended _v_cause;25:25 June mofy;26:26 1 dofm;27:27 the _q;28:28 greatest _a_for;29:29 return _n_of;31:31 any _q;33:33 13 card;34:34 assets _n_1;0:0 Salomon named', '0:0 (Impressionist _n_1;1:1 and _c;2:2 modern _a_1;3:3 paintings _n_of;5:5 tracked _v_1;7:7 Salomon named;9:9 ranked _v_1;10:10 even _x_deg;11:11 higher _a_1;12:12 at _p;13:13 74 card;14:14 % _n_of;0:0 (Impressionist _n_1', '0:0 Salomon named;1:1 moreover _a_1;2:2 gets _v_1;4:4 data _n_1;5:5 on _p;6:6 art _n_1;7:7 appreciation _n_for;8:8 from _p;12:12 prices _n_of;13:13 go _v_1;14:14 up _p_dir;15:15 with _p;16:16 clients _n_1;17:17 like _p;18:18 Mr _n_1;19:19 Paul named;20:20 in _p;0:0 Salomon named', '0:0 The _q;1:1 percentages _n_of;2:2 omit _v_1;3:3 from _p;4:4 consideration _n_of;5:5 the _q;7:7 paintings _n_of;9:9 go _v_ing;10:10 begging _v_1;11:11 at _p;0:0 The _q', '0:1 _n_1 Art indexes;2:2 track _v_1;3:3 winners _n_of;4:4 not _c;0:0 Art _n_1', '0:0 But _c;1:1 art _n_1;4:4 fallen _v_1;5:5 sharply _a_1;6:6 in _p;7:7 value _n_1;9:9 rarely _a_1;10:10 put _v_up;12:12 for _p;0:0 But _c', '0:0 Also _a_1;1:1 at _p;2:2 any _q;4:4 Sotheby named;6:6 auctions _n_of;8:8 old _a_1;9:9 masters _n_1;10:10 roughly _x_deg;12:12 third _n_of;12:12 third card;13:13 to _p;15:15 fifth _n_of;15:15 fifth card;16:16 of _p;19:21 _v_1 offered sell;22:22 at _p;23:23 any _q;0:0 Also _a_1', '1:1 \'s _v_nv;5:5 aren\'t _v_there;6:6 any _q;7:7 bids _n_1;8:8 but _c;9:9 the _q;10:10 bids _n_1;12:12 meet _v_1;13:13 the _q;14:14 minimum _a_1;15:15 "reserve" _n_1;16:16 prices _n_of;17:17 set _v_1;19:19 the _q;1:1 \'s _v_nv', '0:0 In _p_temp;1:1 January mofy;2:2 the _q;3:3 Preti named;4:4 painting _n_of;6:6 now _a_1;7:7 hangs _v_1;8:8 at _p;9:9 CenTrust named;11:13 _v_1 expected bring;14:14 no _x_deg;17:17 $ _n_1;18:18 700,000 card;19:19 at _p;20:20 auction _n_of;21:21 until _x_h;22:22 Mr _n_1;23:23 Paul named;24:24 came _v_along;26:26 with _p;28:28 $ _n_1;29:30 card 1.15 million', '0:0 Mr _n_1;1:1 Hall named;2:2 of _p;3:3 the _q;4:4 Colnaghi named;5:5 gallery _n_of;6:6 says _v_to;7:7 $ _n_1;8:9 card 1.15 million;10:10 "would _v_modal;12:12 been _v_id;13:13 an _q;14:14 impossible _a_for;15:15 price _n_of;17:17 anyone _q;19:19 ask _v_1;20:20 for _p;21:21 a _q;22:22 Preti named;23:23 four card;24:24 years _n_1;0:0 Mr _n_1', '0:0 Mr _n_1;1:1 Hall named;2:2 of _p;3:3 the _q;4:4 Colnaghi named;5:5 gallery _n_of;6:6 says _v_to;7:7 $ _n_1;8:9 card 1.15 million;10:10 "would _v_modal;12:12 been _v_id;13:13 an _q;14:14 impossible _a_for;15:15 price _n_of;17:17 anyone _q;19:19 ask _v_1;20:20 for _p;21:21 a _q;22:22 Preti named;23:23 four card;24:24 years _n_1;0:0 Mr _n_1', '0:0 But _c;1:1 from _p_state;3:3 vantage _n_1;4:4 point _n_of;6:6 isn\'t _v_nv;8:8 Mr _n_1;9:9 Paul named;10:10 a _q;11:11 customer _n_of;14:14 too _a_also;15:15 overpaid _v_unknown;16:16 for _p;17:17 the _q;18:18 work _n_1;19:19 "a _q;20:20 gargantuan _a_unknown;21:21 painting _n_of;22:22 by _p;23:23 an _q;24:24 artist _n_1;26:26 is _v_id;28:28 a _q;29:29 household _n_1;0:0 But _c', '0:0 (The _q;1:1 painting _n_of;3:3 10 card;4:4 feet _n_1;5:5 wide _a_1;6:6 seven card;7:7 feet _n_1;0:0 (The _q', '0:2 _a_1 Rather just;3:3 shows _v_1;4:4 things _n_of-about;0:0 Rather _a_1', '0:0 Mr _n_1;1:1 Paul named;2:2 boasts _v_to;5:5 spotted _v_1;6:6 bargains _n_1;7:7 in _p;8:8 old _a_1;9:9 masters _n_1;10:10 just _x_deg;11:11 before _x_h;13:13 took _v_1;14:14 an _q;15:15 upward _a_1;0:0 Mr _n_1', '1:1 went _v_1;2:2 up _p_dir;3:3 51 card;4:4 % _n_of;5:5 last _a_1;6:6 year _n_1;7:7 and _c;10:10 do _v_1;12:12 again _a_1;13:13 this _q_dem;14:14 year," _n_1;1:1 went _v_1', '1:1 were _v_id;2:2 a _q;1:1 were _v_id', '2:2 out _p;3:3 buying _v_1;2:2 out _p', '2:2 vice _n_1;3:3 president _n_of;4:5 named Diana Levitt;6:6 says _v_to;7:7 the _q;8:9 _n_of auction house;12:12 "assisting" _v_1;13:13 Mr _n_1;14:14 Paul named;15:15 in _p;16:16 selling _v_1;17:17 the _q;2:2 vice _n_1', '0:0 Mr _n_1;1:1 Paul named;2:5 _v_1 claims sold;6:6 three card;7:7 paintings _n_of;8:8 at _p;11:11 a _q;12:12 10 card;13:13 % _n_of;0:0 Mr _n_1', "0:0 That _q_dem;1:1 isn't _v_id;2:2 51 card;3:3 % _n_of;4:4 and _c;5:5 the _q;6:6 claim _n_of;0:0 That _q_dem", '1:1 furthermore _a_1;2:2 denies _v_to;5:5 relied _v_on;7:7 heavily _a_1;11:11 or _c;12:12 Mr _n_1;1:1 furthermore _a_1', '0:0 Mr _n_1;1:1 Paul named;2:2 says _v_to;4:4 had _v_1;6:6 one card;7:7 but _c_not;8:8 four card;9:9 advisers _n_1;10:10 and _c;13:13 never _a_1;14:14 bid _v_1;0:0 Mr _n_1', '0:1 After all _a_1;3:3 had _v_1;4:4 the _q;5:5 counsel _n_1;6:6 of _p;7:7 "curators _n_unknown;8:8 from _p;9:9 the _q;11:11 reputable _a_unknown;12:12 museums _n_of;13:13 in _p;14:14 the _q;0:1 After all _a_1', '1:1 says _v_to;3:5 _v_1 expects sell;6:6 the _q;7:7 collection _n_of;9:9 including _v_1;10:10 the _q;11:11 controversial _a_1;12:12 Rubens named;13:13 - excl;14:14 "carefully _a_with-about-of;15:15 and _c;16:16 prudently _a_for;17:17 just _x_deg;18:18 as _x_subord;1:1 says _v_to', '0:0 But _c;1:1 in _p_state;2:2 art- _n_1;3:3 world _n_of;4:4 parlance _n_unknown;5:5 Mr _n_1;6:6 Paul named;8:8 holdings _n_1;0:0 But _c', '0:1 That is _a_1;5:7 _v_1 compelled put;9:9 on _p;10:10 the _q;11:11 market _n_1;13:13 soon _p;14:14 and _c;16:16 already _a_1;17:17 gotten _v_1;18:18 offers _n_1;20:20 are _v_id;24:24 paid _v_for;26:28 _q some the;29:30 _n_1 art works', '0:0 "After _p;1:2 a few _a_1;3:3 years _n_1;5:5 can _v_modal;6:6 argue _v_with;9:9 been _v_there;10:10 natural _a_for;11:11 appreciation," _n_for;12:12 says _v_to;13:14 named Susan Theran;15:15 the _q;16:16 publisher _n_1;18:20 named Leonard Annual;21:22 named Price Index;23:23 of _p;24:25 named Art Auctions', '0:0 But _c;1:1 quick _a_1;2:2 turnover _n_1;3:3 in _p;4:4 artwork _n_1;6:6 "like _p;7:7 pawning _v_1;9:9 jewelry _n_1;12:12 end _v_up;14:14 with _p;15:15 50 card;0:0 But _c', '0:0 People _n_of;1:1 hold _v_out;3:3 and _c;4:6 _v_1 try get;7:7 a _q;0:0 People _n_of', '2:2 defends _v_from;4:4 and _c;5:5 Mr _n_1;6:6 Paul named;7:7 in _p;8:8 the _q;2:2 defends _v_from', '0:0 Mr _n_1;1:1 Wachter named;2:2 says _v_to;3:3 Mr _n_1;4:4 Paul named;5:5 was _v_id;6:6 a _q;7:7 quick _a_1;8:8 study _n_of;10:10 worked _v_1;11:11 intensely _a_1;12:12 and _c;13:13 bought _v_1;14:14 the _q;15:15 best _a_at-for-of;16:16 pictures _n_of;17:17 available _a_1;18:18 at _p_temp;19:19 the _q;0:0 Mr _n_1', '0:0 "On _p_state;1:1 occasion _n_1;3:3 paid _v_for;4:4 a _q;5:5 high _a_1;6:6 price," _n_of;7:7 Mr _n_1;8:8 Wachter named;9:9 concedes _v_to;10:10 but _c;12:12 says _v_to;13:13 those _q_dem;15:15 bid _v_1;17:17 and _c;18:18 dropped _v_out;20:20 were _v_id;21:21 dealers _n_1;23:23 would _v_modal;24:24 then _a_1;26:26 marked _v_1;27:27 up _p;28:28 the _q;29:29 paintings _n_of;30:30 to _x;31:31 resell _v_1;31:31 resell _re-_a_again;33:33 at _p;34:34 a _q;35:37 _n_1 profit collectors', '0:1 named Naomi Bernhard;2:2 Levinson named;3:3 a _q;5:5 arts _n_1;5:5 arts _fine_a_for;6:6 appraiser _n_unknown;7:7 at _p;8:9 named Bernhard Associates;10:10 in _p;11:12 named San Francisco;13:13 considers _v_1;15:15 "definite _a_1;16:16 conflict _n_1;17:17 of _p;18:18 interest _n_in;19:19 for _p;20:20 an _q;21:22 _n_of auction house;23:23 to _x;24:24 both _a_1;25:25 advise _v_1;26:26 a _q;27:27 client _n_1;28:28 on _p;29:29 purchases _n_of;30:30 and _c;31:31 to _x;32:32 set _v_1;33:34 _n_of price estimates;35:35 on _p;36:36 the _q;37:37 paintings _n_of;0:0 Naomi named', '3:3 says _v_to;5:5 "wearing _v_1;6:6 both _q;6:6 both card;3:3 says _v_to', '1:1 can\'t _v_modal;2:2 see _v_1;3:3 why _p;5:5 would _v_modal;6:6 be _v_there;7:7 a _q;8:8 conflict _n_1;9:9 of _p;10:10 interest," _n_in;11:11 says _v_to;12:12 Sotheby named;14:14 Ms _n_1;1:1 can\'t _v_modal', '0:0 "Estimates _n_of;2:2 based _v_1;3:3 on _p;4:4 the _q;5:5 previous _a_1;6:6 price _n_of;8:8 similar _a_to;9:9 works _n_1;10:10 sold _v_1;11:11 at _p;12:12 auction _n_of;13:13 and _c;14:14 current _a_1;15:16 _n_1 market conditions;17:17 and _c;20:20 affected _v_1;22:22 any _q;23:23 knowledge _n_of;26:26 the _q;27:27 potential _a_1;28:28 buyer _n_of;29:29 could _v_modal;0:0 "Estimates _n_of', '0:0 Frequently _a_1;1:1 clients _n_1;2:2 express _v_to;3:3 interest _n_in;5:5 paintings _n_of;6:6 but _c;8:8 end _v_up;10:10 bidding _v_1;12:12 adds _v_to;13:13 "so _x;16:16 know _v_1;18:18 the _q;19:19 potential _a_1;20:20 buyer _n_of;0:0 Frequently _a_1', '0:0 Mr _n_1;1:1 Paul named;2:2 in _p;3:3 selling _v_off;6:6 paintings _n_of;8:8 seeking _v_1;9:10 at least _x_deg;11:11 a _q;12:12 15 card;13:14 _n_of % return;15:15 on _p;16:16 the _q;17:17 bank _n_of;19:19 investment _n_1;20:21 so as _so+as+to_x;23:23 prove _v_to;25:25 the _q;26:26 venture _n_1;0:0 Mr _n_1', '0:0 Mr _n_1;1:1 Paul named;2:2 says _v_to;5:5 has _v_prd;6:6 feelers _n_1;7:8 _p out over;11:11 the _q;12:12 globe _n_1;13:13 and _c;15:15 potential _a_1;16:16 buyers _n_of;17:17 from _p;19:19 far _x_deg;20:21 _p away as;22:22 Japan named;23:23 and _c;24:24 Italy named;26:26 examined _v_1;27:27 the _q;0:0 Mr _n_1', '0:1 Because of _p;2:2 the _q;3:3 pressure _n_1;4:4 on _p;5:5 CenTrust named;7:7 sell _v_1;8:8 dealers _n_1;9:9 and _c;10:10 collectors _n_1;13:15 _v_1 trying get;16:16 the _q;17:17 paintings _n_of;18:18 at _p;19:20 _n_1 bargain- basement;0:1 Because of _p', '0:0 But _c;2:2 far _a_1;3:3 Mr _n_1;4:4 Paul named;5:5 and _c;7:7 advisers _n_1;9:9 holding _v_1;0:0 But _c', '0:0 One card;1:1 dealer _n_1;2:3 named Martin Zimet;4:4 of _p;5:5 French named;6:7 & Co _n_1;8:8 in _p;9:10 named New York;11:11 says _v_to;13:13 "would _v_modal;15:15 loved _v_1;17:17 buy" _v_from;18:18 a _q;19:20 named Jan Davids;21:22 named de Heem;23:23 painting _n_of;25:25 the _q;0:0 One card', "1:1 tried _v_1;3:3 steal _v_from;4:4 the _q;5:5 picture _n_of;7:7 to _x;8:8 buy _v_1;10:10 attractively _a_1;12:12 and _c;13:13 {Sotheby named;15:15 wouldn't _v_modal;1:1 tried _v_1", '2:2 protecting _v_1;2:2 protecting _v_1', '0:0 Meanwhile _p;1:1 Mr _n_1;2:2 Paul named;3:3 and _c;4:4 CenTrust named;5:5 executives _n_1;7:7 getting _v_state;8:8 squeamish _a_unknown;9:9 about _p;0:0 Meanwhile _p', '0:0 Mr _n_1;1:1 Paul named;4:4 characterized _v_as;6:6 "the _q;7:8 named Great Gatsby;9:9 or _c;10:10 something," _q;11:11 complains _v_to-about;12:13 named Karen E;14:14 Brinkman named;15:15 an _q;16:17 _n_1 executive vice;18:18 president _n_of;0:0 Mr _n_1', '0:0 The _q;1:1 media _n_1;3:3 says _v_to;5:5 distorted _v_cause;7:7 personal _a_1;0:0 The _q', '0:0 Mr _n_1;1:1 Paul named;2:2 nods _v_1;3:3 in _p;0:0 Mr _n_1', '2:4 _v_1 think have;5:5 a _q;6:7 _n_of life style;10:10 frankly _a_1;12:12 flamboyant," _a_unknown;2:2 think _v_1', '0:0 But _c;1:1 at _p_temp;2:2 just _x_deg;3:3 that _q_dem;4:4 moment _n_1;7:7 interrupted _v_1;8:8 in _p;10:10 office _n_of;11:11 by _p;12:12 a _q;13:13 servant _n_1;14:14 in _p;15:15 tuxedo _n_unknown;17:17 pours _v_1;18:18 coffee _n_1;19:19 from _p;20:20 silver _n_1;21:21 into _p;22:22 a _q;23:23 cup _n_of;25:25 china named;26:26 and _c;27:27 dabs _v_1;28:28 the _q;29:29 brim _n_of;30:30 with _p;0:0 But _c', '0:0 Mr _n_1;1:1 Paul named;2:2 says _v_to;4:4 the _q;5:5 ceiling _n_1;6:6 in _p;8:9 _n_1 executive suite;10:10 is _v_id;11:12 _n_1 gold- leaf;0:0 Mr _n_1', "1:1 say _v_to;3:3 's _v_id;4:4 a _q;5:6 _n_1 gold ceiling", '0:0 Just _a_1;1:1 say _v_to;2:2 the _q;3:3 offices _n_of;5:5 tastefully _a_1;6:6 appointed," _v_1;0:0 Just _a_1', '0:0 "Otherwise _a_1;1:1 the _q;2:2 regulators _n_1;4:4 take _v_1;6:6 for _p;7:7 decadence _n_unknown;8:8 and _c;9:9 nowadays _a_1;12:12 got _v_to;0:0 "Otherwise _a_1', '0:0 Figures _n_1;2:2 include _v_1;3:3 taxes _n_1;4:4 or _c;5:6 _n_1 transaction costs', '0:0 Companies _n_of;1:1 listed _v_1;2:2 below _p;3:3 reported _v_to;4:4 quarterly _a_1;5:5 profit _n_1;6:6 substantially _a_1;7:7 different _a_than-from;9:9 the _q;10:10 average _n_of;12:12 analysts _n_1;0:0 Companies _n_of', '0:0 The _q;1:1 companies _n_of;3:3 followed _v_1;5:6 at least _x_deg;7:7 three card;8:8 analysts _n_1;9:9 and _c;10:10 had _v_1;11:11 a _q;12:12 minimum _a_1;14:14 cent _n_1;14:14 cent card;15:15 change _n_of;16:16 in _p;17:17 actual _a_1;18:18 earnings _n_1;19:19 per _p;0:0 The _q', '0:0 Estimated _v_at;1:1 and _c;2:2 actual _a_1;3:3 results _n_of;4:4 involving _v_1;5:5 losses _n_of;0:0 Estimated _v_at', '0:0 The _q;1:1 percent _n_of;2:2 difference _n_1;3:3 compares _v_with;4:4 actual _a_1;5:5 profit _n_1;7:7 the _q;9:9 day _n_of;9:9 day card;10:10 estimate _n_of;12:13 at least _x_deg;14:14 three card;15:15 analysts _n_1;16:16 have _v_1;17:18 _n_1 issues forecasts;19:19 in _p;20:20 the _q;21:21 past _a_1;22:22 30 card;0:0 The _q', '0:1 _a_1 Otherwise actual;2:2 profit _n_1;4:4 compared _v_with;6:6 the _q;8:8 day _n_of;8:8 day card;0:0 Otherwise _a_1', '0:0 {During _p;2:3 _n_1 centennial year;4:4 The _q;5:6 named Wall Street;7:7 Journal named;9:9 report _v_to;10:10 events _n_item;11:11 of _p;12:12 the _q;13:13 past _a_1;14:14 century _n_1;16:16 stand _v_1;17:17 as _p;18:18 milestones _n_unknown;19:19 of _p;20:20 American _a_1;21:21 business _n_1;0:0 {During _p', '0:1 named CREATIVE ACCOUNTING;2:2 mostly _x_deg;3:3 by _p;4:4 conglomerates _n_unknown;5:5 forced _v_1;6:6 CPAs _n_unknown;8:8 change _v_cause;10:10 way _n_of;12:12 setting _v_1;13:13 standards _n_1;16:16 followed _v_1;18:18 corporations _n_1;19:19 reporting _v_to;20:20 financial _a_1;21:21 results _n_of;22:22 standards _n_1;25:25 become _v_id;26:27 all too _x_deg;0:0 CREATIVE named', '0:1 named CREATIVE ACCOUNTING;2:2 mostly _x_deg;3:3 by _p;4:4 conglomerates _n_unknown;5:5 forced _v_1;6:6 CPAs _n_unknown;8:8 change _v_cause;10:10 way _n_of;12:12 setting _v_1;13:13 standards _n_1;16:16 followed _v_1;18:18 corporations _n_1;19:19 reporting _v_to;20:20 financial _a_1;21:21 results _n_of;22:22 standards _n_1;25:25 become _v_id;26:27 all too _x_deg;0:0 CREATIVE named', '0:0 The _q;1:1 new _a_1;2:3 named Financial Accounting;4:5 named Standards Board;6:6 (FASB) named;8:8 created _v_1;9:9 in _p_temp;10:10 1972 yofc;11:11 to _x;12:12 replace _v_with;13:13 the _q;14:15 named Accounting Principles;16:16 Board _n_of;18:18 the _q;19:20 named American Institute;21:21 of _p;22:23 named Certified Public;0:0 The _q', '0:0 The _q;1:1 new _a_1;2:5 _n_of board member;5:5 member card;6:6 structure _n_1;7:7 kept _v_unspec;8:8 four card;9:9 CPAs _n_unknown;10:10 but _c;11:11 the _q;12:12 others _n_1;14:14 from _p;15:15 industry _n_1;16:16 and _c;0:0 The _q', '0:1 named Francis M;2:2 Wheat named;3:3 a _q;4:4 former _a_1;5:5 Securities named;6:6 and _c;7:8 named Exchange Commission;9:9 member _n_of;10:10 headed _v_1;11:11 the _q;12:12 panel _n_of;15:15 studied _v_1;16:16 the _q;17:17 issues _n_1;18:18 for _p;19:19 a card;20:20 year _n_1;21:21 and _c;22:22 proposed _v_to;23:23 the _q;24:24 FASB named;25:25 on _p_temp;26:26 March mofy;27:27 30 dofm;0:0 Francis named', '0:0 The _q;1:1 former _a_1;2:2 board _n_of;4:4 produced _v_1;5:5 "21 card;6:6 opinions _n_of;7:7 and _c;8:8 1,000 card;9:9 critics" _n_of;10:10 in _p;13:13 year _n_1;13:13 year card;14:16 _n_of life chairman;0:0 The _q', '0:0 The _q;1:1 climate _n_1;3:3 right _a_1;4:4 for _p;5:5 the _q;6:6 new _a_1;0:0 The _q', '0:0 In _p_state;1:1 the _q;2:2 late _a_for;3:3 1960s year_range;4:4 some _q;5:5 CPAs _n_unknown;6:8 _v_1 failed correct;9:9 such _q;10:10 abuses _n_of;11:11 as _p;12:12 clients _n_1;13:13 picking _v_1;14:14 permissive _a_unknown;15:15 rules _n_of;17:17 hyped _v_1;18:18 earnings _n_1;19:19 and _c;20:20 stock _n_1;0:0 In _p_state', '0:0 And _c;1:1 in _p_temp;2:2 November mofy;3:3 1970 yofc;4:4 Congress named;6:6 passed _v_1;7:7 a _q;8:8 special _a_1;9:9 act _n_of;11:11 overrule _v_unknown;12:12 one card;13:14 _n_of board rule', '0:0 Keepers _n_1;1:1 of _p;2:2 the _q;3:3 books _n_of;4:4 dubbed _v_1;5:6 _n_1 "figure filberts,";7:7 loathed _v_1;8:8 the _q;0:0 Keepers _n_1', '0:0 The _q;1:1 FASB named;2:2 had _v_1;4:4 initial _a_1;5:5 meeting _n_of;6:6 on _p_temp;7:7 March mofy;8:8 28 dofm;0:0 The _q', '0:0 On _p_temp;1:1 Dec mofy;2:2 13 dofm;3:3 1973 yofc;5:5 issued _v_1;7:7 first ord;8:8 rule; _n_of;10:10 required _v_of;11:11 companies _n_of;13:13 disclose _v_to;14:14 foreign _a_1;15:16 _n_1 currency translations;17:17 in _p;18:18 U.S named_n;0:0 On _p_temp', '0:0 The _q;1:1 FASB named;2:2 since _p;3:3 then _p_temp;5:5 issued _v_1;6:6 102 card;7:7 rules _n_of;8:8 and _c;9:9 some _q;10:10 still _a_1;11:11 rile _v_unknown;0:0 The _q', '0:0 Since _p;1:1 late _a_for;2:2 1987 yofc;7:7 put _v_off;9:9 a _q;10:10 rule _n_of;11:11 dealing _v_with;13:13 deferred _v_1;14:15 _n_1 income taxes;16:17 because of _p;18:18 the _q;19:19 continuing _v_1;20:20 controversy _n_1;21:21 over _p;22:22 the _q;0:0 Since _p', '0:1 named Amcast Industrial;2:2 Corp _n_1;3:3 said _v_to;5:7 _v_1 plans repurchase;7:7 repurchase _re-_a_again;8:8 500,000 card;9:9 shares _n_of;10:10 or _c;11:11 about _x_deg;12:12 7 card;13:13 % _n_of;14:14 of _p;16:16 shares _n_of;17:17 outstanding _a_1;18:18 in _p;19:19 open _a_1;20:21 _n_1 market transactions', '0:0 The _q;1:2 _n_1 metal products;3:3 concern _n_1;4:4 currently _a_1;5:5 has _v_1;6:7 card 7.2 million;8:8 common _a_for;9:9 shares _n_of;0:0 The _q', '0:0 The _q;1:1 company _n_of;2:2 named _v_1;3:4 named Dillon Read;5:6 & Co _n_1;7:7 as _p;9:9 exclusive _a_of;10:10 agent _n_of;11:11 for _p;12:12 the _q;13:14 _n_1 stock buy- back;0:0 The _q', '0:0 A _q;1:1 seat _n_1;2:2 on _p;3:3 the _q;4:5 named Chicago Board;6:6 of _p;7:7 Trade named;9:9 sold _v_1;10:10 for _p;11:11 $ _n_1;12:12 390,000 card;13:13 down _p;14:14 $ _n_1;15:15 5,000 card;16:16 from _p;17:17 the _q;18:18 previous _a_1;19:19 sale _n_of;20:20 last _a_1;0:0 A _q', '0:0 Seats _n_1;1:1 currently _a_1;3:3 quoted _v_1;4:4 at _p;5:5 $ _n_1;6:6 353,500 card;7:7 bid _v_1;8:8 $ _n_1;9:9 405,000 card;0:0 Seats _n_1', '0:0 The _q;1:2 _n_of record price;3:3 for _p;4:4 a _q;5:5 full _a_of;6:6 membership _n_1;7:7 on _p;8:8 the _q;9:9 exchange _n_1;10:10 is _v_id;11:11 $ _n_1;12:12 550,000 card;13:13 set _v_1;14:14 Aug mofy;15:15 31 dofm;0:0 The _q', '0:0 An _q;1:2 _n_of associate member;3:3 seat _n_1;5:5 sold _v_1;6:6 for _p;7:7 $ _n_1;8:8 228,000 card;9:9 up _p;10:10 $ _n_1;11:11 8,000 card;12:12 from _p;13:13 the _q;14:14 previous _a_1;15:15 sale _n_of;16:16 Oct mofy;0:0 An _q', '0:1 _n_of Associate member;2:2 seats _n_1;3:3 currently _a_1;5:5 quoted _v_1;6:6 at _p;7:7 $ _n_1;8:8 225,000 card;9:9 bid _v_1;10:10 $ _n_1;11:11 256,000 card;0:0 Associate _n_of', '0:0 The _q;1:2 _n_of record price;3:3 for _p;4:4 associate _n_of;5:5 membership _n_1;6:6 is _v_id;7:7 $ _n_1;8:8 275,000 card;9:9 set _v_1;10:10 Aug mofy;11:11 30 dofm;0:0 The _q', '0:1 named CAE Industries;2:2 Ltd _n_1;3:3 said _v_to;5:6 named Link Flight;7:7 Simulation named;8:8 division _n_of;10:10 awarded _v_1;11:11 a _q;12:12 contract _n_1;13:13 by _p;14:14 the _q;15:15 U.S named_n;16:16 Army _n_of;17:17 for _p;18:18 two card;19:20 _n_1 helicopter simulators;22:22 the _q;23:23 company _n_of;24:24 valued _v_1;25:25 at _p;29:30 card 37 million;31:31 Canadian _a_1;32:33 _n_1 dollars (US$;34:35 card 31.5 million)', '0:0 CAE named;1:1 said _v_to;2:2 the _q;3:3 fixed _a_1;4:4 price _n_of;5:5 for _p;6:6 the _q;7:7 first ord;9:9 the _q;10:11 named AH-64 Apache;12:13 _n_1 combat mission;14:14 simulators _n_1;15:15 is _v_id;16:16 C$ _n_1;17:18 card 19 million', '2:2 scheduled _v_1;3:3 for _p;4:4 delivery _n_of;5:5 in _p_temp;6:6 late _a_for;2:2 scheduled _v_1', '0:0 The _q;1:1 price _n_of;3:3 the _q;4:4 second ord;5:5 simulator _n_1;6:6 ranges _v_1;7:7 between _p;8:8 C$ _n_1;9:10 card 16.4 million;11:11 and _c;12:12 C$ _n_1;13:14 card 18 million;15:15 CAE named;16:16 said _v_to;17:17 depending _v_on-upon;20:20 the _q;21:21 Army named;22:22 exercises _v_1;0:0 The _q', '0:0 CAE named;1:1 is _v_id;2:2 a _q;3:3 Toronto- named;4:4 based _v_1;5:5 maker _n_of;7:7 commercial _a_1;8:8 and _c;9:9 military _a_1;10:11 _n_1 aircraft simulators;12:12 and _c;13:13 training _v_1;0:0 CAE named', '0:0 Total _a_1;1:1 value _n_1;2:2 of _p;3:3 the _q;4:4 contract _n_1;5:5 could _v_modal;6:6 be _v_id;7:7 $ _n_1;8:9 card 100 million;10:10 Helionetics named;11:11 said _v_to;12:12 and _c;13:13 work _n_1;14:14 on _p;15:15 the _q;16:16 project _n_1;17:17 would _v_modal;19:19 about _x_deg;20:20 evenly _a_1;0:0 Total _a_1', '0:0 As _x_prd;1:1 previously _p;2:2 reported _v_to;3:3 Helionetics named;4:4 emerged _v_1;5:5 from _p;6:6 Chapter named;7:7 11 card;8:9 _n_1 bankruptcy- law;10:10 protection _n_of;11:11 in _p_temp;0:0 As _x_prd', '0:0 This _q_dem;1:2 named Los Angeles;3:3 company _n_of;4:4 and _c;6:7 named Union Federal;8:9 named Savings Bank;10:10 subsidiary _n_1;11:11 said _v_to;14:14 99 card;15:15 % _n_of;16:16 of _p;18:18 7 card;19:19 1/4 fraction;20:20 % _n_of;21:21 convertible _a_1;22:22 subordinated _v_1;23:23 debentures _n_unknown;24:24 due _a_1;25:25 2011 card;27:27 tendered _v_1;28:28 for _p;29:29 conversion _n_1;30:30 into _p;31:31 UnionFed named;32:32 common _a_for;0:0 This _q_dem', '0:0 The _q;1:1 conversion _n_1;2:2 increased _v_cause;3:3 total _a_1;4:5 _n_1 equity capital;6:6 by _p;7:7 about _x_deg;8:8 $ _n_1;9:10 card 38.5 million;11:11 to _p;12:12 a _q;13:13 total _n_of;15:15 $ _n_1;16:17 card 156.8 million', '0:1 named Union Federal;2:2 a _q;3:3 federally _a_1;4:4 insured _v_for;5:5 savings _n_1;6:6 bank _n_of;7:7 has _v_1;8:8 $ _n_1;9:10 card 2.4 billion;11:11 in _p;0:0 Union named', '0:1 named David D;2:2 Lung named;4:4 appointed _v_1;5:5 president _n_of;6:6 and _c;7:7 chief _n_1;8:8 operating _v_1;9:9 officer _n_1;10:10 of _p;11:11 this _q_dem;12:12 maker _n_of;14:15 _n_1 building materials;16:16 for _p;17:17 manufactured _v_1;18:18 homes _n_of-n;19:19 and _c;20:20 recreational _a_1;0:0 David named', '0:0 As _p_nbar;1:1 president _n_of;2:2 Mr _n_1;3:3 Lung named;4:4 42 card;5:5 years _n_1;6:6 old _a_1;7:7 succeeds _v_1;9:9 father _n_of;10:11 named Mervin D;12:12 Lung named;13:13 66 card;15:15 founded _v_1;16:16 the _q;17:17 company _n_of;18:18 in _p_temp;0:0 As _p_nbar', '0:1 named Mervin Lung;2:2 remains _v_1;3:3 chairman _n_of;4:4 and _c;5:6 _n_1 chief executive;0:0 Mervin named', '0:1 named David Lung;4:4 with _p;5:5 Patrick named;6:6 since _p;7:7 1970 yofc;8:8 and _c;10:10 served _v_as;11:11 as _p_nbar;12:12 vice _n_1;13:13 president _n_of;14:14 for _p;15:15 administration _n_of;16:16 and _c;17:17 purchasing _v_1;18:18 since _p;0:0 David named', '0:1 named General Dynamics;2:2 Services named;3:3 Co _n_1;4:4 a _q;5:5 unit _n_of;7:8 named General Dynamics;9:9 Corp _n_1;10:10 won _v_1;11:11 a _q;12:12 $ _n_1;13:14 card 48.2 million;15:15 Army named;16:16 contract _n_1;18:18 establish _v_1;19:20 _n_1 maintenance facilities;21:21 for _p;22:22 tracked _v_1;23:23 vehicles _n_1;24:24 in _p;0:0 General named', '0:0 Grumman named;1:1 Corp _n_1;3:3 given _v_1;4:4 a _q;5:5 $ _n_1;6:7 card 15 million;8:8 Navy named;9:9 contract _n_1;10:10 for _p;11:12 _n_1 aircraft- electronics;0:0 Grumman named', '0:1 named Hughes Aircraft;2:2 Co _n_1;3:3 a _q;4:4 unit _n_of;6:7 named General Motors;8:8 Corp _n_1;9:9 got _v_1;10:10 a _q;11:11 $ _n_1;12:13 card 10.3 million;14:15 named Air Force;16:16 contract _n_1;17:17 for _p;19:19 radar _n_1;19:19 radar _airborne_a_1;0:0 Hughes named', '0:1 named Reynolds Metals;2:2 Co _n_1;3:3 said _v_to;5:5 quarter _n_temp;5:5 quarter ord;6:7 _n_1 net income;8:8 dropped _v_1;9:9 nearly _x_deg;10:10 10 card;11:11 % _n_of;12:12 to _p;13:13 $ _n_1;14:15 card 123.7 million;16:16 or _c;17:17 $ _n_1;18:18 2.10 card;19:19 a _p_per;20:20 share _n_of;21:21 from _p;22:22 $ _n_1;23:24 card 137.2 million;25:25 or _c;26:26 $ _n_1;27:27 2.56 card;28:28 a _p_per;29:29 share _n_of;30:30 a _q;31:31 year _n_1;0:0 Reynolds named', '0:0 The _q;1:1 latest _a_for;2:2 earnings _n_1;3:3 reflect _v_1;4:4 an _q;5:5 increase _n_1;7:7 about _x_deg;8:9 card 5.5 million;10:10 in _p;11:11 common _a_for;12:12 shares _n_of;0:0 The _q', '0:0 Revenue _n_1;1:1 rose _v_1;2:2 3 card;3:3 % _n_of;4:4 to _p;5:5 $ _n_1;6:7 card 1.52 billion;8:8 from _p;9:9 $ _n_1;10:11 card 1.48 billion', '0:0 Reynolds named;1:1 is _v_id;2:2 the _q;3:3 third ord;4:4 big _a_1;5:5 aluminum _n_1;6:6 company _n_of;7:7 since _p;8:8 Friday dofw;10:10 report _v_to;11:11 disappointing _v_1;0:0 Reynolds named', '0:0 The _q;1:1 No _n_of;2:2 1 card;3:3 domestic _a_1;4:4 aluminum _n_1;5:5 producer _n_of;6:6 Aluminum named;7:7 Co _n_1;8:8 of _p;9:9 America named;10:10 Friday dofw;11:11 said _v_to;13:13 earnings _n_1;14:14 fell _v_1;15:15 3.2 card;16:16 % _n_of;17:17 to _p;18:18 $ _n_1;19:20 card 219 million;21:21 or _c;22:22 $ _n_1;23:23 2.46 card;24:24 a _p_per;0:0 The _q', '0:0 And _c;1:2 named Alcan Aluminium;3:3 Ltd _n_1;4:4 yesterday _a_1;5:5 reported _v_to;6:7 _n_1 net income;8:8 slid _v_1;9:9 30 card;10:10 % _n_of;11:11 to _p;12:12 $ _n_1;13:14 card 180 million;15:15 or _c;16:16 77 card;17:17 cents _n_1;18:18 a _p_per;19:19 share _n_of;20:20 from _p;21:21 $ _n_1;22:23 card 258 million;24:24 or _c;25:25 $ _n_1;26:26 1.07 card;27:27 a _p_per;0:0 And _c', '0:0 Analysts _n_1;1:1 on _p;2:2 average _n_of;5:5 expecting _v_1;6:6 about _x_deg;7:7 $ _n_1;8:8 2.70 card;9:9 for _p;10:10 Alcoa named;11:11 and _c;12:12 $ _n_1;13:13 1 card;14:14 for _p;0:0 Analysts _n_1', '1:1 \'s _v_itcleft;2:2 a _q;3:3 good _a_at-for-of;4:4 indication _n_of;5:5 that _q_dem;6:6 level _n_1;7:7 of _p;8:8 profitability _n_1;10:10 peaked _v_1;11:11 for _p;12:12 the _q;13:13 industry," _n_1;14:14 says _v_to;15:16 named Vahid Fathi;17:18 _n_1 metals analyst;19:19 with _p;20:21 named Prescott Ball;22:22 & _c;23:24 named Turben Inc;27:27 estimated _v_at;28:28 Reynolds named;29:29 would _v_modal;30:30 earn _v_1;31:31 about _x_deg;32:32 $ _n_1;33:33 2.35 card;34:34 a _p_per;1:1 \'s _v_itcleft', '0:0 The _q;1:3 _n_of nation No;4:4 2 card;5:5 aluminum _n_1;6:6 company _n_of;7:7 said _v_to;8:8 earnings _n_1;10:10 hurt _v_1;12:12 lower _a_1;13:13 prices _n_of;14:14 for _p;15:15 certain _a_of;16:16 fabricated _v_1;17:18 _n_1 aluminum products;20:20 typically _a_of;21:21 follow _v_1;22:22 price _n_of;23:23 fluctuations _n_in;24:24 of _p;25:25 primary _a_1;0:0 The _q', '0:0 The _q;1:2 _n_1 base metal;3:3 price _n_of;5:5 dropped _v_1;6:6 30.3 card;7:7 % _n_of;8:8 from _p;9:9 a card;10:10 year _n_1;11:11 earlier _a_1;12:12 to _p;13:13 78 card;14:14 cents _n_1;15:15 a _p_per;0:0 The _q', '2:2 the _q;3:3 price _n_of;4:4 decline _n_1;7:7 blamed _v_on;9:9 a _q;10:10 slowing _v_1;11:11 economy _n_1;12:12 and _c;13:13 the _q;14:14 third ord;15:15 quarter _n_of;16:16 is _v_id;17:17 typically _a_of;18:18 the _q;19:19 industry _n_1;21:21 slowest _a_1;2:2 the _q', '0:0 But _c;1:2 named William O;3:3 Bourke named;4:4 chairman _n_of;5:5 and _c;6:7 _n_1 chief executive;8:8 officer _n_1;9:9 said _v_to;10:10 the _q;11:11 ingot _n_unknown;12:12 price _n_of;13:13 "appears _v_to;0:0 But _c', '1:1 said _v_to;2:2 shipments _n_of-to;4:4 continuing _v_1;5:5 at _p;6:6 a _q;7:7 "healthy" _a_1;8:8 pace _n_1;9:9 and _c;10:10 the _q;11:11 company _n_of;12:12 has _v_1;13:13 no _q;14:15 _n_1 excess inventory', '0:0 Aluminum _n_1;1:1 shipments _n_of-to;3:3 329,600 card;4:4 metric _a_1;5:5 tons _n_1;7:8 _a_to nearly equal;10:10 the _q;11:12 year- earlier _a_1;13:13 period _n_of;14:14 the _q;15:15 company _n_of;0:0 Aluminum _n_1', '0:0 Nevertheless _a_1;1:1 the _q;2:2 company _n_of;3:3 said _v_to;5:5 in _p_temp;6:6 the _q;7:7 latest _a_for;8:8 quarter _n_temp;10:10 were _v_there;11:11 increased _v_cause;12:12 material _n_1;13:13 and _c;14:15 _n_1 labor costs;16:16 including _v_1;17:17 a _q;18:18 new _a_1;19:19 employee _n_of;20:20 profit- _n_1;21:21 sharing _v_1;0:0 Nevertheless _a_1', '0:0 In _p_state;1:1 composite _n_1;2:2 trading _v_1;3:3 on _p;4:4 the _q;5:6 named New York;7:8 _n_1 Stock Exchange;9:9 Reynolds named;10:10 closed _v_1;11:11 at _p;12:12 $ _n_1;13:13 55.375 card;14:14 up _p;15:15 $ _n_1;0:0 In _p_state', '0:0 No _q;1:1 strikeout _n_unknown;2:2 but _c;3:3 certainly _a_1;4:4 no _q;5:5 home _n_of-n;0:0 No _q', "0:0 That _q_dem;1:1 's _v_id;3:3 the _q;4:4 stock- _n_1;5:5 picking _v_1;6:6 game _n_1;8:8 shaping _v_up;10:10 for _p;11:11 the _q;12:12 months _n_1;13:14 _p ahead according to;16:16 money _n_1;17:17 managers _n_of;18:18 and _c;19:20 a few _a_1;0:0 That _q_dem", '0:0 Yesterday _a_1;3:3 point _n_of;3:3 point card;4:4 recovery _n_1;5:5 from _p;6:6 Friday dofw;8:8 megadrop _n_unknown;9:9 in _p;10:10 the _q;11:12 named Dow Jones;13:13 industrials _n_unknown;14:14 had _v_prd;16:16 brokerage _n_unknown;17:17 houses _n_of;18:18 proclaiming _v_to;20:20 stocks _n_1;21:21 are _v_id;22:22 a _q;23:23 good _a_at-for-of;24:24 bargain _n_1;0:0 Yesterday _a_1', '0:0 But _c;1:1 quite _x;2:3 a few _a_1;4:4 money _n_1;5:5 managers _n_of;0:0 But _c', '0:0 Weakening _v_1;1:1 corporate _a_1;2:2 earnings _n_1;4:4 say _v_to;5:5 are _v_id;6:6 no _q;7:7 prescription _n_unknown;8:8 for _p;9:9 a _q;10:11 _n_1 bull market', '0:0 "The _q;1:2 _n_1 stock market;4:4 going _v_qmodal;6:6 do _v_1;9:9 anything" _q;10:10 for _p;11:12 a while _awhile_n_1;13:13 says _v_to;14:15 named John Neff;16:16 of _p;17:18 named Wellington Management;20:20 runs _v_1;21:21 the _q;22:22 $ _n_1;23:24 card 8.3 billion;25:25 Windsor named;0:0 "The _q', '1:1 suspects _v_of;3:3 Friday dofw;5:6 _n_1 market decline;7:7 may _v_modal;8:8 have _v_1;9:9 a _q;10:10 second ord;11:11 leg _n_1;12:12 perhaps _x_deg;13:13 a _q;14:14 10 card;15:15 % _n_of;17:17 15 card;18:19 _n_of % drop;1:1 suspects _v_of', '0:0 Mr _n_1;1:1 Neff named;2:2 says _v_to;3:3 the _q;4:5 _n_1 stock market;7:7 lost _v_1;8:8 some _q;9:9 powerful _a_1;10:10 driving _v_1;11:11 forces _n_1;12:12 namely _p;13:13 earnings _n_1;14:14 growth _n_of;15:15 and _c;16:16 the _q;17:18 _n_unknown "LBO sweepstakes";20:22 _n_1 buy- out fever;24:24 induced _v_1;25:25 investors _n_1;27:27 bid _v_1;28:28 up _p;29:29 whole _a_1;30:30 groups _n_of;31:31 of _p;32:32 stocks _n_1;33:34 such as _p;35:35 media _n_1;36:36 and _c;0:0 Mr _n_1', '0:0 After _p;1:1 sitting _v_1;2:2 with _p;3:3 20 card;4:4 % _n_of;5:5 of _p;7:7 fund _n_1;8:8 in _p;9:9 cash _n_1;10:10 before _p;11:11 Friday dofw;13:15 _n_1 sell- off Mr;16:16 Neff named;17:17 says _v_to;19:19 bought _v_1;20:20 "a _q;21:21 narrow _a_1;22:22 list _n_of;24:24 stocks" _n_1;0:0 After _p', '0:0 With _x_subord;1:2 _a_1 flat corporate;3:3 profits _n_1;4:4 on _p;5:5 the _q;6:6 horizon _n_1;7:7 for _p;8:8 1990 yofc;9:9 money _n_1;10:10 managers _n_of;11:11 say _v_to;12:12 price- _n_of;13:14 _n_1 earnings multiples;16:16 look _v_seem-to;17:18 _a_1 cheap today;19:19 might _v_modal;20:20 go _v_on;23:23 cheap _a_1;24:24 for _p;25:25 a _q;26:26 long _a_1;0:0 With _x_subord', '0:0 "This _q_dem;1:1 is _v_id;3:3 a _q;4:4 grossly _a_unknown;5:5 overvalued _v_unknown;6:6 market _n_1;7:7 but _c;11:11 cheap _a_1;12:12 either," _a_also;13:13 says _v_to;14:15 named George Collins;16:16 president _n_of;18:18 the _q;19:19 mutual _a_1;20:20 fund _n_1;21:21 company _n_of;22:23 named T Rowe;24:25 named Price Associates;26:26 in _p;0:0 "This _q_dem', '0:0 Before _p;1:1 the _q;2:2 1987 yofc;3:3 crash _n_1;4:4 the _q;5:7 named P E;8:8 was _v_id;10:10 than _p_ind;0:0 Before _p', '0:0 The _q;1:1 common _a_for;2:2 view _n_of;3:3 says _v_to;4:5 named Abby Cohen;6:6 strategist _n_unknown;7:7 for _p;8:9 named Drexel Burnham;10:10 Lambert named;11:11 is _v_nv;15:15 be _v_there;16:17 _a_1 "mild economic;18:18 growth _n_of;19:19 modest _a_1;20:21 _n_1 profit expansion;22:22 and _c;23:23 things _n_of-about;25:25 going _v_qmodal;28:28 hunky- _n_unknown;0:0 The _q', '1:1 view _n_of;2:2 is _v_nv;5:5 may _v_modal;6:6 see _v_1;7:7 a _q;8:9 _n_1 profit decline."', '0:0 Some _q;1:1 think _v_1;2:2 investors _n_1;3:3 should _v_modal;4:4 sell _v_1;5:5 into _p;0:0 Some _q', '0:0 The _q;1:1 market _n_1;3:3 going _v_qmodal;5:5 wind _v_down;7:7 says _v_to;8:9 named Gerald W;10:10 Perritt named;11:11 a _q;12:12 Chicago named;13:13 money _n_1;0:0 The _q', '0:0 "Things _n_of-about;2:3 a little _x_deg;5:5 overpriced" _a_unknown;6:6 after _p;7:7 Friday dofw;9:9 jolt _n_1;10:10 in _p;11:11 the _q;0:0 "Things _n_of-about', '1:1 expects _v_1;2:2 stocks _n_1;4:4 decline _v_1;5:5 an _q;6:6 additional _a_1;7:7 5 card;8:8 % _n_of;9:9 to _p;10:10 30 card;11:11 % _n_of;12:12 with _x_subord;13:13 the _q;14:14 Dow named;16:16 bottoming _v_out;18:18 between _p;19:19 2000 card;20:20 and _c;21:21 2100 yofc;22:22 "between _p;23:23 now _a_1;24:24 and _c;1:1 expects _v_1', '0:0 After _p;1:1 Friday dofw;3:4 _n_1 decline Mr;5:5 Perritt named;7:7 firm _n_1;8:8 ran _v_1;9:9 statistical _a_1;10:10 tests _n_of;11:11 on _p;12:12 100 card;14:14 quality _n_1;14:14 quality _high_a_1;15:15 stocks _n_1;16:16 using _v_1;17:18 old- fashioned _a_1;19:20 _n_1 value criteria;21:21 devised _v_1;23:24 named Benjamin Graham;25:25 an _q;26:26 analyst _n_1;27:27 and _c;28:28 author _n_of;29:29 in _p;30:30 the _q;31:31 1930s year_range;32:32 and _c;33:33 1940s year_range;36:36 widely _a_1;37:37 considered _v_1;39:39 be _v_id;40:40 the _q;41:41 father _n_of;43:43 modern _a_1;44:44 securities _n_1;0:0 After _p', '1:1 found _v_1;2:2 85 card;3:3 still _a_1;4:4 overvalued _v_unknown;5:5 and _c;6:6 15 card;7:7 fairly _a_1;1:1 found _v_1', '0:1 named Nicholas Parks;2:2 a _q;3:4 named New York;5:5 money _n_1;6:6 manager _n_of;7:7 expects _v_1;8:8 the _q;9:9 market _n_1;11:11 decline _v_1;12:12 about _x_deg;13:13 15 card;0:0 Nicholas named', '2:2 been _v_id;4:4 thirds _n_of;4:4 thirds card;5:5 in _p;6:6 cash _n_1;7:7 since _p;8:8 July mofy;9:9 and _c;11:11 continue _v_2;13:15 _v_1 think having;16:16 a _q;17:17 defensive _a_1;18:18 position _n_of;20:20 appropriate," _a_to;2:2 been _v_id', '0:0 Companies _n_of;2:2 piled _v_on;4:4 debt _n_1;5:5 in _p;6:6 leveraged _v_1;7:8 buy- outs _n_1;9:9 during _p;10:10 the _q;11:11 past _a_1;12:12 two card;13:13 years _n_1;15:15 continue _v_2;17:17 surface _v_1;18:18 as _p;19:19 business _n_1;0:0 Companies _n_of', '0:0 "Generalizations _n_1;1:1 about _p;2:2 value _n_1;4:4 useful," _a_for;5:5 says _v_to;6:7 named New York;8:8 money _n_1;9:9 manager _n_of;10:11 named John LeFrere;12:12 of _p;13:14 named Delta Capital;0:0 "Generalizations _n_1', '0:1 For instance _a_1;3:3 says _v_to;4:5 named International Business;6:6 Machines named;7:7 and _c;8:8 Unisys named;9:9 might _v_modal;10:10 look _v_seem-to;11:11 cheap _a_1;12:12 but _c;13:13 investors _n_1;14:14 might _v_modal;15:15 continue _v_2;17:17 do _v_1;18:18 better _a_1;19:19 with _p;20:20 stocks _n_1;21:21 like _p;22:23 named Walt Disney;27:27 and _c;28:29 named Coca- Cola;30:30 strong _a_1;31:31 performers _n_1;32:32 in _p_temp;33:33 recent _a_1;0:1 For instance _a_1', '1:1 are _v_there;2:2 still _a_1;3:3 bulls _n_1;4:4 out _p;1:1 are _v_there', '2:2 doing _v_1;3:4 a little _q;5:5 buying" _v_1;6:6 in _p;7:7 some _q;8:8 stocks _n_1;11:11 really _a_1;13:13 smashed _v_1;2:2 doing _v_1', '1:1 brokerage _n_unknown;2:2 house _n_of;3:3 officials _n_1;4:6 _a_1 also optimistic', '0:0 Yesterday _a_1;1:2 named Goldman Sachs;3:4 named Merrill Lynch;5:5 and _c;6:7 named Dean Witter;8:8 all _a_1;9:9 increased _v_cause;10:10 the _q;11:11 proportion _n_1;12:12 of _p;13:13 assets _n_1;15:15 recommend _v_1;16:16 investors _n_1;17:17 commit _v_to;0:0 Yesterday _a_1', '0:0 Some _q;1:1 investors _n_1;2:2 say _v_to;3:3 Friday dofw;5:6 sell- off _n_1;7:7 was _v_id;8:8 a _q;9:9 good _a_at-for-of;10:10 thing _n_of-about;11:11 because _x;13:13 deflated _v_cause;14:14 a _q;15:15 lot _n_of;17:17 crazy _a_1;18:18 takeover _n_of;0:0 Some _q', '1:1 was _v_id;2:2 a _q;3:3 healthy _a_1;4:4 cleansing," _v_of;5:5 says _v_to;6:7 named Michael Holland;9:9 runs _v_1;10:11 named Salomon Brothers;12:13 named Asset Management;14:14 in _p;15:16 named New York', '0:0 From _p_state;1:1 here _a_1;2:2 out _p;3:3 these _q_dem;4:4 investors _n_1;5:5 see _v_1;6:6 a _q;7:7 return _n_of;8:8 to _p;9:10 old- fashioned _a_1;11:12 _v_1 investing based;13:13 on _p;14:14 a _q;15:15 company _n_of;17:17 ability _n_1;19:19 show _v_1;20:20 profit _n_1;0:0 From _p_state', '0:0 "The _q;1:1 fundamentals _n_1;3:3 pretty _x;4:4 strong," _a_1;5:5 Mr _n_1;6:6 Dreman named;0:0 "The _q', '2:2 see _v_as;3:3 this _q_dem;5:5 a _q;6:7 _n_1 bear market;2:2 see _v_as', "1:1 's _v_id;2:2 a _q;3:3 recognition _n_1;6:6 was _v_there;7:7 much _x_deg;10:10 fluff _n_unknown;11:11 in _p;12:12 the _q;13:13 LBO _n_unknown;1:1 's _v_id", '0:0 Friday dofw;2:2 big _a_1;3:3 fall _n_1;4:4 was _v_id;5:5 "just _a_1;6:6 a _q;7:7 blunder _n_1;8:8 by _p;9:9 the _q;10:11 _n_1 stock market,";12:12 says _v_to;13:14 named John Connolly;15:15 chief _n_1;16:16 strategist _n_unknown;17:17 for _p;18:19 named Dean Witter', '1:1 was _v_id;2:2 an _q;3:3 overreaction _n_1;4:4 to _p;5:5 an _q;6:6 event _n_item;7:7 {the _q;8:8 failure _n_1;9:9 of _p;10:10 a _q;11:11 management _n_1;12:12 and _c;13:14 _n_of union group;16:16 get _v_1;17:17 bank _n_of;18:18 financing _v_1;19:19 for _p;20:20 a _q;21:21 takeover _n_of;23:23 UAL} named;26:26 mean _v_1;27:27 that _x_deg;29:29 to _p;30:30 lots _n_of;1:1 was _v_id', '1:1 investors _n_1;2:3 _v_1 have nagging;1:1 investors _n_1', '0:0 Newspapers _n_of;2:2 full _a_of;4:4 headlines _n_1;5:5 about _p;6:6 companies _n_of;7:7 defaulting _v_on;10:10 debts _n_1;11:11 and _c;12:12 banks _n_of;13:13 writing _v_off;15:17 _n_1 real estate loans', '0:0 That _q_dem;1:1 hurts _v_1;2:4 _n_1 investors confidence;5:5 in _p;6:6 the _q;7:7 economy _n_1;8:8 and _c;0:0 That _q_dem', '0:1 Not even _x_deg;2:3 _q all the;4:4 brokerage _n_unknown;5:5 firms _n_1;6:6 see _v_1;7:7 clear _a_of;8:8 sailing _v_1;0:1 Not even _x_deg', '0:0 "Disappointing _v_1;1:1 profits _n_1;3:3 likely _a_1;5:5 get _v_state;6:6 worse _a_1;7:7 in _p_temp;8:8 the _q;9:9 next _a_1;10:10 two card;11:11 quarters," _n_temp;12:12 says _v_to;13:14 named Mary Farrell;15:15 a _q;16:16 market _n_1;17:17 strategist _n_unknown;18:18 at _p;0:0 "Disappointing _v_1', '1:1 thinks _v_1;2:2 the _q;3:3 market _n_1;4:4 could _v_modal;5:5 drop _v_1;6:6 about _x_deg;7:7 10 card;8:8 % _n_of;9:9 in _p;10:10 the _q;11:11 next _x_deg;13:13 months _n_1;14:14 then _c;15:15 recover _v_1;16:16 and _c;17:17 go _v_state;1:1 thinks _v_1', '0:0 Companies _n_of;1:1 with _p;2:2 steady _a_1;3:3 earnings _n_1;4:4 growth _n_of;5:5 could _v_modal;6:6 do _v_1;7:7 well _a_1;9:9 says _v_to;10:10 while _x;11:11 others _n_1;12:12 with _p;13:13 high _a_1;14:14 debt _n_1;15:15 or _c;16:16 poor _a_1;17:17 earnings _n_1;18:18 could _v_modal;19:19 see _v_1;21:21 shares _n_of;22:22 decline _v_1;23:23 far _x_deg;24:25 more than _p;26:26 10 card;0:0 Companies _n_of', '0:0 The _q;1:1 turmoil _n_1;2:2 on _p;3:4 named Wall Street;5:5 may _v_modal;6:6 benefit _v_1;7:7 some _q;8:8 retailers _n_1;9:11 _v_1 attempting lead;12:12 leveraged _v_1;13:14 buy- outs _n_1;15:15 of _p;17:17 specialty _n_1;18:18 and _c;19:20 _n_1 department- store;21:21 chains _n_of;22:23 _n_1 investment bankers;24:24 and _c;25:25 retailers _n_1;0:0 The _q', '0:0 Managers _n_of;1:1 at _p;2:2 five card;3:3 chains _n_of;5:5 said _v_to;6:6 in _p_temp;7:7 recent _a_1;8:8 weeks _n_1;11:11 intend _v_for;13:13 bid _v_1;14:14 for _p;0:0 Managers _n_of', '0:0 Hooker named;2:2 based _v_1;3:3 in _p;4:5 named Sydney Australia', '0:0 "The _q;1:1 pricing _n_1;3:3 become _v_id;5:5 realistic _a_1;7:7 should _v_modal;8:8 help _v_1;9:9 management," _n_1;10:10 said _v_to;11:12 named Bruce Rosenthal;13:13 a _q;14:15 named New York;16:17 _n_1 investment banker;18:18 with _p;19:19 Nathan named;20:20 S _n_1;21:21 Jonas named;0:0 "The _q', '0:0 "Investors _n_1;2:2 going _v_qmodal;5:5 throwing _v_1;6:6 money _n_1;7:7 at _p;8:10 _q any the;11:11 proposed _v_to;12:12 LBOs _n_unknown;13:13 but _c;14:14 doing _v_1;15:15 deals _n_1;16:16 on _p;17:17 the _q;18:18 basis _n_of;20:20 ridiculous _a_1;21:21 assumptions _n_1;22:22 never _a_1;23:23 made _v_1;24:24 sense _n_of;0:0 "Investors _n_1', '0:0 Earlier _a_1;1:1 this _q_dem;2:3 _n_1 year bankers;4:4 and _c;5:5 other _a_1;6:6 investors _n_1;8:8 willing _a_1;10:11 _v_1 provide financing;12:12 because _x;14:14 assumed _v_1;16:16 would _v_modal;17:17 be _v_there;18:18 major _a_1;19:19 gains _n_1;20:20 in _p;22:22 profitability _n_1;23:23 and _c;24:24 sales _n_of;25:25 Mr _n_1;26:26 Rosenthal named;0:0 Earlier _a_1', '0:0 Those _q_dem;1:1 days _n_of;3:3 over _p_dir;4:4 now _a_1;0:0 Those _q_dem', '0:0 "Competition _n_1;1:1 from _p;2:2 third ord;3:3 parties _n_of-to;5:5 have _v_1;6:6 cash _n_1;7:7 and _c;9:9 prepared _a_1;11:11 buy _v_1;13:13 always _a_1;14:14 existed _v_1;15:15 and _c;17:17 continue," _v_1;18:18 added _v_to;19:19 Mr _n_1;0:0 "Competition _n_1', '0:0 "But _c;1:1 when _x_subord;2:2 prices _n_of;4:7 _a_1 crazy even;8:8 harder _a_for;10:10 do _v_1;11:11 an _q;0:0 "But _c', '0:0 Bankers _n_1;1:1 believed _v_in;3:3 the _q;5:5 fool _n_1;5:5 fool _great_a_for;6:6 theory _n_of;8:8 says _v_to;9:9 somebody _q;10:12 _a_1 else always;13:13 willing _a_1;0:0 Bankers _n_1', '0:0 This _q_dem;2:3 no longer _a_1;4:4 true _a_of;0:0 This _q_dem', '0:0 At _p;1:2 named Saks Fifth;3:4 named Avenue Paul;5:5 Leblang named;6:6 senior _a_1;7:7 vice _n_1;8:8 president _n_of;9:9 marketing _v_1;10:10 agreed _v_with;12:12 lower _a_1;13:13 prices _n_of;15:15 help _v_1;17:17 management _n_1;18:18 team _n_of;19:19 in _p;21:21 proposed _v_to;0:0 At _p', '0:0 "Having _v_qmodal;2:2 take _v_on;5:5 debt _n_1;6:6 would _v_modal;7:7 certainly _a_1;8:8 be _v_id;9:9 an _q;10:10 advantage," _n_1;11:11 said _v_to;12:12 Mr _n_1;0:0 "Having _v_qmodal', '1:1 would _v_modal;2:2 also _a_1;3:3 help _v_1;5:5 in _p;7:7 search _n_of;8:8 for _p;9:10 _n_1 equity partners', '0:0 To _x;1:1 make _v_cause;2:2 an _q;3:3 LBO _n_unknown;4:4 work _v_1;5:5 now _a_1;8:8 going _v_qmodal;10:10 need _v_1;13:13 just _x_deg;14:15 _n_1 junk bonds', '0:0 " None _q;1:1 believe _v_1;2:2 the _q;3:3 proposed _v_to;4:4 management _n_1;5:5 LBOs _n_unknown;8:8 easy _a_for;10:10 complete _v_2;11:11 especially _x_deg;12:12 at _p;13:14 named B Altman;15:16 & Co _n_1;19:19 under _p;20:20 Chapter named;21:21 11 card;22:22 bankruptcy _n_1;0:0 " None _q', '0:0 Further _a_1;2:2 institutions _n_1;3:3 today _a_1;4:4 holding _v_1;5:5 troubled _a_1;6:8 _n_1 retailers debt;9:9 securities _n_1;12:12 reticent _a_unknown;13:13 to _x;14:14 consider _v_1;15:15 additional _a_1;16:16 retailing _v_1;0:0 Further _a_1', '2:2 called _v_name;3:3 bad _a_at;4:4 money _n_1;5:5 driving _v_out;7:7 good _a_at-for-of;8:8 money," _n_1;9:9 said _v_to;10:10 one card;11:11 retailing _v_1;2:2 called _v_name', '0:0 "Institutions _n_1;2:2 usually _a_1;3:3 buy _v_1;4:5 _n_1 retail paper;6:6 have _v_qmodal;0:0 "Institutions _n_1', '0:0 However _a_1;1:1 the _q;2:2 lower _a_1;3:3 prices _n_of;4:4 these _q_dem;5:5 retail _n_1;6:6 chains _v_to;8:8 now _a_1;9:11 _v_1 expected bring;12:12 should _v_modal;13:13 make _v_cause;15:15 easier _a_for;17:17 managers _n_of;19:19 raise _v_cause;20:20 the _q;21:21 necessary _a_for;22:22 capital _n_1;23:23 and _c;24:24 pay _v_for;25:25 back _p;26:26 the _q;27:27 resulting _v_1;0:0 However _a_1', '0:0 In _p_state;1:1 addition _n_1;2:2 the _q;3:3 fall season;4:4 selling _v_1;5:5 season season;7:7 generally _a_1;8:8 been _v_id;9:9 a _q;10:10 good _a_at-for-of;11:11 one card;12:12 especially _x_deg;13:13 for _p;14:14 those _q_dem;15:15 retailers _n_1;16:16 dependent _a_on;18:18 apparel _n_unknown;19:19 sales _n_of;20:20 for _p;21:21 the _q;22:22 majority _n_of;0:0 In _p_state', '2:2 encouraging _a_for;3:3 about _p;4:4 this _q_dem;5:5 is _v_nv;7:7 retail _n_1;8:8 chains _n_of;11:11 sold _v_1;12:12 on _p;13:13 the _q;14:17 _n_of basis sales;18:18 and _c;19:19 earnings _n_1;20:20 not _c;21:21 liquidation _n_unknown;22:22 values," _n_1;23:23 said _v_to;24:25 named Joseph E;26:26 Brooks named;27:27 chairman _n_of;28:28 and _c;29:30 _n_1 chief executive;31:31 officer _n_1;32:32 of _p;33:34 named Ann Taylor;35:35 Inc _n_1;36:36 a _q;37:37 specialty _n_1;2:2 encouraging _a_for', '0:0 "Retailers _n_1;2:2 had _v_1;3:3 good _a_at-for-of;4:4 track _n_1;5:5 records _n_of;7:7 producing _v_1;8:8 profits _n_1;10:10 have _v_1;11:11 a _q;12:12 better _a_at-for-of;13:13 chance _n_of;15:15 buy _v_back;0:0 "Retailers _n_1', '0:0 Still _a_disc;1:1 most _q;2:2 retailing _v_1;3:3 observers _n_1;4:4 expect _v_1;6:7 _q all the;8:8 proposed _v_to;9:9 retailing _v_1;10:10 LBOs _n_unknown;12:12 depend _v_on-upon;13:13 partly _a_1;15:15 the _q;16:16 sale _n_of;18:19 _n_1 junk bonds;20:20 a _q;21:21 market _n_1;22:22 already _a_1;23:23 in _p;24:24 tumult _n_unknown;25:26 in part _x_deg;27:28 because of _p;29:29 concerns _n_1;30:30 associated _v_with;32:32 bonds _n_1;33:33 issued _v_1;35:35 the _q;36:36 Federated named;37:37 and _c;38:38 Allied named;39:39 units _n_of;0:0 Still _a_disc', '0:0 "Prices _n_of;1:1 for _p;2:2 retail _n_1;3:3 chains _n_of;5:6 _a_1 lower today;10:10 last _a_1;11:11 week _n_1;14:14 help _v_1;15:15 management," _n_1;16:16 said _v_to;17:18 named Gilbert Harrison;19:19 chairman _n_of;21:21 Financo named;22:22 Inc _n_1;23:23 an _q;24:25 _n_1 investment- banking;26:26 firm _n_1;27:27 specializing _v_in;29:29 retailing _v_1;0:0 "Prices _n_of', '0:0 "But _c;1:1 the _q;2:2 hurdle _n_unknown;3:3 of _p;4:4 financing _v_1;5:5 still _a_1;6:6 has _v_qmodal;0:0 "But _c', '0:0 Potential _a_1;1:1 bondholders _n_unknown;4:4 look _v_for;6:6 greater _a_for;7:7 equity _n_1;8:8 participation _n_in;9:11 on behalf of _p;12:12 management _n_1;13:13 or _c;14:14 insist _v_1;15:15 the _q;16:17 _n_1 equity component;18:18 of _p;19:19 the _q;20:20 deals _n_1;22:22 substantially _a_1;23:23 greater _a_for;25:25 in _p;26:26 the _q;0:0 Potential _a_1', '0:0 Potential _a_1;1:1 bondholders _n_unknown;4:4 look _v_for;6:6 greater _a_for;7:7 equity _n_1;8:8 participation _n_in;9:11 on behalf of _p;12:12 management _n_1;13:13 or _c;14:14 insist _v_1;15:15 the _q;16:17 _n_1 equity component;18:18 of _p;19:19 the _q;20:20 deals _n_1;22:22 substantially _a_1;23:23 greater _a_for;25:25 in _p;26:26 the _q;0:0 Potential _a_1', '0:0 Sony named;1:1 Corp _n_1;2:2 won _v_1;3:3 a _q;4:4 pretrial _n_1;4:4 pretrial _pre-_a_ante;5:5 order _n_of;6:6 blocking _v_1;7:7 U.S named_n;8:8 sales _n_of;10:11 named Justin Products;12:12 Inc _n_1;14:15 named "My Own";16:16 line _n_of;18:18 portable _a_1;19:19 audio _n_1;20:20 players _n_of;21:21 for _p;0:0 Sony named', '0:0 Judge _n_of;1:2 named John E;3:3 Sprizzo named;4:4 issued _v_1;5:5 the _q;6:6 order _n_of;7:7 in _p;8:8 Manhattan named;9:9 federal _a_1;10:10 court _n_of;12:12 Sony named;14:14 accused _v_of;15:15 the _q;16:16 tiny _a_1;17:17 company _n_of;19:19 illegally _a_for;20:20 knocking _v_off;22:22 the _q;23:24 named "My First;25:25 Sony" named;0:0 Judge _n_of', '0:0 The _q;1:1 judge _n_of;2:2 held _v_1;4:4 the _q;5:5 combination _n_of;7:7 colors _n_1;8:8 used _v_1;9:9 for _p;10:10 the _q;11:11 Sony named;12:12 products _n_1;14:14 distinctive _a_1;15:15 and _c;16:16 subject _a_to;18:18 protection _n_of;19:19 under _p;20:21 named New York;22:22 state _n_of;23:23 law _n_1;24:25 rather than _c;26:26 federal _a_1;0:0 The _q', '0:0 The _q;1:1 legal _a_1;2:2 fight _n_1;3:3 was _v_id;4:4 the _q;5:5 subject _n_of;7:7 a _q;8:9 named Wall Street;10:10 Journal named;11:11 story _n_of;0:0 The _q', '0:0 Justin named;2:2 attorney _n_1;3:4 named Charles E;5:5 Baxley named;6:6 said _v_to;7:7 Justin named;8:8 would _v_modal;9:9 ask _v_1;10:10 an _q;11:11 appeals _n_1;12:12 court _n_of;14:14 set _v_aside;16:16 the _q;17:17 order _n_of;18:19 _a_1 temporarily pending;20:20 an _q;21:21 expedited _v_1;0:0 Justin named', '1:1 also _a_1;2:2 repeated _v_1;3:3 Justin named;5:5 denial _n_of;7:7 Sony named;1:1 also _a_1', '1:1 likelihood _n_of;3:3 reversing _v_1;6:6 very _x_deg;7:7 slim," _a_1;8:8 said _v_to;9:10 named Lewis H;11:12 named Eslinger Sony;14:14 attorney _n_1;16:16 said _v_to;18:18 doubts _v_1;19:19 Justin named;21:21 go _v_ahead;23:23 with _p;24:24 a _q;1:1 likelihood _n_of', '0:1 named CONTINENTAL MORTGAGE;2:2 & _c;3:4 named EQUITY TRUST;5:5 said _v_to;8:8 resume _v_1;9:9 dividend _n_1;10:10 payments _n_of;11:11 with _p;12:12 a _q;17:17 payout _n_1;18:18 on _p_temp;19:19 Nov mofy;20:20 6 dofm;21:21 to _p;22:24 _n_of shares record;25:25 Oct mofy;0:0 CONTINENTAL named', '0:0 The _q;1:1 Dallas named;2:4 _n_1 real estate investment;5:5 trust _n_1;6:6 last _a_1;7:7 paid _v_for;8:8 a _q;9:9 dividend _n_1;10:10 on _p_temp;11:11 Dec mofy;12:12 31 dofm;13:13 1987 yofc;15:15 shareholders _n_1;16:16 received _v_1;17:17 $ _n_1;18:18 1 card;19:19 a _p_per;0:0 The _q', '0:0 Despite _p;1:1 continuing _v_1;2:2 troubles _n_1;3:3 with _p;4:4 problem _n_of;5:5 assets _n_1;6:6 and _c;7:7 nonperforming _a_unknown;8:8 loans _n_1;9:9 the _q;10:10 trust _n_1;11:11 said _v_to;13:13 expects _v_1;16:16 able _a_1;18:18 maintain _v_1;19:19 or _c;20:20 increase _v_cause;21:21 the _q;22:22 rate _n_of;24:24 distributions _n_1;25:26 because of _p;27:27 operations _n_of;30:30 venture _n_1;30:30 venture _joint_a_1;0:0 Despite _p', '0:0 The _q;1:1 court _n_of;2:2 in _p;3:3 a _q;4:6 card 3 0;7:7 ruling _n_1;8:8 threw _v_out;10:10 a _q;11:11 deadline _n_of;12:12 set _v_1;14:14 the _q;15:16 named Federal Energy;17:18 named Regulatory Commission;19:19 for _p;20:20 settling _v_1;21:21 old _a_1;22:23 _n_1 contract disputes;24:24 over _p;25:25 gas _n_1;27:27 the _q;28:28 pipeline _n_1;29:29 companies _n_of;30:30 reserved _v_1;31:31 but _c;0:0 The _q', '0:0 A _q;1:1 majority _n_of;3:3 old _a_1;4:4 contracts _n_1;6:6 renegotiated _v_1;6:6 renegotiated _re-_a_again;7:7 by _p;8:8 the _q;9:9 deadline _n_of;10:10 and _c;11:11 settled _v_1;12:12 at _p;13:13 steep _a_1;0:0 A _q', '0:0 But _c;1:1 pipeline _n_1;2:2 companies _n_of;3:3 estimate _v_at;5:5 still _a_1;6:6 face _v_1;7:7 $ _n_1;8:9 card 2.4 billion;10:10 in _p;11:11 liabilities _n_1;12:12 from _p;13:13 unresolved _v_1;13:13 unresolved _un-_a_rvrs;14:14 disputes _n_1;15:15 including _v_1;16:16 $ _n_1;17:18 card 1 billion;20:20 fear _v_1;24:24 able _a_1;26:26 pass _v_on;0:0 But _c', '0:1 According to _p;2:3 _n_1 industry lawyers;4:4 the _q;5:5 ruling _n_1;6:6 gives _v_1;7:7 pipeline _n_1;8:8 companies _n_of;9:9 an _q;10:10 important _a_for;11:11 second ord;12:12 chance _n_of;14:14 resolve _v_1;15:15 remaining _a_1;16:16 disputes _n_1;17:17 and _c;18:18 take _v_of-i;19:19 advantage _n_i;21:21 the _q;22:22 cost- _n_1;23:23 sharing _v_1;0:1 According to _p', '0:0 The _q;1:1 court _n_of;2:2 left _v_open;5:5 FERC named;6:6 could _v_modal;7:7 reimpose _v_on;7:7 reimpose _re-_a_again;8:8 a _q;9:9 new _a_1;10:10 deadline _n_of;0:0 The _q', '0:0 The _q;1:1 court _n_of;2:2 ordered _v_1;3:3 FERC named;5:5 justify _v_to;6:6 within _p;7:7 60 card;8:8 days _n_of;10:10 only _x_deg;12:12 cost- _n_1;13:13 sharing _v_1;14:14 deadline _n_of;15:15 but _c_not;16:17 _a_1 other major;18:18 elements _n_1;19:19 of _p;21:21 proposed _v_to;22:22 regulation _n_1;23:23 for _p;24:24 introducing _v_to;26:26 competition _n_1;27:27 into _p;29:29 gas _n_1;29:29 gas _natural_a_for;0:0 The _q', '0:0 The _q;1:1 complex _a_1;2:2 regulation _n_1;3:3 known _v_as;4:4 in _p;5:5 the _q;6:6 industry _n_1;8:8 Order named;9:9 500 card;12:12 hotly _a_unknown;13:13 contested _v_1;15:15 all _q;16:16 sides _n_1;17:17 including _v_1;19:19 gas _n_1;19:19 gas _natural_a_for;20:20 producers _n_of;21:21 pipelines _n_1;22:22 local _a_1;23:23 distribution _n_1;24:24 companies _n_of;25:25 and _c;0:0 The _q', '0:0 The _q;1:1 court _n_of;3:3 decision _n_1;4:4 would _v_modal;5:5 allow _v_1;6:6 FERC named;8:8 change _v_cause;9:9 some _q;12:12 provisions _n_1;13:13 but _c;14:18 _v_1 ensures reviewed;19:20 _a_1 again quickly;21:21 by _p;22:22 the _q;0:0 The _q', '0:0 MEDUSA named;1:1 Corp _n_1;2:2 said _v_to;4:4 voluntarily _a_1;5:5 prepaid _pay_v_for;5:5 prepaid _a_ante;6:6 $ _n_1;7:8 card 7 million;9:9 on _p;11:11 original _a_1;12:12 $ _n_1;13:14 card 75 million;15:15 term _n_of;16:16 loan _n_1;17:17 bringing _v_1;18:18 the _q;19:19 total _a_1;20:20 debt _n_1;21:21 reduction _n_of;22:22 for _p;23:23 the _q;24:24 year _n_1;25:25 to _p;26:26 $ _n_1;27:28 card 18 million', '0:0 After _p;1:1 the _q;2:2 payment _n_of;3:3 the _q;4:4 Cleveland named;5:5 company _n_of;6:6 owes _v_1;7:7 $ _n_1;8:9 card 57 million;10:10 on _p;11:11 the _q;0:0 After _p', '0:0 The _q;1:1 cement _n_1;2:2 producer _n_of;3:3 said _v_to;4:4 the _q;5:5 payment _n_of;7:7 made _v_1;8:8 from _p;9:10 _n_1 excess cash;0:0 The _q', '0:1 named NATIONAL INCOME;2:3 named REALTY TRUST;4:4 said _v_to;7:7 resume _v_1;8:8 dividend _n_1;9:9 payments _n_of;10:10 with _p;11:11 a _q;16:16 dividend _n_1;19:19 paid _v_for;20:20 Nov mofy;21:21 6 dofm;23:25 _n_of shares record;26:26 Oct mofy;0:0 NATIONAL named', '0:0 The _q;1:1 mortgage _n_1;2:2 and _c;3:4 _n_1 equity real estate;6:7 _n_1 investment trust;8:8 last _a_1;9:9 paid _v_for;10:10 a _q;11:11 dividend _n_1;12:12 on _p_temp;13:13 Aug mofy;14:14 1 dofm;15:15 1988 yofc;17:17 holders _n_of;18:18 received _v_1;19:19 75 card;20:20 cents _n_1;21:21 a _p_per;0:0 The _q', '0:0 Despite _p;1:1 continuing _v_1;2:2 troubles _n_1;3:3 with _p;4:4 problem _n_of;5:5 properties _n_1;6:6 and _c;7:7 nonperforming _a_unknown;8:8 loans _n_1;9:9 the _q;10:10 Dallas named;11:11 trust _n_1;12:12 said _v_to;15:15 rebuilt _v_1;15:15 rebuilt _re-_a_again;16:16 reserves _n_1;17:17 abandoned _v_1;18:18 properties _n_1;19:19 with _p;21:21 potential _n_1;22:22 and _c;23:23 experienced _v_1;24:24 improved _v_cause;25:25 operating _v_1;26:26 results _n_of;27:27 from _p;28:28 joint _a_1;0:0 Despite _p', '0:0 MLX named;1:1 Corp _n_1;2:2 said _v_to;4:4 reached _v_1;5:5 a _q;6:6 preliminary _a_1;7:7 agreement _n_1;8:8 with _p;9:9 senior _a_1;10:10 lenders _n_unknown;11:11 to _p;13:13 refrigeration _n_1;14:14 and _c;15:15 air- _n_1;16:16 conditioning _v_1;17:17 group _n_of;19:19 restructure _v_cause;20:20 the _q;21:21 $ _n_1;22:23 card 188.5 million;24:24 of _p;25:26 _n_1 credit facilities;27:27 the _q;28:28 lenders _n_unknown;29:29 provide _v_1;31:31 the _q;0:0 MLX named', '0:0 MLX named;2:2 also _a_1;3:3 makes _v_1;4:4 aircraft _n_1;5:5 and _c;7:7 duty _n_1;7:7 duty _heavy_a_1;8:9 _n_1 truck parts;10:10 said _v_to;11:11 the _q;12:12 debt _n_1;14:14 accumulated _v_cause;15:15 during _p;17:17 acquisition _n_of;19:19 nine card;20:20 businesses _n_1;22:22 make _v_up;24:24 the _q;25:25 group _n_of;26:26 the _q;27:27 biggest _a_1;28:28 portion _n_of;32:32 related _v_to;34:34 the _q;35:35 1986 yofc;36:36 purchase _n_of;38:38 a _q;39:39 Hillman named;40:40 Co _n_1;0:0 MLX named', '0:0 Among _p_state;1:1 other _a_1;2:2 things _n_of-about;3:3 the _q;4:4 restructured _v_cause;5:5 facilities _n_1;7:7 substantially _a_1;8:8 reduce _v_1;9:9 the _q;10:10 group _n_of;12:12 required _v_of;13:13 amortization _n_unknown;14:14 of _p;15:15 the _q;16:16 term _n_of;17:17 loan _n_1;18:18 portion _n_of;20:20 the _q;21:22 _n_1 credit facilities;23:23 through _p;24:24 September mofy;25:25 1992 yofc;26:26 MLX named;0:0 Among _p_state', '0:0 Certain _a_of;1:1 details _n_1;2:2 of _p;3:3 the _q;4:4 restructured _v_cause;5:5 facilities _n_1;6:9 _v_1 remain negotiated', '0:0 The _q;1:1 agreement _n_1;3:3 subject _a_to;5:5 completion _n_of;7:7 a _q;8:8 definitive _a_1;9:9 amendment _n_of;10:10 and _c;11:11 appropriate _a_to;0:0 The _q', '0:1 named William P;2:3 named Panny MLX;4:4 chairman _n_of;5:5 and _c;6:7 _n_1 chief executive;8:8 said _v_to;9:9 the _q;10:10 pact _n_1;12:12 provide _v_1;13:13 MLX named;14:14 with _p;15:15 the _q;16:16 additional _a_1;17:17 time _n_of;18:18 and _c;19:19 flexibility _n_1;20:20 necessary _a_for;22:22 complete _v_2;23:23 the _q;24:24 restructuring _v_1;25:25 of _p;26:26 the _q;27:27 company _n_of;29:30 _n_1 capital structure."', '0:0 MLX named;2:2 filed _v_1;3:3 a _q;4:5 _n_of registration statement;6:6 with _p;7:7 the _q;8:8 Securities named;9:9 and _c;10:11 named Exchange Commission;12:12 covering _v_1;13:13 a _q;14:14 proposed _v_to;15:15 offering _n_of;17:17 $ _n_1;18:19 card 120 million;20:20 in _p;22:22 term _n_of;22:22 term _long_a_1;23:23 senior _n_1;24:24 subordinated _v_1;25:25 notes _n_1;26:26 and _c;0:0 MLX named', '0:1 named Dow Jones;2:3 & Co _n_1;4:4 said _v_to;6:6 acquired _v_1;7:7 a _q;8:8 15 card;9:9 % _n_of;10:10 interest _n_in;12:12 DataTimes named;13:13 Corp _n_1;14:14 a _q;15:15 subsidiary _n_1;16:16 of _p;17:18 named Oklahoma Publishing;19:19 Co _n_1;20:20 Oklahoma named;21:21 City _n_1;23:23 provides _v_1;24:24 electronic _a_1;25:26 _n_1 research services', '0:0 Terms _n_of;0:0 Terms _n_of', '0:0 Customers _n_of;3:3 DataTimes named;4:4 or _c;5:6 named Dow Jones;7:7 News named;9:9 Retrieval _n_of;11:11 able _a_1;13:13 access _v_1;14:14 the _q;15:15 information _n_on-about;17:17 both _q;17:17 both card;0:0 Customers _n_of', '0:1 named Dow Jones;2:2 is _v_id;3:3 the _q;4:4 publisher _n_1;6:6 The _q;7:8 named Wall Street;0:0 Dow named', '0:0 Flowers _n_1;1:1 Industries named;2:2 Inc _n_1;3:6 _v_to said report;7:7 a _q;8:8 charge _n_of;10:10 eight card;11:11 cents _n_1;12:12 to _p;13:13 10 card;14:14 cents _n_1;15:15 a _p_per;16:16 share _n_of;17:17 for _p;19:19 fiscal _a_1;20:20 first ord;21:21 quarter _n_temp;22:22 ended _v_cause;23:23 Sept mofy;24:24 23 dofm;25:25 from _p;26:26 the _q;27:27 sale _n_of;29:29 two card;30:30 bakeries _n_1;31:31 in _p;32:33 named High Point;34:34 N.C named;35:35 and _c;36:37 named Gadsden Ala', '0:0 Flowers _n_1;1:1 Industries named;2:2 Inc _n_1;3:6 _v_to said report;7:7 a _q;8:8 charge _n_of;10:10 eight card;11:11 cents _n_1;12:12 to _p;13:13 10 card;14:14 cents _n_1;15:15 a _p_per;16:16 share _n_of;17:17 for _p;19:19 fiscal _a_1;20:20 first ord;21:21 quarter _n_temp;22:22 ended _v_cause;23:23 Sept mofy;24:24 23 dofm;25:25 from _p;26:26 the _q;27:27 sale _n_of;29:29 two card;30:30 bakeries _n_1;31:31 in _p;32:33 named High Point;34:34 N.C named;35:35 and _c;36:37 named Gadsden Ala', '0:0 The _q;1:2 _n_1 convenience- food;3:3 company _n_of;4:4 said _v_to;6:6 sold _v_1;7:7 the _q;8:8 bakeries _n_1;10:11 named Mills Family;12:12 Bakery named;13:13 for _p;14:14 an _q;15:15 undisclosed _v_to;15:15 undisclosed _un-_a_rvrs;0:0 The _q', '1:1 said _v_to;2:2 the _q;3:3 sales _n_of;4:4 were _v_id;5:7 _q part a;8:8 1983 yofc;9:10 named Federal Trade;11:12 named Commission Consent;1:1 said _v_to', '0:0 A _q;1:1 year _n_1;2:2 earlier _a_1;3:3 Flowers named;4:4 had _v_1;5:5 fiscal _a_1;7:7 quarter _n_temp;7:7 quarter ord;8:9 _n_1 net income;10:10 of _p;11:11 $ _n_1;12:13 card 8 million;14:14 or _c;15:15 23 card;16:16 cents _n_1;17:17 a _p_per;18:18 share _n_of;19:19 on _p;20:20 revenue _n_1;21:21 of _p;22:22 $ _n_1;23:24 card 170.4 million', '1:1 steel _n_1;1:1 steel _raw_a_1;2:2 production _n_of;3:3 by _p;4:4 the _q;5:5 nation _n_of;7:7 mills _n_1;8:8 decreased _v_1;9:9 0.8 card;10:10 % _n_of;11:11 last _a_1;12:12 week _n_1;13:13 to _p;14:14 1,828,000 card;15:15 tons _n_1;16:16 from _p;17:17 1,843,000 card;18:18 tons _n_1;19:19 the _q;20:20 previous _a_1;21:21 week _n_1;22:22 the _q;23:24 named American Iron;25:25 and _c;26:27 named Steel Institute;1:1 steel _n_1', '0:0 Last _a_1;1:1 week _n_1;3:3 output _n_of;4:4 rose _v_1;5:5 1.4 card;6:6 % _n_of;7:7 from _p;8:8 the _q;9:9 1,802,000 card;10:10 tons _n_1;11:11 produced _v_1;12:12 a _q;13:13 year _n_1;0:0 Last _a_1', '0:0 The _q;1:1 industry _n_1;2:2 used _v_1;3:3 82.2 card;4:4 % _n_of;5:5 of _p;7:7 capability _n_to;8:8 last _a_1;9:9 week _n_1;10:10 compared _v_with;12:12 82.8 card;13:13 % _n_of;14:14 the _q;15:15 previous _a_1;16:16 week _n_1;17:17 and _c;18:18 84 card;19:19 % _n_of;20:20 a card;21:21 year _n_1;0:0 The _q', '0:0 The _q;1:1 capability _n_to;2:2 utilization _n_1;3:3 rate _n_of;4:4 is _v_id;5:5 a _q;6:6 calculation _n_of;7:9 _v_1 designed indicate;10:10 at _p;11:11 what _q;12:15 _n_of percent production;16:16 capability _n_to;17:17 the _q;18:18 industry _n_1;20:20 operating _v_1;21:21 in _p_temp;22:22 a _q;23:23 given _v_1;0:0 The _q', '0:1 named Selwyn B;2:2 Kossuth named;4:4 named _v_1;5:5 executive _n_1;6:6 director _n_of;8:8 the _q;9:9 commission _n_of;10:11 _a_1 effective early;0:0 Selwyn named', '0:0 Mr _n_1;1:1 Kossuth named;2:2 52 card;3:3 years _n_1;4:4 old _a_1;5:5 succeeds _v_1;6:7 named Ermanno Pascutto;8:8 36 card;10:10 resigned _v_1;11:11 to _x;12:12 join _v_1;13:14 named Hong Kong;16:16 Securities named;17:17 and _c;18:19 named Futures Commission', '0:0 Mr _n_1;1:1 Kossuth named;2:2 was _v_id;3:3 vice _n_1;4:4 president _n_of;5:5 and _c;6:6 director _n_of;7:7 corporate _a_1;8:8 finance _n_1;9:9 of _p;10:11 named Nesbitt Thomson;12:12 Deacon named;13:13 Inc _n_1;14:14 a _q;15:15 Toronto named;16:17 _n_1 investment dealer', '0:0 Dun named;1:1 & _c;2:2 Bradstreet named;3:3 Corp _n_1;5:6 named Market Data;7:7 Retrieval named;8:8 unit _n_of;9:9 said _v_to;11:11 acquired _v_1;12:12 School named;13:13 and _c;14:15 named College Construction;16:16 Reports named;17:17 service _n_1;18:18 from _p;19:19 Intelligence _n_1;20:20 for _p;21:21 Education named;0:0 Dun named', '0:0 Terms _n_of;0:0 Terms _n_of', '0:0 The _q;1:1 service _n_1;2:2 supplies _v_1;3:3 weekly _a_1;4:4 reports _n_of;5:5 on _p;6:6 school _n_1;7:7 and _c;8:9 _n_of college construction;0:0 The _q', '0:1 named Market Data;2:2 Retrieval named;3:3 is _v_id;4:4 a _q;5:5 compiler _n_unknown;6:6 of _p;7:7 educational _a_1;8:8 information _n_on-about;9:9 and _c;10:10 provides _v_1;11:11 related _v_to;0:0 Market named', '0:0 Closely _a_to;1:1 held _v_1;2:2 Intelligence _n_1;3:3 in _p;4:4 Education named;5:5 of _p;6:7 named Larchmont N.Y;8:8 is _v_id;9:9 an _q;10:10 educational _a_1;11:11 publisher _n_1;12:12 and _c;0:0 Closely _a_to', '0:0 A _q;1:1 battle _n_1;3:3 raging _v_1;4:4 in _p;5:5 Venice named;6:6 over _p;7:7 plans _n_1;9:9 have _v_cause;10:10 the _q;11:11 1,200- _n_unknown;12:12 year- _n_1;13:14 _a_1 old Italian;15:15 city _n_1;16:16 be _v_id;17:17 the _q;18:18 site _n_1;19:19 for _p;20:20 a _q;21:21 universal _a_1;22:22 exposition _n_of;23:23 in _p_temp;0:0 A _q', '0:0 The _q;1:1 plans _n_1;2:2 include _v_1;3:3 a _q;4:4 subway _n_1;5:5 system _n_of;6:6 a _q;7:7 congress _n_1;8:8 center _n_of;9:9 floating _v_1;10:10 trees _n_of;11:11 fanciful _a_unknown;12:12 fountains _n_1;14:14 and _c;18:18 60,000 card;19:19 additional _a_1;20:20 tourists _n_1;21:21 a _p_per;0:0 The _q', '0:0 But _c;1:1 opponents _n_1;2:2 fear _v_1;0:0 But _c', '0:0 Three card;1:1 gambling _v_1;2:2 casinos _n_1;4:4 opened _v_1;5:5 in _p;0:0 Three card', '0:0 The _q;1:1 three card;2:2 establishments _n_of;4:4 two card;5:5 in _p;6:6 Warsaw named;7:7 and _c;8:8 one card;9:9 in _p;10:10 Krakow named;12:12 accept _v_1;13:13 only _x_deg;14:14 foreign _a_1;15:15 currency _n_1;16:16 and _c;17:17 are _v_id;18:18 joint _a_1;19:19 ventures _n_1;20:20 between _p;21:21 Polish _a_1;22:22 firms _n_1;23:23 and _c;24:24 Western named;0:0 The _q', '0:0 Not _x_deg;1:1 all _q;2:2 Poles named;0:0 Not _x_deg', '3:3 want _v_1;4:4 casinos _n_1;5:5 for _p;6:6 when _x_subord;9:9 got _v_1;10:10 anything _q;11:11 in _p;12:12 the _q;13:13 shops?" _n_of;14:14 one card;15:15 housewife _n_unknown;3:3 want _v_1', '0:0 But _c;1:2 named Bogdan Gumkowski;4:4 runs _v_1;5:5 the _q;6:6 casino _n_1;7:7 at _p;8:10 named Warsaw Marriott;11:11 Hotel named;12:12 said _v_to;13:13 the _q;14:14 ventures _n_1;15:15 would _v_modal;16:16 help _v_to;17:17 Poland named;18:18 service _v_1;20:20 $ _n_1;21:22 card 39 billion;23:23 foreign _a_1;24:24 debt _n_1;25:25 by _p_means;26:26 pouring _v_1;27:27 dollars _n_1;28:28 into _p;29:29 the _q;30:30 state _n_of;31:31 firms _n_1;32:32 in _p;33:33 the _q;34:34 joint _a_1;35:35 ventures _n_1;37:37 the _q;38:38 LOT named;39:39 airline _n_1;40:40 and _c;41:41 Orbis named;42:43 _n_1 tourist organization', '0:0 Algeria named;1:1 plans _v_1;3:3 increase _v_cause;5:5 gas _n_1;5:5 gas _natural_a_for;6:6 sales _n_of;7:7 to _p;8:8 Europe named;9:9 and _c;10:10 the _q;0:0 Algeria named', '0:1 According to _p;2:2 the _q;3:4 named Middle East;5:6 named Economic Survey;7:7 the _q;8:9 named North African;10:10 nation _n_of;12:12 holding _v_1;13:13 talks _n_of-on;14:14 with _p;15:15 Italy named;16:16 for _p;17:17 adding _v_to;18:18 a _q;19:19 fourth ord;20:20 pipe _n_1;22:22 a _q;23:23 section _n_of;25:25 the _q;26:26 Trans- named;27:27 Mediterranean named_n;28:28 pipeline _n_1;29:29 expanding _v_cause;30:30 capacity _n_1;31:31 by _p;32:33 up to _x_deg;34:35 card six billion;36:36 cubic _a_1;37:37 meters _n_1;38:38 a _p_per;39:39 year _n_1;40:40 from _p;41:42 card 12.5 billion', '0:0 Algeria named;1:1 also _a_1;2:4 _v_1 wants build;5:5 a _q;6:6 pipeline _n_1;7:7 through _p;8:8 Morocco named;9:9 and _c;10:10 across _p;11:11 the _q;12:12 Strait named;13:13 of _p;14:14 Gibraltar named;15:15 to _x;16:16 supply _v_1;17:18 named Spain France;19:19 and _c;20:21 named West Germany;22:22 with _p;23:24 up to _x_deg;25:26 card 15 billion;27:27 cubic _a_1;28:28 meters _n_1;29:29 a _p_per;30:30 year _n_1;31:31 by _p;32:32 the _q;33:33 late _a_for;0:0 Algeria named', '0:1 named South Africa;3:4 named National Union;5:5 of _p;6:6 Mineworkers named;7:9 _v_1 agreed suspend;10:10 the _q;11:11 strike _n_1;12:12 by _p;13:14 _n_1 diamond workers;15:15 and _c;16:16 resume _v_1;17:17 negotiations _n_of;18:18 with _p;19:20 named De Beers;21:22 named Consolidated Mines;23:23 Ltd _n_1;24:24 over _p;26:27 _n_1 wage dispute;28:29 named De Beers;0:0 South named', '1:1 also _a_1;2:2 said _v_to;3:3 the _q;4:4 union _n_of;6:8 _v_1 agreed meet;9:9 the _q;10:10 company _n_of;11:11 for _p;12:12 further _a_1;13:13 talks _n_of-on;1:1 also _a_1', '0:0 The _q;1:1 strike _n_1;2:2 at _p;3:3 five card;4:5 named De Beers;6:6 mines _n_of;7:7 began _v_1;8:8 last _a_1;9:9 Thursday dofw;10:10 with _p;11:11 9,500 card;12:13 out of _p;14:14 a _q;15:15 total _a_1;16:16 10,000 card;17:17 NUM named;18:18 members _n_of;19:19 employed _v_1;20:20 on _p;21:22 named De Beers;23:23 mines _n_of;24:24 participating _v_in;25:26 according to _p;27:27 the _q;28:28 union _n_of;29:29 while _x;30:31 named De Beers;32:32 said _v_to;34:34 were _v_there;35:35 7,800 card;0:0 The _q', '0:0 The _q;1:1 union _n_of;3:3 demanded _v_of;4:4 a _q;5:5 37.6 card;6:6 % _n_of;7:7 increase _n_1;8:8 in _p;9:9 the _q;10:10 minimum _a_1;11:11 wage _n_1;12:12 while _x;13:14 named De Beers;16:16 final _a_1;17:17 offer _n_1;18:18 was _v_id;19:19 an _q;20:20 increase _n_1;21:21 of _p;22:22 17 card;0:0 The _q', '0:0 A _q;2:2 nation _n_of;2:2 nation card;3:3 environmental _a_1;4:4 conference _n_of;5:5 opened _v_1;6:6 in _p;7:8 named Sofia Bulgaria', '0:0 The _q;1:1 gathering _n_of;3:3 expected _v_1;5:5 focus _v_on;7:7 curbing _v_1;8:8 the _q;9:9 fouling _v_1;11:11 rivers _n_of;12:12 and _c;13:13 lakes _n_1;14:14 limiting _v_1;15:15 damage _n_to;16:16 from _p;17:17 industrial _a_1;18:18 accidents _n_1;19:19 and _c;20:20 improving _v_cause;21:21 the _q;22:22 handling _v_1;24:24 harmful _a_to;0:0 The _q', '0:0 West named;1:1 German _a_1;2:2 Environment named;3:3 Minister _n_of;4:5 named Klaus Toepfer;6:6 said _v_to;7:7 Bonn named;9:9 convinced _v_of;11:11 the _q;12:12 need _n_of;13:13 for _p;14:14 cooperation _n_1;15:15 "especially _x_deg;16:16 with _p;18:18 neighbors _n_1;19:19 in _p;20:20 the _q;21:21 East named;22:22 because _x;25:25 directly _a_1;26:26 affected _v_1;29:29 ecological _a_1;30:30 progress _n_1;31:31 or _c;32:32 lack _n_1;0:0 West named', '0:0 The _q;1:1 U.S named_n;2:2 and _c;3:3 Canada named;4:4 joined _v_1;5:5 every _q;6:6 European _a_1;7:7 country _n_of;8:8 except _p;9:9 Albania named;10:10 at _p;11:11 the _q;0:0 The _q', '0:0 The _q;1:1 Swedish _a_1;2:2 publishers _n_1;4:4 a _q;5:5 new _a_1;6:6 Estonian- named;7:7 language _n_1;8:8 newspaper _n_of;9:9 rushed _v_1;10:10 an _q;11:11 extra _a_1;12:12 edition _n_1;13:13 across _p;14:14 the _q;15:15 Baltic named_n;16:16 on _p_temp;17:17 Oct mofy;18:18 10 dofm;19:19 after _x_h;20:20 the _q;21:21 first ord;22:22 run _n_of;23:23 sold _v_out;25:25 in _p;26:26 one card;0:0 The _q', '0:1 named Editor Hasse;2:2 Olsson named;3:3 said _v_to;4:4 plans _n_1;6:6 called _v_for;8:8 7,000 card;9:9 copies _n_of;11:11 the _q;12:12 monthly _a_1;13:14 named Are Paev;15:16 named (Business Paper);19:19 sold _v_1;20:20 at _p;21:21 newsstands _n_unknown;22:22 and _c;23:24 an additional _q;25:25 3,000 card;26:26 promotion _n_of;27:27 issues _n_1;30:30 sent _v_1;32:32 direct _a_1;0:0 Editor named', '1:1 said _v_to;2:2 13,000 card;4:4 copies _n_of;6:6 sent _v_1;7:7 to _p;8:8 Estonia named;9:10 because of _p;11:11 strong _a_1;1:1 said _v_to', '0:0 The _q;1:1 Swedish _a_1;2:2 publishing _v_1;3:3 company _n_of;4:4 Bonniers named;5:5 owns _v_1;6:6 51 card;7:7 % _n_of;8:8 of _p;9:10 named Are Paev;11:11 and _c;12:12 the _q;13:13 Estonian _a_unknown;14:14 management _n_1;15:15 company _n_of;16:16 Minor named;17:17 owns _v_1;18:18 49 card;0:0 The _q', '0:1 named Angel Gurria;2:2 Mexico named;4:4 top _a_1;5:5 debt _n_1;6:6 negotiator _n_unknown;7:7 said _v_to;8:8 the _q;9:9 country _n_of;11:11 creditor _n_1;12:12 banks _n_of;14:14 responding _v_to;15:15 positively _a_1;17:17 Mexico named;19:19 debt- _n_1;20:21 _n_of reduction package', '0:0 Mr _n_1;1:1 Gurria named;3:3 optimism _n_1;4:4 contrasts _v_with;6:6 some _q;7:7 bankers _n_1;9:9 views _n_of;11:11 the _q;12:12 deal _n_1;13:13 may _v_modal;14:14 require _v_of;15:15 a _q;16:16 lot _n_of;18:18 arm _n_1;19:19 twisting _v_1;20:20 by _p;21:21 the _q;22:22 U.S named_n;23:23 Treasury named;24:25 in order _in+order+to_x;0:0 Mr _n_1', '0:0 An _q;1:1 increasing _v_1;2:4 _n_of number banks;5:5 appear _v_to;8:8 considering _v_1;9:9 the _q;10:10 option _n_1;11:11 under _p;12:12 the _q;13:13 deal _n_1;16:16 can _v_modal;17:17 swap _v_for;19:19 Mexican _a_1;20:23 _n_1 loans year;23:23 year card;24:24 bonds _n_1;25:25 with _p;26:26 a _q;27:28 _n_1 face value;29:29 discounted _v_1;31:31 35 card;32:32 % _n_of;33:33 Mr _n_1;34:34 Gurria named;0:0 An _q', '0:0 The _q;1:1 other _a_1;2:2 two card;3:3 options _n_1;4:4 consist _v_of;6:6 swapping _v_for;7:9 _n_1 loans bonds;10:10 with _p;11:11 6.25 card;12:12 % _n_of;13:13 interest _n_in;14:14 rates _n_of;15:15 or _c;16:16 providing _v_1;17:17 fresh _a_1;0:0 The _q', '0:0 The _q;1:1 accord _n_1;3:3 covers _v_1;4:4 $ _n_1;5:6 card 52.7 billion;7:7 of _p;8:8 Mexico named;10:10 medium- _n_1;11:11 and _c;13:13 term _n_of;13:13 term _long_a_1;14:14 debt _n_1;16:18 _v_1 expected go;19:19 into _p;20:20 effect _n_1;21:21 in _p_dir;0:0 The _q', '0:0 China named;2:2 top _a_1;3:4 _n_1 film actress;5:6 named Liu Xiaoqing;7:7 paid _v_for;8:8 $ _n_1;9:9 4,555 card;10:10 in _p;11:12 _n_1 back taxes;13:13 and _c;14:14 fines _n_1;15:15 in _p;16:16 Shandong named;17:17 province _n_1;18:18 the _q;19:21 named People Daily;0:0 China named', '0:0 China named;2:2 spend _v_1;3:3 $ _n_1;4:5 card 9.45 million;6:6 for _p;7:7 urgent _a_for;8:8 maintenance _n_1;9:9 on _p;10:12 named Tibet Potala;13:13 Palace named;14:14 former _a_1;15:15 home _n_of-n;16:16 of _p;17:17 the _q;18:19 named Dalai Lama;20:20 the _q;21:22 named China News;23:23 Service named;0:0 China named', '0:0 The _q;1:2 named Dalai Lama;5:5 just _a_1;6:6 awarded _v_1;7:7 the _q;8:9 named Nobel Peace;10:10 Prize named;11:11 lives _v_1;12:12 in _p;13:13 exile _n_1;14:14 in _p;0:0 The _q', '0:1 named George W;2:2 Koch named;3:3 63 card;4:4 years _n_1;5:5 old _a_1;6:6 president _n_of;7:7 and _c;8:9 _n_1 chief executive;10:10 officer _n_1;11:11 of _p;12:13 named Grocery Manufacturers;14:14 of _p;15:16 named America Inc;18:18 elected _v_1;19:19 a _q;20:20 director _n_of;22:22 this _q_dem;23:23 maker _n_of;25:26 _n_1 spices seasonings;27:27 and _c;28:29 _n_1 specialty foods;30:30 succeeding _v_1;31:32 named Erskin N;33:34 named White Jr;35:35 65 card;0:0 George named', '0:1 named American Business;2:2 Computer named;3:3 Corp _n_1;4:4 said _v_to;6:6 privately _a_1;7:7 placed _v_1;8:8 1,035,000 card;9:9 common _a_for;10:10 shares _n_of;11:11 at _p;12:12 $ _n_1;13:13 2.50 card;14:14 a _p_per;0:0 American named', '0:0 The _q;1:1 placement _n_of;3:3 made _v_1;4:4 through _p;5:6 named Gray Seifert;7:8 named Securities New;9:9 York named;10:10 to _p;11:11 institutional _a_1;0:0 The _q', '0:0 Proceeds _n_1;3:3 used _v_1;5:5 commercialize _v_unknown;6:6 recently _a_1;7:7 patented _v_1;8:8 technology _n_1;9:9 and _c;10:10 support _v_1;11:11 the _q;12:12 company _n_of;14:14 international _a_1;0:0 Proceeds _n_1', '0:0 The _q;1:1 company _n_of;2:2 develops _v_cause;3:3 and _c;4:4 markets _v_1;5:5 products _n_1;6:6 for _p;7:7 the _q;8:9 _n_1 food service;0:0 The _q', '0:0 THE _q;1:2 named R.H MACY;3:5 _n_1 & CO department-;6:6 store _n_1;7:7 chain _n_of;0:0 THE _q', '0:0 In _p_state;1:1 yesterday _a_1;3:3 edition _n_1;6:6 incorrectly _a_1;7:7 included _v_1;8:8 with _p;9:9 a _q;10:10 list _n_of;12:13 named New York;14:14 chains _n_of;15:16 _p up for;0:0 In _p_state', '0:0 Korean _a_1;1:1 car _n_1;2:2 exports _n_of;4:4 slid _v_1;5:5 about _x_deg;6:6 40 card;7:7 % _n_of;9:9 far _a_1;10:10 this _q_dem;11:11 year _n_1;12:12 but _c;13:13 auto _n_1;14:14 makers _n_of;15:15 here _a_1;0:0 Korean _a_1', '2:2 enjoying _v_1;3:3 domestic _a_1;4:4 sales _n_of;7:8 more than _a_1;9:9 making _v_up-for;11:11 for _p;12:12 lost _v_1;13:13 overseas _a_1;2:2 enjoying _v_1', '0:0 South named;1:1 Korean _a_1;2:2 consumers _n_of;4:6 _v_1 expected buy;7:7 almost _x_deg;8:8 500,000 card;9:10 _n_1 passenger cars;11:11 this _q_dem;12:12 year _n_1;13:13 up _p;14:14 60 card;15:15 % _n_of;16:16 from _p;0:0 South named', "0:1 In fact _a_1;2:2 some _q;3:4 _n_1 auto executives;5:5 suggest _v_to;7:7 slackened _v_1;8:8 demand _n_1;9:9 for _p;11:11 cars _n_1;12:12 in _p;13:13 the _q;14:14 U.S named_n;15:15 and _c;16:16 Canada named;17:17 is _v_id;18:18 a _q;19:19 blessing; _n_1;20:20 otherwise _a_1;22:22 wouldn't _v_modal;24:24 able _a_1;26:26 keep _v_up;28:28 with _p;29:29 demand _n_1;30:30 in _p;31:31 the _q;33:33 profitable _a_for;34:34 local _a_1;0:1 In fact _a_1", '2:2 very _x_deg;3:3 lucky _a_1;5:5 easily _a_for;6:6 change _v_from-to;7:7 an _q;8:9 _n_of export loss;11:11 domestic _a_1;12:12 plus," card;13:13 says _v_to;14:15 named Hong Tu;16:16 Pyo named;17:17 managing _v_1;18:18 director _n_of;20:20 domestic _a_1;21:21 marketing _v_1;22:22 for _p;23:24 named Hyundai Motor;2:2 very _x_deg', '0:0 As _x_subord;3:3 waiting _v_1;4:4 lists _n_of;6:6 a card;7:7 month _n_1;9:9 unusual _a_about;10:10 for _p;11:11 popular _a_for;0:0 As _x_subord', '0:0 Demand _n_1;3:3 strong _a_1;5:7 _q all the;8:8 domestic _a_1;9:9 makers _n_of;11:12 named Hyundai Kia;13:13 Motors named;14:14 Corp _n_1;15:16 named Daewoo Motor;17:17 Co _n_1;18:18 and _c;19:19 even _x_deg;20:20 upstart _n_1;21:22 named SsangYong Motor;23:23 Co _n_1;25:27 _v_1 plan build;0:0 Demand _n_1', '0:1 _n_1 Industry analysts;2:2 predict _v_1;4:4 by _p_temp;5:5 1995 yofc;6:7 named South Korea;10:10 building _v_1;11:12 card three million;13:13 cars _n_1;14:14 a _p_per;15:15 year _n_1;17:17 about _x_deg;18:18 half _q;20:20 that _q_dem;21:21 for _p;0:0 Industry _n_1', '0:0 But _c;1:1 South named;2:2 Korean _a_1;3:3 auto _n_1;4:4 makers _n_of;6:6 confident _a_of-about;8:8 the _q;9:9 export _n_of;10:10 market _n_1;12:12 bounce _v_back;14:14 and _c;16:16 demand _n_1;17:17 in _p;18:18 Korea named;20:20 stay _v_prd;0:0 But _c', '0:0 Currently _a_1;1:1 only _x_deg;2:2 one card;3:3 in _p;4:4 38 card;5:5 South named;6:6 Koreans _n_1;7:7 owns _v_1;8:8 a _q;9:9 car _n_1;10:11 _p up from;12:12 one card;13:13 in _p_temp;14:14 200 yofc;15:15 a card;16:16 decade _n_1;0:0 Currently _a_1', '0:0 "In _p_temp;1:1 the _q;2:2 year _n_1;3:3 2000 yofc;6:6 be _v_id;7:7 one card;8:8 car _n_1;9:9 per _p;0:0 "In _p_temp', '0:0 At _p;1:1 that _q_dem;2:2 point _n_of;3:3 domestic _a_1;4:4 sales _n_of;6:6 slow _v_down;8:8 says _v_to;9:10 named Kim Yoon;11:11 Kwon named;12:12 director _n_of;14:14 marketing _v_1;15:15 for _p;16:17 named Daewoo Motor', '0:0 The _q;1:1 reason _n_for;3:3 the _q;4:4 tremendous _a_1;5:5 demand _n_1;7:7 simple _a_for;8:8 : _p_namely;9:10 named South Koreans;11:11 suddenly _a_1;12:12 have _v_1;13:14 a lot _x;0:0 The _q', '1:1 never _a_1;2:2 thought _v_1;4:4 \'d _v_modal;5:5 own _v_1;6:6 a _q;7:7 car," _n_1;8:8 says _v_to;9:9 Kwang named;10:10 Ok _n_1;11:11 Kyong named;13:13 just _a_1;14:14 bought _v_1;15:15 a _q;16:17 named Daewoo LeMans;18:18 on _p;19:19 a _q;21:21 year _n_1;21:21 year card;1:1 never _a_1', '1:1 and _c;3:3 husband _n_1;4:4 started _v_cause;5:5 a _q;6:6 small _a_1;7:7 printing _v_1;8:8 business _n_1;9:9 and _c;10:10 need _v_1;11:11 the _q;12:12 car _n_1;13:13 for _p;14:14 work _n_1;15:17 as well as _c;18:18 for _p;19:19 weekend _n_of;1:1 and _c', '0:1 _n_1 Pay raises;2:2 of _p;3:3 60 card;4:4 % _n_of;5:5 over _p;6:6 the _q;7:7 past _a_1;8:8 three card;9:9 years _n_1;11:11 given _v_1;13:13 South named;14:14 Koreans _n_1;15:15 the _q;16:16 money _n_1;18:18 enjoy _v_1;19:19 the _q;20:20 things _n_of-about;23:23 supplying _v_1;24:24 the _q;25:25 rest _n_of;27:27 the _q;0:0 Pay _n_1', '0:0 The _q;1:1 success _n_1;2:2 of _p;3:3 newcomer _n_1;4:5 named SsangYong Motor;6:6 shows _v_1;7:7 the _q;8:8 strength _n_1;9:9 of _p;10:10 the _q;11:12 _n_1 auto market;13:13 and _c;15:15 growing _v_1;0:0 The _q', '0:0 SsangYong named;1:2 _v_1 began making;3:3 variations _n_of;5:5 the _q;6:6 Jeep- named;6:6 Jeep- _a_1;8:8 "Korando" named;0:0 SsangYong named', '0:0 The _q;2:2 popular _a_for;3:3 style _n_of;4:4 is _v_id;5:5 the _q;6:6 stretched _v_1;7:7 "Family," named;9:9 resembles _v_1;10:10 a _q;11:12 named Ford Bronco;13:13 or _c;14:15 named Chevy Blazer', '0:0 The _q;2:2 wheel- _n_1;2:2 wheel- card;3:4 _n_1 drive vehicles;5:5 start _v_1;6:6 at _p;7:7 $ _n_1;8:8 15,000; card;9:9 a _q;10:10 Family named;11:11 can _v_modal;12:12 cost _v_1;13:13 over _x_deg;14:14 $ _n_1;0:0 The _q', '0:0 SsangYong named;2:2 has _v_1;3:3 only _a_1;4:4 about _x_deg;5:5 3 card;6:6 % _n_of;7:7 of _p;8:8 the _q;9:9 domestic _a_1;10:10 market _n_1;12:12 sell _v_1;13:13 about _x_deg;14:14 18,000 card;17:17 models _n_of;18:18 this _q_dem;19:19 year _n_1;20:20 twice _a_1;21:23 _p as as;24:24 last _a_1;0:0 SsangYong named', '1:1 sees _v_1;2:2 sales _n_of;3:3 rising _v_1;4:4 45 card;5:5 % _n_of;6:6 to _p;7:7 26,000 card;8:8 units _n_2;9:9 next _a_1;1:1 sees _v_1', '0:0 The _q;1:1 company _n_of;2:2 plans _v_1;4:4 expand _v_cause;5:6 _n_1 plant capacity;7:7 50 card;8:8 % _n_of;9:9 by _p_temp;0:0 The _q', '0:1 _p_temp By then;3:3 also _a_1;4:6 _v_1 hopes begin;7:7 producing _v_1;8:8 a _q;9:10 _n_1 passenger car;11:11 based _v_1;12:12 on _p;13:13 the _q;14:14 Volvo named;15:15 240 card;16:16 and _c;17:17 selling _v_1;18:18 for _p;19:19 about _x_deg;20:20 $ _n_1;0:0 By _p_temp', '0:0 Hyundai named;1:1 and _c;2:2 Daewoo named;3:3 seem _v_to;4:4 unconcerned _v_1;4:4 unconcerned _un-_a_rvrs;5:5 about _p;6:6 the _q;7:7 SsangYong named;8:8 threat _n_1;9:9 but _c;10:10 Kia named;11:11 the _q;12:12 scrappy _a_unknown;13:13 No.3 named;14:14 auto _n_1;15:15 maker _n_of;17:17 selling _v_1;19:19 wheel- _n_1;19:19 wheel- card;20:21 _n_1 drive vehicles;22:22 through _p;24:24 Asia named;0:0 Hyundai named', '1:3 _v_1 plans sell;4:4 1,700 card;5:5 units _n_2;6:6 in _p_temp;1:1 plans _v_1', '0:0 Kia named;1:1 the _q;2:3 _a_1 only Korean;4:4 car _n_1;5:5 maker _n_of;8:8 seen _v_1;10:10 overseas _a_1;11:11 sales _n_of;12:12 grow _v_1;13:13 in _p_temp;14:14 1989 yofc;15:15 aims _v_at;17:17 Korea named;19:19 common _a_for;0:0 Kia named', '1:1 advantage _n_1;3:3 been _v_id;4:4 the _q;5:5 peppy _a_unknown;6:6 little _a_1;7:7 Pride named;8:8 sold _v_1;9:9 as _p;10:10 the _q;11:12 named Ford Festiva;13:13 in _p;14:14 the _q;1:1 advantage _n_1', '0:1 Along with _p;2:2 two card;3:3 larger _a_1;4:4 models _n_of;5:5 the _q;6:6 company _n_of;7:7 claims _v_1;8:8 18 card;9:9 % _n_of;10:10 of _p;11:11 the _q;12:12 domestic _a_1;0:1 Along with _p', '0:1 named Ford Motor;2:2 Co _n_1;3:3 and _c;4:6 named Japan Mazda;7:7 Motor named;8:8 Corp _n_1;9:9 have _v_1;10:10 equity _n_1;11:11 interests _n_in;0:0 Ford named', '0:0 Kia named;1:1 is _v_id;2:2 the _q;4:4 aggressive _a_1;6:6 the _q;7:7 Korean _a_1;8:8 Big named;9:9 Three card;10:10 in _p;11:12 _v_1 offering financing', '0:0 Loans _n_1;1:1 for _p;3:3 long _a_1;5:5 five card;6:6 years _n_1;7:7 make _v_cause;8:8 the _q;9:9 cars _n_1;10:10 very _x_deg;11:11 accessible _a_1;12:12 with _p;13:13 monthly _a_1;14:14 payments _n_of;16:16 low _a_on;18:18 80,000 card;19:19 won _v_1;20:20 or _c;21:21 $ _n_1;0:0 Loans _n_1', '0:1 named Daewoo Motor;2:2 a _q;6:6 joint _a_1;7:7 venture _n_1;8:8 with _p;9:10 named General Motors;11:11 Corp _n_1;12:12 and _c;13:13 the _q;14:15 named Daewoo Group;16:16 conglomerate _n_unknown;17:17 is _v_id;18:18 the _q;19:19 only _a_1;20:20 auto _n_1;21:21 maker _n_of;23:23 appears _v_to;0:0 Daewoo named', '0:0 Shipments _n_of-to;3:3 Lemans named;4:4 to _p;5:7 named GM Pontiac;8:8 division _n_of;10:10 off _p;11:11 about _x_deg;12:12 65 card;13:13 % _n_of;14:14 from _p;15:15 a card;16:16 year _n_1;17:18 _p ago versus;19:19 a _q;20:20 44 card;21:21 % _n_of;22:22 decline _n_1;23:23 for _p;24:24 Hyundai named;25:25 and _c;26:26 an _q;27:27 18 card;28:28 % _n_of;29:29 increase _n_1;30:30 for _p;0:0 Shipments _n_of-to', '0:0 Moreover _a_1;1:1 Daewoo named;3:3 domestic _a_1;4:4 sales _n_of;6:6 grown _v_1;7:9 _a_1 half fast;11:11 sales _n_of;0:0 Moreover _a_1', '0:0 The _q;1:1 big _a_1;2:2 problem _n_of;3:3 for _p;4:4 Daewoo named;6:6 holds _v_1;7:7 about _x_deg;8:8 21 card;9:9 % _n_of;10:10 of _p;11:11 the _q;12:12 market _n_1;13:13 is _v_id;14:14 the _q;15:15 long _a_1;16:16 series _n_of;18:18 labor _n_1;19:19 disruptions _n_of;21:21 suffered _v_1;22:22 this _q_dem;0:0 The _q', '0:0 But _c;1:1 Daewoo named;3:3 expanding _v_1;0:0 But _c', '0:1 In fact _a_1;2:2 a _q;3:4 _n_of sister company;5:6 named Daewoo Shipbuilding;7:7 and _c;8:9 named Heavy Machinery;10:12 _v_1 plans build;13:13 240,000 card;14:14 minicars _n_unknown;15:15 by _p;16:16 the _q;18:18 1990s year_range;0:1 In fact _a_1', '0:0 Hyundai named;1:1 the _q;2:2 Korean _a_1;3:3 market _n_1;4:4 leader _n_of;5:5 with _p;6:6 a _q;7:7 58 card;8:9 _n_of % share;10:10 also _a_1;11:13 _v_1 plans jump;14:14 into _p;15:15 minicars _n_unknown;16:16 at _p_temp;17:17 the _q;18:18 same _a_as;0:0 Hyundai named', '1:1 has _v_1;2:2 a _q;3:3 similar _a_to;4:4 project _n_1;5:5 for _p;6:6 200,000 card;7:7 cars _n_1;8:8 a _p_per;1:1 has _v_1', '0:0 Kia named;2:3 _a_1 reportedly also;4:4 considering _v_1;5:6 such a _q;0:0 Kia named', '0:0 Even _x_deg;1:1 giant _n_1;2:3 named Samsung Group;5:5 rumored _v_1;6:6 in _p;7:7 the _q;8:8 Korean _a_1;9:9 press _n_1;12:12 considering _v_1;13:13 getting _v_state;14:14 into _p;15:15 the _q;16:16 auto- _n_1;17:17 making _v_1;18:18 business; _n_1;19:19 a _q;20:20 company _n_of;21:21 spokesman _n_1;22:22 had _v_1;23:23 no _q;0:0 Even _x_deg', '0:1 named Robert P;2:2 Bulseco named;3:3 44 card;4:4 years _n_1;5:5 old _a_1;7:7 named _v_1;8:8 president _n_of;9:9 and _c;10:10 chief _n_1;11:11 administrative _a_1;12:12 officer _n_1;13:13 of _p;14:14 this _q_dem;15:16 _a_1 regional commercial;0:0 Robert named', '0:0 Both _q;0:0 Both card;1:1 posts _n_of;0:0 Both _q', '0:1 named Robert Robie;2:2 51 card;4:4 named _v_1;5:5 to _p;6:6 the _q;7:7 new _a_1;8:8 positions _n_of;10:10 vice _n_1;11:11 chairman _n_of;12:12 and _c;13:14 _n_1 chief credit;0:0 Robert named', '1:1 skittish _a_unknown;2:2 mutual _a_1;3:4 _n_1 fund investors;5:5 picked _v_up;7:7 the _q;8:8 phone _n_1;9:9 yesterday _a_1;10:10 but _c;11:11 decided _v_1;14:14 cash _v_in;17:17 chips _n_1;1:1 skittish _a_unknown', '0:0 As _x_subord;1:1 the _q;2:3 _n_1 stock market;4:4 bounced _v_back;6:6 withdrawals _n_of;8:8 money _n_1;9:9 from _p;10:11 _n_1 stock funds;12:12 amounted _v_to;14:14 a _q;15:15 mere _a_1;16:16 trickle _n_unknown;17:17 compared _v_with;19:19 Black named;20:20 Monday dofw;22:22 investors _n_1;23:23 dumped _v_1;24:24 $ _n_1;25:26 card 2.3 billion;27:27 or _c;28:28 about _x_deg;29:29 2 card;30:30 % _n_of;31:31 of _p;32:33 _n_1 stock- fund;0:0 As _x_subord', '0:1 _n_1 Net outflows;2:2 from _p;3:3 Fidelity named;5:6 _n_1 stock funds;7:7 stood _v_1;8:8 at _p;11:11 $ _n_1;12:13 card 300 million;14:14 or _c;15:15 below _p;16:16 15 card;17:17 % _n_of;18:18 of _p;19:19 the _q;20:20 $ _n_1;21:22 card 2 billion;23:23 cash _n_1;24:24 position _n_of;25:25 of _p;26:26 the _q;27:29 _n_1 firm stock;0:0 Net _n_1', '2:2 the _q;3:3 money _n_1;5:5 switched _v_cause;6:6 into _p;7:7 the _q;8:10 _n_1 firm money;11:12 _n_1 market funds', '0:1 _a_1 Other mutual;2:2 fund _n_1;3:3 companies _n_of;4:4 reported _v_to;5:5 even _x_deg;6:6 lighter _a_1;7:7 withdrawal _n_of;0:0 Other _a_1', '0:0 And _c;1:1 some _q;2:2 investors _n_1;3:3 at _p;4:4 Fidelity named;5:5 and _c;6:7 _a_1 elsewhere even;8:9 _v_1 began buying;10:11 _n_1 stock funds;12:12 during _p;13:13 the _q;0:0 And _c', '0:0 "Two card;1:1 years _n_1;2:2 ago _p;4:4 was _v_there;5:5 a _q;6:6 lot _n_of;8:9 _n_1 redemption activity;10:10 and _c;11:11 trouble _n_1;12:12 with _p;13:13 people _n_of;14:14 getting _v_through;16:16 on _p;17:17 the _q;18:18 phone," _n_1;19:19 said _v_to;20:21 named Kathryn McGrath;22:22 head _n_of;24:24 the _q;25:26 _n_1 investment management;27:27 division _n_of;29:29 the _q;30:30 Securities named;31:31 and _c;32:33 named Exchange Commission', '0:0 This _q_dem;1:1 time _n_of;4:4 have _v_1;5:5 that _q_dem;0:0 This _q_dem', '0:1 Of course _a_1;2:2 the _q;3:3 relative _a_to;4:4 calm _n_1;5:5 could _v_modal;7:7 jolted _v_1;8:8 if _x_then;9:9 the _q;10:10 market _n_1;11:11 plunges _v_into;0:1 Of course _a_1', '0:0 And _c;1:1 any _q;2:2 strong _a_1;3:3 surge _n_1;4:4 in _p;5:5 redemptions _n_1;6:6 could _v_modal;7:7 force _v_1;8:8 some _q;9:9 funds _n_1;11:11 dump _v_1;12:12 stocks _n_1;13:13 to _x;14:14 raise _v_cause;15:15 cash _n_1;16:16 as _x_subord;17:17 some _q;19:19 during _p;20:20 Black named;0:0 And _c', '0:0 But _c;1:1 funds _n_1;2:2 generally _a_1;4:4 better _x_deg;5:5 prepared _a_1;6:6 this _q_dem;7:7 time _n_of;0:0 But _c', '0:0 As _p;1:1 a _q;2:2 group _n_of;4:4 cash _n_1;5:5 position _n_of;6:6 of _p;7:7 10.2 card;8:8 % _n_of;9:9 of _p;10:10 assets _n_1;11:11 in _p_temp;12:12 August mofy;14:14 the _q;15:15 latest _a_for;16:16 figure _n_1;17:17 available _a_1;20:20 14 card;21:21 % _n_of;22:22 higher _a_1;24:24 two card;25:25 years _n_1;0:0 As _p', '1:1 fund _n_1;2:2 managers _n_of;4:4 boosted _v_to;6:7 _n_1 cash levels;8:8 in _p_temp;9:9 recent _a_1;1:1 fund _n_1', '0:0 The _q;1:1 biggest _a_1;2:2 flurry _n_of;4:5 _n_1 investor activity;6:6 came _v_1;7:7 early _a_1;8:8 in _p_temp;9:9 the _q;0:0 The _q', '0:1 named Vanguard Group;2:2 Inc _n_1;3:3 saw _v_1;4:4 heavy _a_1;5:5 exchanges _n_1;6:6 from _p;7:8 _n_1 stock funds;9:9 into _p;10:11 _n_1 money market;12:12 funds _n_1;13:13 after _x_h;14:14 the _q;15:15 telephone _n_1;16:16 lines _n_of;17:17 opened _v_1;18:18 at _p_temp;19:19 8:30 numbered_hour;0:0 Vanguard named', '0:0 "In _p_temp;1:1 the _q;2:2 first ord;3:3 hour _n_1;4:4 the _q;5:5 real _x_deg;6:6 nervous _a_about;7:7 folks _n_1;8:8 came _v_along;10:10 a _q;11:11 spokesman _n_1;0:0 "In _p_temp', '0:0 "But _c;1:1 the _q;2:2 horrendous _a_to;3:3 pace _n_1;4:4 of _p;5:5 call _n_1;6:6 volume _n_of;7:7 in _p_temp;8:8 the _q;9:9 first ord;10:11 half- hour _n_1;12:12 slowed _v_1;0:0 "But _c', '0:0 At _p;1:2 named Scudder Stevens;3:3 & _c;4:5 named Clark Inc;6:7 _n_1 phone calls;8:8 came _v_in;10:10 at _p;11:11 40 card;12:12 % _n_of;13:14 more than _p;15:15 the _q;16:16 normal _a_1;17:17 pace _n_1;18:18 through _p;19:19 early _a_1;0:0 At _p', '0:0 Most _q;2:2 that _q_dem;3:3 increase _n_1;4:4 came _v_1;5:5 in _p_temp;6:6 the _q;7:7 first ord;8:8 hour _n_1;9:9 after _p;10:10 the _q;11:11 phone _n_1;12:12 lines _n_of;13:13 opened _v_cause;14:14 at _p_temp;15:15 8 numbered_hour;0:0 Most _q', '0:0 As _x_subord;1:1 stocks _n_1;2:2 rose _v_1;3:4 in fact _a_1;5:5 some _q;6:6 investors _n_1;7:7 changed _v_cause;8:8 course _n_i;9:9 and _c;10:10 reversed _v_1;12:12 sell _n_1;0:0 As _x_subord', '1:1 funds _n_1;2:2 allow _v_1;3:3 investors _n_1;5:5 void _v_1;6:6 orders _n_of;7:7 before _p;8:8 the _q;9:9 close _n_of;1:1 funds _n_1', '0:0 Because _x;1:1 mutual _a_1;2:2 fund _n_1;3:3 trades _n_of;5:5 take _v_1;6:6 effect _n_1;7:7 until _p;8:8 the _q;9:9 market _n_1;10:10 close _n_of;12:12 in _p;13:13 this _q_dem;14:14 case _n_of;15:15 at _p_temp;16:16 4 numbered_hour;17:17 p.m _x;19:19 these _q_dem;20:20 shareholders _n_1;21:21 effectively _a_1;22:22 stayed _v_prd;0:0 Because _x', '0:0 At _p;1:1 Fidelity named;3:3 office _n_of;4:4 in _p;5:5 downtown _a_1;6:7 named Boston Gerald;8:8 Sherman named;9:9 walked _v_1;10:10 in _p_dir;11:11 shortly _x_deg;12:12 after _p;13:13 7:30 numbered_hour;14:14 a.m _x;15:15 and _c;16:16 placed _v_1;17:17 an _q;18:18 order _n_of;20:20 switch _v_cause;22:23 _n_of retirement accounts;24:25 out of _p;26:26 three card;27:28 _n_1 stock funds;29:29 and _c;30:30 into _p;31:31 a _q;32:33 _n_1 money market;0:0 At _p', '1:1 \'s _v_id;2:2 a _q;3:3 nice _a_of-for;4:4 feeling _n_of;5:5 to _x;6:6 know _v_1;8:8 things _n_of-about;9:9 stabilized," _v_1;10:10 said _v_to;11:11 Mr _n_1;12:12 Sherman named;13:13 the _q;15:15 year- _n_1;15:15 year- card;16:16 old _a_1;17:17 co-owner _n_unknown;18:18 of _p;19:19 a _q;20:21 _n_1 discount department;1:1 \'s _v_id', '0:0 Shareholders _n_1;3:3 steadily _a_1;4:4 bailing _v_out+of;7:8 _a_1 several big;9:10 _n_1 junk funds;11:11 the _q;12:13 _a_1 past several;14:14 weeks _n_1;15:15 as _x_subord;16:16 the _q;17:17 $ _n_1;18:19 card 200 billion;20:20 market _n_1;22:22 jolted _v_1;24:24 a _q;25:26 _n_1 cash crunch;27:27 at _p;28:28 Campeau named;29:29 Corp _n_1;30:30 and _c;31:31 steadily _a_1;32:32 declining _v_1;0:0 Shareholders _n_1', '2:2 the _q;3:3 money _n_1;6:6 switched _v_cause;7:7 into _p;8:9 _n_1 money market;10:11 _n_1 funds fund;12:12 executives _n_1;2:2 the _q', '0:1 Instead of _p;2:2 selling _v_1;3:3 bonds _n_1;4:4 to _x;5:5 meet _v_1;6:6 redemptions _n_1;8:8 some _q;9:9 funds _n_1;11:11 borrowed _v_from;13:13 banks _n_of;14:14 to _x;15:15 meet _v_1;16:16 withdrawal _n_of;0:1 Instead of _p', '0:0 This _q_dem;1:1 avoids _v_1;2:2 knocking _v_down;4:4 prices _n_of;0:0 This _q_dem', '0:0 The _q;1:1 $ _n_1;2:3 card 1.1 billion;4:5 named T Rowe;6:7 named Price High;8:8 Yield named;9:9 Fund _n_1;11:11 among _p;12:12 the _q;13:13 funds _n_1;15:15 borrowed _v_from;16:16 during _p;17:17 the _q;18:18 Campeau named;19:19 crisis _n_1;20:20 says _v_to;21:22 named George J;23:23 Collins named;24:24 president _n_of;26:27 named T Rowe;28:29 named Price Associates;0:0 The _q', '0:0 That _q_dem;1:1 way _n_of;2:2 Mr _n_1;3:3 Collins named;4:4 says _v_to;7:7 have _v_qmodal;9:9 sell _v_1;10:10 securities _n_1;11:11 in _p;12:12 a _q;13:13 sloppy _a_about;0:0 That _q_dem', '0:0 When _x_subord;1:1 the _q;2:2 market _n_1;3:3 stabilized _v_1;5:5 added _v_to;6:6 the _q;7:7 firm _n_1;8:8 sold _v_1;9:9 the _q;10:10 bonds _n_1;11:11 and _c;12:12 quickly _a_1;13:13 paid _v_for;14:14 the _q;15:15 loans _n_1;0:0 When _x_subord', '0:1 named Tom Herman;2:2 contributed _v_to;4:4 this _q_dem;0:0 Tom named', '0:1 named Amcore Financial;2:2 Inc _n_1;3:3 said _v_to;5:7 _v_1 agreed acquire;8:8 Central named;9:9 of _p;10:10 Illinois named;11:11 Inc _n_1;12:12 in _p;13:13 a _q;14:14 stock _n_1;0:0 Amcore named', '0:0 Shareholders _n_1;1:1 of _p;2:2 Central named;3:3 a _q;4:4 bank _n_of;5:5 holding _n_1;6:6 company _n_of;7:7 based _v_1;8:8 in _p;9:10 named Sterling Ill;12:12 receive _v_1;13:13 Amcore named;14:14 stock _n_1;15:15 equal _a_to;17:17 10 card;18:18 times _n_1;19:19 Central named;21:21 1989 yofc;22:22 earnings _n_1;23:23 Amcore named;0:0 Shareholders _n_1', '0:0 For _p;1:1 the _q;2:2 first ord;3:3 nine card;4:4 months _n_1;5:5 of _p;6:6 1989 yofc;7:7 Central named;8:8 earned _v_1;9:9 $ _n_1;10:11 card 2 million', '0:0 Amcore named;1:1 also _a_1;2:2 a _q;3:3 bank _n_of;4:4 holding _n_1;5:5 company _n_of;6:6 has _v_1;7:7 assets _n_1;8:8 of _p;9:9 $ _n_1;10:11 card 1.06 billion', '0:0 Central named;2:2 assets _n_1;3:3 are _v_id;4:4 $ _n_1;5:6 card 240 million', '0:0 (During _p;2:3 _n_1 centennial year;4:4 The _q;5:6 named Wall Street;7:7 Journal named;9:9 report _v_to;10:10 events _n_item;11:11 of _p;12:12 the _q;13:13 past _a_1;14:14 century _n_1;16:16 stand _v_1;17:17 as _p;18:18 milestones _n_unknown;19:19 of _p;20:20 American _a_1;21:21 business _n_1;0:0 (During _p', '0:0 SOFT _a_1;1:2 _n_1 CONTACT LENSES;3:3 WON _v_1;4:4 federal _a_1;5:5 blessing _v_1;6:6 on _p_temp;7:7 March mofy;8:8 18 dofm;9:9 1971 yofc;10:10 and _c;11:11 quickly _a_1;12:12 became _v_id;13:14 _n_1 eye openers;15:15 for _p;0:0 SOFT _a_1', '0:0 The _q;1:1 Food named;2:2 and _c;3:4 named Drug Administration;5:5 that _q_dem;6:6 day _n_of;7:7 said _v_to;8:8 Bausch named;9:9 & _c;10:10 Lomb named;11:11 could _v_modal;12:13 _v_1 start selling;15:15 in _p;16:16 the _q;0:0 The _q', '0:0 The _q;1:1 cornflake- _n_unknown;2:3 _n_1 size product;6:6 comfortable _a_1;7:7 and _c;9:9 prone _a_to;11:11 falling _v_1;12:12 out _p_dir;14:14 hard _a_for;15:16 _n_1 contact lenses;20:21 _p around since;0:0 The _q', '0:0 Bausch named;1:1 & _c;2:2 Lomb named;3:3 sold _v_1;4:4 the _q;5:5 softies _n_unknown;6:6 under _p;7:7 a _q;8:8 sublicense _n_unknown;9:9 from _p;10:11 named National Patent;12:12 Development named;15:15 gained _v_1;16:16 the _q;17:17 rights _n_1;18:18 from _p;19:19 the _q;20:21 named Czechoslovakia Academy;22:22 of _p;0:0 Bausch named', '0:1 named Otto Wichterle;2:2 a _q;3:3 Czech named;4:4 invented _v_1;6:6 in _p_temp;0:0 Otto named', '0:0 The _q;1:1 plastic _a_1;2:2 lens _n_1;3:3 wraps _v_1;5:5 over _p;6:6 the _q;7:7 cornea _n_unknown;8:8 absorbing _v_1;9:10 _n_1 eye moisture;11:11 while _x;12:12 permitting _v_1;13:13 oxygen _n_1;15:15 pass _v_1;0:0 The _q', '0:0 But _c;1:1 the _q;2:2 new _a_1;3:3 lens _n_1;4:4 became _v_id;5:5 the _q;6:6 eye _n_1;7:7 of _p;8:8 a _q;0:0 But _c', '0:0 In _p_temp;1:1 September mofy;2:2 1971 yofc;3:3 California named;4:4 officials _n_1;5:5 seized _v_1;6:6 "bootlegged" _v_unknown;7:7 lenses _n_1;9:9 made _v_1;11:11 unlicensed _v_to;11:11 unlicensed _un-_a_rvrs;12:12 companies _n_of;14:14 after _x_h;15:15 some _q;16:16 showed _v_1;17:17 traces _n_of;0:0 In _p_temp', '0:0 In _p_temp;1:1 October mofy;2:2 doctors _n_1;4:4 debating _v_1;5:5 the _q;6:8 _n_1 product safety;9:9 some _q;10:12 _v_1 claiming caused;0:0 In _p_temp', '0:0 And _c;2:2 were _v_there;3:3 Senate named_n;4:4 hearings _n_1;5:5 on _p;6:6 the _q;7:7 questions _n_about;8:8 in _p_temp;9:9 July mofy;0:0 And _c', '0:0 The _q;1:1 product _n_1;2:2 overcame _v_1;3:3 the _q;4:4 bad _a_at;5:5 publicity _n_1;6:6 and _c;7:7 kept _v_on;0:0 The _q', '0:0 The _q;1:2 _a_1 early soft;3:3 lenses _n_1;5:5 cost _v_1;6:6 $ _n_1;7:7 300 card;8:8 a _p_per;9:9 set _n_of;11:13 _v_1 expected last;14:14 for _p;15:15 a card;0:0 The _q', '0:0 Eighteen card;1:1 months _n_1;2:2 ago _p;3:3 a _q;4:4 "disposable" _a_unknown;6:6 day _n_of;6:6 day card;7:7 model _n_of;8:8 bowed; _v_1;9:9 a card;10:12 _n_1 year supply;13:13 costs _v_1;14:14 about _x_deg;15:15 $ _n_1;0:0 Eighteen card', '0:0 Last _a_1;1:1 month _n_1;2:2 the _q;3:3 FDA named_n;4:4 and _c;5:6 named Contact Lens;7:7 Institute named;8:8 cautioned _v_1;9:9 users _n_of;11:11 serious _a_1;12:13 _n_1 eye infections;14:14 could _v_modal;15:15 result _v_from;17:17 wearing _v_1;18:18 lenses _n_1;19:20 more than _p;21:21 seven card;22:22 days _n_of;23:23 at _p;24:24 a _q;0:0 Last _a_1', '0:0 Today _a_1;1:2 card 20 million;4:4 the _q;5:6 card 25 million;7:7 Americans _n_1;8:8 using _v_1;9:10 _n_1 contact lenses;12:12 using _v_1;13:13 the _q;14:14 soft _a_1;0:0 Today _a_1', '0:0 Including _v_1;1:1 the _q;2:2 accesory _n_unknown;3:4 _n_1 eye care;5:6 _n_1 products contacts;7:7 account _v_for;9:9 $ _n_1;10:11 card 2 billion;12:12 in _p;13:13 annual _a_1;14:14 retail _n_1;0:0 Including _v_1', '0:0 Although _x;1:1 Bausch named;2:2 remains _v_1;3:3 the _q;4:4 leader _n_of;5:5 among _p;6:6 the _q;7:7 six card;8:8 majors _n_1;9:9 Johnson named;10:10 & _c;11:11 Johnson named;12:12 with _p;14:14 new _a_1;15:15 disposables _n_unknown;17:17 coming _v_on;0:0 Although _x', '0:0 The _q;1:2 _n_1 roller- coaster;3:4 _n_1 stock market;6:6 making _v_cause;7:7 life _n_of;8:8 tougher _a_for;9:9 for _p;10:10 small _a_1;11:11 companies _n_of;12:12 trying _v_1;14:14 raise _v_cause;0:0 The _q', '0:0 In _p_state;1:1 the _q;2:2 wake _n_1;3:3 of _p;4:4 Friday dofw;6:6 plunge _n_of;7:7 and _c;8:8 yesterday _a_1;10:10 rebound _n_1;11:11 some _q;12:12 companies _n_of;14:14 already _a_1;15:15 postponing _v_1;16:16 deals _n_1;17:17 and _c;18:18 others _n_1;19:19 wish _v_1;0:0 In _p_state', '0:0 As _x_prd;1:1 in _p_temp;2:2 other _a_1;3:3 jittery _a_unknown;4:4 times _n_of;6:6 small _a_1;7:7 businesses _n_1;8:8 expect _v_1;9:9 a _q;10:11 _a_1 particularly rough;12:12 time _n_of;13:13 raising _v_cause;14:14 funds _n_1;15:15 as _x_subord;16:16 investors _n_1;17:17 shun _v_1;18:18 risky _a_for;19:19 deals _n_1;20:20 seeking _v_1;21:21 safety _n_1;22:22 in _p;23:23 bigger _a_1;0:0 As _x_prd', '0:0 Even _x_deg;1:1 if _x_then;2:2 stock _n_1;3:3 prices _n_of;4:4 fully _a_of;5:5 recover _v_from;7:7 Friday dofw;9:9 sharp _a_1;10:10 decline _n_1;11:11 the _q;12:12 unsettled _a_1;13:13 conditions _n_1;15:15 frighten _v_1;0:0 Even _x_deg', '0:0 "The _q;1:1 implication _n_of;3:3 an _q;4:4 unsettled _a_1;5:5 situation _n_1;6:6 is _v_nv;8:8 the _q;9:9 thing _n_of-about;10:10 could _v_modal;11:11 drop _v_1;12:12 dramatically," _a_1;13:13 says _v_to;14:15 named Henry Linsert;16:16 Jr named;17:17 chairman _n_of;19:19 Martek named;20:20 Corp _n_1;21:21 a _q;23:23 year- _n_1;23:23 year- card;24:24 old _a_1;25:25 biotechnology _n_unknown;26:26 company _n_of;29:29 planning _v_1;30:30 a _q;31:31 private _a_1;32:32 placement _n_of;0:0 "The _q', '0:0 Earlier _a_1;1:1 this _q_dem;2:2 month _n_1;3:3 Staples named;4:4 Inc _n_1;5:5 a _q;6:7 named Newton Mass;8:8 office- _n_of;9:9 supplies _n_1;10:10 discounter _n_unknown;11:11 said _v_to;13:13 would _v_modal;14:14 accelerate _v_cause;15:16 _n_1 expansion plans;17:17 nationwide _a_1;18:18 and _c;19:19 offer _v_1;23:23 stock _n_1;25:25 the _q;0:0 Earlier _a_1', '0:0 At _p_temp;1:1 the _q;2:4 _n_of time shares;6:6 selling _v_1;7:7 above _p;9:9 initial _a_1;10:11 _n_of offering price;13:13 $ _n_1;14:14 19 card;15:15 and _c;16:16 bankers _n_1;17:17 believed _v_1;18:18 Staples named;19:19 would _v_modal;20:20 sell _v_1;21:21 new _a_1;22:22 stock _n_1;23:23 without _p;24:24 a _q;0:0 At _p_temp', '0:0 But _c;1:1 with _x_subord;2:2 the _q;3:5 _n_of company shares;6:6 standing _v_1;7:7 at _p;8:8 $ _n_1;9:9 15 card;10:10 yesterday _a_1;11:11 a _q;12:12 new _a_1;13:13 offering _n_of;14:14 seems _v_to;15:15 unlikely _a_for;16:16 company _n_of;17:17 officials _n_1;0:0 But _c', '0:0 Business _n_1;2:2 continues _v_2;5:5 "robust," _a_1;6:6 and _c;7:7 the _q;8:9 _n_1 stock market;11:11 affected _v_1;12:12 the _q;13:15 _n_1 concern expansion;16:16 plans _n_1;17:17 says _v_to;18:19 named Todd Krasnow;20:20 a _q;21:21 senior _a_1;0:0 Business _n_1', "0:0 Other _a_1;1:1 companies _n_of;2:2 figure _v_1;4:4 can't _v_modal;5:5 avoid _v_1;6:6 the _q;0:0 Other _a_1", '1:1 have _v_1;2:3 _n_1 capital requirements,";4:4 says _v_to;5:5 Mr _n_1;6:6 Linsert named;7:7 "so _x;9:9 have _v_qmodal;11:11 go _v_ahead;13:13 with _p;14:14 a _q;15:15 planned _v_1;16:16 $ _n_1;17:18 card 1.5 billion;19:19 private _a_1;1:1 have _v_1', '0:0 Unless _x;1:1 the _q;2:2 market _n_1;3:3 goes _v_1;4:4 right _x_deg;5:6 _p back up;8:8 says _v_to;10:10 may _v_modal;11:11 take _v_2;13:15 card six nine;16:16 months _n_1;18:18 find _v_1;19:19 the _q;20:20 money _n_1;21:22 instead of _p;0:0 Unless _x', '0:0 And _c;1:1 the _q;2:3 named Columbia Md;4:4 company _n_of;5:5 may _v_modal;6:6 have _v_qmodal;8:8 settle _v_1;9:9 for _p;10:10 a _q;11:11 lower _a_1;12:12 price _n_of;0:0 And _c', '0:0 Life _n_of;2:2 particularly _a_1;3:3 nerve- _n_1;4:4 racking _v_1;5:5 for _p;6:6 companies _n_of;9:9 planned _v_1;11:11 go _v_state;12:12 public _a_1;13:13 this _q_dem;0:0 Life _n_of', '0:0 Hand- _n_1;1:1 holding _v_1;2:3 _v_id is becoming;4:4 an _q;5:6 _n_1 investment- banking;7:7 job _n_of;0:0 Hand- _n_1', '0:1 named Robertson Stephens;2:3 & Co _n_1;4:4 a _q;5:6 named San Francisco;7:8 _n_1 investment banking;9:9 concern _n_1;10:10 has _v_1;11:11 a _q;12:12 client _n_1;14:14 looked _v_forward-to;17:17 making _v_1;19:20 _a_1 initial public;21:21 offering _n_of;0:0 Robertson named', '0:0 As _x_subord;1:1 the _q;2:2 market _n_1;3:3 dropped _v_1;4:4 Friday dofw;5:6 named Robertson Stephens;7:7 slashed _v_at;8:8 the _q;9:9 value _n_1;10:10 of _p;11:11 the _q;12:12 offering _n_of;13:13 by _p;14:14 7 card;0:0 As _x_subord', '0:0 Yesterday _a_1;1:1 when _x_subord;2:2 similar _a_to;3:3 securities _n_1;4:6 _v_1 rebounded bumped;7:7 the _q;8:8 valuation _n_1;9:9 up _p;0:0 Yesterday _a_1', '0:2 _p As of late;3:3 yesterday _a_1;4:4 the _q;5:5 IPO _n_unknown;7:7 still _a_1;0:1 As of _p', '0:0 For _p;2:2 the _q;3:3 situation _n_1;4:4 is _v_id;5:5 especially _x_deg;6:6 discouraging _v_1;7:7 because _x;8:8 the _q;9:9 market _n_1;10:10 for _p;11:11 IPOs _n_unknown;13:13 showing _v_1;14:14 signs _n_of;16:16 strengthening _v_1;17:17 after _p;18:18 several _a_1;19:19 years _n_1;20:20 of _p;0:0 For _p', '2:2 just _a_1;3:3 beginning _v_1;5:5 look _v_at;7:7 the _q;8:8 increase _n_1;9:9 in _p;10:10 IPOs _n_unknown;11:11 seeing _v_1;12:12 the _q;13:13 light _n_1;14:14 at _p;15:15 the _q;16:16 end _n_of;18:18 the _q;19:19 tunnel," _n_1;20:20 says _v_to;21:22 named Frank Kline;23:23 Jr named;24:24 partner _n_1;25:25 in _p;26:27 named Lambda Funds;28:28 a _q;29:30 named Beverly Hills;31:31 Calif named;32:33 _n_1 venture capital;2:2 just _a_1', '0:0 "But _c;1:1 the _q;2:2 tunnel _n_1;4:4 just _a_1;5:5 gotten _v_state;0:0 "But _c', '0:0 Companies _n_of;1:1 planning _v_1;3:3 go _v_state;4:6 _a_1 public definitely;7:7 taking _v_1;8:8 a _q;9:9 second ord;10:10 look," _n_at;11:11 says _v_to;12:13 named Allen Hadhazy;14:14 senior _a_1;15:15 analyst _n_1;16:16 at _p;17:17 the _q;18:18 Institute named;19:19 for _p;20:21 named Econometric Research;22:23 named Fort Lauderdale;24:24 Fla named;26:26 publishes _v_1;27:27 the _q;28:29 named New Issues;30:30 newsletter _n_of;31:31 on _p;0:0 Companies _n_of', '1:1 calculates _v_1;3:3 the _q;4:4 recent _a_1;5:6 _n_1 market slide;7:7 translated _v_into;9:9 a _q;10:10 5 card;11:11 % _n_of;13:13 7 card;14:15 _n_of % reduction;16:16 in _p;17:17 IPO _n_unknown;18:18 proceeds _n_1;19:19 to _p;1:1 calculates _v_1', '1:1 companies _n_of;1:1 companies _n_of', '0:0 Exabyte named;1:1 Corp _n_1;4:6 _v_1 planning sell;7:7 10 card;8:8 % _n_of;9:9 of _p;11:11 stock _n_1;12:12 this _q_dem;13:13 week _n_1;14:14 in _p;15:15 an _q;16:16 IPO _n_unknown;18:18 would _v_modal;19:19 raise _v_cause;20:21 up to _x_deg;22:22 $ _n_1;23:24 card 28.5 million', '0:0 But _c;1:1 now _a_1;2:3 named Peter Behrendt;4:4 president _n_of;5:5 says _v_to;8:8 making _v_1;9:9 decisions _n_1;10:10 on _p;11:11 a _q;12:14 day- to- day _a_1;0:0 But _c', '0:0 Debt- _n_1;1:1 free _a_1;2:2 and _c;3:3 profitable _a_for;4:4 the _q;5:6 named Boulder Colo;7:8 _n_1 computer- products;9:9 concern _n_1;10:10 could _v_modal;11:11 borrow _v_from;12:12 funds _n_1;13:13 if _x_then;15:15 decides _v_against;17:17 an _q;18:18 IPO _n_unknown;19:19 now _a_1;0:0 Debt- _n_1', '0:0 KnowledgeWare named;1:1 Inc _n_1;2:2 an _q;3:3 Atlanta named;4:5 _n_1 computer- software;6:6 concern _n_1;7:7 says _v_to;10:10 still _a_1;11:11 planning _v_1;13:13 go _v_ahead;15:15 with _p;17:17 IPO _n_unknown;18:18 this _q_dem;19:19 week _n_1;20:20 or _c;21:21 next _a_1;23:23 unless _x;24:24 conditions _n_1;0:0 KnowledgeWare named', '0:0 Delayed _v_1;1:1 financings _n_unknown;2:2 also _a_1;3:3 would _v_modal;4:4 affect _v_1;5:5 the _q;6:9 _n_of operations companies', '0:1 named Sierra Tucson;2:2 Cos _n_1;3:3 a _q;4:5 named Tucson Ariz;6:6 operator _n_of;8:8 addiction- _n_to;9:10 _n_of treatment centers;11:11 has _v_1;12:12 a _q;13:14 _v_1 planned doubling;15:15 of _p;16:16 capacity _n_1;17:17 riding _v_1;18:18 on _p;19:19 an _q;20:20 IPO _n_unknown;21:21 scheduled _v_1;22:22 for _p;23:23 next _a_1;0:0 Sierra named', "0:1 named William O'Donnell;2:2 president _n_of;3:3 says _v_to;5:5 still _a_1;6:6 thinks _v_1;7:7 the _q;8:8 IPO _n_unknown;0:0 William named", '0:0 If _x_then;4:4 says _v_to;5:5 the _q;6:6 company _n_of;7:7 would _v_modal;8:8 have _v_qmodal;10:10 change _v_cause;12:12 expansion _n_1;0:0 If _x_then', '0:0 But _c;1:1 the _q;2:3 _n_1 market turmoil;4:4 could _v_modal;6:7 _a_1 partially beneficial;8:8 for _p;9:9 some _q;10:10 small _a_1;0:0 But _c', '0:0 In _p_state;1:1 a _q;2:2 sagging _v_1;3:3 market _n_1;4:4 the _q;5:6 named Federal Reserve;7:7 System named;8:8 "might _v_modal;9:9 flood _v_cause;10:10 the _q;11:11 market _n_1;12:12 with _p;13:13 funds _n_1;14:14 and _c;15:15 that _q_dem;16:16 should _v_modal;17:17 bring _v_down;18:18 interest _n_in;19:19 rates _n_of;21:21 says _v_to;22:23 named Leonard T;24:24 Anctil named;25:25 vice _n_1;26:26 president _n_of;28:28 the _q;29:29 Bank _n_of;31:32 named New England;0:0 In _p_state', '0:1 named James G;2:2 Zafris named;3:3 president _n_of;5:6 named Danvers Savings;7:8 named Bank Danvers;9:9 Mass named;10:10 says _v_to;11:11 the _q;12:13 _n_1 market turmoil;14:14 "is _v_id;15:15 an _q;16:16 absolute _a_1;18:18 event _n_item;18:18 event _non_a_1;19:19 for _p;20:20 small _a_1;0:0 James named', '0:0 Mr _n_1;1:1 Zafris named;2:2 thinks _v_1;3:3 rates _n_of;5:5 heading _v_dir;6:6 down _p;7:7 helping _v_1;8:8 small _a_1;0:0 Mr _n_1', '0:1 named Peter Drake;2:2 biotechnology _n_unknown;3:3 analyst _n_1;4:4 for _p;5:6 named Vector Securities;7:8 named International Chicago;9:9 thinks _v_1;10:11 _n_1 market uncertainty;12:12 may _v_modal;13:13 encourage _v_1;14:14 small _a_1;15:15 companies _n_of;17:17 form _v_cause;19:19 strategic _a_1;20:20 alliances _n_1;21:21 with _p;22:22 big _a_1;0:0 Peter named', '0:0 Partly _x_deg;1:1 because _x;2:2 the _q;3:3 1987 yofc;4:5 _n_1 market crash;6:6 made _v_cause;8:8 harder _a_for;12:13 _v_1 find financing;16:16 technology _n_1;16:16 technology _high_a_1;17:17 concerns _n_1;19:19 made _v_1;20:20 such _q;21:21 alliances _n_1;0:0 Partly _x_deg', '0:0 Some _q;1:1 even _a_1;2:2 see _v_1;3:3 a _q;4:4 silver _a_1;5:5 lining _n_1;6:6 in _p;7:7 the _q;8:8 dark _a_1;0:0 Some _q', '0:1 named Alan Wells;2:2 president _n_of;4:5 named Bollinger Wells;6:6 Lett named;7:8 & Co _n_1;9:9 a _q;10:11 named New York;12:13 _n_1 merger specialist;14:14 thinks _v_1;15:15 panicky _a_unknown;16:16 investors _n_1;17:17 may _v_modal;18:18 lose _v_1;20:20 enthusiasm _n_1;21:21 for _p;22:22 leveraged _v_1;23:24 buy- out _n_1;25:25 and _c;26:26 giant _a_1;27:27 takeover _n_of;0:0 Alan named', '0:0 Instead _p;2:2 could _v_modal;3:3 turn _v_1;4:4 to _p;5:5 investing _v_in;7:7 smaller _a_1;8:8 deals _n_1;9:9 involving _v_1;10:10 smaller _a_1;11:11 companies _n_of;0:0 Instead _p', '0:0 This _q_dem;2:2 add _v_to;4:4 the _q;5:5 appeal _n_1;6:6 of _p;7:7 small _a_1;8:8 business _n_1;10:10 says _v_to;12:12 investors _n_1;13:13 often _a_1;14:14 have _v_1;15:15 a _q;16:16 degree _n_of;0:0 This _q_dem', "0:1 named Bay Financial;2:2 Corp _n_1;3:3 hurt _v_1;5:5 high _a_1;6:6 debts _n_1;7:7 and _c;8:8 deteriorating _v_1;9:11 _n_1 real estate investments;12:12 reported _v_to;13:13 a _q;14:14 wider _a_1;15:15 loss _n_of;16:16 for _p;17:17 the _q;18:18 fourth ord;19:19 quarter _n_temp;20:20 and _c;21:21 said _v_to;23:23 might _v_modal;25:27 _v_1 forced seek;28:28 a _q;29:29 bankruptcy- _n_1;30:30 court _n_of;31:31 reorganization _n_1;32:32 if _x_then;34:34 can't _v_modal;35:35 renegotiate _v_1;35:35 renegotiate _re-_a_again;0:0 Bay named", '0:0 Thus _a_1;3:3 coming _v_up;5:5 short _a_of;6:6 on _p;7:7 a _q;8:8 big _a_1;9:9 bet _n_1;11:11 quick _a_1;12:12 sales _n_of;13:13 at _p;14:14 higher _a_1;15:15 prices _n_of;16:16 would _v_modal;17:17 enable _v_1;20:20 keep _v_up;22:22 with _p;23:23 mortgage _n_1;24:24 and _c;25:25 other _a_1;26:26 debt _n_1;0:0 Thus _a_1', '0:0 The _q;1:1 company _n_of;2:2 said _v_to;4:4 had _v_1;5:5 a _q;6:6 net _n_1;7:7 loss _n_of;8:8 in _p_temp;10:10 fourth ord;11:11 quarter _n_temp;12:12 ended _v_cause;13:13 June mofy;14:14 30 dofm;15:15 of _p;16:16 $ _n_1;17:18 card 36.2 million;19:19 or _c;20:20 $ _n_1;21:21 9.33 card;22:22 a _p_per;23:23 share _n_of;24:24 on _p;25:25 revenue _n_1;26:26 of _p;27:27 $ _n_1;28:29 card 13.1 million', '0:0 A _q;1:1 year _n_1;2:2 earlier _a_1;3:3 the _q;4:4 company _n_of;5:5 had _v_1;6:6 a _q;7:7 loss _n_of;9:9 $ _n_1;10:11 card 10.8 million;12:12 or _c;13:13 $ _n_1;14:14 3.04 card;15:15 a _p_per;16:16 share _n_of;17:17 on _p;18:18 revenue _n_1;19:19 of _p;20:20 $ _n_1;21:22 card 10.8 million', '0:0 For _p;1:1 the _q;2:2 year _n_1;4:4 had _v_1;5:5 a _q;6:6 net _n_1;7:7 loss _n_of;9:9 $ _n_1;10:11 card 62 million;12:12 or _c;13:13 $ _n_1;14:14 15.97 card;15:15 a _p_per;16:16 share _n_of;17:17 on _p;18:18 revenue _n_1;19:19 of _p;20:20 $ _n_1;21:22 card 44.3 million', '0:0 In _p_temp;1:1 the _q;2:2 previous _a_1;3:3 year _n_1;5:5 had _v_1;6:6 a _q;7:7 loss _n_of;9:9 $ _n_1;10:11 card 22.5 million;12:12 or _c;13:13 $ _n_1;14:14 6.52 card;15:15 a _p_per;16:16 share _n_of;17:17 on _p;18:18 revenue _n_1;19:19 of _p;20:20 $ _n_1;21:22 card 41.1 million', '0:0 Although _x;3:3 having _v_1;4:4 serious _a_1;5:5 cash- _n_1;6:7 _n_of flow problems;8:8 Bay named;9:9 said _v_to;10:10 the _q;12:12 market _n_1;12:12 market _fair_a_1;13:13 value _n_1;14:14 of _p;16:16 holdings _n_1;17:17 minus _c;18:18 debt _n_1;20:20 equal _a_to;22:22 $ _n_1;23:23 6.02 card;24:24 a _p_per;25:25 share _n_of;26:26 at _p_temp;27:27 June mofy;28:28 30 dofm;29:29 based _v_1;30:30 on _p;31:31 a _q;32:32 recent _a_1;0:0 Although _x', '0:0 Book _n_of;1:1 value _n_1;2:2 per _p;3:3 share _n_of;6:6 based _v_1;7:7 on _p;8:8 investments _n_1;9:9 at _p;10:10 cost _n_1;11:11 was _v_id;12:12 a _q;13:13 negative _a_1;14:14 $ _n_1;15:15 6.69 card;16:16 a _p_per;0:0 Book _n_of', '0:0 A _q;1:1 year _n_1;2:2 earlier _a_1;4:4 market _n_1;4:4 market _fair_a_1;5:5 value _n_1;6:6 per _p;7:7 share _n_of;8:8 was _v_id;9:9 $ _n_1;10:10 26.02 card;11:11 and _c;12:12 book _n_of;13:13 value _n_1;14:14 was _v_id;15:15 $ _n_1;16:16 9.43 card;17:17 a _p_per;0:0 A _q', '0:0 Annualized _v_unknown;1:1 interest _n_in;2:2 rates _n_of;3:3 on _p;4:4 certain _a_of;5:5 investments _n_1;6:6 as _p;7:7 reported _v_to;9:9 the _q;10:11 named Federal Reserve;12:12 Board named;13:13 on _p;14:14 a _q;15:15 weekly- _n_1;16:16 average _a_1;17:17 basis _n_of;18:18 : _p_namely;19:19 1989 yofc;20:20 and _c;21:21 Wednesday dofw;22:22 October mofy;23:23 4 dofm;0:0 Annualized _v_unknown', '0:0 c named;2:2 Yields _n_1;3:3 adjusted _v_to;4:4 for _p;5:5 constant _a_1;0:0 c named', '0:0 TRW named;1:1 Inc _n_1;2:2 reported _v_to;3:3 a _q;4:4 12 card;5:5 % _n_of;6:6 decline _n_1;7:7 in _p;9:9 quarter _n_temp;9:9 quarter ord;10:11 _n_1 net income;12:12 but _c;13:13 the _q;14:14 company _n_of;15:15 said _v_to;17:17 excluding _v_1;18:18 unusual _a_about;19:19 gains _n_1;20:20 in _p_temp;21:21 both _q;21:21 both card;22:22 quarters _n_temp;23:23 operating _v_1;24:24 profit _n_1;25:25 rose _v_1;26:26 16 card;0:0 TRW named', '0:0 The _q;1:1 electronics _n_1;2:2 automotive _a_unknown;3:3 and _c;4:5 _n_1 aerospace concern;6:6 said _v_to;8:8 quarter _n_temp;8:8 quarter ord;9:9 net _n_1;10:10 was _v_id;11:11 $ _n_1;12:13 card 60 million;14:14 or _c;15:15 98 card;16:16 cents _n_1;17:17 a _p_per;18:18 share _n_of;19:20 _p down from;21:21 $ _n_1;22:23 card 68 million;24:24 or _c;25:25 $ _n_1;26:26 1.11 card;27:27 a _p_per;28:28 share _n_of;29:29 a _q;30:30 year _n_1;0:0 The _q', '0:0 Share _n_of;1:1 earnings _n_1;3:3 reported _v_to;4:4 on _p;5:5 a _q;6:6 fully _a_of;7:7 diluted _v_1;8:8 basis _n_of;9:9 by _p;10:10 company _n_of;0:0 Share _n_of', '0:0 Results _n_of;1:1 for _p;2:2 the _q;3:3 1988 yofc;4:4 quarter _n_temp;5:5 included _v_1;6:6 a _q;7:7 gain _n_1;8:8 of _p;9:9 $ _n_1;10:10 1.05 card;11:11 a _p_per;12:12 share _n_of;13:13 from _p;14:14 sale _n_of;16:16 the _q;17:18 named Reda Pump;19:19 and _c;20:21 named Oilwell Cable;22:22 units _n_of;23:23 partly _a_1;24:24 offset _v_1;26:26 a _q;27:27 charge _n_of;29:29 69 card;30:30 cents _n_1;31:31 a _p_per;32:32 share _n_of;33:33 for _p;34:34 recall _n_of;36:36 faulty _a_1;37:37 truck _n_1;38:38 steering _v_1;0:0 Results _n_of', '0:0 The _q;1:1 latest _a_for;2:2 quarter _n_temp;3:3 included _v_1;4:4 a _q;5:5 gain _n_1;6:6 of _p;7:7 11 card;8:8 cents _n_1;9:9 a _p_per;10:10 share _n_of;11:11 as _p;12:12 a _q;13:13 partial _a_1;14:14 reversal _n_of;16:16 the _q;17:18 _n_of recall charge;19:19 because _x;20:20 the _q;21:21 reserve _n_1;22:22 established _v_1;23:23 last _a_1;24:24 year _n_1;25:25 exceeded _v_1;26:26 the _q;27:27 actual _a_1;28:28 recall _n_of;0:0 The _q', '0:0 Sales _n_of;1:1 for _p;2:2 the _q;3:3 quarter _n_temp;4:4 rose _v_1;5:5 8.3 card;6:6 % _n_of;7:7 to _p;8:8 $ _n_1;9:10 card 1.79 billion;11:11 from _p;12:12 $ _n_1;13:14 card 1.65 billion;15:15 with _x_subord;16:16 all _q;17:17 three card;18:18 major _a_1;19:19 product _n_1;20:20 groups _n_of;21:21 reporting _v_to;0:0 Sales _n_of', '0:0 Automotive _a_unknown;1:1 sales _n_of;2:2 jumped _v_1;3:3 16 card;4:4 % _n_of;5:5 to _p;6:6 $ _n_1;7:8 card 791 million;9:9 mainly _x_deg;10:11 because of _p;12:12 higher _a_1;13:13 sales _n_of;15:15 air _n_1;16:16 bags _n_of;17:17 and _c;18:18 other _a_1;19:20 _n_1 passenger restraint;21:21 systems _n_of;22:22 TRW named;0:0 Automotive _a_unknown', '0:0 The _q;1:1 group _n_of;2:2 had _v_1;3:3 an _q;4:4 operating _v_1;5:5 profit _n_1;6:6 of _p;7:7 $ _n_1;8:9 card 65 million;10:10 against _p;11:11 a _q;12:12 loss _n_of;14:14 $ _n_1;15:16 card 13 million;17:17 a _q;18:18 year _n_1;0:0 The _q', '0:0 However _a_1;1:1 excluding _v_1;2:2 the _q;3:4 year- earlier _a_1;5:5 charge _n_of;6:6 for _p;7:7 recall _n_of;9:9 steering _v_1;10:10 gear _n_1;11:11 operating _v_1;12:12 profit _n_1;13:13 in _p_temp;14:14 the _q;15:15 latest _a_for;16:16 quarter _n_temp;17:17 declined _v_1;18:18 14 card;19:19 % _n_of;20:20 reflecting _v_1;21:21 higher _a_1;22:23 start- up _startup_n_1;24:24 and _c;25:25 product _n_1;26:26 development _n_of;27:27 expenses _n_1;28:28 in _p;29:30 _n_1 passenger- restraint;0:0 However _a_1', '0:0 Materials _n_1;1:1 and _c;2:2 production _n_of;3:3 costs _n_1;4:4 also _a_1;5:5 rose _v_1;6:6 TRW named;0:0 Materials _n_1', '0:0 An _q;1:1 acquisition _n_of;2:2 accounted _v_for;4:5 _q half the;6:6 sales _n_of;7:7 rise _n_1;8:8 TRW named;0:0 An _q', '0:0 Operating _v_1;1:1 profit _n_1;2:2 rose _v_1;3:3 threefold _a_1;4:4 to _p;5:5 $ _n_1;6:7 card 18 million;8:8 from _p;9:9 $ _n_1;10:11 card 6 million', '0:0 For _p;1:1 the _q;2:2 nine card;3:3 months _n_1;4:4 TRW named;6:6 net _n_1;7:7 was _v_id;8:8 $ _n_1;9:10 card 199 million;11:11 or _c;12:12 $ _n_1;13:13 3.22 card;14:14 a _p_per;15:15 share _n_of;16:16 down _p;17:17 3 card;18:18 % _n_of;19:19 from _p;20:20 $ _n_1;21:22 card 205 million;23:23 or _c;24:24 $ _n_1;25:25 3.33 card;26:26 a _p_per;27:27 share _n_of;28:28 a _q;29:29 year _n_1;0:0 For _p', '0:0 Sales _n_of;1:1 rose _v_1;2:2 2.9 card;3:3 % _n_of;4:4 to _p;5:5 $ _n_1;6:7 card 5.42 billion;8:8 from _p;9:9 $ _n_1;10:11 card 5.27 billion', '0:0 a _q;1:1 tragicomic _a_unknown;2:2 monologue _n_unknown;3:3 by _p;4:4 an _q;5:5 idealistic _a_unknown;6:6 not _c;7:7 unheroic _a_unknown;8:8 though _c;9:9 sadly _a_1;10:10 self- _n_1;11:11 deceived _v_1;12:12 English _a_1;13:13 butler _n_1;14:14 in _p;16:16 sixties card;18:18 proceeds _v_1;19:20 as if _x;21:21 the _q;22:23 _a_1 realistic English;24:26 _n_of novel manners;27:27 like _p;28:28 Britannia named;30:30 still _a_1;31:31 ruled _v_1;32:32 the _q;0:0 a _q', '1:1 implies _v_1;3:3 the _q;4:5 named British Empire;7:7 rooted _v_1;8:8 in _p;10:10 subjects _n_of;12:12 minds _n_1;13:13 manners _n_of;14:14 and _c;15:15 morals _n_1;16:16 and _c;17:17 argues _v_with;18:18 tacitly _a_unknown;21:21 self- _n_1;22:22 destructive _a_1;23:23 flaws _n_1;25:25 embodied _v_1;26:26 in _p;27:27 the _q;28:28 defensive _a_1;29:29 snobbery _n_unknown;30:30 willful _a_1;31:31 blindness _n_1;32:32 role- _n_of;33:33 playing _v_1;34:34 and _c;35:35 especially _x_deg;36:36 the _q;37:37 locutions _n_unknown;38:38 of _p;40:40 domestic _a_1;1:1 implies _v_1', '0:0 As _x_subord;1:1 the _q;2:2 narrator _n_1;3:3 Stevens named;4:4 the _q;5:5 solitary _a_1;6:6 butler _n_1;7:7 of _p;8:9 named Darlington Hall;10:10 mulls _v_over;12:12 such _q;13:13 hallowed _a_unknown;14:14 terms _n_of;15:15 as _p;16:17 _n_1 "greatness," "dignity,";18:18 "service" _n_1;19:19 and _c;20:20 "loyalty," _n_to;22:22 see _v_1;24:24 pious _a_1;25:25 cant _n_1;26:26 subverts _v_from;27:27 the _q;0:0 As _x_subord', '0:0 Stevens named;2:2 dutiful _a_unknown;3:3 conflation _n_unknown;4:4 of _p;5:5 the _q;6:6 public _a_1;7:7 and _c;8:8 private _a_1;9:9 realms _n_1;11:11 like _p;13:13 beloved _a_1;14:14 master _n_1;17:17 destroys _v_1;18:18 all _q;21:23 _v_1 designed preserve', '0:0 Such _q;1:1 armor _n_1;2:2 crushes _v_cause;3:3 the _q;0:0 Such _q', '0:0 The _q;1:1 mask _n_1;2:2 cuts _v_1;3:3 to _p;4:4 the _q;0:0 The _q', "1:1 's _v_id;2:2 1956 yofc;3:3 the _q;4:4 year _n_1;5:5 the _q;6:6 Suez named;7:7 crisis _n_1;8:8 marked _v_1;9:9 the _q;10:10 final _a_1;11:11 end _n_of;1:1 's _v_id", '1:1 is _v_itcleft;2:2 the _q;3:3 very _a_1;4:4 lack _n_1;5:5 of _p;6:7 _n_1 obvious drama;8:8 or _c;9:9 spectacle _n_of;11:11 sets _v_apart;12:12 the _q;13:13 beauty _n_1;14:14 of _p;1:1 is _v_itcleft', '2:2 pertinent _a_unknown;3:3 is _v_id;4:4 the _q;5:5 calmness _n_unknown;6:6 of _p;7:7 that _q_dem;8:8 beauty _n_1;10:10 sense _n_of;2:2 pertinent _a_unknown', '1:1 is _v_nv;2:3 as though _x;4:4 the _q;5:5 land _n_of;6:6 knows _v_of;9:9 own _a_1;10:10 beauty _n_1;11:11 of _p;13:13 own _a_1;14:14 greatness _n_1;15:15 and _c;16:16 feels _v_1;17:17 no _q;18:18 need _n_of;1:1 is _v_nv', '0:0 An _q;1:1 effusive _a_1;0:0 An _q', '0:0 An _q;2:2 mannered _a_1;2:2 mannered _ill_a_1;0:0 An _q', '0:0 Such _q;1:1 dignity _n_1;2:2 "has _v_qmodal;4:4 do _v_1;5:5 crucially _a_for;6:6 with _p;7:7 a _q;8:10 _n_1 butler ability;13:13 abandon _v_1;14:14 the _q;15:15 professional _a_1;16:16 being _n_1;0:0 Such _q', '1:3 _v_1 see know;5:5 father _n_of;6:6 would _v_modal;8:8 wished _v_1;11:11 carry _v_on;13:13 just _x_deg;1:1 see _v_1', '1:1 is _v_itcleft;2:2 this _q_dem;3:3 kind _n_of-n;5:5 dignity _n_1;6:6 and _c;7:7 restraint _n_1;9:9 allows _v_1;10:10 Stevens named;12:12 declare _v_to;13:13 : _p_namely;14:14 "For _p;15:15 all _q;17:17 sad _a_1;18:18 associations _n_of;21:21 recall _v_1;22:22 that _q_dem;23:23 evening _n_of;24:24 today _a_1;26:26 find _v_1;28:28 do _v_so;30:30 with _p;31:31 a _q;32:32 large _a_1;33:33 sense _n_of;1:1 is _v_itcleft', '1:1 note _v_to;2:2 the _q;3:4 _a_1 imperial public;5:5 word _n_of;6:6 used _v_1;8:8 deny _v_to;9:9 private _a_1;10:10 rage _n_1;11:11 and _c;1:1 note _v_to', '0:0 Mr _n_1;1:1 Ishiguro named;3:3 ability _n_1;5:5 create _v_1;6:6 a _q;7:7 fallible _a_unknown;8:8 narrative _a_1;9:9 voice _n_1;11:14 _v_1 permits explore;15:15 such _q;16:16 intertwining _a_unknown;17:18 _a_1 domestic cultural;19:19 and _c;20:20 political _a_1;21:21 themes _n_of;23:23 abundantly _a_1;24:24 clear _a_of;25:25 in _p;27:27 previous _a_1;28:28 novel _n_of;29:30 named "An Artist;31:31 of _p;32:32 the _q;33:34 named Floating World,";35:35 set _v_1;36:36 in _p;37:37 Japan named;38:38 after _p;39:39 the _q;0:0 Mr _n_1', '0:0 Now _a_1;1:1 shifting _v_1;3:3 scene _n_of;4:4 from _p_time;5:5 the _q;6:6 country _n_of;8:8 left _v_1;9:9 at _p_temp;10:10 five numbered_hour;11:11 to _p;12:12 the _q;13:13 England named;16:16 lived _v_1;17:18 _p in for;19:19 nearly _x_deg;20:20 30 card;21:21 years _n_1;24:24 fashioned _v_into;25:25 a _q;26:26 novel _n_of;27:27 in _p;28:28 the _q;29:29 mode _n_1;30:30 of _p;31:32 named Henry James;33:33 and _c;34:35 named E.M Forster', '0:0 `This _q_dem;1:1 employer _n_1;2:2 embodies _v_1;3:3 all _q;6:6 find _v_1;7:7 noble _a_1;8:8 and _c;0:0 `This _q_dem', '2:2 hereafter _a_unknown;3:3 devote _v_to;2:2 hereafter _a_unknown', '0:0 In _p_state;1:1 the _q;2:2 end _n_of;3:3 after _p;4:4 meeting _v_1;5:5 with _p;6:6 the _q;7:7 former _a_1;8:8 housekeeper _n_unknown;9:9 Stevens named;10:10 sits _v_1;11:11 by _p;12:12 the _q;13:13 seashore _n_1;14:14 at _p_temp;15:15 dusk numbered_hour;16:16 thinking _n_1;17:17 of _p;19:19 and _c;20:20 of _p;22:22 employer _n_1;23:23 and _c;24:24 declares _v_to;0:0 In _p_state', "1:1 can't _v_modal;2:2 even _a_1;3:3 say _v_to;5:5 made _v_1;7:7 own _a_1;1:1 can't _v_modal", '0:0 Really _a_1;2:2 one _n_1;3:3 has _v_qmodal;5:5 ask _v_1;7:7 - excl;8:8 what _q;9:9 dignity _n_1;10:10 is _v_there;12:12 in _p;0:0 Really _a_1', '1:1 is _v_id;1:1 is _v_id', '1:1 is _v_id;1:1 is _v_id', '1:1 understand _v_by;2:2 such _q;3:3 rueful _a_unknown;4:4 wisdom _n_1;5:5 must _v_modal;6:6 be _v_id;7:7 retrospective _n_unknown;8:8 : _p_namely;9:9 The _q;10:10 owl _n_1;11:11 of _p;12:12 Minerva named;13:13 only _a_1;14:14 spreads _v_1;16:16 wings _n_1;17:17 at _p_temp;1:1 understand _v_by', '0:0 But _c;1:1 as _x_subord;2:3 named "The Remains;4:4 of _p;5:5 the _q;6:6 Day" named;7:8 _a_1 so eloquently;9:9 demonstrates _v_to;10:10 with _p;11:11 quiet _a_1;12:12 virtuosity _n_unknown;13:13 such _q;14:14 wisdom _n_1;15:15 can _v_modal;17:17 movingly _a_unknown;18:18 embodied _v_1;19:19 in _p;0:0 But _c', '0:0 Mr _n_1;1:1 Locke named;2:2 teaches _v_1;3:3 English named;4:4 and _c;5:5 comparative _a_1;6:6 literature _n_1;7:7 at _p;8:9 named Columbia University', '0:0 UGI named;1:1 Corp _n_1;2:2 said _v_to;4:4 AmeriGas named;5:5 subsidiary _n_1;6:6 completed _v_2;7:7 the _q;8:8 previously _p;9:9 announced _v_to;10:10 sale _n_of;13:14 _n_1 air separation;15:15 plant _n_1;16:16 and _c;17:17 related _v_to;18:18 assets _n_1;19:19 in _p;20:21 named Waukesha Wis;22:22 to _p;23:24 named AGA Gas;25:26 named Inc Cleveland', '0:0 The _q;1:1 price _n_of;0:0 The _q', '0:0 The _q;1:1 transaction _n_1;2:2 is _v_id;3:3 part _q;5:5 UGI named;7:7 continuing _v_1;8:8 program _n_1;10:10 shed _v_1;11:11 AmeriGas named;13:13 industrial _a_1;14:14 gas _n_1;15:15 interests _n_in;16:16 and _c;17:17 expand _v_cause;18:18 the _q;19:21 _n_1 subsidiary propane;0:0 The _q', '0:0 Since _p;1:1 June mofy;2:2 AmeriGas named;4:4 netted _v_1;7:7 $ _n_1;8:9 card 100 million;10:10 from _p;11:11 industrial _a_1;12:12 gas _n_1;13:13 divestitures _n_unknown;14:14 and _c;15:15 reinvested _v_1;15:15 reinvested _re-_a_again;18:18 $ _n_1;19:20 card 50 million;21:21 to _x;22:22 acquire _v_1;23:23 three card;24:24 propane _n_1;0:0 Since _p', '0:0 UGI named;1:1 is _v_id;2:2 a _q;3:3 gas _n_1;4:4 and _c;5:5 electric _a_1;6:6 utility _n_1;7:7 and _c;8:8 distributes _v_to;9:9 propane _n_1;10:10 nationally _a_1;11:11 through _p;13:13 AmeriGas named;0:0 UGI named', '0:1 named Stanislav Ovcharenko;3:3 represents _v_1;4:4 the _q;5:5 Soviet named;6:6 airline _n_1;7:7 Aeroflot named;8:8 here _a_1;9:9 has _v_1;10:10 some _q;11:11 visions _n_1;14:14 wild _a_1;15:15 even _x_deg;16:16 by _p;17:17 the _q;18:18 current _a_1;19:19 standards _n_1;20:20 of _p;0:0 Stanislav named', '0:0 In _p_state;2:2 office _n_of;3:3 overlooking _v_1;4:4 the _q;5:5 runway _n_1;6:6 of _p;7:8 named Shannon Airport;9:9 Mr _n_1;10:10 Ovcharenko named;11:11 enthusiastically _a_1;12:12 throws _v_out;16:16 calls _v_name;17:17 "just _x_deg;18:18 ideas" _n_of;0:0 In _p_state', '0:0 First named;2:2 suggests _v_to;3:4 named GPA Group;5:5 Ltd _n_1;6:6 the _q;7:7 international _a_1;8:8 aircraft _n_1;9:9 leasing _v_1;10:10 company _n_of;11:11 based _v_1;12:12 in _p;13:13 Ireland named;14:14 could _v_modal;15:15 lease _v_to;16:16 some _q;19:19 Boeing named;20:20 jetliners _n_unknown;22:22 the _q;23:23 Soviet named;0:0 First named', '0:0 Then _a_1;1:2 named Aer Lingus;3:3 the _q;4:4 Irish _a_1;5:5 flag _n_1;6:6 carrier _n_of;7:7 could _v_modal;8:8 teach _v_1;9:9 Aeroflot named;10:10 pilots _n_of;12:12 fly _v_to;13:13 the _q;14:14 Boeings named;15:15 and _c;16:16 the _q;17:17 fleet _n_1;18:18 could _v_modal;20:20 based _v_1;21:21 here _a_1;22:22 at _p;23:24 named Shannon Airport', "0:0 That _q_dem;1:1 's _v_id;3:3 all _q;0:0 That _q_dem", '0:1 named Aer Rianta;2:2 the _q;3:3 Irish _a_1;4:5 _n_1 airport authority;6:6 could _v_modal;7:7 build _v_1;8:8 a _q;9:10 _n_1 cargo terminal;11:11 in _p;12:12 the _q;13:14 named Soviet Union', '0:0 Aeroflot named;1:1 could _v_modal;2:2 lease _v_to;3:3 some _q;6:7 _n_1 cargo planes;9:10 named Aer Lingus;11:11 through _p;12:12 GPA named;13:13 for _p;14:14 a _q;16:16 venture _n_1;16:16 venture _joint_a_1;17:18 _n_1 cargo airline', '0:0 And _c;1:1 then _a_1;3:3 is _v_there;5:5 notion _n_of;7:7 an _q;9:9 Soviet named;9:9 Soviet _a_1;10:11 _n_1 charter airline;13:13 ferry _v_1;14:14 Armenians named;15:15 to _p;16:17 named Los Angeles;18:18 via _p;0:0 And _c', '1:1 the _q;2:2 freedoms _n_1;3:3 of _p;4:4 glasnost _u_unknown;5:5 gone _v_1;6:6 to _p;7:7 Mr _n_1;8:8 Ovcharenko named;1:1 the _q', '0:0 Hardly _a_1', '0:0 The _q;2:2 Soviet named;2:2 Soviet _a_1;3:3 aviation _n_1;4:4 connection _n_of;6:6 alive _a_1;7:7 and _c;8:8 well _a_2;9:9 here _a_1;10:10 at _p;11:12 named Shannon Airport', '0:0 GPA named;2:2 indeed _a_1;3:3 talking _v_about;4:4 about _p;5:5 leasing _v_to;6:6 Western named;7:7 planes _n_1;9:9 Aeroflot named;10:10 and _c;11:11 even _x_deg;12:12 about _p;13:13 buying _v_1;14:14 Soviet- named;15:15 built _v_1;16:17 named Tupolev 204s', '0:1 named Aer Lingus;3:3 in _p;4:4 discussions _n_1;5:5 with _p;6:6 the _q;7:7 Soviet named;8:8 carrier _n_of;9:9 about _p;10:10 a _q;11:12 _n_1 cargo venture;13:13 and _c;14:14 other _a_1;0:0 Aer named', '0:1 named Aer Rianta;2:2 already _a_1;3:3 has _v_1;6:6 ventures _n_1;7:7 with _p;8:8 Aeroflot named;11:12 _n_1 chief executive;14:14 studying _v_1;0:0 Aer named', '0:0 And _c;1:1 as _x_subord;2:2 Aeroflot named;3:3 struggles _v_1;5:5 boost _v_to;7:8 _n_1 service standards;9:9 upgrade _v_1;11:11 fleet _n_1;12:12 and _c;13:13 pursue _v_1;14:14 commercial _a_1;15:15 opportunities _n_1;16:16 the _q;17:17 Irish _a_1;18:19 _n_1 aviation industry;20:20 seems _v_to;21:23 _v_1 poised benefit', '0:0 "Irish _a_1;1:1 and _c;2:2 Soviet _a_1;3:3 people _n_of;5:5 similar," _a_to;6:6 says _v_to;7:7 Mr _n_1;0:0 "Irish _a_1', '1:1 look _v_seem-to;1:1 look _v_seem-to', '2:2 very _x_deg;2:2 very _x_deg', '0:0 Moreover _a_1;2:2 says _v_to;3:3 Irish _a_1;4:4 companies _n_of;6:6 small _a_1;7:7 but _c;0:0 Moreover _a_1', '1:1 have _v_qmodal;3:3 study _v_1;5:5 experience _n_with;6:6 very _x_deg;7:7 well," _a_1;1:1 have _v_qmodal', '1:1 must _v_modal;2:2 find _v_1;3:3 any _q;4:4 way _n_of;6:6 get _v_1;1:1 must _v_modal', '0:0 The _q;1:1 two card;2:2 groups _n_of;5:5 working _v_1;6:7 _p together since;8:8 the _q;9:9 late _a_for;10:10 1970s year_range;11:11 long _a_1;12:12 before _x_h;13:13 Soviet named;14:14 joint _a_1;15:15 ventures _n_1;16:16 were _v_id;17:17 the _q;18:18 rage _n_1;19:19 in _p;20:20 the _q;0:0 The _q', '0:0 Aeroflot named;1:1 carried _v_1;2:2 about _x_deg;3:4 card 125 million;5:5 passengers _n_1;6:6 last _a_1;7:7 year _n_1;8:8 and _c;9:10 named Shannon Airport;11:11 the _q;12:12 airline _n_1;14:14 largest _a_1;15:16 _n_1 transit airport;17:17 outside _p;18:18 the _q;19:20 named Soviet Union;21:21 saw _v_1;22:22 1,400 card;23:23 Aeroflot named;24:24 flights _n_1;25:25 and _c;26:26 250,000 card;27:27 passengers _n_1;28:28 pass _v_1;0:0 Aeroflot named', '0:0 An _q;1:1 apartment _n_1;2:2 complex _n_of;3:3 down _p;4:4 the _q;5:5 road _n_1;6:6 is _v_id;7:7 the _q;8:9 _n_of crew- rest;10:10 and _c;11:11 staging _v_1;12:12 area _n_of;13:13 for _p;16:16 130 card;17:17 Aeroflot named;18:18 pilots _n_of;19:19 and _c;20:21 _n_1 flight attendants', '0:0 The _q;1:1 airport _n_1;3:3 biggest _a_1;4:4 supplier _n_of;6:7 _n_1 aircraft fuel;8:8 is _v_id;9:9 the _q;10:11 named Soviet Union', '0:0 Tankers _n_1;1:1 from _p;2:2 the _q;3:3 Latvian _a_unknown;4:4 port _n_of;6:6 Ventspils named;7:7 each _q;8:8 year _n_1;9:9 unload _v_1;10:11 card 25 million;12:12 gallons _n_1;13:13 of _p;14:14 fuel _n_1;15:15 into _p;16:16 a _q;17:17 special _a_1;18:19 _n_1 tank farm;20:20 at _p;21:21 the _q;0:0 Tankers _n_1', '1:1 Aeroflot named;3:3 pour _v_1;4:4 into _p;6:6 own _a_1;7:7 gas- _n_1;8:8 guzzling _a_unknown;9:9 Ilyushins named;11:11 bartered _v_unknown;12:12 to _p;13:13 the _q;14:15 _n_1 airport authority;17:17 resells _v_1;17:17 resells _re-_a_again;20:20 11 card;21:21 Western named;22:22 carriers _n_of;23:23 including _v_1;24:25 named Air France;26:27 named Trans World;28:28 Airlines named;29:29 and _c;30:31 named Pakistan International;1:1 Aeroflot named', '0:0 Aeroflot named;1:1 thus _a_1;2:2 pays _v_for;4:5 _n_1 landing fees;6:6 ground- _n_1;7:7 handling _v_1;8:8 and _c;9:9 catering _v_1;10:10 bills _n_of;11:11 with _p;12:12 fuel _n_1;13:13 preserving _v_1;15:15 hard _a_for;0:0 Aeroflot named', "0:0 That _q_dem;1:1 isn't _v_id;0:0 That _q_dem", '0:0 Last _a_1;1:1 year _n_1;2:2 the _q;3:3 Irish _a_1;4:5 _n_1 airport authority;6:6 in _p;7:7 a _q;8:8 joint _a_1;9:9 venture _n_1;10:10 with _p;11:11 Aeroflot named;12:12 opened _v_cause;13:13 four card;15:15 currency _n_1;15:15 currency _hard_a_for;16:16 duty- _n_1;17:17 free _a_1;18:18 shops _n_of;19:19 at _p;20:22 named Moscow Sheremetyevo;0:0 Last _a_1', '0:1 named Aer Rianta;2:2 now _a_1;3:3 manages _v_1;4:4 duty- _n_1;5:5 free _a_1;6:6 sales _n_of;7:7 on _p;8:8 all _q;9:9 Aeroflot named;10:10 international _a_1;11:11 flights _n_1;12:13 out of _p;0:0 Aer named', '0:0 Duty- _n_1;1:1 free _a_1;2:2 shops _n_of;3:3 in _p;4:6 named Leningrad Pulkova;7:7 Airport named;8:8 opened _v_1;9:9 in _p_temp;10:10 July mofy;11:11 and _c;13:13 currency _n_1;13:13 currency _hard_a_for;14:14 shops _n_of;15:15 in _p;16:16 Leningrad named;17:17 hotels _n_1;18:18 and _c;19:19 on _p;20:20 the _q;21:21 Soviet- named;22:22 Finnish _a_1;23:23 frontier _n_1;25:25 coming _v_1;0:0 Duty- _n_1', '0:1 named Aer Rianta;3:3 talking _v_about;4:4 about _p;5:5 similar _a_to;6:6 joint _a_1;7:7 ventures _n_1;8:8 in _p;9:9 Tashkent named;10:10 and _c;11:11 in _p;12:12 Sochi named;13:13 a _q;14:15 named Black Sea;16:16 resort _n_1;17:17 and _c;18:18 even _a_1;19:19 has _v_prd;20:20 a _q;21:21 computer- _n_1;22:22 assembly _n_of;23:23 project _n_1;24:24 cooking _v_1;25:25 with _p;26:26 the _q;27:27 Georgian _a_1;28:28 city _n_1;29:29 of _p;0:0 Aer named', '0:0 Aeroflot named;2:2 international _a_1;3:3 fleet _n_1;4:4 of _p;5:5 285 card;6:6 planes _n_1;9:9 repainted _v_1;9:9 repainted _re-_a_again;10:10 and _c;11:11 refurbished _v_1;12:12 at _p;13:14 named Shannon Airport', '0:1 Thanks to _p;2:2 a _q;3:3 new _a_1;4:5 _n_1 air- traffic;6:6 agreement _n_1;7:7 and _c;8:8 the _q;9:9 ability _n_1;10:10 of _p;11:11 Irish _a_1;12:12 travel _n_1;13:13 agents _n_of;15:15 issue _v_1;16:16 Aeroflot named;17:18 _n_1 tickets tourists;19:19 here _a_1;21:21 taking _v_of-i;22:22 advantage _n_i;24:24 Aeroflot named;26:26 reasonable _a_for;27:27 prices _n_of;28:28 to _x;29:29 board _v_1;30:30 flights _n_1;31:31 in _p;32:32 Shannon named;33:33 for _p;34:34 holidays _n_1;35:35 in _p;36:37 named Havana Kingston;38:38 and _c;39:40 named Mexico City', '0:0 The _q;2:2 trip _n_of;2:2 trip _round_a_1;3:3 fare _n_1;4:4 to _p;5:5 Havana named;6:6 is _v_id;7:7 410 card;8:8 Irish _a_1;9:10 _n_1 punts ($;0:0 The _q', '0:0 Jamaica named;1:1 costs _v_1;2:2 504 card;0:0 Jamaica named', '0:0 A _q;1:1 formal _a_1;2:2 blessing _v_1;4:4 sorts _n_of;6:6 bestowed _v_on;8:8 this _q_dem;9:9 friendship _n_1;10:10 in _p_temp;11:11 April mofy;12:12 when _x_subord;13:13 Mikhail named;14:14 and _c;15:16 named Raisa Gorbachev;17:17 stopped _v_1;18:18 here _a_1;19:19 for _p;20:20 talks _n_of-on;21:21 with _p;22:22 Irish _a_1;23:24 Prime Minister _n_of;25:26 named Charles Haughey', '0:0 New _a_1;1:1 trade _n_of;2:2 accords _n_1;0:0 New _a_1', '1:1 all _a_1;2:2 started _v_1;3:3 with _p;1:1 all _a_1', '0:0 When _x_subord;2:2 opened _v_1;3:3 in _p_temp;4:4 1939 yofc;5:5 Shannon named;6:6 was _v_id;7:7 the _q;8:8 first ord;9:9 landfall _n_unknown;10:10 in _p;11:11 Europe named;12:12 for _p;13:13 thirsty _a_1;14:14 airplanes _n_1;15:15 flying _v_1;16:16 from _p;17:18 named North America', "0:0 Advances _n_of;1:1 in _p;2:3 _n_1 aircraft fuel;4:4 efficiency _n_1;5:5 over _p;6:6 the _q;7:7 years _n_1;8:8 made _v_cause;9:9 a _q;10:10 Shannon named;11:11 stop _n_1;12:12 unnecessary _a_for;13:13 for _p;14:14 most _q;15:15 Western named;16:17 _n_1 air fleets;18:18 but _c;19:19 Aeroflot named;20:20 still _a_1;21:21 flies _v_to;22:22 inefficient _a_1;23:23 Ilyushins named;25:25 can't _v_modal;26:26 make _v_1;28:28 from _p_time;29:29 Moscow named;30:30 to _p;31:31 Managua named;32:32 on _p;33:33 one card;0:0 Advances _n_of", '0:0 As _p;1:1 a _q;2:2 result _n_of;3:3 Ireland named;5:5 spurn _v_1;6:6 the _q;7:7 Soviets _n_1;8:8 after _x_h;10:10 shot _v_down;12:12 a _q;13:13 Korean _a_1;14:15 named Air Lines;16:16 jetliner _n_unknown;17:17 over _p;18:18 the _q;19:19 Sea _n_of;21:21 Japan named;22:22 in _p_temp;23:23 1983 yofc;24:24 though _x;26:26 suspended _v_1;27:27 direct _a_1;28:29 named Moscow- Shannon;30:30 flights _n_1;31:31 for _p;32:32 two card;0:0 As _p', '0:1 In fact _a_1;2:3 named Aer Lingus;4:5 _v_1 started ferrying;6:6 Russians _n_1;7:7 from _p_time;8:8 Shannon named;9:9 to _p;10:11 named New York;12:12 when _x_subord;13:13 Washington named;14:14 stripped _v_of;15:15 Aeroflot named;18:18 U.S named_n;19:19 landing _v_1;0:1 In fact _a_1', '0:0 Today _a_1;1:2 named Aer Rianta;4:4 making _v_1;5:5 a _q;6:6 heap _n_1;7:7 of _p;8:8 money _n_1;9:9 from _p;11:11 Soviet named;0:0 Today _a_1', '0:0 And _c;1:1 with _x_subord;2:2 those _q_dem;3:3 contacts _n_1;4:4 in _p;5:5 place _n_of;7:7 could _v_modal;9:9 relatively _x_deg;10:10 simple _a_for;11:11 to _x;12:12 add _v_to;13:14 named Aer Lingus;15:15 and _c;16:16 GPA named;18:18 the _q;0:0 And _c', "0:1 _a_1 Then perhaps;2:2 Mr _n_1;3:3 Ovcharenko named;5:5 ideas _n_of;6:6 wouldn't _v_modal;7:7 sound _v_seem-to;8:8 like _p;0:0 Then _a_1", '0:0 Britain named;2:2 industrial _a_1;3:3 production _n_of;4:4 rose _v_1;5:5 1.5 card;6:6 % _n_of;7:7 in _p_temp;8:8 August mofy;9:9 from _p;10:10 July mofy;11:11 and _c;13:13 up _p;14:14 0.9 card;15:15 % _n_of;16:16 from _p;17:17 August mofy;18:18 1988 yofc;19:20 according to _p;21:21 provisional _a_1;22:22 data _n_1;23:23 from _p;24:24 the _q;25:26 named Central Statistical;0:0 Britain named', '0:0 Output _n_of;1:1 in _p;2:2 the _q;3:4 _n_1 energy sector;6:6 can _v_modal;7:7 vary _v_1;8:8 greatly _a_1;9:9 with _p;10:10 swings _n_1;11:11 in _p;12:12 the _q;13:14 _n_1 oil market;15:15 rose _v_1;16:16 3.8 card;17:17 % _n_of;18:18 in _p_temp;19:19 August mofy;20:20 from _p;21:21 May mofy;22:22 but _c;24:24 down _p;25:25 7.1 card;26:26 % _n_of;27:27 from _p;28:28 a card;29:29 year _n_1;0:0 Output _n_of', '0:0 The _q;1:1 latest _a_for;2:2 figures _n_1;3:3 compare _v_with;5:5 July mofy;7:7 4.5 card;8:8 % _n_of;9:11 month- to- month _a_1;12:12 rise _n_1;13:13 and _c;14:14 11.3 card;15:15 % _n_of;16:18 year- to- year _a_1;0:0 The _q', '0:0 When _x_subord;1:1 Nucor named;2:2 Corp _n_1;3:3 begins _v_1;4:4 shipping _v_cause;5:5 steel _n_1;6:6 from _p;7:7 the _q;8:8 world _n_of;10:10 first ord;12:12 slab _n_1;12:12 slab _thin_a_1;13:13 plant _n_1;14:14 this _q_dem;15:15 month _n_1;18:19 _v_1 begin testing;20:20 the _q;21:21 competitive _a_1;22:22 mettle _n_1;23:23 of _p;25:25 giant _a_1;0:0 When _x_subord', '0:0 The _q;1:1 new _a_1;2:2 technology _n_1;4:4 creates _v_1;5:5 a _q;6:6 very _x_deg;7:7 thin _a_1;8:8 piece _n_of;10:10 steel _n_1;11:11 radically _a_1;12:12 reduces _v_1;13:13 the _q;14:14 costs _n_1;15:15 of _p;16:16 making _v_1;17:17 flat- _n_1;18:18 rolled _v_cause;0:0 The _q', '0:0 An _q;1:1 ebullient _a_unknown;2:3 named Kenneth Iverson;4:4 Nucor named;6:6 chairman _n_of;7:7 says _v_to;8:8 the _q;9:9 company _n_of;11:11 plant _n_1;12:12 eventually _a_1;14:14 make _v_1;15:15 a card;16:16 ton _n_1;17:17 of _p;18:18 steel _n_1;19:19 in _p;20:20 1.5 card;21:22 _n_1 man hours;23:23 compared _v_with;25:27 card four six;28:29 _n_1 man hours;30:30 at _p;31:31 a _q;32:32 conventional _a_1;0:0 An _q', '2:2 had _v_1;3:3 the _q;4:4 Russians _n_1;5:5 and _c;6:6 Chinese _n_1;7:7 and _c;8:8 people _n_of;9:9 from _p;10:10 India named;11:11 visiting _v_1;13:13 Mr _n_1;14:14 Iverson named;2:2 had _v_1', '1:1 in _p;2:2 the _q;3:3 world _n_of;5:5 watching _v_1;7:7 very _x_deg;1:1 in _p', '0:0 Especially _x_deg;2:2 neighbors _n_1;3:3 the _q;4:4 major _a_1;5:5 U.S named_n;0:0 Especially _x_deg', '0:0 Already _a_1;1:1 USX named;2:2 Corp _n_1;3:3 and _c;4:4 Armco named;5:5 Inc _n_1;7:7 studying _v_1;8:8 Nucor named;10:10 technology _n_1;11:11 to _x;12:12 see _v_1;15:15 can _v_modal;0:0 Already _a_1', '0:0 Says _v_to;1:1 the _q;2:3 _n_1 chief executive;4:4 officer _n_1;5:5 of _p;6:6 a _q;7:7 major _a_1;8:8 Midwest named;9:9 steel _n_1;10:10 company _n_of;11:11 : _p_namely;14:14 damn _x_deg;0:0 Says _v_to', '0:1 _a_1 New efficient;2:2 and _c;3:3 sophisticated _a_1;4:4 processes _n_of;5:5 make _v_cause;7:7 easier _a_for;9:9 smaller _a_1;11:11 cash- _n_1;12:12 rich _a_in;13:13 companies _n_of;15:15 make _v_1;16:16 steel _n_1;17:17 at _p;18:18 a _q;19:19 fraction _n_of;22:23 named Big Steel;24:24 paid _v_for;25:25 decades _n_1;0:0 New _a_1', '0:1 _a_1 New efficient;2:2 and _c;3:3 sophisticated _a_1;4:4 processes _n_of;5:5 make _v_cause;7:7 easier _a_for;9:9 smaller _a_1;11:11 cash- _n_1;12:12 rich _a_in;13:13 companies _n_of;15:15 make _v_1;16:16 steel _n_1;17:17 at _p;18:18 a _q;19:19 fraction _n_of;22:23 named Big Steel;24:24 paid _v_for;25:25 decades _n_1;0:0 New _a_1', '1:1 also _a_1;2:2 enables _v_1;3:3 minimills _n_unknown;4:4 finally _a_1;6:6 get _v_1;7:7 a _q;8:8 toehold _n_unknown;9:9 in _p;10:10 the _q;11:11 flat- _n_1;12:12 rolled _v_cause;13:14 _n_1 steel market;16:16 the _q;17:17 major _a_1;18:18 steelmakers _n_unknown;20:20 largest _a_1;22:22 prized _v_1;23:23 and _c;24:24 until _p;25:25 now _a_1;26:26 untouchable _a_unknown;1:1 also _a_1', '0:0 But _c;1:1 such _q;3:3 slab _n_1;3:3 slab _thin_a_1;4:4 technology _n_1;5:5 is _v_id;6:6 only _a_1;7:7 the _q;0:0 But _c', '0:0 Eager _a_1;1:1 engineers _n_1;2:2 espouse _v_1;4:4 steelmaking _n_unknown;4:4 steelmaking _direct_a_1;5:5 and _c;6:6 direct _a_1;7:7 casting _v_1;9:9 by _p;10:10 the _q;11:11 end _n_of;13:13 the _q;14:14 1990s year_range;16:16 enable _v_1;17:17 production _n_of;18:18 without _p;19:20 _n_1 coke ovens;21:21 and _c;22:23 _n_1 blast furnaces', '0:0 Those _q_dem;1:1 massive _a_1;2:2 structures _n_1;3:3 while _x;4:4 posing _v_as;5:5 cost _n_1;6:6 and _c;7:7 environmental _a_1;8:8 headaches _n_1;9:9 effectively _a_1;10:10 locked _v_out;12:12 all _q;13:13 but _p_except;15:15 pocketed _v_1;15:15 pocketed _deep_x_deg;16:16 giants _n_1;17:17 from _p;0:0 Those _q_dem', '1:1 \'s _v_there;2:2 a _q;3:3 revolution _n_of;4:5 ahead of _p;9:9 ultimately _a_1;10:10 change _v_cause;11:11 the _q;12:12 way _n_of;14:14 market _v_1;15:15 and _c;16:16 distribute _v_to;17:17 steel," _n_1;18:18 says _v_to;19:20 named William Dennis;21:21 vice _n_1;22:22 president _n_of;23:23 manufacturing _v_1;24:24 and _c;25:25 technology _n_1;26:26 for _p;27:27 the _q;28:29 named American Iron;30:30 Ore named;31:31 and _c;32:32 Steel named;1:1 \'s _v_there', "1:1 isn't _v_nv;3:3 major _a_1;4:4 steelmakers _n_unknown;6:6 blithely _a_unknown;7:7 ignored _v_1;8:8 high _a_1;1:1 isn't _v_nv", '0:1 In fact _a_1;4:4 spent _v_1;5:5 billions card;7:7 dollars _n_1;8:8 to _x;9:9 boost _v_to;10:10 the _q;11:11 percentage _n_of;13:13 continously _a_unknown;14:14 cast _v_1;15:15 steel _n_1;17:17 60.9 card;18:18 % _n_of;19:19 in _p_temp;20:20 1988 yofc;21:21 from _p;22:22 39.6 card;23:23 % _n_of;24:24 five card;25:25 years _n_1;0:1 In fact _a_1', '0:0 But _c;1:1 that _q_dem;0:0 But _c', "1:1 's _v_id;2:3 no longer _a_1;4:4 enough _q;5:5 to _x;6:6 beat _v_to;7:7 the _q;8:8 guy _n_1;9:9 down _p;10:10 the _q;1:1 's _v_id", '1:1 have _v_qmodal;3:3 beat _v_to;5:5 around _p;6:6 the _q;7:7 world," _n_of;8:8 says _v_to;9:9 Mr _n_1;1:1 have _v_qmodal', '1:3 _v_1 wants see;4:4 steelmakers _n_unknown;6:6 involved _a_1;7:7 in _p;8:8 computers _n_1;9:9 and _c;10:10 artificial _a_1;1:1 wants _v_1', '0:0 The _q;1:1 problem _n_of;2:2 : _p_namely;5:5 saddled _v_with;7:7 huge _a_1;8:8 plants _n_1;10:10 require _v_of;11:11 costly _a_1;0:0 The _q', '0:0 And _c;1:2 _v_1 try plying;3:3 new _a_1;4:4 dollars _n_1;5:5 free _a_1;6:6 in _p;7:7 a _q;8:8 market _n_1;11:12 _v_1 softening hurt;14:14 a _q;15:15 strong _a_1;16:16 dollar _n_1;17:17 and _c;18:18 concerned _a_about;20:20 overcapacity _n_unknown;22:22 the _q;23:23 industry _n_1;25:26 named Darth Vadar', '0:0 "The _q;1:1 technology _n_1;2:2 revolution _n_of;4:4 going _v_qmodal;7:7 very _x_deg;8:8 threatening _a_1;9:9 to _p;10:10 established _v_1;11:11 producers," _n_of;12:12 says _v_to;13:14 named Peter Marcus;15:15 an _q;16:16 analyst _n_1;17:17 with _p;18:18 PaineWebber named;0:0 "The _q', "2:2 got _v_state;5:5 invested _v_in;7:7 the _q;8:8 old _a_1;9:9 stuff _n_1;10:10 and _c;12:12 can't _v_modal;13:13 get _v_cause;15:15 workers _n_1;2:2 got _v_state", '0:1 No one _no_q;2:2 expects _v_1;3:3 minimills _n_unknown;5:5 eclipse _v_1;6:6 major _a_1;7:7 integrated _v_into;8:8 steelmakers _n_unknown;10:10 remain _v_1;11:11 the _q;12:12 undisputed _v_1;12:12 undisputed _un-_a_rvrs;13:13 kings _n_of;16:16 quality _n_1;16:16 quality _high_a_1;17:17 steel _n_1;18:18 used _v_1;19:19 for _p;20:20 autos _n_1;21:21 and _c;0:1 No one _no_q', '0:0 Moreover _a_1;1:1 the _q;2:2 process _n_of;4:4 without _p;0:0 Moreover _a_1', '0:0 Because _x;1:1 all _q;2:2 operations _n_of;4:4 connected _v_to-with;5:5 one card;6:7 _n_1 equipment failure;8:8 forces _v_1;9:9 a _q;10:10 complete _a_with;11:11 plant _n_1;0:0 Because _x', '0:0 On _p_temp;1:1 some _q;2:2 days _n_of;3:3 the _q;4:4 Nucor named;5:5 plant _n_1;7:7 produce _v_1;0:0 On _p_temp', '0:0 "At _p;1:1 this _q_dem;2:2 point _n_of;3:3 the _q;4:4 minimill _n_unknown;5:5 capacity _n_1;7:7 make _v_1;8:8 a _q;9:9 great _a_for;10:10 dent _n_1;11:11 in _p;12:12 the _q;13:13 integrated _v_into;14:14 market _n_1;15:15 but _c;18:18 challenge _v_1;21:21 develop _v_cause;22:22 new _a_1;23:23 markets," _n_1;24:24 says _v_to;25:26 named James McCall;27:27 vice _n_1;28:28 president _n_of;29:29 materials _n_1;30:30 at _p;31:31 Battelle named;32:32 a _q;33:33 technology _n_1;34:34 and _c;35:36 _n_1 management- research;37:37 giant _n_1;38:38 based _v_1;39:39 in _p;40:41 named Columbus Ohio', '0:0 Indeed _a_1;1:1 with _x_subord;2:2 demand _n_1;3:3 for _p;4:4 steel _n_1;6:6 growing _v_1;7:7 fast _a_1;9:9 to _x;10:10 absorb _v_1;11:11 capacity _n_1;12:12 steelmakers _n_unknown;14:14 have _v_qmodal;16:16 change _v_cause;17:17 the _q;18:18 way _n_of;20:20 do _v_1;0:0 Indeed _a_1', '0:0 In _p_state;1:1 the _q;2:2 past _n_1;3:3 says _v_to;4:4 Armco named;6:7 _n_1 chief economist;8:9 named John Corey;10:10 steelmakers _n_unknown;11:11 made _v_1;12:12 a _q;13:13 product _n_1;14:14 and _c;15:15 set _v_1;17:18 _p out on;19:19 the _q;20:20 loading _v_1;0:0 In _p_state', '0:0 Armco named;2:3 _n_of sales representatives;4:4 visit _v_1;5:5 the _q;6:7 named General Motors;8:8 Corp _n_1;10:10 Fairfax named;11:11 assembly _n_of;12:12 plant _n_1;13:13 in _p;14:15 named Kansas City;16:16 Mo named;17:17 two card;18:18 or _c;19:19 three card;20:20 days _n_of;21:21 a _p_per;0:0 Armco named', '0:0 When _x_subord;2:2 determined _v_1;4:4 GM named;5:5 needed _v_1;6:6 parts _n_1;8:8 quickly _a_1;9:9 Armco named;10:10 convinced _v_of;11:11 a _q;12:13 _n_1 steel service;14:14 center _n_of;16:16 build _v_1;17:17 a _q;18:18 processing _v_1;19:19 plant _n_1;20:20 nearby _p;21:21 so _x;22:22 shipments _n_of-to;23:23 could _v_modal;25:25 delivered _v_to;26:26 within _p;27:27 15 card;0:0 When _x_subord', '0:0 Cementing _v_1;1:1 such _q;2:2 relationships _n_of;3:3 with _p;4:4 major _a_1;5:7 _n_1 clients car;8:8 and _c;9:9 appliance _n_1;10:10 makers _n_of;13:13 a _q;14:14 means _n_of;16:16 survival _n_1;17:17 especially _x_deg;18:18 when _x_subord;19:19 those _q_dem;20:21 _n_1 key clients;23:23 relying _v_on;25:25 a _q;26:26 smaller _a_1;27:29 _n_of pool producers;30:30 and _c;31:31 flirting _v_unknown;32:32 with _p;33:33 plastic _a_1;34:34 and _c;35:35 aluminum _n_1;0:0 Cementing _v_1', '0:1 For example _a_1;2:2 when _x_subord;3:3 Detroit named;4:4 began _v_1;5:5 talking _v_about;6:6 about _p;8:8 bodied _n_1;8:8 bodied _plastic_a_1;9:9 cars _n_1;10:10 the _q;11:12 named American Iron;13:13 and _c;14:15 named Steel Institute;16:16 began _v_1;17:17 a _q;18:18 major _a_1;19:19 lobbying _v_1;20:20 effort _n_1;22:22 show _v_1;23:23 auto _n_1;24:24 makers _n_of;27:27 could _v_modal;28:28 use _v_1;29:29 steel _n_1;31:31 efficiently _a_1;32:32 by _p_means;33:33 simply _a_1;34:34 redesigning _v_1;34:34 redesigning _re-_a_again;36:36 a _q;37:38 _n_1 car door;0:1 For example _a_1', '0:0 But _c;1:1 steelmakers _n_unknown;2:2 must _v_modal;3:3 also _a_1;4:4 find _v_1;5:5 new _a_1;0:0 But _c', '0:0 After _x_h;1:1 letting _v_1;2:2 aluminum- _n_1;3:3 makers _n_of;4:4 take _v_1;5:5 the _q;6:6 recycling _v_1;7:7 lead _n_1;8:8 a _q;9:9 group _n_of;11:11 the _q;12:12 nation _n_of;14:14 largest _a_1;15:15 steelmakers _n_unknown;16:16 started _v_cause;17:17 a _q;18:18 recycling _v_1;19:19 institute _n_of;20:20 to _x;21:21 promote _v_1;22:23 _n_1 steel cans;24:24 to _p;25:25 an _q;26:26 environmentally _a_1;27:27 conscious _a_of;0:0 After _x_h', '0:0 Battelle named;2:2 Mr _n_1;3:3 McCall named;4:4 thinks _v_1;5:5 steelmakers _n_unknown;6:6 should _v_modal;7:7 concentrate _v_on;8:8 more _a_1;0:0 Battelle named', '0:1 named Weirton Steel;2:2 Corp _n_1;3:4 named Weirton W;5:5 Va named;9:9 touting _v_to;11:11 homeowners _n_1;12:12 fashionable _a_1;13:14 _n_1 steel doors;15:15 with _p;16:16 leaded _a_1;17:17 glass _n_of;18:18 inserts _n_1;19:19 as _p;20:20 a _q;21:21 secure _a_1;22:22 and _c;23:23 energy- _n_1;24:24 efficient _a_1;25:25 alternative _n_to;27:27 wooden _a_1;28:28 or _c;29:29 aluminum _n_1;0:0 Weirton named', '0:0 Other _a_1;1:1 steelmakers _n_unknown;2:2 envision _v_1;3:4 _n_1 steel roofs;5:5 covering _v_1;0:0 Other _a_1', '0:0 Still _a_1;1:1 others _n_1;3:3 looking _v_at;5:5 overseas _a_1;0:0 Still _a_1', '0:0 USX named;2:2 funneling _v_unknown;3:3 drilling _v_1;4:4 pipe _n_1;5:5 to _p;6:6 steel- _n_1;7:7 hungry _a_1;8:9 named Soviet Union', '0:0 This _q_dem;1:1 year _n_1;2:2 the _q;3:3 nation _n_of;5:5 largest _a_1;6:6 steelmaker _n_unknown;7:7 reactivated _v_1;7:7 reactivated _re-_a_again;9:9 overseas _a_1;10:11 _n_of sales operation', '0:0 Producers _n_of;1:1 also _a_1;3:5 _v_1 trying differentiate;6:6 by _p_means;7:7 concentrating _v_on;10:10 profit _n_1;10:10 profit _high_a_1;11:11 output _n_of;12:13 such as _p;14:14 coated _v_1;15:15 and _c;16:16 electrogalvanized _a_unknown;17:17 products _n_1;19:19 remain _v_1;20:20 beyond _p;21:21 the _q;22:22 reach _n_1;23:23 of _p;0:0 Producers _n_of', '0:0 Almost _x_deg;1:1 all _q;2:3 _n_1 capital- improvement;4:4 programs _n_of;5:5 announced _v_to;7:7 major _a_1;8:8 steelmakers _n_unknown;9:9 within _p;10:10 the _q;11:11 past _a_1;12:12 year _n_1;13:14 _v_1 involve building;15:15 electrogalvanizing _v_unknown;16:16 lines _n_of;17:19 _v_1 used produce;20:20 steel _n_1;21:21 for _p;22:22 such _q;23:23 products _n_1;24:24 as _p;25:26 _n_1 household appliances;27:27 and _c;28:29 _n_1 car doors', '0:0 But _c;1:1 unfortunately _a_1;2:2 that _q_dem;3:3 segment _n_1;5:5 much _x_deg;6:6 smaller _a_1;8:8 the _q;9:11 _n_1 bread- butter;12:12 flat- _n_1;13:13 rolled _v_cause;0:0 But _c', '2:2 like _p;4:4 climbing _v_1;5:6 out of _p_dir;7:7 the _q;8:9 named QE II;10:10 and _c;11:11 getting _v_state;12:12 into _p;13:13 a _q;14:14 lifeboat," _n_unknown;15:15 says _v_to;16:17 named John Jacobson;18:18 an _q;19:19 analyst _n_1;20:20 with _p;21:21 AUS named;2:2 like _p', '0:0 "After _p;1:2 a while _awhile_n_1;3:3 someone _q;4:4 has _v_qmodal;6:6 go _v_1;7:7 over _p_dir;8:8 the _q;0:0 "After _p', '0:0 Although _x;3:3 expect _v_1;4:4 any _q;5:5 bankruptcies _n_1;8:8 see _v_1;10:10 plants _n_1;12:12 sold _v_1;13:13 or _c;0:0 Although _x', '0:1 named Robert Crandall;2:2 with _p;3:3 the _q;4:5 named Brookings Institute;0:0 Robert named', '0:0 "Unless _x;2:2 is _v_there;3:3 an _q;4:4 enormous _a_1;5:5 rate _n_of;7:7 economic _a_1;8:8 growth _n_of;9:9 or _c;10:10 a _q;11:11 further _a_1;12:12 drop _n_of;13:13 in _p;14:14 the _q;15:15 dollar _n_1;18:18 unlikely _a_for;20:20 consumption _n_1;21:21 of _p;22:22 U.S named_n;23:23 produced _v_1;24:24 steel _n_1;26:26 grow _v_1;27:27 sufficiently _a_for;28:28 to _x;29:29 offset _v_1;30:30 the _q;31:31 growth _n_of;0:0 "Unless _x', '0:2 Not to mention _c;3:3 the _q;4:4 incursion _n_1;5:5 of _p;0:2 Not to mention _c', '0:0 Japanese _a_1;1:1 and _c;2:2 European _a_1;3:3 steelmakers _n_unknown;6:6 led _v_1;7:7 the _q;8:8 recent _a_1;9:9 technology _n_1;10:10 developments _n_of;12:12 anxiously _a_1;13:13 awaiting _v_1;14:14 the _q;15:15 lifting _v_1;16:16 of _p;17:17 trade _n_of;18:18 restraints _n_1;19:19 in _p_temp;0:0 Japanese _a_1', '0:0 Moreover _a_1;1:1 the _q;2:2 U.S named_n;3:3 can _v_modal;4:4 expect _v_1;6:6 competition _n_1;7:7 from _p;9:9 cost _n_1;9:9 cost _low_a_on;10:10 producing _v_1;11:12 named Pacific Rim;13:13 and _c;14:15 named Latin American;0:0 Moreover _a_1', '0:0 A _q;1:1 Taiwanese _a_1;2:2 steelmaker _n_unknown;3:3 recently _a_1;4:4 announced _v_to;5:5 plans _n_1;7:7 build _v_1;8:8 a _q;9:9 Nucor- named;9:9 Nucor- _a_1;0:0 A _q', '0:0 "People _n_of;1:1 think _v_of;3:3 the _q;4:5 _n_1 steel business;6:6 as _p;7:7 an _q;8:8 old _a_1;9:9 and _c;10:10 mundane _a_unknown;11:11 smokestack _n_unknown;12:12 business," _n_1;13:13 says _v_to;14:14 Mr _n_1;0:0 "People _n_of', '2:2 dead _n_1;2:2 dead _n_1', '0:1 named *USX LTV;2:3 named Bethlehem Inland;4:5 named Armco National;0:0 *USX named', '0:1 named *USX LTV;2:3 named Bethlehem Inland;4:5 named Armco National;0:0 *USX named', '0:0 **Projected _v_1', '0:0 Polaroid named;1:1 Corp _n_1;3:3 patent- _n_of;4:4 infringement _n_unknown;5:5 damages _n_1;6:6 case _n_of;7:7 against _p;8:9 named Eastman Kodak;10:11 _n_1 Co one;12:12 of _p;13:13 the _q;14:14 highest _a_1;15:15 stakes _n_1;16:16 corporate _a_1;17:17 trials _n_1;20:20 getting _v_1;21:21 scant _a_1;22:22 attention _n_to;23:23 on _p;24:25 named Wall Street', '0:0 After _p;1:1 78 card;2:2 days _n_of;3:3 of _p;4:4 mind- _n_1;5:5 numbing _v_cause;6:6 testimony _n_1;7:7 in _p;8:8 federal _a_1;9:9 court _n_of;10:10 in _p;11:11 Boston named;12:12 the _q;13:13 trial _n_1;16:17 all but _a_1;18:18 ignored _v_1;20:20 analysts _n_1;21:21 and _c;22:22 patent _n_of;0:0 After _p', '0:0 Most _q;2:2 read _v_1;3:3 the _q;4:4 pre-trial _n_1;4:4 pre-trial _pre-_a_ante;5:5 documents _n_1;6:6 however _a_1;7:7 and _c;8:8 estimate _v_at;9:9 Kodak named;12:12 ordered _v_1;14:14 pay _v_for;15:15 $ _n_1;16:17 card 1 billion;19:19 $ _n_1;20:21 card 1.5 billion;22:22 for _p;23:23 infringing _v_unknown;24:24 on _p;25:25 seven card;26:26 Polaroid named;0:0 Most _q', '0:0 That _q_dem;1:1 may _v_modal;2:2 be _v_id;3:3 the _q;4:4 largest _a_1;5:5 patent _n_of;6:6 award _n_for;8:8 but _c;11:11 well _x_deg;12:12 below _p;13:13 the _q;14:14 $ _n_1;15:16 card 12 billion;17:17 Polaroid named;0:0 That _q_dem', '0:0 The _q;1:1 highest _a_1;2:2 patent _n_of;3:3 damage _n_to;4:4 award _n_for;5:6 to date _p;8:8 in _p_temp;9:9 1986 yofc;10:10 when _x_subord;11:12 named Smith International;13:13 Inc _n_1;15:15 ordered _v_1;17:17 pay _v_for;18:18 $ _n_1;19:20 card 205 million;22:23 named Baker Hughes;24:24 Inc _n_1;25:25 for _p;26:26 infringing _v_unknown;27:27 on _p;28:28 a _q;29:29 patent _n_of;30:30 on _p;31:31 an _q;32:32 oil _n_1;33:33 drilling _v_1;34:35 _n_1 bit seal', '0:0 The _q;1:1 two card;2:2 companies _n_of;3:3 later _a_1;4:6 _v_1 agreed settle;7:7 for _p;8:8 $ _n_1;9:10 card 95 million', '1:1 analysts _n_1;2:2 think _v_1;5:5 worth _a_1;7:7 time _n_of;8:8 to _x;9:9 slog _v_unknown;10:10 through _p;11:11 the _q;12:12 Polaroid named;13:14 _n_1 trial testimony', '2:2 like _p;3:3 panning _v_1;4:4 for _p;5:5 gold _n_1;6:7 outside of _p;8:9 named Grand Central;2:2 like _p', '1:1 might _v_modal;2:2 find _v_1;3:3 something _q;4:4 but _c;5:5 the _q;6:6 chances _n_of;8:8 low," _a_on;9:9 said _v_to;10:11 named Michael Ellman;12:12 an _q;13:13 analyst _n_1;14:14 at _p;15:16 named Wertheim Schroder;1:1 might _v_modal', '0:0 And _c;1:2 named Eugene Glazer;3:3 an _q;4:4 analyst _n_1;5:5 at _p;6:7 named Dean Witter;8:8 Reynolds named;9:9 Inc _n_1;10:10 said _v_to;11:11 : _p_namely;12:12 "If _x_then;14:14 hired _v_1;15:15 an _q;16:16 attorney _n_1;19:19 there _a_1;20:21 _q all the;22:22 time _n_of;23:23 and _c;24:24 give _v_1;26:26 a _q;27:27 (prediction) _n_of;29:29 the _q;30:30 eventual _a_1;31:31 award _n_for;33:33 would _v_modal;35:35 willing _a_1;37:37 bet _v_on;40:40 would _v_modal;42:43 _p off" by;44:44 a _q;0:0 And _c', '0:0 A _q;2:2 day _n_of;2:2 day card;3:3 trial _n_1;4:4 in _p;5:5 the _q;6:6 early _a_1;7:7 1980s year_range;8:8 determined _v_1;10:10 Kodak named;11:11 based _v_1;12:12 in _p;13:14 named Rochester N.Y;15:15 infringed _v_unknown;16:16 on _p;17:17 patents _n_of;18:18 of _p;19:19 Polaroid named;20:20 of _p;21:22 named Cambridge Mass', '0:0 The _q;1:1 main _a_1;2:2 issues _n_1;3:3 remaining _v_1;4:4 are _v_nv;7:7 calculate _v_1;8:8 damages _n_1;9:9 and _c;11:11 the _q;12:12 infringement _n_unknown;14:14 "willful _a_1;15:15 and _c;0:0 The _q', '0:0 If _x_then;2:2 the _q;3:3 damages _n_1;4:4 could _v_modal;0:0 If _x_then', '0:0 Two card;1:1 analysts _n_1;4:4 read _v_1;5:5 the _q;6:6 transcripts _n_of;7:8 named David Nelson;9:9 of _p;10:11 named Shearson Lehman;12:12 Hutton named;13:13 Inc _n_1;14:14 and _c;15:16 named Calvert D;17:17 Crary named;18:18 a _q;19:20 _n_1 litigation analyst;21:21 at _p;22:23 named Labe Simpson;24:25 & Co _n_1;26:26 think _v_1;27:27 Judge _n_of;28:29 named A David;30:30 Mazzone named;32:32 decide _v_1;33:33 in _p;34:34 Kodak named;36:36 favor _n_1;37:37 on _p;38:38 the _q;39:39 "willful _a_1;40:40 and _c;41:41 deliberate" _a_1;0:0 Two card', '0:0 Mr _n_1;1:1 Crary named;2:2 said _v_to;3:3 testimony _n_1;4:4 by _p;5:5 Kodak named;7:7 patent _n_of;8:8 counsel _n_1;9:10 named Francis T;11:11 Carr named;12:12 of _p;13:13 Kenyon named;14:14 & _c;15:15 Kenyon named;16:19 _v_1 showed worked;20:20 with _p;21:21 Kodak named;22:22 continuously _a_1;23:23 from _p;24:24 the _q;25:25 outset _n_1;26:26 of _p;27:27 the _q;28:28 project" _n_1;29:29 in _p;30:30 an _q;31:31 effort _n_1;33:33 avoid _v_1;0:0 Mr _n_1', '0:0 "Carr named;1:1 told _v_1;2:2 Kodak named;3:3 on _p;5:5 occasions _n_of;7:7 avoid _v_1;8:8 various _a_1;9:9 features _n_1;10:11 because of _p;12:12 Polaroid named;14:15 _n_of patent positions,";16:16 and _c;17:17 Kodak named;18:18 followed _v_1;20:20 advice _n_1;21:21 in _p;22:22 every _q;23:23 instance _n_of;24:24 Mr _n_1;25:25 Crary named;0:0 "Carr named', '0:0 But _c;1:2 named Irving Kayton;3:3 a _q;4:4 patent _n_of;5:5 expert _n_1;6:6 at _p;7:8 named George Mason;9:10 named University School;11:11 of _p;12:12 Law named;15:15 familiar _a_with;17:17 the _q;18:18 case _n_of;19:19 said _v_to;20:20 the _q;21:21 fact _n_1;23:23 seven card;24:24 patents _n_of;26:26 infringed _v_unknown;27:27 "suggests _v_to;29:29 infringement _n_unknown;0:0 But _c', '2:2 difficult _a_for;5:5 that _x_deg;6:6 consistently _a_1;2:2 difficult _a_for', '0:0 Polaroid named;1:1 claims _v_1;3:3 could _v_modal;5:5 manufactured _v_1;6:6 and _c;7:7 sold _v_1;8:9 _q all the;10:10 instant _a_1;11:11 cameras _n_1;12:12 and _c;13:13 film _n_1;14:14 sold _v_1;16:16 Kodak named;17:17 if _x_then;18:18 Kodak named;20:20 entered _v_1;21:21 the _q;0:0 Polaroid named', "0:0 Moreover _a_1;1:1 Polaroid named;2:2 contends _v_1;4:4 could _v_modal;6:6 sold _v_1;8:8 at _p;9:9 a _q;10:10 higher _a_1;11:11 price _n_of;13:13 and _c;14:14 thus _a_1;15:15 made _v_1;16:16 higher _a_1;17:17 profits _n_1;19:19 because _x;21:21 wouldn't _v_modal;24:24 forced _v_1;26:26 match _v_cause;27:27 Kodak named;29:29 lower _a_1;0:0 Moreover _a_1", '0:0 Each _q;1:1 side _n_1;3:3 called _v_1;4:4 a _q;5:6 named Harvard Business;7:7 School named;8:8 professor _n_of;9:9 to _x;10:10 testify _v_to;11:11 on _p;12:12 that _q_dem;0:0 Each _q', '0:0 Kodak named;1:1 hired _v_1;2:3 named Robert Buzzell;4:4 and _c;5:5 Polaroid named;6:6 brought _v_in;8:9 named Robert J;0:0 Kodak named', "1:1 's _v_there;2:2 nothing _q;4:4 says _v_to;6:6 people _n_of;7:7 at _p;8:9 named Harvard Business;10:10 school _n_1;11:11 have _v_qmodal;13:13 agree _v_on;14:14 with _p;17:17 said _v_to;18:18 Mr _n_1;1:1 's _v_there", '0:0 Testimony _n_1;2:4 _v_1 expected continue;5:5 until _p;6:6 early _a_1;0:0 Testimony _n_1', '0:0 A _q;1:1 decision _n_1;3:3 expected _v_1;4:4 until _p;5:5 some _q;6:6 time _n_of;7:7 next _a_1;0:0 A _q', '0:0 The _q;1:1 main _a_1;2:2 reason _n_for;3:3 was _v_id;4:4 a _q;5:5 delay _n_1;6:6 in _p;7:7 shipment _n_of-to;9:9 new _a_1;11:11 end _n_of;11:11 end _high_a_1;12:13 _n_1 disk drives;14:14 a _q;15:15 business _n_1;17:17 accounts _v_for;19:19 some _q;20:20 10 card;21:21 % _n_of;23:23 IBM named;25:25 $ _n_1;26:27 card 60 billion;28:28 of _p;29:29 annual _a_1;0:0 The _q', '0:0 IBM named;2:2 telegraphed _v_1;3:3 the _q;4:4 poor _a_1;5:5 results _n_of;6:6 three card;7:7 weeks _n_1;8:8 ago _p;9:9 also _a_1;10:10 cited _v_for;11:11 an _q;12:12 increase _n_1;13:13 in _p;15:15 leasing _v_1;16:16 business _n_1;18:18 tends _v_1;20:20 lock _v_in;22:22 business _n_1;23:24 long- term _a_1;25:25 but _c;26:26 cut _v_1;27:27 revenue _n_1;28:28 in _p;29:29 the _q;30:30 near _a_to;0:0 IBM named', '0:0 In _p_state;1:1 addition _n_1;2:2 IBM named;3:3 noted _v_to;5:5 the _q;6:6 stronger _a_1;7:7 dollar _n_1;9:9 cut _v_1;10:10 the _q;11:11 value _n_1;12:12 of _p;13:13 overseas _a_1;14:14 revenue _n_1;15:15 and _c;16:16 earnings _n_1;17:17 when _x_subord;20:20 translated _v_into;0:0 In _p_state', '0:0 Earnings _n_1;1:1 fell _v_1;2:2 to _p;3:3 $ _n_1;4:5 card 877 million;6:6 or _c;7:7 $ _n_1;8:8 1.51 card;9:9 a _p_per;10:10 share _n_of;11:11 somewhat _x_deg;12:12 below _p;13:14 _n_1 securities analysts;16:16 revised _v_1;17:17 expectations _n_of;19:19 around _x_deg;20:20 $ _n_1;21:21 1.60 card;22:22 a _p_per;0:0 Earnings _n_1', '0:0 That _q_dem;1:1 compared _v_with;3:3 the _q;4:5 year- earlier _a_1;6:6 $ _n_1;7:8 card 1.25 billion;9:9 or _c;10:10 $ _n_1;11:11 2.10 card;12:12 a _p_per;13:13 share _n_of;14:14 - excl;17:17 inflated _v_cause;18:18 by _p;19:19 a _q;21:21 cents- _n_1;21:21 cents- card;23:23 share _n_of;24:24 gain _n_1;25:25 from _p;26:26 the _q;27:27 sale _n_of;29:29 some _q;30:31 named MCI Communications;32:33 _n_1 Corp stock;34:34 and _c;35:35 by _p;36:36 an _q;37:37 unspecified _a_1;38:38 amount _n_of;39:39 from _p;40:40 a _q;41:41 payment _n_of;42:42 by _p;43:43 Fujitsu named;44:44 Ltd _n_1;45:45 relating _v_to;47:47 a _q;48:49 _n_1 software dispute', '0:0 Revenue _n_1;1:1 climbed _v_1;2:2 4.3 card;3:3 % _n_of;4:4 to _p;5:5 $ _n_1;6:7 card 14.31 billion;8:8 from _p;9:9 $ _n_1;10:11 card 13.71 billion', '0:1 named IBM Armonk;2:2 N.Y named;3:3 remained _v_1;0:0 IBM named', '0:0 The _q;1:2 _n_1 computer giant;4:4 U.S named_n;5:5 results _n_of;8:8 dismal _a_1;9:9 for _p;10:10 years _n_1;11:11 noted _v_to;13:13 revenue _n_1;14:14 rose _v_1;15:15 again _a_1;16:16 in _p;17:17 the _q;18:18 U.S named_n;19:19 in _p_temp;20:20 the _q;21:21 third ord;22:22 quarter _n_temp;23:23 following _v_1;24:24 an _q;25:25 increase _n_1;26:26 in _p;27:27 the _q;28:28 second ord;0:0 The _q', '0:0 The _q;1:1 company _n_of;2:2 said _v_to;3:3 in _p;4:4 a _q;5:5 statement _n_of;7:7 "demand _n_1;8:8 for _p;9:9 IBM named;10:10 products _n_1;11:11 and _c;12:12 services _n_1;13:13 continues _v_2;16:16 good _a_at-for-of;17:17 world- _n_of;0:0 The _q', '3:3 see _v_1;4:4 anything _q;5:5 in _p;6:6 the _q;7:7 fundamentals _n_1;8:8 of _p;10:10 business _n_1;12:12 would _v_modal;13:13 cause _v_1;16:16 change _v_cause;18:18 strategy _n_of;20:20 investing _v_1;21:21 for _p;22:22 profitable _a_for;3:3 see _v_1', '0:1 _n_1 Securities analysts;3:3 remained _v_1;0:0 Securities _n_1', '1:1 think _v_1;2:2 1990 yofc;4:4 be _v_id;5:5 another _q;6:6 mediocre _a_unknown;7:7 year," _n_1;8:8 said _v_to;9:10 named Steve Milunovich;11:11 of _p;12:13 named First Boston', '0:1 named Jay Stevens;2:2 of _p;3:4 named Dean Witter;5:5 actually _a_1;6:6 cut _v_1;9:9 share _n_of;9:9 share _per_a_1;10:10 earnings _n_1;11:11 estimate _n_of;12:12 to _p;13:13 $ _n_1;14:14 9 card;15:15 from _p;16:16 $ _n_1;17:17 9.50 card;18:18 for _p;19:19 1989 yofc;20:20 and _c;21:21 to _p;22:22 $ _n_1;23:23 9.50 card;24:24 from _p;25:25 $ _n_1;26:26 10.35 card;27:27 in _p_temp;28:28 1990 yofc;29:29 because _x;31:31 decided _v_1;32:32 sales _n_of;33:33 would _v_modal;35:35 even _x_deg;36:36 weaker _a_1;0:0 Jay named', '0:0 Both _q;0:0 Both card;1:1 estimates _n_of;2:2 would _v_modal;3:3 mark _v_1;4:4 declines _n_1;5:5 from _p;6:6 the _q;7:7 1988 yofc;8:8 net _n_1;9:9 of _p;10:10 $ _n_1;11:12 card 5.81 billion;13:13 or _c;14:14 $ _n_1;15:15 9.80 card;16:16 a _p_per;17:17 share _n_of;21:21 well _x_deg;22:22 below _p;23:23 the _q;24:24 record _n_of;25:25 IBM named;26:26 set _v_1;27:27 in _p_temp;0:0 Both _q', '0:0 Mr _n_1;1:1 Stevens named;2:2 said _v_to;4:4 kept _v_prd;5:5 a _q;6:6 "buy _n_1;8:8 hold" _n_of;9:9 recommendation _n_1;10:10 on _p;11:11 the _q;12:12 stock _n_1;13:13 only _x_deg;14:14 because _x;15:16 _q "all the;17:17 damage _n_to;0:0 Mr _n_1', '1:1 said _v_to;2:2 the _q;3:3 stock _n_1;5:5 traded _v_1;6:6 below _p;7:7 1 card;8:8 1/2 fraction;9:9 times named_n;10:10 book _n_of;11:11 value _n_1;12:12 over _p;13:13 the _q;14:14 past _a_1;15:15 10 card;16:16 years _n_1;18:18 at _p_temp;19:19 the _q;20:20 moment _n_1;21:21 computes _v_1;22:22 to _p;23:23 a _q;24:24 stock _n_1;25:25 price _n_of;27:27 $ _n_1;1:1 said _v_to', '0:0 The _q;1:1 stock _n_1;2:2 closed _v_1;3:3 yesterday _a_1;4:4 at _p;5:5 $ _n_1;6:6 103 card;7:7 a _p_per;8:8 share _n_of;9:9 up _p;10:10 just _x_deg;11:11 $ _n_1;12:12 1 card;13:13 in _p;14:14 composite _n_1;15:15 trading _v_1;16:16 on _p;17:17 the _q;18:19 named New York;20:21 _n_1 Stock Exchange;22:22 as _x_subord;23:23 the _q;24:24 market _n_1;0:0 The _q', '0:0 Analysts _n_1;1:1 worry _v_about;3:3 the _q;4:5 _n_1 disk- drive;6:6 and _c;7:7 leasing _v_1;8:8 problems _n_of;10:10 last _v_1;11:12 at least _x_deg;13:13 through _p;14:14 the _q;15:15 first ord;0:0 Analysts _n_1', '0:0 "And _c;1:1 the _q;2:2 input _n_to;5:5 had _v_1;6:6 from _p;7:7 customers _n_of;8:8 is _v_nv;11:11 still _a_1;12:12 could _v_modal;13:13 be _v_id;14:14 a _q;0:0 "And _c', '0:0 On _p_state;1:1 leasing _v_1;2:3 named Bob Djurdjevic;4:4 at _p;5:6 named Annex Research;7:7 said _v_to;9:9 thinks _v_1;10:10 IBM named;12:12 hurt _v_1;0:0 On _p_state', '1:1 said _v_to;2:2 IBM named;4:4 priced _v_1;6:6 leases _n_1;7:7 aggressively _a_1;8:8 thinking _v_1;9:9 that _q_dem;10:10 would _v_modal;11:11 help _v_to;12:12 win _v_1;1:1 said _v_to', '0:0 But _c;2:2 said _v_to;3:3 IBM named;4:4 would _v_modal;6:6 won _v_1;7:7 the _q;8:8 business _n_1;9:9 anyway _a_1;10:10 as _p;11:11 a _q;12:12 sale _n_of;13:13 to _p;14:14 a _q;15:15 third ord;16:16 party _n_of-to;18:18 would _v_modal;20:20 then _a_1;21:21 leased _v_to;22:22 the _q;23:23 equipment _n_1;25:25 the _q;0:0 But _c', '1:1 said _v_to;2:2 IBM named;4:5 not only _a_1;6:6 hurt _v_1;9:9 term _n_of;9:9 term _short_a_of;10:11 _n_1 revenue outlook;12:12 but _c;14:14 also _a_1;16:16 losing _v_1;17:17 money _n_1;18:18 on _p;1:1 said _v_to', '0:1 named Bob Bardagy;2:3 _n_1 executive vice;4:4 president _n_of;6:6 marketing _v_1;7:7 at _p;8:8 Comdisco named;9:9 Inc _n_1;10:10 a _q;11:11 huge _a_1;12:12 leasing _v_1;13:13 firm _n_1;14:14 said _v_to;15:15 : _p_namely;16:16 "To _x;17:17 put _v_adv;19:19 mildly _a_1;20:21 named IBM Credit;24:24 doing _v_1;25:27 _q some the;28:29 _a_1 worst economic;30:30 deals _n_1;31:31 of _p;32:32 any _q;33:33 leasing _v_1;34:34 company _n_of;37:37 ever _a_1;0:0 Bob named', '0:0 IBM named;2:4 _v_1 expected get;5:5 a _q;6:6 boost _n_1;7:7 soon _p;8:8 when _x_subord;10:10 announces _v_to;11:11 some _q;12:12 new _a_1;13:13 versions _n_of;0:0 IBM named', '0:0 But _c;1:1 the _q;2:2 basic _a_1;3:3 technology _n_1;4:4 in _p;5:5 the _q;6:6 line _n_of;8:8 almost _x_deg;9:9 five card;10:10 years _n_1;11:11 old _a_1;13:13 means _v_to;16:16 long _a_1;17:17 in _p;18:18 the _q;19:19 tooth _n_1;20:20 and _c;21:21 competitors _n_1;23:23 rolling _v_out;25:25 strong _a_1;26:26 products _n_1;27:27 of _p;0:0 But _c', '0:0 IBM named;2:2 gaining _v_1;3:3 momentum _n_1;4:4 in _p;5:5 the _q;7:7 computer _n_1;7:7 computer _personal_a_1;8:8 market _n_1;9:9 and _c;11:11 expected _v_1;13:13 introduce _v_to;14:14 some _q;15:15 impressive _a_1;16:16 workstations _n_1;17:18 _a_1 early next;0:0 IBM named', '0:0 But _c;3:3 hard _a_for;5:5 squeeze _v_1;7:7 profit _n_1;8:9 out of _p;10:10 the _q;12:12 computer _n_1;12:12 computer _personal_a_1;13:13 business _n_1;14:14 these _q_dem;15:15 days _n_of;16:16 and _c;17:17 the _q;18:19 _n_1 workstation market;20:20 while _x;21:21 important _a_for;24:24 small _a_1;26:26 rely _v_on;28:28 for _p;0:0 But _c', '0:0 The _q;1:2 _n_1 disk drives;4:4 doubtless _a_1;5:5 sell _v_1;6:6 well _a_1;7:7 when _x_subord;9:9 finally _a_1;10:10 become _v_id;0:0 The _q', '0:0 But _c;1:1 the _q;2:3 named AS/400 IBM;5:6 _a_1 highly successful;7:7 minicomputer _n_unknown;8:8 line _n_of;10:10 losing _v_1;12:12 momentum _n_1;13:13 and _c;14:14 some _q;15:15 analysts _n_1;16:16 said _v_to;17:17 sales _n_of;18:18 could _v_modal;19:19 even _a_1;20:20 decline _v_1;21:21 in _p_temp;22:22 the _q;23:23 fourth ord;0:0 But _c', '0:0 In _p_state;1:1 addition _n_1;2:2 IBM named;4:4 growth _n_of;5:5 in _p;6:6 software _n_1;7:7 in _p_temp;8:8 the _q;9:9 third ord;10:10 quarter _n_temp;11:11 was _v_id;12:12 just _x_deg;13:13 8.8 card;14:14 % _n_of;15:15 well _x_deg;16:16 below _p;17:17 historical _a_1;18:18 levels _n_1;19:19 even _x_deg;21:21 adjusted _v_to;22:22 to _x;23:23 reflect _v_1;24:24 last _a_1;25:25 year _n_1;27:27 payment _n_of;28:28 from _p;29:29 Fujitsu named;30:30 and _c;31:31 the _q;32:32 stronger _a_1;0:0 In _p_state', '0:0 And _c;1:1 expenses _n_1;2:2 up _p;3:3 7.9 card;4:4 % _n_of;5:5 in _p_temp;6:6 the _q;7:7 quarter _n_temp;9:9 stayed _v_prd;10:10 stubbornly _a_unknown;0:0 And _c', '0:0 In _p_state;1:1 the _q;2:2 nine card;3:3 months _n_1;4:4 IBM named;5:5 earned _v_1;6:6 $ _n_1;7:8 card 3.17 billion;9:9 or _c;10:10 $ _n_1;11:11 5.43 card;12:12 a _p_per;13:13 share _n_of;14:14 down _p;15:15 8.4 card;16:16 % _n_of;17:17 from _p;18:18 the _q;19:20 year- earlier _a_1;21:21 $ _n_1;22:23 card 3.46 billion;24:24 or _c;25:25 $ _n_1;26:26 5.83 card;27:27 a _p_per;0:0 In _p_state', '0:0 Revenue _n_1;1:1 increased _v_1;2:2 6.5 card;3:3 % _n_of;4:4 to _p;5:5 $ _n_1;6:7 card 42.25 billion;8:8 from _p;9:9 $ _n_1;10:11 card 39.68 billion', '0:1 named D Wayne;2:2 Calloway named;3:3 also _a_1;4:5 _n_1 chief executive;6:6 officer _n_1;7:7 of _p;8:8 the _q;9:9 company _n_of;10:13 _v_1 indicated expects;14:14 analysts _n_1;16:16 raise _v_cause;18:18 forecasts _n_1;19:19 for _p;20:20 1989 yofc;21:21 after _x_h;22:22 the _q;23:23 company _n_of;24:24 releases _v_1;26:26 earnings _n_1;0:0 D named', '1:1 far _a_1;2:2 analysts _n_1;4:4 said _v_to;7:7 looking _v_for;9:9 $ _n_1;10:10 3.30 card;11:11 to _p;12:12 $ _n_1;13:13 3.35 card;14:14 a _p_per;1:1 far _a_1', '0:0 After _p;1:1 today _a_1;3:3 announcement _n_of;4:4 that _q_dem;5:5 range _n_of;6:6 could _v_modal;7:7 increase _v_1;8:8 to _p;9:9 $ _n_1;10:10 3.35 card;11:11 to _p;12:12 $ _n_1;13:13 3.40 card;14:14 a _p_per;0:0 After _p', '0:0 The _q;1:1 official _n_1;2:2 said _v_to;4:4 also _a_1;5:5 would _v_modal;7:7 comfortable _a_1;8:8 with _p;9:9 that _q_dem;10:10 new _a_1;0:0 The _q', '0:0 In _p_temp;1:1 1988 yofc;2:2 the _q;4:4 drink _n_1;4:4 drink _soft_a_1;5:5 giant _n_1;6:6 earned _v_1;7:7 $ _n_1;8:8 2.90 card;9:9 a _p_per;0:0 In _p_temp', '0:0 Results _n_of;1:1 for _p;2:2 1989 yofc;4:4 include _v_1;5:5 about _x_deg;6:6 40 card;7:7 cents _n_1;8:8 a _p_per;9:9 share _n_of;10:10 from _p;11:11 the _q;12:12 dilutive _a_unknown;13:13 effects _n_1;14:14 of _p;15:16 _n_1 snack- food;17:17 and _c;18:18 bottling _v_1;19:20 _n_of company acquisitions', '0:0 In _p_state;1:1 composite _n_1;2:2 trading _v_1;3:3 on _p;4:4 the _q;5:6 named New York;7:8 _n_1 Stock Exchange;9:9 the _q;10:10 company _n_of;11:11 closed _v_1;12:12 yesterday _a_1;13:13 at _p;14:14 $ _n_1;15:15 57.125 card;16:16 a _p_per;17:17 share _n_of;18:18 up _p;19:19 $ _n_1;0:0 In _p_state', '0:0 The _q;1:1 company _n_of;2:2 said _v_to;4:4 quarter _n_temp;4:4 quarter ord;5:5 sales _n_of;7:9 _v_1 expected increase;10:10 25 card;11:11 % _n_of;12:12 from _p;13:13 $ _n_1;14:15 card 3.12 billion;16:16 of _p;17:17 last _a_1;18:18 year _n_1;20:20 third ord;0:0 The _q', '0:0 Mr _n_1;1:1 Calloway named;2:2 blamed _v_on;3:3 the _q;4:4 slower _a_1;5:5 volume _n_of;7:7 rainier _a_1;8:8 weather _n_1;9:9 a _q;10:10 dearth _n_1;11:11 of _p;12:12 new _a_1;13:13 products _n_1;14:14 in _p;15:15 the _q;16:16 industry _n_1;17:17 and _c;19:19 to _p;20:20 a _q;21:21 much _x_deg;22:22 lesser _a_1;23:23 extent _n_of;0:0 Mr _n_1', '0:0 PepsiCo named;1:1 said _v_to;4:4 drink _n_1;4:4 drink _soft_a_1;5:5 prices _n_of;7:7 about _x_deg;8:8 2 card;9:9 % _n_of;10:10 higher _a_1;11:11 in _p_temp;12:12 the _q;0:0 PepsiCo named', "0:0 Mr _n_1;1:1 Calloway named;2:2 also _a_1;3:3 noted _v_to;6:6 drink _n_1;6:6 drink _soft_a_1;7:7 volume _n_of;8:8 rose _a_1;9:9 a _q;10:10 hefty _a_1;11:11 9 card;12:12 % _n_of;13:13 in _p_temp;14:14 last _a_1;15:15 year _n_1;16:16 's _v_id;17:17 third ord;18:18 quarter _v_1;19:19 making _v_cause;20:20 the _q;21:21 comparison _n_1;0:0 Mr _n_1", '0:0 International _a_1;2:2 drink _n_1;2:2 drink _soft_a_1;3:3 volume _n_of;5:5 up _p;6:6 about _x_deg;7:7 6 card;0:0 International _a_1', '0:1 _n_1 Snack- food;2:2 tonnage _n_unknown;3:3 increased _v_cause;4:4 a _q;5:5 strong _a_1;6:6 7 card;7:7 % _n_of;8:8 in _p_temp;9:9 the _q;10:10 third ord;11:11 quarter _n_temp;12:12 while _x;13:13 domestic _a_1;14:14 profit _n_1;15:15 increased _v_1;16:16 in _p;17:17 double _a_1;18:19 _n_1 digits Mr;20:20 Calloway named;0:0 Snack- _n_1', '0:0 Excluding _v_1;1:1 the _q;2:2 British _a_1;3:4 _n_1 snack- food;5:5 business _n_1;6:6 acquired _v_1;7:7 in _p_temp;8:8 July mofy;9:10 _n_1 snack- food;11:11 international _a_1;12:12 tonnage _n_unknown;13:13 jumped _v_1;14:14 40 card;15:15 % _n_of;16:16 with _x_subord;17:17 sales _n_of;18:18 strong _a_1;19:19 in _p;20:21 named Spain Mexico;22:22 and _c;0:0 Excluding _v_1', '0:0 Total _a_1;1:2 _n_1 snack- food;3:3 profit _n_1;4:4 rose _v_1;5:5 30 card;0:0 Total _a_1', '0:0 Led _v_1;2:3 named Pizza Hut;4:4 and _c;5:6 named Taco Bell;7:8 _n_1 restaurant earnings;9:9 increased _v_1;10:10 about _x_deg;11:11 25 card;12:12 % _n_of;13:13 in _p_temp;14:14 the _q;15:15 third ord;16:16 quarter _n_temp;17:17 on _p;18:18 a _q;19:19 22 card;20:21 _n_of % sales;0:0 Led _v_1', '1:1 store _n_1;1:1 store _same_a_as;2:2 sales _n_of;3:3 for _p;4:5 named Pizza Hut;6:6 rose _v_1;7:7 about _x_deg;8:8 13 card;9:9 % _n_of;10:10 while _x;11:12 named Taco Bell;14:14 increased _v_1;15:15 22 card;16:16 % _n_of;17:17 as _x_subord;18:18 the _q;19:19 chain _n_of;20:20 continues _v_2;22:22 benefit _v_from;25:25 price- _n_of;26:26 value _n_1;1:1 store _n_1', '0:1 named Taco Bell;3:3 turned _v_around;5:5 declining _v_1;6:6 customer _n_of;7:7 counts _n_1;8:8 by _p_means;9:9 permanently _a_1;10:10 lowering _v_cause;11:11 the _q;12:12 price _n_of;0:0 Taco named', '0:0 Same _a_as;1:1 store- _n_1;2:2 sales _n_of;3:3 for _p;4:5 named Kentucky Fried;6:6 Chicken named;9:9 struggled _v_1;10:10 with _p;11:11 increased _v_cause;12:12 competition _n_1;13:13 in _p;14:14 the _q;16:16 food _n_1;16:16 food _fast_a_1;17:18 _n_1 chicken market;19:19 and _c;20:20 a _q;21:21 lack _n_1;22:22 of _p;23:23 new _a_1;24:24 products _n_1;25:25 rose _v_1;26:26 only _x_deg;27:27 1 card;0:0 Same _a_as', '0:0 The _q;1:1 operation _n_of;5:5 slow _a_1;7:7 respond _v_to;9:9 consumers _n_of;11:11 shifting _v_1;12:12 tastes _n_of;13:14 _p away from;15:15 fried _v_cause;16:16 foods _n_1;19:19 developing _v_cause;20:20 a _q;22:22 chicken _n_1;22:22 chicken _grill_v_1;23:23 product _n_1;25:25 may _v_modal;27:27 introduced _v_to;28:28 nationally _a_1;29:29 at _p;30:30 the _q;31:31 end _n_of;33:33 next _a_1;0:0 The _q', '0:0 The _q;1:1 new _a_1;2:2 product _n_1;4:4 performed _v_1;5:5 well _a_1;6:6 in _p;7:7 a _q;8:8 market _n_1;9:9 test _n_of;10:10 in _p;11:12 named Las Vegas;13:13 Nev named;14:14 Mr _n_1;15:15 Calloway named;0:0 The _q', '0:0 But _c;2:2 never _a_1;3:3 can _v_modal;4:4 tell," _v_1;6:6 added _v_to;8:8 have _v_qmodal;10:10 take _v_of-i;11:11 advantage _n_i;0:0 But _c', '0:0 President _n_of;1:1 Bush named;2:2 chose _v_from;3:4 named Martin Allday;5:5 a _q;6:6 longtime _a_unknown;7:7 friend _n_1;8:8 from _p;9:9 Texas named;11:11 be _v_id;12:12 chairman _n_of;13:13 of _p;14:14 the _q;15:16 named Federal Energy;17:18 named Regulatory Commission', '0:0 Mr _n_1;1:1 Allday named;2:2 would _v_modal;3:3 succeed _v_1;4:5 named Martha Hesse;0:0 Mr _n_1', '0:0 The _q;1:1 White named;2:2 House named_n;3:3 said _v_to;4:4 Ms _n_1;5:5 Hesse named;6:6 a _q;7:7 Chicago named;8:8 businesswoman _n_1;10:10 previously _p;11:11 held _v_1;12:12 posts _n_of;13:13 at _p;14:14 the _q;15:16 named Energy Department;17:17 and _c;18:18 FERC named;20:20 leaving _v_1;21:21 to _x;22:22 become _v_id;23:23 a _q;24:24 vice _n_1;25:25 president _n_of;27:28 named First Chicago;0:0 The _q', '1:1 met _v_1;2:2 Mr _n_1;3:3 Bush named;4:4 in _p;5:5 the _q;6:6 1950s year_range;7:7 when _x_subord;8:8 the _q;9:9 president _n_of;10:10 was _v_id;11:11 a _q;12:12 young _a_1;13:14 _n_1 oil man;15:15 in _p;16:16 Midland named;17:17 and _c;18:18 Mr _n_1;19:19 Allday named;20:20 was _v_id;21:21 a _q;22:22 lawyer _n_1;23:23 for _p;24:24 an _q;25:26 _n_1 oil firm', '0:0 The _q;1:1 FERC named;2:2 is _v_id;3:3 a _q;5:5 member _n_of;5:5 member card;6:6 commission _n_of;8:8 regulates _v_1;9:9 billions card;11:11 dollars _n_1;12:12 of _p;13:13 interstate _n_1;14:14 wholesale _a_1;15:16 _n_1 energy transactions', '0:0 Mr _n_1;1:1 Allday named;3:3 appointment _n_1;5:5 subject _a_to;7:7 confirmation _n_of;8:8 by _p;9:9 the _q;0:0 Mr _n_1', '0:0 Administration _n_of;1:1 officials _n_1;2:2 said _v_to;3:3 a _q;4:4 date _n_of;5:5 for _p;6:6 Ms _n_1;7:7 Hesse named;9:9 departure _n_1;0:0 Administration _n_of', '0:1 named CALIFORNIA REAL;2:3 named ESTATE INVESTMENT;4:4 Corp _n_1;5:5 said _v_to;7:7 directors _n_of;8:8 declared _v_to;9:9 a _q;10:10 dividend _n_1;11:11 of _p;12:12 five card;13:13 cents _n_1;14:14 per _p;17:17 common _a_for;18:18 stock _n_1;19:19 payable _a_to;20:20 Nov mofy;21:21 6 dofm;23:23 stock _n_1;24:24 of _p;25:25 record _n_of;26:26 Oct mofy;0:0 CALIFORNIA named', '0:0 The _q;1:1 company _n_of;2:2 said _v_to;4:6 _v_1 hopes resume;8:8 schedule _n_1;9:9 of _p;10:11 _a_1 regular quarterly;12:12 dividends _n_1;13:13 at _p;14:14 the _q;15:15 end _n_of;17:17 this _q_dem;0:0 The _q', '0:1 named Hydro- Quebec;2:2 said _v_to;4:4 notified _v_1;5:6 named Central Maine;7:7 Power named;8:8 Co _n_1;11:11 cancel _v_1;12:12 a _q;13:13 $ _n_1;14:15 card 4 billion;16:16 contract _n_1;18:18 supply _v_1;19:19 electricity _n_1;20:20 to _p;21:21 the _q;22:22 Maine named;0:0 Hydro- named', '0:0 The _q;1:1 provincially _a_unknown;2:2 owned _v_1;3:3 utility _n_1;4:4 said _v_to;7:7 tearing _v_up;9:9 the _q;10:10 deal _n_1;11:11 because _x;12:12 "the _q;13:15 _n_1 contract objectives;16:16 can\'t _v_modal;0:0 The _q', '0:1 named Hydro- Quebec;2:2 said _v_to;3:3 Maine named;4:4 regulators _n_1;6:6 refusal _n_of;8:8 approve _v_1;9:9 the _q;10:10 contract _n_1;11:11 earlier _a_1;12:12 this _q_dem;13:13 year _n_1;14:14 halted _v_1;15:15 work _n_1;16:16 on _p;17:18 _n_of transmission lines;19:19 and _c;20:20 stopped _v_1;21:21 negotiations _n_of;22:22 for _p;23:23 resale _n_unknown;24:24 of _p;25:25 electricity _n_1;26:26 carried _v_1;27:27 through _p;28:28 Maine named;29:29 to _p;30:30 other _a_1;0:0 Hydro- named', '1:1 would _v_modal;2:4 _a_1 now physically;5:5 impossible _a_for;7:7 begin _v_1;8:8 deliveries _n_of;9:9 in _p_temp;10:10 1992," yofc;11:11 a _q;12:13 named Hydro- Quebec;14:14 official _n_1;1:1 would _v_modal', '0:0 The _q;1:1 contract _n_1;4:4 run _v_1;5:5 from _p_time;6:6 1992 yofc;7:7 to _p;0:0 The _q', '0:1 named Hydro- Quebec;2:2 said _v_to;3:3 Maine named;4:4 regulators _n_1;6:6 refusal _n_of;8:8 approve _v_1;9:9 the _q;10:10 contract _n_1;11:11 means _v_to;12:13 named Central Maine;14:14 Power named;16:16 lost _v_1;18:18 place _n_of;19:19 in _p;0:0 Hydro- named', '2:2 sign _v_1;3:3 any _q;4:4 new _a_1;5:5 contracts _n_1;6:6 {with _p;7:7 deliveries} _n_of;8:8 beginning _v_1;9:9 earlier _a_1;11:11 2000," yofc;12:12 the _q;13:14 named Hydro- Quebec;15:15 official _n_1;2:2 sign _v_1', '1:1 said _v_to;2:3 named Hydro- Quebec;4:4 already _a_1;5:5 has _v_1;6:6 some _q;7:7 "customers _n_of;8:8 in _p;9:9 mind" _n_1;10:10 for _p;11:11 the _q;12:12 power _n_1;17:17 delivered _v_to;1:1 said _v_to', '0:0 "Nothing _q;2:2 happened _v_1;3:3 since _x_subord;5:5 signed _v_1;6:6 the _q;7:7 contract _n_1;8:8 to _x;9:9 undermine _v_1;11:11 conviction _n_of;13:14 named Hydro- Quebec;15:15 was _v_id;16:16 the _q;18:18 cost _n_1;18:18 cost _low_a_1;20:20 environmentally _a_1;21:21 acceptable _a_for;22:22 choice _n_of;23:23 for _p;24:24 meeting _v_1;25:25 a _q;26:26 part _n_1;27:27 of _p;29:29 customers _n_of;31:31 energy _n_1;32:32 needs _n_of;33:33 through _p;34:34 the _q;35:35 year _n_1;36:36 2020," card;37:37 said _v_to;38:39 named Central Maine;40:40 senior _a_1;41:41 vice _n_1;42:42 president _n_of;43:44 named Donald F;0:0 "Nothing _q', '0:1 named Central Maine;2:2 said _v_to;5:5 evaluating _v_1;7:8 _n_1 energy options";9:9 to _x;10:10 make _v_up-for;12:12 for _p;13:13 the _q;14:14 lost _v_1;15:15 future _a_1;16:16 power _n_1;17:17 including _v_1;18:18 new _a_1;19:19 energy _n_1;20:20 generation _n_of;21:21 and _c;22:23 _n_1 management proposals;24:24 from _p;25:26 named New England;27:27 and _c;28:28 possibly _x_deg;29:30 _a_1 new Canadian;0:0 Central named', '0:0 CHICAGO named;2:3 _n_1 Options traders;5:5 among _p;6:6 the _q;7:7 big _a_1;8:8 victims _n_of;10:10 Friday dofw;12:12 plunging _v_into;13:14 _n_1 stock market;15:15 including _v_1;16:16 one card;17:17 small _a_1;18:18 firm _n_1;20:20 required _v_of;21:21 an _q;22:23 _n_1 emergency $;24:25 card 50 million;0:0 CHICAGO named', '0:0 While _x;1:1 Monday dofw;3:3 rebounding _v_1;3:3 rebounding _re-_a_again;4:4 markets _n_1;5:5 helped _v_to;6:6 other _a_1;7:7 investors _n_1;8:8 recoup _v_unknown;9:9 losses _n_of;11:11 options _n_1;12:12 customers _n_of;13:13 and _c;14:14 professional _a_1;15:15 traders _n_1;16:16 in _p;17:18 _n_1 stock- index;19:19 options _n_1;20:20 and _c;21:21 the _q;22:22 options _n_1;23:23 on _p;24:24 takeover _n_of;25:25 stocks _n_1;27:27 left _v_1;28:28 with _p;29:29 multimillion- _n_unknown;30:30 dollar _n_1;31:31 losses _n_of;32:32 traders _n_1;33:33 here _a_1;34:34 and _c;35:35 in _p;36:37 named New York;0:0 While _x', '0:1 _n_1 Options traders;3:3 hurt _v_1;4:4 worse _a_1;6:6 others _n_1;7:7 on _p_temp;8:8 Friday dofw;9:10 because of _p;11:11 the _q;12:13 _a_1 highly volatile;14:14 nature _n_1;15:15 of _p;16:16 options _n_1;18:18 often _a_1;19:19 rise _v_1;20:20 or _c;21:21 fall _v_1;22:22 in _p;23:23 value _n_1;24:24 several _a_1;25:25 times _n_1;26:26 the _q;27:27 amount _n_of;28:28 of _p;29:29 the _q;30:31 _n_of price change;32:32 in _p;33:33 the _q;34:34 individual _a_1;35:35 stock _n_1;36:36 or _c;37:37 index _n_1;38:38 of _p;39:39 stocks _n_1;40:40 on _p_state;0:0 Options _n_1', '0:0 Thus _a_1;1:2 _n_1 options traders;3:3 Friday dofw;5:5 stuck _a_with;7:7 losses _n_of;9:11 _a_1 also several;12:12 times _n_of;13:13 larger _a_1;15:15 those _q_dem;16:16 suffered _v_1;17:17 by _p;19:20 _n_1 stock traders;21:21 in _p;22:23 named New York', '0:1 named Jeffrey Miller;2:2 of _p;3:4 named Miller Tabak;5:5 Hirsch named;6:7 & Co _n_1;8:8 said _v_to;10:10 given _v_1;11:11 the _q;12:12 high _a_1;13:13 degree _n_of;15:15 leverage _n_1;16:16 in _p;17:17 the _q;18:19 _n_1 options market;22:22 "very _x_deg;23:23 easy _a_for;25:25 these _q_dem;26:26 guys _n_1;28:28 get _v_state;0:0 Jeffrey named', '0:0 That _q_dem;1:1 may _v_modal;2:2 just _a_1;3:3 be _v_id;4:4 the _q;5:5 nature _n_1;6:6 of _p;7:7 these _q_dem;8:8 highly _a_1;9:9 leveraged _v_1;10:10 little _a_1;0:0 That _q_dem', '1:1 then _a_1;2:2 had _v_1;3:3 no _q;4:4 choice _n_of;5:5 in _p;7:7 cases _n_of;8:8 but _x;10:10 sell _v_1;11:11 the _q;12:12 contracts _n_1;13:13 at _p;14:14 prevailing _v_1;15:15 prices _n_of;17:17 in _p;18:18 most _q;19:19 cases _n_of;20:20 at _p;21:21 a _q;22:22 substantial _a_1;1:1 then _a_1', '0:0 The _q;1:1 latest _a_for;2:4 _n_of round losses;6:6 likely _a_1;8:8 be _v_id;9:9 a _q;10:10 serious _a_1;11:11 blow _n_1;12:12 to _p;13:13 the _q;14:15 named Chicago Board;16:17 named Options Exchange;20:20 never _a_1;21:21 fully _a_of;22:22 recovered _v_from;24:24 the _q;25:25 aftershock _n_1;26:26 of _p;27:27 Black named;28:28 Monday dofw;30:30 investors _n_1;31:31 fled _v_from;32:32 the _q;33:33 market _n_1;34:35 because of _p;36:36 huge _a_1;0:0 The _q', '0:0 Making _v_cause;1:1 matters _n_of;2:2 worse _a_1;4:4 the _q;5:5 fact _n_1;7:7 late _p;8:8 Friday dofw;9:9 afternoon _n_of;10:10 the _q;11:11 CBOE named;12:12 halted _v_1;13:14 _n_1 stock- index;15:15 options _n_1;16:16 trading _v_1;17:17 in _p;18:18 step _n_1;19:19 with _p;20:20 the _q;21:22 named Chicago Mercantile;23:23 Exchange named;25:25 halt _n_1;26:26 in _p;27:28 _n_1 stock- index;0:0 Making _v_cause', '0:0 But _c;1:1 while _x;2:2 the _q;3:3 Merc named;4:4 reopened _v_1;4:4 reopened _re-_a_again;5:5 a _q;6:7 half hour _n_1;8:8 later _a_for;9:9 the _q;10:10 CBOE named;11:11 remained _v_1;12:12 closed _a_1;13:13 leaving _v_1;15:16 _n_1 options traders;17:17 unable _a_1;19:19 make _v_1;20:20 trades _n_of;22:22 might _v_modal;24:24 reduced _v_1;25:25 the _q;0:0 But _c', '0:0 CBOE named;1:1 Chairman _n_of;2:3 named Alger "Duke";4:4 Chapman named;5:5 said _v_to;7:7 unlike _p;8:8 the _q;9:10 _n_1 futures market;11:11 the _q;12:13 _n_1 options exchange;14:14 has _v_qmodal;16:16 open _v_1;17:17 in _p;18:18 a _q;19:19 rotation _n_of;21:21 allows _v_1;22:22 each _q;23:23 different _a_than-from;24:24 options _n_1;25:25 series _n_of;0:0 CBOE named', "0:1 _n_1 Exchange officials;2:2 reasoned _v_1;5:5 wouldn't _v_modal;8:8 able _a_1;10:10 make _v_1;11:12 such a _q;13:13 rotation _n_of;14:14 with _p;15:15 the _q;16:16 time _n_of;17:17 remaining _v_1;18:18 Friday dofw;19:19 afternoon _n_of;20:20 and _c;21:21 with _x_subord;22:22 the _q;23:24 _n_1 stock- index;25:25 futures _n_1;26:26 on _p;27:27 the _q;28:28 verge _n_1;29:29 of _p;30:30 closing _v_1;31:31 for _p;32:32 a _q;33:33 second _n_1;34:34 and _c;35:35 final _a_1;36:36 time _n_of;37:37 the _q;38:38 CBOE named;39:39 reasoned _v_1;42:42 best _a_at-for-of;43:43 course _n_of;46:46 remain _v_1;0:0 Exchange _n_1", '0:0 The _q;1:1 damage _n_to;4:4 bad _a_at;5:5 at _p;6:6 Fossett named;7:7 Corp _n_1;8:8 an _q;9:9 options _n_1;10:10 trading _v_1;11:11 firm _n_1;12:12 here _a_1;16:16 forced _v_1;18:18 transfer _v_from-to;20:20 accounts _n_of;22:23 named First Options;24:24 of _p;25:25 Chicago named;26:26 a _q;27:27 unit _n_of;29:30 named Continental Bank;31:31 Corp _n_1;32:32 as _p;33:33 a _q;34:34 result _n_of;36:36 options _n_1;37:37 trading _v_1;0:0 The _q', '0:0 Fosset named;2:2 far _a_1;3:3 is _v_id;4:4 the _q;5:5 only _a_1;6:6 member _n_of;8:8 a _q;9:9 financial _a_1;10:10 exchange _n_1;13:13 forced _v_1;16:16 taken _v_over;18:18 by _p;19:19 another _q;20:20 firm _n_1;21:21 as _p;22:22 a _q;23:23 result _n_of;25:25 Friday dofw;0:0 Fosset named', '0:0 Fossett named;1:1 still _a_1;2:2 had _v_1;3:4 card several million;5:5 dollars _n_1;6:6 in _p;7:7 capital _n_1;8:8 left _v_1;9:9 after _p;10:10 Friday dofw;12:12 close _n_of;14:14 trading _v_1;15:16 but not _c;17:17 enough _q;19:19 regulators _n_1;20:20 worried _a_about;22:22 another _q;23:23 potential _a_1;24:24 market _n_1;25:25 plunge _n_of;26:26 yesterday _a_1;27:27 would _v_modal;28:30 _v_1 let reopen;30:30 reopen _re-_a_again;31:31 for _p;32:32 trading _v_1;33:34 _n_1 options exchange;35:35 officials _n_1;0:0 Fossett named', '0:0 "Steve named;1:1 and _c;3:3 firm _n_1;5:6 _a_1 still worth;7:7 a _q;8:8 lot _n_of;10:11 _n_1 money," Mr;12:12 Rawls named;0:0 "Steve named', '0:0 "A _q;1:1 package _n_of;3:3 credit _n_1;4:4 support _n_of;6:6 put _v_together;9:9 including _v_1;10:10 the _q;11:11 assets _n_1;12:12 of _p;13:13 Steve named;14:14 and _c;0:0 "A _q', '0:0 The _q;1:1 bailout _n_unknown;3:3 cobbled _a_1;4:5 _p together over;6:6 the _q;7:7 weekend _n_of;8:8 with _p;9:9 officials _n_1;10:10 from _p;11:11 the _q;12:13 named Federal Reserve;14:15 named Board Securities;16:16 and _c;17:18 named Exchange Commission;19:19 Comptroller named;20:20 of _p;21:21 the _q;22:22 Currency named;23:23 and _c;24:24 Treasury named;25:27 as well as _c;28:28 the _q;29:30 _n_1 options exchanges', '2:2 great _a_for;4:4 have _v_1;5:5 the _q;6:6 luxury _n_1;7:7 of _p;8:8 time," _n_of;9:9 Mr _n_1;10:10 Rawls named;2:2 great _a_for', '0:0 At _p;1:1 one card;2:2 point _n_of;3:3 an _q;4:5 _n_1 options industry;6:6 official _n_1;7:7 had _v_qmodal;9:9 talk _v_into;10:10 the _q;11:12 named Federal Reserve;13:13 Bank named;14:14 of _p;15:15 Chicago named;17:17 night _n_of;18:18 watchman _n_1;20:20 giving _v_1;22:22 the _q;23:23 home _n_of-n;24:24 phone _n_1;25:25 number _n_of;27:28 named Silas Keene;29:29 Chicago named;30:30 Fed named_n;0:0 At _p', '0:1 named First Options;3:3 have _v_qmodal;5:5 put _v_1;6:6 any _q;7:7 money _n_1;8:8 into _p;9:9 the _q;0:0 First named', '0:0 Yesterday _a_1;2:2 rally _n_1;3:3 in _p;4:4 the _q;5:6 _n_1 stock futures;7:7 and _c;8:9 _n_1 options markets;10:10 led _v_1;11:11 CBOE named;12:12 and _c;13:13 Amex named;14:14 officials _n_1;16:16 conclude _v_1;18:18 the _q;19:19 $ _n_1;20:21 card 50 million;22:22 in _p;23:23 guarantees _n_of;24:24 almost _x_deg;25:25 certainly _a_1;27:30 _v_1 need tapped;32:33 named First Options', '0:0 The _q;1:1 Fossett named;2:2 firm _n_1;3:3 had _v_1;4:4 some _q;5:5 losses _n_of;6:6 and _c;7:7 liquidity _n_1;8:8 problems _n_of;9:9 during _p;10:10 the _q;11:11 October mofy;12:12 1987 yofc;13:13 crash _n_1;15:15 well _a_1;16:16 Mr _n_1;17:17 Rawls named;0:0 The _q', '0:0 A _q;1:1 federal _a_1;2:2 official _n_1;3:3 said _v_to;5:6 named Continental Bank;7:7 worked _v_1;8:8 with _p;9:9 securities _n_1;10:10 and _c;11:12 _n_1 banking regulators;13:13 over _p;14:14 the _q;15:15 weekend _n_of;16:16 to _x;17:17 fashion _v_into;18:18 the _q;19:19 Fossett named;20:20 bailout _n_unknown;21:21 but _c;23:23 conditions _n_1;25:25 dictated _v_to;27:27 those _q_dem;0:0 A _q', '1:1 was _v_id;3:4 _n_1 business decision,";5:5 the _q;6:6 official _n_1;1:1 was _v_id', '0:0 Officials _n_1;1:1 at _p;2:3 named Options Clearing;4:4 Corp _n_1;6:6 processes _v_1;7:7 all _q;8:8 options _n_1;9:9 trades _n_of;10:10 for _p;11:11 U.S named_n;12:12 exchanges _n_1;13:13 said _v_to;15:15 the _q;16:16 $ _n_1;17:18 card 50 million;19:19 guarantee _n_of;21:21 unprecedented _a_1;22:22 but _c;24:24 necessary _a_for;25:25 to _x;26:26 help _v_to;27:27 insure _v_for;28:28 the _q;29:29 integrity _n_1;30:30 of _p;31:31 the _q;32:33 _n_1 options markets', '1:1 was _v_id;2:2 an _q;3:3 extraordinary _a_to;4:4 situation _n_1;6:6 needed _v_1;7:7 extraordinary _a_to;8:8 steps," _n_1;9:9 said _v_to;10:11 named Paul Stevens;12:12 OCC named;13:13 president _n_of;14:14 and _c;15:15 chief _n_1;16:16 operating _v_1;1:1 was _v_id', '0:0 Mr _n_1;1:1 Stevens named;2:4 _v_1 declined give;5:5 the _q;6:6 specific _a_to;7:7 contributions _n_of;8:8 to _p;9:9 the _q;10:10 $ _n_1;11:12 card 50 million;13:13 guarantee _n_of;14:14 from _p;15:15 each _q;0:0 Mr _n_1', '0:0 But _c;1:1 CBOE named;2:2 and _c;3:3 Amex named;4:4 officials _n_1;5:5 said _v_to;7:8 named Options Clearing;9:9 Corp _n_1;10:10 contributed _v_to;11:11 $ _n_1;12:13 card 20 million;15:15 the _q;16:16 guarantee _n_of;17:17 the _q;18:18 CBOE named;19:19 put _v_up;21:21 $ _n_1;22:23 card 8 million;24:24 the _q;25:25 Amex named;26:26 added _v_to;27:27 $ _n_1;28:29 card 4 million;30:30 and _c;31:31 $ _n_1;32:33 card 18 million;34:34 came _v_1;35:35 from _p_dir;36:36 Mr _n_1;37:37 Fossett named;39:39 own _a_1;0:0 But _c', "0:0 Mr _n_1;1:1 Fossett named;2:2 couldn't _v_modal;4:4 reached _v_1;5:5 to _p;0:0 Mr _n_1", '0:1 named Debora Foster;2:2 takes _v_x-off;5:5 necklace _n_1;6:6 settles _v_1;8:8 on _p;9:9 a _q;10:10 padded _v_1;11:11 chair _n_1;12:12 and _c;13:13 gently _a_1;14:14 leans _v_towards;0:0 Debora named', '0:0 With _x_subord;1:1 a _q;2:3 _n_1 jazz- piano;4:4 tape _n_1;5:5 playing _v_1;6:6 softly _a_1;7:7 in _p;8:8 the _q;9:9 background _n_1;10:10 the _q;11:11 soothing _v_1;12:12 hands _n_1;13:13 of _p;14:15 named Sabina Vidunas;16:16 begin _v_1;18:18 work _v_on;20:20 Ms _n_1;21:21 Foster named;23:23 neck _n_1;24:24 and _c;0:0 With _x_subord', '2:2 like _p;3:3 an _q;4:4 oasis _n_1;5:5 in _p;6:6 this _q_dem;7:8 _n_1 room," Ms;9:9 Foster named;2:2 like _p', '0:0 The _q;1:1 room _n_1;2:2 in _p;3:3 question _n_about;4:4 is _v_id;5:5 the _q;6:6 directors _n_of;8:8 lounge _n_1;9:9 of _p;10:11 named H.J Heinz;12:12 Co _n_1;13:13 60 card;14:14 floors _n_1;15:15 above _p;16:16 the _q;17:17 bustle _n_1;18:18 of _p;0:0 The _q', '0:0 There _a_1;1:1 amid _p;2:2 oil _n_1;3:3 paintings _n_of;4:4 and _c;5:6 _n_1 marble tables;7:7 massages _n_of;9:9 administered _v_to;10:10 every _q;0:0 There _a_1', '0:0 "On _p_temp;1:1 days _n_of;5:5 really _x_deg;6:6 busy," _a_1;7:7 says _v_to;8:8 Ms _n_1;9:9 Foster named;11:11 works _v_1;12:12 in _p;13:13 public _a_1;14:14 relations _n_of;15:15 for _p;16:16 the _q;17:17 company _n_of;19:19 seems _v_to;20:20 decadent _a_unknown;21:21 to _x;22:22 take _v_x-off;23:23 time _n_of;25:25 for _p;26:26 a _q;0:0 "On _p_temp', '0:0 In _p_state;1:1 some _q;2:2 companies _n_of;3:3 middle _a_1;4:4 managers _n_of;5:5 sneak _v_1;6:6 massage _n_of;7:7 therapists _n_1;8:8 into _p;9:9 the _q;10:10 office _n_of;11:11 fearful _a_1;14:14 level _n_1;14:14 level _upper_a_1;15:15 executives _n_1;0:0 In _p_state', '0:0 Ms _n_1;1:1 Foster named;3:3 indulgence _n_unknown;4:4 is _v_id;5:5 nothing _q;6:6 like _p;7:7 the _q;8:8 oily _a_1;9:9 hour- _n_1;10:10 long _a_1;11:11 rubfests _n_unknown;12:12 enjoyed _v_1;14:14 spa _n_1;0:0 Ms _n_1', '0:0 The _q;1:1 massages _n_of;2:2 last _v_1;3:3 15 card;4:4 minutes _n_1;5:5 and _c;6:6 typically _a_of;7:7 cost _v_1;8:8 about _x_deg;9:9 $ _n_1;0:0 The _q', '0:0 Some _q;1:1 companies _n_of;2:2 including _v_1;3:3 Heinz named;4:4 even _a_1;5:5 pay _v_for;6:8 _q part the;0:0 Some _q', '0:0 Ms _n_1;1:1 Vidunas named;4:4 seeing _v_1;5:5 some _q;6:6 15 card;7:7 clients _n_1;8:8 a _p_per;9:9 visit _n_to;10:10 since _x_subord;11:11 the _q;12:12 program _n_of;14:14 started _v_cause;15:15 at _p;16:16 Heinz named;17:17 last _a_1;0:0 Ms _n_1', "0:1 named Anthony J.F;2:2 O'Reilly named;3:3 the _q;4:6 _n_of company chairman;7:7 swears _v_1;8:8 by _p;10:10 firm _a_1;11:11 touch _n_1;12:12 saying _v_to;13:13 regular _a_1;14:14 massages _n_of;15:15 are _v_id;16:16 a _q;17:17 balm _n_1;18:18 for _p;20:20 old _a_1;21:22 _n_1 football injuries", '0:0 Massage _n_of;1:1 advocates _n_1;2:2 say _v_to;4:4 kneading _v_unknown;5:5 the _q;6:6 head _n_of;7:8 _n_1 shoulders neck;9:9 and _c;10:10 back _n_1;11:11 can _v_modal;12:12 go _v_1;13:13 a _q;14:14 long _a_1;15:15 way _n_of;16:16 toward _p;17:17 easing _v_1;18:18 tension _n_1;19:19 and _c;20:20 improving _v_cause;0:0 Massage _n_of', '1:1 also _a_1;2:4 _v_1 insist touching;6:6 a _q;7:7 basic _a_1;8:8 need _n_of;10:10 powerful _a_1;12:12 the _q;13:13 need _n_of;14:14 for _p;15:15 food _n_1;16:16 or _c;17:17 sleep _n_1;18:18 and _c;20:20 the _q;21:21 office _n_of;22:22 is _v_id;24:24 good _a_at-for-of;25:25 a _q;26:26 place _n_of;28:28 any _q;29:29 to _x;1:1 also _a_1', '0:0 "The _q;1:1 blood _n_1;2:2 flows _v_1;3:3 to _p;5:5 head _n_of;7:7 feel _v_seem-about;8:8 lightheaded _a_unknown;9:9 and _c;12:12 feel _v_1;13:13 tension _n_1;14:14 around _p;15:15 the _q;16:16 head _n_of;17:17 or _c;18:18 neck," _n_1;19:19 says _v_to;20:21 named Minnie Morey;22:22 an _q;23:24 _n_of operations supervisor;25:25 at _p;26:26 the _q;27:28 named Social Security;29:29 office _n_of;30:30 in _p;31:32 named Grand Rapids;33:33 Mich named;35:35 massages _n_of;36:36 began _v_1;37:37 last _a_1;0:0 "The _q', '0:0 "When _x_subord;2:2 leave _v_1;3:3 the _q;4:4 room _n_1;5:5 after _p;7:8 _n_of massage people;9:9 say _v_to;11:15 _v_1 look glowing."', '0:0 Adds _v_to;1:2 named Candice Ohlman;3:3 the _q;5:5 year- _n_1;5:5 year- card;6:6 old _a_1;7:7 masseuse _n_1;9:9 plies _v_1;11:11 trade _n_of;12:12 in _p;13:13 the _q;14:15 named Grand Rapids;16:16 office _n_of;18:18 fall _v_1;19:19 in _p;20:20 love _n_of-for;21:21 with _p;0:0 Adds _v_to', '0:0 Not _x_deg;4:4 at _p;5:5 ease _n_1;6:6 with _p;7:8 _n_of office massage', '0:0 Three card;1:1 years _n_1;2:2 ago _p;3:3 the _q;4:5 named Internal Revenue;6:6 Service named;8:8 office _n_of;9:9 in _p;10:11 named San Jose;12:12 Calif named;13:13 opened _v_cause;15:15 doors _n_1;16:16 to _p;17:18 on- site _a_1;0:0 Three card', '0:0 And _c;1:1 even _x_deg;2:2 though _x;3:3 employees _n_of;4:4 paid _v_for;5:5 the _q;6:6 bill _n_of;7:7 taxpayers _n_unknown;0:0 And _c', '0:0 Last _a_1;1:1 month _n_1;2:2 the _q;3:3 complaints _n_1;4:4 intensified _v_1;5:5 and _c;6:6 the _q;7:7 massages _n_of;0:0 Last _a_1', '0:0 "Now _a_1;3:3 looking _v_for;5:5 a _q;6:6 room _n_1;7:7 with _p;8:8 thicker _a_1;9:9 walls," _n_of;10:10 Ms _n_1;11:11 Banks named;0:0 "Now _a_1', '0:0 Massage _n_of;1:1 also _a_1;2:2 has _v_1;3:3 an _q;4:5 _n_of image problem;0:0 Massage _n_of', '0:0 Last _a_1;1:1 year _n_1;2:2 the _q;3:3 research _n_1;4:4 and _c;5:6 _n_of development division;8:8 Weyerhaeuser named;9:9 Co _n_1;10:10 the _q;11:11 large _a_1;12:13 _n_1 wood- products;14:14 concern _n_1;15:15 invited _v_to;16:16 a _q;17:17 masseuse _n_1;20:21 named Tacoma Wash;0:0 Last _a_1', '0:1 named Phil Harms;2:2 a _q;3:4 _n_1 software engineer;5:5 was _v_id;6:6 an _q;7:7 eager _a_1;0:0 Phil named', '1:1 build _v_up;3:3 a _q;4:4 lot _n_of;6:6 tension _n_1;7:7 working _v_1;8:8 at _p;9:9 a _q;10:10 terminal _n_1;11:11 all _a_1;12:12 day," _n_of;1:1 build _v_up', '0:0 But _c;1:1 after _p;2:2 about _x_deg;3:3 eight card;4:4 months _n_1;5:5 the _q;6:6 vice _n_1;7:7 president _n_of;9:9 the _q;10:10 division _n_of;11:12 named Ed Soule;13:13 learned _v_about;15:15 the _q;16:16 sessions _n_1;17:17 and _c;18:18 brought _v_1;20:20 to _p;21:21 a _q;0:0 But _c', '0:0 Mr _n_1;1:1 Soule named;2:2 says _v_to;4:4 only _a_1;5:5 beef _n_1;6:6 was _v_nv;8:8 the _q;9:9 massages _n_of;12:12 given _v_1;13:13 in _p;14:14 a _q;15:16 _n_of company conference;17:17 room; _n_1;18:18 the _q;19:19 department _n_1;21:21 supervised _v_1;22:23 _n_1 health facility;24:24 would _v_modal;0:0 Mr _n_1', '0:0 "In _p_state;2:3 _n_of view {massages};4:4 should _v_modal;6:6 managed _v_1;7:7 with _p;8:8 an _q;9:9 appropriate _a_to;10:10 mixture _n_of;12:12 males _n_1;13:13 and _c;14:14 females _n_1;15:15 around," _p;0:0 "In _p_state', '0:0 Given _v_1;1:1 such _q;2:2 attitudes _n_1;3:3 some _q;4:4 corporate _a_1;5:5 masseurs _n_1;6:6 prefer _v_to;8:8 go _v_about;11:11 business _n_1;0:0 Given _v_1', '1:1 visits _v_1;2:2 the _q;3:3 same _a_as;4:4 department _n_1;5:5 every _x_deg;6:6 two card;7:7 or _c;8:8 three card;1:1 visits _v_1', '1:1 massage _n_of;2:2 chair _n_1;4:4 kept _v_prd;5:5 in _p;6:6 a _q;7:7 closet _n_1;8:8 and _c;9:9 a _q;10:10 secretary _n_of;11:11 escorts _v_1;13:13 past _p;1:1 massage _n_of', '0:0 "This _q_dem;2:2 common _a_for;3:3 with _p;4:4 a _q;5:5 lot _n_of;7:7 large _a_1;8:8 companies," _n_of;9:9 says _v_to;10:10 Mr _n_1;11:11 Borner named;13:13 worked _v_for;15:16 named American Telephone;17:17 & _c;18:18 Telegraph named;19:19 Co _n_1;20:20 for _p;21:21 23 card;22:22 years _n_1;23:23 before _p;24:24 choosing _v_from;26:26 current _a_1;0:0 "This _q_dem', '1:1 vision _n_1;4:4 change _v_cause;5:5 human _a_1;6:6 consciousness _n_1;7:7 towards _p;1:1 vision _n_1', '1:1 attitude _n_1;3:3 : _p_namely;6:6 come _v_1;7:8 out of _p_dir;9:9 the _q;1:1 attitude _n_1', '0:0 Occasionally _a_1;1:1 all _q;4:4 needed _v_1;5:5 is _v_id;6:7 a little _q;0:0 Occasionally _a_1', '0:1 named Elisa Byler;2:2 a _q;3:3 St _n_1;4:4 Louis named;5:5 masseuse _n_1;6:6 won _v_over;8:8 officials _n_1;9:9 at _p;10:11 named Emerson Electric;12:12 Co _n_1;13:13 a _q;14:14 maker _n_of;16:16 electrical _a_1;17:17 and _c;18:18 electronic _a_1;19:19 equipment _n_1;20:20 by _p_means;21:21 providing _v_1;22:22 documents _n_1;23:23 and _c;24:24 other _a_1;25:25 articles _n_of;26:26 trumpeting _v_cause;27:27 the _q;28:28 therapeutic _a_1;29:31 _n_of benefits massage', '1:1 notes _v_to;4:4 also _a_1;5:5 stresses _v_to;6:6 professionalism _n_1;7:7 during _p;9:9 weekly _a_1;1:1 notes _v_to', '1:1 pull _v_back;3:3 hair _n_1;5:5 wear _v_1;6:7 a little _q;8:8 makeup _n_1;9:9 and _c;10:10 look _v_seem-to;11:11 corporate," _a_1;12:12 says _v_to;13:13 Ms _n_1;14:14 Byler named;18:18 visiting _v_1;19:19 Emerson named;20:20 since _p;1:1 pull _v_back', '0:0 "If _x_then;2:2 go _v_1;3:3 in _p;4:4 there _a_1;5:5 as _x_subord;7:7 normally _a_1;8:8 dress _v_in;10:10 \'d _v_modal;11:11 ask _v_1;13:13 is _v_id;14:14 this _q_dem;15:15 hippie _n_unknown;0:0 "If _x_then', '0:0 The _q;1:1 self- _n_1;2:2 proclaimed _v_as;3:3 father _n_of;5:6 on- site _a_1;7:7 massage _n_of;8:8 is _v_id;9:10 named David Palmer;11:11 a _q;13:13 year- _n_1;13:13 year- card;14:14 old _a_1;15:16 named San Francisco;17:19 _n_1 masseur mission;22:22 save _v_1;23:23 the _q;24:24 touch- _n_1;25:25 starved _v_cause;0:0 The _q', '0:0 "The _q;1:1 culture _n_1;4:4 ready _a_1;6:6 take _v_x-off;9:9 clothes _n_1;10:10 lie _v_2;11:11 down _p;12:12 and _c;14:14 touched _v_1;15:15 for _p;16:16 an card;17:17 hour _n_1;18:18 for _p;19:19 $ _n_1;20:20 45," card;0:0 "The _q', '0:0 "The _q;1:1 idea _n_of;3:3 to _x;4:4 keep _v_unspec;5:5 the _q;6:6 clothes _n_1;7:7 on _p_dir;8:8 and _c;10:10 keep _v_prd;11:11 people _n_of;0:0 "The _q', '0:0 The _q;1:1 chair _n_1;2:2 is _v_id;3:3 a _q;4:4 way _n_of;6:6 package _v_1;0:0 The _q', '0:0 Sitting _v_1;1:1 in _p;2:2 one card;4:4 Mr _n_1;5:5 Palmer named;7:7 chairs _n_1;9:9 cost _v_1;10:10 $ _n_1;11:11 425 card;12:12 and _c;14:14 since _p;16:16 copied _v_1;18:18 others _n_1;20:21 a bit _a_1;22:22 like _p;23:23 straddling _v_1;24:24 a _q;0:0 Sitting _v_1', '0:0 Customers _n_of;1:1 lean _v_towards;2:2 forward _p_dir;3:3 rest _v_1;5:5 knees _n_1;6:6 on _p;7:7 side _n_1;8:8 supports _n_of;9:9 and _c;10:10 bury _v_1;12:12 face _n_1;13:13 in _p;14:14 padding _v_1;15:15 on _p;16:16 the _q;17:17 back _n_1;18:18 of _p;19:19 the _q;0:0 Customers _n_of', '0:1 named Scot MacInnis;2:2 a _q;3:3 masseur _n_1;4:4 in _p;5:6 named Boulder Colo;7:7 had _v_1;8:8 a _q;9:9 scary _a_for;10:10 experience _n_with;11:11 while _x;12:12 massaging _v_1;13:13 a _q;14:14 man _n_1;15:15 in _p;16:16 a _q;18:18 foods _n_1;18:18 foods _natural_a_for;19:19 supermarket _n_1;20:20 as _p;21:23 _q part a;24:24 store _n_1;0:0 Scot named', '0:0 Three card;1:1 minutes _n_1;2:2 into _p;3:3 the _q;4:4 massage _n_of;5:5 the _q;6:6 man _n_1;7:7 curled _v_1;8:8 up _p_dir;9:10 _v_1 began shaking;11:11 and _c;12:12 turned _v_1;0:0 Three card', '0:0 Paramedics _n_unknown;0:0 Paramedics _n_unknown', '0:0 A _q;1:1 week _n_1;2:2 later _a_for;3:3 the _q;4:4 man _n_1;5:5 told _v_1;6:6 Mr _n_1;7:7 MacInnis named;10:10 suffered _v_1;11:11 a _q;12:12 mild _a_1;13:14 _n_1 heart attack;15:15 unrelated _a_to;17:17 the _q;0:0 A _q', '1:1 was _v_id;2:2 a _q;3:3 powerful _a_1;4:4 point _n_of;5:5 in _p;7:7 career," _n_1;8:8 says _v_to;9:9 the _q;13:13 Mr _n_1;14:14 MacInnis named;17:17 since _p;18:18 taken _v_out;20:20 a _q;21:21 $ _n_1;22:23 card 1 million;24:24 liability _n_1;25:25 policy _n_of-on;26:26 for _p;1:1 was _v_id', '0:0 "But _c;2:2 pulled _v_through;4:4 and _c;5:5 after _x_h;6:6 the _q;7:7 ambulance _n_1;8:8 left _v_1;10:10 were _v_there;11:11 still _a_1;12:12 six card;13:13 people _n_of;14:14 in _p;15:15 line _n_of;16:16 waiting _v_for;18:18 a _q;0:0 "But _c', '0:0 The _q;1:1 next _a_1;2:2 woman _n_1;4:4 older _a_1;5:5 and _c;8:8 afraid _a_of;0:0 The _q', '0:0 But _c;3:3 like _p;4:4 falling _v_1;5:5 off _p_dir;6:6 a _q;7:7 horse _n_1;8:8 and _c;9:9 getting _v_state;10:10 back _p;0:0 But _c', '0:0 Despite _p;1:1 the _q;2:2 number _n_of;4:4 fans _n_1;6:7 _n_of office massage;9:9 won _v_1;10:10 some _q;11:11 purists _n_unknown;12:12 look _v_1;13:14 _p down on;16:16 arguing _v_with;18:18 naked _a_1;20:20 body _n_1;20:20 body _full_a_of;21:21 rubs _n_1;22:22 are _v_id;23:23 the _q;24:24 only _a_1;25:25 way _n_of;0:0 Despite _p', '0:1 named Linda Aldridge;3:3 does _v_1;5:5 body _n_1;5:5 body _full_a_of;6:6 work _n_1;7:7 in _p;8:8 Pittsburgh named;9:9 says _v_to;11:11 while _x;12:13 on- site _a_1;14:14 massage _n_of;16:16 better _a_at-for-of;18:18 nothing _q;19:19 tired _a_of;20:20 workers _n_1;21:21 should _v_modal;22:22 realize _v_1;24:24 is _v_id;25:25 only _a_1;26:26 the _q;27:27 tip _n_1;28:28 of _p;29:29 the _q;0:0 Linda named', '0:0 "Whole _a_1;1:1 areas _n_of;4:4 bodies _n_1;6:6 neglected," _v_1;8:9 _v_to says adding;11:11 clothes _n_1;12:12 ruin _v_1;13:13 the _q;0:0 "Whole _a_1', "1:1 's _v_there;2:2 nothing _q;3:3 like _p;4:4 skin _n_1;5:5 to _p;1:1 's _v_there", '0:0 In _p_state;3:3 believed _v_1;5:5 be _v_id;6:6 the _q;7:7 first ord;8:8 cancellation _n_of;10:10 a _q;11:11 loan _n_1;12:12 to _p;13:13 China named;14:14 since _p;15:15 the _q;16:16 June mofy;17:17 4 dofm;18:18 killings _n_1;19:19 in _p;20:20 Beijing named;21:21 an _q;22:22 international _a_1;23:23 bank _n_of;24:24 syndicate _n_1;26:26 terminated _v_cause;27:27 a _q;28:28 $ _n_1;29:30 card 55 million;31:31 credit _n_1;32:32 for _p;33:33 a _q;34:34 Shanghai named;35:36 _n_1 property project', '0:0 The _q;1:1 syndicate _n_1;2:2 led _v_1;4:5 named Schroders Asia;6:6 Ltd _n_1;7:7 agreed _v_1;8:8 last _a_1;9:9 November mofy;11:11 provide _v_1;12:12 the _q;13:13 loan _n_1;15:16 named Asia Development;17:17 Corp _n_1;18:18 a _q;19:19 U.S named_n;20:20 property _n_1;0:0 The _q', '0:0 But _c;1:1 several _a_1;2:2 weeks _n_1;3:4 _p ago in;5:5 the _q;6:6 wake _n_1;7:7 of _p;8:8 the _q;9:9 Beijing named;10:10 killings _n_1;11:11 the _q;12:12 loan _n_1;14:14 canceled _v_1;15:16 according to _p;17:17 bankers _n_1;18:18 and _c;19:19 executives _n_1;20:21 close to _p;22:22 the _q;0:0 But _c', '0:1 named Asia Development;2:2 and _c;3:3 Schroders named;4:4 declined _v_1;6:6 comment _v_on;8:8 the _q;0:0 Asia named', '0:0 Lenders _n_unknown;1:1 had _v_1;2:2 doubts _n_1;3:3 about _p;4:4 the _q;5:5 project _n_1;6:6 even _x_deg;7:7 before _p;8:8 June mofy;9:9 4 dofm;10:10 but _c;11:11 the _q;12:12 harsh _a_1;13:13 crackdown _n_unknown;15:15 caused _v_1;17:17 businesses _n_1;19:19 reassess _v_1;19:19 reassess _re-_a_again;21:21 China named;22:22 transactions _n_1;23:23 "gave _v_1;24:24 the _q;25:25 banks _n_of;26:26 the _q;27:27 out _n_1;29:29 wanted," _v_1;30:30 says _v_to;31:31 an _q;32:32 official _n_1;33:33 close _a_to;35:35 the _q;36:36 Shanghai named;0:0 Lenders _n_unknown', '0:0 The _q;1:1 decision _n_1;3:3 cancel _v_1;4:4 the _q;5:5 loan _n_1;6:6 exemplifies _v_1;7:7 the _q;8:8 tough _a_for;9:10 _n_1 attitude bankers;12:12 taken _v_1;13:13 toward _p;14:14 China named;15:15 since _p;16:16 June mofy;0:0 The _q', '0:0 While _x;1:1 some _q;2:2 commercial _a_1;3:3 lending _n_1;5:5 resumed _v_1;6:6 international _a_1;7:7 lenders _n_unknown;8:8 remain _v_1;9:9 nervous _a_about;11:11 China named;13:13 economic _a_1;14:14 troubles _n_1;15:15 and _c;16:16 foreign _a_1;17:19 _n_1 debt $;20:21 card 40 billion;22:22 at _p;23:23 the _q;24:24 end _n_of;0:0 While _x', '1:1 bankers _n_1;2:2 view _v_as;3:4 _n_1 property- sector;5:5 loans _n_1;7:7 particularly _a_1;1:1 bankers _n_1', '0:0 The _q;1:1 canceled _v_1;2:2 Shanghai named;3:3 loan _n_1;4:4 leaves _v_1;5:6 named Asia Development;7:7 a _q;8:8 small _a_1;9:9 concern _n_1;10:10 saddled _v_with;12:12 a _q;14:14 completed _v_2;14:14 completed _half_a_1;16:16 story _n_1;16:16 story card;17:18 _n_1 apartment building;19:19 and _c;20:20 heavy _a_1;0:0 The _q', '0:0 The _q;1:1 company _n_of;2:2 owes _v_1;3:3 $ _n_1;4:5 card 11 million;6:6 to _p;7:7 the _q;8:9 named Shui On;10:10 Group named;11:11 the _q;12:12 project _n_1;14:15 named Hong Kong;16:16 contractor _n_1;17:17 and _c;18:18 a _q;19:19 significant _a_for;20:20 though _c;21:21 unspecified _a_1;22:22 amount _n_of;23:23 in _p;24:24 legal _a_1;25:25 fees _n_1;26:26 to _p;27:28 named Coudert Brothers;29:29 a _q;30:30 U.S named_n;31:32 _n_1 law firm;33:33 the _q;34:34 sources _n_of;0:0 The _q', '0:0 The _q;1:1 project _n_1;2:2 known _v_as;4:5 named Lotus Mansion;8:8 mired _v_1;9:9 in _p;0:0 The _q', '0:0 When _x_subord;1:1 the _q;2:3 _n_1 loan agreement;5:5 announced _v_to;8:8 hailed _v_as;10:10 one card;12:12 the _q;13:13 first ord;14:14 Western- named;15:15 style _n_of;16:16 financing _v_1;17:17 transactions _n_1;18:18 ever _a_1;19:19 used _v_1;20:20 in _p;0:0 When _x_subord', '0:0 Unlike _p;1:1 most _q;2:2 loans _n_1;3:3 to _p;4:4 China named;6:6 was _v_there;7:7 no _q;8:8 Chinese _a_1;0:0 Unlike _p', '0:0 Instead _p;1:1 the _q;2:2 banks _n_of;3:3 secured _v_1;4:4 a _q;5:5 promise _n_of;6:6 from _p;7:7 state- _n_of;8:8 owned _v_1;9:9 Bank named;10:10 of _p;11:11 Communications named;14:14 would _v_modal;15:15 lend _v_1;16:17 named Asia Development;18:18 the _q;19:19 entire _a_1;20:20 $ _n_1;21:22 card 55 million;23:23 at _p;24:24 maturity _n_1;25:25 to _x;26:26 finance _v_1;27:27 repayment _n_of;29:29 the _q;30:30 original _a_1;0:0 Instead _p', '0:0 The _q;1:1 loan _n_1;5:5 matured _v_1;6:6 in _p;7:7 just _x_deg;8:10 card two three;11:11 years _n_1;13:13 soon _p;15:15 construction _n_of;0:0 The _q', '0:0 But _c;1:1 in _p_state;2:2 a _q;3:3 letter _n_of;4:4 sent _v_1;5:5 in _p_temp;6:6 August mofy;7:7 to _p;8:9 named Asia Development;10:10 Schroders named;11:11 said _v_to;12:12 the _q;13:13 loan _n_1;15:15 terminated _v_cause;16:16 because _x;17:17 the _q;18:18 developer _n_of;20:20 failed _v_1;22:22 deliver _v_to;23:24 _a_1 adequate financial;25:25 data _n_1;26:26 and _c;27:27 pay _v_for;28:28 certain _a_of;29:29 fees _n_1;31:31 the _q;32:33 _n_1 loan- management;34:34 committee _n_of;35:36 on time _a_1;37:38 according to _p;39:39 officials _n_1;40:40 close _a_to;42:42 the _q;0:0 But _c', '0:0 Creditors _n_1;1:1 involved _v_in;3:3 the _q;4:4 project _n_1;5:5 contend _v_1;6:6 however _a_1;8:8 the _q;9:9 termination _n_of;10:10 actually _a_1;11:11 had _v_1;12:12 nothing _q;14:14 do _v_1;15:15 with _p;16:16 these _q_dem;17:17 technical _a_1;0:0 Creditors _n_1', '0:0 The _q;1:1 bank _n_of;2:2 syndicate _n_1;4:4 made _v_up-of;6:8 _a_1 mostly European;9:9 banks _n_of;10:10 but _c;12:12 includes _v_1;13:13 China named;15:15 state- _n_of;16:16 owned _v_1;17:18 named Citic Industrial;0:0 The _q', '0:0 The _q;1:1 11 card;2:2 banks _n_of;3:3 in _p;4:4 the _q;5:5 syndicate _n_1;6:6 sustained _v_1;7:7 no _q;8:8 monetary _a_1;9:9 losses _n_of;10:10 because _x;11:13 _q none the;14:15 _n_1 credit facility;0:0 The _q', '0:1 named K mart;2:2 Corp _n_1;3:5 _v_1 agreed acquire;6:7 named Pace Membership;8:8 Warehouse named;9:9 Inc _n_1;10:10 for _p;11:11 $ _n_1;12:12 23 card;13:13 a _p_per;14:14 share _n_of;15:15 or _c;16:16 $ _n_1;17:18 card 322 million;19:19 in _p;20:20 a _q;21:21 move _n_1;23:23 expand _v_cause;25:25 presence _n_1;26:26 in _p;27:27 the _q;28:28 rapidly _a_1;29:29 growing _v_1;30:30 warehouse- _n_of;31:32 _n_1 club business', '0:0 The _q;1:1 proposed _v_to;2:2 merger _n_1;3:3 comes _v_1;4:4 as _x_subord;5:5 K named;6:6 mart _n_unknown;8:8 profit _n_1;10:10 declining _v_1;11:11 and _c;12:12 sales _n_of;13:13 at _p;15:16 _n_1 core discount;17:17 stores _n_1;19:19 rising _v_1;21:21 slowly _a_1;23:23 at _p;24:24 such _q;25:25 competitors _n_1;26:26 as _p;27:28 named Wal- Mart;29:29 Stores named;0:0 The _q', '0:1 named K mart;2:2 based _v_1;3:3 in _p;4:5 named Troy Mich;6:6 recently _a_1;7:7 said _v_to;8:9 _n_1 net income;10:10 would _v_modal;11:11 fall _v_1;12:12 for _p;13:13 the _q;14:14 third ord;15:15 consecutive _a_1;16:16 quarter _n_temp;17:17 after _p;18:18 a _q;19:19 16 card;20:21 _n_of % drop;22:22 in _p;23:23 the _q;24:24 first ord;25:25 half _n_of;28:29 _a_1 current fiscal;0:0 K named', '0:0 "The _q;1:1 membership _n_1;2:2 warehouse- _n_of;3:4 _n_1 club concept;5:5 has _v_1;6:6 great _a_for;7:7 potential," _n_1;8:8 the _q;9:11 _n_of company chairman;12:13 named Joseph E;14:14 Antonini named;15:15 said _v_to;16:16 in _p;17:17 a _q;0:0 "The _q', '0:0 Shoppers _n_1;4:4 operate _v_1;5:5 small _a_1;6:6 businesses _n_1;7:7 pay _v_for;8:8 annual _a_1;9:10 _n_1 membership fees;12:12 provide _v_1;13:13 an _q;14:15 _n_1 income base;16:16 for _p;17:17 the _q;0:0 Shoppers _n_1', '0:1 named K mart;2:2 tested _v_1;3:3 the _q;4:4 warehouse- _n_of;5:6 _n_1 club sector;7:7 last _a_1;8:8 year _n_1;9:9 with _p;11:11 acquisition _n_of;13:13 a _q;14:14 51 card;15:15 % _n_of;16:16 interest _n_in;18:18 Makro named;0:0 K named', '0:0 But _c;1:1 the _q;2:2 Makro named;3:3 chain _n_of;5:5 operates _v_1;6:6 as _p;7:7 a _q;8:8 joint _a_1;9:9 venture _n_1;10:10 between _p;11:12 named K mart;13:13 and _c;14:15 named SHV Holdings;16:16 N.V named;17:17 of _p;18:18 the _q;19:19 Netherlands named_n;20:20 has _v_1;21:21 only _x_deg;22:22 six card;23:23 stores _n_1;24:24 and _c;25:25 annual _a_1;26:26 sales _n_of;28:28 one card;29:29 analyst _n_1;30:30 estimated _v_at;32:32 about _x_deg;33:33 $ _n_1;34:35 card 300 million', '1:1 year- _n_1;1:1 year- card;2:2 old _a_1;3:3 Pace named;4:4 based _v_1;5:5 in _p;6:7 named Aurora Colo;8:8 operates _v_1;9:9 41 card;10:10 warehouse- _n_of;11:12 _n_1 club stores', '0:0 The _q;1:1 company _n_of;2:2 had _v_1;3:3 losses _n_of;4:4 for _p;5:5 several _a_1;6:6 years _n_1;7:7 before _p;8:8 turning _v_prd;9:9 profitable _a_for;10:10 in _p_temp;11:11 fiscal _a_1;0:0 The _q', '0:0 In _p_temp;1:1 the _q;2:2 year _n_1;3:3 ended _v_cause;4:4 Jan mofy;5:5 31 dofm;6:6 Pace named;7:7 rang _v_up;9:9 profit _n_1;10:10 of _p;11:11 $ _n_1;12:13 card 9.4 million;14:14 or _c;15:15 72 card;16:16 cents _n_1;17:17 a _p_per;18:18 share _n_of;19:19 after _p;20:20 a _q;21:21 tax- _n_1;22:22 loss _n_of;23:24 carry- forward _n_1;25:25 on _p;26:26 sales _n_of;28:28 $ _n_1;29:30 card 1.3 billion;31:31 and _c;32:32 analysts _n_1;33:33 expect _v_1;35:35 results _n_of;37:37 continue _v_2;0:0 In _p_temp', '0:0 "The _q;1:1 company _n_of;2:2 turned _v_1;3:3 the _q;4:4 corner _n_of;5:5 fairly _x;6:6 recently _a_1;7:7 in _p;8:8 profitability," _n_1;9:9 said _v_to;10:11 named Margo McGlade;12:12 of _p;13:13 PaineWebber named;14:14 Inc _n_1;18:18 forecasting _v_1;19:19 a _q;20:20 46 card;21:21 % _n_of;22:22 jump _n_1;23:23 in _p;24:24 Pace named;26:27 _n_1 net income;28:28 from _p;29:29 operations _n_of;30:30 this _q_dem;31:31 year _n_1;32:32 and _c;33:33 another _q;34:34 42 card;35:35 % _n_of;36:36 increase _n_1;37:37 next _a_1;0:0 "The _q', '0:0 "Warehouse _n_of;1:1 productivity _n_1;3:3 really _a_1;4:4 beginning _v_1;0:0 "Warehouse _n_of', '0:0 But _c;1:1 some _q;2:2 analysts _n_1;3:3 contend _v_1;4:5 named K mart;7:7 agreed _v_1;9:9 pay _v_for;0:0 But _c', '0:0 "Even _x_deg;1:1 if _x_then;3:3 look _v_at;6:6 as _p;7:7 a _q;8:9 _n_1 turnaround situation;12:12 expensive," _a_1;13:13 said _v_to;14:15 named Wayne Hood;16:16 of _p;17:18 named Prudential- Bache;19:19 Securities named;0:0 "Even _x_deg', '0:0 "In _p_state;2:2 opinion _n_of;4:4 would _v_modal;5:5 only _a_1;6:6 pay _v_for;7:7 that _q_dem;8:8 kind _n_of-n;10:10 price _n_of;11:11 if _x_then;14:14 getting _v_1;15:15 a _q;16:16 premier _a_1;17:17 player _n_of;18:18 in _p;19:19 the _q;0:0 "In _p_state', '0:0 Ms _n_1;1:1 McGlade named;2:2 of _p;3:3 PaineWebber named;4:4 raised _v_cause;5:5 a _q;7:7 fundamental _a_1;8:8 question _n_about;10:10 the _q;0:0 Ms _n_1', '0:0 "If _x_then;1:1 K named;2:2 mart _n_unknown;3:3 can\'t _v_modal;4:4 get _v_state;6:6 act _n_of;7:8 _p together in;9:9 discounting _v_1;10:10 why _p;13:13 spending _v_1;14:14 time _n_of;15:15 worrying _v_about;17:17 other _a_1;18:18 growing _v_1;0:0 "If _x_then', '0:0 At _p;1:1 that _q_dem;2:2 point _n_of;3:3 perhaps _a_1;4:4 diversification _n_1;5:5 would _v_modal;0:0 At _p', '0:0 But _c;1:1 K named;2:2 mart _n_unknown;4:4 Mr _n_1;5:5 Antonini named;7:7 intent _a_on;9:9 pushing _v_1;10:10 the _q;11:11 company _n_of;12:12 into _p;13:13 new _a_1;14:15 _n_1 retail businesses', '0:1 For instance _a_1;2:3 named K mart;5:5 opening _v_cause;6:6 big _a_1;7:7 food _n_1;8:8 and _c;9:9 general _a_1;10:11 _n_1 merchandise stores;12:12 called _v_name;13:13 hypermarkets _n_unknown;14:14 and _c;15:15 warehouse- _n_of;16:16 type _n_of-n;17:17 stores _n_1;18:18 specializing _v_in;20:20 office _n_of;21:21 products _n_1;22:22 and _c;23:24 _n_1 sporting goods', '1:1 also _a_1;2:2 operates _v_1;3:4 named Waldenbooks Pay;5:6 named Less Drug;7:7 Stores named;8:8 and _c;9:10 named Builders Square;11:11 home _n_of-n;12:13 _n_1 improvement stores', '0:0 In _p_state;1:1 composite _n_1;2:2 trading _v_1;3:3 on _p;4:4 the _q;5:6 named New York;7:8 _n_1 Stock Exchange;9:10 named K mart;11:11 closed _v_1;12:12 yesterday _a_1;13:13 at _p;14:14 $ _n_1;15:15 36 card;16:16 a _p_per;17:17 share _n_of;18:18 up _p;19:19 12.5 card;0:0 In _p_state', '0:0 Pace named;1:1 rose _v_1;2:2 $ _n_1;3:3 2.625 card;4:4 to _x;5:5 close _v_1;6:6 at _p;7:7 $ _n_1;8:8 22.125 card;9:9 a _p_per;10:10 share _n_of;11:11 in _p;12:13 _a_1 national over- the- counter;0:0 Pace named', '0:0 A _q;1:1 K named;2:2 mart _n_unknown;3:3 spokesman _n_1;4:4 said _v_to;5:5 the _q;6:6 acquisition _n_of;7:7 would _v_modal;9:9 financed _v_1;10:10 with _p;12:12 term _n_of;12:12 term _short_a_of;0:0 A _q', '0:0 Under _p_state;1:1 terms _n_of;3:3 the _q;4:4 agreement _n_1;5:5 a _q;6:6 K named;7:7 mart _n_unknown;8:8 subsidiary _n_1;10:10 soon _a_1;11:11 make _v_1;12:12 a _q;13:14 _n_1 tender offer;15:15 for _p;16:16 Pace named;0:0 Under _p_state', '0:1 named G William;2:2 Ryan named;3:3 president _n_of;5:6 named Post- Newsweek;7:7 Stations named;9:9 named _v_1;10:11 _n_1 chief executive;12:12 officer _n_1;13:13 of _p;14:14 the _q;15:15 unit _n_of;17:17 this _q_dem;18:18 media _n_1;19:19 company _n_of;20:20 effective _a_1;21:21 Jan mofy;0:0 G named', '2:2 succeed _v_1;3:4 named Joel Chaseman;7:7 remain _v_1;8:8 a _q;9:9 vice _n_1;10:10 president _n_of;12:12 the _q;13:13 company _n_of;14:14 and _c;15:15 continue _v_2;17:17 represent _v_1;18:19 named Post- Newsweek;20:20 stations _n_1;21:21 in _p;22:22 several _a_1;23:24 _n_1 industry organizations;25:25 the _q;26:26 company _n_of;2:2 succeed _v_1', '0:0 literally _a_1', '0:0 Traders _n_1;1:1 nervously _a_about;2:2 watching _v_1;4:4 Quotron named;6:6 data _n_1;6:6 data _electronic_a_1;7:7 machines _n_1;8:8 yesterday _a_1;9:9 morning _n_of;11:11 stunned _v_1;12:12 to _x;13:13 see _v_1;14:14 the _q;15:16 named Dow Jones;17:18 named Industrial Average;19:19 plummet _v_1;20:20 99 card;21:21 points _n_of;22:22 in _p_temp;0:0 Traders _n_1', '0:0 A card;1:1 minute _n_1;2:2 later _a_for;4:4 soared _v_1;5:5 128 card;6:6 points _n_of;7:7 then _c;8:8 zoomed _v_1;9:10 _p back down;11:11 113 card;12:12 points _n_of;13:13 69 card;14:14 below _p;15:15 Friday dofw;0:0 A card', '2:2 crazy," _a_1;3:3 said _v_to;4:5 named Neil Weisman;6:6 general _a_1;7:7 partner _n_1;8:8 of _p;9:10 named Chilmark Capital;2:2 crazy," _a_1', '2:2 like _p;3:3 flying _v_1;4:4 without _p;5:5 a _q;6:6 pilot _n_of;7:7 in _p;8:8 the _q;9:9 front _n_1;10:10 of _p;11:11 the _q;2:2 like _p', '0:0 But _c;1:1 those _q_dem;3:3 said _v_to;4:4 "This _q_dem;5:5 can\'t _v_modal;7:7 happening" _v_1;0:0 But _c', '0:0 The _q;1:1 Quotrons named;0:0 The _q', '0:1 named Quotron Systems;2:2 Inc _n_1;3:3 a _q;4:4 Citicorp named;5:5 unit _n_of;6:6 blamed _v_on;7:7 the _q;9:9 minute _n_1;9:9 minute card;10:10 foul- _a_1;10:10 foul- _up_p;13:13 "a _q;14:14 timing _v_1;15:15 problem _n_of;16:16 in _p;18:18 software" _n_1;19:19 caused _v_1;21:21 the _q;22:23 _a_1 enormous early;24:24 volume _n_of;26:26 about _x_deg;27:28 card 145 million;29:29 shares _n_of;30:30 in _p_temp;31:31 the _q;32:32 first ord;33:33 hour _n_1;34:34 of _p;35:36 named New York;37:38 _n_1 Stock Exchange;0:0 Quotron named', '0:0 The _q;1:1 prices _n_of;3:3 the _q;4:4 individual _a_1;5:5 stocks _n_1;7:7 make _v_up;9:9 the _q;10:10 average _n_of;12:12 correct _a_1;13:13 Quotron named;14:14 said _v_to;15:15 but _c;16:16 the _q;17:17 average _n_of;0:0 The _q', '0:0 Meanwhile _p;2:2 was _v_there;3:3 an _q;4:4 awful _a_1;5:5 lot _n_of;0:0 Meanwhile _p', '0:0 At _p_temp;1:1 about _x_deg;2:2 10:40 numbered_hour;3:3 a.m _x;4:4 on _p;5:5 the _q;6:8 over- the- counter _a_1;9:9 trading _v_1;10:10 desk _n_1;11:11 at _p;12:12 a _q;13:13 major _a_1;14:14 brokerage _n_unknown;15:15 firm _n_1;16:16 a _q;17:17 veteran _n_of;18:18 trader _n_1;20:20 buys _v_1;21:21 and _c;22:22 sells _v_1;23:25 _q some the;27:27 active _a_1;28:28 stocks _n_1;29:29 looked _v_at;31:31 a _q;32:32 senior _a_1;33:33 official _n_1;34:34 and _c;35:35 asked _v_1;0:0 At _p_temp', '1:1 the _q;2:2 market _n_1;3:3 up _p;4:4 or _c;1:1 the _q', '0:0 At _p_temp;1:1 the _q;2:2 time _n_of;3:3 Quotron named;5:5 reporting _v_to;7:7 the _q;8:8 industrial _a_1;9:9 average _n_of;11:11 down _p;12:12 70 card;0:0 At _p_temp', '0:0 In _p_state;1:1 fact _n_1;4:4 up _p;0:0 In _p_state', '0:1 named Holly Stark;2:2 a _q;3:3 vice _n_1;4:4 president _n_of;6:6 heads _v_1;7:7 the _q;8:8 trading _v_1;9:9 desk _n_1;10:10 at _p;11:12 named Dillon Read;13:13 Capital named;14:14 Corp _n_1;15:15 said _v_to;17:17 once _x_subord;19:19 figured _v_out;21:21 the _q;22:22 Quotron named;23:23 numbers _n_of;25:25 wrong _a_with;27:27 called _v_1;28:28 brokers _n_1;29:29 to _x;0:0 Holly named', '5:5 annoying _a_1;6:6 to _x;7:7 say _v_1;8:9 the least," _q;5:5 annoying _a_1', '0:0 To _x;1:1 confuse _v_1;2:2 matters _n_of;3:3 further _a_1;5:5 UAL named;6:7 _n_1 Corp stock;8:8 finally _a_1;9:9 opened _v_1;10:10 on _p;11:11 the _q;12:13 named New York;14:15 _n_1 Stock Exchange;16:16 at _p_temp;17:17 11:08 numbered_hour;18:18 a.m _x;19:19 the _q;20:20 price _n_of;22:22 listed _v_1;23:23 at _p;24:24 $ _n_1;25:25 324.75 card;26:26 a _p_per;27:27 share _n_of;28:28 up _p;29:29 about _x_deg;30:30 $ _n_1;31:31 45 card;32:32 from _p;33:33 Friday; dofw;34:35 in fact _a_1;37:37 true _a_of;38:38 price _n_of;39:39 was _v_id;40:40 $ _n_1;41:41 224.75 card;42:42 down _p;43:43 $ _n_1;0:0 To _x', '0:0 That _q_dem;1:1 was _v_id;2:2 the _q;3:4 named New York;5:6 _n_1 Stock Exchange;0:0 That _q_dem', '0:0 A _q;1:1 spokesman _n_1;2:2 cited _v_for;3:3 a _q;4:4 "technical _a_1;5:5 error" _n_1;6:6 and _c;7:9 _v_1 declined elaborate', '0:0 And _c;2:2 were _v_there;3:3 other _a_1;0:0 And _c', '0:0 When _x_subord;1:1 the _q;2:2 market _n_1;3:3 opened _v_1;4:4 at _p_temp;5:5 9:30 numbered_hour;6:6 a.m _x;7:7 EST timezone_p;8:8 a _q;9:9 reporter _n_of;10:10 for _p;11:11 the _q;12:12 Reuters named;13:13 newswire _n_unknown;14:14 miscalculated _v_1;14:14 miscalculated _mis-_a_error;15:15 the _q;16:16 industrial _a_1;17:19 _n_of average drop;20:20 as _p;21:21 a _q;22:22 4 card;23:23 % _n_of;24:24 decline _n_1;25:25 when _x_subord;27:27 really _a_1;29:29 down _p;30:30 0.7 card;0:0 When _x_subord', '1:1 was _v_id;2:2 a _q;3:3 case _n_of;5:5 human _a_1;6:6 error _n_1;9:9 found _v_1;10:10 almost _x_deg;11:11 immediately _a_1;12:12 and _c;13:13 corrected," _v_1;14:14 a _q;15:15 spokesman _n_1;16:16 for _p;17:17 Reuter named;18:18 in _p;19:20 named New York;1:1 was _v_id', '0:0 Meanwhile _p;1:1 some _q;2:3 _n_1 currency traders;4:4 at _p;5:5 West named;6:6 German _a_1;7:7 banks _n_of;8:8 in _p;9:9 Frankfurt named;10:10 said _v_to;12:12 sold _v_1;13:13 dollars _n_1;14:14 on _p;15:15 the _q;16:16 news _n_of;17:17 and _c;18:18 had _v_qmodal;20:20 buy _v_back;23:23 later _a_for;24:24 at _p;25:25 higher _a_1;0:0 Meanwhile _p', '0:0 But _c;2:2 was _v_itcleft;3:3 the _q;4:4 Quotron named;5:5 problems _n_of;7:8 _v_1 had lingering;0:0 But _c', '1:1 was _v_id;2:2 the _q;3:3 second ord;4:4 time _n_of;5:5 in _p;8:8 a card;9:9 week _n_1;11:11 Quotron named;13:13 had _v_1;14:14 problems _n_of;15:15 calculating _v_1;16:16 the _q;17:17 industrial _a_1;1:1 was _v_id', '0:0 At _p;1:1 the _q;2:2 start _n_1;3:3 of _p;4:4 trading _v_1;5:5 last _a_1;6:6 Wednesday dofw;7:7 the _q;8:8 average _n_of;9:9 appeared _v_to;11:11 plunge _v_into;12:13 more than _p;14:14 200 card;0:0 At _p', '0:0 Actually _a_1;3:3 down _p;4:4 only _x_deg;5:6 a few _a_1;7:7 points _n_of;8:8 at _p_temp;9:9 the _q;0:0 Actually _a_1', '0:0 Quotron named;1:1 said _v_to;2:2 that _q_dem;3:3 snafu _n_unknown;5:5 lasted _v_1;6:6 nine card;7:7 minutes _n_1;8:8 resulted _v_from;10:10 a _q;11:11 failure _n_1;13:13 adjust _v_to;14:14 for _p;15:15 a _q;19:20 _n_1 stock split;21:21 at _p;22:23 named Philip Morris;0:0 Quotron named', '0:0 A _q;1:1 Quotron named;2:2 spokeswoman _n_unknown;3:3 said _v_to;4:4 recent _a_1;5:5 software _n_1;6:6 changes _n_of;7:7 may _v_modal;9:9 contributed _v_to;11:11 yesterday _a_1;0:0 A _q', '1:1 said _v_to;2:2 Quotron named;3:3 switched _v_from-to;5:5 a _q;6:6 backup _n_1;7:7 system _n_of;8:8 until _x_h;9:9 the _q;10:10 problems _n_of;1:1 said _v_to', '0:0 "Today _a_1;1:1 of _p;2:2 all _q;3:3 days," _n_of;0:0 "Today _a_1', '0:0 "The _q;1:1 eyes _n_1;2:2 of _p;3:3 the _q;4:4 world _n_of;0:0 "The _q', '0:1 named Steven F;2:2 Kaplan named;4:4 named _v_1;5:5 a _q;6:6 senior _a_1;7:7 vice _n_1;8:8 president _n_of;10:10 this _q_dem;11:12 _n_1 graphics equipment;0:0 Steven named', '0:0 Houston named;1:1 attorney _n_1;2:3 named Dale Friend;4:4 representing _v_1;5:5 a _q;6:6 plaintiff _n_1;7:7 in _p;8:8 a _q;9:9 damage _n_to;10:10 suit _n_of;11:11 says _v_to;14:14 negotiated _v_1;15:15 a _q;16:16 settlement _n_of;19:19 strike _v_1;20:20 a _q;21:21 blow _n_1;22:22 for _p;0:0 Houston named', '0:0 Literally _a_1', '1:1 turns _v_out;3:3 Mr _n_1;4:4 Friend named;6:6 client _n_1;7:8 named Machelle Parks;9:9 of _p;10:10 Cincinnati named;12:12 like _v_1;13:13 the _q;14:14 way _n_of;15:16 _n_1 defense attorney;17:18 named Tom Alexander;19:19 acted _v_seem+to;20:20 during _p;21:21 the _q;22:22 legal _a_1;1:1 turns _v_out', '0:0 So _a_thus;3:3 agreed _v_1;5:5 forgo _v_unknown;6:6 monetary _a_1;7:7 damages _n_1;8:8 against _p;9:9 Mr _n_1;10:10 Alexander named;12:12 client _n_1;13:13 in _p;14:14 return _n_of;15:15 for _p;16:16 the _q;17:17 right _n_1;19:19 punch _v_1;20:20 the _q;0:0 So _a_thus', '0:1 named Ms Parks;3:3 mother _n_of;4:4 also _a_1;5:5 gets _v_to;7:7 cuff _v_1;8:8 Mr _n_1;0:0 Ms named', '0:0 So _a_1;1:1 does _v_1;2:2 Mr _n_1;3:3 Friend named;4:4 and _c;6:7 _n_1 law partner;8:9 named Nick Nichols', '0:0 Last _a_1;1:2 _n_1 month Mr;3:3 Friend named;4:4 says _v_to;5:5 Mr _n_1;6:6 Alexander named;8:8 associate _n_of;9:9 agreed _v_with;11:11 Derr named;12:12 would _v_modal;13:13 pay _v_for;14:14 $ _n_1;15:15 50,000 card;16:16 as _p;17:19 _q part an;20:20 overall _a_1;0:0 Last _a_1', '0:0 But _c;1:1 Mr _n_1;2:2 Alexander named;3:3 scuttled _v_1;4:4 the _q;5:5 deal _n_1;6:6 at _p_temp;7:7 the _q;8:8 last _a_1;9:9 minute _n_1;10:10 angering _v_1;11:11 the _q;12:14 _n_1 plaintiff side', '1:1 never _a_1;2:2 agreed _v_to;5:5 Mr _n_1;6:6 Alexander named;7:8 _v_to says adding;13:13 necessary _a_for;15:15 pay _v_for;16:16 these _q_dem;17:17 nuisance _n_1;1:1 never _a_1', '0:0 When _x_subord;1:1 Ms _n_1;2:2 Parks named;3:3 and _c;5:5 mother _n_of;6:6 heard _v_about;10:10 happened _v_1;11:11 Mr _n_1;12:12 Friend named;13:13 says _v_to;15:15 volunteered _v_1;16:16 that _q_dem;18:18 would _v_modal;19:21 _v_1 like give;22:22 Mr _n_1;23:23 Alexander named;24:24 a _q;25:25 good _a_at-for-of;0:0 When _x_subord', '0:0 Mr _n_1;1:1 Friend named;2:2 says _v_to;4:4 passed _v_1;5:5 that _q_dem;6:6 along _p;9:9 adversary _n_1;10:10 and _c;11:11 soon _p;14:14 talking _v_about;15:15 about _p;16:16 the _q;17:17 ground _n_1;18:18 rules _n_of;19:19 under _p_state;21:21 Derr named;22:22 could _v_modal;23:23 keep _v_unspec;25:25 money _n_1;26:26 and _c;27:27 the _q;28:28 plaintiffs _n_1;29:29 could _v_modal;30:30 take _v_1;31:31 a _q;32:32 shot _n_of;33:33 at _p;34:34 Mr _n_1;0:0 Mr _n_1', '0:0 Mr _n_1;1:1 Friend named;2:2 says _v_to;4:6 _v_1 agreed strike;7:7 Mr _n_1;8:8 Alexander named;9:9 above _p;10:10 the _q;0:0 Mr _n_1', '0:0 Ms _n_1;1:1 Parks named;2:2 and _c;4:4 mother _n_of;5:7 _v_1 indicated want;9:9 "catch _v_1;11:11 unawares _a_1;12:13 _p from behind,";0:0 Ms _n_1', "0:0 Mr _n_1;1:1 Alexander named;2:2 for _p;4:4 part _n_1;5:5 insisted _v_1;7:7 the _q;8:8 punchers _n_unknown;9:9 can't _v_modal;10:10 assign _v_to;12:12 pummeling _n_unknown;13:13 rights _n_1;15:15 anyone _q;16:16 else _a_1;17:17 can't _v_modal;18:18 use _v_1;19:19 a _q;20:20 blunt _a_1;21:21 instrument _n_of;22:22 and _c;23:23 can't _v_modal;24:24 take _v_1;25:25 a _q;26:26 running _v_1;0:0 Mr _n_1", '0:0 Mr _n_1;1:1 Alexander named;2:2 says _v_to;4:4 regards _v_as;5:5 the _q;6:6 agreement _n_1;10:10 submitted _v_to;12:12 a _q;13:13 judge _n_of;15:15 something _q;16:16 of _p;17:17 a _q;0:0 Mr _n_1', '0:0 However _a_1;2:2 acknowledges _v_to;4:4 "have _v_1;5:5 the _q;6:6 option _n_1;7:7 of _p;8:8 taking _v_1;9:9 a _q;10:10 swat _n_unknown;11:11 at _p;13:13 if _x_then;15:15 really _a_1;0:0 However _a_1', '0:0 Mr _n_1;1:1 Friend named;2:2 says _v_to;4:6 _n_1 side "dead;0:0 Mr _n_1', '0:0 Although _x;3:3 contemplate _v_1;4:4 delivering _v_to;5:5 any _q;6:6 disabling _v_1;7:7 blows _n_1;9:9 says _v_to;11:11 Mr _n_1;12:12 Alexander named;15:17 _v_1 asked sign;18:18 a _q;19:19 release _n_of;20:20 from _p;21:21 liability _n_1;0:0 Although _x', '0:0 As _x_subord;1:1 financial _a_1;2:2 markets _n_1;3:4 _v_1 rebounded trading;5:5 volume _n_of;6:6 in _p;7:7 the _q;8:9 named Chicago Mercantile;10:10 Exchange named;12:12 huge _a_1;17:17 500 card;18:19 _n_1 stock- index;20:21 _n_1 futures pit;22:23 _v_1 soared reaching;25:25 record _n_of;25:25 record _near_a_to;26:26 levels _n_1;27:27 for _p;28:28 the _q;29:29 first ord;30:30 time _n_of;31:31 since _p;32:32 October mofy;0:0 As _x_subord', '0:0 The _q;1:1 sudden _a_1;2:2 influx _n_1;3:3 of _p;4:4 liquidity _n_1;5:5 enabled _v_1;6:6 several _a_1;7:7 traders _n_1;9:9 reap _v_1;11:11 figure _n_1;11:11 figure card;12:12 windfalls _n_unknown;13:13 in _p;14:14 a _q;15:15 matter _n_of;17:17 minutes _n_1;18:18 as _x_subord;19:19 prices _n_of;20:20 soared _v_1;21:21 traders _n_1;0:0 The _q', '0:0 "Guys _n_1;2:2 minting _v_1;3:3 money _n_1;4:4 in _p;5:6 _a_1 there today,";7:7 said _v_to;8:9 named John Legittino;10:10 a _q;11:12 _n_1 futures broker;13:13 for _p;14:15 named Elders Futures;16:16 Inc _n_1;17:17 in _p;0:0 "Guys _n_1', '0:0 The _q;1:1 S&P _n_unknown;2:2 500 card;3:4 _n_1 futures contract;6:6 moves _v_1;7:7 in _p;8:8 fractions _n_of;9:9 of _p;10:10 an card;11:11 index _n_1;12:12 point _n_of;13:13 under _p;14:14 normal _a_1;15:15 conditions _n_1;16:16 jumped _v_1;17:19 card two three;20:20 points _n_of;21:21 in _p_temp;22:22 seconds _n_1;23:24 _a_1 early yesterday;25:25 after _p;26:26 an _q;27:27 initial _a_1;28:28 downturn _n_unknown;29:29 then _c;30:30 moved _v_1;31:32 _a_1 strongly higher;33:33 the _q;34:34 rest _n_of;36:36 the _q;0:0 The _q', '0:0 Each _q;1:1 index _n_1;2:2 point _n_of;3:3 represents _v_1;4:4 a _q;5:5 $ _n_1;6:6 500 card;7:7 profit _n_1;8:8 for _p;9:9 each _q;10:10 S&P named;11:11 500 card;12:12 contract _n_1;0:0 Each _q', '0:0 For _p;1:1 the _q;2:2 first ord;3:3 time _n_of;4:4 since _p;5:5 the _q;6:6 1987 yofc;7:8 _n_1 crash traders;9:9 said _v_to;13:13 able _a_1;15:15 trade _v_1;16:17 card several hundred;18:18 S&P _n_unknown;19:19 500 card;20:20 contracts _n_1;21:21 at _p;22:22 a card;23:23 time _n_of;24:24 in _p;25:25 a _q;26:27 _a_1 highly liquid;0:0 For _p', '1:1 institutions _n_1;2:2 and _c;3:3 individual _a_1;4:4 investors _n_1;6:6 shied _v_away;8:8 from _p;9:10 _n_1 stock- index;11:11 futures _n_1;12:12 blaming _v_for;15:15 speeding _v_1;16:16 the _q;17:18 _n_1 stock market;19:19 crash _n_1;20:20 on _p_temp;21:21 Black named;22:22 Monday dofw;23:23 two card;24:24 years _n_1;1:1 institutions _n_1', '0:0 Since _p;1:1 the _q;2:4 _n_1 crash futures;5:5 traders _n_1;7:7 assumed _v_1;8:8 large _a_1;9:9 positions _n_of;10:10 for _p;11:11 fear _n_1;13:13 the _q;14:14 S&P named;15:15 500 card;16:16 market _n_1;17:17 with _p;21:22 _n_of customer order;23:23 flow _n_of;24:24 missing _v_1;25:25 would _v_modal;26:26 dry _v_up;28:28 if _x_then;29:29 prices _n_of;30:30 turned _v_1;0:0 Since _p', '2:2 400 card;3:3 traders _n_1;4:4 jammed _v_1;5:5 the _q;6:6 S&P _n_unknown;7:7 500 card;8:9 _n_1 futures pit;10:10 to _x;11:11 await _v_1;12:12 the _q;13:13 opening _v_1;2:2 400 card', '0:0 Traders _n_1;2:2 shouting _v_1;3:3 bids _n_1;4:4 and _c;5:5 offers _n_1;6:7 a full _x_deg;8:8 five card;9:9 minutes _n_1;10:10 before _p;11:11 the _q;12:12 start _n_1;13:13 of _p;14:14 trading _v_1;15:15 at _p_temp;16:16 8:30 numbered_hour;0:0 Traders _n_1', '0:0 The _q;1:1 contract _n_1;2:2 fell _v_1;3:3 five card;4:4 points _n_of;5:5 at _p;6:6 the _q;7:7 open _n_1;8:8 to _p;9:9 323.85 card;10:10 the _q;11:11 maximum _a_1;12:12 opening _v_1;13:13 move _n_1;14:14 allowed _v_1;15:15 under _p;16:16 safeguards _n_1;17:17 adopted _v_1;19:19 the _q;20:20 Merc named;21:21 to _x;22:22 stem _v_1;23:23 a _q;24:25 _n_1 market slide', '0:0 But _c;1:1 several _a_1;2:2 traders _n_1;3:3 quickly _a_1;4:4 stepped _v_up;6:6 and _c;7:7 bid _v_1;8:8 for _p;9:9 contracts _n_1;10:10 driving _v_1;11:11 prices _n_of;12:13 _a_1 sharply higher', '0:0 The _q;1:1 market _n_1;2:2 hovered _v_1;3:3 near _p;4:4 Friday dofw;6:6 closing _v_1;7:7 price _n_of;9:9 328.85 named;10:10 for _p;11:11 about _x_deg;14:14 hour _n_1;15:15 moving _v_1;16:16 several _a_1;17:17 index _n_1;18:18 points _n_of;19:19 higher _a_1;20:20 or _c;21:21 lower _a_1;22:22 in _p_temp;23:23 seconds _n_1;24:24 then _c;25:25 broke _v_1;26:26 higher _a_1;27:27 and _c;29:29 look _v_1;0:0 The _q', '0:0 The _q;1:1 S&P named;2:2 500 card;3:3 contract _n_1;5:5 expires _v_1;6:6 in _p_temp;7:7 December mofy;8:8 closed _v_1;9:9 up _p;10:10 a _q;11:11 record _n_of;12:12 15.65 card;13:13 points _v_1;14:14 on _p;15:15 volume _n_of;17:17 nearly _x_deg;18:18 80,000 card;0:0 The _q', '1:1 could _v_modal;2:2 buy _v_1;3:3 at _p;4:4 the _q;5:5 bid _n_1;6:6 and _c;7:7 sell _v_1;8:8 at _p;9:9 the _q;10:10 offer _n_1;11:11 and _c;12:12 make _v_1;13:13 a _q;14:14 fortune," _n_1;1:1 could _v_modal', '0:0 Several _a_1;2:3 named Wall Street;5:5 largest _a_1;6:7 _n_1 securities firms;8:8 including _v_1;9:10 named Salomon Brothers;11:11 Inc _n_1;12:12 and _c;13:13 PaineWebber named;14:14 Inc _n_1;15:15 were _v_id;16:17 _a_1 also large;18:18 buyers _n_of;19:19 traders _n_1;0:0 Several _a_1', '0:1 named Salomon Brothers;3:3 among _p;4:4 the _q;5:5 largest _a_1;6:6 sellers _n_of;8:9 _n_1 stock- index;10:10 futures _n_1;11:11 last _a_1;12:13 _n_1 week traders;0:0 Salomon named', '0:0 Brokerage named;1:1 firms _n_1;2:2 as _p;3:3 a _q;4:4 rule _n_of;6:6 comment _v_on;9:10 _n_1 market activity', '0:0 Unlike _p;1:1 the _q;2:2 week _n_1;3:3 following _v_1;4:4 Black named;5:5 Monday dofw;6:6 two card;7:7 years _n_1;8:8 ago _p;9:9 individual _a_1;10:10 traders _n_1;11:11 in _p;12:12 the _q;13:13 S&P named;14:14 500 card;15:15 pit _n_1;17:17 also _a_1;19:20 _a_unknown uncharacteristically circumspect;21:21 about _p;24:24 day _n_of;24:24 day card;0:0 Unlike _p', '0:0 "With _x_subord;1:1 the _q;2:2 FBI named;3:3 around _p;4:4 here _a_1;5:5 bragging _v_to;6:6 rights _n_1;7:7 are _v_id;8:8 a _q;9:9 thing _n_of-about;11:11 the _q;12:12 past," _n_1;13:13 said _v_to;14:14 one card;15:15 trader _n_1;16:16 referring _v_to;18:18 the _q;19:19 federal _a_1;20:20 investigation _n_1;21:21 of _p;22:22 futures _n_1;23:23 trading _v_1;26:26 far _a_1;28:28 resulted _v_in;30:30 46 card;31:31 indictments _n_1;32:32 lodged _v_1;33:33 against _p;34:34 individuals _n_1;35:35 on _p;36:36 the _q;37:37 Merc named;38:38 and _c;39:39 the _q;40:41 named Chicago Board;42:42 of _p;0:0 "With _x_subord', '0:0 The _q;1:1 market _n_1;2:2 for _p;3:3 $ _n_1;4:5 card 200 billion;6:6 of _p;8:8 yield _n_1;8:8 yield _high_a_1;9:10 _n_1 junk bonds;11:11 regained _v_1;12:12 some _q;15:15 footing _n_1;16:16 as _x_subord;17:17 the _q;18:19 named Dow Jones;20:21 named Industrial Average;22:22 rebounded _v_1;23:23 from _p;24:24 Friday dofw;0:0 The _q', '0:0 But _c;1:1 the _q;2:3 _n_1 junk recovery;4:4 led _v_1;6:6 the _q;7:7 bellwether _n_unknown;8:9 named RJR Holdings;10:10 bonds _n_1;0:0 But _c', '0:0 No _q;1:2 _v_1 trading existed;3:3 for _p;4:4 the _q;5:5 vast _a_1;6:6 majority _n_of;8:9 _n_1 junk bonds;10:11 _n_1 securities industry;12:12 officials _n_1;0:0 No _q', '0:0 On _p_temp;1:1 Friday dofw;2:2 trading _v_1;3:3 in _p;4:4 practically _x_deg;5:5 every _q;6:6 issue _n_1;7:7 ground _v_1;8:8 to _p;9:9 a _q;10:10 halt _n_1;11:11 as _x_subord;12:12 potential _a_1;13:13 buyers _n_of;14:14 fled _v_from;15:15 and _c;16:16 brokerage _n_unknown;17:17 firms _n_1;19:19 unwilling _a_1;21:21 provide _v_1;22:22 bid _n_1;23:23 and _c;24:24 offer _n_1;25:25 prices _n_of;26:26 for _p;27:27 most _q;0:0 On _p_temp', '0:0 "But _c;2:2 had _v_1;3:3 a _q;4:5 _a_1 fairly active;6:6 day _n_of;0:0 "But _c', '0:0 At _p;1:2 named Drexel Burnham;3:3 Lambert named;4:4 Inc _n_1;5:5 the _q;6:6 leading _v_1;7:7 underwriter _n_1;8:8 of _p;9:10 _n_1 junk bonds;13:13 prepared _a_1;16:16 in _p;17:17 a _q;18:18 very _x_deg;19:19 bad _a_at;20:20 mood _n_1;21:21 tonight," _a_1;22:22 said _v_to;23:24 named David Feinman;25:25 a _q;26:27 _n_1 junk bond;0:0 At _p', '0:0 "Now _a_1;2:2 feel _v_1;3:3 maybe _a_1;5:5 \'s _v_there;6:6 a _q;8:8 bit _n_of;0:0 "Now _a_1', "0:0 But _c;1:1 before _x_h;2:2 the _q;3:4 _n_1 stock market;5:5 rebounded _v_1;6:6 from _p;7:7 a _q;8:9 _a_1 sharp early;10:11 sell- off _n_1;12:12 yesterday _a_1;14:14 said _v_to;16:16 couldn't _v_modal;17:17 buy _v_1;18:19 _n_1 {junk bonds};20:20 and _c;22:22 couldn't _v_modal;0:0 But _c", '0:0 Yesterday _a_1;2:2 rally _n_1;4:4 led _v_1;6:7 named RJR Holdings;8:8 13 card;9:9 3/4 fraction;10:10 % _n_of;11:11 bonds _n_1;13:13 initially _a_1;14:14 tumbled _v_1;15:15 three card;16:16 points _n_of;17:17 or _c;18:18 $ _n_1;19:19 30 card;20:20 for _p;21:21 each _q;22:22 $ _n_1;23:23 1,000 card;24:24 face _n_1;25:25 amount _n_of;26:26 to _p;27:27 96 card;28:28 1/4 fraction;29:29 before _p;30:30 rebounding _v_1;31:31 to _p;32:32 99 card;0:0 Yesterday _a_1', '0:0 Bonds _n_1;1:1 issued _v_1;3:4 named Kroger Duracell;5:5 Safeway named;6:6 and _c;7:8 named American Standard;9:9 also _a_1;10:10 showed _v_1;11:11 big _a_1;12:12 gains _n_1;13:13 recovering _v_1;14:14 almost _x_deg;15:15 all _q;17:17 losses _n_of;18:18 from _p;19:19 Friday dofw;20:20 and _c;21:22 _a_1 early yesterday', '0:0 But _c;1:1 traders _n_1;2:2 said _v_to;3:3 the _q;4:5 _n_1 junk bond;6:6 market _n_1;7:7 increasingly _a_1;9:9 separating _v_from;10:10 into _p;11:11 a _q;13:13 tier _n_1;13:13 tier _top_a_1;14:14 group _n_of;15:15 in _p_state;17:17 trades _n_of;18:18 can _v_modal;20:20 executed _v_1;21:21 easily _a_for;22:22 and _c;23:23 a _q;24:24 larger _a_1;25:25 group _n_of;28:28 quality _n_1;28:28 quality _low_a_1;29:29 bonds _n_1;30:30 in _p_state;32:32 liquidity _n_1;34:34 or _c;35:35 the _q;36:36 ability _n_1;38:38 trade _v_1;39:39 without _p;42:42 difficulty _n_1;45:45 steadily _a_1;46:46 deteriorated _v_1;47:47 this _q_dem;0:0 But _c', '0:0 "Liquidity _n_1;2:2 returned _v_1;3:3 to _p;4:4 the _q;5:6 _a_1 vast middle;7:7 ground _n_1;8:8 of _p;9:9 the _q;10:10 market," _n_1;11:11 said _v_to;12:12 Mr _n_1;13:13 Minella named;14:14 of _p;0:0 "Liquidity _n_1', '0:0 "The _q;1:1 deadbeats _n_unknown;2:2 are _v_id;3:3 still _a_1;4:4 deadbeats," _n_unknown;5:5 said _v_to;6:6 Mr _n_1;7:7 Feinman named;8:8 of _p;0:0 "The _q', '0:0 Analysts _n_1;2:2 concerned _a_about;6:6 the _q;8:8 yield _n_1;8:8 yield _high_a_1;9:9 market _n_1;11:11 remain _v_1;12:12 treacherous _a_1;13:13 for _p;0:0 Analysts _n_1', '0:1 named Mark Bachmann;2:2 a _q;3:3 senior _a_1;4:4 vice _n_1;5:5 president _n_of;6:6 at _p;11:11 Corp _n_1;12:12 confirms _v_as;15:15 is _v_there;16:16 "increasing _v_1;17:17 concern _n_1;18:18 about _p;19:19 the _q;20:20 future _a_1;21:21 liquidity _n_1;22:22 of _p;23:23 the _q;24:25 _n_1 junk bond;0:0 Mark named', '0:1 _n_1 "Junk bonds;2:2 are _v_id;3:3 a _q;4:4 highly _a_1;5:5 stratified _a_unknown;6:6 market," _n_1;7:7 said _v_to;8:9 named Lewis Glucksman;10:10 vice _n_1;11:11 chairman _n_of;13:14 named Smith Barney;15:16 named Harris Upham;0:0 "Junk _n_1', "1:1 's _v_there;2:2 a _q;3:3 whole _a_1;4:4 bunch _n_of;6:9 _n_1 stuff money;10:10 good _a_at-for-of;11:11 and _c;12:12 a _q;13:13 whole _a_1;14:14 bunch _n_of;16:16 stuff _n_1;1:1 's _v_there", '0:0 Analysts _n_1;1:1 at _p;6:6 say _v_to;7:8 _n_1 junk bond;9:9 offerings _n_of;10:10 by _p;11:11 "tightly _a_1;12:12 stretched" _v_1;13:13 issuers _n_unknown;14:14 seem _v_to;0:0 Analysts _n_1', '1:1 could _v_modal;2:2 still _a_1;3:3 have _v_prd;4:4 some _q;5:5 very _x_deg;6:6 bad _a_at;7:7 times _n_of;8:8 ahead," _p;9:9 said _v_to;10:10 Mr _n_1;1:1 could _v_modal', '2:2 possible _a_for;4:4 have _v_1;5:5 a _q;6:6 10 card;7:7 % _n_of;8:8 default _n_1;9:9 rate _n_of;10:10 in _p;11:11 one card;12:12 year _n_1;13:13 because _x;16:16 already _a_1;17:17 seeing _v_1;18:18 big _a_1;19:19 problems _n_of;20:20 in _p;21:21 the _q;22:22 midst _n_of;24:24 a _q;25:25 pretty _x;26:26 strong _a_1;2:2 possible _a_for', '2:4 _a_1 certainly comfortable;5:5 saying _v_to;8:8 seen _v_1;9:9 the _q;2:2 certainly _a_1', '0:0 But _c;1:1 yesterday _a_1;3:3 rally _n_1;4:4 among _p;5:5 "good" _a_at-for-of;6:6 junk _n_1;7:7 was _v_id;8:8 a _q;9:9 badly _a_1;10:10 needed _v_1;11:11 tonic _n_1;12:12 for _p;13:13 the _q;0:0 But _c', '1:1 issues _n_1;2:2 "bounced _v_1;3:3 off _p;4:4 the _q;5:6 _n_1 floor," Mr;7:7 Minella named;8:8 said _v_to;9:9 and _c;10:10 benchmark _n_unknown;11:12 _n_1 junk issues;13:13 "recovered _v_1;14:14 all _q;17:17 losses" _n_of;18:18 from _p;19:19 Friday dofw;20:20 and _c;21:22 _a_1 early yesterday', '0:0 In _p;1:1 contrast _n_1;3:3 says _v_to;4:4 "The _q;5:6 _n_1 stock market;7:7 gained _v_1;8:8 back _p;9:10 _x_deg only about;11:11 half _q;14:14 lost _v_1;15:15 Friday dofw;16:16 and _c;17:17 the _q;18:18 {government} _n_of;19:20 _n_1 bond market;21:21 lost _v_1;22:22 about _x_deg;23:23 half _q;26:26 gained _v_1;0:0 In _p', '0:0 Traders _n_1;1:1 said _v_to;2:2 yesterday _a_1;4:4 rally _n_1;6:6 fueled _v_unknown;8:8 insurance _n_1;9:9 companies _n_of;10:10 looking _v_for;12:12 bargains _n_1;13:13 after _p;14:14 a _q;15:15 drastic _a_1;16:16 slide _n_1;17:17 in _p;18:18 prices _n_of;19:19 the _q;20:20 past _a_1;0:0 Traders _n_1', '0:0 In _p_state;1:1 addition _n_1;2:2 mutual _a_1;3:3 funds _n_1;5:5 appear _v_to;7:7 be _v_id;8:8 major _a_1;9:9 sellers _n_of;12:12 yield _n_1;12:12 yield _high_a_1;13:13 securities _n_1;14:14 as _x_subord;0:0 In _p_state', '0:0 "Sometimes _a_1;1:1 a _q;2:2 shakeout _n_unknown;4:4 healthy," _a_1;5:5 said _v_to;6:6 Drexel named;8:8 Mr _n_1;0:0 "Sometimes _a_1', '0:0 "People _n_of;2:2 learn _v_1;0:0 "People _n_of', '0:0 If _x_then;2:2 do _v_1;3:3 good _a_at-for-of;4:4 credit _n_1;5:5 analysis _n_of;8:8 avoid _v_1;9:9 the _q;10:11 _n_1 hand grenades', '1:1 think _v_1;2:2 the _q;3:3 market _n_1;5:5 in _p;6:6 good _a_at-for-of;1:1 think _v_1', '0:0 Should _v_modal;2:2 really _a_1;3:3 own _v_1;0:0 Should _v_modal', "0:0 That _q_dem;1:1 's _v_id;2:2 a _q;3:3 question _n_about;4:4 a _q;5:7 _n_of lot people;9:10 _v_1 asking following;11:11 the _q;12:13 _n_1 stock market;15:15 stunning _v_1;16:16 display _n_of;0:0 That _q_dem", "0:0 That _q_dem;1:1 's _v_id;2:2 a _q;3:3 question _n_about;4:4 a _q;5:7 _n_of lot people;9:10 _v_1 asking following;11:11 the _q;12:13 _n_1 stock market;15:15 stunning _v_1;16:16 display _n_of;0:0 That _q_dem", '0:0 Whipsawed _a_unknown;1:1 financially _a_1;2:2 and _c;3:3 emotionally _a_1;4:4 by _p;5:5 Friday dofw;7:7 heartstopping _a_unknown;9:9 point _n_of;9:9 point card;10:10 drop _n_of;11:11 in _p;12:12 the _q;13:14 named Dow Jones;15:16 named Industrial Average;17:17 and _c;18:18 yesterday _a_1;21:21 point _n_of;21:21 point card;22:22 rebound _n_1;25:25 wondering _v_1;27:27 an _q;28:28 individual _n_1;29:29 has _v_1;30:30 any _q;31:31 business _n_1;33:33 in _p;34:34 the _q;0:0 Whipsawed _a_unknown', '0:0 The _q;1:1 answer _n_to;2:2 say _v_to;3:3 academic _a_1;4:4 researchers _n_of;5:5 money _n_1;6:6 managers _n_of;7:7 and _c;8:9 _n_1 investment specialists;11:11 yes _a_1;13:15 as long as _x;17:17 approach _v_1;18:18 the _q;19:20 _n_1 stock market;21:21 as _p;22:22 an _q;0:0 The _q', "0:0 But _c;2:2 say _v_to;3:3 people _n_of;4:4 shouldn't _v_modal;5:5 try _v_1;7:7 be _v_id;8:8 traders _n_1;10:10 buy _v_1;11:11 and _c;12:12 sell _v_1;13:13 in _p;14:14 an _q;15:15 effort _n_1;17:17 ride _v_1;18:18 the _q;19:19 latest _a_for;20:20 economic _a_1;21:21 trend _n_1;22:22 or _c;23:23 catch _v_1;24:24 the _q;25:26 _a_1 next hot;0:0 But _c", '0:0 The _q;1:1 case _n_of;2:2 for _p;3:3 owning _v_1;4:4 stocks _n_1;5:5 over _p;6:6 the _q;8:8 term _n_of;8:8 term _long_a_1;0:0 The _q', '0:0 A card;1:1 dollar _n_1;2:2 invested _v_in;4:4 the _q;5:6 _n_1 stock market;7:7 in _p_temp;8:8 1926 yofc;9:9 would _v_modal;11:11 grown _v_1;12:12 to _p;13:13 $ _n_1;14:14 473.29 card;15:15 by _p;16:16 the _q;17:17 end _n_of;19:19 last _a_1;20:20 June mofy;21:22 according to _p;23:24 named Laurence Siegel;25:25 managing _v_1;26:26 director _n_of;27:27 at _p;28:29 named Ibbotson Associates;0:0 A card', '0:0 But _c;1:1 a _q;2:2 dollar _n_1;3:3 invested _v_in;6:6 term _n_of;6:6 term _long_a_1;7:7 bonds _n_1;8:8 in _p_temp;9:9 1926 yofc;10:10 would _v_modal;12:12 grown _v_1;13:13 to _p;14:14 only _x_deg;15:15 $ _n_1;16:16 16.56 card;17:17 and _c;18:18 a _q;19:19 dollar _n_1;20:20 put _v_1;21:21 in _p;22:22 Treasury named;23:23 bills _n_of;24:24 would _v_modal;25:25 equal _v_1;26:26 a _q;27:27 meager _a_1;28:28 $ _n_1;0:0 But _c', '0:0 Over _p_state;1:1 time _n_of;2:2 the _q;3:3 odds _n_1;4:4 increasingly _a_1;5:5 favor _v_1;6:6 the _q;7:7 investor _n_1;8:8 with _p;9:9 a _q;10:10 diversified _v_1;0:0 Over _p_state', '0:0 "If _x_then;3:3 need _v_1;4:4 the _q;5:5 money _n_1;6:6 for _p;7:7 10 card;8:8 years _n_1;10:10 \'s _v_there;11:11 a _q;12:13 clear- cut _a_1;14:14 case _n_of;15:15 for _p;16:16 sticking _v_to;18:18 a _q;19:19 steady _a_1;20:20 core _n_1;21:21 of _p;22:23 _n_1 stocks," Mr;24:24 Gregory named;0:0 "If _x_then', '0:1 _n_1 Stock- market;2:2 investments _n_1;3:3 also _a_1;4:4 help _v_to;5:5 balance _v_1;6:6 the _q;7:7 other _a_1;8:8 assets _n_1;9:9 an _q;10:10 individual _n_1;11:11 owns _v_1;12:12 says _v_to;13:14 named John Blankenship;15:15 Jr _n_1;16:16 president _n_of;18:18 the _q;19:19 Institute _n_of;21:22 named Certified Financial;0:0 Stock- _n_1', '1:1 are _v_there;2:2 some _q;3:3 important _a_for;4:4 caveats _n_1;5:5 : _p_namely;6:6 Before _p;7:7 investing _v_in;9:10 _n_1 stocks individuals;11:11 should _v_modal;12:12 have _v_prd;13:14 at least _x_deg;15:17 card three six;18:18 months _n_1;19:19 of _p;20:21 _n_1 living expenses;22:22 set _v_aside;24:24 in _p;25:25 the _q;26:26 bank _n_of;27:27 most _q;28:29 _n_1 investment advisers;1:1 are _v_there', '0:0 Individuals _n_1;1:1 also _a_1;2:2 should _v_modal;3:3 focus _v_on;5:5 building _v_1;6:6 equity _n_1;7:7 in _p;8:8 a _q;9:9 home _n_of-n;11:11 provides _v_1;12:12 some _q;13:13 protection _n_of;14:14 against _p;15:15 inflation _n_of;16:18 as well as _c;19:19 a _q;20:20 nest- _n_of;21:21 egg _n_1;23:23 can _v_modal;25:25 cashed _v_in;27:28 _p late in;29:29 life _n_of;30:30 to _x;31:31 help _v_to;32:32 cover _v_1;33:33 the _q;34:34 cost _n_1;35:35 of _p;36:36 retirement _n_of;0:0 Individuals _n_1', "0:0 People _n_of;1:1 also _a_1;2:2 shouldn't _v_modal;3:3 invest _v_in;4:6 _n_1 money stocks;10:10 need _v_1;11:11 in _p;12:12 the _q;13:13 near _a_to;14:14 future _n_1;16:17 for example _a_1;18:18 for _p;19:19 college _n_of;20:20 tuition _n_1;21:21 payments _n_of;22:22 or _c;23:23 retirement _n_of;0:0 People _n_of", '1:1 may _v_modal;2:2 have _v_qmodal;4:4 sell _v_1;6:6 stocks _n_1;7:7 at _p_temp;8:8 a _q;9:9 time _n_of;11:11 the _q;12:12 market _n_1;13:13 takes _v_1;14:14 a _q;15:15 plunge," _n_of;16:16 says _v_to;17:17 Mr _n_1;18:18 Blankenship named;19:19 a _q;20:20 Del named;21:21 Mar mofy;22:22 Calif named;23:23 financial _a_1;1:1 may _v_modal', '0:0 But _c;1:1 once _x_subord;2:2 the _q;3:3 basics _n_1;5:5 covered _v_1;6:6 "then _a_1;8:8 would _v_modal;9:11 _v_1 start invest;12:12 even _x_deg;13:13 if _x_then;15:15 \'s _v_id;19:19 $ _n_1;20:20 1,000," card;21:21 says _v_to;22:23 named Michael Lipper;24:24 president _n_of;26:27 named Lipper Analytical;28:28 Services named;0:0 But _c', '1:1 says _v_to;2:2 individuals _n_1;3:3 should _v_modal;4:4 consider _v_1;5:6 not just _x_deg;7:7 stocks _n_1;8:8 but _c;9:9 other _a_1;11:11 term _n_of;11:11 term _long_a_1;12:12 investments _n_1;13:14 such as _p;16:16 quality _n_1;16:16 quality _high_a_1;1:1 says _v_to', "0:0 Despite _p;1:1 the _q;2:2 strong _a_1;3:3 case _n_of;4:4 for _p;5:5 stocks _n_1;7:7 most _q;8:8 pros _n_1;9:9 warn _v_1;11:11 individuals _n_1;12:12 shouldn't _v_modal;13:13 try _v_1;15:15 profit _v_from;18:18 term _n_of;18:18 term _short_a_of;0:0 Despite _p", '2:2 very _x_deg;3:3 difficult _a_for;5:5 do," _v_1;6:6 says _v_to;7:8 named Donald Holt;9:9 a _q;10:10 market _n_1;11:11 strategist _n_unknown;12:12 for _p;13:14 named Wedbush Morgan;15:15 Securities named;16:16 a _q;17:18 named Los Angeles;19:19 brokerage _n_unknown;2:2 very _x_deg', "1:1 markets _n_1;2:2 move _v_1;4:4 fast _a_1;5:5 and _c;9:9 volatile _a_1;11:11 's _v_there;12:12 no _q;13:13 way _n_of;14:14 the _q;15:15 average _a_1;16:16 investor _n_1;17:17 can _v_modal;18:18 compete _v_1;19:19 with _p;20:20 the _q;1:1 markets _n_1", '0:0 Individual _a_1;1:1 investors _n_1;2:2 face _v_1;3:3 high _a_1;4:5 _n_1 transaction costs;6:6 of _p;7:7 moving _v_1;8:8 in _p;9:9 and _c;10:11 out of _p;12:12 the _q;0:0 Individual _a_1', '0:0 The _q;1:1 cost _n_1;2:2 of _p;3:3 executing _v_1;4:4 stock _n_1;5:5 orders _n_of;6:6 varies _v_1;7:7 from _p_time;8:8 brokerage _n_unknown;9:9 to _p;10:10 brokerage _n_unknown;11:11 and _c;12:12 with _p;13:13 the _q;14:14 size _n_1;15:15 of _p;16:16 the _q;17:17 order _n_of;18:18 but _c;19:19 2 card;20:20 % _n_of;21:21 of _p;22:22 the _q;23:23 order _n_of;25:25 value _n_1;26:26 is _v_id;27:27 an _q;28:28 average _n_of;29:29 says _v_to;30:31 named Stephen Boesel;32:32 manager _n_of;34:35 named T Rowe;36:38 named Price Growth;39:39 and _c;40:40 Income named;41:41 mutual _a_1;0:0 The _q', '0:0 And _c;1:1 assuming _v_1;3:3 first ord;4:4 investment _n_1;6:6 successful _a_1;7:7 investors _n_1;9:9 have _v_qmodal;11:11 pay _v_for;12:12 taxes _n_1;13:13 on _p;0:0 And _c', '0:0 That _q_dem;1:1 can _v_modal;2:2 reduce _v_1;3:3 returns _n_of;4:4 by _p;5:5 a card;6:6 third _n_of;7:7 or _c;9:9 once _x_subord;10:10 local _a_1;11:11 taxes _n_1;13:13 included _v_1;14:14 Mr _n_1;15:15 Lipper named;0:0 That _q_dem', '0:0 After _p;1:1 that _q_dem;2:2 individual _a_1;3:3 traders _n_1;4:4 face _v_1;5:5 the _q;6:6 risk _n_of;8:8 the _q;9:9 new _a_1;10:10 investment _n_1;12:12 choose _v_from;14:14 perform _v_1;15:15 well _a_1;17:17 so _x;19:19 trading _v_1;20:20 costs _n_1;21:21 could _v_modal;23:23 sustained _v_1;24:24 for _p;0:0 After _p', '1:1 should _v_modal;2:2 really _a_1;3:3 think _v_1;4:4 twice _a_1;5:5 if _x_then;7:9 _v_1 think can;11:11 smart named;11:11 smart _p;12:12 the _q;1:1 should _v_modal', '0:0 Then _p_temp;1:1 too _a_also;3:3 individual _a_1;4:4 investors _n_1;5:5 lack _v_1;6:6 the _q;7:8 _a_1 sturdy emotional;9:10 _n_1 makeup professionals;11:11 say _v_to;13:13 needed _v_1;14:14 to _x;15:15 plunge _v_into;16:16 in _p;17:17 and _c;18:19 out of _p;20:20 the _q;0:0 Then _p_temp', "0:0 So _a_thus;2:2 's _v_id;3:3 the _q;4:4 best _a_at-for-of;5:5 way _n_of;7:7 buy _v_1;0:0 So _a_thus", '0:0 "Unless _x;1:1 an _q;2:2 individual _n_1;3:3 has _v_1;4:4 a _q;5:5 minimum _n_of;8:8 $ _n_1;9:9 50,000 card;10:10 and _c_btwn;11:11 $ _n_1;12:12 100,000 card;14:14 invest _v_in;16:16 stocks _n_1;19:20 _a_1 still better off;22:22 in _p;23:23 mutual _a_1;24:24 funds _n_1;26:26 in _p;27:27 individual _a_1;28:28 stocks _n_1;29:29 in _p;30:30 terms _n_of;32:32 getting _v_1;33:33 enough _q;34:34 attention _n_to;35:35 from _p;36:36 a _q;37:37 competent _a_1;38:38 broker," _n_1;39:39 says _v_to;40:40 Mr _n_1;0:0 "Unless _x', '0:0 Still _a_disc;2:2 adds _v_to;4:4 could _v_modal;5:6 _v_1 see owning;7:7 both _q;7:7 both card;8:9 given that _x_subord;10:10 individuals _n_1;11:11 often _a_1;12:12 have _v_1;13:13 an _q;14:14 advantage _n_1;15:15 over _p;16:16 big _a_1;17:17 investors _n_1;18:18 in _p;19:19 spotting _v_1;20:20 special _a_1;21:21 situations _n_1;22:22 based _v_1;23:23 on _p;25:25 own _a_1;26:26 insights," _n_into;0:0 Still _a_disc', '0:0 This _q_dem;1:1 growth _n_of;2:2 sector _n_1;4:4 usually _a_1;5:5 carries _v_1;6:6 a _q;7:7 price _n_of;9:10 _n_1 earnings multiple;11:11 about _p;12:12 twice _q;13:13 that _q_dem;14:14 of _p;15:15 the _q;16:16 Standard named;17:17 & _c;18:18 Poor named;20:20 500 card;21:23 _v_1 happens include;24:26 _q some the;27:27 market _n_1;30:30 attractive _a_1;31:31 bargains _n_1;32:32 right _x_deg;0:0 This _q_dem', '2:2 now _a_1;3:3 selling _v_1;4:4 at _p;5:5 a _q;6:6 multiple _n_1;7:7 about _x_deg;8:8 even _a_1;9:9 with _p;10:10 the _q;11:11 market," _n_1;12:12 says _v_to;13:13 Mr _n_1;2:2 now _a_1', '0:0 Moreover _a_1;1:1 Mr _n_1;2:2 Douglas named;3:3 sees _v_1;4:4 a _q;5:5 revival _n_of;7:7 institutional _a_1;8:8 interest _n_in;10:10 smaller _a_1;11:11 growth _n_of;12:12 stocks _n_1;14:14 could _v_modal;15:15 boost _v_to;16:16 the _q;17:17 performance _n_1;18:18 of _p;19:19 these _q_dem;20:20 stocks _n_1;21:21 in _p;22:22 the _q;23:23 medium _a_1;0:0 Moreover _a_1', '1:1 big _a_1;2:3 named Wall Street;4:4 brokerage _n_unknown;5:5 firms _n_1;7:7 eliminated _v_from;9:10 _n_1 research effort;11:11 in _p;12:12 stocks _n_1;13:13 of _p;14:14 emerging _v_1;15:16 _n_of growth companies;17:18 a few _a_1;19:19 years _n_1;20:20 ago _p;22:22 now _a_1;23:23 resuming _v_1;24:24 coverage _n_of;26:26 this _q_dem;27:27 area _n_of;1:1 big _a_1', '2:2 seeing _v_1;3:3 a _q;4:4 real _a_1;5:5 turnaround _n_1;6:6 in _p;7:7 interest _n_in;9:9 small _a_1;10:10 growth _n_of;11:11 stocks," _n_1;2:2 seeing _v_1', '0:0 The _q;1:1 pros _n_1;2:2 strenuously _a_1;3:3 advise _v_1;4:4 individuals _n_1;6:6 stay _v_prd;7:8 _p away from;9:9 the _q;10:10 latest _a_for;11:11 investment _n_1;0:0 The _q', "1:1 say _v_to;2:2 that _q_dem;4:4 especially _a_1;5:5 important _a_for;6:6 this _x_deg;7:8 _p late in;9:9 the _q;10:10 growth _n_of;11:11 phase _n_1;12:12 of _p;13:13 the _q;14:14 economic _a_1;15:15 cycle _n_1;16:16 when _x_subord;18:18 's _v_there;19:19 no _q;20:20 robust _a_1;21:22 _n_1 bull market;24:24 bail _v_out+of;25:29 _n_1 investors mistakes", '0:0 "Buy _v_1;1:1 stocks _n_1;2:2 on _p;3:3 weakness _n_1;4:4 for _p;7:7 term _n_of;7:7 term _long_a_1;8:8 fundamentals," _n_1;0:0 "Buy _v_1', '0:0 In _p_state;1:1 the _q;2:2 long _a_1;3:3 run _n_of;4:5 _n_1 investment advisers;6:6 say _v_to;7:7 most _q;8:8 investors _n_1;11:12 better off _a_1;13:13 using _v_1;14:14 the _q;15:16 _n_1 dollar- cost;17:17 averaging _v_1;18:18 method _n_of;20:20 buying _v_1;0:0 In _p_state', '0:0 In _p_state;1:1 this _q_dem;2:2 method _n_of;3:3 a _q;4:4 person _n_1;5:5 invests _v_1;6:6 a _q;7:7 regular _a_1;8:8 amount _n_of;9:9 every _q;10:10 month _n_1;11:11 or _c;12:12 quarter _n_temp;13:13 into _p;14:14 the _q;15:16 _n_1 stock market;17:17 whether _x;18:18 the _q;19:19 market _n_1;21:21 up _p;22:22 or _c;0:0 In _p_state', '0:0 That _q_dem;1:1 cuts _v_1;2:2 the _q;3:3 risk _n_of;4:4 Mr _n_1;5:5 Gregory named;6:6 the _q;7:8 named San Francisco;9:9 money _n_1;10:10 manager _n_of;0:0 That _q_dem', '0:0 "When _x_subord;1:1 the _q;2:2 market _n_1;4:4 low _a_on;7:7 buying _v_1;9:9 shares _n_of;10:10 and _c;11:11 when _x_subord;14:14 high _a_1;17:17 buying _v_1;18:18 fewer _a_1;19:19 shares," _n_of;0:0 "When _x_subord', '0:0 Otherwise _a_1;1:1 if _x_then;3:3 put _v_in;4:4 all _q;6:6 money _n_1;8:8 at _p_temp;9:9 one card;10:10 time _n_of;11:11 by _p_state;12:12 sheer _a_1;13:13 bad _a_at;14:14 luck _n_1;16:16 might _v_modal;17:17 pick _v_1;18:18 a _q;19:19 terrible _a_for;20:20 time _n_of;21:21 and _c;22:22 have _v_qmodal;24:24 wait _v_1;25:25 three card;26:26 years _n_1;27:27 to _x;28:28 get _v_state;29:29 even _a_1;30:30 Mr _n_1;31:31 Gregory named;0:0 Otherwise _a_1', '0:0 A _q;1:1 disciplined _v_1;2:2 program _n_of;4:4 work _v_1;5:6 the best _best_a_1;7:7 Mr _n_1;8:8 Boesel named;0:0 A _q', '0:0 "One card;2:2 the _q;3:3 hardest _a_for;4:4 things _n_of-about;6:9 _v_1 do buy;10:10 stocks _n_1;11:11 when _x_subord;12:12 the _q;13:13 market _n_1;15:15 down," _p;0:0 "One card', '0:0 "But _c;1:1 that _q_dem;2:2 \'s _v_id;3:3 just _a_1;4:4 the _q;5:5 time _n_of;8:8 should _v_modal;0:0 "But _c', '0:0 Compound _n_1;1:1 annual _a_1;2:2 returns _n_of;3:3 including _v_1;4:5 _n_of price changes;6:6 and _c;7:7 income _n_1;8:8 from _p;9:9 interest _n_in;10:10 and _c;0:0 Compound _n_1', '0:0 *Actual _a_1;1:1 performance _n_1;0:0 *Actual _a_1', '0:0 Source _n_2;1:1 : _v_id;2:3 named Ibbotson Associates;0:0 Source _n_2', '0:1 named Inefficient- Market;2:2 Fund _n_1;3:3 Inc named;4:4 initial _a_1;5:5 offering _n_of;7:8 card five million;9:9 common _a_for;10:10 shares _n_of;11:11 via _p;12:13 named Smith Barney;14:15 named Harris Upham;0:0 Inefficient- named', '0:1 named Donald Trump;3:4 _v_1 faced rising;5:5 doubt _n_1;6:6 about _p;8:8 bid _n_1;9:9 for _p;10:11 named American Airlines;12:12 parent _n_of;13:13 AMR named;14:14 Corp _n_1;15:15 even _x_deg;16:16 before _x_h;17:17 a _q;18:19 named United Airlines;20:21 buy- out _n_1;22:22 came _v_1;23:23 apart _a_1;24:24 Friday dofw;25:25 withdrew _v_1;27:27 $ _n_1;28:29 card 7.54 billion;0:0 Donald named', '0:0 Separately _a_from;1:1 bankers _n_1;2:2 representing _v_1;3:3 the _q;4:4 group _n_of;5:7 _v_1 trying buy;8:8 United named;10:10 parent _n_of;11:11 UAL named;12:12 Corp _n_1;13:13 met _v_1;14:14 with _p;15:15 other _a_1;16:16 banks _n_of;17:17 about _p;18:18 reviving _v_cause;19:19 that _q_dem;20:20 purchase _n_of;21:21 at _p;22:22 a _q;23:23 lower _a_1;24:24 price _n_of;25:25 possibly _x_deg;26:26 around _p;27:27 $ _n_1;28:28 250 card;29:29 a _p_per;30:30 share _n_of;31:31 or _c;32:32 $ _n_1;33:34 card 5.65 billion', '0:0 But _c;1:1 a _q;2:2 lower _a_1;3:3 bid _n_1;4:4 could _v_modal;5:5 face _v_1;6:6 rejection _n_1;7:7 by _p;8:8 the _q;9:9 UAL named;0:0 But _c', '0:0 Mr _n_1;1:1 Trump named;3:3 vowed _v_to;4:4 Wednesday dofw;6:6 "go _v_1;7:7 forward" _p_dir;8:8 with _p;9:9 the _q;10:10 bid _n_1;11:11 said _v_to;14:14 dropping _v_cause;16:16 "in _p;17:17 light _n_1;18:18 of _p;19:19 the _q;20:20 recent _a_1;21:21 change _n_of;22:22 in _p;23:24 _n_1 market conditions."', '1:1 said _v_to;3:3 might _v_modal;4:4 now _a_1;5:5 sell _v_1;7:7 AMR named;8:8 stake _n_1;9:9 buy _v_1;11:11 shares _n_of;12:12 or _c;13:13 make _v_1;14:14 another _q;15:15 offer _n_1;16:16 at _p;17:17 a _q;18:18 lower _a_1;1:1 said _v_to', '0:0 News _n_of;1:1 about _p;2:2 UAL named;3:3 and _c;4:4 AMR named;6:6 shares _n_of;7:7 never _a_1;8:8 reopened _v_1;8:8 reopened _re-_a_again;9:9 after _x_h;10:12 _v_1 trading halted;13:13 Friday dofw;14:14 for _p;15:15 the _q;16:16 UAL named;17:17 announcement _n_of;18:18 sent _v_1;19:19 both _q;19:19 both card;20:20 stocks _n_1;21:21 nosediving _v_1;22:22 in _p;23:23 composite _n_1;24:24 trading _v_1;25:25 on _p;26:26 the _q;27:28 named New York;29:30 _n_1 Stock Exchange', '0:0 UAL named;1:1 tumbled _v_1;2:2 $ _n_1;3:3 56.875 card;4:4 to _p;5:5 $ _n_1;6:6 222.875 card;7:7 on _p;8:8 volume _n_of;10:11 card 2.3 million;12:12 shares _n_of;13:13 and _c;14:14 AMR named;15:15 declined _v_1;16:16 by _p;17:17 $ _n_1;18:18 22.125 card;19:19 to _p;20:20 $ _n_1;21:21 76.50 card;22:22 as _x_subord;23:24 card 4.7 million;25:25 shares _n_of;26:26 changed _v_cause;0:0 UAL named', '0:0 Together _p_state;1:1 the _q;2:2 two card;3:3 stocks _n_1;4:4 wreaked _v_unknown;5:5 havoc _n_unknown;6:6 among _p;7:7 takeover _n_of;8:9 _n_1 stock traders;10:10 and _c;11:11 caused _v_1;12:12 a _q;13:13 7.3 card;14:15 _n_of % drop;16:16 in _p;17:17 the _q;18:19 named Dow Jones;20:21 named Transportation Average;22:22 second ord;23:23 in _p;24:24 size _n_1;25:25 only _x_deg;26:26 to _p;27:27 the _q;28:29 _n_1 stock- market;30:30 crash _n_1;31:31 of _p;32:32 Oct mofy;33:33 19 dofm;0:0 Together _p_state', '0:0 Some _q;1:1 said _v_to;2:2 Friday dofw;4:4 market _n_1;5:5 debacle _n_unknown;7:7 given _v_1;8:8 Mr _n_1;9:9 Trump named;10:10 an _q;11:11 excuse _n_1;13:13 bail _v_1;14:15 out of _p;16:16 an _q;17:17 offer _n_1;19:19 showed _v_1;20:20 signs _n_of;22:22 stalling _v_1;23:23 even _x_deg;24:24 before _x_h;25:25 problems _n_of;26:26 emerged _v_1;27:27 with _p;28:28 the _q;29:29 UAL named;0:0 Some _q', '0:0 After _p;1:1 reaching _v_1;2:2 an _q;3:3 intraday _n_unknown;4:4 high _n_1;5:5 of _p;6:6 $ _n_1;7:7 107.50 card;8:8 the _q;9:9 day _n_of;10:10 Mr _n_1;11:11 Trump named;12:12 disclosed _v_to;14:14 bid _n_1;15:15 Oct mofy;16:16 5 dofm;17:17 AMR named;19:19 stock _n_1;21:21 retreated _v_1;23:23 low _a_1;25:25 $ _n_1;26:26 97.75 card;27:27 last _a_1;0:0 After _p', '0:0 Some _q;1:1 takeover _n_of;2:3 _n_1 stock traders;6:6 betting _v_1;7:7 against _p;8:8 Mr _n_1;9:9 Trump named;10:10 because _x;12:12 has _v_1;13:13 a _q;14:14 record _n_of;16:16 disclosing _v_to;17:17 stakes _n_1;18:18 in _p;19:19 companies _n_of;21:21 are _v_id;22:22 potential _a_1;23:24 _n_of takeover targets;25:25 then _c;26:26 selling _v_1;27:27 at _p;28:28 a _q;29:29 profit _n_1;30:30 without _p;31:31 making _v_1;32:32 a _q;0:0 Some _q', '1:1 still _a_1;3:3 proven _v_to;5:5 mettle _n_1;6:6 as _p;7:7 a _q;9:9 league _n_of;9:9 league _big_a_1;10:12 _n_1 take- out artist,";13:13 said _v_to;14:15 _n_1 airline analyst;16:17 named Kevin Murphy;18:18 of _p;19:20 named Morgan Stanley;1:1 still _a_1', '2:2 done _v_1;3:3 this _q_dem;4:4 thing _n_of-about;8:8 buy _v_1;9:9 a _q;10:10 little _a_1;11:11 bit _n_of;13:13 a _q;14:14 company _n_of;15:16 and then _c;17:17 trade _v_1;2:2 done _v_1', '2:2 written _v_to;3:3 this _q_dem;4:4 book _n_of;5:6 named `The Art;7:7 of _p;8:8 the _q;2:2 written _v_to', '0:0 Mr _n_1;1:1 Trump named;2:2 withdrew _v_1;4:4 bid _n_1;5:5 before _x_h;6:6 the _q;7:7 AMR named;8:8 board _n_of;11:11 due _a_1;13:13 meet _v_1;14:15 _a_1 tomorrow ever;16:16 formally _a_1;0:0 Mr _n_1', '0:0 AMR named;2:2 weighed _v_1;3:3 a _q;4:4 wide _a_1;5:5 range _n_of;7:7 possible _a_for;8:8 responses _n_to;9:9 from _p_time;10:10 flat _a_1;11:11 rejection _n_1;12:12 to _p;13:13 recapitalizations _n_unknown;14:14 and _c;15:15 leveraged _v_1;16:17 buy- outs _n_1;19:19 might _v_modal;21:21 included _v_1;23:23 employees _n_of;24:24 a _q;25:25 friendlier _a_unknown;26:26 buyer _n_of;27:28 such as _p;29:29 Texas named;30:30 billionaire _n_1;31:32 named Robert Bass;33:33 or _c;34:34 both _q;0:0 AMR named', '0:0 AMR named;2:2 also _a_1;3:5 _v_1 sought foil;6:6 Mr _n_1;7:7 Trump named;8:8 in _p;9:9 Congress named;10:10 by _p_means;11:11 lobbying _v_1;12:12 for _p;13:13 legislation _n_1;15:15 would _v_modal;17:17 bolstered _v_1;18:18 the _q;19:19 authority _n_1;20:20 of _p;21:21 the _q;22:23 named Transportation Department;25:25 reject _v_as;26:27 _n_1 airline buy- outs', '0:0 Yesterday _a_1;1:1 Mr _n_1;2:2 Trump named;3:5 _v_1 tried put;6:6 the _q;7:7 blame _n_1;8:8 for _p;9:9 the _q;10:10 collapse _n_1;11:11 of _p;12:12 the _q;13:13 UAL named;14:14 deal _n_1;15:15 on _p;16:16 Congress named;17:17 saying _v_to;20:20 rushing _v_1;21:21 through _p;22:22 a _q;23:23 bill _n_of;25:25 protect _v_1;26:26 AMR named;0:0 Yesterday _a_1', '1:1 believe _v_1;3:3 the _q;4:6 _n_1 perception legislation;7:7 in _p;8:8 this _q_dem;9:9 area _n_of;10:10 may _v_modal;12:12 hastily _a_1;13:13 approved _v_1;14:14 contributed _v_to;16:16 the _q;17:17 collapse _n_1;18:18 of _p;19:19 the _q;20:20 UAL named;21:21 transaction _n_1;22:22 and _c;23:23 the _q;24:24 resulting _v_1;25:25 disruption _n_of;26:26 in _p;27:27 the _q;28:28 financial _a_1;29:29 markets _n_1;30:30 experienced _v_1;31:31 this _q_dem;32:32 past _a_1;33:33 Friday," dofw;34:34 Mr _n_1;35:35 Trump named;36:36 wrote _v_to;37:37 members _n_of;1:1 believe _v_1', '0:0 AMR named;1:1 declined _v_1;3:3 comment _v_on;4:4 and _c;5:5 Mr _n_1;6:6 Trump named;8:8 respond _v_to;10:10 requests _n_for;0:0 AMR named', '0:2 _a_1 However only;3:3 received _v_1;4:4 federal _a_1;5:5 clearance _n_1;7:7 buy _v_1;10:10 $ _n_1;11:12 card 15 million;13:13 of _p;14:14 the _q;15:15 stock _n_1;16:16 on _p_temp;17:17 Sept mofy;18:18 20 dofm;19:19 when _x_subord;20:20 the _q;21:21 price _n_of;22:22 rose _v_1;23:23 $ _n_1;24:24 2 card;25:25 a _p_per;26:26 share _n_of;27:27 to _p;28:28 $ _n_1;0:0 However _a_1', '0:0 Between _p;1:1 then _p_temp;2:2 and _c;4:4 bid _n_1;5:5 on _p_temp;6:6 Oct mofy;7:7 5 dofm;8:8 the _q;9:9 price _n_of;10:10 fluctuated _v_1;11:11 between _p;12:12 $ _n_1;13:13 75.625 card;14:14 and _c;15:15 $ _n_1;0:0 Between _p', '0:0 In _p_state;1:1 an _q;2:2 attempt _n_1;4:4 persuade _v_of;5:8 _n_1 investors bid;9:9 wasn\'t _v_id;10:10 just _x_deg;11:11 "a _q;12:13 _n_1 stock play,";14:14 Mr _n_1;15:15 Trump named;16:16 promised _v_1;17:17 last _a_1;18:18 week _n_1;20:20 notify _v_of;21:21 the _q;22:22 market _n_1;23:23 before _p;24:24 selling _v_1;25:25 any _q;0:0 In _p_state', '0:0 AMR named;2:2 trading _v_1;3:3 at _p;4:4 around _x_deg;5:5 $ _n_1;6:6 84 card;7:7 yesterday _a_1;8:8 before _p;10:11 _n_of withdrawal announcement;12:12 then _c;13:13 immediately _a_1;14:14 fell _v_1;15:15 to _p;16:16 about _x_deg;17:17 $ _n_1;0:0 AMR named', '0:0 Assuming _v_1;3:3 paid _v_for;4:4 a _q;5:6 _a_1 rough average;7:7 price _n_of;9:9 $ _n_1;10:10 80 card;11:11 a _p_per;12:12 share _n_of;13:13 and _c;14:17 _v_1 assuming sell;18:18 before _x_h;20:20 announcement _n_of;21:21 reached _v_1;22:22 the _q;23:24 _n_1 market Mr;25:25 Trump named;26:26 could _v_modal;28:28 sitting _v_1;29:29 with _p;30:30 a _q;31:31 modest _a_1;32:32 loss _n_of;33:33 with _p;34:34 the _q;35:35 stock _n_1;36:36 at _p;37:37 $ _n_1;0:0 Assuming _v_1', '0:0 Some _q;1:1 analysts _n_1;2:2 said _v_to;3:3 AMR named;4:4 Chairman _n_of;5:6 named Robert Crandall;7:7 might _v_modal;8:8 seize _v_1;9:9 the _q;10:10 opportunity _n_1;11:11 presented _v_to;13:13 the _q;14:14 stock _n_1;15:16 _n_of price drop;17:17 to _x;18:18 protect _v_1;19:19 the _q;20:20 nation _n_of;22:22 largest _a_1;23:23 airline _n_1;24:24 with _p;25:25 a _q;26:26 defensive _a_1;27:27 transaction _n_1;28:29 such as _p;30:30 the _q;31:31 sale _n_of;33:33 stock _n_1;34:34 to _p;35:35 a _q;36:36 friendly _a_to;37:37 holder _n_of;38:38 or _c;39:40 _n_of company employees', "0:0 Some _q;1:1 analysts _n_1;2:2 said _v_to;4:4 believed _v_1;5:5 Mr _n_1;6:6 Trump named;8:8 towering _v_1;9:9 ego _n_1;12:12 viewed _v_as;14:16 _q some a;17:17 reason _n_for;19:19 believe _v_1;21:21 wouldn't _v_modal;22:22 back _v_out;24:24 might _v_modal;25:25 come _v_back;27:27 with _p;28:28 a _q;29:29 lower _a_1;0:0 Some _q", '0:1 named Ray Neidl;2:2 of _p;3:4 named Dillon Read;5:6 & Co _n_1;7:7 said _v_to;8:8 Mr _n_1;9:9 Trump named;11:11 stepping _v_1;12:12 back _p;13:13 and _c;14:14 waiting _v_1;16:16 the _q;17:17 dust _n_1;0:0 Ray named', '2:2 sure _a_of;4:4 still _a_1;5:5 wants _v_1;2:2 sure _a_of', '0:0 But _c;1:1 others _n_1;2:2 remained _v_1;0:0 But _c', '2:2 never _a_1;3:3 sure _a_of;4:5 named Donald Trump;6:6 really _a_1;7:9 _v_1 wanted take;10:10 AMR," named;11:11 said _v_to;12:13 named John Mattis;14:14 a _q;15:16 _n_1 bond analyst;17:17 with _p;18:19 named Shearson Lehman;20:20 Hutton named;2:2 never _a_1', '1:1 happened _v_1;2:2 with _p;3:3 United named;4:4 was _v_id;5:5 a _q;6:6 gracious _a_1;7:7 way _n_of;1:1 happened _v_1', '0:0 Mr _n_1;1:1 Trump named;2:2 never _a_1;3:4 _v_1 obtained financing;5:5 for _p;0:0 Mr _n_1', '0:0 Meanwhile _p;1:1 Citicorp named;2:2 and _c;3:4 named Chase Manhattan;5:5 Corp _n_1;6:6 the _q;7:7 two card;8:8 lead _n_1;9:9 lenders _n_unknown;10:10 on _p;11:11 the _q;12:12 UAL named;13:14 buy- out _n_1;15:15 met _v_1;16:16 with _p;17:17 other _a_1;18:18 banks _n_of;19:19 yesterday _a_1;20:20 to _x;21:21 determine _v_1;24:24 would _v_modal;26:26 willing _a_1;28:28 finance _v_1;29:29 the _q;30:31 buy- out _n_1;32:32 at _p;33:33 a _q;34:34 lower _a_1;0:0 Meanwhile _p', '0:0 Officials _n_1;1:1 familiar _a_with;3:3 the _q;4:4 talks _n_of-on;5:5 said _v_to;6:6 Citicorp named;8:8 discussed _v_1;9:9 lowering _v_cause;10:10 the _q;11:11 offer _n_1;12:12 to _p;13:13 $ _n_1;14:14 250 card;15:15 a _p_per;16:16 share _n_of;17:17 but _c;18:18 said _v_to;20:20 price _n_of;21:21 was _v_id;22:22 a _q;23:23 talking _v_about;24:24 point _n_of;25:25 and _c;27:27 no _q;28:28 decision _n_1;0:0 Officials _n_1', '0:0 At _p;1:1 $ _n_1;2:2 250 card;3:3 a _p_per;4:4 share _n_of;5:5 the _q;6:6 group _n_of;7:7 would _v_modal;8:8 have _v_qmodal;10:10 borrow _v_from;11:11 about _x_deg;12:12 $ _n_1;13:14 card 6.1 billion;15:15 from _p;0:0 At _p', "0:0 The _q;1:1 first ord;2:2 UAL named;3:3 deal _n_1;4:4 unraveled _v_1;5:5 after _x_h;6:6 Citibank named;7:7 and _c;8:8 Chase named;9:9 couldn't _v_modal;10:10 raise _v_cause;11:11 $ _n_1;12:13 card 7.2 billion", '0:0 Citibank named;1:1 and _c;2:2 Chase named;4:4 agreed _v_1;6:6 commit _v_to;7:7 $ _n_1;8:9 card 3 billion;10:10 and _c;11:11 said _v_to;14:14 "highly _a_1;15:15 confident" _a_of-about;17:17 raising _v_cause;18:18 another _x_deg;19:19 $ _n_1;20:21 card 4.2 billion', '0:0 Together _p_state;1:1 Citicorp named;2:2 and _c;3:3 Chase named;4:4 received _v_1;5:5 $ _n_1;6:7 card 8 million;8:8 in _p;9:9 fees _n_1;10:10 to _x;11:11 raise _v_cause;12:12 the _q;13:13 rest _n_of;15:15 the _q;0:0 Together _p_state', '0:0 But _c;1:1 other _a_1;2:2 banks _n_of;3:3 balked _v_at;5:5 the _q;6:6 low _a_on;7:7 interest _n_in;8:8 rate _n_of;9:9 and _c;10:11 _n_1 banking fees;12:12 the _q;13:13 UAL named;14:14 group _n_of;16:16 willing _a_1;0:0 But _c', '0:0 Nor _x_subord;2:2 the _q;3:3 group _n_of;4:4 likely _a_1;6:6 come _v_1;7:7 forward _p_dir;8:8 with _p;9:9 a _q;10:10 revised _v_1;11:11 offer _n_1;12:12 within _p;13:13 the _q;14:14 next _a_1;15:15 48 card;16:16 hours _n_1;17:17 despite _p;18:18 the _q;19:19 hopes _n_1;20:20 of _p;0:0 Nor _x_subord', '0:0 The _q;1:1 group _n_of;3:3 advisers _n_1;4:4 want _v_1;6:6 make _v_i;7:7 certain _a_of;9:9 have _v_1;10:10 firm _a_1;11:11 bank _n_of;12:12 commitments _n_to;13:13 the _q;14:14 second ord;15:15 time _n_of;0:0 The _q', '0:0 Even _x_deg;1:1 if _x_then;2:2 the _q;3:4 buy- out _n_1;5:5 group _n_of;7:7 able _a_1;9:10 _v_1 obtain financing;11:11 the _q;12:12 transaction _n_1;13:13 still _a_1;14:14 faces _v_1;0:0 Even _x_deg', "0:0 UAL named;2:2 board _n_of;3:3 could _v_modal;4:4 reject _v_as;5:5 the _q;6:6 new _a_1;7:7 price _n_of;10:10 low _a_on;11:11 especially _x_deg;12:12 since _x_subord;14:14 aren't _v_there;15:15 any _q;16:16 competing _v_1;0:0 UAL named", '0:1 named Los Angeles;2:2 investor _n_1;3:4 named Marvin Davis;6:6 $ _n_1;7:7 275- card;9:9 share _n_of;9:9 share _a_p_per;10:10 offer _n_1;12:12 rejected _v_as;14:14 UAL named;16:16 board _n_of;18:18 shown _v_1;19:19 signs _n_of;21:21 pursuing _v_1;22:22 a _q;23:23 $ _n_1;24:24 300- card;26:26 share _n_of;26:26 share _a_p_per;27:28 back- up _backup_n_1;29:29 bid _n_1;31:31 made _v_1;32:32 last _a_1;0:0 Los named', '0:0 In _p_state;1:1 addition _n_1;2:2 the _q;3:3 coalition _n_1;4:4 of _p;5:5 labor _n_1;6:6 and _c;7:7 management _n_1;8:8 longtime _a_unknown;9:9 enemies _n_1;11:11 joined _v_1;12:12 forces _n_1;13:13 only _x_deg;14:14 under _p;15:15 the _q;16:16 threat _n_1;17:17 of _p;18:18 Mr _n_1;19:19 Davis named;21:21 bid _n_1;22:22 could _v_modal;23:23 break _v_1;24:25 _a_1 apart now', '0:0 The _q;1:1 group _n_of;3:3 resilience _n_unknown;4:4 gets _v_1;6:6 first ord;7:7 test _n_of;8:8 today _a_1;9:9 when _x_subord;10:10 30 card;11:11 top _a_1;12:13 _n_of pilot union;14:14 leaders _n_of;15:15 convene _v_1;16:16 outside _p;17:17 Chicago named;18:18 in _p;19:19 a _q;20:20 previously _p;21:21 scheduled _v_1;0:0 The _q', '0:0 The _q;1:2 _n_of pilot union;4:4 vowing _v_1;5:5 to _x;6:6 pursue _v_1;7:7 an _q;8:8 acquisition _n_of;10:10 the _q;11:11 board _n_of;0:0 The _q', '0:0 But _c;1:1 if _x_then;2:2 the _q;3:3 board _n_of;4:4 rejects _v_as;5:5 a _q;6:6 reduced _v_1;7:7 bid _n_1;8:8 and _c;9:11 _v_1 decides explore;12:12 other _a_1;13:13 alternatives _n_to;15:15 could _v_modal;16:16 transform _v_into;19:19 been _v_id;20:20 a _q;21:21 harmonious _a_1;22:22 process _n_of;24:24 an _q;25:25 adversarial _a_unknown;0:0 But _c', '0:0 The _q;1:1 pilots _n_of;2:2 could _v_modal;3:3 play _v_1;4:4 hardball _n_unknown;5:5 by _p_means;6:6 noting _v_to;9:9 crucial _a_for;10:10 to _p;11:11 any _q;12:12 sale _n_of;13:13 or _c;14:14 restructuring _v_1;15:15 because _x;17:17 can _v_modal;18:18 refuse _v_1;20:20 fly _v_to;21:21 the _q;0:0 The _q', "0:0 If _x_then;4:4 insist _v_on;6:6 a _q;7:7 low _a_on;8:8 bid _n_1;9:9 of _p;11:11 $ _n_1;12:12 200 card;13:13 a _p_per;14:14 share _n_of;15:15 the _q;16:16 board _n_of;17:17 mightn't _v_modal;19:19 able _a_1;21:21 obtain _v_1;22:22 a _q;23:23 higher _a_1;24:24 offer _n_1;25:25 from _p;26:26 other _a_1;27:27 bidders _n_1;28:28 because _x;29:29 banks _n_of;30:30 might _v_modal;31:33 _v_1 hesitate finance;34:34 a _q;35:35 transaction _n_1;36:36 the _q;37:37 pilots _n_of;0:0 If _x_then", '0:0 Also _a_1;1:1 because _x;2:2 UAL named;3:3 Chairman _n_of;4:5 named Stephen Wolf;6:6 and _c;7:7 other _a_1;8:8 UAL named;9:9 executives _n_1;11:11 joined _v_1;12:12 the _q;13:13 pilots _n_of;15:15 bid _n_1;16:16 the _q;17:17 board _n_of;18:18 might _v_modal;20:22 _v_1 forced exclude;24:24 from _p;26:26 deliberations _n_1;27:28 in order _in+order+to_x;31:31 fair _a_1;32:32 to _p;33:33 other _a_1;0:0 Also _a_1', '0:0 That _q_dem;1:1 could _v_modal;2:2 cost _v_1;4:4 the _q;5:5 chance _n_of;7:7 influence _v_1;8:8 the _q;9:9 outcome _n_1;10:10 and _c;12:12 join _v_1;13:13 the _q;14:14 winning _v_1;0:0 That _q_dem']

surf_bert_nc = ['0:0 Consumers _n_of;1:1 may _v_modal;2:2 want _v_1;4:4 move _v_cause;6:6 telephones _n_1;7:8 _x_deg a little;9:9 closer _a_to;11:11 the _q;12:12 TV _n_1;0:0 Consumers _n_of', '0:1 _n_1 Couch- potato;2:2 jocks _n_1;3:3 watching _v_1;4:4 ABC named;6:6 "Monday dofw;7:8 named Night Football";9:9 can _v_modal;10:10 now _a_1;11:11 vote _v_for;12:12 during _p;13:13 halftime _n_unknown;15:15 the _q;16:16 greatest _a_for;17:17 play _n_1;18:18 in _p;19:19 20 card;20:20 years _n_1;21:22 _p from among;23:23 four card;24:24 or _c;25:25 five card;26:26 filmed _v_1;0:0 Couch- _n_1', '0:0 Two card;1:1 weeks _n_1;2:2 ago _p;3:3 viewers _n_of;5:5 several _a_1;6:6 NBC named;7:7 daytime _n_1;8:8 consumer _n_of;9:9 segments _n_1;10:11 _v_1 started calling;12:12 a _q;13:13 900 yofc;14:14 number _n_of;15:15 for _p;16:16 advice _n_1;17:17 on _p;18:18 various _a_1;19:20 _n_of life- style;0:0 Two card', '0:0 And _c;1:1 the _q;2:2 new _a_1;3:3 syndicated _v_to;4:4 "reality" _n_1;5:5 show _n_of;6:7 named "Hard Copy";8:8 records _v_1;9:11 _n_of viewers opinions;12:12 for _p;13:13 possible _a_for;14:14 airing _v_1;15:15 on _p;16:16 the _q;17:17 next _a_1;18:20 _n_of day show', '0:0 Interactive _a_1;1:2 _n_1 telephone technology;4:4 taken _v_1;5:5 a _q;6:6 new _a_1;7:7 leap _n_1;8:8 in _p;9:9 sophistication _n_1;10:10 and _c;11:11 television _n_1;12:12 programmers _n_of;14:14 racing _v_1;15:15 to _x;16:16 exploit _v_1;17:17 the _q;0:0 Interactive _a_1', '0:0 Eventually _a_1;1:1 viewers _n_of;2:2 may _v_modal;3:3 grow _v_to;4:4 bored _a_with;5:5 with _p;6:6 the _q;7:7 technology _n_1;8:8 and _c;9:9 resent _v_1;10:10 the _q;0:0 Eventually _a_1', '0:0 But _c;1:1 right _x_deg;2:2 now _a_1;3:3 programmers _n_of;5:5 figuring _v_1;7:7 viewers _n_of;10:10 busy _a_1;11:11 dialing _v_up;13:13 a _q;14:14 range _n_of;16:16 services _n_1;17:17 may _v_modal;18:18 put _v_down;21:21 remote _a_1;22:22 control _n_of;23:23 zappers _n_1;24:24 and _c;25:25 stay _v_prd;0:0 But _c', '3:3 spending _v_1;4:4 a _q;5:7 _n_of lot time;8:8 in _p;9:10 named Los Angeles;11:11 talking _v_about;12:12 to _p;13:13 TV _n_1;14:15 _n_of production people,";16:16 says _v_to;17:18 named Mike Parks;19:19 president _n_of;21:22 named Call Interactive;24:24 supplied _v_1;25:25 technology _n_1;26:26 for _p;28:29 named ABC Sports;30:30 and _c;31:31 NBC named;33:33 consumer _n_of;3:3 spending _v_1', '0:0 "With _p;1:1 the _q;2:2 competitiveness _n_unknown;3:3 of _p;4:4 the _q;5:6 _n_1 television market;7:7 these _q_dem;8:8 days _n_of;11:11 looking _v_for;13:13 a _q;14:14 way _n_of;16:16 get _v_state;17:17 viewers _n_of;0:0 "With _p', '0:0 One card;2:2 the _q;3:3 leaders _n_of;4:4 behind _p;5:5 the _q;6:6 expanded _v_cause;7:7 use _n_of;9:9 900 card;10:10 numbers _n_of;12:13 named Call Interactive;14:14 a _q;15:15 joint _a_1;16:16 venture _n_1;17:17 of _p;18:19 named giants American;20:20 Express named;21:21 Co _n_1;22:22 and _c;23:24 named American Telephone;25:25 & _c;26:26 Telegraph named;0:0 One card', '0:0 Formed _v_cause;1:1 in _p_temp;2:2 August mofy;3:3 the _q;4:4 venture _n_1;5:5 weds _v_unknown;6:6 AT&T named;8:8 newly _a_1;9:9 expanded _v_cause;10:10 900 yofc;11:11 service _n_1;12:12 with _p;13:13 200 card;14:14 voice- _n_1;15:15 activated _v_1;16:16 computers _n_1;17:17 in _p;18:19 named American Express;21:22 named Omaha Neb;23:23 service named;0:0 Formed _v_cause', '0:0 Other _a_1;2:2 distance _n_of;2:2 distance _short_a_of;3:3 carriers _n_of;5:5 also _a_1;6:7 _v_1 begun marketing;8:8 enhanced named;9:9 900 yofc;10:10 service _n_1;11:11 and _c;12:12 special _a_1;13:13 consultants _n_1;15:15 springing _v_up;17:17 to _x;18:18 exploit _v_1;19:19 the _q;20:20 new _a_1;0:0 Other _a_1', '0:1 named Blair Entertainment;2:2 a _q;3:4 named New York;5:5 firm _n_1;7:7 advises _v_1;8:9 _n_1 TV stations;10:10 and _c;11:11 sells _v_1;12:12 ads _n_1;13:13 for _p;16:16 just _a_1;17:17 formed _v_cause;18:18 a _q;19:19 subsidiary _n_1;21:21 900 card;22:22 Blair named;24:24 to _x;25:25 apply _v_to;26:26 the _q;27:29 _n_1 technology television', '0:0 The _q;1:1 use _n_of;3:3 900 yofc;4:4 toll _n_1;5:5 numbers _n_of;8:8 expanding _v_1;9:9 rapidly _a_1;10:10 in _p_temp;11:11 recent _a_1;0:0 The _q', '0:0 For _p;1:2 _a_1 a while;4:4 cost _n_1;4:4 cost _high_a_1;5:5 pornography _n_unknown;6:6 lines _n_of;7:7 and _c;8:8 services _n_1;10:10 tempt _v_1;11:11 children _n_1;13:13 dial _v_1;14:14 (and _c;15:15 redial) _v_unknown;16:16 movie _n_of;17:17 or _c;18:18 music _n_1;19:19 information _n_on-about;20:20 earned _v_1;21:21 the _q;22:22 service _n_1;23:23 a _q;24:24 somewhat _x_deg;25:25 sleazy _a_unknown;26:26 image _n_of;27:27 but _c;28:29 _a_1 new legal;30:30 restrictions _n_of;32:32 aimed _v_at;34:34 trimming _v_1;0:0 For _p', '0:0 The _q;1:1 cost _n_1;2:2 of _p;3:3 a _q;4:4 900 yofc;5:5 call _n_1;7:7 set _v_1;9:9 the _q;10:10 originator _n_unknown;12:13 named ABC Sports;14:14 for _p;17:17 with _x_subord;18:18 the _q;19:19 cheapest _a_1;20:20 starting _v_1;21:21 at _p;22:22 75 card;0:0 The _q', '0:0 Billing _n_1;2:2 included _v_1;3:3 in _p;4:4 a _q;5:5 caller _n_of;7:7 regular _a_1;8:8 phone _n_1;0:0 Billing _n_1', '0:0 From _p_state;1:1 the _q;2:2 fee _n_1;3:3 the _q;4:4 local _a_1;5:5 phone _n_1;6:6 company _n_of;7:7 and _c;8:8 the _q;10:10 distance _n_1;10:10 distance _high_a_1;11:11 carrier _n_of;12:12 extract _v_from;14:14 costs _n_1;15:15 to _x;16:16 carry _v_1;17:17 the _q;18:18 call _n_1;19:19 passing _v_1;20:20 the _q;21:21 rest _n_of;23:23 the _q;24:24 money _n_1;26:26 the _q;27:27 originator _n_unknown;29:29 must _v_modal;30:30 cover _v_1;31:31 advertising _n_1;32:32 and _c;33:33 other _a_1;0:0 From _p_state', '0:0 In _p_temp;1:1 recent _a_1;2:2 months _n_1;3:3 the _q;4:4 technology _n_1;6:6 become _v_id;8:8 flexible _a_1;9:9 and _c;10:10 able _a_1;12:12 handle _v_1;13:13 much _x_deg;0:0 In _p_temp', '0:0 Before _x_h;1:1 callers _n_of;3:3 900 card;4:4 numbers _n_of;5:5 would _v_modal;6:6 just _a_1;7:7 listen _v_to;8:9 _c and not;10:10 talk _v_about;11:11 or _c;13:13 \'d _v_modal;14:14 vote _v_1;15:15 "yes" _a_1;16:16 or _c;17:17 "no" _a_1;18:18 by _p_means;19:19 calling _v_1;20:22 card one two;0:0 Before _x_h', '0:0 (People _n_of;1:1 in _p;2:2 the _q;3:4 _n_1 phone business;5:5 call _v_name;6:6 this _q_dem;7:7 technology _n_1;8:8 "900 card;0:0 (People _n_of', '0:0 Now _a_1;1:1 callers _n_of;3:3 led _v_1;4:4 through _p;5:5 complex _a_1;6:6 menus _n_1;7:7 of _p;8:8 choices _n_of;10:10 retrieve _v_1;11:11 information _n_on-about;13:13 want _v_1;14:14 and _c;15:15 the _q;16:16 hardware _n_1;17:17 can _v_modal;18:18 process _v_1;19:19 10,000 card;20:20 calls _n_1;21:21 in _p;22:22 90 card;0:0 Now _a_1', '0:1 _p Up to;2:2 now _a_1;3:3 900 card;4:4 numbers _n_of;6:6 mainly _a_1;8:8 used _v_1;9:9 on _p;10:10 local _a_1;11:12 _n_1 TV stations;13:13 and _c;14:15 _n_1 cable channels', '0:0 MTV named;1:1 used _v_1;2:2 one card;4:4 give _v_off;6:6 the _q;7:7 house _n_of;9:10 _n_1 rock star;11:12 named Jon Bon;13:13 Jovi named;14:14 grew _v_up;0:0 MTV named', '0:0 The _q;1:1 newest _a_1;2:2 uses _n_of;4:4 the _q;6:6 interactive _a_1;7:7 technology _n_1;8:8 demonstrate _v_to;9:9 the _q;10:10 growing _v_1;11:11 variety _n_of-n;0:0 The _q', '0:1 named Capital Cities;3:3 ABC named;4:4 Inc _n_1;5:5 CBS named;6:6 Inc _n_1;7:7 and _c;8:9 named General Electric;10:10 Co _n_1;12:13 named National Broadcasting;14:14 Co _n_1;15:15 unit _n_of;17:17 expected _v_1;19:19 announce _v_to;20:20 soon _p;21:21 a _q;22:22 joint _a_1;23:23 campaign _n_1;25:25 raise _v_cause;26:26 awareness _n_1;27:27 about _p;0:0 Capital named', '0:0 The _q;1:1 subject _n_of;4:4 written _v_into;6:6 the _q;7:10 _n_of plots time;11:11 shows _n_of;12:12 and _c;13:13 viewers _n_of;16:16 given _v_1;17:17 a _q;18:18 900 yofc;19:19 number _n_of;0:0 The _q', '0:0 Callers _n_of;3:3 sent _v_1;4:4 educational _a_1;5:5 booklets _n_of;6:6 and _c;7:7 the _q;8:8 call _n_1;10:10 modest _a_1;11:11 cost _n_1;13:13 be _v_id;14:14 an _q;15:15 immediate _a_1;16:16 method _n_of;18:18 raising _v_cause;0:0 Callers _n_of', '0:0 Other _a_1;1:2 _n_of network applications;3:3 have _v_1;4:4 very _x_deg;5:5 different _a_than-from;0:0 Other _a_1', '0:1 named ABC Sports;3:3 looking _v_for;5:5 ways _n_of;7:7 lift _v_cause;8:8 deflated _v_1;9:9 halftime _n_unknown;10:10 ratings _n_1;11:11 for _p;12:12 "Monday dofw;13:13 Night _n_of;0:0 ABC named', '0:1 named Kurt Sanger;2:3 named ABC Sports;5:5 marketing _v_1;6:6 director _n_of;7:7 says _v_to;9:9 now _a_1;10:12 card "tens thousands";14:14 fans _n_1;15:15 call _v_name;17:17 900 yofc;18:18 number _n_of;19:19 each _q;20:20 week _n_1;21:21 to _x;22:22 vote _v_for;24:24 the _q;25:25 best _a_at-for-of;26:26 punt _n_1;27:27 return _n_of;28:28 quarterback _n_unknown;29:29 sack _n_1;0:0 Kurt named', '0:0 Jackets _n_1;1:1 may _v_modal;3:3 sold _v_1;0:0 Jackets _n_1', '0:0 Meanwhile _p;1:2 named NBC Sports;3:3 recently _a_1;4:4 began _v_1;5:6 named "Scores Plus,";7:7 a _q;8:9 _a_1 year- round;11:11 hour _n_1;11:11 hour card;12:12 900 yofc;13:13 line _n_of;14:14 providing _v_1;15:15 a _q;16:16 complex _a_1;17:17 array _n_1;18:18 of _p;19:20 _n_of scores analysis;21:21 and _c;22:22 fan _n_1;0:0 Meanwhile _p', '0:0 A _q;1:1 spokesman _n_1;2:2 said _v_to;4:4 purpose _n_1;7:7 bolster _v_1;8:8 the _q;9:9 impression _n_of;11:12 named NBC Sports;14:15 _a_1 always there;16:16 for _p;0:0 A _q', '0:0 NBC named;2:3 _a_1 "On- Line";4:4 consumer _n_of;5:5 minutes _n_1;7:7 increased _v_cause;8:8 advertiser _n_unknown;9:9 spending _v_1;10:10 during _p;11:11 the _q;12:12 day _n_of;13:13 the _q;14:14 network _n_of;16:16 weakest _a_1;0:0 NBC named', '0:0 Each _q;1:1 weekday _n_1;2:2 matches _v_cause;3:3 a _q;4:4 sponsor _n_of;5:5 and _c;6:6 a _q;7:7 topic _n_of;8:8 : _p_namely;9:9 On _p_temp;10:10 Mondays dofw;11:12 named Unilever N.V;14:15 named Lever Bros;16:16 sponsors _v_1;17:17 tips _n_1;18:18 on _p;19:19 diet _n_1;20:20 and _c;21:21 exercise _n_1;22:22 followed _v_1;24:24 a _q;26:26 second _n_1;26:26 second card;27:28 named Lever Bros;0:0 Each _q', '0:0 If _x_then;1:1 the _q;2:2 caller _n_of;3:3 stays _v_prd;4:4 on _p;5:5 the _q;6:6 line _n_of;7:7 and _c;8:8 leaves _v_1;9:9 a _q;10:10 name _n_of;11:11 and _c;12:12 address _n_1;13:13 for _p;14:14 the _q;15:15 sponsor _n_of;16:16 coupons _n_1;17:17 and _c;18:18 a _q;19:19 newsletter _n_of;22:22 mailed _v_1;23:23 and _c;24:24 the _q;25:25 sponsor _n_of;28:28 able _a_1;30:30 gather _v_down;31:31 a _q;32:32 list _n_of;34:34 desirable _a_for;35:35 potential _a_1;0:0 If _x_then', '0:1 named Diane Seaman;2:2 an _q;3:4 named NBC- TV;5:5 vice _n_1;6:6 president _n_of;7:7 says _v_to;8:8 NBC named;11:11 able _a_1;13:13 charge _v_1;14:14 premium _a_1;15:15 rates _n_of;16:16 for _p;17:17 this _q_dem;18:18 ad _n_1;0:0 Diane named', "1:1 wouldn't _v_modal;2:2 say _v_to;4:4 the _q;5:5 premium _n_1;6:6 is _v_id;7:7 but _c;10:10 believed _v_1;12:12 be _v_id;13:13 about _x_deg;14:14 40 card;15:15 % _n_of;16:16 above _p;17:17 regular _a_1;18:18 daytime _n_1;1:1 wouldn't _v_modal", '2:2 able _a_1;4:4 get _v_cause;5:5 advertisers _n_unknown;7:7 use _v_1;9:9 promotion _n_of;10:10 budget _n_1;11:11 for _p;12:12 this _q_dem;13:13 because _x;15:15 get _v_1;16:16 a _q;17:17 chance _n_of;19:19 do _v_1;20:20 couponing," _n_unknown;21:21 says _v_to;22:22 Ms _n_1;2:2 able _a_1', '0:0 "And _c;3:3 able _a_1;5:5 attract _v_1;6:6 some _q;7:7 new _a_1;8:8 advertisers _n_unknown;9:9 because _x;10:10 this _q_dem;11:11 is _v_id;12:12 something _q;0:0 "And _c', '0:0 Mr _n_1;1:1 Parks named;2:2 of _p;3:4 named Call Interactive;5:5 says _v_to;6:7 _n_1 TV executives;9:9 considering _v_1;10:10 the _q;11:11 use _n_of;13:13 900 card;14:14 numbers _n_of;15:15 for _p;16:16 "talk _n_of-on;17:17 shows _n_of;18:18 game _n_1;19:20 _n_of shows news;21:21 and _c;22:22 opinion _n_of;0:0 Mr _n_1', '0:0 Experts _n_1;2:2 predicting _v_1;3:3 a _q;4:4 big _a_1;5:5 influx _n_1;6:6 of _p;7:7 new _a_1;8:8 shows _n_of;9:9 in _p_temp;10:10 1990 yofc;12:12 a _q;13:13 service _n_1;14:14 called _v_name;15:15 "automatic _a_1;16:16 number _n_of;17:17 information" _n_on-about;19:19 become _v_id;20:21 _a_1 widely available', '0:0 This _q_dem;1:1 service _n_1;2:2 identifies _v_1;3:3 each _q;4:4 caller _n_of;6:6 phone _n_1;7:7 number _n_of;8:8 and _c;10:10 can _v_modal;12:14 _v_1 used generate;15:15 instant _a_1;16:16 mailing _n_1;0:0 This _q_dem', '0:0 "For _p;1:1 a _q;2:2 piece _n_of;3:3 on _p;4:4 local _a_1;5:5 heroes named;6:6 of _p;7:8 named World War;9:9 II named;11:11 can _v_modal;12:12 ask _v_1;13:13 people _n_of;15:15 leave _v_1;16:16 the _q;17:17 name _n_of;18:18 and _c;19:19 number _n_of;21:21 anyone _q;23:25 _v_1 know won;26:26 a _q;27:27 medal," _n_of;0:0 "For _p', '0:0 "That _q_dem;2:2 save _v_1;4:4 time _n_of;5:5 and _c;6:6 get _v_state;7:7 people _n_of;0:0 "That _q_dem', '0:0 But _c;1:1 Mr _n_1;2:2 Monsky named;3:3 sees _v_1;4:4 much _x_deg;5:5 bigger _a_1;6:6 changes _n_of;0:0 But _c', '0:0 "These _q_dem;1:1 are _v_id;2:2 just _x_deg;3:4 _n_1 baby steps;5:5 toward _p;6:7 _a_1 real interactive;8:8 video _n_of;11:11 believe _v_1;13:13 be _v_id;14:14 the _q;15:15 biggest _a_1;16:16 thing _n_of-about;17:17 yet _a_1;19:19 affect _v_1;20:20 television," _n_1;0:0 "These _q_dem', '0:0 Although _x;2:2 would _v_modal;4:4 costly _a_1;5:5 to _x;6:6 shoot _v_1;7:7 multiple _a_1;8:8 versions _n_of;9:9 TV _n_1;10:10 programmers _n_of;11:11 could _v_modal;12:12 let _v_1;13:13 audiences _n_1;14:14 vote _v_1;15:15 on _p;16:16 different _a_than-from;17:17 endings _n_of;18:18 for _p;19:19 a _q;0:0 Although _x', '0:1 named Fox Broadcasting;2:2 experimented _v_with;4:4 this _q_dem;5:5 concept _n_1;6:6 last _a_1;7:7 year _n_1;8:8 when _x_subord;9:9 viewers _n_of;11:12 named "Married  . ;13:13 With _p;14:14 Children" named;15:15 voted _v_on;18:18 Al named;19:19 should _v_modal;20:22 _v_1 say love;24:24 to _x;25:25 Peg _v_1;26:26 on _p_temp;0:0 Fox named', '0:0 Someday _a_1;1:1 viewers _n_of;2:2 may _v_modal;3:3 also _a_1;4:4 choose _v_from;5:5 different _a_than-from;6:8 _n_of depths news;0:0 Someday _a_1', '0:0 "A _q;1:1 menu _n_1;2:2 by _p_means;3:3 phone _n_1;4:4 could _v_modal;5:7 _v_1 let decide;10:10 interested _a_in;12:12 just _x_deg;13:13 the _q;14:14 beginning _v_1;16:17 _n_of story No;18:18 1 card;19:19 and _c;21:21 want _v_1;22:23 _n_of story No;24:24 2 card;25:25 in _p;26:26 depth," _n_of;27:27 Mr _n_1;28:28 Monsky named;0:0 "A _q', '2:4 _v_1 start see;5:7 _n_of shows viewers;8:8 program _v_1;9:9 the _q;2:2 start _v_1', '0:1 named Integrated Resources;2:2 Inc _n_1;3:3 the _q;4:4 troubled _a_1;6:6 services _no_q;7:7 company _n_of;11:13 _v_1 trying sell;15:15 core _n_1;16:16 companies _n_of;17:17 to _x;18:18 restructure _v_cause;19:19 debt _n_1;20:20 said _v_to;21:21 talks _n_of-on;22:22 with _p;23:23 a _q;24:24 potential _a_1;25:25 buyer _n_of;0:0 Integrated named', '0:0 Integrated named;2:2 identify _v_1;3:3 the _q;4:4 party _n_of-to;5:5 or _c;6:6 say _v_to;7:7 why _p;8:8 the _q;9:9 talks _n_of-on;0:0 Integrated named', '0:0 Last _a_1;1:1 week _n_1;2:2 another _q;3:3 potential _a_1;4:4 buyer _n_of;5:6 named Whitehall Financial;7:7 Group named;11:11 agreed _v_1;12:12 in _p_temp;13:13 August mofy;15:15 purchase _v_1;16:16 most _q;18:18 Integrated named;20:20 core _n_1;21:21 companies _n_of;22:22 for _p;23:23 $ _n_1;24:25 card 310 million;27:27 ended _v_cause;28:28 talks _n_of-on;29:29 with _p;0:0 Last _a_1', '0:0 A _q;1:1 price _n_of;0:0 A _q', '0:0 Integrated named;1:1 also _a_1;2:2 said _v_to;4:4 expects _v_1;6:6 report _v_to;7:7 a _q;9:9 quarter _n_temp;9:9 quarter ord;10:10 loss _n_of;11:11 wider _a_1;13:13 the _q;14:14 earlier _a_1;15:15 estimate _n_of;17:17 about _x_deg;18:18 $ _n_1;19:20 card 600 million', '0:0 Meanwhile _p;1:1 a _q;2:2 number _n_of;4:4 top _a_1;5:6 _n_of sales producers;7:7 from _p;8:9 named Integrated Resources;10:10 Equity named;12:12 meet _v_1;13:13 this _q_dem;14:14 afternoon _n_of;15:15 in _p;16:16 Chicago named;17:17 to _x;18:18 discuss _v_1;0:0 Meanwhile _p', '0:0 The _q;1:1 unit _n_of;2:2 is _v_id;3:3 a _q;4:4 loosely _a_1;5:5 constructed _v_1;6:6 group _n_of;8:8 about _x_deg;9:9 3,900 card;10:10 independent _a_of;11:11 brokers _n_1;12:12 and _c;13:13 financial _a_1;14:14 planners _n_1;16:16 sell _v_1;17:17 insurance _n_1;18:18 annuities _n_unknown;19:19 limited _v_1;20:20 partnerships _n_1;21:21 mutual _a_1;22:22 funds _n_1;23:23 and _c;24:24 other _a_1;25:25 investments _n_1;26:26 for _p;27:27 Integrated _v_into;28:28 and _c;29:29 other _a_1;0:0 The _q', '0:0 The _q;1:1 sales _n_of;2:2 force _n_1;4:4 viewed _v_as;6:6 a _q;7:7 critical _a_of;8:8 asset _n_1;9:9 in _p;10:10 Integrated named;12:12 attempt _n_1;14:14 sell _v_1;16:16 core _n_1;0:0 The _q', '0:0 Whitehall named;1:1 cited _v_for;2:2 concerns _n_1;3:3 about _p;5:5 long _a_1;6:6 Integrated named;7:7 would _v_modal;9:9 able _a_1;11:11 hold _v_back;13:13 the _q;14:14 sales _n_of;15:15 force _n_1;17:17 one card;18:18 reason _n_for;20:20 talks _n_of-on;21:21 with _p;22:22 Integrated named;0:0 Whitehall named', '0:0 In _p_state;1:1 composite _n_1;2:2 trading _v_1;3:3 on _p;4:4 the _q;5:6 named New York;7:8 _n_1 Stock Exchange;9:9 yesterday _a_1;10:10 Integrated named;11:11 closed _v_1;12:12 at _p;13:13 $ _n_1;14:14 1.25 card;15:15 a _p_per;16:16 share _n_of;17:17 down _p;18:18 25 card;0:0 In _p_state', '0:0 Integrated named;3:5 _v_1 struggling avoid;6:6 a _q;7:8 _n_1 bankruptcy- law;9:9 filing _n_1;10:10 since _p;11:11 June mofy;14:16 _v_1 failed make;17:17 interest _n_in;18:18 payments _n_of;19:19 on _p;20:20 nearly _x_deg;21:21 $ _n_1;22:23 card 1 billion;24:24 of _p;0:0 Integrated named', '0:0 Integrated named;1:1 senior _a_1;2:2 and _c;3:3 junior _a_1;4:4 creditors _n_1;6:6 owed _v_1;7:7 a _q;8:8 total _n_of;10:10 about _x_deg;11:11 $ _n_1;12:13 card 1.8 billion', '0:0 AN _q;1:1 EARTHQUAKE _n_1;2:2 STRUCK _v_1;3:3 Northern _a_1;4:4 California named;5:5 killing _v_1;8:8 50 card;0:0 AN _q', '0:0 The _q;1:1 violent _a_1;2:2 temblor _n_1;4:4 lasted _v_1;5:5 about _x_deg;6:6 15 card;7:7 seconds _n_1;8:8 and _c;9:9 registered _v_1;10:10 6.9 card;11:11 on _p;12:12 the _q;13:13 Richter named;14:14 scale _n_1;15:15 also _a_1;16:16 caused _v_1;17:17 the _q;18:18 collapse _n_1;19:19 of _p;20:20 a _q;22:22 foot _n_1;22:22 foot card;23:23 section _n_of;25:25 the _q;26:27 named San Francisco-;28:29 named Oakland Bay;30:30 Bridge named;31:31 and _c;32:32 shook _v_1;33:34 named Candlestick Park', '0:0 The _q;1:1 tremor _n_1;3:3 centered _v_1;4:4 near _p;5:5 Hollister named;6:6 southeast _a_1;8:9 named San Francisco;10:10 and _c;12:12 felt _v_1;14:14 far _a_1;16:16 200 card;17:17 miles _n_1;0:0 The _q', '0:0 Numerous _a_1;1:1 injuries _n_1;0:0 Numerous _a_1', '0:0 Some _q;1:1 buildings _n_1;2:2 collapsed _v_cause;3:3 gas _n_1;4:4 and _c;5:5 water _n_1;6:6 lines _n_of;7:7 ruptured _v_1;8:8 and _c;9:9 fires _n_1;0:0 Some _q', '0:0 The _q;1:1 quake _n_1;3:3 also _a_1;4:4 caused _v_1;5:5 damage _n_to;6:6 in _p;7:8 named San Jose;9:9 and _c;10:10 Berkeley named;11:11 knocked _v_out;13:13 electricity _n_1;14:14 and _c;15:15 telephones _n_1;16:16 cracked _v_cause;17:17 roadways _n_unknown;18:18 and _c;19:19 disrupted _v_1;20:21 _n_1 subway service;22:22 in _p;23:23 the _q;24:24 Bay named;0:0 The _q', '0:0 Major _a_1;1:1 injuries _n_1;3:3 reported _v_to;4:4 at _p;5:6 named Candlestick Park;8:8 the _q;9:9 third ord;10:10 game _n_1;11:11 of _p;12:14 named baseball World;15:15 Series named;17:17 canceled _v_1;18:18 and _c;19:19 fans _n_1;20:20 evacuated _v_1;21:21 from _p;22:22 the _q;0:0 Major _a_1', '0:0 Bush named;1:1 vowed _v_to;3:3 veto _v_1;4:4 a _q;5:5 bill _n_of;6:6 allowing _v_1;7:7 federal _a_1;8:8 financing _v_1;9:9 for _p;10:10 abortions _n_of;11:11 in _p;12:12 cases _n_of;14:14 rape _n_1;15:15 and _c;16:16 incest _n_unknown;17:17 saying _v_to;18:19 _n_1 tax dollars;20:20 shouldn\'t _v_modal;22:24 _v_1 used "compound;25:25 a _q;26:26 violent _a_1;27:27 act _n_of;28:28 with _p;29:29 the _q;30:30 taking _v_1;32:32 an _q;33:33 unborn _a_1;0:0 Bush named', '1:1 pledge _n_1;2:2 in _p;3:3 a _q;4:4 letter _n_of;5:5 to _p;6:6 Democratic named;7:7 Sen _n_1;8:8 Byrd named;9:9 came _v_1;10:11 _p ahead of;12:12 an _q;13:13 expected _v_1;14:14 Senate named_n;15:15 vote _n_1;16:16 on _p;17:17 spending _v_1;18:18 legislation _n_1;19:19 containing _v_1;20:20 the _q;1:1 pledge _n_1', '0:1 named East Germany;3:3 Politburo named;4:4 met _v_1;5:5 amid _p;6:6 speculation _n_about;8:8 the _q;9:9 ruling _v_1;10:10 body _n_1;11:11 would _v_modal;12:12 oust _v_1;14:14 line _n_of;14:14 line _high_a_1;15:15 leader _n_of;16:16 Honecker named;18:18 rule _n_of;21:21 challenged _v_to;23:23 mass _n_of;24:24 emigration _n_unknown;25:25 and _c;26:26 calls _n_1;28:28 democratic _a_1;0:0 East named', '0:0 Meanwhile _p;1:1 about _x_deg;2:2 125 card;3:3 refugees _n_1;4:4 flew _v_1;5:5 to _p;6:7 named Duesseldorf West;8:8 Germany named;9:9 from _p;10:10 Warsaw named;11:11 the _q;12:12 first ord;13:13 airlift _n_1;14:14 in _p;15:16 named East Germany;18:19 _n_1 refugee exodus', '0:0 The _q;1:2 named World Psychiatric;3:3 Association named;4:4 voted _v_1;5:5 at _p;6:6 an _q;7:8 named Athens parley;10:10 conditionally _a_unknown;11:11 readmit _v_unknown;12:12 the _q;13:14 named Soviet Union', '0:0 Moscow named;2:2 left _v_1;3:3 the _q;4:4 group _n_of;5:5 in _p_temp;6:6 1983 yofc;7:7 to _x;8:8 avoid _v_1;9:9 explusion _n_unknown;10:10 over _p;11:11 allegations _n_1;13:13 political _a_1;14:14 dissidents _n_unknown;17:17 certified _v_as;19:19 insane _a_1;20:20 could _v_modal;22:22 suspended _v_1;23:23 if _x_then;24:24 the _q;25:25 misuse _n_of;27:27 psychiatry _n_1;28:28 against _p;29:29 dissenters _n_unknown;31:31 discovered _v_1;32:32 during _p;33:33 a _q;34:34 review _n_of;35:35 within _p;36:36 a card;0:0 Moscow named', '0:0 NASA named;1:1 postponed _v_1;2:2 the _q;3:3 liftoff _n_1;4:4 of _p;5:5 the _q;6:6 space _n_of;7:7 shuttle _n_1;8:8 Atlantis _n_unknown;9:10 _p because of;11:11 rain _n_1;12:12 near _p;13:13 the _q;14:14 site _n_1;15:15 of _p;16:16 the _q;17:18 _n_1 launch pad;19:19 in _p;20:21 named Cape Canaveral;0:0 NASA named', '0:0 The _q;1:1 flight _n_1;3:3 rescheduled _v_1;3:3 rescheduled _re-_a_again;4:4 for _p;0:0 The _q', '0:0 The _q;1:1 spacecraft _n_1;3:3 five card;4:4 astronauts _n_1;7:7 dispatch _v_to;8:8 the _q;10:10 powered _v_1;11:11 Galileo named;12:12 space _n_of;13:13 probe _n_1;14:14 on _p;15:15 an _q;16:16 exploratory _a_unknown;17:17 mission _n_1;18:18 to _p;0:0 The _q', '0:0 Senate named_n;1:1 Democratic named;2:2 leaders _n_of;3:3 said _v_to;5:5 had _v_1;6:6 enough _q;7:7 votes _n_1;8:8 to _x;9:9 defeat _v_1;10:10 a _q;11:11 proposed _v_to;12:12 constitutional _a_1;13:13 amendment _n_of;15:15 ban _v_from;16:16 flag _n_1;0:0 Senate named_n', '0:0 The _q;1:1 amendment _n_of;3:3 aimed _v_at;5:5 skirting _v_1;6:6 a _q;7:8 named Supreme Court;9:9 ruling _n_1;11:11 threw _v_out;13:13 the _q;14:14 conviction _n_of;16:16 a _q;17:17 Texas named;18:19 _n_1 flag- burner;20:20 on _p;21:24 _n_1 grounds freedom;25:25 of _p;26:26 speech _n_1;0:0 The _q', '0:0 Federal _a_1;1:1 researchers _n_of;2:2 said _v_to;3:4 _n_1 lung- cancer;5:5 mortality _n_1;6:6 rates _n_of;7:7 for _p;8:8 people _n_of;9:9 under _p;10:10 45 card;11:11 years _n_1;12:12 of _p;13:13 age _n_1;15:17 _v_1 begun decline;18:18 particularly _x_deg;19:19 for _p;20:20 white _a_1;0:0 Federal _a_1', '0:0 The _q;1:2 named National Cancer;3:3 Institute named;4:4 also _a_1;5:5 projected _v_1;7:7 overall _a_1;8:8 U.S named_n;9:9 mortality _n_1;10:10 rates _n_of;11:11 from _p;12:13 _n_1 lung cancer;14:14 should _v_modal;15:17 _v_1 begin drop;18:18 in _p_temp;19:19 several _a_1;20:20 years _n_1;21:21 if _x_then;22:22 cigarette _n_1;23:23 smoking _v_1;24:24 continues _v_2;0:0 The _q', '0:0 Bush named;1:1 met _v_1;2:2 with _p;3:3 South named;4:4 Korean _a_1;5:5 President _n_of;6:6 Roh named;8:8 indicated _v_1;10:10 Seoul named;11:11 plans _v_1;13:13 further _a_1;14:14 ease _v_1;15:16 _n_of trade rules;17:17 to _x;18:18 ensure _v_1;21:21 economy _n_1;22:22 becomes _v_id;24:24 open _a_1;26:26 the _q;27:28 _a_1 other industrialized;29:29 nations _n_of;30:30 by _p;31:31 the _q;0:0 Bush named', '0:0 Bush named;1:1 assured _v_of;2:2 Roh named;4:4 the _q;5:5 U.S named_n;6:6 would _v_modal;7:7 stand _v_1;8:8 by _p_dir;10:10 security _n_1;11:11 commitments _n_to;12:13 _x "as long;14:14 as _x;16:16 is _v_there;17:17 a _q;18:18 threat" _n_1;19:19 from _p;20:20 Communist _a_1;21:22 named North Korea', '0:0 The _q;1:1 Bush named;2:2 administration _n_of;4:4 seeking _v_1;5:5 an _q;6:6 understanding _n_of;7:7 with _p;8:8 Congress named;10:10 ease _v_1;11:11 restrictions _n_of;12:12 on _p;13:13 U.S named_n;14:14 involvement _n_in;16:16 foreign _a_1;17:17 coups _n_1;19:19 might _v_modal;20:20 result _v_in;22:22 the _q;23:23 death _n_1;24:24 of _p;25:25 a _q;26:28 _n_of country leader', '0:0 A _q;1:1 White named;2:2 House named_n;3:3 spokesman _n_1;4:4 said _v_to;6:6 while _x;7:7 Bush named;8:8 wouldn\'t _v_modal;9:9 alter _v_1;10:10 a _q;11:11 longstanding _a_unknown;12:12 ban _n_1;13:13 on _p;14:14 such _q;15:15 involvement _n_in;17:17 \'s _v_there;18:18 a _q;19:19 clarification _n_of;20:20 needed" _v_1;21:21 on _p;0:0 A _q', '0:2 named India Gandhi;3:3 called _v_for;5:5 parliamentary _a_1;6:6 elections _n_of-for;7:7 next _a_1;0:0 India named', '0:0 The _q;1:2 _v_1 balloting considered;3:3 a _q;4:4 test _n_of;5:5 for _p;6:6 the _q;7:8 _n_of prime minister;9:9 and _c;10:10 the _q;11:11 ruling _v_1;12:13 named Congress (I);14:14 Party _n_of-to;15:15 comes _v_1;16:16 amid _p;17:17 charges _n_of;19:19 inept _a_1;20:20 leadership _n_1;21:21 and _c;22:22 government _n_of;0:0 The _q', '0:0 Gandhi named;2:2 family _n_of;4:4 ruled _v_1;5:5 independent _a_of;6:6 India named;7:7 for _p;8:8 all _q;9:9 but _c_btwn;10:10 five card;11:11 years _n_1;12:12 of _p;15:15 year _n_1;15:15 year card;0:0 Gandhi named', '0:0 The _q;1:2 named Soviet Union;3:3 abstained _v_from;5:5 a _q;6:6 U.N named_n;7:7 General _a_1;8:8 Assembly named;9:9 vote _n_1;11:11 reject _v_as;12:12 Israel named;0:0 The _q', '1:1 was _v_id;2:2 the _q;3:3 first ord;4:4 time _n_of;5:5 in _p;6:6 seven card;7:7 years _n_1;9:9 Moscow named;11:11 joined _v_1;12:12 efforts _n_1;13:13 led _v_1;15:15 Moslem named;16:16 nations _n_of;18:18 expel _v_1;19:19 Israel named;20:20 from _p;21:21 the _q;22:22 world _n_of;23:23 body _n_1;24:24 and _c;26:26 viewed _v_as;28:28 a _q;29:29 sign _n_of;31:31 improving _v_cause;32:33 named Soviet- Israeli;1:1 was _v_id', '0:0 Israel named;2:2 seated _v_1;3:3 by _p;4:4 a _q;5:5 vote _n_1;6:6 of _p;7:9 card 95 37;10:10 with _p;11:11 15 card;0:0 Israel named', "0:0 Black _a_1;1:1 activist _n_1;2:3 named Walter Sisulu;4:4 said _v_to;5:5 the _q;6:7 named African National;8:8 Congress named;9:9 wouldn't _v_modal;10:10 reject _v_as;11:11 violence _n_1;13:13 a _q;14:14 way _n_of;16:16 pressure _v_1;17:17 the _q;18:19 named South African;20:20 government _n_of;21:21 into _p;22:22 concessions _n_1;24:24 might _v_modal;25:25 lead _v_to;27:27 negotiations _n_of;28:28 over _p;0:0 Black _a_1", '0:0 The _q;2:2 year- _n_1;2:2 year- card;3:3 old _a_1;4:4 Sisulu named;6:6 among _p;7:7 eight card;8:9 _a_1 black political;10:10 activists _n_1;11:11 freed _v_from;12:12 Sunday dofw;13:13 from _p;0:0 The _q', '0:0 London named;2:2 concluded _v_1;4:4 Austrian named;5:5 President _n_of;6:6 Waldheim named;8:8 responsible _a_for;10:10 the _q;11:11 execution _n_1;12:12 of _p;13:13 six card;14:14 British _a_1;15:15 commandos _n_1;16:16 in _p;17:18 named World War;19:19 II named;20:20 although _x;22:22 probably _a_1;24:24 aware _a_of;26:26 the _q;0:0 London named', '0:0 The _q;1:1 report _n_of;2:2 by _p;3:3 the _q;4:5 named Defense Ministry;6:6 also _a_1;7:7 rejected _v_as;8:8 allegations _n_1;10:10 Britain named;11:11 covered _v_up;13:13 evidence _n_of;15:15 Waldheim named;17:17 activities _n_1;18:18 as _p;19:19 a _q;20:20 German _a_1;21:21 army _n_of;0:0 The _q', '0:0 An _q;1:1 international _a_1;2:2 group _n_of;3:3 approved _v_1;4:4 a _q;5:5 formal _a_1;6:6 ban _n_1;7:7 on _p;8:8 ivory _n_1;9:9 trade _n_of;10:10 despite _p;11:11 objections _n_to;12:12 from _p;13:13 southern named;14:14 African _a_1;15:15 governments _n_of;17:19 _v_1 threatened find;20:20 alternative _a_1;21:21 channels _n_1;22:22 for _p;23:23 selling _v_1;24:25 _n_1 elephant tusks', '0:0 The _q;1:1 move _n_1;2:2 by _p;3:3 the _q;4:4 Convention named;5:5 on _p;6:6 Trade named;7:7 in _p;8:9 named Endangered Species;10:10 meeting _n_of;11:11 in _p;12:12 Switzerland named;13:13 places _v_1;14:14 the _q;15:15 elephant named;16:16 on _p;17:17 the _q;19:19 species _n_of;19:19 species card;0:0 The _q', '0:0 An _q;1:1 assassin _n_1;2:2 in _p;3:3 Colombia named;4:4 killed _v_1;5:5 a _q;6:6 federal _a_1;7:7 judge _n_of;8:8 on _p;9:9 a _q;10:10 Medellin named;0:0 An _q', '0:0 An _q;1:1 anonymous _a_1;2:2 caller _n_of;3:3 to _p;4:4 a _q;5:5 local _a_1;6:7 _n_1 radio station;8:8 said _v_to;9:9 cocaine _n_1;10:10 traffickers _n_unknown;12:12 slain _v_1;13:13 the _q;14:14 magistrate _n_1;15:15 in _p;16:16 retaliation _n_unknown;17:17 for _p;18:18 the _q;19:19 extraditions _n_unknown;20:20 of _p;21:21 Colombians _n_unknown;22:22 wanted _v_1;23:23 on _p;24:24 drug _n_1;25:25 charges _n_of;26:26 in _p;27:27 the _q;0:0 An _q', '0:0 Libyan named;1:1 leader _n_of;2:2 Gadhafi named;3:3 met _v_1;4:4 with _p;5:5 Egypt named;7:7 President _n_of;8:8 Mubarak named;9:9 and _c;10:10 the _q;11:11 two card;12:12 officials _n_1;13:15 _v_1 pledged respect;16:16 each _q;17:19 _n_1 other laws;20:20 security _n_1;21:21 and _c;0:0 Libyan named', '1:1 stopped _v_1;2:2 short _a_of;4:4 resuming _v_1;5:5 diplomatic _a_1;6:6 ties _n_1;7:7 severed _v_1;8:8 in _p_temp;1:1 stopped _v_1', '0:0 The _q;1:1 reconciliation _n_unknown;2:2 talks _n_of-on;3:3 in _p;4:4 the _q;5:5 Libyan _a_1;6:7 _n_1 desert town;8:8 of _p;9:9 Tobruk named;10:10 followed _v_1;11:11 a _q;12:12 meeting _n_of;13:13 Monday dofw;14:14 in _p;15:15 the _q;16:16 Egyptian _a_1;17:17 resort _n_1;18:18 of _p;19:20 named Mersa Metruh', '0:0 The _q;1:1 new _a_1;2:2 notes _n_1;4:4 bear _v_1;5:5 interest _n_in;6:6 at _p;7:7 5.5 card;8:8 % _n_of;9:9 through _p;10:10 July mofy;11:11 31 dofm;12:12 1991 yofc;13:13 and _c;14:15 _p thereafter at;16:16 10 card;0:0 The _q', '0:0 Under _p_state;1:1 the _q;2:2 original _a_1;3:3 proposal _n_1;4:4 the _q;5:5 maker _n_of;7:7 specialty _n_1;8:8 coatings _n_of;9:9 and _c;10:10 a _q;11:11 developer _n_of;13:13 information- _n_on-about;14:14 display _n_of;15:15 technologies _n_1;16:16 offered _v_1;17:17 $ _n_1;18:18 400 card;19:19 of _p;20:20 notes _n_1;21:21 due _a_1;22:22 1996 yofc;23:23 10 card;24:24 common _a_for;25:25 shares _n_of;26:26 and _c;27:27 $ _n_1;28:28 175 card;29:29 in _p;30:30 cash _n_1;31:31 for _p;32:32 each _q;33:33 $ _n_1;34:34 1,000 card;35:35 face _n_1;0:0 Under _p_state', '0:0 Completion _n_of;2:2 the _q;3:4 _n_1 exchange offer;6:6 subject _a_to;8:8 the _q;9:9 tender _n_1;10:10 of _p;11:12 _x_deg at least;13:13 80 card;14:14 % _n_of;15:15 of _p;16:16 the _q;17:17 debt _n_1;18:18 among _p;19:19 other _a_1;0:0 Completion _n_of', '0:0 The _q;1:1 stock _n_1;2:2 of _p;3:3 UAL named;4:4 Corp _n_1;5:5 continued _v_2;8:8 pounded _v_1;9:9 amid _p;10:10 signs _n_of;12:13 named British Airways;14:14 may _v_modal;15:15 balk _v_at;17:17 any _q;18:18 hasty _a_1;19:19 reformulation _n_unknown;20:20 of _p;21:21 the _q;22:22 aborted _v_1;23:23 $ _n_1;24:25 card 6.79 billion;26:27 _n_1 buy- out;28:28 of _p;29:30 named United Airlines;0:0 The _q', '0:0 The _q;1:1 plunge _n_of;2:2 followed _v_1;3:3 a _q;4:4 drop _n_of;6:6 $ _n_1;7:7 56.875 card;8:8 Monday dofw;9:9 amid _p;10:10 indications _n_of;11:11 the _q;12:12 takeover _n_of;13:13 may _v_modal;14:14 take _v_2;15:15 weeks _n_1;0:0 The _q', '0:0 The _q;1:1 stock _n_1;3:3 fallen _v_1;4:4 $ _n_1;5:5 87.25 card;6:6 or _c;7:7 31 card;8:8 % _n_of;9:9 in _p_temp;10:10 the _q;11:11 three card;12:12 trading _v_1;13:13 days _n_of;14:14 since _p;15:15 announcement _n_of;17:17 the _q;18:18 collapse _n_1;19:19 of _p;20:20 the _q;21:21 $ _n_1;22:22 300- card;24:24 share _n_of;24:24 share _a_p_per;25:25 takeover _n_of;26:26 jolted _v_1;27:27 the _q;28:28 entire _a_1;29:30 _n_1 stock market;31:31 into _p;34:34 worst _a_1;34:34 worst ord;0:0 The _q', '0:0 "This _q_dem;1:1 is _v_id;2:2 a _q;3:3 total _a_1;4:4 bloodbath" _n_1;5:5 for _p;6:6 takeover- _n_of;7:8 _n_1 stock traders;9:9 one card;10:11 _n_1 investment banker;0:0 "This _q_dem', '0:1 named Los Angeles;2:2 financier _n_unknown;3:4 named Marvin Davis;6:6 put _v_1;7:7 United named;8:8 in _p;9:9 play _n_1;10:10 with _p;11:11 a _q;12:12 $ _n_1;13:14 card 5.4 billion;15:15 bid _n_1;16:16 two card;17:17 months _n_1;18:18 ago _p;19:19 last _a_1;20:20 night _n_of;21:21 proffered _v_1;23:23 a _q;24:24 ray _n_1;25:25 of _p;26:26 hope _n_1;27:27 and _c;28:28 an _q;29:29 extra _a_1;30:30 element _n_1;31:31 of _p;32:32 uncertainty _n_1;33:33 by _p_means;34:34 saying _v_to;36:36 remains _v_1;37:37 interested _a_in;39:39 acquiring _v_1;0:0 Los named', '0:0 But _c;2:2 dropped _v_cause;4:4 earlier _a_1;5:5 $ _n_1;6:6 300- card;8:8 share _n_of;8:8 share _a_p_per;9:10 _n_1 back- up;11:11 bid _n_1;12:12 saying _v_to;14:14 must _v_modal;15:15 first _a_1;16:16 explore _v_1;17:17 bank _n_of;0:0 But _c', '0:0 Even _x_deg;1:1 as _x_subord;2:2 Citicorp named;3:3 and _c;4:5 named Chase Manhattan;6:6 Corp _n_1;7:7 scrambled _v_1;8:8 to _x;9:9 line _v_up;11:11 bank _n_of;12:12 financing _v_1;13:13 for _p;14:14 a _q;15:15 revised _v_1;16:16 version _n_of;18:18 the _q;19:19 lapsed _v_1;20:21 _n_1 labor- management;22:22 bid _n_1;23:24 named British Airways;25:25 a _q;26:26 15 card;27:27 % _n_of;28:28 partner _n_1;29:29 in _p;30:30 the _q;31:31 buying _v_1;32:32 group _n_of;33:35 _v_1 indicated wants;37:37 start _v_1;38:38 from _p;0:0 Even _x_deg', '1:1 partners _n_1;2:2 are _v_id;3:3 United named;5:5 pilots _n_of;9:9 own _v_1;10:10 75 card;11:11 % _n_of;12:12 and _c;13:13 UAL named;14:14 management _n_1;15:15 at _p;16:16 10 card;1:1 partners _n_1', '0:0 A _q;1:1 similar _a_to;2:2 demand _n_1;4:4 made _v_1;6:6 a _q;7:7 group _n_of;9:9 represents _v_1;10:10 some _q;12:12 United named;14:14 26,000 card;15:15 noncontract _a_unknown;0:0 A _q', '0:1 named John Peterpaul;2:2 Machinists named;3:3 union _n_of;4:4 general _a_1;5:5 vice _n_1;6:6 president _n_of;7:7 attacked _v_as;8:8 Mr _n_1;9:9 Wolf named;11:11 "greedy _a_1;12:12 and _c;13:13 irresponsible" _a_unknown;14:14 for _p;15:15 pursuing _v_1;16:16 the _q;17:18 _n_1 buy- out', '0:0 Although _x;1:1 Mr _n_1;2:2 Wolf named;3:3 and _c;4:5 named John Pope;6:6 UAL named;8:8 chief _n_1;9:9 financial _a_1;10:10 officer _n_1;11:13 _v_1 stood pocket;14:14 $ _n_1;15:16 card 114.3 million;17:17 for _p;18:18 stock _n_1;19:19 and _c;20:20 options _n_1;21:21 in _p;22:22 the _q;23:24 _n_1 buy- out;25:25 UAL named;26:26 executives _n_1;27:29 _v_1 planned reinvest;29:29 reinvest _re-_a_again;30:30 only _x_deg;31:31 $ _n_1;32:33 card 15 million;35:35 the _q;36:36 new _a_1;0:0 Although _x', '0:0 The _q;2:2 collar _n_1;2:2 collar _high_a_1;3:3 machinists _n_1;4:4 longtime _a_unknown;5:5 rivals _n_1;6:6 of _p;7:7 the _q;9:9 collar _n_1;9:9 collar _high_a_1;10:10 pilots _n_of;11:11 say _v_to;12:12 the _q;13:13 buyout _n_1;14:14 would _v_modal;15:15 load _v_cause;16:16 the _q;17:17 company _n_of;18:18 with _p;19:19 debt _n_1;20:20 and _c;21:21 weaken _v_cause;0:0 The _q', '0:0 Confusion _n_1;1:1 about _p;2:2 the _q;3:3 two card;4:4 banks _n_of;6:6 hurried _v_1;7:7 efforts _n_1;9:9 round _v_up;11:11 financing _v_1;12:12 for _p;13:13 a _q;14:14 new _a_1;15:15 bid _n_1;17:17 the _q;18:18 UAL named;19:19 board _n_of;21:21 even _a_1;22:22 seen _v_1;23:23 yet _a_1;24:24 helped _v_to;25:25 send _v_1;26:26 UAL named;27:27 stock _n_1;28:28 spiraling _v_1;0:0 Confusion _n_1', '0:0 And _c;1:1 rumors _n_1;2:2 of _p;3:4 _v_1 forced selling;5:5 by _p;6:6 takeover- _n_of;7:8 _n_1 stock traders;9:9 triggered _v_1;10:10 a _q;12:12 point _n_of;12:12 point card;13:13 downdraft _n_unknown;14:14 in _p;15:15 the _q;16:17 named Dow Jones;18:19 named Industrial Average;20:20 around _p;21:21 11:15 numbered_hour;22:22 a.m _x;23:23 EDT holiday;0:0 And _c', '0:0 Yesterday _a_1;2:3 _v_1 selling began;4:4 after _x_h;5:5 a _q;6:6 Japanese _a_1;7:7 news _n_of;8:8 agency _n_1;9:9 reported _v_to;11:11 Japanese _a_1;12:12 banks _n_of;14:14 balked _v_at;16:16 the _q;17:17 first ord;18:18 bid _n_1;20:20 ready _a_1;22:22 reject _v_as;23:23 a _q;24:24 revised _v_1;25:25 version _n_of;26:26 at _p;27:27 around _x_deg;28:28 $ _n_1;29:29 250 card;30:30 a _p_per;31:31 share _n_of;32:32 or _c;33:33 $ _n_1;34:35 card 5.65 billion', '0:0 Citicorp named;1:2 _p for example;3:3 said _v_to;4:4 only _a_1;7:7 had _v_1;8:8 "expressions _n_1;9:9 of _p;10:10 interest _n_in;11:11 of _p;12:12 a _q;13:13 transaction _n_1;14:14 from _p;16:16 the _q;17:17 borrowers _n_unknown;18:18 and _c;19:19 the _q;20:20 banks," _n_of;21:21 but _c;23:23 have _v_1;24:24 an _q;0:0 Citicorp named', '0:0 Late _p;1:1 in _p_temp;2:2 the _q;3:3 day _n_of;4:4 Mr _n_1;5:5 Wolf named;6:6 issued _v_1;7:7 a _q;8:8 onepage _n_unknown;9:9 statement _n_of;10:10 calling _v_1;11:11 Mr _n_1;12:12 Peterpaul named;14:14 blast _n_1;15:15 "divisive _a_unknown;16:16 and _c;17:17 uncalled _a_unknown;0:0 Late _p', '0:0 But _c;2:2 gave _v_1;4:4 details _n_1;5:5 on _p;6:6 the _q;7:7 progress _n_1;8:8 toward _p;9:9 a _q;10:10 new _a_1;11:11 bid _n_1;12:12 saying _v_to;13:13 only _a_1;16:16 working _v_1;17:17 toward _p;18:18 a _q;19:19 revised _v_1;20:20 proposal _n_1;21:21 for _p;22:23 _n_of majority employee;0:0 But _c', '0:0 Meanwhile _p;1:1 in _p_state;2:2 another _q;3:3 sign _n_of;5:5 a _q;6:6 new _a_1;7:7 bid _n_1;9:9 imminent _a_1;12:12 learned _v_1;14:14 the _q;15:15 UAL named;16:16 board _n_of;17:17 held _v_1;18:18 a _q;19:19 telephone _n_1;20:20 meeting _n_of;21:21 Monday dofw;22:22 to _x;23:23 hear _v_1;24:24 an _q;25:25 update _n_of;26:26 on _p;27:27 the _q;28:28 situation _n_1;29:29 but _c;31:31 a _q;32:32 formal _a_1;33:34 _n_of board meeting;36:36 likely _a_1;39:39 convened _v_cause;40:40 until _p;41:42 _a_1 early next;0:0 Meanwhile _p', '0:0 The _q;1:1 effort _n_1;3:3 revive _v_cause;4:4 the _q;5:5 bid _n_1;7:7 complicated _v_1;9:9 the _q;10:10 unwieldy _a_1;11:11 nature _n_1;12:12 of _p;13:13 the _q;15:15 party _n_of;15:15 party card;16:16 buying _v_1;0:0 The _q', '0:0 The _q;1:1 pilots _n_of;3:3 meeting _v_1;4:4 outside _p;5:5 Chicago named;0:0 The _q', '0:0 But _c;1:2 named British Air;7:7 supplied _v_1;8:8 $ _n_1;9:10 card 750 million;11:12 _p out of;13:13 $ _n_1;14:15 card 965 million;16:16 in _p;17:17 equity _n_1;18:18 financing _v_1;19:19 apparently _a_1;21:21 involved _v_in;23:23 the _q;24:24 second ord;25:25 proposal _n_1;26:26 and _c;27:27 could _v_modal;28:28 well _a_1;29:29 reject _v_as;31:31 even _x_deg;32:32 if _x_then;33:33 banks _n_of;34:35 _v_1 obtain financing', '0:0 The _q;1:1 machinists _n_1;2:2 also _a_1;3:3 asked _v_for;5:5 an _q;6:6 investigation _n_1;7:7 by _p;8:8 the _q;9:9 Securities named;10:10 and _c;11:12 named Exchange Commission;13:13 into _p;14:14 possible _a_for;15:16 _n_1 securities- law;17:17 violations _n_of;18:18 in _p;19:19 the _q;20:20 original _a_1;21:21 bid _n_1;22:22 for _p;23:23 UAL named;24:24 by _p;25:25 Mr _n_1;26:26 Davis named;27:28 _c as well;29:29 as _c;30:30 in _p;31:31 the _q;32:32 response _n_to;33:33 by _p;0:0 The _q', '0:0 Last _a_1;1:1 week _n_1;2:2 just _x_deg;3:3 before _x_h;4:4 the _q;5:5 bank _n_of;6:6 commitments _n_to;8:8 due _a_1;9:9 the _q;10:10 union _n_of;11:11 asked _v_1;12:12 the _q;13:13 U.S named_n;14:15 named Labor Department;17:17 study _v_1;19:19 the _q;20:20 bid _n_1;21:21 violated _v_1;22:22 legal _a_1;23:23 standards _n_1;24:24 of _p;25:25 fairness _n_1;26:26 governing _v_1;27:27 employee _n_of;28:29 _n_1 investment funds', '0:0 In _p_state;2:2 statement _n_of;3:3 Mr _n_1;4:4 Wolf named;5:5 said _v_to;7:7 continue _v_2;9:9 believe _v_1;11:11 approach _n_to;13:13 sound _a_1;14:14 and _c;18:18 far _x_deg;19:19 better _a_at-for-of;20:20 for _p;21:21 all _q;22:22 employees _n_of;24:24 the _q;25:25 alternative _n_to;26:26 of _p;27:27 having _v_cause;28:28 an _q;29:29 outsider _n_1;30:30 own _v_1;31:31 the _q;32:32 company _n_of;33:33 with _x_subord;34:34 employees _n_of;35:35 paying _v_for;38:38 just _x_deg;39:40 _a_1 the same."', '0:0 The _q;1:1 huge _a_1;2:2 drop _n_of;3:3 in _p;4:4 UAL named;5:5 stock _n_1;6:6 prompted _v_1;7:7 one card;8:8 takeover _n_of;9:10 _n_1 stock trader;11:12 named George Kellner;13:13 managing _v_1;14:14 partner _n_1;15:15 of _p;16:17 named Kellner DiLeo;18:19 _n_1 & Co;21:21 deny _v_to;22:22 publicly _a_1;23:26 _n_1 rumors firm;28:28 going _v_1;29:30 _p_dir out of;0:0 The _q', '0:0 Mr _n_1;1:1 Kellner named;2:2 said _v_to;4:4 despite _p;5:5 losses _n_of;6:6 on _p;7:7 UAL named;8:10 _n_1 stock firm;12:12 health _n_1;0:0 Mr _n_1', '0:0 The _q;1:3 _n_1 stock decline;4:4 also _a_1;6:6 left _v_1;7:7 the _q;8:8 UAL named;9:9 board _n_of;10:10 in _p;11:11 a _q;0:0 The _q', "0:0 Although _x;2:2 may _v_modal;5:5 legally _a_1;6:8 _v_1 obligated sell;9:9 the _q;10:10 company _n_of;11:11 if _x_then;12:12 the _q;13:14 _n_1 buy- out;15:15 group _n_of;16:16 can't _v_modal;17:17 revive _v_cause;19:19 bid _n_1;21:21 may _v_modal;22:22 have _v_qmodal;24:24 explore _v_1;25:25 alternatives _n_to;26:26 if _x_then;27:27 the _q;28:28 buyers _n_of;29:29 come _v_back;31:31 with _p;32:32 a _q;33:33 bid _n_1;34:34 much _x_deg;35:35 lower _a_1;37:37 the _q;38:38 group _n_of;40:40 original _a_1;41:41 $ _n_1;42:42 300- card;44:44 share _n_of;44:44 share _a_p_per;0:0 Although _x", '0:0 The _q;1:1 takeover- _n_of;2:3 _n_1 stock traders;5:5 hoping _v_1;7:7 Mr _n_1;8:8 Davis named;9:9 or _c;10:10 one card;12:12 the _q;13:13 other _a_1;14:14 interested _a_in;15:15 parties _n_of-to;16:16 might _v_modal;17:17 re-emerge _v_1;17:17 re-emerge _re-_a_again;18:18 with _p;19:19 the _q;20:20 situation _n_1;21:21 in _p;22:22 disarray _n_unknown;23:23 or _c;25:25 the _q;26:26 board _n_of;27:27 might _v_modal;28:28 consider _v_1;29:29 a _q;0:0 The _q', '0:0 Meanwhile _p;1:1 Japanese _a_1;2:2 bankers _n_1;3:3 said _v_to;6:7 _a_1 still hesitant;8:8 about _p;9:9 accepting _v_1;10:10 Citicorp named;12:12 latest _a_for;0:0 Meanwhile _p', '0:0 Macmillan named;1:1 Inc _n_1;2:2 said _v_to;4:4 plans _v_1;5:5 a _q;6:6 public _a_1;7:7 offering _n_of;9:10 card 8.4 million;11:11 shares _n_of;12:12 of _p;14:15 named Berlitz International;16:16 Inc _n_1;17:17 unit _n_of;18:18 at _p;19:19 $ _n_1;20:20 19 card;21:21 to _p;22:22 $ _n_1;23:23 21 card;24:24 a _p_per;0:0 Macmillan named', '0:0 The _q;1:1 offering _n_of;2:2 for _p;3:3 the _q;4:5 _n_1 language school;6:6 unit _n_of;8:8 announced _v_to;10:11 named Robert Maxwell;12:12 chairman _n_of;13:13 and _c;14:15 _n_1 chief executive;16:16 officer _n_1;17:17 of _p;18:18 London- named;19:19 based _v_1;20:21 named Maxwell Communication;22:22 Corp _n_1;24:24 owns _v_1;0:0 The _q', '0:0 After _x_h;1:1 the _q;2:2 offering _n_of;4:4 completed _v_2;5:5 Macmillan named;7:7 own _v_1;8:8 about _x_deg;9:9 56 card;10:10 % _n_of;11:11 of _p;12:12 the _q;13:13 Berlitz named;14:14 common _a_for;15:15 stock _n_1;0:0 After _x_h', '0:1 card Five million;2:2 shares _n_of;5:5 offered _v_1;6:6 in _p;7:7 the _q;8:8 U.S named_n;9:9 and _c;10:11 card 3.4 million;12:12 additional _a_1;13:13 shares _n_of;16:16 offered _v_1;17:17 in _p;18:19 _a_1 concurrent international;20:20 offerings _n_of;21:21 outside _p;22:22 the _q;0:0 Five card', '0:1 named Goldman Sachs;2:3 _n_1 & Co;5:5 manage _v_1;6:6 the _q;0:0 Goldman named', '0:0 Macmillan named;1:1 said _v_to;2:2 Berlitz named;3:5 _v_for intends pay;6:6 quarterly _a_1;7:7 dividends _n_1;8:8 on _p;9:9 the _q;0:0 Macmillan named', '0:0 The _q;1:1 company _n_of;2:2 said _v_to;4:4 expects _v_1;6:6 pay _v_for;7:7 the _q;8:8 first ord;9:9 dividend _n_1;10:10 of _p;11:11 12.5 card;12:12 cents _n_1;13:13 a _p_per;14:14 share _n_of;15:15 in _p_temp;16:16 the _q;17:17 1990 yofc;18:18 first ord;0:0 The _q', '0:0 Berlitz named;2:2 borrow _v_from;3:3 an _q;4:4 amount _n_of;5:5 equal _a_to;8:8 expected _v_1;9:10 _n_1 net proceeds;12:12 the _q;13:13 offerings _n_of;14:14 plus _c;15:15 $ _n_1;16:17 card 50 million;18:18 in _p;19:19 connection _n_of;20:20 with _p;21:21 a _q;22:23 _n_1 credit agreement;24:24 with _p;0:0 Berlitz named', '0:0 The _q;1:1 total _n_of;2:2 borrowing _v_from;4:4 be _v_id;5:5 about _x_deg;6:6 $ _n_1;7:8 card 208 million;9:9 the _q;10:10 company _n_of;0:0 The _q', '0:0 Proceeds _n_1;1:1 from _p;2:2 the _q;3:3 borrowings _n_1;4:4 under _p;5:5 the _q;6:7 _n_1 credit agreement;10:10 used _v_1;12:12 pay _v_for;13:13 an _q;14:14 $ _n_1;15:16 card 80 million;17:18 _n_1 cash dividend;20:20 Macmillan named;21:21 and _c;23:23 lend _v_1;24:24 the _q;25:25 remainder _n_of;27:27 about _x_deg;28:28 $ _n_1;29:30 card 128 million;32:33 named Maxwell Communications;34:34 in _p;35:35 connection _n_of;36:36 with _p;37:37 a _q;38:38 promissory _a_1;0:0 Proceeds _n_1', '0:0 Proceeds _n_1;1:1 from _p;2:2 the _q;3:3 offering _n_of;6:8 _v_1 used repay;9:9 borrowings _n_1;10:10 under _p;11:11 the _q;13:13 term _n_of;13:13 term _long_a_1;14:14 parts _n_1;15:15 of _p;16:16 a _q;17:18 _n_1 credit agreement', '0:0 Berlitz named;3:3 based _v_1;4:4 in _p;5:6 named Princeton N.J;7:7 provides _v_1;8:8 language _n_1;9:9 instruction _n_of;10:10 and _c;11:11 translation _n_of;12:12 services _n_1;13:13 through _p;16:16 260 card;17:17 language _n_1;18:18 centers _n_of;19:19 in _p;20:20 25 card;0:0 Berlitz named', '0:0 In _p_state;1:1 the _q;2:2 past _a_1;3:3 five card;4:4 years _n_1;7:7 68 card;8:8 % _n_of;9:9 of _p;11:11 sales _n_of;14:14 outside _p;15:15 the _q;0:0 In _p_state', '0:0 Macmillan named;2:2 owned _v_1;3:3 Berlitz named;4:4 since _p;0:0 Macmillan named', '0:0 In _p_state;1:1 the _q;2:2 first ord;3:3 six card;4:4 months _n_1;5:5 of _p;6:6 this _q_dem;7:7 year _n_1;8:8 Berlitz named;9:9 posted _v_1;10:11 _n_1 net income;12:12 of _p;13:13 $ _n_1;14:15 card 7.6 million;16:16 on _p;17:17 sales _n_of;19:19 $ _n_1;20:21 card 106.2 million;22:22 compared _v_with;24:25 _n_1 net income;26:26 of _p;27:27 $ _n_1;28:29 card 8.2 million;30:30 on _p;31:31 sales _n_of;33:33 $ _n_1;34:35 card 90.6 million', '0:0 Right _x_deg;1:1 away _p;3:3 notice _v_1;4:4 the _q;5:5 following _a_1;6:6 things _n_of-about;7:7 about _p;8:8 a _q;9:9 Philip named;10:10 Glass _n_of;0:0 Right _x_deg', '1:1 attracts _v_1;2:2 people _n_of;3:3 with _p;4:4 funny _a_1;5:5 hair _n_1;6:6 (or _c;7:7 with _p;8:8 no _q;9:9 hair _n_1;11:12 _p in front;13:13 of _p;15:15 a _q;16:16 girl _n_1;17:17 with _p;18:18 spiked _v_1;19:19 locks _n_of;20:20 sat _v_1;21:21 beside _p;22:22 a _q;23:23 boy _n_1;1:1 attracts _v_1', '0:0 People _n_of;1:1 in _p;2:3 _n_of Glass houses;4:4 tend _v_1;6:6 look _v_seem-to;0:0 People _n_of', '0:0 And _c;1:1 if _x_then;2:2 still _a_1;3:3 conscious _a_of;4:4 at _p;5:5 the _q;6:8 _n_of evening end;10:10 notice _v_1;11:11 something _q;12:12 else _a_1;13:13 : _p_namely;14:14 The _q;15:15 audience _n_1;16:17 _a_1 at first;18:18 entranced _v_1;19:19 and _c;20:20 hypnotized _v_unknown;22:22 the _q;23:23 music _n_1;24:24 releases _v_1;26:27 _a_1 pent- up;28:28 feelings _n_of;29:29 in _p;30:30 collective _a_1;0:0 And _c', '1:1 sits _v_1;3:3 at _p;4:4 the _q;5:5 piano _n_1;6:6 and _c;1:1 sits _v_1', '0:0 And _c;0:0 And _c', '0:0 Either _q;1:1 one _n_1;2:2 likes _v_1;4:4 or _c;0:0 Either _q', '0:0 The _q;1:1 typical _a_of;2:2 Glass _n_of;3:3 audience _n_1;7:7 likely _a_1;10:10 composed _a_of;12:12 music _n_1;13:16 _n_of students teachers;0:0 The _q', '0:0 The _q;1:1 work _n_1;3:3 sounds _v_seem-to;4:4 like _p;5:5 Muzak _n_unknown;6:6 for _p;0:0 The _q', '1:1 success _n_1;3:3 easy _a_for;1:1 success _n_1', '0:0 Softly _a_1;1:1 introducing _v_to;2:2 and _c;3:3 explaining _v_to;5:5 pieces _n_of;6:6 Mr _n_1;7:7 Glass named;8:8 looks _v_seem-to;9:9 and _c;10:10 sounds _v_seem-to;12:12 like _p;13:13 a _q;14:14 shaggy _a_1;15:15 poet _n_1;16:16 describing _v_to;18:18 work _n_1;20:20 a _q;21:21 classical _a_1;22:22 pianist _n_1;23:23 playing _v_1;24:24 a _q;0:0 Softly _a_1', '1:1 is _v_id;2:2 music _n_1;3:3 for _p;4:4 people _n_of;6:8 _v_1 want hear;9:9 something _q;10:10 different _a_than-from;11:11 but _c;13:15 _v_1 want work;16:16 especially _x_deg;17:17 hard _a_for;18:18 at _p;19:19 the _q;1:1 is _v_id', '0:0 Far _x_deg;1:1 from _p;3:3 minimalist _a_unknown;4:4 the _q;5:5 music _n_1;6:6 unabatingly _a_unknown;7:7 torments _v_1;9:9 with _p;10:10 apparent _a_to;11:11 novelties _n_1;14:14 cleverly _a_1;15:15 disguised _v_1;16:16 in _p;17:17 the _q;18:18 simplicities _n_1;19:19 of _p;20:20 4/4 fraction;21:21 time _n_of;22:22 octave _n_1;23:23 intervals _n_of;24:24 and _c;25:25 ragtime _n_unknown;26:26 or _c;27:28 _n_1 gospel chord;0:0 Far _x_deg', '0:0 "Opening" named;1:1 (1981) card;2:2 from _p;3:3 Glassworks named;4:4 introduces _v_to;5:5 the _q;6:6 audience _n_1;8:8 the _q;9:9 Glass _n_of;10:10 technique _n_1;11:11 : _p_namely;12:12 Never _a_1;13:13 straying _v_from;15:15 far _a_1;16:16 from _p;17:17 the _q;18:18 piano _n_1;20:20 center _n_of;21:21 Mr _n_1;22:22 Glass named;23:23 works _v_1;24:24 in _p;25:25 the _q;26:26 two card;27:27 octaves _n_1;28:28 on _p;29:29 either _q;30:30 side _n_1;32:33 named middle C;34:34 and _c;36:36 fingers _n_1;37:37 seldom _a_1;38:38 leave _v_1;39:39 the _q;0:0 "Opening" named', '1:1 is _v_there;2:2 a _q;3:4 _a_1 recognizable musical;5:5 style _n_of;6:6 here _a_1;7:8 _c but not;9:9 a _q;10:10 particular _a_1;11:11 performance _n_1;1:1 is _v_there', '0:0 The _q;1:1 music _n_1;4:4 especially _a_1;5:5 pianistic; _a_unknown;6:6 indeed _a_1;9:9 hard _a_for;11:11 imagine _v_1;12:12 a _q;13:13 bad _a_at;14:14 performance _n_1;0:0 The _q', '0:0 Nothing _q;1:1 bravura named;2:2 no _q;3:3 arpeggios _n_unknown;4:4 no _q;5:5 ticklish _a_1;6:6 fingering _v_1;7:7 problems _n_of;8:8 challenge _v_to;9:9 the _q;0:0 Nothing _q', '1:1 hear _v_1;3:3 may _v_modal;4:4 think _v_1;5:5 inner _a_1;6:6 voices _n_1;7:7 but _c;9:9 all _a_1;10:10 seem _v_to;13:13 saying _v_1;14:14 the _q;15:15 same _a_as;1:1 hear _v_1', '0:0 With _p;1:2 named "Planet News,";3:3 music _n_1;4:6 _v_1 meant accompany;7:7 readings _n_of;9:10 named Allen Ginsberg;12:13 named "Wichita Vortex;14:14 Sutra," named;15:15 Mr _n_1;16:16 Glass named;17:17 gets _v_state;0:0 With _p', '1:1 hands _n_1;2:2 sit _v_1;3:3 farther _x_deg;4:4 apart _a_1;5:5 on _p;6:6 the _q;1:1 hands _n_1', '0:0 Seventh ord;1:1 chords _n_1;2:2 make _v_cause;4:4 feel _v_1;8:8 may _v_modal;9:9 break _v_1;11:11 a _q;12:12 (very _x_deg;13:13 slow) _a_1;14:14 improvisatory _a_unknown;0:0 Seventh ord', '0:0 The _q;1:1 chords _n_1;2:2 modulate _v_1;3:3 but _c;5:5 is _v_there;7:7 filigree _n_unknown;8:8 even _x_deg;9:9 though _x;11:11 fingers _n_1;12:14 _v_1 begin wander;15:15 over _p_dir;18:18 the _q;0:0 The _q', '0:0 Contrasts _n_1;1:1 predictably _a_1;2:2 accumulate _v_1;3:3 : _p_namely;4:4 First _a_1;5:5 the _q;6:6 music _n_1;8:8 loud _a_1;9:9 then _c;11:11 becomes _v_id;12:12 soft _a_1;13:13 then _c;15:15 realize) _v_1;17:17 becomes _v_id;18:19 _a_1 louder again', '0:0 "The _q;1:1 Fourth ord;2:3 named Knee Play,";4:4 an _q;5:5 interlude _n_1;6:6 from _p;7:8 named "Einstein on;9:9 the _q;10:10 Beach," named;12:12 like _p;13:13 a _q;14:14 toccata _n_unknown;15:15 but _c;18:18 seem _v_to;20:20 move _v_1;21:21 much _x_deg;22:22 beyond _p;25:25 hand _q;26:26 ground _n_1;27:27 in _p;28:28 "Three card;29:30 named Blind Mice."', '0:0 When _x_subord;1:1 Mr _n_1;2:2 Glass named;3:3 decides _v_1;5:5 get _v_state;6:6 really _x_deg;7:7 fancy _a_1;9:9 crosses _v_1;11:11 hands _n_1;12:12 and _c;13:13 hits _v_1;14:14 a _q;15:15 resonant _a_1;16:16 bass _n_1;17:17 note _n_of;18:18 with _p;20:20 right _a_1;0:0 When _x_subord', '1:1 does _v_1;2:2 this _q_dem;3:3 in _p;4:5 _x_deg at least;6:6 three card;9:9 solo _n_1;1:1 does _v_1', '1:1 might _v_modal;2:2 call _v_name;4:4 a _q;5:5 leitmotif _n_unknown;6:6 or _c;7:7 a _q;8:8 virtuoso _a_1;1:1 might _v_modal', '3:3 the _q;4:4 typical _a_of;5:5 Glasswork _n_1;7:7 ended _startup_n_1;10:11 _a_1 also often;12:12 multiple _a_1;13:13 in _p;3:3 the _q', '0:1 named "Mad Rush";2:2 began _v_1;4:4 life _n_of;5:5 as _p;6:6 the _q;7:7 accompaniment _n_1;8:8 to _p;9:9 the _q;10:11 named Dalai Lama;13:13 first ord;14:14 public _a_1;15:15 address _n_1;16:16 in _p;17:17 the _q;18:18 U.S named_n;19:19 when _x_subord;20:20 Mr _n_1;21:21 Glass named;22:22 played _v_1;24:24 on _p;25:25 the _q;26:26 organ _n_1;27:27 at _p;28:29 named New York;31:31 Cathedral _n_1;32:32 of _p;33:33 St _n_1;34:34 John named;35:35 the _q;0:0 "Mad named', '0:0 Later _a_for;3:3 performed _v_1;4:4 on _p;5:6 named Radio Bremen;7:7 in _p;8:8 Germany named;9:10 _c and then;11:12 named Lucinda Childs;13:13 took _v_1;15:15 for _p;16:16 one card;19:19 dance _n_1;0:0 Later _a_for', '0:0 The _q;1:1 point _n_of;2:2 is _v_nv;4:4 any _q;5:5 piece _n_of;6:6 can _v_modal;8:8 used _v_as;10:11 _n_1 background music;12:12 for _p;13:13 virtually _x_deg;0:0 The _q', '0:0 The _q;1:1 evening _n_of;2:2 ended _v_1;3:3 with _p;4:4 Mr _n_1;5:5 Glass named;7:7 "Metamorphosis," _n_unknown;8:8 another _q;9:9 multiple _a_1;0:0 The _q', '0:0 Parts _n_1;1:2 card 1 2;3:3 and _c;4:4 5 card;5:5 come _v_1;6:6 from _p_dir;7:7 the _q;8:8 soundtrack _n_1;9:9 of _p;10:11 named Errol Morris;13:13 acclaimed _a_1;14:14 film _n_1;15:16 named "The Thin;17:18 named Blue Line,";19:19 and _c;20:20 the _q;21:21 two card;22:22 other _a_1;23:23 parts _n_1;24:24 from _p;25:25 incidental _a_unknown;26:26 music _n_1;27:27 to _p;28:28 two card;29:29 separate _a_from;30:30 dramatizations _n_unknown;31:31 of _p;32:32 the _q;33:33 Kafka named;34:34 story _n_of;36:36 the _q;37:37 same _a_as;0:0 Parts _n_1', '0:0 When _x_subord;1:1 used _v_as;3:3 background _n_1;4:4 in _p;5:5 this _q_dem;6:6 way _n_of;7:7 the _q;8:8 music _n_1;9:9 has _v_1;10:10 an _q;11:11 appropriate _a_to;12:12 eeriness _n_1;13:13 as _x_subord;15:15 a _q;17:17 note _n_1;17:17 note card;18:18 phrase _n_1;19:19 a _q;20:20 descending _v_1;21:21 minor _a_1;22:22 third ord;23:23 accompanies _v_1;24:24 the _q;25:26 _a_1 seemingly endless;27:27 litany _n_unknown;28:28 of _p;29:29 reports _n_of;30:30 interviews _n_1;31:31 and _c;32:32 confessions _n_1;33:33 of _p;34:34 witnesses _n_1;35:35 in _p;36:36 the _q;37:37 Morris named;0:0 When _x_subord', '0:0 Served _v_up;2:2 as _p;3:3 a _q;4:4 solo _n_1;6:6 the _q;7:7 music _n_1;8:8 lacks _v_1;9:9 the _q;10:10 resonance _n_1;11:11 provided _v_1;13:13 a _q;14:14 context _n_1;15:15 within _p;16:16 another _q;0:0 Served _v_up', '0:0 Admirers _n_1;1:1 of _p;2:2 Mr _n_1;3:3 Glass named;4:4 may _v_modal;5:5 agree _v_on;6:6 with _p;7:7 the _q;8:8 critic _n_of;9:10 named Richard Kostelanetz;12:12 sense _n_of;14:14 the _q;15:15 1974 yofc;16:16 "Music _n_1;17:17 in _p;18:18 Twelve card;19:19 Parts" _n_1;22:22 encyclopedic _a_1;23:23 and _c;24:24 weighty _a_1;26:26 "The _q;28:28 Tempered _v_1;28:28 Tempered _long_a_1;0:0 Admirers _n_1', '0:0 But _c;1:1 while _x;2:2 making _v_1;3:3 the _q;4:4 obvious _a_to;5:5 point _n_of;7:7 both _q;7:7 both card;8:8 composers _n_of;9:9 develop _v_cause;10:10 variations _n_of;11:11 from _p;12:12 themes _n_of;13:13 this _q_dem;14:14 comparison _n_1;15:15 ignores _v_1;16:16 the _q;17:18 _a_1 intensely claustrophobic;19:19 nature _n_1;20:20 of _p;21:21 Mr _n_1;22:22 Glass named;0:0 But _c', '1:2 _a_1 supposedly austere;3:3 minimalism _n_unknown;4:4 overlays _v_1;5:5 a _q;6:6 bombast _n_unknown;8:8 makes _v_cause;9:9 one _n_1;10:10 yearn _v_unknown;12:12 the _q;13:13 astringency _n_unknown;14:14 of _p;15:15 neoclassical _a_1;16:16 Stravinsky named;17:17 the _q;18:18 genuinely _a_of;19:19 radical _a_1;20:20 minimalism _n_unknown;21:21 of _p;22:22 Berg named;23:23 and _c;24:24 Webern named;25:25 and _c;27:27 in _p;28:28 retrospect _n_unknown;29:29 even _a_1;30:30 seems _v_to;31:31 like _p;32:32 concision _n_1;33:33 in _p;1:1 supposedly _a_1', '0:0 Mr _n_1;1:1 Spiegelman named;2:2 is _v_id;3:3 professor _n_of;5:5 English named;6:6 at _p;7:8 named Southern Methodist;9:9 University named;10:10 and _c;11:11 editor _n_1;12:12 of _p;13:13 the _q;14:15 named Southwest Review', '0:0 Honeywell named;1:1 Inc _n_1;2:2 said _v_to;4:4 hopes _v_1;6:6 complete _v_2;7:7 shortly _a_1;8:8 the _q;9:9 first ord;11:11 two card;12:14 _n_of sales shares;15:15 in _p;17:18 _a_1 Japanese joint;19:19 venture _n_1;20:21 named Yamatake- Honeywell;22:22 for _p;23:23 about _x_deg;24:24 $ _n_1;25:26 card 280 million', "0:0 The _q;1:1 company _n_of;2:2 wouldn't _v_modal;3:3 disclose _v_to;4:4 the _q;5:5 buyer _n_of;7:7 the _q;8:8 initial _a_1;9:9 16 card;10:10 % _n_of;0:0 The _q", '0:0 Proceeds _n_1;1:1 of _p;2:2 the _q;3:3 sale _n_of;4:4 expected _v_1;7:7 completed _v_2;8:8 next _a_1;9:9 week _n_1;10:10 would _v_modal;12:14 _v_1 used repurchase;14:14 repurchase _re-_a_again;18:19 card 10 million;20:20 shares _n_of;22:22 Honeywell named;23:23 stock _n_1;24:24 the _q;25:25 company _n_of;0:0 Proceeds _n_1', '0:0 Honeywell named;1:1 said _v_to;4:4 negotiating _v_1;5:5 the _q;6:6 sale _n_of;8:8 a _q;9:9 second ord;10:10 stake _n_1;11:11 in _p;12:13 named Yamatake- Honeywell;14:14 but _c;15:15 indicated _v_1;17:17 intends _v_for;19:19 hold _v_1;20:21 _x_deg at least;22:22 20 card;23:23 % _n_of;24:24 of _p;25:25 the _q;26:26 joint _a_1;27:29 _n_1 venture stock;30:30 long _a_1;0:0 Honeywell named', '0:0 A _q;1:1 20 card;2:2 % _n_of;3:3 stake _n_1;4:4 would _v_modal;5:5 allow _v_1;6:6 Honeywell named;8:8 include _v_1;9:9 Yamatake named;10:10 earnings _n_1;11:11 in _p;0:0 A _q', '0:0 Honeywell named;1:1 previously _p;2:2 said _v_to;4:4 intended _v_for;6:6 reduce _v_1;8:8 holding _n_1;9:9 in _p;10:10 the _q;11:11 Japanese _a_1;12:12 concern _n_1;13:13 as _p;14:16 _q part a;17:17 restructuring _v_1;18:18 plan _n_1;20:20 also _a_1;21:21 calls _v_for;23:23 a _q;24:24 reduction _n_of;26:26 dependence _n_1;27:27 on _p;28:28 weapons _n_1;0:0 Honeywell named', '0:0 Yesterday _a_1;1:1 a _q;2:2 spokeswoman _n_unknown;3:3 said _v_to;4:4 the _q;5:5 company _n_of;7:7 "pleased _a_with;10:10 progress" _n_1;11:11 in _p;12:12 that _q_dem;13:13 regard _n_1;14:14 and _c;15:17 _v_1 "hopes provide;18:18 additional _a_1;19:19 details _n_1;0:0 Yesterday _a_1', '0:0 Honeywell named;1:1 said _v_to;3:3 Defense _n_1;4:4 and _c;5:6 named Marine Systems;7:7 group _n_of;8:8 incurred _v_1;9:9 delays _n_1;10:10 in _p;11:11 shipping _v_cause;12:12 some _q;13:13 undisclosed _v_1;13:13 undisclosed _un-_a_rvrs;14:14 contracts _n_1;15:15 during _p;16:16 the _q;17:17 third ord;18:18 quarter _n_temp;19:19 resulting _v_in;21:21 lower _a_1;22:22 operating _v_1;23:23 profit _n_1;24:24 for _p;25:25 that _q_dem;0:0 Honeywell named', '0:0 Overall _a_1;1:1 Honeywell named;2:2 reported _v_to;3:3 earnings _n_1;4:4 of _p;5:5 $ _n_1;6:7 card 74.4 million;8:8 or _c;9:9 $ _n_1;10:10 1.73 card;11:11 a _p_per;12:12 share _n_of;13:13 for _p;14:14 the _q;15:15 three card;16:16 months _n_1;17:17 ended _v_cause;18:18 Oct mofy;19:19 1 dofm;20:20 compared _v_with;22:22 a _q;23:23 loss _n_of;25:25 $ _n_1;26:27 card 41.4 million;28:28 or _c;29:29 98 card;30:30 cents _n_1;31:31 a _p_per;32:32 share _n_of;33:33 a _q;34:34 year _n_1;0:0 Overall _a_1', '0:0 The _q;1:1 previous _a_1;2:4 _n_of period results;5:5 included _v_1;6:6 a _q;7:7 $ _n_1;8:9 card 108 million;10:10 pretax _n_1;10:10 pretax _pre-_a_ante;11:11 charge _n_of;12:12 related _v_to;14:14 unrecoverable _a_unknown;15:16 _n_1 contract costs;17:17 and _c;18:18 a _q;19:19 $ _n_1;20:21 card 12.3 million;22:22 pretax _n_1;22:22 pretax _pre-_a_ante;23:23 gain _n_1;24:24 on _p;25:26 _n_1 real estate;0:0 The _q', '0:0 Sales _n_of;1:1 for _p;2:2 the _q;3:3 latest _a_for;4:4 quarter _n_temp;6:6 flat _a_1;7:7 at _p;8:8 $ _n_1;9:10 card 1.72 billion', '0:0 For _p;1:1 the _q;2:2 nine card;3:3 months _n_1;4:4 Honeywell named;5:5 reported _v_to;6:6 earnings _n_1;7:7 of _p;8:8 $ _n_1;9:10 card 212.1 million;11:11 or _c;12:12 $ _n_1;13:13 4.92 card;14:14 a _p_per;15:15 share _n_of;16:16 compared _v_with;18:18 earnings _n_1;19:19 of _p;20:20 $ _n_1;21:22 card 47.9 million;23:23 or _c;24:24 $ _n_1;25:25 1.13 card;26:26 a _p_per;27:27 share _n_of;28:28 a _q;29:29 year _n_1;0:0 For _p', '0:0 Sales _n_of;1:1 declined _v_1;2:2 slightly _a_1;3:3 to _p;4:4 $ _n_1;5:6 card 5.17 billion', '0:1 _a_1 Once again;3:3 editorial _a_1;4:4 page _n_1;5:5 misstates _v_unknown;6:6 the _q;7:7 law _n_1;8:8 to _x;9:9 conform _v_to;12:12 almost _x_deg;13:13 beatific _a_unknown;0:0 Once _a_1', '0:0 This _q_dem;1:1 statement _n_of;2:2 surely _a_1;3:3 buttresses _v_1;5:5 editorial _a_1;6:6 viewpoint _n_1;8:8 environmental _a_1;9:9 protection _n_of;11:12 _a_1 generally silly;13:13 or _c;14:14 excessive _a_1;15:15 but _c;18:18 simply _a_1;0:0 This _q_dem', '0:0 The _q;1:2 named Clean Water;3:3 Act named;4:4 contains _v_1;5:5 no _q;6:6 "legal _a_1;7:7 standard" _n_1;8:8 of _p;9:9 zero _n_1;0:0 The _q', '1:1 requires _v_of;3:3 "discharges _n_of;5:5 pollutants" _n_1;6:6 into _p;7:7 the _q;8:8 "waters _n_1;9:9 of _p;10:10 the _q;11:12 named United States";14:14 authorized _v_1;16:16 permits _n_1;18:18 reflect _v_1;19:19 the _q;20:20 effluent _a_unknown;21:21 limitations _n_1;22:22 developed _v_cause;23:23 under _p;24:24 section _n_of;1:1 requires _v_of', '1:1 may _v_modal;2:2 be _v_id;3:3 the _q;4:4 problems _n_of;5:5 with _p;6:6 this _q_dem;7:7 system _n_of;9:9 scarcely _a_1;10:10 reflects _v_1;11:11 "zero _n_1;12:12 risk" _n_of;13:13 or _c;14:14 "zero _n_1;1:1 may _v_modal', '0:0 Perhaps _a_1;1:1 Mr _n_1;2:2 Greve named;4:4 confused _a_with;6:6 Congress named;8:8 meaningless _a_1;9:9 statement _n_of;11:11 "the _q;12:13 named national goal";14:14 in _p;15:15 section named;16:16 101 card;18:18 indeed _a_1;19:19 calls _v_for;21:21 the _q;22:22 elimination _n_1;23:23 of _p;24:24 discharges _n_of;26:26 by _p_temp;27:27 1985 yofc;28:28 no _x_deg;0:0 Perhaps _a_1', '0:0 This _q_dem;1:1 fatuous _a_unknown;2:2 statement _n_of;5:5 taken _v_1;6:6 seriously _a_1;7:7 when _x_subord;8:8 enacted _v_1;9:9 in _p_temp;10:10 1972 yofc;11:11 and _c;12:12 should _v_modal;14:14 now _a_1;16:16 confused _v_with;18:18 the _q;19:19 operative _a_1;20:20 provisions _n_1;21:21 of _p;22:22 the _q;0:0 This _q_dem', '0:1 named Robert J;0:0 Robert named', '0:0 To _x;1:1 hear _v_1;2:2 advertisers _n_unknown;3:3 tell _v_1;5:5 the _q;6:6 magazine _n_1;7:7 just _a_1;9:9 kept _v_up;11:11 with _p;12:12 the _q;0:0 To _x', '0:0 Despite _p;1:1 renewed _a_1;2:2 interest _n_in;3:3 by _p;4:4 the _q;5:5 public _n_1;6:6 in _p;7:7 such _q;8:8 topics _n_of;9:9 as _p;10:10 the _q;11:11 environment _n_1;12:12 and _c;13:13 the _q;14:15 named Third World;19:19 able _a_1;21:21 shake _v_1;23:23 reputation _n_1;24:24 as _x_subord;25:25 a _q;26:27 _n_1 magazine boys;28:30 _v_1 like flip;32:32 in _p;33:33 search _n_of;35:35 topless _a_unknown;36:36 tribe named;0:0 Despite _p', '0:0 Worse _a_1;2:2 lagged _v_1;3:3 behind _p;4:4 competitors _n_1;5:5 in _p;6:6 offering _v_1;8:8 standard _n_1;8:8 standard _high_a_1;9:9 gimmicks _n_unknown;10:10 from _p;11:11 regional _a_1;12:12 editions _n_1;13:13 to _p;14:14 discounts _n_1;15:15 for _p;16:16 frequent _a_1;0:0 Worse _a_1', '0:0 But _c;1:1 now _a_1;2:2 the _q;3:3 magazine _n_1;5:5 attempting _v_1;7:7 fight _v_off;9:9 with _p;10:10 an _q;11:11 ambitious _a_1;12:12 plan _n_1;13:13 including _v_1;14:14 a _q;15:15 revamped _v_1;15:15 revamped _re-_a_again;16:17 _n_of sales strategy;18:18 and _c;19:19 a _q;20:21 _a_1 surprisingly aggressive;22:23 _n_1 ad campaign', '0:0 Advertisers _n_unknown;2:2 think _v_of;4:4 the _q;5:5 magazine _n_1;6:6 first _a_1;7:7 says _v_to;8:9 named Joan McCraw;11:11 joined _v_1;12:12 in _p_temp;13:13 April mofy;14:14 as _p_nbar;15:15 national _a_1;16:16 advertising _n_1;0:0 Advertisers _n_unknown', '2:4 _v_1 want do;5:5 is _v_nv;6:6 take _v_1;7:7 a _q;9:9 aggressive _a_1;2:2 want _v_1', '0:0 People _n_of;2:2 believe _v_1;5:5 in _p;6:6 tune _n_1;7:7 with _p;8:8 the _q;9:9 marketplace _n_of;10:10 and _c;11:11 in _p;0:0 People _n_of', '0:0 The _q;2:2 year- _n_1;2:2 year- card;3:3 old _a_1;4:4 magazine _n_1;6:6 never _a_1;7:7 had _v_qmodal;9:9 woo _v_unknown;10:10 advertisers _n_unknown;11:11 with _p;12:12 quite _x;15:15 fervor _n_unknown;0:0 The _q', '0:0 The _q;1:1 magazine _n_1;2:2 had _v_1;4:4 best _a_at-for-of;5:5 year _n_1;6:6 yet _a_1;7:7 in _p_temp;8:8 1988 yofc;9:9 when _x_subord;11:11 celebrated _v_1;13:13 centennial _n_1;14:14 and _c;15:15 racked _v_up;17:17 a _q;18:18 17 card;19:19 % _n_of;20:20 gain _n_1;21:21 in _p;22:23 _n_1 ad pages;24:24 to _p;0:0 The _q', '0:0 But _c;1:1 this _q_dem;2:2 year _n_1;3:3 when _x_subord;4:4 the _q;5:5 hullabaloo named;6:6 surrounding _v_1;8:8 centennial _n_1;9:9 died _v_1;10:11 _a_1 so too;13:13 some _q_indiv;14:14 advertiser _n_unknown;0:0 But _c', '0:0 The _q;1:1 reason _n_for;2:3 _n_1 ad executives;4:4 say _v_to;5:5 is _v_nv;7:7 the _q;8:8 entire _a_1;9:10 _n_1 magazine business;13:13 soft _a_1;15:15 and _c;16:17 named National Geographic;18:18 has _v_1;19:19 some _q;20:20 quirks _n_1;22:22 make _v_cause;24:25 _a_1 especially unattractive;26:26 during _p;27:27 a _q;28:28 soft _a_1;0:0 The _q', '0:0 Perhaps _x_deg;1:1 the _q;2:2 biggest _a_1;4:4 those _q_dem;5:5 factors _n_1;6:6 is _v_id;8:8 high _a_1;9:9 ad _n_1;10:10 prices _n_of;12:12 $ _n_1;13:13 130,000 card;14:14 for _p;15:15 a _q;17:17 color _n_1;17:17 color card;18:18 page _n_1;19:19 vs _c;20:20 $ _n_1;21:21 47,000 card;22:22 for _p;23:23 the _q;24:24 Smithsonian named;25:25 a _q;26:26 comparable _a_1;27:27 publication _n_1;28:28 with _p;29:29 a _q;30:30 far _x_deg;31:31 smaller _a_1;0:0 Perhaps _x_deg', '0:0 When _x_subord;1:2 _n_1 ad dollars;4:4 tight _a_1;5:5 the _q;6:6 high _a_1;7:8 _n_1 page cost;9:9 is _v_id;10:10 a _q;11:11 major _a_1;12:12 deterrent _n_unknown;13:13 for _p;14:14 advertisers _n_unknown;16:16 generally _a_1;17:19 _v_1 want appear;20:20 regularly _a_1;21:21 in _p;22:22 a _q;23:23 publication _n_1;24:24 or _c;26:27 _a_1 at all', '0:0 To _x;1:1 combat _v_1;2:2 that _q_dem;3:3 problem _n_of;4:5 named National Geographic;6:6 like _p;7:7 other _a_1;8:8 magazines _n_1;9:10 _v_1 began offering;11:11 regional _a_1;12:12 editions _n_1;13:13 allowing _v_1;14:14 advertisers _n_unknown;16:16 appear _v_1;17:17 in _p;18:18 only _x_deg;19:19 a _q;20:20 portion _n_of;23:23 magazines _n_1;25:26 _a_1 for example;27:27 ads _n_1;28:28 can _v_modal;29:29 run _v_1;30:30 only _x_deg;31:31 in _p;32:32 the _q;33:33 magazines _n_1;34:34 sent _v_1;35:35 to _p;36:36 subscribers _n_to;37:37 in _p;38:38 the _q;39:39 largest _a_1;40:40 25 card;0:0 To _x', '0:0 Time named;1:1 magazine _n_1;2:3 _p for example;4:4 has _v_1;7:7 100 card;8:8 separate _a_from;9:9 editions _n_1;10:10 going _v_1;11:11 to _p;12:12 different _a_than-from;13:13 regions _n_of;14:14 top _a_1;15:15 management _n_1;16:16 and _c;17:17 other _a_1;0:0 Time named', '0:0 Another _q;1:1 sticking _v_1;2:2 point _n_of;3:3 for _p;4:4 advertisers _n_unknown;5:5 was _v_id;6:7 named National Geographic;9:9 tradition _n_1;10:10 of _p;11:11 lumping _v_down;13:13 ads _n_1;14:14 together _p;15:15 usually _x_deg;16:16 at _p;17:17 the _q;18:18 beginning _v_1;19:19 or _c;20:20 end _n_of;22:22 the _q;23:23 magazine _n_1;24:25 _c rather than;26:26 spreading _v_out;27:27 ads _n_1;29:29 among _p;31:31 articles _n_of;32:32 as _x_subord;33:33 most _q;0:0 Another _q', '0:0 But _c;1:1 Ms _n_1;2:2 McCraw named;3:3 says _v_to;4:4 the _q;5:5 magazine _n_1;0:0 But _c', '1:1 now _a_1;2:2 offers _v_1;3:3 30 card;4:4 regional _a_1;5:5 editions _n_1;7:7 very _x_deg;8:8 recently _a_1;9:10 _v_1 began running;11:11 ads _n_1;12:13 _p adjacent to;14:14 articles _n_of;15:15 and _c;19:19 beefing _v_up;22:22 sales _n_of;1:1 now _a_1', '0:0 And _c;2:2 just _a_1;3:3 launched _v_cause;4:4 a _q;5:5 promotional _a_1;6:6 campaign _n_1;8:8 tell _v_1;9:10 _n_1 chief executives;11:11 marketing _v_1;12:12 directors _n_of;13:13 and _c;14:15 _n_1 media executives;16:16 just _x_deg;0:0 And _c', '0:0 The _q;1:1 campaign _n_1;2:2 created _v_1;4:5 named Omnicom Group;7:8 named DDB Needham;9:9 agency _n_1;10:10 takes _v_of-i;11:11 advantage _n_i;13:13 the _q;14:14 eye- _n_1;15:15 catching _v_1;16:16 photography _n_1;18:19 named National Geographic;21:21 known _v_as;0:0 The _q', '0:0 In _p_state;1:1 one card;2:2 ad _n_1;3:3 a _q;4:4 photo _n_1;5:5 of _p;6:6 the _q;7:7 interior _n_1;8:8 of _p;9:9 the _q;10:11 named Sainte- Chapelle;12:12 in _p;13:13 Paris named;15:15 paired _v_1;16:16 with _p;17:17 the _q;18:18 headline _n_1;19:19 "The _q;20:20 only _a_1;21:21 book _n_of;23:27 _v_1 respected accept;0:0 In _p_state', '0:0 Another _q;1:1 ad _n_1;2:2 pictures _v_1;3:3 a _q;4:4 tree _n_of;5:5 ant _n_1;6:6 magnified _v_1;7:7 80 card;8:8 times _n_of;9:9 with _p;10:10 the _q;11:11 headline _n_1;12:12 "For _p;13:13 impact named;14:14 far _x_deg;15:15 beyond _p;17:17 size _n_1;18:18 consider _v_1;20:20 regional _a_1;0:0 Another _q', '0:0 Ms _n_1;1:1 McCraw named;2:2 says _v_to;4:4 wants _v_1;5:5 the _q;6:6 campaign _n_1;8:8 help _v_to;9:9 attract _v_1;10:10 advertisers _n_unknown;11:11 in _p;12:12 10 card;13:13 categories _n_of;14:14 including _v_1;15:16 _a_1 corporate financial;17:17 services _n_1;18:18 consumer _n_of;19:20 _n_1 electronics insurance;21:21 and _c;0:0 Ms _n_1', '1:1 goal _n_1;2:2 : _v_id;4:4 top _v_1;5:5 300 card;6:7 _n_1 ad pages;8:8 in _p_temp;9:9 1990 yofc;10:11 _p up from;12:12 about _x_deg;13:13 274 card;14:14 this _q_dem;1:1 goal _n_1', '0:0 Whether _x;2:2 can _v_modal;3:3 meet _v_1;4:4 that _q_dem;5:5 ambitious _a_1;6:6 goal _n_1;8:8 still _a_1;9:10 _x_deg far from;0:0 Whether _x', '0:0 "The _q;1:2 _n_1 ad campaign;4:4 meant _v_1;6:6 contemporize _v_unknown;7:7 the _q;8:8 thought _n_of;10:11 _a_1 National Geographic,";0:0 "The _q', "1:1 want _v_1;4:4 be _v_id;5:5 a _q;6:6 '90s _n_unknown;7:7 kind _n_of-n;1:1 want _v_1", '0:1 named WCRS Group;2:2 hopes _v_1;4:4 announce _v_to;5:5 perhaps _x_deg;6:6 today _a_1;7:7 an _q;8:8 agreement _n_1;10:10 sell _v_1;11:11 the _q;12:12 majority _n_of;15:15 ad _n_1;16:16 unit _n_of;18:18 Paris- named;19:19 based _v_1;20:20 Eurocom named;21:21 a _q;22:22 European _a_1;23:24 _n_1 ad executive;0:0 WCRS named', '0:0 WCRS named;3:3 in _p;4:4 discussions _n_1;5:5 with _p;6:6 Eurocom named;7:7 for _p;8:8 several _a_1;0:0 WCRS named', '0:0 However _a_1;1:1 when _x_subord;2:2 negotiations _n_of;3:3 bogged _v_down;5:5 recently _a_1;6:6 WCRS named;8:9 _n_1 chief executive;10:11 named Peter Scott;12:12 met _v_1;13:13 in _p;14:14 Paris named;15:15 with _p;16:16 another _q;17:17 French _a_1;18:18 firm _n_1;19:20 named Boulet Dru;21:22 named Dupuy Petit;23:23 or _c;0:0 However _a_1', '0:1 _p According to;2:2 the _q;3:3 executive _n_1;4:4 BDDP named;6:6 involvement _n_in;7:7 prompted _v_1;8:8 renewed _a_1;9:9 vigor _n_1;10:10 in _p;11:11 the _q;12:13 named WCRS- Eurocom;14:14 talks _n_of-on;15:15 and _c;16:16 the _q;17:17 two card;18:18 agencies _n_1;20:20 hoping _v_1;22:22 hammer _v_out;24:24 details _n_1;25:25 by _p_temp;0:0 According _p', "0:0 Executives _n_1;1:1 of _p;2:2 the _q;3:3 two card;4:4 agencies _n_1;5:5 couldn't _v_modal;7:7 reached _v_1;8:8 last _a_1;0:0 Executives _n_1", '0:0 NEW _a_1;1:1 ACCOUNT _n_of;2:2 : _v_id;6:6 Co _n_1;7:7 Cincinnati named;8:8 awarded _v_1;9:9 the _q;10:10 ad _n_1;11:14 _n_of accounts line;16:16 Professional _a_1;17:17 Crisco _n_unknown;18:18 vegetable _n_1;19:19 shortening _v_1;20:20 and _c;21:22 _n_1 oil products;24:25 named Northlich Stolley;26:27 named LaWarre Cincinnati', '0:0 Billings _n_1;0:0 Billings _n_1', '0:0 Professional _a_1;1:1 Crisco named;2:2 products _n_1;4:4 specially _a_1;5:5 made _v_1;6:6 for _p;7:7 the _q;8:8 foodservice _n_unknown;0:0 Professional _a_1', '1:1 was _v_id;2:3 _n_1 executive vice;4:5 _n_of president director;7:7 broadcast _n_1;1:1 was _v_id', '0:0 The _q;1:2 named Commodity Futures;3:4 named Trading Commission;5:5 plans _v_1;7:7 restrict _v_to;8:8 dual _a_1;9:9 trading _v_1;10:10 on _p;11:12 _n_1 commodity exchanges;13:13 a _q;14:14 move _n_1;15:15 almost _a_1;16:16 certain _a_of;18:18 infuriate _v_1;19:20 _n_1 exchange officials;21:21 and _c;0:0 The _q', '0:0 The _q;1:1 CFTC named;2:5 _v_to said propose;6:6 the _q;7:7 restrictions _n_of;8:8 after _p;9:9 the _q;10:10 release _n_of;12:12 a _q;13:13 study _n_of;15:15 shows _v_1;17:17 economic _a_1;18:18 benefit _n_of;19:19 resulting _v_from;21:21 dual _a_1;22:22 trading _v_1;23:23 and _c;24:24 cites _v_for;25:25 "problems" _n_of;26:26 associated _v_with;28:28 the _q;0:0 The _q', '0:0 Dual _a_1;1:2 _v_1 trading gives;3:3 an _q;4:5 _n_1 exchange trader;6:6 the _q;7:7 right _n_1;9:9 trade _v_1;11:11 for _p;13:13 own _a_1;14:14 account _n_of;15:15 and _c;16:16 for _p;0:0 Dual _a_1', '0:0 While _x;2:2 specifically _a_to;3:3 mentioned _v_to;4:4 in _p;5:5 the _q;6:6 FBI named;7:7 charges _n_of;8:8 dual _a_1;9:9 trading _v_1;10:10 became _v_id;11:11 a _q;12:12 focus _n_1;13:13 of _p;14:14 attempts _n_1;16:16 tighten _v_cause;17:18 _n_1 industry regulations', '0:0 Critics _n_of;1:1 contend _v_1;3:3 traders _n_1;5:6 _v_1 putting buying;7:7 or _c;8:8 selling _v_1;9:9 for _p;11:11 own _a_1;12:12 accounts _n_of;13:14 _p ahead of;15:15 other _a_1;16:16 traders _n_1;18:19 _n_of customer orders', '0:0 Traders _n_1;2:2 likely _a_1;4:4 oppose _v_1;5:5 such _q;6:6 restrictions _n_of;7:7 because _x;8:8 dual _a_1;9:10 _v_1 trading provides;11:11 a _q;12:12 way _n_of;14:14 make _v_1;15:15 money _n_1;16:16 in _p;17:17 slower _a_1;18:18 markets _n_1;21:21 is _v_there;22:22 a _q;23:25 _n_of shortage customer;0:0 Traders _n_1', '0:0 The _q;1:1 exchanges _n_1;2:2 contend _v_1;4:4 dual _a_1;5:5 trading _v_1;6:6 improves _v_cause;7:7 liquidity _n_1;8:8 in _p;9:9 the _q;10:10 markets _n_1;11:11 because _x;12:12 traders _n_1;13:13 can _v_modal;14:14 buy _v_1;15:15 or _c;16:16 sell _v_1;17:17 even _x_deg;18:18 when _x_subord;21:21 have _v_prd;22:22 a _q;23:24 _n_of customer order;25:25 in _p;0:0 The _q', '0:0 The _q;1:1 exchanges _n_1;2:2 say _v_to;3:3 liquidity _n_1;4:4 becomes _v_id;5:5 a _q;6:6 severe _a_1;7:7 problem _n_of;8:8 for _p;9:9 thinly _a_1;10:10 traded _v_1;11:11 contracts _n_1;12:13 _p such as;14:14 those _q_dem;15:15 with _p;16:16 a _q;17:17 long _a_1;18:18 time _n_of;19:19 remaining _v_1;20:20 before _p;0:0 The _q', '0:0 The _q;1:1 CFTC named;2:2 may _v_modal;3:3 take _v_of-i;4:4 those _q_dem;5:5 arguments _n_1;7:7 account _n_i;8:8 by _p_means;9:9 allowing _v_1;10:10 exceptions _n_to;0:0 The _q', '0:0 The _q;1:1 agency _n_1;3:3 cite _v_for;4:4 specific _a_to;5:5 situations _n_1;7:7 dual _a_1;8:8 trading _v_1;9:9 might _v_modal;11:11 allowed _v_1;12:12 but _c;13:13 smaller _a_1;14:14 exchanges _n_1;15:15 or _c;16:16 contracts _n_1;18:18 need _v_1;19:19 additional _a_1;20:20 liquidity _n_1;22:22 expected _v_1;0:0 The _q', '0:0 The _q;1:1 study _n_of;2:2 by _p;3:3 the _q;4:4 CFTC named;6:6 division _n_of;8:9 named economic analysis;10:10 shows _v_1;12:12 "a _q;13:13 trade _n_of;14:14 is _v_id;15:15 a _q;16:16 trade," _n_of;17:17 a _q;18:18 member _n_of;20:20 the _q;21:22 _n_of study team;0:0 The _q', '0:0 Whether _x;1:1 a _q;2:2 trade _n_of;4:4 done _v_1;5:5 on _p;6:6 a _q;7:7 dual _a_1;8:8 or _c;10:10 dual _a_1;11:11 basis _n_of;12:12 the _q;13:13 member _n_of;14:16 _v_to said seem;18:18 have _v_1;20:20 economic _a_1;0:0 Whether _x', '0:0 "The _q;1:1 tests _n_of;2:2 indicate _v_1;4:4 dual _a_1;5:5 and _c;7:7 dual _a_1;8:8 traders _n_1;10:10 similar _a_to;11:11 in _p;12:12 terms _n_of;14:14 the _q;15:15 trade _n_of;16:16 executions _n_1;17:17 and _c;18:18 liquidity _n_1;20:20 provide _v_1;22:22 the _q;23:24 _n_1 market," Mrs;25:25 Gramm named;26:26 told _v_1;27:27 the _q;28:28 Senate named_n;0:0 "The _q', '0:0 Members _n_of;2:2 Congress named;4:5 _v_to proposed restricting;6:6 dual _a_1;7:7 trading _v_1;8:8 in _p;9:9 bills _n_of;11:11 reauthorize _v_1;11:11 reauthorize _re-_a_again;12:12 CFTC named;0:0 Members _n_of', '0:0 The _q;1:1 House named_n;3:3 bill _n_of;4:4 would _v_modal;5:5 prohibit _v_from;6:6 dual _a_1;7:7 trading _v_1;8:8 in _p;9:9 markets _n_1;10:10 with _p;11:12 _a_1 daily average;13:13 volume _n_of;15:15 7,000 card;16:16 contracts _n_1;17:17 or _c;19:19 comprising _v_1;20:20 those _q_dem;21:21 considered _v_1;23:23 difficult _a_for;25:25 track _v_1;26:26 without _p;27:27 a _q;28:28 sophisticated _a_1;29:29 computer _n_1;0:0 The _q', "0:0 The _q;1:1 Senate named_n;2:2 bill _n_of;3:3 would _v_modal;4:4 force _v_1;5:5 the _q;6:6 CFTC named;8:8 suspend _v_1;9:9 dual _a_1;10:10 trading _v_1;11:11 if _x_then;12:12 an _q;13:13 exchange _n_1;14:14 can't _v_modal;15:15 show _v_1;18:18 oversight _n_1;19:19 system _n_of;20:20 can _v_modal;21:23 _v_1 detect trading;0:0 The _q", '1:1 far _a_1;2:2 one card;3:3 test _n_of;5:5 restricting _v_to;6:6 dual _a_1;7:9 _v_1 trading worked;1:1 far _a_1', '0:0 The _q;1:2 named Chicago Merc;3:3 banned _v_from;4:4 dual _a_1;5:5 trading _v_1;6:6 in _p;8:8 Standard named;9:9 & _c;10:10 Poor named;13:13 stock _n_1;13:13 stock card;14:15 _n_1 index futures;16:16 pit _n_1;17:17 in _p_temp;0:0 The _q', '0:0 Under _p_state;1:1 the _q;2:2 rules _n_of;3:3 traders _n_1;4:4 decide _v_1;5:5 before _x_h;6:6 a _q;7:7 session _n_1;8:12 _v_1 begins trade;13:13 for _p;15:15 own _a_1;16:16 account _n_of;17:17 or _c;18:18 for _p;0:0 Under _p_state', "0:0 Traders _n_1;2:2 stand _v_1;3:3 on _p;4:4 the _q;5:5 pit _n_1;7:7 top _a_1;8:8 step _n_1;10:10 most _q;11:12 _n_of customer orders;14:14 executed _v_1;15:15 can't _v_modal;16:16 trade _v_1;0:0 Traders _n_1", '0:0 A _q;1:1 Merc named;2:2 spokesman _n_1;3:3 said _v_to;4:4 the _q;5:5 plan _n_1;7:7 made _v_1;9:9 difference _n_1;10:10 in _p;11:11 liquidity _n_1;12:12 in _p;13:13 the _q;0:0 A _q', '3:3 soon _p;5:5 tell _v_1;6:6  .  <unk>;7:7 but _c;8:8 people _n_of;10:10 seem _v_to;13:13 unhappy _a_1;14:14 with _p;3:3 soon _p', "1:1 said _v_to;3:3 wouldn't _v_modal;4:4 comment _v_on;6:6 the _q;7:7 CFTC named;8:8 plan _n_1;9:9 until _x_h;10:10 the _q;11:11 exchange _n_1;13:13 seen _v_1;14:14 the _q;15:15 full _a_of;1:1 said _v_to", '0:0 But _c;1:1 at _p;2:2 a _q;3:3 meeting _n_of;4:4 last _a_1;5:5 week _n_1;6:7 named Tom Donovan;8:8 the _q;9:9 Board _n_of;11:11 Trade named;13:13 president _n_of;14:14 told _v_1;15:16 _n_1 commodity lawyers;17:17 : _p_namely;18:18 "Dual _a_1;19:19 trading _v_1;21:22 _a_1 definitely worth;0:0 But _c', '1:1 adds _v_to;2:4 _q something the;1:1 adds _v_to', '0:0 Japanese _a_1;1:1 Firms _n_1;2:2 Push _v_1;3:3 Posh _a_1;4:5 _n_1 Car Showrooms', '0:0 JAPANESE _a_1;1:2 _n_1 luxury- car;3:3 makers _n_of;5:7 _v_1 trying set;8:8 strict _a_1;9:9 design _n_of;10:10 standards _n_1;11:11 for _p;0:0 JAPANESE _a_1', '0:0 But _c;1:1 some _q;2:2 dealers _n_1;4:4 negotiating _v_1;5:5 looser _a_1;6:6 terms _n_of;7:7 while _x;8:8 others _n_1;9:11 _v_1 decline deal;12:13 _a_1 at all', '0:1 named Nissan Motor;2:2 Co _n_1;4:4 Infiniti named;5:5 division _n_of;6:8 _v_1 likes insist;10:10 every _q;11:11 dealer _n_1;12:12 construct _v_1;13:13 and _c;14:14 furnish _v_1;15:15 a _q;16:16 building _n_1;17:17 in _p;18:18 a _q;19:19 Japanese _a_1;0:0 Nissan named', '0:1 named Nissan Motor;2:2 Co _n_1;4:4 Infiniti named;5:5 division _n_of;6:8 _v_1 likes insist;10:10 every _q;11:11 dealer _n_1;12:12 construct _v_1;13:13 and _c;14:14 furnish _v_1;15:15 a _q;16:16 building _n_1;17:17 in _p;18:18 a _q;19:19 Japanese _a_1;0:0 Nissan named', '0:0 Specifications _n_for;1:1 include _v_1;2:2 a _q;3:3 polished _a_1;4:5 _n_1 bronze sculpture;6:6 at _p;7:7 the _q;8:8 center _n_of;10:10 each _q;11:11 showroom _n_1;12:12 and _c;13:13 a _q;14:15 _n_1 tile bridge;16:16 spanning _v_1;17:17 a _q;18:18 stream _n_of;20:20 flows _v_1;21:21 into _p;22:22 the _q;23:23 building _n_1;24:25 _p from outside', '0:0 "Infiniti named;1:1 has _v_prd;3:4 _p down to;5:5 the _q;6:6 ashtrays," _n_unknown;7:7 says _v_to;8:9 named Jay Ferron;10:10 a _q;11:11 partner _n_1;12:12 at _p;13:14 named J.D Power;15:15 & _c;16:16 Associates named;17:17 an _q;18:19 _n_1 auto research;0:0 "Infiniti named', '0:1 named Toyota Motor;2:2 Corp _n_1;4:4 Lexus named;5:5 division _n_of;6:6 also _a_1;7:7 provides _v_1;0:0 Toyota named', '0:0 But _c;1:1 only _x_deg;3:3 thirds _n_of;3:3 thirds card;4:4 of _p;5:5 Lexus named;6:6 dealers _n_1;8:8 constructing _v_1;9:9 new _a_1;10:10 buildings _n_1;11:12 _p according to;13:13 the _q;14:14 Lexus named;0:0 But _c', '0:0 Some _q;2:2 even _a_1;3:3 coming _v_up;5:5 with _p;7:7 own _a_1;8:9 _n_of novel designs', '0:0 In _p_state;1:2 named Louisville Ky;5:6 named David Peterson;8:8 built _v_1;9:9 a _q;10:10 Lexus named;11:11 dealership _n_1;12:12 with _p;13:13 the _q;14:14 showroom _n_1;15:15 on _p;16:16 the _q;17:17 second ord;0:0 In _p_state', '0:0 Yet _c;1:1 some _q;2:2 dealers _n_1;4:4 turned _v_off;6:6 Infiniti named;7:7 or _c;8:8 Lexus named;9:9 franchises _n_1;10:10 because _x;13:13 unwilling _a_1;14:14 or _c;15:15 unable _a_1;17:17 meet _v_1;18:18 the _q;19:19 design _n_of;0:0 Yet _c', '0:1 named Lee Seidman;2:2 of _p;3:3 Cleveland named;4:4 says _v_to;5:5 Infiniti named;6:6 "was _v_id;7:7 a _q;8:8 bear _n_1;9:9 on _p;10:10 interiors" _n_1;11:11 but _c;12:13 _a_1 at least;14:16 _v_1 let retrofit;16:16 retrofit _re-_a_again;17:17 an _q;18:18 existing _v_1;19:19 building _n_1;21:21 without _p;22:22 the _q;0:0 Lee named', '0:0 Mr _n_1;1:1 Seidman named;2:2 says _v_to;4:4 turned _v_down;6:6 a _q;7:7 Lexus named;8:8 franchise _n_1;9:9 in _p;10:10 part _n_1;11:11 because _x;12:12 "the _q;13:13 building _n_1;15:15 gorgeous _a_1;16:16 but _c;17:17 very _x_deg;0:0 Mr _n_1', '0:0 To _x;1:1 head _v_off;3:3 arguments _n_1;4:4 Infiniti named;5:5 offers _v_1;6:7 _n_1 dealers cash;8:8 bonuses _n_unknown;9:9 and _c;11:11 interest _n_of;11:11 interest _high_a_1;12:12 construction _n_of;0:0 To _x', '0:0 Dictation _n_of;1:3 _n_1 Device Saga;4:4 Plays _v_down;6:6 a _q;0:0 Dictation _n_of', '0:0 PRODUCTS _n_1;2:2 have _v_qmodal;4:4 be _v_id;5:5 first _a_1;6:6 to _x;7:7 be _v_id;0:0 PRODUCTS _n_1', "0:0 That _q_dem;1:1 's _v_id;2:2 the _q;3:3 lesson _n_1;4:4 offered _v_1;5:5 through _p;6:6 one card;7:8 _n_of case study;9:9 featured _v_cause;10:10 in _p;11:11 a _q;12:12 design _n_of;0:0 That _q_dem", '0:0 Blocked _v_1;2:3 _n_of patent protection;4:4 from _p;5:5 following _v_1;6:6 suit _n_i;7:7 Dictaphone named;8:10 _v_1 decided go;11:11 a card;12:12 step _n_item;13:13 further _a_1;14:14 and _c;15:15 cut _v_1;16:16 the _q;17:17 cassette _n_1;18:18 in _p;19:19 half _q;20:20 again _a_1;22:23 _p down to;24:24 the _q;25:25 length _n_of;27:27 a _q;0:0 Blocked _v_1', '0:0 By _p_temp;1:1 1979 yofc;2:2 designers _n_of;3:3 and _c;4:4 engineers _n_1;5:5 at _p;6:6 Dictaphone named;7:7 a _q;8:9 named Pitney Bowes;10:10 subsidiary _n_1;12:12 produced _v_1;13:13 a _q;14:14 working _v_1;15:15 model _n_of;17:17 a _q;18:18 "picocassette" _n_unknown;0:0 By _p_temp', '0:0 By _p_temp;1:1 1982 yofc;3:3 the _q;4:4 patent _n_of;5:5 status _n_of-on;7:7 the _q;8:8 Lanier _a_1;9:9 microcassette _n_unknown;11:12 _v_1 changed permitting;13:13 Dictaphone named;15:15 develop _v_cause;17:18 _a_1 own competitive;19:19 micro _n_1;20:20 system _n_of;0:0 By _p_temp', '0:0 Marketing _v_1;1:1 and _c;2:2 sales _n_of;3:3 departments _n_1;4:4 then _a_1;5:5 urged _v_1;6:6 abandonment _n_of;7:7 of _p;8:8 the _q;9:9 pico named;0:0 Marketing _v_1', '0:0 But _c;1:1 others _n_1;2:2 said _v_to;3:3 pico named;4:4 should _v_modal;0:0 But _c', '0:0 Both _q;0:0 Both card;0:0 Both _q', '0:0 Dictaphone named;1:1 went _v_back;3:3 and _c;4:4 introduced _v_to;5:5 the _q;6:6 pico named;7:7 in _p_temp;8:8 1985 yofc;9:9 but _c;12:12 sold _v_1;0:0 Dictaphone named', '0:1 _p To date;2:2 says _v_to;3:4 named Emil Jachmann;5:5 a _q;6:6 Dictaphone _n_unknown;7:7 vice _n_1;8:8 president _n_of;11:11 "broken _v_1;12:12 even _a_1;13:13 or _c;14:14 shown _v_1;15:15 a _q;16:16 small _a_1;0:0 To _p', '0:0 Nevertheless _a_1;1:1 the _q;2:2 device _n_1;5:5 successful _a_1;6:6 in _p;7:7 other _a_1;0:0 Nevertheless _a_1', '1:1 helped _v_to;2:2 Dictaphone _n_unknown;3:3 attract _v_1;4:4 better _a_at-for-of;5:5 engineers _n_1;6:6 and _c;8:8 provided _v_1;9:9 new _a_1;10:10 technology _n_1;11:11 for _p;12:12 other _a_1;13:13 company _n_of;1:1 helped _v_to', '0:0 The _q;1:1 picocassette named;2:2 recorder _n_of;3:3 also _a_1;4:4 helped _v_to;5:5 transform _v_into;6:6 the _q;7:7 company _n_of;9:9 reputation _n_1;10:10 from _p;11:11 follower _n_of;12:12 to _p;13:13 leading- _v_1;14:14 edge _n_of;0:0 The _q', '0:2 named Dictaphone picocassette;3:3 recorder _n_of;4:4 is _v_id;5:7 card one 13;8:9 _n_of case studies;10:10 in _p;11:11 the _q;12:12 TRIAD named;13:13 Design _n_of;14:14 Project _n_1;15:15 sponsored _v_1;17:17 the _q;18:19 named Design Management;20:22 named Institute Boston;23:23 and _c;24:25 named Harvard Business;0:0 Dictaphone named', '0:0 The _q;1:1 studies _n_of;3:3 on _p;4:4 exhibit _n_1;5:5 at _p;6:6 Harvard named;7:7 this _q_dem;8:8 month _n_1;9:9 and _c;11:11 travel _v_1;12:12 to _p;13:13 Chicago named;15:15 Institute _n_of;16:16 of _p;17:17 Design named;18:18 and _c;19:19 the _q;20:20 University _n_of;22:22 California named;23:23 at _p;0:0 The _q', '0:0 A _q;1:1 Rake named;3:3 Progress _n_1;4:4 Means _v_1;0:0 A _q', '0:0 ONE card;1:1 DAY _n_of;2:3 named Carl Barrett;4:4 of _p;5:6 named Mobile Ala;8:8 raking _v_1;9:9 some _q;10:10 sycamore _n_unknown;11:11 leaves _n_1;12:12 but _c;13:13 the _q;14:14 rake _n_1;15:15 kept _v_on;16:16 riding _v_1;17:17 up _p_dir;18:18 over _p;19:19 the _q;0:0 ONE card', '1:1 crude _a_1;2:2 device _n_1;3:3 worked _v_1;4:4 : _p_namely;5:5 The _q;6:6 lower _a_1;7:7 teeth _n_1;8:8 gathered _v_cause;9:9 the _q;10:10 leaves _n_1;11:11 into _p;12:12 a _q;13:13 pile _n_of;14:14 while _x;15:15 the _q;16:16 higher _a_1;17:17 harder _a_for;18:18 teeth _n_1;19:19 moved _v_cause;20:20 the _q;21:21 top _n_1;22:22 of _p;23:23 the _q;1:1 crude _a_1', '0:0 Now _a_1;1:1 incorporated _v_1;2:2 into _p;3:3 a _q;4:4 polypropylene _n_unknown;5:5 rake _n_1;6:6 the _q;8:8 inch _n_1;8:8 inch card;9:9 prongs _n_1;10:10 or _c;11:11 "wonderbars," _n_unknown;12:14 _a_1 also supposed;16:16 aid _v_1;17:17 in _p;18:18 picking _v_up;0:0 Now _a_1', '0:0 One card;1:1 customer _n_of;2:3 named Donald Blaggs;4:4 of _p;5:5 Mobile named;6:6 says _v_to;7:7 the _q;8:9 named Barrett Rake;10:13 _v_1 allowed do;15:15 lawn _n_1;16:16 in _p;17:17 2 card;18:18 1/2 fraction;19:19 hours _n_1;20:20 two card;21:21 hours _n_1;22:23 _p less than;0:0 One card', '0:0 But _c;1:1 other _a_1;2:2 rake named;3:3 makers _n_of;4:4 have _v_1;0:0 But _c', '0:1 named Richard Mason;2:2 president _n_of;4:4 Ames named;5:5 Co _n_1;6:6 in _p;7:8 named Parkersburg W;9:9 Va named;10:10 says _v_to;11:11 the _q;12:13 named Barrett rake;14:14 "makes _v_1;15:15 sense," _n_of;16:16 but _c;18:18 would _v_modal;20:20 "tough" _a_for;22:22 explain _v_to;23:23 to _p;0:0 Richard named', '0:1 named John Stoner;2:2 marketing _v_1;3:3 director _n_of;4:4 for _p;5:6 named True Temper;7:7 Corp _n_1;8:8 a _q;9:9 subsidiary _n_1;10:10 of _p;11:11 Black named;12:12 & _c;13:13 Decker named;14:14 says _v_to;15:15 people _n_of;17:17 want _v_1;19:19 move _v_cause;20:20 a _q;21:21 leaf _n_1;0:0 John named', '2:2 pick _v_up;6:6 says _v_to;7:7 "or _c;9:10 _v_1 start pulling;11:11 from _p;12:12 a _q;13:13 fresh _a_1;2:2 pick _v_up', '0:0 Odds _n_1;1:1 and _c;0:0 Odds _n_1', '0:0 Four card;1:1 tubular _a_1;2:2 steel _n_1;3:3 "Bedfellows," _n_unknown;4:4 each _p;5:5 roughly _x_deg;6:6 in _p;7:7 the _q;8:8 shape _n_of;9:9 of _p;10:10 a _q;11:11 "W," named;13:13 attached _v_1;14:14 to _p;15:15 the _q;16:16 bottom _n_1;17:17 of _p;18:18 the _q;19:19 box _n_of;20:20 spring _n_1;21:21 in _p;22:22 a _q;23:23 recessed _v_1;24:24 position _n_of;25:25  . .  _n_1;0:0 Four card', '0:0 Nearly _x_deg;1:1 half _q;3:3 U.S named_n;4:4 consumers _n_of;5:5 say _v_to;8:8 pay _v_for;9:10 _x_deg up to;11:11 5 card;12:12 % _n_of;14:14 for _p;15:15 packaging _v_1;17:17 can _v_modal;19:19 recycled _v_1;20:20 or _c;22:22 biodegradable _a_unknown;23:24 _p according to;25:25 a _q;26:26 survey _n_1;27:27 commissioned _v_1;29:29 the _q;30:31 named Michael Peters;32:32 Group named;33:33 a _q;34:34 design _n_of;0:0 Nearly _x_deg', '0:0 The _q;1:1 Pentagon named;2:2 is _v_id;3:3 a _q;4:4 haunted _v_1;0:0 The _q', '0:0 Living _v_1;1:1 there _a_1;2:2 for _p;3:3 six card;4:4 years _n_1;6:6 really _x_deg;0:0 Living _v_1', '0:0 The _q;1:1 ghosts _n_1;2:2 of _p;3:3 the _q;4:4 past _n_1;7:7 : _p_namely;10:10 kept _v_prd;11:11 at _p;12:12 bay _n_i;13:13